/*!****************************************************************************
 @File          OGLESShadowTechniques.cpp

 @Title         ShadowTechniques

 @Author        PowerVR

 @Date          14/01/2005

 @Copyright     Copyright 2005 by Imagination Technologies Limited.
                All rights reserved. No part of this software, either
                material or conceptual may be copied or distributed,
                transmitted, transcribed, stored in a retrieval system
                or translated into any human or computer language in any
                form by any means, electronic, mechanical, manual or
                other-wise, or disclosed to third parties without the
                express written permission of Imagination Technologies
                Limited, Unit 8, HomePark Industrial Estate,
                King's Langley, Hertfordshire, WD4 8LZ, U.K.

 @Platform      Independant

 @Description   Demonstrates MBX shadow techniques.
				Requires the OGLESShell.

******************************************************************************/

/*************************
**		  Includes	    **
*************************/

#include <stdlib.h>
#include <math.h>

#include "OGLESShell.h"
#include "OGLESTools.h"

/* Enable the following 2 lines for memory leak checking - also see InitApplication function */
//#define _CRTDBG_MAP_ALLOC
//#include <crtdbg.h>

/* -- Monster Geometry and Textures -- */

/* Remove this line to load MD2 files from file rather than include files */
#define LOADFROMHEADER

/* The following 2 defines are used to select the MD2 model to use - only include one of the two */
#define MONSTER1
//#define MONSTER2

/* Following includes and defines select the correct MD2 model and textures */
#if defined MONSTER1
	#if defined LOADFROMHEADER
		#include "Monster1_Geo.h"
		#define MONSTER Monster1_Geo
	#else
		#define MONSTERFILENAME "./../../../Media/Monster1.md2"
	#endif

	#include "Monster1_Tex.h"
	#define MONSTERTEXTURE	Monster1_Tex
#else if defined MONSTER2
	#if defined LOADFROMHEADER
		#include "Monster2_Geo.h"
		#define MONSTER Monster2_Geo
	#else
		#define MONSTERFILENAME "./../../../Media/Monster2.md2"
	#endif

	#include "Monster2_Tex.h"
	#define MONSTERTEXTURE	Monster2_Tex
#endif

/* -- Scene Geometry -- */

#include "Boxes.H"

/* -- Scene Textures -- */

#include "Blob.h"
#include "Crate.h"
#include "Floor.h"
#include "LightSource.h"

/* -- Defines -- */

#ifndef PI
#define PI 3.14159f
#endif

#define OFFSETA -10.0f		// Polygon Offset Values - Might need to be tweaked for various HW
#define OFFSETB -15.0f		// Polygon Offset Values - These avoid ZFighting between floor and shadow

#define	FLOORSIZE	10.0f	// Used to generate the Floor Plane
#define FLOORHEIGHT	3.0f	// Used to position the Floor Plane

#define	CHARWIDTH	2.2f	// Used to draw the basic blob shadow

#define	LIGHTPOSSCALE	1.25f	// Light Position Scale used to position the light at correct distance

#define SELECTEDANIMATION		0		// MD2 Animation 0 = STAND - 1 = RUN - 2 = SHOOT - 6 = JUMP - 7 = SCRATCH - Etc...
#define ANIMATIONSTEP			0.125f	// Animation Time Increase Per Frame - Higher = Faster Animation
#define ENABLE_GL_LIGHTING		0		// Toggle for OGL Lighting
#define ENABLE_TEXTURES			1		// Toggle Textures
#define ENABLE_OBJ_ROT			0		// Rotate the whole scene
#define ENABLE_LIGHT_ROT		1		// Animate the Light Position

/* -- Build Options to Force a certain shadow type -- */

//#define DRAW_ADV_BLOB_SHADOW		1		// Force Adv Blob Shadow
//#define DRAW_PROJ_SHADOW			1		// Force Projected Shadow
//#define DRAW_BASE_BLOB_SHADOW		1		// Force Basic Blob Shadow
//#define DRAW_RENDER2TEX_SHADOW	1		// Force Render To Texture Shadow

/* -- PBuffer Options -- */

#define PBUFFERSIZE				128		// Render To Texture Size
#define ENABLE_PBUFFER_USAGE	0		// Do context swap with render to texture (PBuffer) or use main framebuffer to render to texture

/* -- Geometry drawing option -- */

#define USE_FANS_N_STRIPS		0		// Use Indexed Triangle List to draw or use Fans/Strips to draw

/* -- Light Animation Params -- */

#define MAXANIMATIONSTEPS		7		// Number of Entries in the Animation Table which controls the ShadowTechs shown

/* -- Number of frames for each render 2 texture shadow re-generation - helps with performance -- */

#define PBUFFERUPDATESKIP		2

/* Enums */

enum
{
  X, Y, Z, W
};

enum
{
  A, B, C, D
};

/* Set Matrix Enums */

enum MatrixEnums
{
	CAMERA,
	LIGHT,
	PROJTEX,
	PROJGEOM,
	BILLBOARD
};

/* Shadow Modes */

enum ShadowModes
{
	BASEBLOBMODE,
	ADVANCEDBLOBMODE,
	PROJGEOMMODE,
	R2TEXMODE
};

/* Actual Source Code Starts Here */

/****************************************************************************
** Class: OGLESShadowTechniques
****************************************************************************/
class OGLESShadowTechniques : public OGLESShell{

	/* -- Print3D Class Object -- */
	CPVRTPrint3D 	AppPrint3D;

	/* -- Texture IDs -- */
	GLuint gMonsterMap;
	GLuint gFloorMap;
	GLuint gBlobMap;
	GLuint gLightSourceMap;
	GLuint gCrateMap;

	/* -- Meshes Objects-- */
	CMD2Model		MonsterObj;
	unsigned short*	MonsterTriangleList;

	/* -- Shadow Projection on Floor -- */

	GLfloat floorVertices[4][3];
	GLfloat floorPlane[4];
	GLfloat floorShadow[16];

	/* -- Animation -- */
	int		gnObjRotCntr;
	float	gnLightRotCntr;
	bool	bAnimated;
	float	MyTime;
	
	int		gAnimTbl[8][3];
	float	fAnimationCounter;
	int		gnCurrentAnimation;
	ShadowModes		gnCurrentMode;

	unsigned long	OldTime;
	float			fFactor;
	float			fOldFactor;
	float			fOlderFactor;

	/* -- Camera -- */
	PVRTVECTOR3 MyFrom;
	PVRTVECTOR3 MyTo;
	PVRTVECTOR3 MyUp;

	PVRTMATRIX	MyRotMatrix,MyViewMatrix;
	PVRTMATRIX	MyRotViewMatrix;

	/* -- PBuffer Update Counter -- */
	int		gnPBufferUpdateCounter;

	/* -- Lighting -- */

	float fCurrentLightPos[4];
	float lightcoloramb[4];
	float lightcolordiff[4];
	float lightcolorspec[4];

	float shininess;

	/* Bounding Ellips Data */
	GLfloat Radius;
	GLfloat YMin;
	GLfloat YMax;

	/* -- Options -- */

	bool bColorEnable;
	bool bNormalEnable;
	bool bTexCoordEnable;

	/* -- PBuffer -- */

	EGLSurface MyPBufSurface;
	EGLDisplay CurrentDisplay;
	EGLContext CurrentContext;
	EGLSurface CurrentSurface;

	bool firsttime;				// Used to make sure that some PBuffer init work is only done once
	GLuint MyPBufferTexture;	// Texture ID for PBuffer

	/* Header Object to Lite Conversion */
	HeaderStruct_Mesh_Type** Meshes;

	public:
		OGLESShadowTechniques()
	{
		/* -- Init vars to default and/or start values -- */
		/* -- Textures IDs -- */
		gBlobMap		= 0;
		gCrateMap		= 0;
		gFloorMap		= 0;
        gLightSourceMap = 0;
		gMonsterMap		= 0;

		/* -- Data for Shadow Projection on Floor -- */
		floorVertices[0][0] = -20.0f;	floorVertices[0][1] = -3.0f;	floorVertices[0][2] =  20.0f;
		floorVertices[1][0] =  20.0f;	floorVertices[1][1] = -3.0f;	floorVertices[1][2] =  20.0f;
		floorVertices[2][0] =  20.0f;	floorVertices[2][1] = -3.0f;	floorVertices[2][2] = -20.0f;
		floorVertices[3][0] = -20.0f;	floorVertices[3][1] = -3.0f;	floorVertices[3][2] = -20.0f;

		/* -- Animation -- */
		gnObjRotCntr = 0;
		gnLightRotCntr = -145;
		bAnimated = true;
		MyTime=1.0f;
		gnCurrentAnimation=0;
		fAnimationCounter=0;

		gnPBufferUpdateCounter=0;

		/* -- Animation Table -- */
		gAnimTbl[0][0] = -145;	gAnimTbl[0][1] =   45;	gAnimTbl[0][2] = BASEBLOBMODE;
		gAnimTbl[1][0] =   45;	gAnimTbl[1][1] = -145;	gAnimTbl[1][2] = ADVANCEDBLOBMODE;
		gAnimTbl[2][0] = -145;	gAnimTbl[2][1] =   45;	gAnimTbl[2][2] = PROJGEOMMODE;
		gAnimTbl[3][0] =   45;	gAnimTbl[3][1] =  100;	gAnimTbl[3][2] = PROJGEOMMODE;		// Transistion
		gAnimTbl[4][0] =  100;	gAnimTbl[4][1] =  250;	gAnimTbl[4][2] = R2TEXMODE;
		gAnimTbl[5][0] =  250;	gAnimTbl[5][1] =  100;	gAnimTbl[5][2] = R2TEXMODE;
		gAnimTbl[6][0] =  100;	gAnimTbl[6][1] =   45;	gAnimTbl[6][2] = PROJGEOMMODE;		// Transistion
		gAnimTbl[7][0] =   45;	gAnimTbl[7][1] = -145;	gAnimTbl[7][2] = PROJGEOMMODE;		// Transistion

		gnCurrentMode = ((ShadowModes)gAnimTbl[0][2]);

		/* -- Lighting -- */
		fCurrentLightPos[0] = 10.0f;	fCurrentLightPos[1] = 10.0f;	fCurrentLightPos[2] = -10.0f;	fCurrentLightPos[3] = 0.0f;

		lightcoloramb[0] = 0.1f;	lightcoloramb[1] = 0.1f;	lightcoloramb[2] = 0.1f;	lightcoloramb[3] = 0.1f;
		lightcolordiff[0] = 0.2f;	lightcolordiff[1] = 0.2f;	lightcolordiff[2] = 0.2f;	lightcolordiff[3] = 1.0f;
		lightcolorspec[0] = 0.2f;	lightcolorspec[1] = 0.2f;	lightcolorspec[2] = 0.2f;	lightcolorspec[3] = 1.0f;

		shininess = 2.0f;

		Radius = 0;
		YMin = 0;
		YMax = 0;

		MyUp.x=0.0f;
		MyUp.y=1.0f;
		MyUp.z=0.0f;

		/* Set options based on Defines */
		if (ENABLE_GL_LIGHTING)
		{
			bColorEnable		=	false;
			bNormalEnable		=	true;
		}
		else
		{
			bColorEnable		=	true;
			bNormalEnable		=	false;
		}

		if (ENABLE_TEXTURES)
		{
			bTexCoordEnable	=	true;
		}
		else
		{
			bTexCoordEnable	=	false;
		}

		/* PBuffer Init */
		MyPBufSurface = EGL_NO_SURFACE;
		CurrentDisplay	= EGL_NO_DISPLAY;
		CurrentContext	= EGL_NO_CONTEXT;
		CurrentSurface = EGL_NO_SURFACE;

		firsttime=true;		// Need to do init once
	}

	/* OGLESShell functions */
	virtual bool InitApplication(int argc, char*argv[], unsigned int uWidth, unsigned int uHeight);
	virtual bool InitView(unsigned int uWidth, unsigned int uHeight);
	virtual void ReleaseView();
	virtual bool QuitApplication();
	virtual bool RenderScene();

	/****************************************************************************
	** Function Definitions
	****************************************************************************/
	void shadowMatrix(GLfloat shadowMat[16],  GLfloat groundplane[4],  GLfloat lightpos[4]);
	void findPlane(GLfloat plane[4],  GLfloat v0[3], GLfloat v1[3], GLfloat v2[3]);
	void DrawFloor();
	void RenderMD2(CMD2Model *Model, float time, bool bColorEnable, bool bNormalEnable, bool bTexCoordEnable, bool bComputeBBox, unsigned short* Indices, bool bFanStripMode);
	void DrawBlob();
	void DrawBaseBlob();
	void GetBoundingEllips(VERTTYPE *Vertices, int nNumOfVertices);
	void DrawProjectedShadow();
	void DrawAdvancedBlobShadow();
	EGLConfig SelectEGLConfig(long type);
	void DrawProjectedFloor();
	void DrawBillBoardQuad (PVRTVECTOR3 MyFrom, PVRTVECTOR3 MyTo,float x,float y,float z,float Size, GLuint pTexture);
	void DrawProjectedMeshes();
	void UpdateLightPosition();
	void SetupMatrices(MatrixEnums MatrixSelected);
	void DrawMeshes();
};

/*******************************************************************************
 * Function Name  : InitApplication
 * Inputs		  : argc, *argv[], uWidth, uHeight
 * Returns        : true if no error occured
 * Description    : Code in InitApplication() will be called by the Shell ONCE per 
 *					run, early on in the execution of the program.
 *					Used to initialize variables that are not dependant on the
 *					rendering context (e.g. external modules, loading meshes, etc.)
 *******************************************************************************/
bool OGLESShadowTechniques::InitApplication(int argc, char*argv[], unsigned int uWidth, unsigned int uHeight)
{
	// Enable the following lines for memory leak checking
	/* Check for memory leaks also checks for writing into free'd memory */
	//_CrtSetDbgFlag(_CRTDBG_DELAY_FREE_MEM_DF | _CRTDBG_LEAK_CHECK_DF | _CrtSetDbgFlag(_CRTDBG_REPORT_FLAG));

	return true;
}

/*******************************************************************************
 * Function Name  : QuitApplication
 * Returns        : true if no error occured
 * Description    : Code in QuitApplication() will be called by the Shell ONCE per 
 *					run, just before exiting the program.
 *******************************************************************************/
bool OGLESShadowTechniques::QuitApplication()
{
	return true;
}

/*******************************************************************************
 * Function Name  : InitView
 * Inputs		  : uWidth, uHeight
 * Returns        : true if no error occured
 * Description    : Code in InitView() will be called by the Shell upon a change 
 *					in the rendering context.
 *					Used to initialize variables that are dependant on the rendering 
 *					context (e.g. textures, vertex buffers, etc.)
 *******************************************************************************/
bool OGLESShadowTechniques::InitView(unsigned int uWidth, unsigned int uHeight)
{
	int err;
	SPVRTContext Context;

	/* Init Print3D Textures */
	err = AppPrint3D.SetTextures(&Context,uWidth,uHeight);
	if(err == false)
	{
		OGLESShellOutputDebug ("ERROR: Cannot initialise Print3D\n");
		return false;
	}

	/* Load Textures */
	err = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) MONSTERTEXTURE ,
		ShellTextureFilterModeLinear, &gMonsterMap);
	if(err)
		return false;

	err = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) Floor,
		ShellTextureFilterModeLinear, &gFloorMap);
	if(err)
		return false;

	err = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) Blob,
		ShellTextureFilterModeLinear, &gBlobMap);
	if(err)
		return false;

	err = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) LightSource,
		ShellTextureFilterModeLinear, &gLightSourceMap);
	if(err)
		return false;

	err = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) Crate,
		ShellTextureFilterModeLinear, &gCrateMap);
	if(err)
		return false;

	/* Load MD2 Models either from File or from Header Files included */
	#if(OGLESLITE)
	
		#if defined LOADFROMHEADER
			if(!MonsterObj.LoadModelFixedFromHeader((char*)MONSTER))
			{
				OGLESShellOutputDebug("Failed to load Model !\n");
				return false;
			}
		#else
			if(!MonsterObj.LoadModelFixedFromFile(MONSTERFILENAME))
			{
				OGLESShellOutputDebug("Failed to load Model !\n");
				return false;
			}
		#endif
	
	#else
	
		#if defined LOADFROMHEADER
			if(!MonsterObj.LoadModelFloatFromHeader((char*)MONSTER))
			{
				OGLESShellOutputDebug("Failed to load Model !\n");
				return false;
			}
		#else
			if(!MonsterObj.LoadModelFloatFromFile(MONSTERFILENAME))
			{
				OGLESShellOutputDebug("Failed to load Model !\n");
				return false;
			}
		#endif
	
	#endif

	/* Set Selected Animation and Global Scale Factor */
	MonsterObj.SetAnim( SELECTEDANIMATION	);
	MonsterObj.ScaleModel( 0.125f );

	/* Generate Indexed Triangle List data for rendering MD2 Model */
	MonsterTriangleList = MonsterObj.GenerateTriangleList();

	/* Load 3DSMax Geometry for scenery */
	/* Header Objects */
	if(!OGLESShellInitIsFromLostContext())
	{
		Meshes = new HeaderStruct_Mesh_Type*[NUM_MESHES];
		for(int i = 0; i < NUM_MESHES; i++)
		{
			Meshes[i] = PVRTLoadHeaderObject(&Mesh[i]);
		}
	}	

	/* OpenGL Lighting Setup and Init */
	glEnable ( GL_LIGHTING );
	glEnable ( GL_LIGHT0 );

	myglLightv( GL_LIGHT0, GL_POSITION, fCurrentLightPos );
	myglLightv( GL_LIGHT0, GL_AMBIENT, lightcoloramb);
	myglLightv( GL_LIGHT0, GL_DIFFUSE, lightcolordiff );
	myglLightv( GL_LIGHT0, GL_SPECULAR, lightcolorspec );

	myglMaterialv(GL_FRONT_AND_BACK,GL_SHININESS,&shininess);
	myglMaterialv(GL_FRONT_AND_BACK,GL_SPECULAR, lightcolorspec);

	/* General OpenGL Setup of States */

	// smoothing polygons
	glShadeModel( GL_SMOOTH );

	// polygon offset for shadow to avoid ZFighting between the shadow and floor
	myglPolygonOffset(OFFSETA,OFFSETB);

	/* PBuffer Init Setup */

	EGLint list[5];

	/* Set PBuffer size in options */
	list[0]=EGL_WIDTH;
	list[1]=PBUFFERSIZE;
	list[2]=EGL_HEIGHT;
	list[3]=PBUFFERSIZE;
	list[4]=EGL_NONE;

	if (ENABLE_PBUFFER_USAGE)
	{
		/* Create PBuffer */
		MyPBufSurface = eglCreatePbufferSurface(eglGetCurrentDisplay(), SelectEGLConfig(2), list);
		if(MyPBufSurface==EGL_NO_SURFACE)
		{
			return false;
		}

		/* Get and Store current Display, Context and Surface - required to toggle between PBuffer and Main Render Surface */
		CurrentDisplay = eglGetCurrentDisplay();
		if(CurrentDisplay == EGL_NO_DISPLAY)
		{
			return false;
		}

		CurrentContext = eglGetCurrentContext();
		if (CurrentContext == EGL_NO_CONTEXT)
		{
			return false;
		}

		CurrentSurface = eglGetCurrentSurface(EGL_DRAW);
		if (CurrentSurface == EGL_NO_SURFACE)
		{
			return false;
		}
	}

	/* Setup floor plane for projected shadow calculations. */
	findPlane(floorPlane, floorVertices[1], floorVertices[2], floorVertices[3]);

	/* Init Time related variables */

	OldTime = OGLESShellGetTime();
	fFactor = 1.0f;
	fOldFactor = 1.0f;
	fOlderFactor = 1.0f;

	return true;
}

/*******************************************************************************
 * Function Name  : ReleaseView
 * Returns        : Nothing
 * Description    : Code in ReleaseView() will be called by the Shell before 
 *					changing to a new rendering context.
 *******************************************************************************/
void OGLESShadowTechniques::ReleaseView()
{
	/* Release All Textures : Print3D, PBuffer, Loaded from Header */
	AppPrint3D.ReleaseTextures();

	OGLESShellReleaseHeaderTexture(gMonsterMap);
	OGLESShellReleaseHeaderTexture(gFloorMap);
	OGLESShellReleaseHeaderTexture(gBlobMap);
	OGLESShellReleaseHeaderTexture(gLightSourceMap);
	OGLESShellReleaseHeaderTexture(gCrateMap);

	if (ENABLE_PBUFFER_USAGE)
	{
		/* Release PBuffer Texture */
		glDeleteTextures(1,&MyPBufferTexture);

		/* Destroy PBuffer */
		eglDestroySurface(CurrentDisplay,MyPBufSurface);
	}

	/* Header Objects */
	if(!OGLESShellInitIsFromLostContext())
	{
		for(int i = 0; i < NUM_MESHES; i++)
		{
			PVRTUnloadHeaderObject(Meshes[i]);
		}
	}
}


/*******************************************************************************
 * Function Name  : RenderScene
 * Returns		  : 0 if no error occured
 * Description    : Main rendering loop function of the program. The shell will
 *					call this function every frame.
 *******************************************************************************/
bool OGLESShadowTechniques::RenderScene()
{
	unsigned long CurrTime;

	/* Update Light Position and Rotation Params */
	UpdateLightPosition();

	/* If required do Render 2 Texture - must be correct mode and not a skipping frame*/
	if((gnCurrentMode==R2TEXMODE) && ((gnPBufferUpdateCounter%PBUFFERUPDATESKIP)==0))
	{
		/* Setup PBuffer Render or alternatively use Back Buffer*/
		if (ENABLE_PBUFFER_USAGE)
		{
			if (!eglMakeCurrent(CurrentDisplay, MyPBufSurface, MyPBufSurface, CurrentContext))
			{
				OGLESShellOutputDebug("Unable to make the pbuffer context current");
				return false;
			}
		}

		/* Set Viewport and do a Clear */
		glViewport (0, 0, PBUFFERSIZE, PBUFFERSIZE);
		myglClearColor(1,1,1,1);
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

		/* Setup Matrices to render from the point of view of the lightsource */
        SetupMatrices(LIGHT);

		glEnable(GL_DEPTH_TEST);

		// Disable Texture
		glActiveTexture(GL_TEXTURE0);
		glDisable(GL_TEXTURE_2D);

		// Set the Shadow Color and Alpha
		myglColor4(0.25f,0.25f,0.25f,0.0f);

		// Render the objects into the texture
		RenderMD2(  &MonsterObj,bAnimated ? MyTime : 0.0f, false, false, false, false, MonsterTriangleList,USE_FANS_N_STRIPS );

		/* Only do the following setup work the first time we render to the PBuffer */
		if (firsttime)
		{
			/* Create Texture ID */
			glGenTextures(1,&MyPBufferTexture);

			/* Bind and Enable Texture ID */
			glActiveTexture(GL_TEXTURE0);
			glBindTexture(GL_TEXTURE_2D,MyPBufferTexture);
			glEnable(GL_TEXTURE_2D);

			/* If Tex Params are not set glCopyTexImage2D will fail ! */
			myglTexParameter(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
			myglTexParameter(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);

			myglTexParameter(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_CLAMP_TO_EDGE);
			myglTexParameter(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_CLAMP_TO_EDGE);

			/* Initialise the Texture */
			glTexImage2D(GL_TEXTURE_2D,0,GL_RGB,PBUFFERSIZE,PBUFFERSIZE,0,GL_RGB,GL_UNSIGNED_BYTE,0);

			/* Only do this once */
			firsttime=false;
		}

		/* Make the copy using the pbuffer as source ... or backbuffer if pbuffer is disabled */
		glActiveTexture(GL_TEXTURE0);
		glEnable(GL_TEXTURE_2D);
		glBindTexture(GL_TEXTURE_2D,MyPBufferTexture);
		glCopyTexImage2D(GL_TEXTURE_2D, 0, GL_RGB, 0, 0, PBUFFERSIZE, PBUFFERSIZE, 0);

		if (ENABLE_PBUFFER_USAGE)
		{
			/* Switch back to the backbuffer for rendering etc. */
			if (!eglMakeCurrent(CurrentDisplay, CurrentSurface, CurrentSurface, CurrentContext))
			{
				OGLESShellOutputDebug("Unable to make the main context current");
				return false;
			}
		}
	}

	/* Keep track of the PBuffer updates */
	gnPBufferUpdateCounter++;

	/* Setup and Clear for the backbuffer */
	glViewport(0,0,OGLESShellGetShellSizeX(),OGLESShellGetShellSizeY());
	myglClearColor(0,0,0,1);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* Setup the Matrices to render from the Camera point of view */
	SetupMatrices(CAMERA);

	glEnable(GL_DEPTH_TEST);

	if (ENABLE_GL_LIGHTING)
	{
		glEnable(GL_LIGHTING);
	}

	if (ENABLE_TEXTURES)
	{
		glEnable(GL_TEXTURE_2D);
	}
	else
	{
		glDisable(GL_TEXTURE_2D);
	}

	myglColor4(1.0f,1.0f,1.0f,1.0f);

	/* Draw the Monster */
	/* Set Texture*/
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D,gMonsterMap);

	/* Draw Geometry */
	RenderMD2(&MonsterObj,bAnimated ? MyTime : 0.0f, bColorEnable, bNormalEnable, bTexCoordEnable, (gnCurrentMode==ADVANCEDBLOBMODE) ? true:false, MonsterTriangleList,USE_FANS_N_STRIPS );

	/* Draw the Environment and Shadows */

	/* Draw the Floor */

	glDisable(GL_LIGHTING);

	if(gnCurrentMode==R2TEXMODE)
	{
		/* Set correct Texture*/
		glActiveTexture(GL_TEXTURE0);
		glBindTexture(GL_TEXTURE_2D, MyPBufferTexture);

		/* Draw Geometry with Projected Shadow Texture Layer */

		/* Setup Second Layer for Texturing Floor*/
		glActiveTexture(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D,gFloorMap);
		glEnable(GL_TEXTURE_2D);

		/* Draw Floor Geometry Dual Textured */
        DrawProjectedFloor();

		/* Setup Second Layer for Texturing Crates*/
		glActiveTexture(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D,gCrateMap);
		glEnable(GL_TEXTURE_2D);

		/* Draw Crates Geometry Dual Textured */
		DrawProjectedMeshes();

		/* Disable Second Texture Layer */
		glActiveTexture(GL_TEXTURE1);
		glBindTexture(GL_TEXTURE_2D,NULL);
		glDisable(GL_TEXTURE_2D);
	}
	else
	{
		/* Set correct Texture*/
		glActiveTexture(GL_TEXTURE0);
		glBindTexture(GL_TEXTURE_2D, gFloorMap);

		/* Draw Geometry: Crates and Floor */
		DrawFloor();

		/* Set Correct Texture */
		glBindTexture(GL_TEXTURE_2D,gCrateMap);
		DrawMeshes();
	}

	/* Render Different Shadow Types */
	if (gnCurrentMode==ADVANCEDBLOBMODE)
	{
		/* Setup work for Projecting the Shadow */
		SetupMatrices(PROJGEOM);

		/* Draw Shadow*/
		DrawAdvancedBlobShadow();

	} else 	if (gnCurrentMode==PROJGEOMMODE)
	{
		/* Setup work for Projecting the Shadow */
		SetupMatrices(PROJGEOM);

		/* Draw Shadow */
		DrawProjectedShadow();

	} else 	if(gnCurrentMode==BASEBLOBMODE)
	{
		/* Draw Geometry */
		DrawBaseBlob();
	}

	/* Update Time for animation */

	/* Grab Current Time in ms */
	CurrTime = OGLESShellGetTime();

	/* Calculate a Time Scale Factor using the Current and Old Time */
	fFactor = (((float)(CurrTime-OldTime))*0.03333f);

	/* Average out the Factor Change to avoid spike behaviour */
	fFactor = 0.3333f*(fOlderFactor+fOldFactor+fFactor);
	fOlderFactor = fOldFactor;
	fOldFactor = fFactor;
	
	/* Update the actual Animation Time used by the drawing routines */
	MyTime+=(ANIMATIONSTEP*fFactor);

	/* Store Current Time as Old Time for Next Frame */
	OldTime = CurrTime;

	/* Draw Light Source with Alpha Blending Enabled at the end - no lighting */

	if (ENABLE_GL_LIGHTING)
	{
		glDisable(GL_LIGHTING);
	}

	/* Setup Matrices to render from the point of view of the camera */
	SetupMatrices(CAMERA);

	/* Enable and Setup Blend Mode */
	glEnable(GL_BLEND);
	glBlendFunc(GL_ONE,GL_ONE);

	/* Draw Billboard geometry */
	DrawBillBoardQuad(MyFrom, MyTo, 0.75f*fCurrentLightPos[0],0.75f*fCurrentLightPos[1],0.75f*fCurrentLightPos[2],1.0f,gLightSourceMap);

	/* Disable Blending*/
	glDisable(GL_BLEND);

	/* Check for possible problems */
	if (glGetError())
	{
		_RPT0(_CRT_WARN,"glError Tracker : Error prior to Print3D Call found !\n");
	}

	/* Display info text. */
	switch(gnCurrentMode)
	{
		case BASEBLOBMODE:		AppPrint3D.DisplayDefaultTitle("ShadowTechniques", "Basic Static Blob", PVR_LOGO);
								break;
		case ADVANCEDBLOBMODE:	AppPrint3D.DisplayDefaultTitle("ShadowTechniques", "Dynamic Blob", PVR_LOGO);
								break;
		case PROJGEOMMODE:		AppPrint3D.DisplayDefaultTitle("ShadowTechniques", "Projected Geometry", PVR_LOGO);
								break;
		case R2TEXMODE:			AppPrint3D.DisplayDefaultTitle("ShadowTechniques", "Projected Render to Texture", PVR_LOGO);
								break;
	}

	AppPrint3D.Flush();

	return true;
}

/*******************************************************************************
 * Function Name  : shadowMatrix
 * Inputs		  : GroundPlane Equation and Lightposition
 * Outputs		  : Matrix
 * Description    : Create a matrix that will project the desired shadow.
 *******************************************************************************/
void OGLESShadowTechniques::shadowMatrix(GLfloat shadowMat[16],  GLfloat groundplane[4],  GLfloat lightpos[4])
{
  GLfloat dot;

  /* Find dot product between light position vector and ground plane normal. */
  dot = groundplane[X] * lightpos[X] +
    groundplane[Y] * lightpos[Y] +
    groundplane[Z] * lightpos[Z] +
    groundplane[W] * lightpos[W];

  shadowMat[0] = dot - lightpos[X] * groundplane[X];
  shadowMat[4] = 0.f - lightpos[X] * groundplane[Y];
  shadowMat[8] = 0.f - lightpos[X] * groundplane[Z];
  shadowMat[12] = 0.f - lightpos[X] * groundplane[W];

  shadowMat[1] = 0.f - lightpos[Y] * groundplane[X];
  shadowMat[5] = dot - lightpos[Y] * groundplane[Y];
  shadowMat[9] = 0.f - lightpos[Y] * groundplane[Z];
  shadowMat[13] = 0.f - lightpos[Y] * groundplane[W];

  shadowMat[2] = 0.f - lightpos[Z] * groundplane[X];
  shadowMat[6] = 0.f - lightpos[Z] * groundplane[Y];
  shadowMat[10] = dot - lightpos[Z] * groundplane[Z];
  shadowMat[14] = 0.f - lightpos[Z] * groundplane[W];

  shadowMat[3] = 0.f - lightpos[W] * groundplane[X];
  shadowMat[7] = 0.f - lightpos[W] * groundplane[Y];
  shadowMat[11] = 0.f - lightpos[W] * groundplane[Z];
  shadowMat[15] = dot - lightpos[W] * groundplane[W];

}

/*******************************************************************************
 * Function Name  : findPlane
 * Inputs		  : 3 Points
 * Outputs		  : Plane Equations
 * Description    : Find the plane equation given 3 points.
 *******************************************************************************/
void OGLESShadowTechniques::findPlane(GLfloat plane[4],  GLfloat v0[3], GLfloat v1[3], GLfloat v2[3])
{
  GLfloat vec0[3], vec1[3];

  /* Need 2 vectors to find cross product. */
  vec0[X] = v1[X] - v0[X];
  vec0[Y] = v1[Y] - v0[Y];
  vec0[Z] = v1[Z] - v0[Z];

  PVRTMatrixVec3Normalize((PVRTVECTOR3*)vec0,(PVRTVECTOR3*)vec0);

  vec1[X] = v2[X] - v0[X];
  vec1[Y] = v2[Y] - v0[Y];
  vec1[Z] = v2[Z] - v0[Z];

  PVRTMatrixVec3Normalize((PVRTVECTOR3*)vec1,(PVRTVECTOR3*)vec1);

  /* find cross product to get A, B, and C of plane equation */
  plane[A] = vec0[Y] * vec1[Z] - vec0[Z] * vec1[Y];
  plane[B] = -(vec0[X] * vec1[Z] - vec0[Z] * vec1[X]);
  plane[C] = vec0[X] * vec1[Y] - vec0[Y] * vec1[X];

  plane[D] = -(plane[A] * v0[X] + plane[B] * v0[Y] + plane[C] * v0[Z]);
}

/*******************************************************************************
 * Function Name  : RenderMD2
 * Inputs		  : Model data and a whole bunch of options (color, normals, texcoord, compute bounding box enables and render mode)
 * Description    : Draw an MD2 model using the options provided
 *******************************************************************************/
void OGLESShadowTechniques::RenderMD2(CMD2Model *Model, float time, bool bColorEnable, bool bNormalEnable, bool bTexCoordEnable, bool bComputeBBox, unsigned short* Indices, bool bFanStripMode)		// Render the model using DrawArrays - Fill Memory with Data and Render
{
	VERTTYPE *pPosition;
	VERTTYPE *pTexCoord;
	VERTTYPE *pNormal;
	unsigned char *pColor;
	int NumberOfFans;
	int NumberOfStrips;
	int Offset2Strips;
	int *pFanLengths;
	int *pStripLengths;

	int CurrentIndex = 0;
	int j;

	/* rotate the MD2 Model so its upright - model is stored in a non traditional orientation */
	glPushMatrix();
	myglRotate( -90.0, 1.0, 0.0, 0.0 );
	myglRotate( 90.0, 0.0, 0.0, 1.0 );

	/* Generate and Grab the Data */
	#if(OGLESLITE)
	{
		Model->GetModelDataPntrsFixed(time, &pPosition, bTexCoordEnable ? &pTexCoord:NULL, bNormalEnable ? &pNormal:NULL, bColorEnable ? &pColor:NULL, &NumberOfFans, &NumberOfStrips, &Offset2Strips, &pFanLengths, &pStripLengths);
	}
	#else
	{
		Model->GetModelDataPntrsFloat(time, &pPosition, bTexCoordEnable ? &pTexCoord:NULL, bNormalEnable ? &pNormal:NULL, bColorEnable ? &pColor:NULL, &NumberOfFans, &NumberOfStrips, &Offset2Strips, &pFanLengths, &pStripLengths);
	}
	#endif

	/* If required update the bounding box info */
	if (bComputeBBox)
	{
		GetBoundingEllips(pPosition, Model->GetNumberOfVertices());
	}

	/* Setup some render states */
	glFrontFace( GL_CW );

	/* Enable BackFace Culling */
	glEnable( GL_CULL_FACE );
	glCullFace( GL_BACK );

	/* Draw the Model */
	/* Enable the Client States and set Data Pointers as required by the options */
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,pPosition);

	if (bColorEnable)
	{
		glEnableClientState(GL_COLOR_ARRAY);
		glColorPointer(4,GL_UNSIGNED_BYTE,0,pColor);
	}

	if (bNormalEnable)
	{
		glEnableClientState(GL_NORMAL_ARRAY);
		glNormalPointer(VERTTYPEENUM,0,pNormal);
	}

	if (bTexCoordEnable)
	{
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2,VERTTYPEENUM,0,pTexCoord);
	}

	/* Draw the object either using fans and strips or using indexed triangles */
	if (bFanStripMode)
	{
		/*  Draw Fans */
		for (j=0;j<NumberOfFans; j++)
		{
			glDrawArrays(GL_TRIANGLE_FAN,CurrentIndex,pFanLengths[j]);
			CurrentIndex+=pFanLengths[j];
		}

		/* Draw Strips */
		for (j=0;j<NumberOfStrips; j++)
		{
			glDrawArrays(GL_TRIANGLE_STRIP,CurrentIndex,pStripLengths[j]);
			CurrentIndex+=pStripLengths[j];
		}
	}
	else
	{
		/* Draw using Indexed Triangles */
        glDrawElements(GL_TRIANGLES,Model->TriangleCount*3,GL_UNSIGNED_SHORT,Indices);
	}

	/* Disable required Client States */
	glDisableClientState(GL_VERTEX_ARRAY);

	if (bColorEnable)
	{
		glDisableClientState(GL_COLOR_ARRAY);
	}

	if (bNormalEnable)
	{
		glDisableClientState(GL_NORMAL_ARRAY);
	}

	if (bTexCoordEnable)
	{
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}

	/* Recover state */
	glDisable( GL_CULL_FACE );
	glPopMatrix();

}

/*******************************************************************************
 * Function Name  : DrawFloor
 * Inputs		  : None
 * Description    : Draw Simple Floor Object
 *******************************************************************************/
void OGLESShadowTechniques::DrawFloor()
{
	static VERTTYPE	Vertices[] = {
			f2vt(-FLOORSIZE)	, f2vt(-FLOORHEIGHT)	, f2vt(-FLOORSIZE),
			f2vt(FLOORSIZE)		, f2vt(-FLOORHEIGHT)	, f2vt(-FLOORSIZE),
			f2vt(-FLOORSIZE)	, f2vt(-FLOORHEIGHT)	, f2vt(FLOORSIZE),
	 		f2vt(FLOORSIZE)		, f2vt(-FLOORHEIGHT)	, f2vt(FLOORSIZE)
		};

	static VERTTYPE	Colours[] = {
			f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f),
			f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f),
			f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f),
	 		f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f)
		};

	static VERTTYPE	UVs[] = {
			f2vt(0.0f), f2vt(0.0f),
			f2vt(1.0f), f2vt(0.0f),
			f2vt(0.0f), f2vt(1.0f),
	 		f2vt(1.0f), f2vt(1.0f)
		};

	VERTTYPE *pVertices = ( (VERTTYPE*)&Vertices );
	VERTTYPE *pColours  = ( (VERTTYPE*)&Colours );
	VERTTYPE *pUV       = ( (VERTTYPE*)&UVs );

	/* Enable Client States and Set Data Pointers */
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,pVertices);

	glEnableClientState(GL_COLOR_ARRAY);
	glColorPointer(4,VERTTYPEENUM,0,pColours);

	glClientActiveTexture(GL_TEXTURE0);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2,VERTTYPEENUM,0,pUV);

	/* Draw Geometry */
	glDrawArrays(GL_TRIANGLE_STRIP,0,4);

	/* Disable Client States */
	glDisableClientState(GL_VERTEX_ARRAY);

	glDisableClientState(GL_COLOR_ARRAY);

	glClientActiveTexture(GL_TEXTURE0);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

}

/*******************************************************************************
 * Function Name  : DrawProjectedFloor
 * Inputs		  : None
 * Description    : Draw Floor Object with projected texture layer
 *******************************************************************************/
void OGLESShadowTechniques::DrawProjectedFloor()
{
	static VERTTYPE	Vertices[] = {
			f2vt(-FLOORSIZE)	, f2vt(-FLOORHEIGHT)	, f2vt(-FLOORSIZE),
			f2vt(FLOORSIZE)		, f2vt(-FLOORHEIGHT)	, f2vt(-FLOORSIZE),
			f2vt(-FLOORSIZE)	, f2vt(-FLOORHEIGHT)	, f2vt(FLOORSIZE),
	 		f2vt(FLOORSIZE)		, f2vt(-FLOORHEIGHT)	, f2vt(FLOORSIZE)
		};

	static VERTTYPE	Colours[] = {
			f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f),
			f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f),
			f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f),
	 		f2vt(1.0f), f2vt(1.0f), f2vt(1.0f), f2vt(1.0f)
		};

	static VERTTYPE	UV[] = {
			f2vt(-FLOORSIZE)	, f2vt(-FLOORHEIGHT)	, f2vt(-FLOORSIZE), f2vt(1.0f),
			f2vt(FLOORSIZE)		, f2vt(-FLOORHEIGHT)	, f2vt(-FLOORSIZE), f2vt(1.0f),
			f2vt(-FLOORSIZE)	, f2vt(-FLOORHEIGHT)	, f2vt(FLOORSIZE), f2vt(1.0f),
	 		f2vt(FLOORSIZE)		, f2vt(-FLOORHEIGHT)	, f2vt(FLOORSIZE), f2vt(1.0f)
		};

	static VERTTYPE	UVSecond[] = {
			f2vt(0.0f), f2vt(0.0f),
			f2vt(1.0f), f2vt(0.0f),
			f2vt(0.0f), f2vt(1.0f),
	 		f2vt(1.0f), f2vt(1.0f)
		};

	VERTTYPE *pVertices = ( (VERTTYPE*)&Vertices );
	VERTTYPE *pColours  = ( (VERTTYPE*)&Colours );
	VERTTYPE *pUV		= ( (VERTTYPE*)&UV );
	VERTTYPE *pUVSecond = ( (VERTTYPE*)&UVSecond );

	/* Enable Client States and Set Data Pointers */
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,pVertices);

	glEnableClientState(GL_COLOR_ARRAY);
	glColorPointer(4,VERTTYPEENUM,0,pColours);

	glClientActiveTexture(GL_TEXTURE1);
	glActiveTexture(GL_TEXTURE1);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2,VERTTYPEENUM,0,pUVSecond);
	glMatrixMode(GL_TEXTURE);
	glLoadIdentity();

	glClientActiveTexture(GL_TEXTURE0);
	glActiveTexture(GL_TEXTURE0);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(4,VERTTYPEENUM,0,pUV);

	/* Setup Matrix for Projection */
	SetupMatrices(PROJTEX);

	/* Draw Geometry */
	glDrawArrays(GL_TRIANGLE_STRIP,0,4);

	/* Disable Client States */
	glDisableClientState(GL_VERTEX_ARRAY);

	glDisableClientState(GL_COLOR_ARRAY);

	glClientActiveTexture(GL_TEXTURE1);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	glClientActiveTexture(GL_TEXTURE0);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	/* Disable Texture Matrix */
	glMatrixMode(GL_TEXTURE);
	glLoadIdentity();

}

/*******************************************************************************
 * Function Name  : DrawProjectedMeshes
 * Inputs		  : None
 * Description    : Draw Objects with projected texture layer
 *******************************************************************************/
void OGLESShadowTechniques::DrawProjectedMeshes()
{
	for(int i = 0; i < NUM_MESHES; i++)
	{
		VERTTYPE *pVertices = ( (VERTTYPE*)Meshes[i]->pVertex );
		VERTTYPE *pUV		= ( (VERTTYPE*)Meshes[i]->pUV );
		VERTTYPE *pUVSecond = ( (VERTTYPE*)Meshes[i]->pVertex );

		/* Enable Client States and Setup Data Pointers */
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3,VERTTYPEENUM,0,pVertices);

		glClientActiveTexture(GL_TEXTURE1);
		glActiveTexture(GL_TEXTURE1);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2,VERTTYPEENUM,0,pUV);
		glMatrixMode(GL_TEXTURE);
		glLoadIdentity();

		glClientActiveTexture(GL_TEXTURE0);
		glActiveTexture(GL_TEXTURE0);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(3,VERTTYPEENUM,0,pUVSecond);

		/* Setup Texture Matrix for Projection */
		SetupMatrices(PROJTEX);

		/* Draw Geometry */
		glDrawElements(GL_TRIANGLES,Meshes[i]->nNumFaces*3,GL_UNSIGNED_SHORT,Meshes[i]->pFaces);

		/* Disable Client States */
		glDisableClientState(GL_VERTEX_ARRAY);

		glClientActiveTexture(GL_TEXTURE1);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);

		glClientActiveTexture(GL_TEXTURE0);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);

		/* Disable Texture Matrix */
		glMatrixMode(GL_TEXTURE);
		glLoadIdentity();
	}
}

/*******************************************************************************
 * Function Name  : DrawMeshes
 * Inputs		  : None
 * Description    : Draw Objects
 *******************************************************************************/
void OGLESShadowTechniques::DrawMeshes()
{
	for(int i = 0; i < NUM_MESHES; i++)
	{
		VERTTYPE *pVertices = ( (VERTTYPE*)Meshes[i]->pVertex );
		VERTTYPE *pUV		= ( (VERTTYPE*)Meshes[i]->pUV );

		/* Enable Client States and Setup Data Pointers */
		glEnableClientState(GL_VERTEX_ARRAY);
		glVertexPointer(3,VERTTYPEENUM,0,pVertices);

		glClientActiveTexture(GL_TEXTURE0);
		glActiveTexture(GL_TEXTURE0);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2,VERTTYPEENUM,0,pUV);
		glMatrixMode(GL_TEXTURE);
		glLoadIdentity();

		/* Draw Geometry */
		glDrawElements(GL_TRIANGLES,Meshes[i]->nNumFaces*3,GL_UNSIGNED_SHORT,Meshes[i]->pFaces);

		/* Disable Client States */
		glDisableClientState(GL_VERTEX_ARRAY);

		glClientActiveTexture(GL_TEXTURE0);
		glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}
}

/*******************************************************************************
 * Function Name  : DrawBlob
 * Inputs		  : None
 * Description    : Draw Blob Object
 *******************************************************************************/
void OGLESShadowTechniques::DrawBlob()
{
	/* Blob Data based on bounding box data and orientation based on position of the light */
	float lightdirX = -0.9f*(float)-cos(atan2(fCurrentLightPos[2],fCurrentLightPos[0]));
	float lightdirZ = -0.9f*(float)-sin(atan2(fCurrentLightPos[2],fCurrentLightPos[0]));

	float perplightdirX = (float)cos(atan2(fCurrentLightPos[2],fCurrentLightPos[0])+PI/2);
	float perplightdirZ = (float)sin(atan2(fCurrentLightPos[2],fCurrentLightPos[0])+PI/2);

	VERTTYPE	Vertices[] = {
			f2vt(Radius*(lightdirX-perplightdirX))	, f2vt(YMin)	, f2vt(Radius*(lightdirZ-perplightdirZ)),
			f2vt(Radius*(lightdirX+perplightdirX))	, f2vt(YMin)	, f2vt(Radius*(lightdirZ+perplightdirZ)),
			f2vt(Radius*(lightdirX-perplightdirX))	, f2vt(YMax)	, f2vt(Radius*(lightdirZ-perplightdirZ)),
	 		f2vt(Radius*(lightdirX+perplightdirX))	, f2vt(YMax)	, f2vt(Radius*(lightdirZ+perplightdirZ))
		};

	static VERTTYPE	UVs[] = {
			f2vt(0.0f), f2vt(0.0f),
			f2vt(1.0f), f2vt(0.0f),
			f2vt(0.0f), f2vt(1.0f),
	 		f2vt(1.0f), f2vt(1.0f)
		};

	VERTTYPE *pVertices = ( (VERTTYPE*)&Vertices );
	VERTTYPE *pUV       = ( (VERTTYPE*)&UVs );

	/* Enable Client States and Setup Data Pointers */
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,pVertices);

	glClientActiveTexture(GL_TEXTURE0);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2,VERTTYPEENUM,0,pUV);

	/* Draw Geometry */
	glDrawArrays(GL_TRIANGLE_STRIP,0,4);

	/* Disable Client States */
	glDisableClientState(GL_VERTEX_ARRAY);

	glClientActiveTexture(GL_TEXTURE0);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

}

/*******************************************************************************
 * Function Name  : GetBoundingEllips
 * Inputs		  : Vertex List and Number of Vertices in the List
 * Description    : Calculate a bounding ellips based on the vertices in the list
 *******************************************************************************/
void OGLESShadowTechniques::GetBoundingEllips(VERTTYPE *Vertices, int nNumOfVertices)
{
	/* Allocate and Init some vars */
	int i;
	float CenterX = 0, CenterY = 0, CenterZ = 0;

	float XMinTmp=9999999.99f;
	float XMaxTmp=-9999999.99f;

	float YMinTmp=9999999.99f;
	float YMaxTmp=-9999999.99f;

	float ZMinTmp=9999999.99f;
	float ZMaxTmp=-9999999.99f;

	float OverallMinTmp=9999999.99f;
	float OverallMaxTmp=-9999999.99f;

	float tmpMax;

	float TempVtxZero, TempVtxOne, TempVtxTwo;

	/* Run through Vertices finding the required data */
	for (i=0; i<nNumOfVertices; i++)
	{
		TempVtxZero = vt2f(Vertices[i*3+0]);
		TempVtxOne  = vt2f(Vertices[i*3+1]);
		TempVtxTwo  = vt2f(Vertices[i*3+2]);

		CenterX+=TempVtxZero;
		CenterY+=TempVtxOne;
		CenterZ+=TempVtxTwo;


		if (TempVtxZero>XMaxTmp)
		{
			XMaxTmp=TempVtxZero;
		}

		if (TempVtxZero<XMinTmp)
		{
			XMinTmp=TempVtxZero;
		}

		if (TempVtxOne>YMaxTmp)
		{
			YMaxTmp=TempVtxOne;
		}

		if (TempVtxOne<YMinTmp)
		{
			YMinTmp=TempVtxOne;
		}

		if (TempVtxTwo>ZMaxTmp)
		{
			ZMaxTmp=TempVtxTwo;
		}

		if (TempVtxTwo<ZMinTmp)
		{
			ZMinTmp=TempVtxTwo;
		}

		tmpMax=max(TempVtxZero,TempVtxOne);
		tmpMax=max(tmpMax,TempVtxTwo);

		if (tmpMax>OverallMaxTmp)
		{
			OverallMaxTmp=tmpMax;
		}

		tmpMax=min(TempVtxZero,TempVtxOne);
		tmpMax=min(tmpMax,TempVtxTwo);

		if (tmpMax<OverallMinTmp)
		{
			OverallMinTmp=tmpMax;
		}
	}

	/* Store the results */
	Radius=(float)max((fabs(XMaxTmp)+fabs(XMinTmp))*0.5f,(fabs(YMaxTmp)+fabs(YMinTmp))*0.5f);
	YMin=ZMinTmp;
	YMax=ZMaxTmp;
}

/*******************************************************************************
 * Function Name  : DrawAdvancedBlobShadow
 * Inputs		  : None
 * Description    : Draw Advanced Bob Shadow - this is a dynamic shadow based on bounding info.
 *******************************************************************************/
void OGLESShadowTechniques::DrawAdvancedBlobShadow()
{
	// Enable Polygon offset to avoid ZFighting between floor and shadow
	glEnable(GL_POLYGON_OFFSET_FILL);

	// Enable Blending for Transparent Blob
	glEnable (GL_BLEND);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	// Bind Blob Texture
	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D,gBlobMap);

	// Set Base Blend color to influence how transparent the shadow is
	myglColor4(0.0f,0.0f,0.0f,0.5f);

	// Draw the bounding box blob which gets flattened into the floor
	DrawBlob();

	// Disable blending
	glDisable (GL_BLEND);

	// Disable Polygon offset
	glDisable(GL_POLYGON_OFFSET_FILL);
}

/*******************************************************************************
 * Function Name  : DrawProjectedShadow
 * Inputs		  : None
 * Description    : Draw Projected Geometry Shadow
 *******************************************************************************/
void OGLESShadowTechniques::DrawProjectedShadow()
{
	// Enable Polygon offset to avoid ZFighting between floor and shadow
	glEnable(GL_POLYGON_OFFSET_FILL);

	// Disable Blending since alpha blend does not work with projection
	glDisable (GL_BLEND);

	// Disable Texture
	glDisable(GL_TEXTURE_2D);

	// Set the Shadow Color and Alpha
	myglColor4(0.0f,0.0f,0.0f,0.0f);

	// Render the objects which will be slammed into the floor plane
	RenderMD2(&MonsterObj,bAnimated ? MyTime : 0.0f, false, false, false, false, MonsterTriangleList, USE_FANS_N_STRIPS );

	// Disable Polygon offset
	glDisable(GL_POLYGON_OFFSET_FILL);
}

/*******************************************************************************
 * Function Name  : DrawBaseBlob
 * Inputs		  : None
 * Description    : Draw Static Blob Shadow underneath object
 *******************************************************************************/
void OGLESShadowTechniques::DrawBaseBlob()
{
	static VERTTYPE	Vertices[] = {
			f2vt(-CHARWIDTH)	, f2vt(-FLOORHEIGHT)	, f2vt(-CHARWIDTH),
			f2vt(CHARWIDTH)		, f2vt(-FLOORHEIGHT)	, f2vt(-CHARWIDTH),
			f2vt(-CHARWIDTH)	, f2vt(-FLOORHEIGHT)	, f2vt(CHARWIDTH),
	 		f2vt(CHARWIDTH)		, f2vt(-FLOORHEIGHT)	, f2vt(CHARWIDTH)
		};

	static VERTTYPE	UVs[] = {
			f2vt(0.0f), f2vt(0.0f),
			f2vt(1.0f), f2vt(0.0f),
			f2vt(0.0f), f2vt(1.0f),
	 		f2vt(1.0f), f2vt(1.0f)
		};

	// Enable Polygon offset to avoid ZFighting between floor and shadow
	glEnable(GL_POLYGON_OFFSET_FILL);

	// Enable Blending for Transparent Blob
	glEnable (GL_BLEND);
	glBlendFunc (GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA);

	// Setup Matrix
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	myglLoadMatrix(MyRotViewMatrix.f);

	// Bind Blob Texture
	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D,gBlobMap);

	// Set Base Blend color to influence how transparent the shadow is
	myglColor4(0.0f,0.0f,0.0f,0.5f);

	// Draw Blob - in this case the object is "static" so blob position is "static" as well
	// In a Game the Blob position would be calculated from the Character Position.

	VERTTYPE *pVertices = ( (VERTTYPE*)&Vertices );
	VERTTYPE *pUV       = ( (VERTTYPE*)&UVs );

	/* Enable Client States and Setup Data Pointers */
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,pVertices);

	glClientActiveTexture(GL_TEXTURE0);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2,VERTTYPEENUM,0,pUV);

	/* Draw Geometry */
	glDrawArrays(GL_TRIANGLE_STRIP,0,4);

	/* Disable Client States */
	glDisableClientState(GL_VERTEX_ARRAY);

	glClientActiveTexture(GL_TEXTURE0);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	// Disable blending
	glDisable (GL_BLEND);

	// Disable Polygon offset to avoid ZFighting between floor and shadow
	glDisable(GL_POLYGON_OFFSET_FILL);
}

/*******************************************************************************
 * Function Name  : SelectEGLConfig
 * Inputs		  : Mode Selection
 * Description    : Finds an EGL config with required options based on Mode Requested - for PBuffer
 *******************************************************************************/
EGLConfig OGLESShadowTechniques::SelectEGLConfig(long type)
{
    EGLint num_config;
    EGLint conflist[32];
    int i;
	EGLConfig  gEglConfig;

    i = 0;

	/* Setup required options : essential is the PBuffer Bit */
    conflist[i++] = EGL_LEVEL;
    conflist[i++] = 0;

    conflist[i++] = EGL_NATIVE_RENDERABLE;
    conflist[i++] = EGL_FALSE;

	conflist[i++] = EGL_SURFACE_TYPE;
	conflist[i++] = EGL_WINDOW_BIT | EGL_PBUFFER_BIT ;

	switch(shellData->gFSAAMode)
	{
		case 1:
			conflist[i++] = EGL_SAMPLE_BUFFERS;
			conflist[i++] = 1;
			conflist[i++] = EGL_SAMPLES;
			conflist[i++] = 2;
		break;

		case 2:
			conflist[i++] = EGL_SAMPLE_BUFFERS;
			conflist[i++] = 1;
			conflist[i++] = EGL_SAMPLES;
			conflist[i++] = 4;
		break;

		default:
			conflist[i++] = EGL_SAMPLE_BUFFERS;
			conflist[i++] = 0;
	}

	conflist[i++] = EGL_NONE;

	/* Find and return the config */
    if (!eglChooseConfig(eglGetCurrentDisplay(), conflist, &gEglConfig, 1, &num_config) || num_config != 1) {
		return 0;
    }

    return gEglConfig;
}

/*******************************************************************************
 * Function Name  : DrawBillBoardQuad
 * Input		  : Size, (x,y,z) and texture pntr, From and To camera vectors
 * Description    : Draw a Billboard in location X,Y,Z with a certain size and texture
 *******************************************************************************/
void OGLESShadowTechniques::DrawBillBoardQuad (PVRTVECTOR3 MyFrom, PVRTVECTOR3 MyTo, float x,float y,float z,float Size, GLuint pTexture)
{
	float bbmatrix[16];

	PVRTVECTOR3 LookAt,Right,Up;

	/* Find Matrix based on Camera Vectors such that the Billboard always faces the camera */
	LookAt.x = MyFrom.x - MyTo.x;
	LookAt.y = MyFrom.y - MyTo.y;
	LookAt.z = MyFrom.z - MyTo.z;

	Up.x = 0.0f;
	Up.y = 1.0f;
	Up.z = 0.0f;

	Right.x = 1.0f;
	Right.y = 0.0f;
	Right.z = 0.0f;

	PVRTMatrixVec3Normalize(&LookAt,&LookAt);
	PVRTMatrixVec3CrossProduct(&Right, &Up, &LookAt);
	PVRTMatrixVec3CrossProduct(&Up, &Right, &LookAt);

	bbmatrix[0]  = Right.x;
	bbmatrix[1]  = Right.y;
	bbmatrix[2]  = Right.z;
	bbmatrix[3]  = 0;
	bbmatrix[4]  = Up.x;
	bbmatrix[5]  = Up.y;
	bbmatrix[6]  = Up.z;
	bbmatrix[7]  = 0;
	bbmatrix[8]  = LookAt.x;
	bbmatrix[9]  = LookAt.y;
	bbmatrix[10] = LookAt.z;
	bbmatrix[11] = 0;
	bbmatrix[12] = x;
	bbmatrix[13] = y;
	bbmatrix[14] = z;
	bbmatrix[15] = 1;

	x=0;
	y=0;
	z=0;

	/* Set the special billboard matrix */
	glPushMatrix();
	myglMultMatrix(bbmatrix);

	/* Set Texture and Texture Options */
	glActiveTexture(GL_TEXTURE0);
	glBindTexture(GL_TEXTURE_2D, pTexture);
	glEnable(GL_TEXTURE_2D);

	/* Set Base Color to White */
	myglColor4(1.0f,1.0f,1.0f,1.0f);

	/* Vertex Data */
	VERTTYPE verts[] =		{	f2vt(x-Size), f2vt(y-Size), f2vt(z),
								f2vt(x-Size), f2vt(y+Size), f2vt(z),
								f2vt(x+Size), f2vt(y-Size), f2vt(z),
								f2vt(x+Size), f2vt(y+Size), f2vt(z)
							};

	VERTTYPE texcoords[] =	{	f2vt(0.0f), f2vt(1.0f),
								f2vt(0.0f), f2vt(0.0f),
								f2vt(1.0f), f2vt(1.0f),
								f2vt(1.0f), f2vt(0.0f)
							};

	/* Set Arrays - Only need Vertex Array and Tex Coord Array*/
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,verts);

	glClientActiveTexture(GL_TEXTURE0);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2,VERTTYPEENUM,0,texcoords);

	/* Draw Strip */
	glDrawArrays(GL_TRIANGLE_STRIP,0,4);

	/* Disable Arrays */
	glClientActiveTexture(GL_TEXTURE0);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);

	glDisableClientState(GL_VERTEX_ARRAY);

	/* Recover previous Matrix - no more billboard stuff */
	glPopMatrix();

}

/*******************************************************************************
 * Function Name  : UpdateLightPosition
 * Input		  : None
 * Description    : Update the Light Position and select the current active shadow technique
 *******************************************************************************/
void OGLESShadowTechniques::UpdateLightPosition()
{
	float	gfAnimationBlend;
	float	gfInvAnimationBlend;

	int Range = abs(gAnimTbl[gnCurrentAnimation][1] - gAnimTbl[gnCurrentAnimation][0]);

	/* Determine Blend Factors for Animation Table */
	gfAnimationBlend = (((float)fAnimationCounter)/Range);
	gfInvAnimationBlend = 1.0f - gfAnimationBlend;

	/* Calculate current Animation Position */
	gnLightRotCntr = ((gfInvAnimationBlend * ((float)gAnimTbl[gnCurrentAnimation][0]) + gfAnimationBlend * ((float)gAnimTbl[gnCurrentAnimation][1])));

	/* Calculate Light Position from current Animation Position */
	if (ENABLE_LIGHT_ROT)
	{
		fCurrentLightPos[0] = LIGHTPOSSCALE * 5.35f * (float)cos(gnLightRotCntr/100.0f);
		fCurrentLightPos[1] = LIGHTPOSSCALE * 5.35f;
		fCurrentLightPos[2] = -LIGHTPOSSCALE * 5.35f * (float)sin(gnLightRotCntr/100.0f);
		fCurrentLightPos[3] = 0.0f;
	}

	myglLightv( GL_LIGHT0, GL_POSITION, fCurrentLightPos );

	/* Animation Update use the time based Factor calculated in the main render loop */
	fAnimationCounter=fAnimationCounter+fFactor;

	if (fAnimationCounter>Range)
	{
		fAnimationCounter=0;
		gnPBufferUpdateCounter=0;
		gnCurrentAnimation++;

		if (gnCurrentAnimation>MAXANIMATIONSTEPS)
		{
			gnCurrentAnimation=0;
		}

		/* Set current Shadow Mode */
		gnCurrentMode = ((ShadowModes)gAnimTbl[gnCurrentAnimation][2]);
	}

	/* Through Build option its possible to force a certain Shadow Type to always be used
	   Handy for debugging and to see where the techniques break down */

	#if defined DRAW_ADV_BLOB_SHADOW
		gnCurrentMode = ADVANCEDBLOBMODE;
	#endif

	#if defined DRAW_PROJ_SHADOW
		gnCurrentMode = PROJGEOMMODE;
	#endif

	#if defined DRAW_BASE_BLOB_SHADOW
		gnCurrentMode = BASEBLOBMODE;
	#endif

	#if defined DRAW_RENDER2TEX_SHADOW
		gnCurrentMode = R2TEXMODE;
	#endif
}

/*******************************************************************************
 * Function Name  : SetupMatrices
 * Input		  : Matrix Type
 * Description    : Setup the Matrix as requested
 *******************************************************************************/
void OGLESShadowTechniques::SetupMatrices(MatrixEnums MatrixSelected)
{
	static MatrixEnums OldEnum = BILLBOARD; /* Initial Safe Setup */

	float height, width, nearPlane, farPlane, fov, aspect;
	float top, bottom, left, right;

	MyTo.x=0.0f;
	MyTo.y=1.0f;
	MyTo.z=0.0f;

	if (OldEnum==PROJGEOM)
	{
		/* Do a POP Operation first to removed the post process */
		glMatrixMode(GL_MODELVIEW);
		glPopMatrix();
	}

	if(MatrixSelected==LIGHT)
	{
		/* Setup Matrices for Render from Light Position */
		/* Projection */

		glMatrixMode(GL_PROJECTION);
		glLoadIdentity();

		/* Calculte Frustum call to match gluPerspective */
		height = PBUFFERSIZE;
		width  = PBUFFERSIZE;
		nearPlane = 1.0f;
		farPlane = 50.0f;
		fov = 45;
		aspect = PBUFFERSIZE/PBUFFERSIZE;

		top = (float) tan(fov*PI/360.0) * nearPlane;
		bottom = -top;
		left = aspect * bottom;
		right = aspect * top;

		myglFrustum(left,right,bottom,top,nearPlane,farPlane);

		/* Model View Matrix */

		glMatrixMode(GL_MODELVIEW);
		glLoadIdentity();

		PVRTMatrixRotationY(&MyRotMatrix,((float)gnObjRotCntr/100.0f));

		MyFrom.x = fCurrentLightPos[0];
		MyFrom.y = fCurrentLightPos[1];
		MyFrom.z = fCurrentLightPos[2];

		PVRTMatrixLookAtRH(&MyViewMatrix,&MyFrom,&MyTo,&MyUp);
		PVRTMatrixMultiply(&MyRotViewMatrix,&MyRotMatrix, &MyViewMatrix);

		myglLoadMatrix(MyRotViewMatrix.f);
	}
	else if (MatrixSelected==PROJTEX)
	{
		/* Setup Texture Matrix for Texture	Projection */

		glMatrixMode(GL_TEXTURE);
		glLoadIdentity();

		myglTranslate(0.5f, 0.6f, 0.0f);
		myglScale(0.15f, 0.15f, 1.0f);

		/*******/

		PVRTMatrixRotationY(&MyRotMatrix,((float)gnObjRotCntr/100.0f));

		MyFrom.x = fCurrentLightPos[0];
		MyFrom.y = fCurrentLightPos[1];
		MyFrom.z = fCurrentLightPos[2];

		PVRTMatrixLookAtRH(&MyViewMatrix,&MyFrom,&MyTo,&MyUp);
		PVRTMatrixMultiply(&MyRotViewMatrix,&MyRotMatrix, &MyViewMatrix);

		myglMultMatrix(MyRotViewMatrix.f);
	}
	else
	{
		/* Setup Default Camera Case */
		if (OldEnum!=CAMERA)
		{
			/* Do Setup */

			/* Projection */
			glMatrixMode(GL_PROJECTION);
			glLoadIdentity();
			if(OGLESShellIsRotated() && OGLESShellIsFullScreen())
				myglRotate(90,0,0,1);

			// Calculte Frustum call to match gluPerspective
			height = 240.0f;
			width  = 320.0f;
			nearPlane = 10.0f;
			farPlane = 500.0f;
			fov = 30;
			aspect = 320.0f/240.0f;

			top = (float) tan(fov*PI/360.0) * nearPlane;
			bottom = -top;
			left = aspect * bottom;
			right = aspect * top;

			myglFrustum(left,right,bottom,top,nearPlane,farPlane);

			/* Model View Matrix */

			glMatrixMode(GL_MODELVIEW);
			glLoadIdentity();

			PVRTMatrixRotationY(&MyRotMatrix,((float)gnObjRotCntr/100.0f));

			MyFrom.x=10.0f;
			MyFrom.y=10.0f;
			MyFrom.z=-10.0f;

			MyTo.x=0.0f;
			MyTo.y=0.0f;
			MyTo.z=0.0f;

			PVRTMatrixLookAtRH(&MyViewMatrix,&MyFrom,&MyTo,&MyUp);
			PVRTMatrixMultiply(&MyRotViewMatrix,&MyRotMatrix, &MyViewMatrix);

			myglLoadMatrix(MyRotViewMatrix.f);
		}

		if (MatrixSelected==PROJGEOM)
		{
			/* Add on Post Process after a Push */
			glMatrixMode(GL_MODELVIEW);
			glPushMatrix();

			shadowMatrix(floorShadow, floorPlane, fCurrentLightPos);
			myglMultMatrix(floorShadow);
		}
	}

	/* Update Old Enum Var */
	OldEnum = MatrixSelected;
}

/*******************************************************************************
 * Function Name  : NewDemo
 * Description    : Called by the Shell to initialize a new instance to the 
 *					demo class.
 *******************************************************************************/
OGLESShell* NewDemo(){
	return new OGLESShadowTechniques();
}

/*****************************************************************************
 End of file (OGLESShadowTechniques.cpp)
*****************************************************************************/