/*!****************************************************************************
 @File          OGLESFiveSpheres.cpp

 @Title         OGLESFiveSpheres

 @Author        PowerVR

 @Date          13/01/2005

 @Copyright     Copyright 2005 by Imagination Technologies Limited.
                All rights reserved. No part of this software, either
                material or conceptual may be copied or distributed,
                transmitted, transcribed, stored in a retrieval system
                or translated into any human or computer language in any
                form by any means, electronic, mechanical, manual or
                other-wise, or disclosed to third parties without the
                express written permission of Imagination Technologies
                Limited, Unit 8, HomePark Industrial Estate,
                King's Langley, Hertfordshire, WD4 8LZ, U.K.

 @Platform      Independant

 @Description   Shows different primitive types applied to a model.
				This is more a test than a demonstration. Programmers new to
				OpenGL ES are invited to start from a simpler and more featured
				demo like e.g. OGLESVase.
******************************************************************************/

/*************************
**		  Includes	    **
*************************/
#include <math.h>
#include "OGLESShell.h"
#include "OGLESTools.h"
#include "GRASSFAR.h"

/* Include model data */
#ifdef OGLESLITE
	/* Fixed-point version */
	#include "OGLESFiveSpheres_i.H"
#else
	/* Float version */
	#include "OGLESFiveSpheres_f.H"
#endif


/*************************
**		  Defines	    **
*************************/
#ifndef PI
#define PI 3.14159f
#endif


/****************************
** Class: OGLESFiveSpheres **
****************************/
class OGLESFiveSpheres : public OGLESShell
{
	/* Print3D class */
	CPVRTPrint3D 	AppPrint3D;

	/* Mesh pointers and other model data */
	HeaderStruct_Mesh_Type* Meshes[NUM_MESHES];
	float meshCentres[NUM_MESHES*4];
	float meshCentresTx[NUM_MESHES*4];
	int drawOrder[NUM_MESHES];

	/* Texture names */
	GLuint texName;

	/* Render width and height */
	int	dwCurrentWidth;
	int	dwCurrentHeight;

	/* Rotation variables */
	float fXAng, fYAng;

public:
	OGLESFiveSpheres()
	{
		fXAng = 36.0f;
		fYAng = -118.0f;
		dwCurrentWidth = 320;
		dwCurrentHeight = 240;
	}

	/* OGLESShell functions */
	virtual bool InitApplication(int argc, char*argv[], unsigned int uWidth, unsigned int uHeight);
	virtual bool InitView(unsigned int uWidth, unsigned int uHeight);
	virtual void ReleaseView();
	virtual bool QuitApplication();
	virtual bool RenderScene();

	/****************************************************************************
	** Function Definitions
	****************************************************************************/
	HeaderStruct_Mesh_Type* MakeUnstrippedMesh(HeaderStruct_Mesh_Type *oldMesh, GLenum mode);
	void DeleteUnstrippedMesh(HeaderStruct_Mesh_Type *oldMesh);
	void RenderPrimitive(VERTTYPE *pVertex, VERTTYPE *pNormals, VERTTYPE *pUVs, int nFirst,int nStripLength, GLenum mode);
	void SortMeshes(PVRTMATRIX &Matrix);
};

/*******************************************************************************
 * Function Name  : InitApplication
 * Inputs		  : argc, *argv[], uWidth, uHeight
 * Returns        : true if no error occured
 * Description    : Code in InitApplication() will be called by the Shell ONCE per 
 *					run, early on in the execution of the program.
 *					Used to initialize variables that are not dependant on the
 *					rendering context (e.g. external modules, loading meshes, etc.)
 *******************************************************************************/
bool OGLESFiveSpheres::InitApplication(int argc, char*argv[], unsigned int uWidth, unsigned int uHeight)
{
	/* Allocate memory for all meshes and copy them */
	for(int i = 0; i < NUM_MESHES; i++)
	{
		Meshes[i] = (HeaderStruct_Mesh_Type *)malloc(sizeof(HeaderStruct_Mesh_Type));
		memcpy(Meshes[i], &Mesh[i], sizeof(HeaderStruct_Mesh));
	}
	
	/* Unstrip all mesh data */
	Meshes[1] = MakeUnstrippedMesh(Meshes[1], GL_TRIANGLES);
	Meshes[2] = MakeUnstrippedMesh(Meshes[2], GL_LINE_STRIP);
	Meshes[3] = MakeUnstrippedMesh(Meshes[3], GL_TRIANGLE_FAN);
	
	/* No problem occured */
	return true;
}

/*******************************************************************************
 * Function Name  : QuitApplication
 * Returns        : true if no error occured
 * Description    : Code in QuitApplication() will be called by the Shell ONCE per 
 *					run, just before exiting the program.
 *******************************************************************************/
bool OGLESFiveSpheres::QuitApplication()
{
	/* Delete strip data */
	DeleteUnstrippedMesh(Meshes[1]);
	DeleteUnstrippedMesh(Meshes[2]);
	DeleteUnstrippedMesh(Meshes[3]);

	/* No problem occured */
	return true;
}

/*******************************************************************************
 * Function Name  : InitView
 * Inputs		  : uWidth, uHeight
 * Returns        : true if no error occured
 * Description    : Code in InitView() will be called by the Shell upon a change 
 *					in the rendering context.
 *					Used to initialize variables that are dependant on the rendering 
 *					context (e.g. textures, vertex buffers, etc.)
 *******************************************************************************/
bool OGLESFiveSpheres::InitView(unsigned int uWidth, unsigned int uHeight)
{
	float			sigx, sigy, sigz;
	int				i;
	int				err;
	PVRTMATRIX		MyPerspMatrix;
	SPVRTContext	Context;

	/* Initialize Print3D textures */
	err = AppPrint3D.SetTextures(&Context,uWidth,uHeight);
	if (err == false)
	{
		OGLESShellOutputDebug ("ERROR: Cannot initialise Print3D\n");
		return false;
	}

	/* Load textures */
	if (OGLESShellLoadTextureFromHeader((PVR_Texture_Header*)GRASSFAR, ShellTextureFilterModeLinear, &texName))
		return false;

	/* Calculate mesh centres */
	for(i = 0; i < NUM_MESHES; i++)
	{
		/* Reset variables */
		sigx = sigy = sigz = 0;

		/* Loop through all vertices and add them together */
		for (unsigned int j = 0; j < Meshes[i]->nNumVertex; j++)
		{
			sigx += Meshes[i]->pVertex[3*j];
			sigy += Meshes[i]->pVertex[3*j + 1];
			sigz += Meshes[i]->pVertex[3*j + 2];
		}

		/* Then divide by number of vertices */
		sigx /= (float) Meshes[i]->nNumVertex;
		sigy /= (float) Meshes[i]->nNumVertex;
		sigz /= (float) Meshes[i]->nNumVertex;

		/* Write results */
		meshCentres[i*4 + 0] = sigx;
		meshCentres[i*4 + 1] = sigy;
		meshCentres[i*4 + 2] = sigz;
		meshCentres[i*4 + 3] = 1;
	}

	/* Create perspective matrix */
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if(OGLESShellIsRotated() && OGLESShellIsFullScreen())
	{
		/* Rotate display by 90 degrees */
		myglRotate(90,0,0,1);

		/* Invert width and height */
		dwCurrentWidth = uHeight;
		dwCurrentHeight = uWidth;
	}
	PVRTMatrixPerspectiveFovRH(&MyPerspMatrix, 20.0f*(PI/180.0f), (float)dwCurrentWidth/(float)dwCurrentHeight, 80.0f, 1200.0f);
	myglMultMatrix(MyPerspMatrix.f);

	/* Set material properties */
	float fObjectMatAmb[] = { 0.1f, 0.1f, 0.1f, 1.0f};
	float fObjectMatDiff[] = { 0.5f, 0.5f, 0.5f, 1.0f};
	float fObjectMatSpec[] = { 1.0f, 1.0f, 1.0f, 1.0f};
	myglMaterialv(GL_FRONT_AND_BACK, GL_DIFFUSE, fObjectMatDiff);
	myglMaterialv(GL_FRONT_AND_BACK, GL_AMBIENT, fObjectMatAmb);
	myglMaterialv(GL_FRONT_AND_BACK, GL_SPECULAR, fObjectMatSpec);
	myglMaterial(GL_FRONT_AND_BACK, GL_SHININESS, 5);

	/* Set lighting properties (light position set in RenderScene()) */
	float fLightAmb[4]  = { 0.0f, 0.0f, 0.0f, 1.0f };
	float fLightDif[4]  = { 1.0f, 1.0f, 1.0f, 1.0f };
	float fLightSpec[4] = { 1.0f, 1.0f, 1.0f, 1.0f };
	float fAmbient[4] =  { 1.0f, 1.0f, 1.0f, 1.0f };
	myglLightv(GL_LIGHT0, GL_AMBIENT, fLightAmb);
	myglLightv(GL_LIGHT0, GL_DIFFUSE, fLightDif);
	myglLightv(GL_LIGHT0, GL_SPECULAR, fLightSpec);
	myglLight(GL_LIGHT0, GL_SPOT_EXPONENT, 5.0f);
	myglLightModelv(GL_LIGHT_MODEL_AMBIENT, fAmbient);

	/* Set point size */
	myglPointSize(2.0f);
	
	/* Set front face direction */
	glFrontFace(GL_CW);

	/* No problem occured */
	return true;
}

/*******************************************************************************
 * Function Name  : ReleaseView
 * Returns        : Nothing
 * Description    : Code in ReleaseView() will be called by the Shell before 
 *					changing to a new rendering context.
 *******************************************************************************/
void OGLESFiveSpheres::ReleaseView()
{
	/* Release textures */
	OGLESShellReleaseHeaderTexture(texName);

	/* Release Print3D textures */
	AppPrint3D.ReleaseTextures(); 
}

/*******************************************************************************
 * Function Name  : RenderScene
 * Returns		  : true if no error occured
 * Description    : Main rendering loop function of the program. The shell will
 *					call this function every frame.
 *******************************************************************************/
bool OGLESFiveSpheres::RenderScene()
{
	PVRTMATRIX TmpX, TmpY, TmpT, TransMatrix;

	/* Set up viewport */
	glViewport(0, 0, OGLESShellGetShellSizeX(), OGLESShellGetShellSizeY());
	
	/* Clear the buffers */
	glEnable(GL_DEPTH_TEST);
	myglClearColor(0.4f, 0.4f, 0.4f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* Ser MODELVIEW transform */
	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();
	myglTranslate(0, 0, -600);
	myglRotate(fXAng, 0, 1, 0);
	myglRotate(fYAng, 1, 0, 0);

	/* Calculate transformation matrix (used to sort objects) */
	PVRTMatrixRotationX(&TmpX, (fXAng*PI)/180.0f);
	PVRTMatrixRotationY(&TmpY, (fYAng*PI)/180.0f);
	PVRTMatrixTranslation(&TmpT, 0.0f, 0.0f, -600.0f);
	PVRTMatrixMultiply(&TransMatrix, &TmpX, &TmpY);
	PVRTMatrixMultiply(&TransMatrix, &TmpT, &TransMatrix);

	/* Enable lighting (light 0) - this needs to be re-enabled every frame because Print3D will disable it */
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	float fLightPos[4]  = {100.0f, 200.0f, 100.0f, 0.0f };
	myglLightv(GL_LIGHT0, GL_POSITION, fLightPos);

	/* Texturing */
	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);
	glBindTexture(GL_TEXTURE_2D, texName);
	myglTexEnv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);
	glActiveTexture(GL_TEXTURE1);
	glDisable(GL_TEXTURE_2D);
	
	/* Cull back faces */
	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);

	/* Sort meshes from back to front */
	SortMeshes(TransMatrix);

	/* Draw the meshes in the order we determined ( back to front ) */
	for (unsigned int ti = 0; ti < NUM_MESHES; ti++)	
	{
		int				vertexNum = 0;
		unsigned int	nMeshToDraw = drawOrder[ti];
		
		/* Loop over defined strips for a given mesh */
		for (unsigned int j = 0; j < Meshes[nMeshToDraw]->nNumStrips; j++)	
		{
			switch (nMeshToDraw)
			{
				case 0:	/* POINTS */

						/* Render primitive */
						RenderPrimitive(Meshes[nMeshToDraw]->pVertex, Meshes[nMeshToDraw]->pNormals, 
										Meshes[nMeshToDraw]->pUV, vertexNum, Meshes[nMeshToDraw]->pStripLength[j]+2,
										GL_POINTS);

						/* Go to next batch */
						vertexNum += Meshes[nMeshToDraw]->pStripLength[j]+2;
						break;

				case 1: /* TRIANGLE LIST */
					
						/* Enable additive blending (ONE-ONE) */
						glEnable(GL_BLEND);
						glBlendFunc(GL_ONE,GL_ONE);

						/* Render primitive */
						RenderPrimitive(Meshes[nMeshToDraw]->pVertex, Meshes[nMeshToDraw]->pNormals,
										Meshes[nMeshToDraw]->pUV, vertexNum, Meshes[nMeshToDraw]->pStripLength[j],
										GL_TRIANGLES);

						/* Go to next batch */
						vertexNum += Meshes[nMeshToDraw]->pStripLength[j];

						/* Disable blending */
						glDisable(GL_BLEND);
						break;

				case 2: /* LINE STRIP */

						/* Enable SRCALPHA - ZERO blending mode */
						glEnable(GL_BLEND);
						glBlendFunc(GL_SRC_ALPHA,GL_ZERO);

						/* Render primitive */
						RenderPrimitive(Meshes[nMeshToDraw]->pVertex, Meshes[nMeshToDraw]->pNormals,
										Meshes[nMeshToDraw]->pUV, vertexNum, Meshes[nMeshToDraw]->pStripLength[j],
										GL_LINE_STRIP);
						
						/* Go to next batch */
						vertexNum += Meshes[nMeshToDraw]->pStripLength[j];

						/* Disable blending */
						glDisable(GL_BLEND);
						break;

				case 3: /* Enable SRCALPHA - ONE blending mode */
						glEnable(GL_BLEND);
						glBlendFunc(GL_SRC_ALPHA,GL_ONE);

						/* Render primitive */
						RenderPrimitive(Meshes[nMeshToDraw]->pVertex, Meshes[nMeshToDraw]->pNormals,
										Meshes[nMeshToDraw]->pUV, vertexNum, Meshes[nMeshToDraw]->pStripLength[j],
										GL_TRIANGLE_FAN);

						/* Go to next batch */
						vertexNum += Meshes[nMeshToDraw]->pStripLength[j];

						/* Disable blending */
						glDisable(GL_BLEND);
                        break;

				case 4:	/* Enable ONE_MINUS_DST_COLOR - ZERO blending mode */
						glEnable(GL_BLEND);
						glBlendFunc(GL_ONE_MINUS_DST_COLOR,GL_ZERO);

						/* Render primitive */
						RenderPrimitive(Meshes[nMeshToDraw]->pVertex, Meshes[nMeshToDraw]->pNormals,
										Meshes[nMeshToDraw]->pUV, vertexNum, Meshes[nMeshToDraw]->pStripLength[j]+2,
										GL_TRIANGLE_STRIP);
						
						/* Go to next batch */
						vertexNum += Meshes[nMeshToDraw]->pStripLength[j]+2;

						/* Disable blending */
						glDisable(GL_BLEND);
						break;
			}
		}
	}

	/* Increase rotation angles */
	fXAng++;
	fYAng++;

	/* Display info text. */
	AppPrint3D.DisplayDefaultTitle("FiveSpheres", "Primitives test.", PVR_LOGO);
	AppPrint3D.Flush();

	/* No problem occured */
	return true;
}


/*******************************************************************************
 * Function Name  : RenderPrimitive
 * Inputs		  : *pVertex, *pNormals, *pUVs, nFirst, nStripLength, mode
 * Description    : Render a primitive with the parameters specified
 *******************************************************************************/
void OGLESFiveSpheres::RenderPrimitive(VERTTYPE *pVertex, VERTTYPE *pNormals, VERTTYPE *pUVs, int nFirst, int nStripLength, GLenum mode)
{
	/* Set vertex data */
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,pVertex);

	/* Enable normal and texture coordinates arrays for polygons only */
	if (mode != GL_POINTS && mode != GL_LINE_STRIP)
	{
		glEnableClientState(GL_NORMAL_ARRAY);
		glNormalPointer(VERTTYPEENUM, 0, pNormals);

		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2, VERTTYPEENUM, 0, pUVs);
	}

    /* Draw primitive */	
	glDrawArrays(mode,nFirst,nStripLength);
	
	/* Disable client arrays */
	glDisableClientState(GL_NORMAL_ARRAY);
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
}


/*******************************************************************************
 * Function Name  : MakeUnstrippedMesh
 * Inputs		  : *oldMesh, mode
 * Description    : Unstrip a mesh
 *******************************************************************************/
HeaderStruct_Mesh_Type* OGLESFiveSpheres::MakeUnstrippedMesh(HeaderStruct_Mesh_Type *oldMesh, GLenum mode)
{
#define COPY3(dst,src)  ((*dst++) = *(src));\
						((*dst++) = *(src + 1));\
						((*dst++) = *(src + 2));
#define COPY2(dst,src)  ((*dst++) = *(src));\
						((*dst++) = *(src + 1));
	
	VERTTYPE		*pVertex = NULL,*pNormals = NULL, *pUVs = NULL;
	VERTTYPE		*pVertexPtr, *pNormalsPtr, *pUVsPtr;
	unsigned short	*pNewStripLengths = 0;
	int				nNumNewStrips = 0, vertexOffset = 0, uvOffset = 0;
	int				i, j, x;

	/* Allocate memory for a new mesh and copy the incoming mesh onto it */
	HeaderStruct_Mesh_Type *newMesh = (HeaderStruct_Mesh_Type*) malloc(sizeof(HeaderStruct_Mesh_Type));
	memcpy(newMesh, oldMesh, sizeof(HeaderStruct_Mesh_Type));
	
	/* Unstrip */

	/* Copy pointers */
	VERTTYPE *pOldVertex	= oldMesh->pVertex;
	VERTTYPE *pOldNormals	= oldMesh->pNormals;
	VERTTYPE *pOldUVs		= oldMesh->pUV;
	
	switch(mode)
	{
		case GL_TRIANGLES:
			pVertexPtr	= pVertex	= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*3*3);	// expansion factor of about 3
			pNormalsPtr	= pNormals	= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*3*3);
			pUVsPtr		= pUVs		= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*2*3);
			pNewStripLengths		= (unsigned short*) 	  malloc(sizeof(unsigned short)*oldMesh->nNumStrips);
			
			for (i = 0; i < (int)oldMesh->nNumStrips; i++)
			{
				for (j = 0; j < (int)oldMesh->pStripLength[i]; j++)
				{
					if (j&1)
					{
						// verts
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 6);

						// normals
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 6);

						// uvs
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 2);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 4);
					}
					else
					{
						// verts
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 6);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);

						// normals
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 6);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);

						// uvs
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 4);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 2);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2);
					}
				}
				pNewStripLengths[nNumNewStrips++] = (oldMesh->pStripLength[i]+2)*3;
				vertexOffset += (oldMesh->pStripLength[i]+2)*3;
				uvOffset += (oldMesh->pStripLength[i]+2)*2;
			}

	break;

	case GL_TRIANGLE_FAN:
			pVertexPtr	= pVertex	= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*3*3);	// expansion factor of about 3
			pNormalsPtr	= pNormals	= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*3*3);
			pUVsPtr		= pUVs		= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*2*3);
			pNewStripLengths		= (unsigned short*) 	  malloc(sizeof(unsigned short)*oldMesh->nNumStrips);

			for(x = 0; x < (int)oldMesh->nNumStrips; x++)
			{
				nNumNewStrips += (oldMesh->pStripLength[x]+2)/5 + (oldMesh->pStripLength[x]+2)%5;
			}
			
			pNewStripLengths		= (unsigned short*) malloc(sizeof(unsigned short)*oldMesh->nNumStrips*nNumNewStrips);	// allocate for worst case expansion
			nNumNewStrips = 0;
			
			for(i = 0; i < (int)oldMesh->nNumStrips; i++)
			{
				x=0;
				for(j = 0; j < (int)oldMesh->pStripLength[i]-2; j+=3,x++)
				{
					if(x&1)
					{
						// verts

						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 6);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 9);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 12);

						// normals
						
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 6);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 9);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 12);

						// uvs

						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 4);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 2);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 6);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 8);
					}
					else
					{
						// verts
						
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 6);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 12);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 9);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);
						

						// normals

						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 6);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 12);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 9);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);
						
						// uvs

						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 4);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 8);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 6);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2 + 2);
						COPY2(pUVsPtr,pOldUVs + vertexOffset + j*2);
					}
					pNewStripLengths[nNumNewStrips++] = 5;
				}
				for(; j < oldMesh->pStripLength[i]; j++,x++)
				{
					if(x&1)
					{
						// verts

						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 6);

						// normals

						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 6);

						// uvs

						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 2);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 4);
					}
					else
					{
						// verts

						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 6);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3 + 3);
						COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);

						// normals

						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 6);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3 + 3);
						COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);

						// uvs

						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 4);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2 + 2);
						COPY2(pUVsPtr,pOldUVs + uvOffset + j*2);
					}
					pNewStripLengths[nNumNewStrips++] = 3;
				}
				
				vertexOffset += (oldMesh->pStripLength[i]+2)*3;
				uvOffset += (oldMesh->pStripLength[i]+2)*2;
			}
	break;
	
	case GL_LINE_STRIP:
		pVertexPtr	= pVertex	= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*3*3);	// expansion factor of about 3
		pNormalsPtr	= pNormals	= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*3*3);
		pUVsPtr		= pUVs		= (VERTTYPE*) malloc(oldMesh->nNumVertex*sizeof(VERTTYPE)*2*3);
		pNewStripLengths		= (unsigned short*) 	  malloc(sizeof(unsigned short)*oldMesh->nNumStrips*3);

		for(i = 0; i < (int)oldMesh->nNumStrips; i++)
		{
			for(x = 0, j = 0; j < (int)oldMesh->pStripLength[i]+2; j++, x++)
			{
				// verts
				COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);
			
				// normals
				COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);
			
				// uvs
				COPY2(pUVsPtr,pOldUVs + uvOffset + j*2);
			}
			pNewStripLengths[nNumNewStrips++] = x;
			
			for(x = 0, j = 1; j <  (int)oldMesh->pStripLength[i]+2; j+=2, x++)
			{
				// verts
				COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);

				// normals
				COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);

				// uvs
				COPY2(pUVsPtr,pOldUVs + uvOffset + j*2);
			}
			pNewStripLengths[nNumNewStrips++] = x;
			
			for(x = 0, j = 0; j < (int)oldMesh->pStripLength[i]+2; j+=2,x++)
			{
				// verts
				COPY3(pVertexPtr,pOldVertex + vertexOffset + j*3);

				// normals
				COPY3(pNormalsPtr,pOldNormals + vertexOffset + j*3);

				// uvs
				COPY2(pUVsPtr,pOldUVs + uvOffset + j*2);
			}
			pNewStripLengths[nNumNewStrips++] = x;
			
			vertexOffset += (oldMesh->pStripLength[i]+2)*3;
			uvOffset += (oldMesh->pStripLength[i]+2)*2;
		}
	break;
	}
	
	newMesh->pVertex = pVertex;
	newMesh->pNormals = pNormals;
	newMesh->pUV = pUVs;
	newMesh->pStripLength = pNewStripLengths;
	newMesh->nNumStrips = nNumNewStrips;
	
	return newMesh;
}

/*******************************************************************************
 * Function Name  : DeleteUnstrippedMesh
 * Inputs		  : *oldMesh, mode
 * Description    : Destroy memory allocated in MakeUnstrippedMesh
 *******************************************************************************/
void OGLESFiveSpheres::DeleteUnstrippedMesh(HeaderStruct_Mesh_Type *oldMesh)
{
	free(oldMesh->pVertex);
	free(oldMesh->pNormals);
	free(oldMesh->pUV);
	free(oldMesh->pStripLength);
	free(oldMesh);
}

/*******************************************************************************
 * Function Name  : SortMeshes
 * Inputs		  : Matrix
 * Description    : Sort all meshes from back to front
 *******************************************************************************/
void OGLESFiveSpheres::SortMeshes(PVRTMATRIX &Matrix)
{
#define LOCALFLTSWAP(x,y) { float tmp; tmp = (x); (x) = (y); (y) = tmp; }
#define LOCALINTSWAP(x,y) { int tmp; tmp = (x); (x) = (y); (y) = tmp; }

	/* Initialize sort with the transformed mesh centres */
	for (int f = 0; f < NUM_MESHES; f++)
	{
		drawOrder[f] = f;
		PVRTTransTransformArray((PVRTVECTOR3 *)&meshCentresTx[f*4+0], (PVRTVECTOR3 *)&meshCentres[f*4+0], 1, &Matrix);
	}

	/* Z-sort the meshes back to front */
	bool bKeepSwapping;
	do
	{
		bKeepSwapping = false;
		for (int i = 0; i < NUM_MESHES-1; i++)
		{
			if (meshCentresTx[i*4+2] < meshCentresTx[i*4+6])
			{
				LOCALFLTSWAP(meshCentresTx[i*4],meshCentresTx[i*4+4]);
				LOCALFLTSWAP(meshCentresTx[i*4+1],meshCentresTx[i*4+5]);
				LOCALFLTSWAP(meshCentresTx[i*4+2],meshCentresTx[i*4+6]);

				LOCALINTSWAP(drawOrder[i],drawOrder[i+1]);

				bKeepSwapping = true;
			}
		}
	}
	while (bKeepSwapping);
}


/*******************************************************************************
 * Function Name  : NewDemo
 * Description    : Called by the Shell to initialize a new instance to the 
 *					demo class.
 *******************************************************************************/
OGLESShell* NewDemo()
{
	return new OGLESFiveSpheres();
}

/*****************************************************************************
 End of file (OGLESFiveSpheres.cpp)
*****************************************************************************/