/*!****************************************************************************
 @File          OGLESMouse.cpp

 @Title         OGLESMouse

 @Author        PowerVR

 @Date          13/01/2005

 @Copyright     Copyright 2005 by Imagination Technologies Limited.
                All rights reserved. No part of this software, either
                material or conceptual may be copied or distributed,
                transmitted, transcribed, stored in a retrieval system
                or translated into any human or computer language in any
                form by any means, electronic, mechanical, manual or
                other-wise, or disclosed to third parties without the
                express written permission of Imagination Technologies
                Limited, Unit 8, HomePark Industrial Estate,
                King's Langley, Hertfordshire, WD4 8LZ, U.K.

 @Platform      Independant

 @Description   Demonstrates cell-shading (cartoon style) using VertexProgram extension.
				Requires the OGLESShell.

******************************************************************************/

// Enable the following line to force software processing of the UVs instead of VGP Hardware
// Can also be used as a build option
// #define FORCE_SOFTWARE_PROC	1

#ifdef FORCE_SOFTWARE_PROC 		
	#define DO_NOT_USE_VERTEXPROG 1
#else
	#define DO_NOT_USE_VERTEXPROG 0
#endif


/*************************
**		  Includes	    **
*************************/
#include <stdlib.h>
#include <math.h>
#include "OGLESShell.h"

#include "OGLESTools.h"

/**** Geometry and Animation Data */
#include "OGLESMouse_trilist.H"

/**** Precompiled Vertex Programs */

#if defined(BUILD_VGPARMVP)
	#include "CSH_VGPARMVP.h"
	#define CSH vgp_CSH_VGPARMVP
#elif defined(BUILD_VGP)
	#include "CSH_VGP.h"
	#define CSH vgp_CSH_VGP
#elif defined(BUILD_VGPLITE)
	#include "CSH_VGPLITE.h"
    #define CSH vgp_CSH_VGPLITE
#else
	// No VGP Hardware on this build target thus define CSH as NULL
	#define CSH NULL
	#define DO_NOT_USE_VERTEXPROG 1
#endif

/*** Textures */
#include "Toon.h"			// Special Texture to generate Toon Shading
#include "MouseToon.h"		// Mouse Base Texture
#include "WallToon.h"		// Wall Texture
#include "FloorToon.h"		// Floor Texture

/****************************************************************************
** Class: OGLESMouse
****************************************************************************/
class OGLESMouse : public OGLESShell
{
	/* Texture IDs */
	GLuint  guiTexturesMap[256];

	/* Animation Related */
	GLfloat gfAnimationFrame, gfAnimationIncr;

	/* VGP Program ID */
	GLuint guiVGPProg;

	/* Camera */
	PVRTVECTOR3	gsCameraPosition;
	PVRTVECTOR3	gsCameraPointing;
	PVRTVECTOR3	gsCameraUpVector;

	/* Data Conversion */
	HeaderStruct_Mesh_Type *gpsDemoMesh[NUM_MESHES];

	/* Print3D and Extension Class Objects */
	CPVRTPrint3D gMyPrint3D;
	CPVRTglesExt gPVRTglesExt;

	bool		NoVertexProgram;
	VERTTYPE	InvCamera[3];
	VERTTYPE	InvLight[3];
	VERTTYPE	TempUVBuffer[2*head_NumVertex];	// Used for software processing - Head is most complex object

public:
	OGLESMouse()
	{
		/* Initialisation to start values*/
		gfAnimationFrame = 1.0f;
		gfAnimationIncr  = 1.0f;

		gsCameraUpVector.x=0.0f;	gsCameraUpVector.y=1.0f;	gsCameraUpVector.z=0.0f;
		gsCameraPosition.x=-1000.0f;	gsCameraPosition.y=800.0f;	gsCameraPosition.z=300.0f;

		NoVertexProgram = DO_NOT_USE_VERTEXPROG;
	}

	/* OGLESShell functions */
	virtual bool  InitApplication(int argc, char*argv[], unsigned int uWidth, unsigned int uHeight);
	virtual bool  InitView(unsigned int uWidth, unsigned int uHeight);
	virtual void ReleaseView();
	virtual bool  QuitApplication();
	virtual bool  RenderScene();

	/****************************************************************************
	** Function Definitions
	****************************************************************************/
	void RenderPrimitive(HeaderStruct_Mesh_Type *object);
	void SetAnimMatrix	(int nObject, unsigned FrameCount);
	int  InitVertexProgram(void);
};

/*******************************************************************************
 * Function Name  : InitApplication
 * Inputs		  : argc, *argv[], uWidth, uHeight
 * Returns        : true if no error occured
 * Description    : Code in InitApplication() will be called by the Shell ONCE per 
 *					run, early on in the execution of the program.
 *					Used to initialize variables that are not dependant on the
 *					rendering context (e.g. external modules, loading meshes, etc.)
 *******************************************************************************/
bool OGLESMouse::InitApplication(int argc, char*argv[], unsigned int uWidth, unsigned int uHeight)
{
	/* Load Model and convert to fixed point if necessary
	 * This load only happens once and it must not be repeated hence in InitApplication
	 */
	
	for(int i=0; i<NUM_MESHES; i++)
	{
		gpsDemoMesh[i] = PVRTLoadHeaderObject(&Mesh[i]);
	}
	
	return true;
}

/*******************************************************************************
 * Function Name  : QuitApplication
 * Returns        : true if no error occured
 * Description    : Code in QuitApplication() will be called by the Shell ONCE per 
 *					run, just before exiting the program.
 *******************************************************************************/
bool OGLESMouse::QuitApplication()
{
	/* Release Model Data */
	for(int i=0; i<NUM_MESHES; i++)
	{
		PVRTUnloadHeaderObject(gpsDemoMesh[i]);
	}

	return true;
}

/*******************************************************************************
 * Function Name  : InitView
 * Inputs		  : uWidth, uHeight
 * Returns        : true if no error occured
 * Description    : Code in InitView() will be called by the Shell upon a change 
 *					in the rendering context.
 *					Used to initialize variables that are dependant on the rendering 
 *					context (e.g. textures, vertex buffers, etc.)
 *******************************************************************************/
bool OGLESMouse::InitView(unsigned int uWidth, unsigned int uHeight)
{
	PVRTMATRIX		PerspectiveMatrix;
	SPVRTContext	sTempContext;
	float			fWidth = (float)uWidth;
	float			fHeight = (float)uHeight;
	int				nErr;
	bool			bErr;
	float			fValue[4];
	

	/* Init Print3D to display text on screen */

	bErr = gMyPrint3D.SetTextures(&sTempContext,uWidth,uHeight);
	if (bErr == false)
	{
		OGLESShellOutputDebug ("ERROR: Cannot initialise Print3D\n");
		return false;
	}

	/* Load Textures */

	/* Special Toon Texture must be Point Sampled (Nearest) */
	nErr = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) Toon, ShellTextureFilterModeNearest, &guiTexturesMap[100]);
	if(nErr)
	{
		OGLESShellOutputDebug ("**ERROR** Cannot load Toon texture.\n");
		return false;
	}

	/* Special Toon Texture Option to Clamp to Edge, no wrap around of tex coords */
	myglTexParameter(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP_TO_EDGE);
	myglTexParameter(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP_TO_EDGE);

	nErr = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) MouseToon, ShellTextureFilterModeLinear, &guiTexturesMap[M_HEAD]);
	if(nErr)
	{
		OGLESShellOutputDebug ("**ERROR** Cannot load MouseToon texture.\n");
		return false;
	}

	nErr = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) FloorToon, ShellTextureFilterModeLinear, &guiTexturesMap[M_FLOOR]);
	if(nErr)
	{
		OGLESShellOutputDebug ("**ERROR** Cannot load FloorToon texture.\n");
		return false;
	}

	nErr = OGLESShellLoadTextureFromHeader((PVR_Texture_Header*) WallToon, ShellTextureFilterModeLinear, &guiTexturesMap[M_WALL]);
	if(nErr)
	{
		OGLESShellOutputDebug ("**ERROR** Cannot load WallToon texture.\n");
		return false;
	}

	guiTexturesMap[M_HAND_LEFT] = guiTexturesMap[M_HAND_RIGHT] = guiTexturesMap[M_BODY] = guiTexturesMap[M_OBJECT02] = guiTexturesMap[M_OBJECT01] = guiTexturesMap[M_HEAD];

	/* Generic Render States */
	glEnable(GL_CULL_FACE);
	glCullFace(GL_BACK);
	glFrontFace(GL_CCW);

	glEnable(GL_DEPTH_TEST);
	glDisable(GL_BLEND);
	glDisable(GL_LIGHTING);

	myglClearColor(1,1,0,1);

	/* Set matrices */
	/* Create perspective matrix */
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	if(OGLESShellIsRotated() && OGLESShellIsFullScreen())
	{
		myglRotate(90,0,0,1);
		fWidth = (float)uHeight;
		fHeight = (float)uWidth;
	}

	PVRTMatrixPerspectiveFovRH(&PerspectiveMatrix, 20.0f*(3.14f/180.0f), fWidth/fHeight, 800.0f, 2800.0f);
	myglMultMatrix(PerspectiveMatrix.f);

	/* Check extensions */
	if(!OGLESShellIsExtensionSupported("GL_IMG_vertex_program"))
	{
		OGLESShellOutputDebug("**Warning** Running this demo without GL_IMG_vertex_program\n");
		NoVertexProgram = true;
	}

	if(!NoVertexProgram)
	{
		/* Extension is available. Init function calls */
		gPVRTglesExt.Init();

		/* Init Vertex Program */
		if(!InitVertexProgram())
		{
			return false;
		}

		/* Setup Camera Position Constant */

		/* Set constants for the Vertex Program */

		fValue[0]=f2vt(0.0f); 
		fValue[1]=f2vt(0.0f); 
		fValue[2]=f2vt(0.0f); 
		fValue[3]=f2vt(1.0f);

		myglProgramLocalParameter4v(gPVRTglesExt, GL_VERTEX_PROGRAM_ARB,0,fValue);  // Camera Pos

		fValue[0]=f2vt(10.0f); 
		fValue[1]=f2vt(0.0f); 
		fValue[2]=f2vt(10.0f);
		fValue[3]=f2vt(1.0f);

		myglProgramLocalParameter4v(gPVRTglesExt, GL_VERTEX_PROGRAM_ARB,1,fValue);  // Light Pos
	}

	/* Base Viewport Setup */
	glViewport(0,0,OGLESShellGetShellSizeX(),OGLESShellGetShellSizeY());

	return true;
}

/*******************************************************************************
 * Function Name  : ReleaseView
 * Returns        : Nothing
 * Description    : Code in ReleaseView() will be called by the Shell before 
 *					changing to a new rendering context.
 *******************************************************************************/
void OGLESMouse::ReleaseView()
{
	/* Release the Print3D textures */
	gMyPrint3D.ReleaseTextures();

	// Release Textures
	for(int i=0; i<4; i++)
	{
		OGLESShellReleaseHeaderTexture(guiTexturesMap[i]);
	}

	if(!NoVertexProgram)
	{
        // Release Vertex Program
		gPVRTglesExt.glDeleteProgramsARB(1, &guiVGPProg);
	}
}


/*******************************************************************************
 * Function Name  : RenderScene
 * Returns		  : true if no error occured
 * Description    : Main rendering loop function of the program. The shell will
 *					call this function every frame.
 *******************************************************************************/
bool OGLESMouse::RenderScene()
{
	/* Clear */
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	/* Enable texturing */
	glActiveTexture(GL_TEXTURE0);
	glEnable(GL_TEXTURE_2D);

	/* Set camera target (following the mouse: Data from the 3DS file) */
	gsCameraPointing.x = *(CameraAnimation[0].Target+(int)gfAnimationFrame*3+0);
	gsCameraPointing.y = 200.0f;
	gsCameraPointing.z = *(CameraAnimation[0].Target+(int)gfAnimationFrame*3+2);

	/* Draw the scene */

	if(!NoVertexProgram)
	{
        /* Enable Vertex Program */
		glEnable(GL_VERTEX_PROGRAM_ARB);
		gPVRTglesExt.glBindProgramARB(GL_VERTEX_PROGRAM_ARB, guiVGPProg);
	}

	/* For All Meshes Do... */
	for(int i=0; i<NUM_MESHES; i++)
	{
		/* Bind correct Texture */
		glBindTexture(GL_TEXTURE_2D,guiTexturesMap[i]);

		/* Second layer multitexture for the cartoon effect.
		 * This layer will draw the black halo around the mesh and generate the banded lighting
		 * using modulate blending.
		 * Dynamic UV mapping is calculated in the Vertex Program 
		 */
		if (guiTexturesMap[i] == guiTexturesMap[M_HEAD]) /* Only the mouse is multitextured */
		{
			/* Enable and Set Correct Texture */
			glActiveTexture(GL_TEXTURE1);
			glEnable(GL_TEXTURE_2D);
			glBindTexture(GL_TEXTURE_2D,guiTexturesMap[100]); /* Texture 100 is the special Toon Shading Texture */
			glActiveTexture(GL_TEXTURE0); // Back to default texture layer

			/* Texture Combine Mode */
			myglTexEnv(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE);			
		}
		
        /* Update Animation Matrix and Vertex Shader Constants and Render Geometry */
		SetAnimMatrix	(i, (int)gfAnimationFrame);
		RenderPrimitive(gpsDemoMesh[i]);

		if (guiTexturesMap[i] == guiTexturesMap[M_HEAD]) /* Only the mouse is multitextured */
		{
			glActiveTexture(GL_TEXTURE1);
			glBindTexture(GL_TEXTURE_2D,NULL);
			glDisable(GL_TEXTURE_2D);
			glActiveTexture(GL_TEXTURE0); // Back to default texture layer
		}
	}

	if(!NoVertexProgram)
	{
        /* Disable Vertex Program since its not used to display text */
		glDisable(GL_VERTEX_PROGRAM_ARB);
	}


	/* Update mouse animation */
	if(gfAnimationFrame>=NUM_FRAMES || gfAnimationFrame<=0)
	{
		gfAnimationIncr=-gfAnimationIncr;
	}
	gfAnimationFrame += gfAnimationIncr;

	/* Draw Text */
	/* Shadow */
	if (NoVertexProgram)
	{
		gMyPrint3D.Print3D(0.0f, 1.0f, 1.205f,  0xFF000000, "Mouse");
		gMyPrint3D.Print3D(0.0f, 8.0f, 0.905f,  0xFF000000, "Toon Shading");

		/* Base*/
		gMyPrint3D.DisplayDefaultTitle("Mouse", "Toon Shading", PVR_LOGO);
	}
	else
	{
		gMyPrint3D.Print3D(0.0f, 1.0f, 1.205f,  0xFF000000, "Mouse");
		gMyPrint3D.Print3D(0.0f, 8.0f, 0.905f,  0xFF000000, "VGP Toon Shading");

		/* Base*/
		gMyPrint3D.DisplayDefaultTitle("Mouse", "VGP Toon Shading", PVR_LOGO);
	}

	gMyPrint3D.Flush();

	return true;
}
/*******************************************************************************
 * Function Name  : InitVertexProgram
 * Returns        : True is everything was ok, false if something went wrong
 * Description    : Load a VGP program and set a few constants
 *******************************************************************************/
int OGLESMouse::InitVertexProgram()
{
	/* Load VGP code */
	gPVRTglesExt.glGenProgramsARB(1, &guiVGPProg);
	gPVRTglesExt.glBindProgramARB(GL_VERTEX_PROGRAM_ARB, guiVGPProg);
	gPVRTglesExt.glProgramStringARB(GL_VERTEX_PROGRAM_ARB, GL_PROGRAM_FORMAT_BINARY_IMG, sizeof(CSH), CSH);
	
	/* Check for problems */
	if (glGetError()!=GL_NO_ERROR)
	{
		OGLESShellOutputDebug("**ERROR** glProgramStringARB failed\n");
		return false;
	}

	return true;
}

/*******************************************************************************
 * Function Name  : RenderPrimitive
 * Input          : Object
 * Returns		  : TRUE if no nError occured
 * Description    : Render the mesh data
 *******************************************************************************/
void OGLESMouse::RenderPrimitive(HeaderStruct_Mesh_Type *object)
{
	VERTTYPE *pVerticies = object->pVertex;
	VERTTYPE *pUV = object->pUV;
	VERTTYPE *pNormals = object->pNormals;
	unsigned short *pIndices = object->pFaces;

	/* Set Data Pointers and enable state*/
	glEnableClientState(GL_VERTEX_ARRAY);
	glVertexPointer(3,VERTTYPEENUM,0,pVerticies);

	glEnableClientState(GL_NORMAL_ARRAY);
	glNormalPointer(VERTTYPEENUM,(sizeof(float))*3,pNormals);

	glClientActiveTexture(GL_TEXTURE0);
	glEnableClientState(GL_TEXTURE_COORD_ARRAY);
	glTexCoordPointer(2,VERTTYPEENUM,0,pUV);

	if(NoVertexProgram)
	{
		// Software processing of the second set of UVs to generate the Toon Tex Coords
		for (unsigned int i=0; i<object->nNumVertex;i++)
		{
            TempUVBuffer[i*2+0]= VERTTYPEMUL(object->pNormals[i*3+0],InvCamera[0]) +
								 VERTTYPEMUL(object->pNormals[i*3+1],InvCamera[1]) +
								 VERTTYPEMUL(object->pNormals[i*3+2],InvCamera[2]);
			TempUVBuffer[i*2+1]= VERTTYPEMUL(object->pNormals[i*3+0],InvLight[0]) +
								 VERTTYPEMUL(object->pNormals[i*3+1],InvLight[1]) +
								 VERTTYPEMUL(object->pNormals[i*3+2],InvLight[2]);
		}

		glClientActiveTexture(GL_TEXTURE1);
		glEnableClientState(GL_TEXTURE_COORD_ARRAY);
		glTexCoordPointer(2,VERTTYPEENUM,0,TempUVBuffer);
	}

	/* Draw */
	glDrawElements(GL_TRIANGLES, (object->nNumFaces)*3, GL_UNSIGNED_SHORT, pIndices);

	/* Disable Active States */
	glDisableClientState(GL_VERTEX_ARRAY);
	glDisableClientState(GL_NORMAL_ARRAY);

	if(NoVertexProgram)
	{
		// Disable second layer texture coords
        glClientActiveTexture(GL_TEXTURE1);
        glDisableClientState(GL_TEXTURE_COORD_ARRAY);
	}

	glClientActiveTexture(GL_TEXTURE0);
	glDisableClientState(GL_TEXTURE_COORD_ARRAY);
}

/*********************************************************************************
 *  Function Name   : SetAnimMatrix
 *  Inputs          : nObject,  FrameCount
 *  Returns         : none
 *  Description     : To set a animation from 3DStudio MAX, feed the transformation matrix
 *					  with the fValues exported by the PVRexp plug-in. And setup VGP Constants.
 **********************************************************************************/
void OGLESMouse::SetAnimMatrix (int nObject, unsigned uFrameCount)
{
#ifdef STRUCT_ANIMATION_DEFINED

	PVRTMATRIX  Matrix, ViewMatrix, InvWorld, TempMatrix;
	PVRTVECTOR3 CameraVector,LightVector;
	PVRTVECTOR4 Result;

	float	fValue[4];

	glMatrixMode(GL_MODELVIEW);
	glLoadIdentity();

	PVRTMatrixLookAtRH(&ViewMatrix,&gsCameraPosition,&gsCameraPointing,&gsCameraUpVector);
	myglMultMatrix(ViewMatrix.f);

	/* If the frame is out of range return the identity matrix */
	if (uFrameCount > Animation[nObject].nNumFrames)
	{
		return;
	}

	/*
	 *  Every chunk has 12 floats so, for example, the data for frame 32 of the object 2 starts at
	 *  (Animation[2].pData + 32*12 + 0) and carrys on for 12 floats.
	 *  Note: M_14 = 0, M_24 = 0, M_34 = 0 and M_44 =1 are fixed fValues.
	 */
	uFrameCount = uFrameCount*12;

    Matrix.f[0] = *(Animation[nObject].pData + uFrameCount + 0);
	Matrix.f[1] = *(Animation[nObject].pData + uFrameCount + 1);
	Matrix.f[2] = *(Animation[nObject].pData + uFrameCount + 2);
	Matrix.f[3] = 0.0f;

	Matrix.f[4] = *(Animation[nObject].pData + uFrameCount + 3);
	Matrix.f[5] = *(Animation[nObject].pData + uFrameCount + 4);
	Matrix.f[6] = *(Animation[nObject].pData + uFrameCount + 5);
	Matrix.f[7] = 0.0f;

	Matrix.f[8] = *(Animation[nObject].pData + uFrameCount + 6);
	Matrix.f[9] = *(Animation[nObject].pData + uFrameCount + 7);
	Matrix.f[10] = *(Animation[nObject].pData + uFrameCount + 8);
	Matrix.f[11] = 0.0f;

	Matrix.f[12] = *(Animation[nObject].pData + uFrameCount + 9);
	Matrix.f[13] = *(Animation[nObject].pData + uFrameCount + 10);
	Matrix.f[14] = *(Animation[nObject].pData + uFrameCount + 11);
	Matrix.f[15] = 1.0f;

	myglMultMatrix(Matrix.f);

#else
	glLoadIdentity();
	PVRTMatrixLookAtRH(&ViewMatrix,&gsCameraPosition,&gsCameraPointing,&gsCameraUpVector);
	myglMultMatrix(ViewMatrix.f);
#endif

	glMatrixMode ( GL_MATRIX0_ARB );
	glLoadIdentity();
	PVRTMatrixLookAtRH(&ViewMatrix,&gsCameraPosition,&gsCameraPointing,&gsCameraUpVector);
	myglMultMatrix(ViewMatrix.f);
	myglMultMatrix(Matrix.f);

	/* Setup VGP Constants */

	CameraVector.x = 0.0f;
	CameraVector.y = 0.0f;
	CameraVector.z = 10000.0f;

	PVRTMatrixMultiply(&TempMatrix,&Matrix,&ViewMatrix);

	PVRTMatrixInverseEx(&InvWorld, &TempMatrix);

	PVRTTransVec3TransformArray(&Result,sizeof(PVRTVECTOR4),&CameraVector,sizeof(PVRTVECTOR3),&InvWorld,1);

	PVRTMatrixVec3Normalize((PVRTVECTOR3*)&Result, (PVRTVECTOR3*)&Result);

	fValue[0]=Result.x;
	fValue[1]=Result.y;
	fValue[2]=Result.z;
	fValue[3]=Result.w;

	if(!NoVertexProgram)
	{
        myglProgramLocalParameter4v(gPVRTglesExt, GL_VERTEX_PROGRAM_ARB,0,fValue);  // Camera Position
	}
	else
	{
		InvCamera[0]=f2vt(fValue[0]);
		InvCamera[1]=f2vt(fValue[1]);
		InvCamera[2]=f2vt(fValue[2]);
	}

	LightVector.x = 1000.0f;
	LightVector.y = 0.0f;
	LightVector.z = 0.0f;

	PVRTTransVec3TransformArray(&Result,sizeof(PVRTVECTOR3),&LightVector,sizeof(PVRTVECTOR3),&InvWorld,1);

	PVRTMatrixVec3Normalize((PVRTVECTOR3*)&Result, (PVRTVECTOR3*)&Result); 

	fValue[0]=Result.x;
	fValue[1]=Result.y;
	fValue[2]=Result.z;
	fValue[3]=Result.w;

	if(!NoVertexProgram)
	{
        myglProgramLocalParameter4v(gPVRTglesExt, GL_VERTEX_PROGRAM_ARB,1,fValue);  // Light Position
	}
	else
	{
		InvLight[0]=f2vt(fValue[0]);
		InvLight[1]=f2vt(fValue[1]);
		InvLight[2]=f2vt(fValue[2]);
	}
}

/*******************************************************************************
 * Function Name  : NewDemo
 * Description    : Called by the Shell to initialize a new instance to the 
 *					demo class.
 *******************************************************************************/
OGLESShell* NewDemo(){
	return new OGLESMouse();
}

