/*
 * arch/arm/kernel/dma-sa1100.c
 *
 * Copyright (C) 2000 Nicolas Pitre
 *
 * DMA subsystem for the SA11x0 chip.
 * (this is work in progress)
 */

#include <linux/module.h>
#include <linux/init.h>
#include <linux/wait.h>
#include <linux/spinlock.h>
#include <linux/malloc.h>
#include <linux/errno.h>

#include <asm/system.h>
#include <asm/irq.h>
#include <asm/hardware.h>
#include <asm/io.h>
#include <asm/dma.h>


// #define DEBUG
#ifdef DEBUG
#define DPRINTK( s, arg... )  printk( "dma<%s>: " s, dma->device_id , ##arg )
#else
#define DPRINTK( x... )
#endif


/*
 * Maximum physical DMA buffer size
 */
#define MAX_DMA_SIZE		0x1fff
#define MAX_DMA_ORDER		12


/*
 * DMA control register structure
 */
typedef struct {
	volatile u_long DDAR;
	volatile u_long SetDCSR;
	volatile u_long ClrDCSR;
	volatile u_long RdDCSR;
	volatile dma_addr_t DBSA;
	volatile u_long DBTA;
	volatile dma_addr_t DBSB;
	volatile u_long DBTB;
} dma_regs_t;


/*
 * DMA buffer structure
 */
typedef struct dma_buf_s {
	int size;		/* buffer size */
	dma_addr_t dma_start;	/* starting DMA address */
	dma_addr_t dma_ptr;	/* current DMA pointer position */
	int ref;		/* number of DMA references */
	void *id;		/* to identify buffer from outside */
	struct dma_buf_s *next;	/* next buffer to process */
} dma_buf_t;


/*
 * DMA channel structure.
 */
typedef struct {
	unsigned int lock;	/* Device is allocated */
	const char *device_id;	/* Device name */
	dma_buf_t *head;	/* where to insert buffers */
	dma_buf_t *tail;	/* where to remove buffers */
	dma_buf_t *curr;	/* buffer currently DMA'ed */
	int ready;		/* 1 if DMA can occur */
	int active;		/* 1 if DMA is actually processing data */
	dma_regs_t *regs;	/* points to appropriate DMA registers */
	dma_callback_t callback;	/* ... to call when buffers are done */
	int spin_size;		/* > 0 when DMA should spin when no more buffers */
	dma_addr_t spin_addr;	/* DMA address to spin onto */
	int spin_ref;		/* number of spinning references */
} dma_t;

static dma_t dma_chan[MAX_DMA_CHANNELS];


/*
 * DMA processing...
 */

static inline int start_dma(dma_t * dma, dma_addr_t dma_ptr, int size)
{
	dma_regs_t *regs = dma->regs;
	int status;
	int use_bufa;

	status = regs->RdDCSR;

	/* If both DMA buffers are started, there's nothing else we can do. */
	if ((status & DCSR_STRTA) && (status & DCSR_STRTB)) {
		DPRINTK("start: st %#x busy\n", status);
		return -EBUSY;
	}

	use_bufa = (((status & DCSR_BIU) && (status & DCSR_STRTB)) ||
		    (!(status & DCSR_BIU) && !(status & DCSR_STRTA)));
	if (use_bufa) {
		regs->ClrDCSR = DCSR_DONEA | DCSR_STRTA;
		regs->DBSA = dma_ptr;
		regs->DBTA = size;
		regs->SetDCSR = DCSR_STRTA | DCSR_IE | DCSR_RUN;
		DPRINTK("start a=%#x s=%d on A\n", dma_ptr, size);
	} else {
		regs->ClrDCSR = DCSR_DONEB | DCSR_STRTB;
		regs->DBSB = dma_ptr;
		regs->DBTB = size;
		regs->SetDCSR = DCSR_STRTB | DCSR_IE | DCSR_RUN;
		DPRINTK("start a=%#x s=%d on B\n", dma_ptr, size);
	}

	return 0;
}


/* This must be called with IRQ disabled */
static void process_dma(dma_t * dma)
{
	dma_buf_t *buf;
	int chunksize;

	for (;;) {
		buf = dma->tail;

		if (!buf) {
			/* no more data available */
			DPRINTK("process: no more buf (dma %s)\n",
				dma->curr ? "active" : "inactive");
			/*
			 * Some devices may require DMA still sending data
			 * at any time for clock reference, etc.
			 * Note: if there is still a data buffer being
			 * processed then the ref count is negative.  This
			 * allows for the DMA termination to be accounted in
			 * the proper order.
			 */
			if (dma->spin_size && dma->spin_ref >= 0) {
				chunksize = dma->spin_size;
				if (chunksize > MAX_DMA_SIZE)
					chunksize = (1 << MAX_DMA_ORDER);
				while (start_dma(dma, dma->spin_addr, chunksize) == 0)
					dma->spin_ref++;
				if (dma->curr != NULL)
					dma->spin_ref = -dma->spin_ref;
			}
			break;
		}

		/* 
		 * Let's try to start DMA on the current buffer.  
		 * If DMA is busy then we break here.
		 */
		chunksize = buf->size;
		if (chunksize > MAX_DMA_SIZE)
			chunksize = (1 << MAX_DMA_ORDER);
		DPRINTK("process: b=%#x s=%d\n", (int) buf->id, buf->size);
		if (start_dma(dma, buf->dma_ptr, chunksize) != 0)
			break;
		dma->active = 1;
		if (!dma->curr)
			dma->curr = buf;
		buf->ref++;
		buf->dma_ptr += chunksize;
		buf->size -= chunksize;
		if (buf->size == 0) {
			/* current buffer is done: move tail to the next one */
			dma->tail = buf->next;
			DPRINTK("process: next b=%#x\n", (int) dma->tail);
		}
	}
}


static void dma_irq(int irq, void *dev_id, struct pt_regs *regs)
{
	dma_t *dma = (dma_t *) dev_id;
	int status = dma->regs->RdDCSR;
	dma_buf_t *buf = dma->curr;

	DPRINTK("IRQ: b=%#x st=%#x\n", (int) buf->id, status);

	dma->regs->ClrDCSR = DCSR_ERROR | DCSR_DONEA | DCSR_DONEB;
	if (!(status & (DCSR_DONEA | DCSR_DONEB)))
		return;

	if (dma->spin_ref > 0) {
		dma->spin_ref--;
	} else if (buf) {
		buf->ref--;
		if (buf->ref == 0 && buf->size == 0) {
			/* 
			 * Current buffer is done.  
			 * Move current reference to the next one and send 
			 * the processed buffer to the callback function, 
			 * then discard it.
			 */
			DPRINTK("IRQ: buf done\n");
			dma->curr = buf->next;
			if (dma->curr == NULL)
				dma->spin_ref = -dma->spin_ref;
			if (dma->head == buf)
				dma->head = NULL;
			buf->size = buf->dma_ptr - buf->dma_start;
			if (dma->callback)
				dma->callback(buf->id, buf->size);
			kfree(buf);
		}
	}

	process_dma(dma);
}


/*
 * DMA interface functions
 */

/* 
 * Get dma list
 * for /proc/dma
 */
int get_dma_list(char *buf)
{
	int i, len = 0;

	for (i = 0; i < MAX_DMA_CHANNELS; i++) {
		if (dma_chan[i].lock)
			len += sprintf(buf + len, "%2d: %s\n",
				       i, dma_chan[i].device_id);
	}
	return len;
}


int sa1100_request_dma(dmach_t * channel, const char *device_id)
{
	dma_t *dma = NULL;
	dma_regs_t *regs;
	int ch, irq, err;

	*channel = -1;		/* to be sure we catch the freeing of a misregistered channel */

	for (ch = 0; ch < MAX_DMA_CHANNELS; ch++) {
		dma = &dma_chan[ch];
		if (xchg(&dma->lock, 1) == 0)
			break;
	}
	if (ch >= MAX_DMA_CHANNELS) {
		printk(KERN_ERR "%s: no free DMA channel available\n",
		       device_id);
		return -EBUSY;
	}

	irq = IRQ_DMA0 + ch;
	err = request_irq(irq, dma_irq, SA_INTERRUPT,
			  device_id, (void *) dma);
	if (err) {
		printk(KERN_ERR
		       "%s: unable to request IRQ %d for DMA channel %d\n",
		       device_id, irq, ch);
		dma->lock = 0;
		return err;
	}

	*channel = ch;
	dma->device_id = device_id;
	dma->callback = NULL;
	dma->spin_size = 0;

	regs = dma->regs;
	regs->ClrDCSR =
	    (DCSR_DONEA | DCSR_DONEB | DCSR_STRTA | DCSR_STRTB | DCSR_IE |
	     DCSR_ERROR | DCSR_RUN);
	regs->DDAR = 0;

	DPRINTK("requested\n");
	return 0;
}


int sa1100_dma_set_callback(dmach_t channel, dma_callback_t cb)
{
	dma_t *dma = &dma_chan[channel];

	dma->callback = cb;
	DPRINTK("cb = %p\n", cb);
	return 0;
}


int sa1100_dma_set_device(dmach_t channel, dma_device_t device)
{
	dma_t *dma = &dma_chan[channel];
	dma_regs_t *regs = dma->regs;

	if (dma->ready)
		return -EINVAL;

	regs->ClrDCSR = DCSR_STRTA | DCSR_STRTB | DCSR_IE | DCSR_RUN;
	regs->DDAR = device;
	DPRINTK("DDAR = %#x\n", device);
	dma->ready = 1;
	return 0;
}


int sa1100_dma_set_spin(dmach_t channel, dma_addr_t addr, int size)
{
	dma_t *dma = &dma_chan[channel];
	int flags;

	DPRINTK("set spin %d at %#x\n", size, addr);
	save_flags_cli(flags);
	dma->spin_addr = addr;
	dma->spin_size = size;
	if (size)
		process_dma(dma);
	restore_flags(flags);
	return 0;
}


int sa1100_dma_queue_buffer(dmach_t channel, void *buf_id,
			    dma_addr_t data, int size)
{
	dma_t *dma;
	dma_buf_t *buf;
	int flags;

	dma = &dma_chan[channel];
	if (!dma->ready)
		return -EINVAL;

	buf = kmalloc(sizeof(*buf), GFP_ATOMIC);
	if (!buf)
		return -ENOMEM;

	buf->next = NULL;
	buf->ref = 0;
	buf->dma_ptr = buf->dma_start = data;
	buf->size = size;
	buf->id = buf_id;
	DPRINTK("queueing b=%#x a=%#x s=%d\n", (int) buf_id, data, size);

	save_flags_cli(flags);
	if (dma->head)
		dma->head->next = buf;
	dma->head = buf;
	if (!dma->tail)
		dma->tail = buf;
	process_dma(dma);
	restore_flags(flags);

	return 0;
}


int sa1100_dma_flush_all(dmach_t channel)
{
	dma_t *dma;
	dma_buf_t *buf, *next_buf;
	int flags;

	dma = &dma_chan[channel];
	save_flags_cli(flags);
	dma->regs->ClrDCSR = DCSR_STRTA | DCSR_STRTB | DCSR_IE | DCSR_RUN;
	buf = dma->curr;
	if (!buf)
		buf = dma->tail;
	while (buf) {
		next_buf = buf->next;
		kfree(buf);
		buf = next_buf;
	}
	dma->head = dma->tail = dma->curr = NULL;
	dma->active = 0;
	dma->spin_ref = 0;
	if (dma->spin_size)
		process_dma(dma);
	restore_flags(flags);
	DPRINTK("flushed\n");
	return 0;
}


void sa1100_free_dma(dmach_t channel)
{
	dma_t *dma;

	if ((unsigned) channel >= MAX_DMA_CHANNELS) {
		return;
	}

	dma = &dma_chan[channel];
	if (!dma->lock) {
		printk(KERN_ERR "Trying to free free DMA%d\n", channel);
		return;
	}

	sa1100_dma_set_spin(channel, 0, 0);
	sa1100_dma_flush_all(channel);
	free_irq(IRQ_DMA0 + channel, dma->device_id);
	dma->lock = 0;
	DPRINTK("freed\n");
}


EXPORT_SYMBOL(sa1100_request_dma);
EXPORT_SYMBOL(sa1100_dma_set_callback);
EXPORT_SYMBOL(sa1100_dma_set_device);
EXPORT_SYMBOL(sa1100_dma_set_spin);
EXPORT_SYMBOL(sa1100_dma_queue_buffer);
EXPORT_SYMBOL(sa1100_dma_flush_all);
EXPORT_SYMBOL(sa1100_free_dma);


int __init init_dma(void)
{
	int channel;
	for (channel = 0; channel < MAX_DMA_CHANNELS; channel++) {
		dma_chan[channel].regs =
		    (dma_regs_t *) io_p2v(_DDAR(channel));
	}
	return 0;
}


int no_dma(void)
{
	return -EINVAL;
}

#define GLOBAL_ALIAS(_a,_b) asm (".set " #_a "," #_b "; .globl " #_a)
GLOBAL_ALIAS(request_dma, no_dma);
GLOBAL_ALIAS(free_dma, no_dma);
