/*
 * Touch screen driver for Tifon
 * uses the sa1100 and ucb1200
 *
 * Copyright 1999 Peter Danielsson
 * (codec routines shamelessly stolen from the audio driver)
 *
 * Update to linux-2.3.99-rmk1-np5
 * 
 * peter@xlinux.com
 * (PhantomCat)
 * chester@linux.org.tw
 *
 */
#include <linux/config.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/mm.h>
#include <linux/malloc.h>
#include <linux/sched.h>
#include <linux/major.h>
#include <linux/ctype.h>
#include <linux/wrapper.h>
#include <linux/errno.h>
#include <linux/poll.h>
#include <linux/errno.h>

#ifdef  CONFIG_PROC_FS
#include <linux/stat.h>
#include <linux/proc_fs.h>
#endif

#ifdef  CONFIG_POWERMGR
#include <linux/powermgr.h>
#endif

#include <asm/segment.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>


#include "mcp_common.h"
#include "mcp-sa1100.h"


/* jiffies between samples */
#define SAMPLE_INTERVAL 10

/* minimum delay to register as double-tap */
#define MINTAP (3*SAMPLE_INTERVAL)


/* codec registers */
#define	CODEC_IO_DATA		(0)
#define	CODEC_IO_DIRECTION	(1)
#define CODEC_RE_INT_ENABLE     (2)
#define CODEC_FE_INT_ENABLE     (3)
#define CODEC_INT_STATCLR       (4)
#define CODEC_TELCOM_CTL_A      (5)
#define CODEC_TELCOM_CTL_B      (6)         
#define	CODEC_AUDIO_CTL_A	(7)
#define	CODEC_AUDIO_CTL_B	(8)
#define CODEC_TOUCH_CTL         (9)
#define	CODEC_ADC_CTL		(10)
#define CODEC_ADC_DATA          (11)
#define CODEC_ID                (12)
#define	CODEC_MODE		(13)

#define TSMX_POW        0x1
#define TSPX_POW        0x2
#define TSMY_POW        0x4
#define TSPY_POW        0x8
#define TSMX_GND        0x10
#define TSPX_GND        0x20
#define TSMY_GND        0x40
#define TSPY_GND        0x80
#define TSC_MODE_INT    0x000
#define TSC_MODE_PRES   0x100
#define TSC_MODE_POS    0x200
#define TSC_BIAS_ENA    0x800
#define TSPX_LOW        0x1000
#define TSMX_LOW        0x2000

/* Register CODEC_ADC_CTL */
#define ADC_SYNC_ENA    0x1
#define ADC_INPUT_TSPX  0x0
#define ADC_INPUT_TSMX  0x4
#define ADC_INPUT_TSPY  0x8
#define ADC_INPUT_TSMY  0xc
#define ADC_INPUT_AD0   0x10
#define ADC_INPUT_AD1   0x14
#define ADC_INPUT_AD2   0x18
#define ADC_INPUT_AD3   0x1c
#define EXT_REF_ENA     0x20
#define ADC_START       0x80
#define ADC_ENA         0x8000

#define PRESSED 0
#define P_DONE 1
#define X_DONE 2
#define P1_DONE 3
#define Y_DONE 4
#define RELEASED 5



/* Register 11 */
#define ADC_DAT_VAL     0x8000
#define GET_DATA(x) (((x)>>5)&0x3ff)

#define ADC_ENA_TYPE (ADC_SYNC_ENA | ADC_ENA)

#define TS_INTERRUPT (TSPX_POW | TSMX_POW | TSPY_GND | \
                      TSMY_GND | TSC_MODE_INT )
#define TS_PRESSURE (TSPX_POW | TSMX_POW | TSPY_GND | TSMY_GND | \
		    TSC_MODE_PRES | TSC_BIAS_ENA )

#define ADC_PRESSURE 0

#define TS_XPOS (TSPX_POW | TSMX_GND | TSC_MODE_POS | TSC_BIAS_ENA )
#define ADC_XPOS (ADC_INPUT_TSMY)
#define TS_YPOS (TSPY_GND | TSMY_POW | TSC_MODE_POS | TSC_BIAS_ENA )
#define ADC_YPOS (ADC_INPUT_TSMX)

//FIXME if you got regisation major number, tell me
//it is only local use

#define TS_IRQ 11
#define TS_MAJOR 60
#define TS_NAME "Touchscreen"

#define BUFSIZE 128 

/* PhantomCat */
#define NODATA (ts.head == ts.tail)
/* End PhantomCat */

struct ts_data {
	int p;
	int x;
	int y;
	int p_raw;
	int x_raw;
	int y_raw;
	unsigned long time;
};

struct data_packet {
	unsigned short p;
	unsigned short x;
	unsigned short y;
};

struct calibration {
	int x_scale;               /* scale value for x-pos*1024 */
	int y_scale;               /* scale value for y-pos*1024 */
	int p_scale;               /* scale value for pressure*1024 */
	int x_offset;              /* offset for x pos */
	int y_offset;
	int p_offset;
	/* PhantomCat */
	int x_factor;              /* offset for x pos */
	int y_factor;
	int p_factor;
	int x_base;              /* offset for x pos */
	int y_base;
	int p_base;
	/* PhantomCat */
};

struct touchinfo {
	mcp_reg *reg;             /* physical address to mcp register */
	int mcp;
	int id;                   /* codec id */
	int count;                /* usage count */
	int state;                /* current state of driver */
	struct timer_list timer;  /* timer used in delays */
	wait_queue_head_t (proc_list);
	struct calibration cal;
	struct ts_data buf[BUFSIZE];
	struct fasync_struct *fasync;
	int head;
	int tail;
};

struct ts_data cur_data;
struct touchinfo ts;
unsigned int irq_count;

void setADC(int);
static void ts_interrupt( int, void *, struct pt_regs *);
static inline void mcp_mcsr_wait_read(mcp_reg *);
static inline void mcp_mcsr_wait_write(mcp_reg *);
static int codec_read(mcp_reg *, int);
static void codec_write(mcp_reg *, int, int);
static int readADC(void);
static int pen_up(void);
void new_data(void);
struct ts_data get_data(void);
void store_p(int);
void store_x(int);
void store_y(int);
void wait_for_action(void);
void start_chain(void);
static void handle_timer(unsigned long);
void start_timer(void);


static unsigned long in_timehandle=0;

/* mcp/codec operations */

static inline void mcp_mcsr_wait_read(mcp_reg *mcp)
{
  /* wait for outstanding read to complete */
  while ((mcp->mcsr & MCSR_M_CRC) == 0)  { /* spin */ }
}

static inline void mcp_mcsr_wait_write(mcp_reg *mcp)
{
  /* wait for outstanding write to complete */
  while ((mcp->mcsr & MCSR_M_CWC) == 0)  { /* spin */ }
}

static int codec_read(mcp_reg *mcp, int addr)
{
  ulong flags;
  int data; 

  /* critical section */
  save_flags_cli(flags);
  {
    mcp_mcsr_wait_write(mcp);
    mcp->mcdr2 = ((addr & 0xf) << MCDR2_V_RN);
    mcp_mcsr_wait_read(mcp);
    data = mcp->mcdr2 & 0xffff;
  }
  restore_flags(flags);

  return(data);
}

static void codec_write(mcp_reg *mcp, int addr, int data)
{
  ulong flags;

  /* critical section */
  save_flags_cli(flags);
  {
    mcp_mcsr_wait_write(mcp);
    mcp->mcdr2 = ((addr & 0xf) << MCDR2_V_RN) | MCDR2_M_nRW | (data & 0xffff);
  }
  restore_flags(flags);
}

static int readADC( void )
{
	int ret;
	/* we have to check if conversion is ready */
	while(!((ret=codec_read(ts.reg, CODEC_ADC_DATA)) & ADC_DAT_VAL));
	return GET_DATA(ret);
}


static int pen_up( void )
{
	int i=0;


	codec_write(ts.reg, CODEC_ADC_CTL, 0 );
	codec_write(ts.reg, CODEC_TOUCH_CTL, (TSPX_POW | TSMX_POW | TSPY_GND | TSMY_GND ));

	i = codec_read(ts.reg, CODEC_TOUCH_CTL ) & (1 << 12);
	i |= codec_read(ts.reg, CODEC_TOUCH_CTL ) & (1 << 13);

	return(i);
}

/* data processing routines */

void new_data( void )
{

#define PPOW 0x100
	if( cur_data.p_raw < PPOW){
		return;
	}

	if( ts.head != ts.tail ){
		int last = ts.head--;
		if( last < 0 )
			last = BUFSIZE-1;
#define PLIMIT 50
#define XLIMIT 50
#define YLIMIT 50
		if( abs(cur_data.p - ts.buf[last].p) < PLIMIT &&
		    abs(cur_data.y - ts.buf[last].y) < YLIMIT &&
		    abs(cur_data.x - ts.buf[last].x) < XLIMIT &&
		    abs(jiffies-ts.buf[last].time) < MINTAP){
			return;
		}
	}
	cur_data.time = jiffies;
	ts.buf[ts.head]=cur_data;
	ts.head++;
	if( ts.head == BUFSIZE )
		ts.head = 0;
	if( ts.head == ts.tail ){
		ts.tail++;
		if( ts.tail == BUFSIZE )
			ts.tail = 0;
	}
	if( ts.fasync )
		kill_fasync( ts.fasync, SIGIO, POLL_IN );

	wake_up_interruptible( &ts.proc_list );

}

struct ts_data get_data(void)
{
	int last; 

	last = ts.tail;

	ts.tail++;

	if(ts.tail == BUFSIZE)
		ts.tail=0;

	return ts.buf[last];
}


void store_p(int p)
{
	cur_data.p_raw = p;
	cur_data.p=p;
/*
	if( p > ts.cal.p_threshold )
		cur_data.p = (p*ts.cal.p_scale)/1024 + ts.cal.p_offset;
*/
}

void store_x(int x)
{
	cur_data.x_raw = x;
	cur_data.x = x;

/* cur_data.x = (x-ts.cal.x_base) * (310-10) / ts.cal.x_scale * ts.cal.x_factor / (240-10);

	if( x > ts.cal.x_threshold )
		cur_data.x = (x*ts.cal.x_scale)/1024 + ts.cal.x_offset;
*/
}

void store_y(int y)
{
	cur_data.y_raw = y;
	cur_data.y = y;

/* cur_data.y = (y-ts.cal.y_base) * (230-10) / ts.cal.y_scale * ts.cal.y_factor / (310-10);
 *
	if( y > ts.cal.y_threshold )
		cur_data.y = (y*ts.cal.y_scale)/1024 + ts.cal.y_offset;
*/
}


/*calibration*/
static void touchcal(struct touchinfo *ts){
#define CAL_N 5

	struct ts_data p[CAL_N*12];
	struct file file;
	int i;
	unsigned j;

	file.f_flags &= ~ O_NONBLOCK;

	for(i=0;i<CAL_N;i++){
		if(ts_read(&file,&p[i],60,NULL)<0){
		}
		ts->tail=ts->head;
		p[i].p=cur_data.y_raw;
		p[i].x=cur_data.y_raw;
		p[i].y=cur_data.y_raw;
		for(j=0;j<30000;j++)
			udelay(1);
	}

	ts->cal.x_base=p[1].x;
	ts->cal.y_base=p[1].y;
	ts->cal.x_scale=p[2].x-p[1].x;
	ts->cal.y_scale=p[3].y-p[1].y;
	ts->cal.x_factor=p[3].y-p[1].y;
	ts->cal.y_factor=p[2].x-p[1].x;

}


/* interrupt handling routines */

void wait_for_action( void )
{
	/* set up for pen_down detection */
	ts.state = 0;
	codec_write(ts.reg, CODEC_TOUCH_CTL, TS_INTERRUPT );
	codec_write(ts.reg, 2, 0x0 );
	codec_write(ts.reg, 3, 0x3000 );
}

void start_chain(void)
{
	/* set up for pressure reading; */
	ts.state = P_DONE;
	codec_write(ts.reg, CODEC_TOUCH_CTL, TS_PRESSURE );
	codec_write(ts.reg, 2, 1 << 11 );
	codec_write(ts.reg, 3, 0 );
	setADC(ADC_PRESSURE);
}


void setADC(int sel)
{
 // unsigned ret;

  codec_write(ts.reg, CODEC_ADC_CTL, ADC_ENA | ADC_SYNC_ENA | sel );
  codec_write(ts.reg, CODEC_ADC_CTL, ADC_ENA | ADC_SYNC_ENA | ADC_START | sel );
  codec_write(ts.reg, CODEC_IO_DATA, codec_read(ts.reg, CODEC_IO_DATA) | 0x200);
  codec_write(ts.reg, CODEC_ADC_CTL, ADC_ENA | ADC_SYNC_ENA | sel );
  codec_write(ts.reg, CODEC_IO_DATA, codec_read(ts.reg, CODEC_IO_DATA) & ~0x200);
}

static void handle_timer( unsigned long arg )
{

  in_timehandle--;

  if( pen_up() ){
    wait_for_action();
    return;
  }
  else{
    start_chain();
  }
}


void start_timer( void )
{
  in_timehandle++;

  init_timer( &ts.timer );
  ts.timer.expires = jiffies + SAMPLE_INTERVAL;
  ts.timer.function = handle_timer;
  add_timer( &ts.timer );
}

static void ts_interrupt( int irq, void *dev_id, struct pt_regs *regs)
{


	codec_write(ts.reg, 4,0);
	codec_write(ts.reg, 4,0xffff);

	GEDR |= GPIO_UCB1300_IRQ;

  	if(in_timehandle > 0)
	  return;

	switch( ts.state ){
	case PRESSED:
		start_chain();
		break;
	case P_DONE:
		/* set up for x reading; */
		store_p( readADC() );
		codec_write(ts.reg, CODEC_TOUCH_CTL, TS_XPOS );
		codec_write(ts.reg, 2, 1 << 11 );
		codec_write(ts.reg, 3, 0 );
		setADC(ADC_XPOS );
		ts.state++;
		break;
	case X_DONE:
		/* set up for y reading */
		store_x( readADC() );
		codec_write(ts.reg, CODEC_TOUCH_CTL, TS_PRESSURE );
		codec_write(ts.reg, 2, 1 << 11 );
		codec_write(ts.reg, 3, 0 );
		setADC(ADC_PRESSURE);
		ts.state++;
		break;

	case P1_DONE:
		codec_write(ts.reg, CODEC_TOUCH_CTL, TS_YPOS );
		codec_write(ts.reg, 2, 1 << 11 );
		codec_write(ts.reg, 3, 0 );
		setADC(ADC_YPOS );
		ts.state++;
		break;

	case Y_DONE:
		store_y( readADC() );
		/* set up for pen_up detection */
		codec_write(ts.reg, CODEC_TOUCH_CTL, TS_INTERRUPT );
		codec_write(ts.reg, 2, 0x3000 );
		codec_write(ts.reg, 3, 0 );
		ts.state++;
		new_data();
		start_timer();
		break;
	case RELEASED:
		wait_for_action();
		break;
	default:
		panic("Unknown Touchscreen state\n");
	}

}


/* interface functions */

static ssize_t ts_read( struct file *file, char *buf,
			size_t count, loff_t *offset)
{
	/* return position values from the buffer */
	int i;
	DECLARE_WAITQUEUE (wait, current);

	if( count % 6 ){
	  printk(KERN_INFO "EIO\n");
		return -EIO;
	}

	if( NODATA ){
		if( file->f_flags & O_NONBLOCK ){
		  printk(KERN_INFO "EAGAIN\n");
			return -EAGAIN;
		}
		add_wait_queue( &ts.proc_list, &wait );

		current->state = TASK_INTERRUPTIBLE;
		while( NODATA && ! signal_pending(current) ){
			schedule();
			current->state = TASK_INTERRUPTIBLE;
		}
		current->state = TASK_RUNNING;
		remove_wait_queue(&ts.proc_list, &wait );
	}
	i=count;
	while( i > 5 && !NODATA ){
		struct data_packet p;
		struct ts_data t = get_data();

		p.x = t.x;
		p.y = t.y;
		p.p = t.p;
		i -= 6;
		copy_to_user( buf, &p, 6 );
		buf += 6;
	}
	if( count-i ){
		//file->f_dentry->d_inode->i_atime = CURRENT_TIME;
		return count - i;
	}
	return 0;	
}

static unsigned int ts_poll( struct file *filp, poll_table *wait)
{
	/* wait for data */
	poll_wait(filp, &ts.proc_list, wait);
	if(!NODATA)
		return POLLIN | POLLRDNORM;
	return 0;
}

static int ts_ioctl( struct inode *inode, struct file *file,
		     unsigned int command, unsigned long argument)
{
	/* serve ioctl requests for calibration */
	return -EINVAL;
}

static int ts_fasync( int fd, struct file *file, int on )
{
	/* handle asynchronous notification */
	int retval;

	retval = fasync_helper( fd, file, on, &ts.fasync );
	if( retval < 0 )
		return retval;
	return 0;
}

static int ts_open( struct inode *inode, struct file *file)
{
	/* initialize mcp and interrupts */
	if( ts.count == 0 ){

	  	init_waitqueue_head (&ts.proc_list);	

		mcp_common_enable(ts.mcp);
		GRER |=   GPIO_UCB1300_IRQ;
		GFER &= ~ GPIO_UCB1300_IRQ;
		GEDR |=   GPIO_UCB1300_IRQ;
		GPDR &= ~ GPIO_UCB1300_IRQ;
		if( request_irq(TS_IRQ, ts_interrupt, SA_SHIRQ,
				"Touchscreen", NULL) ){
			printk(KERN_INFO "error request irq\n");
			return -EBUSY;
		}
		codec_write(ts.reg, CODEC_IO_DIRECTION, 
		    codec_read(ts.reg,CODEC_IO_DIRECTION) | 0x200);
		codec_write(ts.reg, 2, 0 );
		codec_write(ts.reg, 3, 0 );
		ts.head = ts.tail = 0;
		wait_for_action();
		//touchcal(&ts);

	}
	MOD_INC_USE_COUNT;
	ts.count++;

	/* test keyboard */
	return 0;
}

static int ts_close( struct inode *inode, struct file *file)
{
	/* close mcp and free interrupts */
	ts.count--;
	if( ts.count == 0 ){
		mcp_common_disable(ts.mcp);
		free_irq( TS_IRQ, NULL );
	}
	ts_fasync(-1, file, 0);
	MOD_DEC_USE_COUNT;

	return 0;
}


static struct file_operations ts_fops = {
	NULL,            /* seek */
	ts_read,
	NULL,            /* write */
	NULL,            /* readdir */
	ts_poll,
	ts_ioctl,
	NULL,            /* mmap */
	ts_open,
	NULL,            /* flush */
	ts_close,
	NULL,            /* fsync */
	ts_fasync
};

#if 0
int get_tsinfo(char *buffer,off_t offset) 
{
			if(offset >0)
						return 0;

			return srpitf(buffer,
	    		"codec id:					%d\n"
				"usage count: 				%d\n"
				"state of driver			%d\n"
				"head						%d\n"
				"tail						%d\n",
				ts.id,ts.count,ts.state,ts.head,ts.tail);

}
struct proc_dir_entry ts_info_entry = {
		namelen: 		8,
		name:			"ts_info",
		mode:			S_IFREG |S_IRUGO,
		nlink:			1,
		get_info:		get_tsinfo.
};

#endif

static int __init touch_init(void)
{
	int err;
	
	memset(&ts, 0, sizeof(ts));
	memset(&ts.cal, 1, sizeof(struct calibration));
	ts.reg = (mcp_reg *)&Ser4MCCR0;

	if((err = register_chrdev(TS_MAJOR, TS_NAME, &ts_fops))){
		printk("Unable to get major %d for device %s\n",
		       TS_MAJOR, TS_NAME);
		return err;
	}


	if((ts.mcp = mcp_common_register()) == MCP_COMMON_ID_INVALID){
		printk(KERN_INFO "%s could not get access to mcp\n", TS_NAME);
		unregister_chrdev( TS_MAJOR, TS_NAME );
		return -1;
	}

	//update to 2.3.99-rmk1-np5
	BCR_set(BCR_CODEC_RST);
	
	mcp_common_enable(ts.mcp);
	ts.id = codec_read( ts.reg, 12 );
	mcp_common_disable(ts.mcp);
	

	if(ts.id == 0){
		printk(KERN_INFO "%s device id is zero - aborting\n", TS_NAME);
		mcp_common_unregister( ts.mcp );
		unregister_chrdev( TS_MAJOR, TS_NAME );
		return -1;
	}

	printk(KERN_INFO "Touchscreen driver, device type %x, version %d\n",
	       (ts.id>>6) & 0x3ff, ts.id & 0x3f);

	return 0;
}

static void __exit touch_exit( void )
{
    BCR_clear(BCR_CODEC_RST);
	unregister_chrdev(TS_MAJOR, TS_NAME);
	mcp_common_unregister( ts.mcp );
}

module_init(touch_init);
module_exit(touch_exit);

