#include "sa1100_ts.h"
#include <linux/kernel.h>
#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/fs.h>
#include <linux/poll.h>
#include <asm/uaccess.h>
#include <asm-arm/arch-sa1100/hardware.h>
#include <linux/string.h>

#ifdef DEBUG
#define SA_PRINTK(x...) printk(x)
#else 
#define SA_PRINTK(x...)
#endif


#ifdef DEBUG
//------------------------------------------------------------
// output 32 bits in binary

void print_bits(unsigned int *p, unsigned int num) {
  int i;
  for(i=num-1; i>=0; i--) {
    printk("%1X",(*p & (1<<i))?1:0 );
    if ((i%4)==0) printk(" ");
  }  
}

//------------------------------------------------------------
// Displays a series of regiseters

void display_regs(unsigned int *pBase, unsigned int num_regs) {
  int i ;
  unsigned int buf[num_regs];

  for(i=0; i<num_regs; i++) {
     memcpy(&buf, pBase+i, sizeof(unsigned int)*num_regs);
  }

  for(i=0;i<num_regs; i++) {
    printk ("%08X: %08X - ",pBase+i, buf[i]);
    print_bits(&buf[i],32);
    printk("\n");
  }
}

#else
void print_bits(unsigned int *p, unsigned int num) {
}

void display_regs(unsigned int *pBase, unsigned int num_regs) {
}
#endif



SA1100_TS_SETTINGS _g_settings, * g_settings;

/* File operations */
ssize_t sa1100_ts_read(struct file * filp, char * buf, size_t count, 
					   loff_t * l);
int sa1100_ts_ioctl(struct inode * inode, struct file *filp
				   ,unsigned int cmd , unsigned long arg);
unsigned int sa1100_ts_poll(struct file * filp, struct poll_table_struct * wait);
int sa1100_ts_open(struct inode * inode, struct file * filp);
int sa1100_ts_release(struct inode * inode, struct file * filp);

/* Interrupt Handler */
void sa1100_ts_inthandler(int irq, void *dev_id, struct pt_regs *regs); 

/* Convenience functions to read/write UCB1200 registers */
unsigned short sa1100_ts_regread( unsigned char reg);
void sa1100_ts_regwrite(unsigned char reg, unsigned short data);

/* Routines used to measure screen values */
int sa1100_ts_measure_y(void);
int sa1100_ts_measure_x(void);
int sa1100_ts_measure_pressure(void);
int sa1100_ts_measure_ground(void);

int sa1100_ts_readscreen(int * in_x, int * in_y);
int sa1100_ts_normalize(int * in_x, int * in_y);
int sa1100_ts_format(char * buf, int x, int y);
int sa1100_ts_starttimer(void);

unsigned long g_dev_id;

int g_last_x;
int g_last_y;
int g_file_type;
int g_x;
int g_y;
int g_mouse_down;

wait_queue_head_t g_wq;
struct timer_list g_timer;

struct file_operations sa1100_ts_fops = {
		NULL,				/* LSEEK */
		sa1100_ts_read,		/* READ  */
		NULL,				/* WRITE */
		NULL,				/* READDIR */
		sa1100_ts_poll,		/* SELECT */
		sa1100_ts_ioctl,	/* IOCTL */
		NULL,				/* MMAP */
		sa1100_ts_open,		/* OPEN */
		NULL,		
		sa1100_ts_release,	/* CLOSE */
		NULL,
		NULL
};

//------------------------------------------------------------

int sa1100_ts_ioctl(struct inode * inode, struct file *filp,
				   unsigned int cmd , unsigned long arg)
{
	int ret;
	ret = (-EINVAL);

	switch(cmd) {
	case _UGL_SA1100_TS_PRESSURE_THRESHOLD:
		UGL_SA1100_TS_PRESSURE_THRESHOLD = arg;
		ret = 0;
		break;
	case _SA1100_TS_RAW_MAX_X:
		SA1100_TS_RAW_MAX_X = arg;
		ret = 0;
		break;
	case _SA1100_TS_RAW_MAX_Y:
		SA1100_TS_RAW_MAX_Y = arg;
		ret = 0;
		break;
	case _SA1100_TS_RAW_MIN_X:
		SA1100_TS_RAW_MIN_X = arg;
		ret = 0;
		break;
	case _SA1100_TS_RAW_MIN_Y:
		SA1100_TS_RAW_MIN_Y = arg;
		ret = 0;
		break;
	case _SA1100_TS_RES_X:
		SA1100_TS_RES_X = arg;
		ret = 0;
		break;
	case _SA1100_TS_RES_Y:
		SA1100_TS_RES_Y = arg;
		ret = 0;
		break;
	case _SA1100_TS_FUDGE_X:
		SA1100_TS_FUDGE_X = arg;
		ret = 0;
		break;
	case _SA1100_TS_FUDGE_Y:
		SA1100_TS_FUDGE_Y = arg;
		ret = 0;
		break;
	case _SA1100_TS_AVG_SAMPLE:
		SA1100_TS_AVG_SAMPLE = arg;
		ret = 0;
		break;
	case _SA1100_TS_UP_DELAY:
		SA1100_TS_UP_DELAY = arg;
		ret = 0;
		break;
	default:
		break;
	}

	return ret;
}

//------------------------------------------------------------

unsigned int sa1100_ts_poll(struct file * filp, struct poll_table_struct * wait)
{
	poll_wait(filp,&g_wq,wait);
	
	if(g_x != SA1100_TS_NOVALUE || 
		g_y != SA1100_TS_NOVALUE ||
		g_mouse_down != SA1100_TS_NOVALUE)
		return (POLLIN | POLLRDNORM);

	return 0;
}

//------------------------------------------------------------

int sa1100_ts_readscreen(int * in_x, int * in_y)
{
	int ground;
	int pressure;
	int x,y;
	int delta_x, delta_y;
	int i;

	ground = sa1100_ts_measure_ground();
	pressure = sa1100_ts_measure_pressure();

	if((pressure - ground) > UGL_SA1100_TS_PRESSURE_THRESHOLD) {
	
		x = 0;
		y = 0;

		g_mouse_down = SA1100_TS_MOUSEDOWN;

		for (i=0; i<SA1100_TS_AVG_SAMPLE; i++) {

			pressure = sa1100_ts_measure_pressure();
			if((pressure - ground) > UGL_SA1100_TS_PRESSURE_THRESHOLD) {
				x += sa1100_ts_measure_x();
#ifdef CONFIG_SA1100_ASSABET
				/* See Philips' AN809 */
    				sa1100_ts_regwrite(UGL_UCB1200_TS_CR,
					UGL_UCB1200_TS_PRESSURE);
#endif
				y += sa1100_ts_measure_y();
			} else {
				*in_x = SA1100_TS_NOVALUE;
				*in_y = SA1100_TS_NOVALUE;
				g_mouse_down = SA1100_TS_UP_DELAY;
				return 1;
			}
		}

		x /= SA1100_TS_AVG_SAMPLE;
		y /= SA1100_TS_AVG_SAMPLE;

		if(g_file_type & SA1100_TS_FILE_NORMAL)
			sa1100_ts_normalize(&x,&y);
		
		if(!g_last_x && !g_last_y) {
			g_last_x = x;
			g_last_y = y;
			*in_x = x;
			*in_y = y;
			return 1;
		} else {
			delta_x = abs(g_last_x - x);
			delta_y = abs(g_last_y - y);

			if(delta_x > SA1100_TS_FUDGE_X ||
				delta_y > SA1100_TS_FUDGE_Y) {

				g_last_x = x;
				g_last_y = y;
				*in_x = x;
				*in_y = y;
				return 1;
			}

		}
	} else {
		if(g_mouse_down > SA1100_TS_NOMOUSE) {
			*in_x = SA1100_TS_NOVALUE;
			*in_y = SA1100_TS_NOVALUE;
			g_mouse_down--;
			return 1;
		}
	}
	return 0;
}

//------------------------------------------------------------

int sa1100_ts_normalize(int * in_x, int * in_y)
{
#ifdef CONFIG_SA1100_ASSABET
	*in_x = (((SA1100_TS_RAW_MAX_X - *in_x)) * SA1100_TS_RES_X) / (SA1100_TS_RAW_MAX_X - SA1100_TS_RAW_MIN_X);
	*in_y = (((SA1100_TS_RAW_MAX_Y - *in_y)) * SA1100_TS_RES_Y) / (SA1100_TS_RAW_MAX_Y - SA1100_TS_RAW_MIN_Y);
#else
	*in_x = (((*in_x - SA1100_TS_RAW_MIN_X)) * SA1100_TS_RES_X) / (SA1100_TS_RAW_MAX_X - SA1100_TS_RAW_MIN_X);
	*in_y = (((*in_y - SA1100_TS_RAW_MIN_Y)) * SA1100_TS_RES_Y) / (SA1100_TS_RAW_MAX_Y - SA1100_TS_RAW_MIN_Y);
#endif
	return 0;	
}

//------------------------------------------------------------

ssize_t sa1100_ts_read(struct file * filp, char * buf, size_t count, loff_t * l)
{
	int x,y;
	int ret;

	if(g_x != SA1100_TS_NOVALUE || 
		g_y != SA1100_TS_NOVALUE ||
		g_mouse_down != SA1100_TS_NOVALUE) {

		x = g_x;
		y = g_y;
		g_x = SA1100_TS_NOVALUE;
		g_y = SA1100_TS_NOVALUE;
		
		
		ret = sa1100_ts_format(buf,x,y);
		
		if(g_mouse_down == SA1100_TS_NOMOUSE)
			g_mouse_down = SA1100_TS_NOVALUE;
		
		return ret;

	} else {
		return 0;
	}
}

//------------------------------------------------------------

void sa1100_ts_timer(unsigned long data)
{
	int ret;
	int x,y;

	SA_PRINTK("sa1100_ts_timer called\n");

	ret = sa1100_ts_readscreen(&x,&y);

	if(ret) {
		g_x = x;
		g_y = y;
		wake_up_interruptible(&g_wq);
	}
	sa1100_ts_starttimer();

}

//------------------------------------------------------------

int sa1100_ts_starttimer(void)
{
	g_timer.function = sa1100_ts_timer;
	g_timer.data = 0;
	g_timer.expires = jiffies + (HZ / 100);

	add_timer(&g_timer);
	
	return 0;
}

//------------------------------------------------------------

int sa1100_ts_stoptimer(void)
{
	del_timer(&g_timer);
	
	return 0;
}

//------------------------------------------------------------

int sa1100_ts_format(char * buf, int x, int y)
{
	char tbuf[20];
	short _x, _y, _down;

	if(g_file_type & SA1100_TS_FILE_TEXT) {
		sprintf(tbuf,"x:%d y:%d\n",x,y);
		copy_to_user(buf,tbuf,strlen(tbuf));
		return strlen(buf);
	} else if(g_file_type & SA1100_TS_FILE_BIN) {
		_x = (short)x;
		_y = (short)y;
		_down = (short)g_mouse_down;

		copy_to_user(buf,&_x,sizeof(_x));
		copy_to_user(buf+sizeof(_x),&_y,sizeof(_y));
		copy_to_user(buf+sizeof(_x) + sizeof(_y), &_down, sizeof(_down));
		return sizeof(short) * 3;

	} else {
		printk("Unknown output type (%X)\n",g_file_type);
	}

	return 0;
}

//------------------------------------------------------------

int sa1100_ts_open(struct inode * inode, struct file * filp)
{
	int type = inode->i_rdev;
	
	if (type) {
		filp->f_op = &sa1100_ts_fops;
		g_file_type = type;
	} else {
		return -ENODEV;
	}
	
	SA_PRINTK("sa1100_ts_open\n");

	MOD_INC_USE_COUNT;
	return 0;
}

//------------------------------------------------------------

int sa1100_ts_release(struct inode * inode, struct file * filp)
{
	SA_PRINTK("sa1100_ts_release\n");
	MOD_DEC_USE_COUNT;
	return 0;
}

//------------------------------------------------------------

#ifdef DEBUG
void UCB1300_readRegs() {
  unsigned int data = 0;
  int i; 

  for (i=0; i <= 0x000F; i++) {
    data = sa1100_ts_regread(i);
    printk("%04X: ", i); print_bits(&data, 32);
    printk("\n");
  }

}
#endif

//------------------------------------------------------------

int __init sa1100_ts_init(void)
{
	int tmp;
	unsigned int irq;

	SA_PRINTK("sa1100_ts_init\n");

	memset(&_g_settings,0,sizeof(_g_settings));
	g_settings = &_g_settings;

#ifdef CONFIG_SA1100_ASSABET
	g_settings->m_raw_max_x = 944;
	g_settings->m_raw_max_y = 944;
	g_settings->m_raw_min_x = 70;
	g_settings->m_raw_min_y = 70;
	g_settings->m_res_x = 320;
	g_settings->m_res_y = 240;
#else
	g_settings->m_raw_max_x = 885;
	g_settings->m_raw_max_y = 885;
	g_settings->m_raw_min_x = 70;
	g_settings->m_raw_min_y = 70;
	g_settings->m_res_x = 640;
	g_settings->m_res_y = 480;
#endif
	g_settings->m_fudge_x = 1;
	g_settings->m_fudge_y = 1;
	g_settings->m_avg_sample = 10;
	g_settings->m_pressure = 20;
	g_settings->m_up_delay = 10;		/* 1/10th of a second */

	g_x = -1;
	g_y = -1;
	g_last_x = 0;
	g_last_y = 0;
	g_mouse_down = SA1100_TS_MOUSEDOWN;

	init_waitqueue_head(&g_wq);
	init_timer(&g_timer);


	if ( machine_is_assabet() ) {
	  BCR_set(BCR_CODEC_RST);

	  /* Enable the MCP */
	  Ser4MCCR0 = (1 << UGL_SA1100_MCCR0_MCE);
	  Ser4MCCR0 |= (1 << UGL_SA1100_MCCR0_ADM);

	  /* Enable the sa1100 interrupt on a rising edge */
	  /* Clear any old interrupts of sa1100 */

	  GPDR |= (1 << 23);
	  GAFR |= (1 << 23); /* TIC request */
	  GRER |= (1 << 23);
	  GEDR |= (1 << 23);
	} 

	if ( machine_is_graphicsclient()) {
	  /* Enable the MCP */
	  Ser4MCCR0 = (1 << UGL_SA1100_MCCR0_MCE);
	
	  tmp = GRER;
	  tmp |= UGL_SA1100_GRER_TS_PINS;
	  tmp &= ~UGL_SA1100_GRER_TS_NOT_PINS;
	  GRER = tmp;
	  

	  tmp = GEDR;
	  tmp |= UGL_SA1100_GRER_TS_PINS;
	  GEDR = tmp;
	}

#ifdef DEBUG
	UCB1300_readRegs();
#endif
	sa1100_ts_regread(15);

	/* initialize the touchscreen controller */
	/* put it in interrupt mode */
	sa1100_ts_regwrite(UGL_UCB1200_TS_CR,UGL_UCB1200_TS_INT);
	sa1100_ts_regwrite(UGL_UCB1200_IO_FAL_I,UGL_UCB1200_IO_TS_INT);
	
	/* Turn off the ADC */
	sa1100_ts_regwrite(UGL_UCB1200_ADC_CR,0);
	
	/* Clear any pending ints off the ts */
	sa1100_ts_regwrite(UGL_UCB1200_IO_CLR_I,0);
	sa1100_ts_regwrite(UGL_UCB1200_IO_CLR_I,UGL_UCB1200_IO_TS_INT);
	sa1100_ts_regread(UGL_UCB1200_IO_CLR_I);
	sa1100_ts_regwrite(UGL_UCB1200_IO_CLR_I,0);
	sa1100_ts_regwrite(UGL_UCB1200_IO_CLR_I,UGL_UCB1200_IO_TS_INT);
	
	tmp = GEDR;
	tmp |= UGL_SA1100_GRER_TS_PINS;
	GEDR = tmp;
	
	register_chrdev(0,"sa1100-ts",&sa1100_ts_fops);
	irq = -1;
#if defined(CONFIG_SA1100_ASSABET)
	if (machine_is_assabet())
	  irq = CODEC_IRQ_ASSABET;
#endif
#if defined(CONFIG_SA1100_GRAPHICSCLIENT)
	if (machine_is_graphicsclient())
	  irq = CODEC_IRQ_GRAPHICSCLIENT;
#endif
	if(request_irq(irq, sa1100_ts_inthandler, 0, 
		"sa1100-ts",&g_dev_id) != 0) {
		printk("sa1100_ts_init: Could not get CODEC IRQ\n");
	}	

	sa1100_ts_starttimer();
	
	printk("sa1100 TouchScreen Driver Initialized\n");
	return 0;
}

//------------------------------------------------------------

void sa1100_ts_inthandler(int irq, void *dev_id, struct pt_regs *regs)      
{                                                                       
	SA_PRINTK("SA1100_ts Interrupt Fired\n");
}                                                                       

//------------------------------------------------------------

unsigned short sa1100_ts_regread( unsigned char reg)
{
	unsigned short tmp;
	/* Wait till any previous writes complete */
	
	SA_PRINTK("sa1100_ts_regread Waiting to read - 1 \n");
	
	while (1)
	{
		tmp = Ser4MCSR;
		if ((tmp & MCSR_CWC) != 0)
			break;
	}
	
	Ser4MCDR2 = (reg  << FShft(MCDR2_ADD)) | MCDR2_Rd;

	SA_PRINTK("sa1100_ts_regread Waiting to read - 2\n");
	/* Wait till it's ok to read */
	while (1)
	{
		tmp = Ser4MCSR;
		if ((tmp & MCSR_CRC) != 0)
			break;
	}
	return (Ser4MCDR2);
}

//------------------------------------------------------------

void sa1100_ts_regwrite(unsigned char reg, unsigned short data)
{
	unsigned short tmp;
	
	SA_PRINTK("Waiting to write\n");

	while (1)
	{
		tmp = Ser4MCSR;
		if ((tmp & MCSR_CWC) != 0)
			break;
	}
	
	Ser4MCDR2 = ((reg  << FShft(MCDR2_ADD)) | MCDR2_Wr | (data & 0xFFFF));
	
}

//------------------------------------------------------------

int sa1100_ts_measure_y(void)
{
    int y;

	/* Read and ignore the value once, cause the first read is
 	 * often inaccurate 
	 */

    /* TSPY powered, TSMY grounded, measure Y position, bias active */
    sa1100_ts_regwrite(UGL_UCB1200_TS_CR, UGL_UCB1200_TS_Y);
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_Y);  
	
    /* Start ADC on TSPX */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_Y | UGL_UCB1200_ADC_START));
	
    y = (sa1100_ts_regread(UGL_UCB1200_ADC_DR) 
		& ~UGL_UCB1200_ADC_DR_VALID) >> UGL_UCB1200_ADC_DR_SHIFT;
	
	
    /* TSPY powered, TSMY grounded, measure Y position, bias active */
    sa1100_ts_regwrite(UGL_UCB1200_TS_CR, UGL_UCB1200_TS_Y);
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_Y);  
	
    /* Start ADC on TSPX */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_Y | UGL_UCB1200_ADC_START));
	
    y = (sa1100_ts_regread(UGL_UCB1200_ADC_DR) 
		& ~UGL_UCB1200_ADC_DR_VALID) >> UGL_UCB1200_ADC_DR_SHIFT;
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_Y);  
	
    /* Start ADC on TSPX */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
	       (UGL_UCB1200_ADC_Y | UGL_UCB1200_ADC_START));

    /* Take the average of the two values, to compensate for noise */
    y = (y + ((sa1100_ts_regread(UGL_UCB1200_ADC_DR) 
		& ~UGL_UCB1200_ADC_DR_VALID) >> UGL_UCB1200_ADC_DR_SHIFT)) / 2;
	
    return y;
}

//------------------------------------------------------------

int sa1100_ts_measure_x(void)
{
    int x;
	
	/* Read and ignore the value once, cause the first read is
	 * often inaccurate 
	 */
	
    /* TSPX powered, TSMX grounded, measure X position, bias active */
    sa1100_ts_regwrite(UGL_UCB1200_TS_CR, UGL_UCB1200_TS_X);
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_X);  
	
    /* Start ADC on TSPY */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_X | UGL_UCB1200_ADC_START));

    x = (sa1100_ts_regread(UGL_UCB1200_ADC_DR) & ~UGL_UCB1200_ADC_DR_VALID) 
		>> UGL_UCB1200_ADC_DR_SHIFT;
	
	
	
    /* TSPX powered, TSMX grounded, measure X position, bias active */
    sa1100_ts_regwrite(UGL_UCB1200_TS_CR, UGL_UCB1200_TS_X);
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_X);  
	
    /* Start ADC on TSPY */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_X | UGL_UCB1200_ADC_START));

    x = (sa1100_ts_regread(UGL_UCB1200_ADC_DR) & ~UGL_UCB1200_ADC_DR_VALID) 
		>> UGL_UCB1200_ADC_DR_SHIFT;
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_X);  
	
    /* Start ADC on TSPY */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_X | UGL_UCB1200_ADC_START));
	
    /* Take the average of the two values, to compensate for noise */
    x = (x + ((sa1100_ts_regread(UGL_UCB1200_ADC_DR) & 
		~UGL_UCB1200_ADC_DR_VALID) >> UGL_UCB1200_ADC_DR_SHIFT)) / 2;

    return x;
}

//------------------------------------------------------------

int sa1100_ts_measure_pressure(void)
{
    int pressure;
	
    /* TSPX/MX powered, TSPY/MY grounded, measure pressure, bias active */
    sa1100_ts_regwrite(UGL_UCB1200_TS_CR, UGL_UCB1200_TS_PRESSURE);
	
    /* Read and ignore the value once, cause the first read is
	 * often inaccurate 
	 */
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_PRESSURE);  
	
    /* Start ADC on pressure input */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_PRESSURE | UGL_UCB1200_ADC_START));
	
    pressure = (sa1100_ts_regread(UGL_UCB1200_ADC_DR) 
		& ~UGL_UCB1200_ADC_DR_VALID) >> UGL_UCB1200_ADC_DR_SHIFT;
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_PRESSURE);  
	
    /* Start ADC on pressure input */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_PRESSURE | UGL_UCB1200_ADC_START));

    pressure = (sa1100_ts_regread(UGL_UCB1200_ADC_DR) 
		& ~UGL_UCB1200_ADC_DR_VALID) >> UGL_UCB1200_ADC_DR_SHIFT;
	
    /* Clear ADC Start bit */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_PRESSURE);  
	
    /* Start ADC on pressure input */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_PRESSURE | UGL_UCB1200_ADC_START));
	
    /* Take the average of the two values, to compensate for noise */
    pressure = (pressure + ((sa1100_ts_regread(UGL_UCB1200_ADC_DR) 
		& ~UGL_UCB1200_ADC_DR_VALID) 
		>> UGL_UCB1200_ADC_DR_SHIFT)) / 2;
	
    return pressure;
}

//------------------------------------------------------------

int sa1100_ts_measure_ground(void)
{
	/* TSPX powered, TSMX grounded, measure TSMX, bias active */
    sa1100_ts_regwrite(UGL_UCB1200_TS_CR, UGL_UCB1200_TS_GROUND);
	
    /* Enable ADC,  measure TSMX */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, UGL_UCB1200_ADC_GROUND);
	
    /* Allow ADC and bias to become active (20us by default) */
	udelay(UGL_SA1100_TS_DELAY_GROUND);
	
    /* Start ADC on TSMX */
    sa1100_ts_regwrite(UGL_UCB1200_ADC_CR, 
		(UGL_UCB1200_ADC_GROUND | UGL_UCB1200_ADC_START));

    return (sa1100_ts_regread(UGL_UCB1200_ADC_DR) & ~UGL_UCB1200_ADC_DR_VALID) 
		>> UGL_UCB1200_ADC_DR_SHIFT;
}

#ifdef MODULE
int init_module(void)
{
	SA_PRINTK("sa1100_init_module\n");
	return 0;
}

void cleanup_module(void)
{
	SA_PRINTK("sa1100_cleanup_module\n");
}
#endif

