/* 
 * mtdbitsy - an onboard flash mtd device
 * Author: Jamey Hicks <jamey@crl.dec.com>
 * Original Author: Alexander Larsson <alex@cendio.se> 
 *
 * Copyright (c) 2000 Compaq Computer Corp
 *
 */

#include <linux/module.h>
#include <linux/malloc.h>
#include <linux/ioport.h>
#include <linux/init.h>
#include <linux/vmalloc.h>
#include <linux/mtd/mapped.h>
#include <asm/arch/hardware.h>

struct flash_mtd_info {
  struct mtd_info mtd;
  u_long base;
  u_long length;
};


/* Flash mapping start */
#define FLASH_START		0xd0000000
#define FLASH_PARTITIONS	7
static int flash_length[FLASH_PARTITIONS] = 
{ 0x00040000, 0x00080000, 0x00040000, 0x00700000, 0x00800000, 0x00800000, 0x00800000 };

/* Flash commands.. */
#define _CMD( x )  ((x)|((x)<<16))

#define FLASH_CMD_READ		_CMD( 0x00FF )
#define FLASH_CMD_ERASE		_CMD( 0x0020 )
#define FLASH_CMD_CONFIRM	_CMD( 0x00D0 )
#define FLASH_CMD_CLEAR		_CMD( 0x0050 )
#define FLASH_CMD_WRITE		_CMD( 0x0040 )
#define FLASH_CMD_WR_BUFF	_CMD( 0x00e8 )
#define FLASH_CMD_STATUS	_CMD( 0x0070 )
#define FLASH_CMD_ID            _CMD( 0x0090 )

#define FLASH_STATUS_READY	_CMD( 0x0080 )

#define FLASH_WR_BUFF_SIZE	16

#define ERROR_VOLTAGE_RANGE	_CMD( 0x0008 )
#define ERROR_DEVICE_PROTECT	_CMD( 0x0002 )
#define ERROR_PROGRAMMING	_CMD( 0x0010 )

#define USEC		1000000		/* usec per sec */
#define POLL_TIME	1000		/* usec */
#define POLL_LOOP_LIMIT	3*USEC/POLL_TIME  /* 3 sec */

static struct flash_mtd_info *mtd_info[FLASH_PARTITIONS];

int mtdbitsy_point (struct mtd_info *mtd, loff_t from, size_t len, size_t *retlen, u_char **mtdbuf)
{
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;

  if (from + len > mtd->size)
    return -EINVAL;
  
  *mtdbuf = (char*)FLASH_START + priv->base + from;
  *retlen = len;
  return 0;
}

void mtdbitsy_unpoint (struct mtd_info *mtd, u_char *addr)
{
}

int mtdbitsy_read(struct mtd_info *mtd, loff_t from, size_t len,
	     size_t *retlen, u_char *buf)
{
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;

  if (from + len > mtd->size)
    return -EINVAL;

  memcpy (buf, (char*)FLASH_START + priv->base + from, len);

  *retlen=len;
  return 0;
}

/*
 *  Function:   full_status_check
 *  Author:     Stephane Dalton
 *  Parameters: in->    flash status value
 *              out->   0: status ok, statuscode: problem
 */
static inline int full_status_check( unsigned long status_reg )
{
   if( status_reg & ERROR_VOLTAGE_RANGE ){
      printk("Flash driver: programming voltage error!\n");
      return status_reg;
   }
   if( status_reg & ERROR_DEVICE_PROTECT ){
      printk("Flash driver: device is write protected!\n");
      return status_reg;
   }
   if( status_reg & ERROR_PROGRAMMING ){
      printk("Flash driver: programming error!\n");
      return status_reg;
   }
   return 0;
}

/* 
 * Function: flash_status_check
 * 	Poll for status until it's ready.  Return error code if some error 
 * 	detected, 0 otherwise.
 * Note: This really really has to be redesigned to avoid busy waits.
 */

static int flash_status_check( volatile u_long *flash_ptr )
{
  u_long status;
  int timeout = POLL_LOOP_LIMIT;

  for( status = *flash_ptr;
       (status & FLASH_STATUS_READY) != FLASH_STATUS_READY;
       status = *flash_ptr ) 
    {
      if( timeout-- < 0 ){
	printk( "flash_mem: flash seems dead!\n" );
	goto done;
      }
      udelay(POLL_TIME);
    } 
  status = full_status_check(status);
 done:
  *flash_ptr = FLASH_CMD_CLEAR;
  *flash_ptr = FLASH_CMD_READ;
  return status;
}

#define FLASH_PTR(priv, offset) (volatile u_long *)(FLASH_START + (priv)->base + (long)(offset))

int mtdbitsy_write (struct mtd_info *mtd, loff_t to, size_t len, size_t *retlen, const u_char *buf)
{
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;
  loff_t end = to+len;
  volatile u_long *flash_ptr = FLASH_PTR(priv, 0);
  u_long *data_ptr = (u_long *)buf;

  if (to + len > mtd->size) {
     printk(__FILE__ ":" __FUNCTION__ ": invalid write -- to=%08lx len=%08lx mtd->size=%08lx\n", (long)to, (long)len, mtd->size);
    return -EINVAL;
  }
  /* check alignment of target, must be word aligned */
  if (to & (loff_t)3) {
    printk(__FILE__ ":" __FUNCTION__ ": invalid write -- must be word aligned to=%08lx\n", (long)to);
    return -EINVAL;
  }

#ifdef CONFIG_SA1100_BITSY
  set_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif
  
  while (to < end) {
    int pollmax = 1<<20;
#undef USE_WRITE_BUFFER
#ifdef USE_WRITE_BUFFER
    int use_write_buffer = ((to & ((4*FLASH_WR_BUFF_SIZE)-1)) == 0);
#else
    int use_write_buffer = 0;
#endif
    int cnt = (use_write_buffer) ? FLASH_WR_BUFF_SIZE : 1;
    u_long error;
    u_long flags;
    u_long value = *data_ptr;

    flash_ptr = FLASH_PTR(priv, to);

#if 1
    printk("mtdbitsy_write: flash_ptr=%p *flash_ptr=%08lx *data_ptr=%08lx cnt=%d to=%08lx\n", flash_ptr, *flash_ptr, value, cnt, (long)to);
#endif

    /* Make sure we have exclusive access to Flash */
    save_flags(flags);
    cli();

    if (use_write_buffer) {
      *flash_ptr = FLASH_CMD_CLEAR;
      printk("mtdbitsy_write: FLASH_CMD_WR_BUFF -0- status=%08lx\n", *flash_ptr);
      do {
        *flash_ptr = FLASH_CMD_WR_BUFF;
        pollmax--;
      } while( pollmax && (*flash_ptr & FLASH_STATUS_READY) != FLASH_STATUS_READY );
      printk("mtdbitsy_write: FLASH_CMD_WR_BUFF -1- status=%08lx\n", *flash_ptr);
      error = flash_status_check( flash_ptr );
      if( error )  {
        printk("mtdbitsy_write: FLASH_CMD_WR_BUFF error=%08lx\n", error);
        break;
      }

      *flash_ptr = _CMD(cnt-1);
      while( cnt-- ) {
        flash_ptr = FLASH_PTR(priv, to);
        to += sizeof(u_long);
        *flash_ptr = *data_ptr++;
        printk("mtdbitsy_write: FLASH_CMD_WR_BUFF -2- flash_ptr=%p status=%08lx to=%08lx cnt=%d\n", flash_ptr, *flash_ptr, (long)(to-4), cnt);
      }
      *flash_ptr = FLASH_CMD_CONFIRM;
      printk("mtdbitsy_write: FLASH_CMD_WR_BUFF -3- status=%08lx\n", *flash_ptr);
    } else {
      *flash_ptr = FLASH_CMD_WRITE;
      to += sizeof(u_long);
      *flash_ptr = *data_ptr++;
    }
    /* flash_status_check waits for FLASH_STATUS_READY */
    error = flash_status_check( flash_ptr );
    restore_flags(flags);
    printk("                flash_ptr=%p *flash_ptr=%08lx *data_ptr=%08lx\n", flash_ptr, *flash_ptr, value);
    if( error ) {
      printk("mtd_flash_write: error programming: error=%08lx\n", error);
      break;
    }
  }
  flash_ptr = FLASH_PTR(priv, 0);
  *flash_ptr = FLASH_CMD_READ;
  if (retlen != NULL) {
     *retlen = len;
  }
#ifdef CONFIG_SA1100_BITSY
  clr_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif

  return 0;
}

int mtdbitsy_erase (struct mtd_info *mtd, struct erase_info *instr)
{
  int res = 0;
  struct flash_mtd_info *priv = (struct flash_mtd_info *)mtd;
  if (instr->addr + instr->len > mtd->size)
    return -EINVAL;

  /* do the erase */

#ifdef CONFIG_SA1100_BITSY
  set_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif

  {
    u_long sector_mask = ~(mtd->erasesize-1);
    u_long start_sector = instr->addr & sector_mask;
    u_long end_sector = (instr->addr + instr->len + mtd->erasesize-1)&sector_mask;
    u_long sector;
    for (sector = start_sector; sector < end_sector; sector += mtd->erasesize) {
      volatile u_long *flash_ptr = (u_long *)(FLASH_START + priv->base + sector);

      *flash_ptr = FLASH_CMD_READ;
      printk("mtd_erase: erasing sector=%08lx flash_ptr=%p *flash_ptr=%08lx\n", 
	     sector, flash_ptr, *flash_ptr);

      *flash_ptr = FLASH_CMD_CLEAR;
      *flash_ptr = FLASH_CMD_ERASE;
      *flash_ptr = FLASH_CMD_CONFIRM;
      res = flash_status_check(flash_ptr);
      *flash_ptr = FLASH_CMD_READ;
      if (res != 0) {
	printk("status=%08x\n", res);
	return -EIO;
      }
    }
  }

#ifdef CONFIG_SA1100_BITSY
  clr_bitsy_egpio(EGPIO_BITSY_VPP_ON);
#endif

  /* now do the other stuff */
  instr->state = MTD_ERASE_DONE;
  if (instr->callback)
    (*instr->callback)(instr);
  return 0;
}

static void __exit cleanup_mtdbitsy(void)
{
  int i;
  for (i = 0; i < FLASH_PARTITIONS; i++) {
    if (mtd_info[i]) {
      del_mtd_device(&mtd_info[i]->mtd);
      kfree(mtd_info[i]);
    }
  }
}


static int __init init_mtdbitsy(void)
{
  int i;
  u_long base = 0;
  printk("init_mtdbitsy\n");
  for (i = 0; i < FLASH_PARTITIONS; i++) {
    char *name = (char *)kmalloc(32, GFP_KERNEL);
    // Allocate some memory
    mtd_info[i] = (struct flash_mtd_info *)kmalloc(sizeof(struct flash_mtd_info), GFP_KERNEL);
    if (name == 0 || mtd_info[i] == 0)
      return 0;
   
    printk("init_mtdbitsy: partition %d start=%08lx size=%08x\n", i, base, flash_length[i]);
    memset(mtd_info[i], 0, sizeof(struct flash_mtd_info));

    // Setup the MTD structure
    sprintf(name, "mtdbitsy %d", i);
    mtd_info[i]->mtd.name = name;
    mtd_info[i]->mtd.type = MTD_NORFLASH;
    mtd_info[i]->mtd.flags = MTD_CAP_NORFLASH;
    mtd_info[i]->mtd.size = flash_length[i];
    mtd_info[i]->mtd.erasesize = 256*1024;
    mtd_info[i]->base = base;
    mtd_info[i]->length = flash_length[i];
    mtd_info[i]->mtd.module = THIS_MODULE;

    base += flash_length[i];

    mtd_info[i]->mtd.erase = mtdbitsy_erase;
    mtd_info[i]->mtd.point = mtdbitsy_point;
    mtd_info[i]->mtd.unpoint = mtdbitsy_unpoint;
    mtd_info[i]->mtd.read = mtdbitsy_read;
    mtd_info[i]->mtd.write = mtdbitsy_write;

    if (add_mtd_device(&mtd_info[i]->mtd)) {
      /* Clean up */
    }
  }   
  return 0;
}



module_init(init_mtdbitsy);
module_exit(cleanup_mtdbitsy);
