/*======================================================================

    Device driver for the PCMCIA control functionality of StrongARM
    SA-1100 microprocessors.

    The contents of this file are subject to the Mozilla Public
    License Version 1.1 (the "License"); you may not use this file
    except in compliance with the License. You may obtain a copy of
    the License at http://www.mozilla.org/MPL/

    Software distributed under the License is distributed on an "AS
    IS" basis, WITHOUT WARRANTY OF ANY KIND, either express or
    implied. See the License for the specific language governing
    rights and limitations under the License.

    The initial developer of the original code is John G. Dorsey
    <john+@cs.cmu.edu>.  Portions created by John G. Dorsey are
    Copyright (C) 1999 John G. Dorsey.  All Rights Reserved.

    Alternatively, the contents of this file may be used under the
    terms of the GNU Public License version 2 (the "GPL"), in which
    case the provisions of the GPL are applicable instead of the
    above.  If you wish to allow the use of your version of this file
    only under the terms of the GPL and not to allow others to use
    your version of this file under the MPL, indicate your decision
    by deleting the provisions above and replace them with the notice
    and other provisions required by the GPL.  If you do not delete
    the provisions above, a recipient may use your version of this
    file under either the MPL or the GPL.
    
======================================================================*/

#if !defined(_PCMCIA_SA1100_H)
# define _PCMCIA_SA1100_H

#include <pcmcia/cs_types.h>
#include <pcmcia/ss.h>
#include <pcmcia/bulkmem.h>
#include <pcmcia/cistpl.h>
#include "cs_internal.h"

#include <asm/arch/pcmcia.h>


/* MECR: Expansion Memory Configuration Register
 * (SA-1100 Developers Manual, p.10-13; SA-1110 Developers Manual, p.10-24)
 *
 * BS_xx = ( PCMCIA_CYCLE / ( 2 * 3 * CPU_CYCLE ) ) - 1
 *
 * Official PCMCIA cycle times:
 *
 *   - I/O:                  255ns
 *   - attribute:            300ns
 *   - common memory (3.3V): 600, 250, 200, 150, 100ns
 *   - common memory (5V):   250, 200, 150, 100ns
 *
 * MECR layout is:  
 *
 *   FAST1 BSM1<4:0> BSA1<4:0> BSIO1<4:0> FAST0 BSM0<4:0> BSA0<4:0> BSIO0<4:0>
 *
 * (This layout is actually true only for the SA-1110; the FASTn bits are
 * reserved on the SA-1100.)
 */

#define MECR_SOCKET_0_SHIFT (0)
#define MECR_SOCKET_1_SHIFT (16)

#define MECR_BS_MASK        (0x1f)
#define MECR_FAST_MODE_MASK (0x01)

#define MECR_BSIO_SHIFT (0)
#define MECR_BSA_SHIFT  (5)
#define MECR_BSM_SHIFT  (10)
#define MECR_FAST_SHIFT (15)

#define MECR_SET(mecr, sock, shift, mask, bs) \
((mecr)=((mecr)&~(((mask)<<(shift))<<\
                  ((sock)==0?MECR_SOCKET_0_SHIFT:MECR_SOCKET_1_SHIFT)))|\
        (((bs)<<(shift))<<((sock)==0?MECR_SOCKET_0_SHIFT:MECR_SOCKET_1_SHIFT)))

#define MECR_GET(mecr, sock, shift, mask) \
((((mecr)>>(((sock)==0)?MECR_SOCKET_0_SHIFT:MECR_SOCKET_1_SHIFT))>>\
 (shift))&(mask))

#define MECR_BSIO_SET(mecr, sock, bs) \
MECR_SET((mecr), (sock), MECR_BSIO_SHIFT, MECR_BS_MASK, (bs))

#define MECR_BSIO_GET(mecr, sock) \
MECR_GET((mecr), (sock), MECR_BSIO_SHIFT, MECR_BS_MASK)

#define MECR_BSA_SET(mecr, sock, bs) \
MECR_SET((mecr), (sock), MECR_BSA_SHIFT, MECR_BS_MASK, (bs))

#define MECR_BSA_GET(mecr, sock) \
MECR_GET((mecr), (sock), MECR_BSA_SHIFT, MECR_BS_MASK)

#define MECR_BSM_SET(mecr, sock, bs) \
MECR_SET((mecr), (sock), MECR_BSM_SHIFT, MECR_BS_MASK, (bs))

#define MECR_BSM_GET(mecr, sock) \
MECR_GET((mecr), (sock), MECR_BSM_SHIFT, MECR_BS_MASK)

#define MECR_FAST_SET(mecr, sock, fast) \
MECR_SET((mecr), (sock), MECR_FAST_SHIFT, MECR_FAST_MODE_MASK, (fast))

#define MECR_FAST_GET(mecr, sock) \
MECR_GET((mecr), (sock), MECR_FAST_SHIFT, MECR_FAST_MODE_MASK)


/* PPCR: Power Manager Phase-Locked Loop Configuration Register
 * (SA-1100 Developers Manual, p.8-2)
 *
 * The core clock frequency of the processor is determined by the
 * frequency of the crystal oscillator (either 3.6864MHz or 3.5795MHz).
 *
 * We have to know the processor clock rate in order to set PCMCIA 
 * memory access times. Note that this code currently does not support
 * altering of the clock rate after initial configuration.
 *
 * sa1100_pcmcia_ccf_6864() assumes an oscillator frequency of 3.6864MHz
 * (as used in the Itsy), while sa1100_pcmcia_ccf_5795() assumes a
 * frequency of 3.5795MHz. Both return 0 if the PPCR contains an invalid
 * clock code.
 */
static inline unsigned int sa1100_pcmcia_ccf_6864(void){
  switch(FExtr(PPCR, PPCR_CCF)){
  case PPCR_F59_0MHz:  return 59;
  case PPCR_F73_7MHz:  return 73;
  case PPCR_F88_5MHz:  return 88;
  case PPCR_F103_2MHz: return 103;
  case PPCR_F118_0MHz: return 118;
  case PPCR_F132_7MHz: return 132;
  case PPCR_F147_5MHz: return 147;
  case PPCR_F162_2MHz: return 162;
  case PPCR_F176_9MHz: return 176;
  case PPCR_F191_7MHz: return 191;
  case PPCR_F206_4MHz: return 206;
  case PPCR_F221_2MHz: return 221;
  case PPCR_F239_6MHz: return 240;
  case PPCR_F250_7MHz: return 251;
  case PPCR_F265_4MHz: return 265;
  case PPCR_F280_2MHz: return 280;
  }
  return 0;
}

static inline unsigned int sa1100_pcmcia_ccf_5795(void){
  switch(FExtr(PPCR, PPCR_CCF)){
  case PPCR_F57_3MHz:  return 57;
  case PPCR_F71_6MHz:  return 71;
  case PPCR_F85_9MHz:  return 85;
  case PPCR_F100_2MHz: return 100;
  case PPCR_F114_5MHz: return 114;
  case PPCR_F128_9MHz: return 128;
  case PPCR_F143_2MHz: return 143;
  case PPCR_F157_5MHz: return 157;
  case PPCR_F171_8MHz: return 171;
  case PPCR_F186_1MHz: return 186;
  case PPCR_F200_5MHz: return 200;
  case PPCR_F214_8MHz: return 214;
  case PPCR_F229_1MHz: return 229;
  case PPCR_F243_4MHz: return 243;
  case PPCR_F257_7MHz: return 258;
  case PPCR_F272_0MHz: return 272;
  }
  return 0;
}


/* Define this to be the correct lookup function for your board.
 * (This could be a configurable parameter.)
 */
#define SA1100_PCMCIA_CCF sa1100_pcmcia_ccf_6864


/* This function implements the BS value calculation for setting the MECR
 * using integer arithmetic:
 */
static inline unsigned int sa1100_pcmcia_mecr_bs(unsigned int pcmcia_cycle_ns,
						 unsigned int cpu_clock_mhz){
  unsigned int t=((pcmcia_cycle_ns*cpu_clock_mhz)/6)-1000;
  return (t/1000)+(((t%1000)==0)?0:1);
}


/* When configuring I/O maps, Card Services appears to adopt the policy
 * that a memory access time of "0" means "use the default." The default
 * PCMCIA I/O access time is 255ns:
 */
#define SA1100_PCMCIA_IO_ACCESS      (255)


/* This structure encapsulates per-socket state which we might need to
 * use when responding to a Card Services query of some kind.
 */
struct sa1100_pcmcia_socket {
  socket_state_t        cs_state;
  struct pcmcia_state   k_state;
  unsigned int          irq;
  void                  (*handler)(void *, unsigned int);
  void                  *handler_info;
  struct bus_operations bus_ops;
};


/* I/O pins replacing memory pins
 * (PCMCIA System Architecture, 2nd ed., by Don Anderson, p.75)
 *
 * These signals change meaning when going from memory-only to 
 * memory-or-I/O interface:
 */
#define iostschg bvd1
#define iospkr   bvd2


/* These values are chosen in accordance with the assumption mentioned in
 * the Linux PCMCIA Programmer's Guide (sections 9.2.5, 9.2.6):
 */
#define SA1100_PCMCIA_IO_MAP_COUNT  (2)
#define SA1100_PCMCIA_MEM_MAP_COUNT (5)


/*
 * Declaration for all implementation specific low_level operations.
 */
extern struct pcmcia_low_level assabet_pcmcia_ops;
extern struct pcmcia_low_level neponset_pcmcia_ops;
extern struct pcmcia_low_level bitsy_pcmcia_ops;
extern struct pcmcia_low_level cerf_pcmcia_ops;


#endif  /* !defined(_PCMCIA_SA1100_H) */
