/*
*
* Driver for the H3650 Touch Screen and other Atmel controlled devices.
*
* Copyright 2000 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author: Charles Flynn.
*
*/

#ifndef __KERNEL__
#  define __KERNEL__
#endif
#ifndef MODULE
#  define MODULE
#endif

/* This version stuff is done ONCE or there are errors */
#define __NO_VERSION__ /* don't define kernel_verion in module.h */
#include <linux/module.h>
#include <linux/version.h>
/* UTS_RELEASE is defined in version.h */
char kernel_version [] = UTS_RELEASE;

#include <linux/init.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/poll.h>
#include <asm/uaccess.h>        /* get_user,copy_to_user */
#include <linux/string.h>

/* SA1100 serial defines */
#include <asm/arch/hardware.h>
#include <asm/arch/serial_reg.h>
#include <asm/arch/irqs.h>

#include <linux/h3650_ts.h>

/* ++++++++++++local function defines ++++++++++++*/
static int checkDevice(struct inode *pInode);

static int initSerial( void );
static void ClearInterrupt(void);
static int readByte(void);
/* These are the stub functions until I get around to implementing them */
static int xxIoctl(void * , unsigned int , unsigned long );
static int xxRead(struct dev *);
static int xxRes(struct dev *);
static int xxInit(struct dev *);
/* These are the touch screen functions */
static int tsEvent(DEV *);	/* event handler */
static int tsRead(struct dev *);	/* file operations _read() */
static int tsIoctl(void * , unsigned int , unsigned long );
static int tsInit(DEV *);

/* ++++++++++++local function defines ++++++++++++*/


/* +++++++++++++globals ++++++++++++++++*/
int gMajor = 0;			/* TODO dynamic major for now */
/* This is the pointer to the UART receiving the ATMEL serial data */
unsigned long   port = (unsigned long)&Ser1UTCR0;


VERSION_DEV verDev;		/* the raw ISR-managed structure */
VERSION_DEV verRetDev;		/* The structure returned to the client */

KEYBD_DEV keybdDev;
KEYBD_DEV keybdRetDev;

TS_DEV tsDev;
TS_DEV tsRetDev;

EPR_DEV epromRdDev;
EPR_DEV epromRdRetDev;

EPW_DEV epromWrDev;
EPW_DEV epromWrRetDev;

LED_DEV nledDev;
LED_DEV nledRetDev;

BAT_DEV battDev;
BAT_DEV battRetDev;

IICRD_DEV iicRdDev;
IICRD_DEV iicRdRetDev;

IICWR_DEV iicWrDev;
IICWR_DEV iicWrRetDev;

FLT_DEV fliteDev;
FLT_DEV fliteRetDev;


/*
	++++++++++++++ start of driver structures ++++++++++++++
	Each Atmel microcontroller device is represented in this array.
*/
DEV g_Devs[MAX_ID] = {
{	/* ID=0 Get/Set Version */
(void *)&verDev, (void *)&verRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(verDev), sizeof(verRetDev) },

{ 0,0,0,0,0,0,0,0,0,0 },

{	/* ID=1 keybd */
(void *)&keybdDev, (void *)&keybdRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(keybdDev), sizeof(keybdRetDev) },

{	/* ID=3 Touch Screen */
(void *)&tsDev, (void *)&tsRetDev, tsRead, tsIoctl, tsEvent, tsInit,	
0, 0, sizeof(tsDev), sizeof(tsRetDev)
},

{	/* ID=4 EEPROM READ */
(void *)&epromRdDev, (void *)&epromRdRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(epromRdDev), sizeof(epromRdRetDev)
},

{	/* ID=5 EEPROM WRITE */
(void *)&epromWrDev, (void *)&epromWrRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(epromWrDev), sizeof(epromWrRetDev)
},
{ 0,0,0,0,0,0,0,0,0,0 },	/* ID=6 N/A */
{ 0,0,0,0,0,0,0,0,0,0 },	/* ID=7 N/A */

{	/* ID=8 LED */
(void *)&nledDev, (void *)&nledRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(nledDev), sizeof(nledRetDev)
},
{	/* ID=9 Battery */
(void *)&battDev, (void *)&battRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(battDev), sizeof(battRetDev)
},
{ 0,0,0,0,0,0,0,0,0,0 },	/* ID=10 N/A */
{	/* ID=11 IIC Write */
(void *)&iicRdDev, (void *)&iicRdRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(iicRdDev), sizeof(iicRdRetDev)
},
{	/* ID=12 IIC Write */
(void *)&iicWrDev, (void *)&iicWrRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(iicWrDev), sizeof(iicWrRetDev)
},
{	/* ID=13 Front Light */
(void *)&fliteDev, (void *)&fliteRetDev, xxRead, xxIoctl, xxRes, xxInit,
0, 0, sizeof(fliteDev), sizeof(fliteRetDev)
}

};

/* ++++++++++++++ end driver structures ++++++++++++++*/

DEV g_Devs[MAX_ID];
RXDEV rxdev;		/* receive ISR state */
TXDEV txdev;		/* receive ISR state */
/* +++++++++++++globals ++++++++++++++++*/


/* +++++++++++++File operations ++++++++++++++*/
static ssize_t h3650_read(struct file * filp, char * buf, size_t count, 
					   loff_t * l);
static int h3650_ioctl(struct inode * inode, struct file *filp
				   ,unsigned int cmd , unsigned long arg);
static unsigned int h3650_poll(struct file * filp, struct poll_table_struct * wait);
static int h3650_open(struct inode * inode, struct file * filp);
static int h3650_release(struct inode * inode, struct file * filp);

static void eventIsr(int , void *, struct pt_regs *);	/* ISR :*/


struct file_operations ts_fops = {
		NULL,			/* LSEEK */
		h3650_read,		/* READ  */
		NULL,			/* WRITE */
		NULL,			/* READDIR */
		h3650_poll,		/* SELECT */
		h3650_ioctl,		/* IOCTL */
		NULL,			/* MMAP */
		h3650_open,		/* OPEN */
		NULL,		
		h3650_release,		/* CLOSE */
		NULL,
		NULL
};

/* +++++++++++++File operations ++++++++++++++*/

/*
	TODO put back TS_GET/SET_RAW
*/
int h3650_ioctl(struct inode * inode, struct file *filp,
				   unsigned int cmd , unsigned long arg)
{
    int minor;
    DEV dev;

    minor = checkDevice( inode );
    if ( minor == - 1)
	return -ENODEV;
    
    printk("h3650_ioctl:minor=%d\n",minor);

    dev = g_Devs[minor];
    if( dev.processIoctl )
    	return ( (*dev.processIoctl)(dev.pdev,cmd,arg) ) ;
    
    return 0;
}

/*
	This IS generic.
	TODO  for some devices poll will be inappropriate.
*/
unsigned int h3650_poll(struct file * filp, struct poll_table_struct * wait)
{

    int minor;
    DEV dev;


    minor = checkDevice( filp->f_dentry->d_inode);
    if ( minor == - 1)
	return -ENODEV;

    printk("h3650_poll:\n");

    dev = g_Devs[minor];
    poll_wait(filp,&dev.wq,wait);
    /*
	The reason head & tail are in the generic structure is
	to allow us to fast poll
    */
    return ( dev.head == dev.tail) ? 0 : (POLLIN | POLLRDNORM);
}

/*
	This IS generic
*/

ssize_t h3650_read(struct file * filp, char * buf, size_t count, loff_t * l)
{
    int nonBlocking = filp->f_flags & O_NONBLOCK;
    int minor;
    DEV dev;


    minor = checkDevice( filp->f_dentry->d_inode );
    if ( minor == - 1)
	return -ENODEV;

    printk("h3650_read: minor=%d\n",minor);

    dev = g_Devs[minor];

    if( nonBlocking )
    {
	if( ( dev.head != dev.tail ) && dev.processRead )
	{
	    int count;
	    count = (*dev.processRead)(&dev);
	    /* returns length to be copied otherwise errno -Exxx */
	    if( count > 0 )
    		__copy_to_user(buf,(char *)dev.pReturn,count );
	    return count;
	}
	else
	    return -EINVAL;

    }
    else
    {
	int i;
	struct task_struct * curr = get_current();

	printk("ts_read: waiting for I/O to complete\n");		

     	interruptible_sleep_on(&dev.wq);

	/* TODO is this code correct? */
	for (i=0 ; i < _NSIG_WORDS; i++)
	{
		if (curr->signal.sig[i] & ~curr->blocked.sig[i])
		{
			printk("ts_read: -ERESTARTSYS\n");
			return -ERESTARTSYS;
		}
	}
	/* check , when woken, there is a complete event to read */
	if( ( dev.head != dev.tail ) && dev.processRead )
	{
	    int count;
	    count = (*dev.processRead)(&dev);
	    /* returns length to be copied otherwise errno -Exxx */
	    if( count > 0 )
    		__copy_to_user(buf,(char *)dev.pReturn,count );
	    return count;	/* fmtLen if success -Exxx if error */
	}
	else
	{
	    printk("ts_read: PANIC: no events\n");
	    return -EINVAL;
	}

	
    }
}


/*
	TODO TODO do we need an open or release since
	IRQs are enabled from init_module
*/
int h3650_open(struct inode * inode, struct file * filp)
{
    int minor;

    minor = checkDevice( inode );
    if ( minor == - 1)
	return -ENODEV;

    printk("h3650_open:minor=%d\n",minor);

    return 0;
}

int h3650_release(struct inode * inode, struct file * filp)
{
    int minor;
    minor = checkDevice( inode );
    if ( minor == - 1)
	return -ENODEV;
    printk("h3650_release:minor=%d\n",minor);
    return 0;
}


int __init h3650_init(void)
{
	/* TODO TODO what happens if there is an error - see pdriver */
	int result;
	unsigned i;
	DEV dev;
	printk("init_module:\n");

        /* register our character device TODO TODO do we have a static major?*/
        printk("ts_dev: init_module registering char device\n");
	/* TODO TODO what is the top-level NAME of this driver ? */
        result = register_chrdev(0, MOD_NAME, &ts_fops);
        if (result < 0) {
            printk("ts_dev:can't get major number\n");
            return result;
        }

	if( gMajor == 0 )
		gMajor = result;

	initSerial();		/* do this first before irq management */
	
	/* Now register our IRQ - ID must be unique if sharing an IRQ line*/
	free_irq(DEV_IRQ, NULL);
	result = request_irq(DEV_IRQ, eventIsr, SA_SHIRQ | SA_INTERRUPT,
				DEV_IRQ_NAME, DEV_IRQ_ID);
	if (!result)
	{	
       	    printk("init_module successful init major= %d irq=%d\n",
		gMajor,DEV_IRQ);

	    for(i=0 ; i < MAX_ID ; i++)
	    {
		/* Zero device specific structures */
		dev = g_Devs[i];
		memset(dev.pdev,0,dev.lenDev);		
		init_waitqueue_head(&dev.wq);
		memset(dev.pReturn,0,dev.lenRet);		
		if( dev.initDev )
			(*dev.initDev)( &dev );
	    }
	}
	else
		printk("init_module error from irq %d\n", result);
	return 0;
}

void cleanup_module(void)
{
	printk("cleanup_module:\n");

	free_irq(DEV_IRQ, DEV_IRQ_ID);
	unregister_chrdev(gMajor, MOD_NAME);
}


/*
	Returns the Minor number from the inode.
*/
static int checkDevice(struct inode *pInode)
{
	int minor;
	kdev_t dev = pInode->i_rdev;

	if( MAJOR(dev) != gMajor)
	{
		printk("checkDevice bad major = %d\n",MAJOR(dev) );
		return -1;
	}
	minor = MINOR(dev);

	if ( minor < MAX_ID )
		return minor;
	else
	{
		printk("checkDevice bad minor = %d\n",minor );
		return -1;
	}
}

/* +++++++++++++++++ Start of ISR code ++++++++++++++++++++++++++ */


/*
	Sets up serial port 1
	baud rate = 115k
	8 bits no parity 1 stop bit
*/

int initSerial( )
{
    unsigned int status;

    /*
		Do nothing while Transmitter busy flag is TRUE meaning data are
		being transmitted. Note also how status register is read.
    */

    while ( ((volatile unsigned long *)port)[UTSR1_TBY] );

    /*Set all 8 bits to zero in cr3 as it isn't used. */
    ( (volatile unsigned long *)port)[UTCR3]= 0;

    /* 8 bits no parity 1 stop bit */
    ( (volatile unsigned long *)port)[UTCR0] = 0x08;

    /* Set baud rate MS nibble to zero. */
    ( (volatile unsigned long *)port)[UTCR1] = 0;

    /* Set baud rate LS nibble to 115K bits/sec */
    ( (volatile unsigned long *)port)[UTCR2]=0x1;

    /*
	  SR0 has sticky bits, must write ones to them to clear.
	    the only way you can clear a status reg is to write a '1' to clear.
    */
    ( (volatile unsigned long *)port)[UTSR0]=0xff;

    /*
	    enable rx fifo interrupt (SAM 11-116/7)
	    If RIE=0 then sr0.RFS and sr0.RID are ignored ELSE
	    if RIE=1 then whenever sro.RFS OR sr0.RID are high an
		interrupt will be generated.
	    Also enable the transmitter and receiver.
    */
    status = UTCR3_TXE | UTCR3_RXE | UTCR3_RIE;
    ( (volatile unsigned long *)port)[UTCR3]=status;


    /*
	    initialize Serial channel protocol frame
    */
    rxdev.state = STATE_SOF;
    rxdev.event = rxdev.len = 0;
    memset(&rxdev.buf[0],0,sizeof(rxdev.buf));

    return 0;	/* TODO how do we denote an error */

} // End of function initSerial

void ClearInterrupt(void)	/* TODO make this in-line */
{
	/* SR0 has sticky bits, must write ones to them to clear */
	( (volatile unsigned long *)port)[UTSR0]=0xff;
}


int readByte(void)		/* TODO make in-line */
{
	unsigned int status;
	int rxval = RX_NODATA;	/* CF??? guilty until proven otherwise */

	/*
	    If RID is TRUE then -  
		The receiver is enabled AND there is data in the rx buffer
		AND Three frame periods have elapse without reception
	    IF RFS is TRUE then - 
		Receive buffer is at high watermark (33%-66% full) and
		needs serviced.
	*/
	
	status = ( (volatile unsigned long *)port)[UTSR0];
	if( ( status & UTSR0_RFS ) || ( status & UTSR0_RID) )
	{
		if ( ( (volatile unsigned long *)port)[UTSR1] & UTSR1_RNE )
		{
			/*
			Enter here if RNE is TRUE which means the receive
			FIFO has 1 or more bytes to read.
			*/

			rxval = ( (volatile unsigned long *)port)[UTDR];
			if (rxval & 0x700) 
			{
				/*
			 	    Return error if parity,frame
				    or overrun error bits set.
				*/
				rxval = RX_ERR;
			}
		}
	}
	
	return(rxval);
}

static void writeChar(unsigned char txchr)
{
	/*
		TODO TODO make this in-line 
		This routine sends only a single character.
		Loop until tnf=1 (transmitter buffer not full)
		while (c->utsr1.tnf == 0) ;
		Store the byte into the transmitter buffer - presumable an IRQ
		will be sent that will handle the transmit.
	*/
        ( (volatile unsigned long *)port)[UTDR] = txchr;
}

#if 0
static void sendString(unsigned short *str )
{
	/* Send message to serial port */
	while (*str) 
		writeChar((unsigned char)*str++);
}
#endif

/* Called when txdev.buf[] has a frame to send */
void txIsr(void)
{
	unsigned char opval;
	int i;

	writeChar(CHAR_SOF);	/* send SOF */

	opval = (unsigned char)((txdev.event << 4) | txdev.len);

	writeChar(opval);		/* send id & len nibbles */
	txdev.chksum = opval;

	/* Send len bytes */
	for (i=0; i<  txdev.len; i++)
	{
		writeChar(txdev.buf[i]);
		txdev.chksum += txdev.buf[i];
	}

	writeChar(txdev.chksum);	/* send check sum  */
}


/*

	Frame format
  byte	  1	  2		  3 		 len + 4
	+-------+---------------+---------------+--=------------+
	|SOF	|id	|len	| len bytes	| Chksum	|
	+-------+---------------+---------------+--=------------+
  bit	0     7  8    11 12   15 16

	+-------+---------------+-------+
	|SOF	|id	|0	|Chksum	| - Note Chksum does not include SOF
	+-------+---------------+-------+
  bit	0     7  8    11 12   15 16

*/

static void eventIsr(int irq, void *dev_id, struct pt_regs *regs)
{
    int rxval;		/* CF?? this is an integer watch it! */
    unsigned char rxchar;

    if ( strcmp( (char *)dev_id, DEV_IRQ_ID) )
	goto exit;;


    /* read next byte */
    rxval = readByte();	/* TODO put this function in-line */

    if( rxval == RX_NODATA )
	/* Should Not Happen!!! set a counter */
    {
	printk("RX_NODATA\n");
	goto exit;
    }
    else if ( rxval == RX_ERR )
    {
	/* reset the receive state machine and try again */
	rxdev.state = STATE_EOF;
	printk("RX_ERR\n");
	goto exit;
    }
	
    rxchar = (unsigned char)rxval;
    /*
    printk("+++++[%02x:%02x]rxchar=%2x+++++\n",rxdev.state,rxdev.idx,rxchar);
    */
    switch ( rxdev.state )
    {
		/*
			We are looking for the SOF in this state.
		*/
		case STATE_SOF:
			if ( rxchar == CHAR_SOF )
				/* found it - next byte is the id and len */
				rxdev.state=STATE_ID;
			break;
		/*
			We now expect the id and len byte
		*/
		case STATE_ID:
			rxdev.event = (rxchar & 0xf0) >> 4 ;
			rxdev.len = (rxchar & 0x0f);
			if (rxdev.event >= MAX_ID )
			{
				/* increment error counter 1 */
				++rxdev.counter[0];
				rxdev.state = STATE_SOF;
				break;
			}
			rxdev.chksum = rxchar;
			rxdev.state=( rxdev.len > 0 ) ? STATE_DATA : STATE_EOF;
			break;
		/*
			We now expect 'len' data bytes.
		*/
		case STATE_DATA:
			rxdev.chksum += rxchar;
			rxdev.buf[rxdev.idx]= rxchar;
			if( ++rxdev.idx == rxdev.len )
			{
				/* look for the EOF char next */
				rxdev.state = STATE_EOF;
				rxdev.idx=0;
			}
			break;
		/*
			We now expect to find the STATE_EOF
			TODO TODO not sure if this is an STATE_EOF marker
			or simply the checksum byte
		*/
		case STATE_EOF:
			rxdev.state = STATE_SOF;
			if (rxchar == CHAR_EOF || rxchar == rxdev.chksum )
			{
				/* TODO TODO do we have to wake up somebody? */
				DEV dev = g_Devs[(unsigned)rxdev.event];
				if( dev.processEvent )
				{
					printk("+");
					(*dev.processEvent)(&dev);
				}
			}
			else
			    ++rxdev.counter[1];
			break;
		default:
			/* TODO TODO this happens only under catastrphic */
			++rxdev.counter[2];
			break;

    }	/* end switch */

exit:

    ClearInterrupt();	/* TODO make in-line do you have to do this??*/
#if defined PC_PP_PORT
    /* toggle the LEDs attached to the parallel port */
    toggleLed();
#endif

}	//end of function

/* +++++++++++++++++ end of ISR code ++++++++++++++++++++++++++ */

/****************** atmel_gbl.c ***********************/

/*	ID=3 TOUCHSCREEN EVENTS
	TouchScreen event data is formatted as shown below:-

	+-------+-------+-------+-------+
	| Xmsb	| Xlsb	| Ymsb	| Ylsb	|
	+-------+-------+-------+-------+
	byte 0	  1	  2	  3

*/

static int tsEvent(DEV * pDev )		/* touch screen event handler */
{
  unsigned head = pDev->head;
  /* This is where we will store the event */
  TS_DEV * pTsDev = ( TS_DEV * )pDev->pdev;
  ts_event event = pTsDev->events[head];

  if( rxdev.len == TS_DATA_LEN )
  {
    /* data along with the event indicates the pen is DOWN */

    /*
	There are 3 sub-events within the TS_EVENT
	PEN_UP		- user has just lifted pen off screen.
	PEN_FLEETING	- user has just put pen DOWN for the first time.
			- we ignore this event.
	PEN_DOWN	- this is a true sample.

	We assume the PEN is down because the ATMEL sent us data
	now do some more filtering based on the pen's previous status.
    */
    switch(pTsDev->penStatus)
    {
	case PEN_FLEETING:
	    /* TODO how many fleeting before we pronounce a PEN_DOWN? */
	    pTsDev->penStatus = PEN_DOWN;
	    /* OK to use this data -  pass thru to next case*/
	case PEN_DOWN:
	    {
		/* store raw x,y & pressure to ts_dev.event[head] */
		unsigned  short x,y;	/* TODO TODO big or little endian? */
		/* Extract RAW coords from event data  */
		/* TODO TODO this is not same as HTC code !! */
		x = rxdev.buf[1]; x <<= 8; x += rxdev.buf[0];
		y = rxdev.buf[3]; y <<= 8; y += rxdev.buf[2];
		/* dont calibrate it until the user asks for it */
		event.x = x;
		event.y = y;
		event.pressure = 1;
		/* increment head */
		++head;
  		pDev->head = ( head < MAX_TS_EVENTS ) ? head : 0;
	        printk("PEN_DOWN: x=%d y=%d p=%d head=%d\n",
			event.x,event.y,event.pressure,pDev->head);
	    }
	    break;
	case PEN_UP:
	    /* Pen was previously UP dont trust this sample */
	    pTsDev->penStatus = PEN_FLEETING;
	    printk("PEN_FLEETING\n");
	    break;
	default:
	    /* panic TODO increment a counter*/
	    pTsDev->penStatus = PEN_UP;
	    printk("PEN_UNKNOWN\n");
	    return -1;
    }

    return 0;	/* success */
  }
  else if (rxdev.len == 0)	/* TODO should also check sizeof (header) */
  {
    /* PEN_UP - since we have no data in the frame */
    pTsDev->penStatus = PEN_UP;
    event.x = 0;
    event.y = 0;
    event.pressure = 0;
    /* increment jead */
    pDev->head = ( ++head < MAX_TS_EVENTS ) ? head : 0;
    printk("PEN_UP: x=%d y=%d p=%d head=%d\n",
	event.x,event.y,event.pressure,pDev->head);
    return 0;		/* success */
  }
  else
  {
    /* we have a length over or under run */
    printk("LENGTH_UNKNOWN\n");
    return -1;
  }

} /* end of tsEvent() */

/*
	Read() function for the touch screen.
	IN:  raw data
	OUT: Calibrated data pointed to by pReturn.
*/
static int tsRead(DEV * pDev)
{
	/* generic line for all readers */
	TS_DEV * pTsDev = (TS_DEV *)pDev->pdev;


	if( pDev->head != pDev->tail )
	{
	    ts_event raw = pTsDev->events[pDev->tail];
	    ts_fmt * fmt = pDev->pReturn;
	    pDev->tail++;
	    /* TODO TODO calibrate the raw data */
				
	    fmt->x = raw.x;
	    fmt->y = raw.x;
	    fmt->pressure = raw.pressure;
	    /* TODO there may be other stuff to copy to the return buffer */
	}
	return sizeof(tsRetDev);		/* success */
}

/*
	ts_ioctl - ioctl function for touch screen device
*/
static int tsIoctl( void * pDevice, unsigned int cmd, unsigned long arg)
{
    TS_DEV * pdev = (TS_DEV *)pDevice;

    switch(cmd)
    {
	case TS_GET_RATE:
	    return pdev->rate;
	    break;
	case TS_SET_RATE:
	    pdev->rate = arg;
	    break;
	case TS_GET_CAL:
	    __copy_to_user((char *)arg, (char *)&pdev->cal, sizeof(ts_cal) );
	    break;
	case TS_SET_CAL:
	    __copy_from_user( (char *)&pdev->cal,(char *)arg ,sizeof(ts_cal) );
	    
	    printk("TS_SET_CAL: xscale=%d xtrans=%d yscale=%d ytrans=%d\n",
		pdev->cal.xscale,pdev->cal.xtrans,
		pdev->cal.yscale,pdev->cal.ytrans);
	    break;
	default:
	    printk("IOCTL: unknowncmd %04x\n",cmd);
	    return -EINVAL;
    }

    return 0;	/* TODO check this is the correct return */
}

static int tsInit(DEV * pDev)
{
    TS_DEV * pTsDev = (TS_DEV *)pDev->pdev;

    /* TODO TODO any non-zero initialisations here */
    printk("tsInit: not implemented yet\n");

    /* TODO this is zero'd at init_module - need to do it again? */
    pTsDev->penStatus=PEN_DOWN;

    return 0;
}

/* ID=xx
	Test functions 
*/
static int xxIoctl( void * pDevice, unsigned int cmd, unsigned long arg)
{
	printk("xxIoctl: not implemented\n");
	/* TODO TODO this assumes that all returns on xxRead/xxIoctl
		return this value
	*/
	return sizeof(verRetDev);
}
static int xxRead(DEV * pDev)
{
	printk("xxRead: not implemented\n");
	return 0;
}

/* Generic RES function */
static int xxRes(DEV * p_gDev )		/* GET VERSION handler */
{
	printk("xxRes: not implemented\n");
	return 0;
}

/* Generic Init function */
static int xxInit(DEV * p_gDev )		/* GET VERSION handler */
{
	printk("xxInit: not implemented\n");
	return 0;
}
