/*
*
* Driver for the H3650 Touch Screen and other Atmel controlled devices.
*
* Copyright 2000 Compaq Computer Corporation.
*
* Use consistent with the GNU GPL is permitted,
* provided that this copyright notice is
* preserved in its entirety in all copies and derived works.
*
* COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
* AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
* FITNESS FOR ANY PARTICULAR PURPOSE.
*
* Author: Charles Flynn.
*
*/


#ifndef __H3650_TS_H__
#define __H3650_TS_H__


#ifdef __KERNEL__

/* ++++++++++++++++++ ISR related defines ++++++++++++++++ */

#define DEV_IRQ         IRQ_Ser1UART	/* see irqs.h */
#define DEV_IRQ_NAME    "h3650" /* cat/proc/interrupts displays this */
#define DEV_IRQ_ID      "h3650"   /* compared when a shared IRQ is used */

/* The start and end of block characters SOF and EOF */
#define CHAR_SOF		0x02
#define	CHAR_EOF		0x03
		
/* decode States  */
#define	STATE_SOF	0	/* start of FRAME */
#define	STATE_ID	1	/* state where we decode the ID & len */
#define	STATE_DATA	2	/* state where we decode data */
#define	STATE_EOF	3	/* state where we decode checksum or EOF */

/*
	Atmel events and response IDs contained in frame.
	Programmer has no control over these numbers.
	TODO there are holes - specifically  1,6,7,0x0a
*/
#define VERSION_ID		0	/* Get Version (request/respose) */
#define	KEYBD_ID		2	/* Keboard (event) */
#define TOUCHS_ID		3	/* Touch Screen (event)*/
#define EEPROM_READ_ID		4	/* (request/response) */
#define EEPROM_WRITE_ID		5	/* (request/response) */
#define	NLED_ID			8	/* (request/response) */
#define	BATTERY_ID		9	/* (request/response) */
#define IIC_READ_ID		0x0b	/* ( request/response) */
#define IIC_WRITE_ID		0x0c	/* ( request/response) */
#define FLITE_ID		0x0d	/* backlight ( request/response) */

#define	MAX_ID			14

#define TXBUF_MAX	16
/* TODO find out how we use Tx */
typedef struct
{
	char state;			/* context of tx state machine */
	char event;			/* event ID from packet */
	unsigned char chksum;
	char buf[TXBUF_MAX];		/* transmitter buffer */
	short len;			/* tx buffer length */
}TXDEV;



/* event independent structure */
#define RXBUF_MAX	16		/* Do NOT make this bigger!! */
#define MAX_RXDEV_COUNTERS      3       /* array of counters */
typedef struct
{
        unsigned char state;            /* context of rx state machine */
        unsigned char event;            /* event ID from packet */
        unsigned char chksum;
        unsigned char pad;              /* keep structure on a 32 bit bdry */
        unsigned int idx;               /* rx buffer index */
        unsigned int len;               /* rx buffer length */
        unsigned int counter[MAX_RXDEV_COUNTERS];
	/* WARNING making buf[]  signed will cause problems */
        unsigned char buf[RXBUF_MAX];            /* receive event buffer */
} RXDEV;

/* status returns from ReadByte() */	/* TODO */
#define RX_NODATA	-1
#define	RX_ERR		-2

/* ++++++++++++++++++ end of ISR related defines ++++++++++++++++ */

/*
	Define all the module name
	TODO non-contiguous is this a problem?
	TODO do we have a separate tsraw device?
	The following device nodes will also be created
	Minor	Name
	0	ver
	2	keybd
	3	ts
	4	eeprom	( 4=eeprom_read 5=eeprom_write )
	8	led
	9	battery
	11	iic	( 11=iicread 12=iicwrite)
	13	light	( back light )
	
*/
#define MOD_NAME	"h3650_ts"

typedef struct {
  unsigned short pressure;
  unsigned short x;
  unsigned short y;
} ts_event;

/* TODO what will the client want returned - change accordingly */
typedef struct {
  unsigned short pressure;
  unsigned short x;
  unsigned short y;
} ts_fmt;

typedef struct {
  int xscale;
  int xtrans;
  int yscale;
  int ytrans;
} ts_cal;

/* TODO */
typedef struct {
  int dummy1;
  int dummy2;
} ts_raw;


/* ++++++++++++++ declare the generic driver structures ++++++++++++++*/
typedef struct dev
{
    void * pdev;			/* device specific structure */
    void * pReturn;			/* pointer to the return struct */
    int (*processRead)(struct dev * );
    int (*processIoctl)(void * , unsigned int , unsigned long );
    int (*processEvent)(struct dev *);	/* pointer to event handler */
    int (*initDev)(struct dev *);	/* function to initialise test data */
    unsigned head;
    unsigned tail;
    unsigned lenDev;			/* sizeof(*pdev) */
    unsigned lenRet;			/* sizeof(*Return) */
    wait_queue_head_t wq;		/* for 2.2.3.xx kernels */
}DEV;
/* ++++++++++++++ +++++++++++++++++++++++++++++++++++++ */


/* Repeat above for other events example the Dummy event (DM) */
int NullRes(void);	/* place holder for holes in the event table */

typedef struct VER_DEV
{
	int dummy1;
	int dummy2;
}VERSION_DEV;

typedef struct
{
	int dummy1;
	int dummy2;
} KEYBD_DEV;


#define MAX_TS_EVENTS	4		/* how many do we want to buffer */
#define TS_DATA_LEN	4		/* sizeof payload ( in bytes) */

#define	PEN_UP		0	/* default at module load time */
#define	PEN_DOWN	1	/* pen is well and truely down */
#define	PEN_FLEETING	2	/* pen is neither up or down */

typedef struct
{
    unsigned rate;		/* rate in samples per sec from the ATMEL */
    unsigned penStatus;		/* PEN_UP, PEN_DOWN, PEN_SAMPLE */
    ts_cal cal;			/* ts calibration parameters */
    ts_event events[MAX_TS_EVENTS];	/* protect against overrun */
} TS_DEV;

typedef struct
{
	int dummy1;
	int dummy2;
} EPR_DEV;

typedef struct
{
	int dummy1;
	int dummy2;
} EPW_DEV;


typedef struct
{
	int dummy1;
	int dummy2;
} LED_DEV;

typedef struct
{
	int dummy1;
	int dummy2;
} BAT_DEV;

typedef struct
{
	int dummy1;
	int dummy2;
} IICRD_DEV;

typedef struct
{
	int dummy1;
	int dummy2;
} IICWR_DEV;

typedef struct
{
	int dummy1;
	int dummy2;
} FLT_DEV;

#endif		/* ifndef __KERNEL__ */

/* IOCTL cmds  user or kernel space */

/* Use 'f' as magic number */
#define WT_IOC_MAGIC  'f'

#define TS_GET_RATE		_IO(WT_IOC_MAGIC, 1)
#define TS_SET_RATE		_IO(WT_IOC_MAGIC, 2)
#define TS_GET_CAL		_IOR(WT_IOC_MAGIC, 3, ts_cal)
#define TS_SET_CAL		_IOW(WT_IOC_MAGIC, 4, ts_cal)
#define TS_GET_RAW		_IOR(WT_IOC_MAGIC, 5, ts_raw)
#define TS_SET_RAW		_IOW(WT_IOC_MAGIC, 6,ts_raw)

#define WT_IOC_MAXNR 6

#endif
