/****************************************************************************/
/* Copyright 2000 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * bootldr file for Compaq Personal Server Bootloader
 *
 */

/*
 * Maintainer: Jamey Hicks (jamey@crl.dec.com)
 * Original Authors: Edwin Foo, Jamey Hicks, Dave Panariti, Mike Schexnaydre, Chris Joerg
 * June 29, 2000 - save commands and xmodem send added.
 *                             - George France <france@crl.dec.com>
 * July  3, 2000 - add commands to allow the baudrate to be changed.
 *                             - George France <france@crl.dec.com>
 * July  5, 2000 - extended the range of baudrate change.
 *                             - George France <france@crl.dec.com>
 * July  5, 2000 - added PF_LOGICAL.
 *                             - George France <france@crl.dec.com>
 * July  6, 2000 - added display command.
 *                             - George France <france@crl.dec.com>
 * July 31, 2000 - add Architecture field to boot loader.
 *                             - George France <france@crl.dec.com>
 * Aug  04, 2000 - made the cached and uncached flash size the 
 *                 same as the actual flash size.
 *                             - George France <france@crl.dec.com>
 * Aug  04, 2000 - Auto protected the boot sector after the boot
 *                 loader is loaded.
 *                             - George France <france@crl.dec.com> 
*/

static char bootprog_name[] = "Compaq OHH BootLoader";
static char bootprog_date[] = DATE;

static char cvsversion[] = "$Id: bootldr.c,v 1.65.2.18 2001/10/15 17:57:23 bavery Exp $";

#if defined(CONFIG_BITSY)
int do_splash = 1;
int packetize_output = 0;
int ack_commands = 0;
int no_cmd_echo = 0;
int use_extended_getcmd = 1;

int check_for_func_buttons = 1;
long reboot_button_enabled = 0;
#endif

unsigned long last_ram_load_address = 0;

#define USE_XMODEM

#include "bootldr.h"
#include "btpci.h"
#include "btflash.h"
#include "btusb.h"
#include "heap.h"
#include "xmodem.h"
#include "lcd.h"
#include "pbm.h"
#if defined(CONFIG_BITSY)
#include "aux_micro.h"
#include "zlib.h"
#endif
#ifdef __linux__
typedef unsigned long u32;
typedef unsigned short u16;
typedef unsigned char u8;
#include <asm-arm/setup.h>
#endif
#include "sa1100.h"
#include "bsdsum.h"
#include "architecture.h"
#ifdef CONFIG_USE_DATE_CODE
extern char _bootldr_date_code[];
#endif
#ifdef CONFIG_LOAD_KERNEL
#include "lkernel.h"
#endif

#ifdef BZIPPED_KERNELS
#define BZ_NO_STDIO
#include "bzip/bzlib.h"
#endif

#define	DEB(s)	putstr(s)
#define	PLW(x)	putLabeledWord(#x"=0x", (unsigned long)(x))

typedef unsigned long u_int32_t;
typedef unsigned long pd_entry_t;
int lcd_params = LCDP_COLOR;
long paramoldvalue;
#include "cyclone_boot.h"
		
#define ROUNDUP(v,szpwr2) (((v)+((szpwr2)-1))&~(szpwr2-1))

#define	BL_READ_GPIO(off)   (*(int *)(0x21800008 + off))
#define MAX_BOOTARGS_LEN 2048

extern void *SerBase;

extern void getcmd_ex(char*, size_t);

extern void update_autoboot_timeout(struct bootblk_param *param);
extern void update_cmdex(struct bootblk_param *param);

#if 0
void reboot(void);
#endif
void enableMMU(void);
void flushTLB(void);
void writeBackDcache(unsigned long region);
unsigned long readCPR0(void);
unsigned long readCPR1(void);
unsigned long readCPR2(void);
unsigned long readCPR3(void);
unsigned long readCPR5(void);
unsigned long readCPR6(void);
unsigned long readCPR13(void);
unsigned long readCPR14(void);
unsigned long  asmEnableCaches(unsigned long dcacheFlag,unsigned long icacheFlag);

/* Boot Loader Function Prototypes */
void print_help_on_commands(struct bootblk_command *commands);
void command_help(int argc, const char **argv);
void command_boot(int argc, const char **argv);
void command_boot_flash(int argc, const char **argv);
void command_boot_jffs2(int argc, const char **argv);
void command_boot_wince(int argc, const char **argv);
void command_boot_nfsroot(int argc, const char **argv);
void command_display(int argc, const char **argv );
void command_load(int argc, const char **argv);
void command_load_flash(int argc, const char **argv);
void command_load_ram(int argc, const char **argv);
/* helper function */
static void command_load_flash_region(const char *regionName, unsigned long regionBase, unsigned long regionSize, int flags);
void command_save(int argc, const char **argv);
void command_save_all(int argc, const char **argv);
void command_save_flash(int argc, const char **argv);
void command_save_ram(int argc, const char **argv);
void command_save_world(int argc, const char **argv);
void command_peek(int argc, const char **argv);
void command_peek_ram(int argc, const char **argv);
void command_poke(int argc, const char **argv);
void command_poke_ram(int argc, const char **argv);
void command_breakpoint(int argc, const char **argv);
void command_qflash(int argc, const char **argv);
void command_eflash(int argc, const char **argv);
void command_pflash(int argc, const char **argv);
#if 0
void command_reboot(int argc, const char **argv);
#endif
void command_physaddr(int argc, const char **argv);
void command_call(int argc, const char **argv);
void command_set(int argc, const char **argv);
struct bootblk_param *get_param(const char *name);
int get_param_value(const char *name, void *value_p);
void command_flash_type(int argc, const char **argv);
void command_params(int argc, const char **argv);
void command_params_show(int argc, const char **argv);
void command_params_eval(int argc, const char **argv);
void parseParamFile(unsigned char *src,unsigned long size,int just_show);
void params_eval_file(int just_show);
void params_eval_partition(const char* prefix_in,int just_show);
void command_params_save(int argc, const char **argv);
void command_partition(int argc, const char **argv);
void testCirrusCardbusBridge(int argc, const char **argv);
void command_cmdex(int argc, const char **argv);

#if defined(CONFIG_BITSY)
void command_aux_ser(int argc, const char **argv);
void command_lcd_test(int argc,const char* argv[]);
void command_lcd_on(int argc,const char* argv[]);
void command_lcd_off(int argc,const char* argv[]);
void command_lcd_light(int argc,const char* argv[]);
/*  void command_lcd_pal(int argc,const char* argv[]); */
void command_lcd_fill(int argc,const char* argv[]);
void command_lcd_bar(int argc,const char* argv[]);
void command_led_blink(int argc,const char* argv[]);
void command_lcd_image(int argc, const char* argv[]);
void command_lcd_zimage(int argc, const char* argv[]);
void command_ttmode(int argc, const char* argv[]);
void command_ser_con(int argc, const char* argv[]);
void command_irda_con(int argc, const char* argv[]);
void command_splash(int argc, const char* argv[]);
void command_version(int argc, const char* argv[]);
void parseParamName(int argc,const char **argv,char *pName,char *pValue);
#endif

void command_memcpy(int argc, const char* argv[]);
void command_hex_dump(int argc, const char* argv[]);
void command_reset(int argc, const char* argv[]);
void command_enable_caches(int argc, const char* argv[]);
void command_memcmp(int argc, const char* argv[]);
void command_boot_addr(int argc, const char** argv);
void command_testJFFS2(int argc, const char* argv[]);
void body_testJFFS2(char *filename,unsigned long *dest);
void command_infoJFFS2(int argc, const char* argv[]);
void body_infoJFFS2(char *partname);
void command_timeFlashRead(int argc, const char* argv[]);
void body_timeFlashRead(char *partname);
void command_p1_load_file(int argc, const char* argv[]);
long body_p1_load_file(char *partname,char *filename,unsigned char *dest);
void command_p1_ls(int argc, const char* argv[]);
void body_p1_ls(char *dir,char *partname);

void command_cmpKernels(int argc, const char* argv[]);
void body_cmpKernels(char *partname,unsigned long *dstFlash,unsigned long *srcJFFS,unsigned long len);

void command_clearMem(int argc, const char* argv[]);
void command_pef(int argc, const char* argv[]);
void command_lli(int argc, const char* argv[]);
void command_cat(int argc, const char* argv[]);
void body_clearMem(unsigned long num,unsigned short *dst);
unsigned char isWincePartition(unsigned char *p);
void parseargs(char *argstr, int *argc_p, char **argv, char** resid);
void unparseargs(char *argstr, int argc, const char **argv);
void splash_file(char * fileName, char *partName);
void splash();

unsigned long autoboot_timeout = TIMEOUT;

byte awaitkey(unsigned long delay, int *error_p);
void bootmenu(void);

void monitorConfigureMMU(void);

#if defined(CONFIG_GZIP) || defined(CONFIG_GZIP_REGION)
int gunzip_region(char*   src,    char*   dst,
		  long len, const char *name);
#endif

/* Boot Loader Variables/Constants */

static struct bootblk_command commands[] = {
  { "?",      command_help, "?" },
  { "help",   command_help, "help <command>" },
  { "help",   command_help, "<command> help" },
  { "boot",   command_boot, "boot [jffs2|flash|nfsroot]" },
  { "display",command_display, "display" },
  { "load",   command_load, "load [kernel | ramdisk | bootldr | params | ram]" },
  { "upload",   command_save, "upload [all | bootldr | flash | kernel | ramdisk | params | world]" },
  { "peek",   command_peek, "peek ram|flash|int|short|byte <addr>" },
  { "poke",   command_poke, "poke ram|flash|int|short|byte <addr>" },
#if defined(CONFIG_SA1100)
  { "breakpoint", command_breakpoint, "breakpoint" },
#endif
  { "qflash", command_qflash, "qflash [cfi|id] <waddr> -- query flash" },
  { "eflash", command_eflash, "eflash <partitionname>|<sectoraddr> [<size>]|chip -- erase partition, sector, range or chip" },
  { "pflash", command_pflash, "pflash <addr> <len> 0/1 (1 -> protect, 0 -> unprotect all!) -- protect address range" },
#if 0
  { "reboot", command_reboot, "reboot" },
#endif
  { "physaddr", command_physaddr, "physaddr <vaddr> -- returns <paddr>" },
  /*  { "testusb", testUSB, "testusb" },*/
  { "call",   command_call, "call <addr> args" },
  { "jump",   command_call, "jump <addr>" },
  { "set",    command_set, "set <param> <value>" },
  { "show",    command_params_show, "show [<param1>] [<param2>] [<param3>]" },
  { "evalparams", command_params_eval, "evalparams [prefix]"},
  { "params", command_params, "params [eval|show|save [-n]|reset]"},
  { "partition", command_partition, "partition [add <name> <base> <size> <flags>|delete <name>|show|save]  (flags 16 for JFFS2)"},
  { "flash_type", command_flash_type, "flash_type <flashtype>"},
#if 0
  { "cardbus", testCirrusCardbusBridge, "cardbus" },
#endif
  { "cmdex", command_cmdex, "cmdex 0|1 (no|yes extended cmdline editing)"},
  
#if defined(CONFIG_BITSY)
  { "auxser", command_aux_ser, "auxser -- dump aux micro's serial data)"},
  { "lcdt", command_lcd_test, "lcdt -- test lcd"},
  { "lcdon", command_lcd_on, "lcdon -- turn lcd back on"},
  { "lcdoff", command_lcd_off, "lcdoff -- turn lcd off"},
  { "lcdlight", command_lcd_light, "lcdlight level -- adjust lcd backlight"},
  /*  { "lcdpal", command_lcd_pal, "lcdpal -- set lcd pal"}, */
  { "lcdfill", command_lcd_fill, "lcdfill -- fill lcd display"},
  { "lcdbar", command_lcd_bar, "lcdbar -- fill lcd bar with [color] to [percent]"},
  { "lcdimg", command_lcd_image, "lcdimg -- display image"},
  { "lcdzimg", command_lcd_zimage, "lcdzimg len -- display gzipped image"},
  { "ledblink", command_led_blink, "ledblink -- blink the led for [on time] [off time] [off = 0]"},
  { "ttmode", command_ttmode, "ttmode 1|0 -- go into tt mode"},
  { "ser_con", command_ser_con, "ser_con -- start a serial console session"},
  { "irda_con", command_irda_con, "irda_con -- start an irda console session"},
  { "splash", command_splash, "splash -- display splash image on lcd"},
  { "ver", command_version, "ver -- display version info."},

#endif
  
  { "memcpy", command_memcpy, "memcpy -- dst src num [size]"},
  { "memcmp", command_memcmp, "memcmp -- addr1 addr2 num [size]"},
  { "hd", command_hex_dump, "hd -- hexdump memory -- addr [num=16]"},
  { "reset", command_reset, "reset -- do software reset."},
  { "enable_caches", command_enable_caches, "enable_caches dcache[0,1] and icache[0,1]  ."},
  { "baddr", command_boot_addr,
    "baddr <addr> <size> -- boot kernel loaded at <addr> of len <size>."},
  /*{ "testJFFS2", command_testJFFS2, "testJFFS2 -- --old way-- check JFFS2 reading by copying [file] to [dest] "},*/  
  { "infoJFFS2", command_infoJFFS2, "infoJFFS2 -- scan JFFS2 partition for useful info "},
/*  { "timeFlashRead", command_timeFlashRead, "timeFlashRead -- time a scan of the [partition] "},*/
  
  { "p1_load_file", command_p1_load_file, "p1_load_file -- load from [partition] the [file] to [dest] "},
  { "ls", command_p1_ls, "ls files from [dir] on [partition (default=param kernel_partition)]"},
   { "cmpKernels", command_cmpKernels, "cmpKernels -- copy flash kernel from [part] to [dst] and cmp with kernal at [src] of len [n]"},
  { "clearMem", command_clearMem, "clearMem -- write [num] 0x00 to a [dst]"},
  { "pef", command_pef, "pef -- [just_show]"},
  { "cat", command_cat, "cat -- show filename from [partition (default=param kernel_partition)]"},
  { "lli", command_lli, "lli -- low level coprocessor info"},
  { NULL,     NULL, NULL }
};

static struct bootblk_command boot_commands[] = {
  { "flash",  command_boot_flash, "boot flash [bootargs ...]" },
  { "jffs2",  command_boot_jffs2, "boot jffs2 [bootargs ...]" },  
  { "nfsroot",   command_boot_nfsroot, "boot nfsroot [bootargs ...]" },
  { "wince",   command_boot_wince, "boot wince [nothing yet ...]" },
/*  { "ram",  command_boot_ram, "boot ram [bootargs ...]" },*/
  
  { NULL,     NULL, NULL }
};

static struct bootblk_command load_commands[] = {
  { "<partition>", command_load, "load <partition>" },
  { "ram",     command_load_ram, "load ram <dstaddr> <size>" },
  { "flash",   command_load_flash, "load flash <dstaddr>" },
  { NULL,     NULL, NULL }
};

static struct bootblk_command save_commands[] = {
  { "<partition>", command_save,         "save <partition>" },
  { "all",       command_save_all,       "save all" },
  { "flash",     command_save_flash,     "save flash <addr> <len>" },
  { "ram",       command_save_ram,       "save ram <addr> <len>" },
  { "world",     command_save_world,     "save world" },
  { NULL,        NULL,                   NULL }
};

static struct bootblk_command peek_commands[] = {
  { "ram",    command_peek_ram, "peek ram <addr> reads 32 bits" },
  { "byte",   command_peek_ram, "peek byte <addr> reads 8 bits" },
  { "short",   command_peek_ram, "peek short <addr> reads 16 bits" },
  { "int",    command_peek_ram,  "peek int <addr> reads 32 bits" },
  { "flash",  command_peek_ram, "peek flash <offset>" },
  { "gpio",   command_peek_ram, "peek gpio <offset>" },
  { NULL,     NULL, NULL }
};

static struct bootblk_command poke_commands[] = {
  { "ram",    command_poke_ram, "poke ram <addr> <dword>" },
  { "byte",   command_poke_ram, "poke byte <addr> <byte> writes 8 bits" },
  { "short",   command_poke_ram, "poke short <addr> <word> writes 16 bits" },
  { "int",   command_poke_ram, "poke int <addr> <dword> writes 32 bits" },
  { "flash",  command_poke_ram, "poke flash <offset> <dword>" },
  { "gpio",  command_poke_ram, "poke gpio <offset> <dword>" },
  { NULL,     NULL, NULL }
};

static struct bootblk_command params_commands[] = {
  { "eval",  command_params_eval, "params eval" },
  { "show",  command_params_show, "params show" },
  { "save",  command_params_save, "params save" },
  { "reset",  command_params,     "params reset" },
  { NULL,     NULL, NULL }
};

static struct bootblk_param bootldr_params[] = {
#define UPDATE_BAUDRATE
#ifdef UPDATE_BAUDRATE
    { "baudrate", PT_INT, PF_DECIMAL, UART_BAUD_RATE, update_baudrate }, 
#endif
    { "os", PT_STRING, PF_STRING, (long)"autoselect", NULL },  /* "linux", "netbsd", "autoselect" */
    { "boot_type", PT_STRING, PF_STRING, (long)"jffs2", NULL },
    { "kernel_in_ram", PT_INT, PF_HEX, 0xC0008000, NULL }, /* set this to the address of where kernel is loaded in ram (0xC0008000) */
    { "autoboot_kernel_part", PT_STRING, PF_STRING, (long)"kernel", NULL },
    { "force_unzip", PT_INT, PF_HEX, 0, NULL }, 
    { "noerase", PT_INT, PF_HEX, 0, NULL }, 
    { "override", PT_INT, PF_HEX, 0, NULL }, 
#ifdef CONFIG_SKIFF
    { "entry", PT_INT, PF_HEX, 0x00000000, NULL }, 
#else
    { "entry", PT_INT, PF_HEX, 0xC0000000, NULL }, 
#endif
    { "copy_ramdisk", PT_INT, PF_HEX, 0, NULL },
    { "dram_size", PT_INT|PT_READONLY, PF_HEX, DRAM_SIZE, NULL },
    { "memc_ctrl_reg", PT_INT, PF_HEX, 0x110c, NULL },
#ifdef CONFIG_PCI
    { "mem_fclk_21285", PT_INT|PT_READONLY, PF_DECIMAL, 48000000, NULL },
    { "maclsbyte", PT_INT, PF_HEX, 0xFF, program_all_eeprom },
#endif
#ifdef CONFIG_SKIFF
    { "serial_number", PT_INT|PT_READONLY, PF_HEX, 0xFF, set_serial_number },
    { "system_rev", PT_INT|PT_READONLY, PF_HEX, 0x01, set_system_rev },
    { "linuxargs", PT_STRING, PF_STRING, (long)" root=/dev/ram initrd ramdisk_size=8192", NULL },
#else
    { "linuxargs", PT_STRING, PF_STRING, (long)" noinitrd root=/dev/mtdblock1 init=/linuxrc console=ttySA0", NULL },
#endif
#ifdef CONFIG_LOAD_KERNEL
    { "kernel_partition", PT_STRING, PF_STRING, (long)"root", NULL },
    { "kernel_filename", PT_STRING, PF_STRING, (long)"boot/zImage", NULL },
    { "ptable_addr", PT_INT, PF_HEX, 0xC0004242, NULL },
#endif
    { "hostname", PT_STRING, PF_STRING, 0, NULL },
    { "domainname", PT_STRING, PF_STRING, 0, NULL },
    { "ipaddr", PT_STRING, PF_STRING, 0, NULL },
    { "gateway", PT_STRING, PF_STRING, 0, NULL },
    { "netmask", PT_STRING, PF_STRING, 0, NULL },
    { "dns1", PT_STRING, PF_STRING, 0, NULL },
    { "dns2", PT_STRING, PF_STRING, 0, NULL },
    { "netcfg", PT_STRING, PF_STRING, 0, NULL }, /* manual/dhcp/... */
    { "nfs_server_address", PT_STRING, PF_STRING, 0, NULL },
    { "nfsroot", PT_STRING, PF_STRING, 0, NULL },
    { "xmodem", PT_INT, PF_HEX, 1, NULL },
    { "verbose", PT_INT, PF_HEX, 0, NULL },
    { "cmdex", PT_INT, PF_HEX, 1, update_cmdex },
    
    /* for formatting JFFS2 filesystem sectors */
    { "jffs2_sector_marker0", PT_INT, PF_HEX, 0x20031985, NULL },
    { "jffs2_sector_marker1", PT_INT, PF_HEX, 0x0000000C, NULL },
    { "jffs2_sector_marker2", PT_INT, PF_HEX, 0xE41EB0B1, NULL },

    /*
     * button macros.  the value is a string to be executed by the main
     * command processor
     */
    { "recb_cmd", PT_STRING, PF_STRING, 0, NULL},
    { "calb_cmd", PT_STRING, PF_STRING, (long)"ser_con", NULL},
    { "conb_cmd", PT_STRING, PF_STRING, (long)"irda_con", NULL},
    { "qb_cmd", PT_STRING, PF_STRING, (long)"boot", NULL},
    { "startb_cmd", PT_STRING, PF_STRING, 0, NULL},
    { "upb_cmd", PT_STRING, PF_STRING, 0, NULL},
    { "rightb_cmd", PT_STRING, PF_STRING, 0, NULL},
    { "leftb_cmd", PT_STRING, PF_STRING, 0, NULL},
    { "downb_cmd", PT_STRING, PF_STRING, 0, NULL},

    /* suppress the splash screen */
    { "suppress_splash", PT_STRING, PF_STRING, 0, NULL },

    /* send one or repeated NAKs when starting xmodem xfer */
    { "xmodem_one_nak", PT_INT, PF_HEX, 0, NULL },
    { "xmodem_initial_timeout", PT_INT, PF_DECIMAL, 800000, NULL },
    { "xmodem_timeout", PT_INT, PF_DECIMAL, 100000, NULL },

    /* reboot after this interval */
    { "autoboot_timeout", PT_INT, PF_DECIMAL, TIMEOUT, update_autoboot_timeout },

    /* end sentinel */
    { NULL, PT_NONE, PF_HEX, 0, NULL }
};

struct bootblk_param *get_param(const char *name)
{
   struct bootblk_param *param = bootldr_params;
   while (param->name != NULL) {
      int namelen = strlen(param->name);
      if (strncmp(name, param->name, namelen) == 0
          && name[namelen] == 0) {
         return param;
      }
      param++;
   }
   putstr("get_param: could not find parameter "); putstr(name); putstr("\r\n");
   return NULL;
}

int get_param_value(const char *name, void *value_p)
{
   struct bootblk_param *param = get_param(name);
   if (param != NULL) {
      if (value_p != NULL) 
         *(long *)value_p = param->value;
      return 0;
   } else {
      return -1;
   }  
}


/*
 * this variable MUST be initialized explicitly to zero to prevent its
 * placement in the BSS.
 */
enum {
    mmue_enabled = 99,
    mmue_notenabled = 101
};

byte mmuEnabled = mmue_notenabled; /* we set this after we call enableMMU() */
#define isMmuEnabled()	(mmuEnabled == mmue_enabled)
#define setMmuEnabled()	(mmuEnabled =  mmue_enabled)

#ifdef NoLibC
/*
 * very simple memcpy to blast bits around from place to place
 */
void *memcpy(char *dst, const char *src, long n)
{
  if (   (((dword)dst)&3) == 0
	 && (((dword)src)&3) == 0
	 && (n&3) == 0 ) {
    unsigned long *ca,*cb;
    for (ca = (unsigned long *)dst, cb = (unsigned long *)src; n > 0;n -= 4)
      *ca++ = *cb++;
  } else {
    byte *ca,*cb;
    for (ca = (byte *)dst, cb = (byte *)src; n-- > 0;)
      *ca++ = *cb++;
  }
  return dst;
}


/*
 * memset routine: fills first n bytes of dst with value c
 */
void memset(void *dst, char c, unsigned long n)
{
  byte *ca;
  for (ca = (byte *)dst; n-- > 0; )
    *ca++ = c;
}

int strlen(const char *s)
{
   int l = 0;
   while (*s++ != 0)
      l++;
   return l;
}

int isalpha(const char c)
{
    if (((c >= 'a') && (c <= 'z')) ||
	((c >= 'A') && (c <= 'Z')))
	return 1;
    else
	return 0;
    
}

int isalpha(const char c)
{
    if (((c == ' ') || (c == '\t')))
	return 1;
    else
	return 0;
    
}


int isdigit(const char c)
{
    if (((c >= '0') && (c <= '9')))
	return 1;
    else
	return 0;
    
}

int isalnum(const char c)
{
    return (isdigit(c) || isalpha(c));
    
    
}



char *strcat(char *dest, const char *src)
{
   char c;
   int len = strlen(dest);
   dest += len;
   while (( *dest++ = *src++) != 0) {
   }
}

int strncmp(const char *s1, const char *s2, int len)
{
   int i = 0;
   int c1, c2;
   if (s1 == NULL || s2 == NULL)
      return -1;
   for (c1 = s1[i], c2 = s2[i]; i < len; c1 = s1[i], c2 = s2[i], i++) {
      if (c1 < c2)
         return -1;
      else if (c1 > c2)
         return 1;
      else
         continue;
   }
   return 0;
}

int strcmp(const char *s1, const char *s2)
{
   int l1 = strlen(s1);
   int l2 = strlen(s2);
   int l = (l1 < l2) ? l1 : l2;
   int result = strncmp(s1, s2, l);
   if (l == 0 && (l1 != l2)) {
      if (l1 < l2)
         return -1;
      else
         return 1;
   } else {
      return result;
   }
}

int memmem(const char *haystack,int num1, const char *needle,int num2)
{
   int l1 = num1
   int l2 = num2;
   int l = (l1 < l2) ? l1 : l2;
   int i;
   
   
   for (i=0; i < (l1 - l2)+1;i++)
       if (!strncmp((char *)(haystack+i),needle,l2)){
	   //putLabeledWord("returning i = ",i);
	   return (int)(haystack+i);
       }
   
   return (int) NULL;

}


char *strchr(const char *s, int c)
{
   char sc;
   while (((sc = *s) != c) && sc != 0) {
     s++;
   }
   if (sc == c)
      return (char *)s;
   else
      return NULL;
}
#endif /* NoLibC */

#if defined(CONFIG_BITSY)
void
bitsy_putc(
    char    c,
    int	    encap_done)
{
    /*
     * silly person calling putc instead of putstr...
     * we encap the single char
     */
    
    if (!encap_done && packetize_output) {
	char	buf[2];

	buf[0] = c;
	buf[1] = '\0';

	putnstr(buf, 1);
    }
    else
	PrintChar(c, SerBase);
}
#endif

void putc(char c) 
{
    if (squelch_serial())
	return;

    
  /* wait for space in the TX FIFO */
#ifdef CONFIG_SKIFF
  while (CSR_READ_BYTE(UARTFLG_REG) & UART_TX_FIFO_BUSY);
  CSR_WRITE_BYTE(UARTDR_REG,c);
#elif defined(CONFIG_BITSY)
  bitsy_putc(c, 0);
#elif defined(CONFIG_NEPONSET) || defined(CONFIG_JORNADA720)
  while (!((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_TNF)); /* wait until TX FIFO not full */
  *(byte *)SA1100_UART3_UTDR = c;
#elif defined(CONFIG_ASSABET)
  while (!((*(volatile long *)SA1100_UART1_UTSR1) & SA1100_UTSR1_TNF));
  *(byte *)SA1100_UART1_UTDR = c;
#else
#error no definition for putc for this architecture
#endif
}

void
let_uart_drain(
    unsigned long status_addr)
{
    while (CTL_REG_READ(status_addr) & SA1100_UTSR1_TBY)
	;
}

do_putnstr(
    const char* str,
    size_t	n)
{
   while (n && *str != '\0') {
#if defined(CONFIG_BITSY)
         bitsy_putc(*str, 1);
#elif defined(CONFIG_ASSABET) || defined(CONFIG_JORNADA720)
         PrintChar(*str, SerBase) ;
#else
         putc(*str) ;
#endif
      str++ ;
      n--;
   }
}

void
putnstr(
    const char	*str,
    size_t	n)
{
   extern void *SerBase;

   if (str == NULL)
      return;
   
#if defined(CONFIG_BITSY)
   if (packetize_output) {
       int  len;
       char len_str[16];

       len = strlen(str);
       if (n < len)
	   len = n;

       /* add the msg header */
       dwordtodecimal(len_str, len);
       do_putnstr("MSG/", 4);
       len = strlen(len_str);
       len_str[len] = '/';
       ++len;
       len_str[len] = '\0';
       do_putnstr(len_str, len);
   }
#endif   

   do_putnstr(str, n);
}

void putstr(const char *str)
{
    putnstr(str, strlen(str));
}

void
putstr_sync(
    char*   s)
{
    putstr(s);
    let_uart_drain(SA1100_UART3_UTSR1);
}

#ifdef ZBSS
/*
 * zero bss (using no bss vars!)
 */
void
zbss(void)
{
    char*	    start;
    unsigned	    len;
    aout_header*    hdrp;
    
    hdrp = (aout_header*)FLASH_BASE;
    start = (char*)hdrp + hdrp->a_text + hdrp->a_data;
    
    len = hdrp->a_bss;
    if (len) {
	memset(start, 0x00, len);
    }
    
}
#endif /* ZBSS */

extern char HEX_TO_ASCII_TABLE[16];

void dwordtodecimal(char *buf, unsigned long x)
{
  int i = 0;
  int j = 0;
  char localbuf[16];

  if (x != 0) {
    while (x > 0) {
      unsigned long rem = x % 10;
      localbuf[i++] = HEX_TO_ASCII_TABLE[rem];
      x /= 10;
    }
    /* now reverse the characters into buf */
    while (i > 0) {
      i--;
      buf[j++] = localbuf[i];
    }
    buf[j] = '\0';
  } else {
    buf[0] = '0';
    buf[1] = '\0';
  }
}

void binarytohex(char *buf, long x, int nbytes)
{
  int i;
  int s = 4*(2*nbytes - 1);
  if (HEX_TO_ASCII_TABLE[0] != '0')
     putstr("HEX_TO_ASCII_TABLE corrupted\n");
  for (i = 0; i < 2*nbytes; i++) {
    buf[i] = HEX_TO_ASCII_TABLE[(x >> s) & 0xf];
    s -= 4;
  }
  buf[2*nbytes] = 0;
}

byte strtoul_err;
unsigned long strtoul(const char *str, char **endptr, int requestedbase)
{
   unsigned long num = 0;
   int i;
   char c;
   byte digit;
   int base = 10;
   int nchars = 0;
   int leadingZero = 0;

   strtoul_err = 0;

   while ((c = *str) != 0) {
      if (nchars == 0 && c == '0') {
         leadingZero = 1;
         if (0) putLabeledWord("strtoul: leadingZero nchars=", nchars);
         goto step;
      } else if (leadingZero && nchars == 1) {
         if (c == 'x') {
            base = 16;
            if (0) putLabeledWord("strtoul: base16 nchars=", nchars);
            goto step;
         } else if (c = 'o') {
            base = 8;
            if (0) putLabeledWord("strtoul: base8 nchars=", nchars);
            goto step;
         }
      }
      if (0) putLabeledWord("strtoul: c=", c);
      if (c >= '0' && c <= '9') {
         digit = c - '0';
      } else if (c >= 'a' && c <= 'z') {
         digit = c - 'a' + 10;
      } else if (c >= 'A' && c <= 'Z') {
         digit = c - 'A' + 10;
      } else {
         strtoul_err = 3;
         return 0;
      }
      if (digit >= base) {
         strtoul_err = 4;
         return 0;
      }
      num *= base;
      num += digit;
   step:
      str++;
      nchars++;
   }
   return num;
}

void putLabeledWord(const char *msg, unsigned long value)
{
   char buf[9];
   binarytohex(buf,value,4);
   putstr(msg);
   putstr(buf);
   putstr("\r\n");
}

void putLabeledAddrWord(const char *msg, unsigned long *value)
{
   char buf[9];

   putstr(msg);
   binarytohex(buf, (unsigned long) value, 4);
   putstr(" *0x" );
   putstr(buf);
   putstr( " == " );
   binarytohex(buf,*value,4);
   putstr(buf);
   putstr("\r\n");
}


void putHexInt32(unsigned long value)
{
  char buf[9];
  binarytohex(buf,value,4);
  putstr(buf);
}

void putHexInt16(word value)
{
  char buf[9];
  binarytohex(buf,value,2);
  putstr(buf);
}

void putHexInt8(byte value)
{
  char buf[3];
  binarytohex(buf,value,1);
  putstr(buf);
}
  
static struct ebsaboot bootinfo = {
   BT_MAGIC_NUMBER,	/* boot info magic number */
   0x20000000,		/* virtual addr of arg page */
   0x20000000,		/* physical addr of arg page */
   NULL,		/* kernel args string pointer */
   (pd_entry_t *)MMU_TABLE_START,	/* active L1 page table */
   0,			/* start of physical memory */
   DRAM_SIZE,		/* end of physical memory */
   SZ_2M+SZ_1M,         /* start of avail phys memory */
   48000000,		/* fclk frequency */
   UART_BAUD_RATE,    	/* baudrate of serial console */
   1 			/* n stop bits */
};

#define	VER_SEP	    "-"

void print_bootldr_version2(
    char*   prefix)
{
  char vbuf[32];

  if (prefix)
      putstr(prefix);
  putstr(bootprog_name);
  putstr(", Rev ");
  dwordtodecimal(vbuf, VERSION_MAJOR); putstr(vbuf); putstr(VER_SEP);
  dwordtodecimal(vbuf, VERSION_MINOR); putstr(vbuf); putstr(VER_SEP);
  dwordtodecimal(vbuf, VERSION_MICRO); putstr(vbuf);
#ifdef CONFIG_BIG_KERNEL
  putstr(" [BIG_KERNEL]");
#endif
  putstr(" [MONO]");
  putstr("\r\n");
}

void print_bootldr_version(void)
{
    print_bootldr_version2(">> ");
}

void
print_version_verbose(
    char*   prefix)
{
    print_bootldr_version2(prefix);

#if 0
    /*
     * last link date is more useful and is set even if bootldr.c is not
     * recompiled.
     */
    
    if (prefix)
	putstr(prefix);
    putstr(bootprog_date);
    putstr("\n\r");
#endif
    
#ifdef CONFIG_USE_DATE_CODE
    if (prefix)
	putstr(prefix);
    putstr("Last link date: ");
    putstr(_bootldr_date_code);
    putstr("\n\r");
#endif
    if (prefix)
	putstr(prefix);

    putstr("Contact: bootldr@handhelds.org\r\n");
}

void print_banner(void)
{
  long armrev = 0;
  long corelogicrev = 0;
  long cpsr = 0;
  __asm__("mrc p15, 0, %0, c0, c0, 0" : "=r" (armrev));
  __asm__("mrs %0, cpsr" : "=r" (cpsr));
#ifdef CONFIG_SKIFF
  corelogicrev = *(volatile unsigned long *)(DC21285_ARMCSR_BASE + PCI_VENDOR_ID);
#endif

  print_version_verbose(">> ");
  
  putstr("\r\n");
  putLabeledWord(">>  ARM Processor Rev=", armrev);
#if CONFIG_SKIFF
  putLabeledWord(">>  Corelogic Rev=", corelogicrev);
  if (get_param("mem_fclk_21285") != NULL)
    putLabeledWord(">>  Corelogic fclk=", get_param("mem_fclk_21285")->value);
#endif

  putstr(">> (c) 2000 Compaq Cambridge Research Laboratory\r\n");
  putstr("Press Return to start the OS now, any other key for monitor menu\r\n");
}

void
bootldr_reset(
    void)
{
    clr_sleep_reset();
    putstr_sync("Rebooting...");
    CTL_REG_WRITE(SA1100_RSRR, RSRR_SWR);
}
    
void
button_check(
    void)
{
    static int	action_button_was_pressed = 0;
    static int	sleep_button_was_pressed = 0;
    unsigned long   gpio_bits;

    gpio_bits = SA1100_GPIO_GPLR_READ();
    
    if (action_button_was_pressed &&
	(gpio_bits & (1<<18)) != 0) {
	action_button_was_pressed = 0;
	if (reboot_button_is_enabled()) {
	    bootldr_reset();
	}
    }

    if (sleep_button_was_pressed &&
	(gpio_bits & (1<<0)) != 0) {
	sleep_button_was_pressed = 0;
	putstr_sync("Snoozing...");
	bootldr_goto_sleep(NULL);
    }
    
    if ((gpio_bits & (1<<18)) == 0) {
	action_button_was_pressed = 1;	
    }
    if ((gpio_bits & (1<<0)) == 0) {
	sleep_button_was_pressed = 1;	
    }

#if defined(CONFIG_BITSY)
    if (check_for_func_buttons) {
	auxm_serial_check();
    }
#endif
}
    
#ifdef CONFIG_SKIFF
#define	SERIAL_CHAR_READY() (!(CSR_READ_BYTE(UARTFLG_REG) & UART_RX_FIFO_EMPTY))
   /* must read UARTDR_REG before RXSTAT_REG */
#define	SERIAL_READ_CHAR()	CSR_READ_BYTE(UARTDR_REG)
#define	SERIAL_READ_STATUS()	CSR_READ_BYTE(RXSTAT_REG)
#elif defined(CONFIG_BITSY) || defined(CONFIG_NEPONSET) || defined(CONFIG_JORNADA720)
#define	SERIAL_CHAR_READY() ((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_RNE)
   /* must read UARTDR_REG before RXSTAT_REG */
#define	SERIAL_READ_CHAR()	(*(volatile byte *)SA1100_UART3_UTDR)
#define	SERIAL_READ_STATUS()	((*(volatile byte *)SA1100_UART3_UTSR1) & SA1100_UTSR1_ERROR_MASK)
#elif defined(CONFIG_ASSABET)
#define	SERIAL_CHAR_READY() ((*(volatile long *)SA1100_UART1_UTSR1) & SA1100_UTSR1_RNE)
   /* must read UARTDR_REG before RXSTAT_REG */
#define	SERIAL_READ_CHAR()	(*(volatile byte *)SA1100_UART1_UTDR)
#define	SERIAL_READ_STATUS()	((*(volatile byte *)SA1100_UART1_UTSR1) & SA1100_UTSR1_ERROR_MASK)
#else
#error no architecture defined for getc
#endif

int	pushed_chars_len = 0;
int	pushed_chars_idx = 0;
char	pushed_chars[128];


int
push_cmd_chars(
    char*   chars,
    int	    len)
{
    if (pushed_chars_len + len > sizeof(pushed_chars))
	return (-1);

    memcpy(&pushed_chars[pushed_chars_len], chars, len);
    pushed_chars_len += len;
#if 0
    putLabeledWord("pcc, len: 0x", pushed_chars_len);
    putLabeledWord("pcc, idx: 0x", pushed_chars_idx);
#endif
}

int
get_pushed_char(
    void)
{
    int	    ret;

#if 0
    putLabeledWord("gpc, len: 0x", pushed_chars_len);
    putLabeledWord("gpc, idx: 0x", pushed_chars_idx);
#endif
    
    if (pushed_chars_idx >= pushed_chars_len) {
	pushed_chars_idx = 0;
	pushed_chars_len = 0;
	ret = -1;
    }
    else
	ret = pushed_chars[pushed_chars_idx++] & 0xff;

    /*  putLabeledWord("gpc, ret: 0x", ret); */
    
    return(ret);
}

int getc_verbose_errors = 1;
int getc_errno = 0;
byte do_getc(
    vfuncp	    idler,
    unsigned long   timeout,
    int*	    statp)
{
   byte c, rxstat;
   int	do_timeout = timeout != 0;
   int	ch;

   getc_errno = 0; /* reset errno */

   while (!SERIAL_CHAR_READY()) {
       if ((ch = get_pushed_char()) != -1)
	   return (ch);
   
       if (do_timeout) {
	   if (!timeout)
	       break;
	   timeout--;
       }
       
       if (idler)
	   idler();
   }

   /* give priority to pushed chars */
   if ((ch = get_pushed_char()) != -1)
       return (ch);
   
   if (do_timeout && timeout == 0) {
       c = 0;
       rxstat = -1;
   }
   else {
       c = SERIAL_READ_CHAR();
       rxstat = SERIAL_READ_STATUS();
   }
   
   if (rxstat) {
      getc_errno = rxstat;
      if (getc_verbose_errors && !statp) {
         putLabeledWord("RXSTAT error: ", rxstat);
      }
      if (statp)
	  *statp = rxstat;
   }
   return(c);
}

byte
getc(
    void)
{
    return (do_getc(button_check, 0, NULL));
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay iterations checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte
awaitkey(
    unsigned long   delay,
    int*	    error_p)
{
    return (do_getc(button_check, delay, error_p));
}

#define CMDBUFSIZE 256
char cmdBuffer[CMDBUFSIZE];
void getcmd(void) {
   byte curpos = 0; /* curpos position - index into cmdBuffer */
   byte c;
   byte noreturn = 1;

   /* first clear out the buffer.. */
   memset(cmdBuffer, 0, CMDBUFSIZE);

   /* here we go..*/

   while (noreturn) {
      c = getc();
      switch (c)
         {
	 case 0x08:
         case 0x06:
         case 0x07:
         case 0x7E:
         case 0x7F: /* backspace / delete */
            if (curpos) { /* we're not at the beginning of the line */
               curpos--;
               putc(0x08); /* go backwards.. */
               putc(' ');  /* overwrite the char */
               putc(0x08); /* go back again */
            }
            cmdBuffer[curpos] = '\0';
            break;
         case '\r':
         case '\n':
         case '\0':
            noreturn = 0;
            putc('\r');
            putc('\n');
            break;

         case CTL_CH('x'):
	    curpos = 0;
	    break;
	     
         default:
            if (curpos < CMDBUFSIZE) {
               cmdBuffer[curpos] = c;
               /* echo it back out to the screen */
	       if (!no_cmd_echo)
		   putc(c);
               curpos++;
            }
            break;
         }
   }
   /*
     putstr("COMMAND: ");
     putstr(cmdBuffer);
     for (c=0;c<CMDBUFSIZE;c++) {
       putHexInt8(cmdBuffer[c]);
       putc(' ');
     }
     putstr("\r\n");
   */

}

int argc;

enum ParseState {
   PS_WHITESPACE,
   PS_TOKEN,
   PS_STRING,
   PS_ESCAPE
};

enum ParseState stackedState;

void parseargs(char *argstr, int *argc_p, char **argv, char** resid)
{
  const char *whitespace = " \t";
  int argc = 0;
  char c;
  enum ParseState lastState = PS_WHITESPACE;
   
  /* tokenize the argstr */
  while ((c = *argstr) != 0) {
    enum ParseState newState;

    if (c == ';' && lastState != PS_STRING && lastState != PS_ESCAPE)
	break;
    
    if (lastState == PS_ESCAPE) {
      newState = stackedState;
    } else if (lastState == PS_STRING) {
      if (c == '"') {
        newState = PS_WHITESPACE;
        *argstr = 0;
      } else {
        newState = PS_STRING;
      }
    } else if ((c == ' ') || (c == '\t')) {
      /* whitespace character */
      *argstr = 0;
      newState = PS_WHITESPACE;
    } else if (c == '"') {
      newState = PS_STRING;
      *argstr++ = 0;
      argv[argc++] = argstr;
    } else if (c == '\\') {
      stackedState = lastState;
      newState = PS_ESCAPE;
    } else {
      /* token */
      if (lastState == PS_WHITESPACE) {
        argv[argc++] = argstr;
      }      
      newState = PS_TOKEN;
    }

    lastState = newState;
    argstr++;
  }

  if (0) { 
    int i;
    putLabeledWord("parseargs: argc=", argc);
    for (i = 0; i < argc; i++) {
      putstr("   ");
      putstr(argv[i]);
      putstr("\r\n");
    }
  }

  argv[argc] = NULL;
  if (argc_p != NULL)
    *argc_p = argc;

  if (*argstr == ';') {
      *argstr++ = '\0';
  }
  *resid = argstr;
}



// this is getting more compliacated,  this function will averride any of the
// args in argstr with tthe args from argv.  this will allow you to override the
// param linuxargs from the commandline. e.g. init=/myRC will override
// init=linuxRC from the params.
void unparseargs(char *argstr, int argc, const char **argv)
{
   int i;
   char *cutStart;
   char *cutEnd;
   char *paramEnd;
   int delta;   
   int j;
   
   
   for (i = 0; i < argc; i++) {
       if (argv[i] != NULL) {
	   if ((paramEnd = (char *) strchr(argv[i],'=')))// we have an a=b arg
	   {
	       paramEnd++;
	       putstr("haystack = <");
	       putnstr(argstr,strlen(argstr));
	       putstr(">\r\n");
   
	       putstr("needle = <");
	       putnstr(argv[i],(int)(paramEnd - argv[i]));
	       putstr(">\r\n");

	       if ((cutStart = (char *)memmem(argstr,strlen(argstr),argv[i],(int)(paramEnd - argv[i]))) != NULL){
		   // found a match
		   if (!(cutEnd = strchr(cutStart,' ')))
		       cutEnd = strlen(argstr);
		   else
		       cutEnd++; // skip the space
		   
#if 0
		   putstr("found a match\r\n");
		   putLabeledWord("argstr = ",(unsigned long) argstr);
		   putLabeledWord("cutStart = ",(unsigned long) cutStart);
		   putLabeledWord("cutEnd = ",(unsigned long) cutEnd);
		   
		   putstr("argstr: <");putstr(argstr);putstr(">\r\n");
		   putstr("paramEnd: <");putstr(paramEnd);putstr(">\r\n");
		   putstr("cutStart: <");putstr(cutStart);putstr(">\r\n");
		   putstr("cutEnd: <");putstr(cutEnd);putstr(">\r\n");
#endif
		   delta = (int)(cutEnd - cutStart);		   
		   for (j=(int) (cutEnd - argstr);j < strlen(argstr); j++) 
		       argstr[j-delta] = argstr[j];
		   // new end of argstr
		   argstr[strlen(argstr) - delta] = '\0';
		   
	       }
	   }	   
#if 0
	   putstr("pre strcat argstr = <");
	   putstr(argstr);
	   putstr(">\r\n");	   
#endif
	   strcat(argstr, " ");
	   strcat(argstr, argv[i]);
#if 0
	   putstr("post strcat argstr = <");
	   putstr(argstr);
	   putstr(">\r\n");	   
#endif
     }
   }
}

int parsecmd(struct bootblk_command *cmdlist, int argc, const char **argv)
{
   /* find the command name */
   const char *cmdname = argv[0];
   int cmdnum = 0;

   if (argc < 1)
      return -1;
   /* do a string compare for the first offset characters of cmdstr
      against each member of the cmdlist */
   while (cmdlist[cmdnum].cmdstr != NULL) {
      if (strcmp(cmdlist[cmdnum].cmdstr, cmdname) == 0)
         return(cmdnum);
     cmdnum++;
   }
   return(-1);
}

void execcmd(struct bootblk_command *cmdlist, int argc, const char **argv)
{
   int cmdnum = parsecmd(cmdlist, argc, argv);

   if (0) {
      putstr("execcmd: argv[0]="); putstr(argv[0]); putstr("\r\n");
      putLabeledWord("  argc=", argc);
      putLabeledWord("  cmdnum=", cmdnum);
   }
   if (cmdnum >= 0) {
      (*cmdlist[cmdnum].cmdfunc)(argc, argv);
   } else if (strcmp(argv[1], "help") == 0) {
      print_help_on_commands(cmdlist);
   } else {
#if 1
      putstr("Don't understand command>");
      putstr(argv[0]);
      putstr("<\n\r");
      print_help_on_commands(cmdlist);
#else
      /* default to the first command and insert it into the head of argv */
      int i;
      for (i = argc; i >= 0; i--) {
         argv[i+1] = argv[i];
      }
      argc++;
      argv[0] = cmdlist[0].cmdstr;
      (*cmdlist[0].cmdfunc)(argc, argv);
#endif
   }
}

void
exec_string(
    char*   buf)
{
  int argc;
  char *argv[128];
  char*	resid;

  while (*buf) {
      memset(argv, 0, sizeof(argv));
      parseargs(buf, &argc, argv, &resid);
      if (argc > 0) {
	  execcmd(commands, argc, (const char **)argv);
      } else {
	  print_help_on_commands(commands);
      }
      buf = resid;
  }
}

void bootmenu(void)
{
  clr_squelch_serial();

  if (get_param_value("cmdex", (void *)&use_extended_getcmd) == -1)
       use_extended_getcmd = 1;

  while (1) {
    if (!packetize_output)
	putstr("boot> ");
    if (use_extended_getcmd)
	getcmd_ex(cmdBuffer, CMDBUFSIZE);
    else
	getcmd();
    
    if (cmdBuffer[0]) {
        /*
	 * useful if calling a modified bootldr's bootmenu
	 * function from another place.
	 */
	if (strcmp("quit", cmdBuffer) == 0)
	    return;
	
	exec_string(cmdBuffer);
    }
    
    if (ack_commands)
	do_putnstr("ACK0\n", 5);
  }
}

#define PARAM_PREFIX ("bootldr:")
#define DEBUG_COMMAND_PARAMS_EVAL 1 

void params_eval(
    const char*   prefix_in,
    int	    just_show)
{
  unsigned char* p;
  unsigned char* endp;
  const char* prefix;
  int   cmdnum;
  char cmdbuf[1024];
  char* cmdp;
  int my_argc;
  char *my_argv[128];
  struct BootldrFlashPartitionTable *pt = 0;
  int partition_table_size = 0;
  int i;
  unsigned long ptable_addr = 0;

 
  putstr("\r\nparams_eval: prefix_in =");
  putstr(prefix_in);
  putLabeledWord("\r\nparams_eval: just_show =",just_show);

  // XXX this presumes that the params sector follows the bootloader sector!!
  p = ((char*)flashword) + flashDescriptor->bootldr.base + flashDescriptor->bootldr.size;
  

  pt = (struct BootldrFlashPartitionTable *)p;
  if (pt->magic == BOOTLDR_PARTITION_MAGIC) {
    params_eval_partition(prefix_in,just_show);
  }else if (isWincePartition(p)) {
  //else if ((pt->magic & 0xFFFF0000) == 0xEA000000) {
    /* guess that wince is still installed and do not look for params */
  } else {
      params_eval_file(just_show);
  }
  // now we have our new params, lets copy them to the right place.
  get_param_value("ptable_addr", &ptable_addr);
  partition_table_size = (sizeof(struct BootldrFlashPartitionTable) 
			  + partition_table->npartitions*sizeof(struct FlashRegion));

  //memcpy(ptable_addr,partition_table,partition_table_size);
  
}

/* pretty simple, really.  we just need to
* 1) load the /boot/params file to the 0xC0008000 space (this should be more than large enough)
* 2) then go through and build up each line and parse it
*/

#define TMP_PARTITION_NAME "throwaway_bootldr_partition"
void params_eval_file(
    int just_show)
{

    long kernel_in_ram = 0xC0008000;
    long size = 0;
    struct part_info part;
    struct FlashRegion *tmpPart;
    unsigned long bootldr_size  ;
    unsigned long bootldr_end ;
    
    
    bootldr_size = flashDescriptor->bootldr.size;
    bootldr_end = flashDescriptor->bootldr.base + bootldr_size;
    
    // given that there is no params sector, we need a temporary partition
    // covering the rest of flash to make a file system out of.
    btflash_define_partition(TMP_PARTITION_NAME, bootldr_end,
			     0x0,
			     LFR_EXPAND | LFR_JFFS2);
    
    tmpPart = btflash_get_partition(TMP_PARTITION_NAME);

    part.offset = ((char *) flashword) + tmpPart->base;
    part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
    part.size = tmpPart->size;
    
    putLabeledWord("params_eval_file: tmpPart->base  =",tmpPart->base);
    putLabeledWord("params_eval_file: tmpPart->size  =",tmpPart->size);

    
    
    size = jffs2_1pass_load(kernel_in_ram,&part,"/boot/params");

    putLabeledWord("pef: loaded a file of size ",size);
    
    
    if (size > 0)
	parseParamFile(kernel_in_ram,size,just_show);
    else
	putstr("failed to load params file /boot/params ...\r\n");


    // cleanup
    btflash_delete_partition(TMP_PARTITION_NAME);
}

void params_eval_partition(
    const char*   prefix_in,
    int	    just_show)
{
  unsigned char* p;
  unsigned char* endp;
  const char* prefix;
  int   cmdnum;
  char cmdbuf[1024];
  char* cmdp;
  int my_argc;
  char *my_argv[128];
  struct BootldrFlashPartitionTable *pt = 0;
  int partition_table_size = 0;
  int i;
  
  putstr("eval param blk\r\n");
  // presumes that the params sector follows the bootloader sector and is no larger!
  p = ((char*)flashword) + flashDescriptor->bootldr.base + flashDescriptor->bootldr.size;
  endp = p + flashDescriptor->bootldr.size;

  pt = (struct BootldrFlashPartitionTable *)p;
  if (pt->magic == BOOTLDR_PARTITION_MAGIC) {
    putstr("found partition table in params sector\r\n");
    putLabeledWord("pt->npartitions=", pt->npartitions);
    partition_table_size = 
      (sizeof(struct BootldrFlashPartitionTable) 
       + pt->npartitions * sizeof(struct FlashRegion));
    putLabeledWord("partition_table_size=", partition_table_size);
    for (i = 0; i < pt->npartitions; i++) {
      struct FlashRegion *partition = &pt->partition[i];
      btflash_define_partition(partition->name, partition->base, partition->size, partition->flags);
    }
  }
  p += partition_table_size;

  /* stops at end of sector or first unwritten byte */
  while (p < endp && *p != 0xff && *p != 0) {
#ifdef DEBUG_COMMAND_PARAMS_EVAL
    putLabeledWord("top of loop, p=", (dword)p);
#endif
    prefix = prefix_in;
    while (p < endp && *prefix && *p == *prefix) {
      p++;
      prefix++;
    }
#ifdef DEBUG_COMMAND_PARAMS_EVAL
    putLabeledWord("past prefix check, p=", (dword)p);
#endif

    if (*prefix != '\0') {
#ifdef DEBUG_COMMAND_PARAMS_EVAL
      putstr("no prefix match\r\n");
#endif

      skip_to_eol: 
      /* skip to end of line */
      while (p < endp && *p != '\n')
	p++;

#ifdef DEBUG_COMMAND_PARAMS_EVAL
      putLabeledWord("skipped line, p=", (dword)p);
      putLabeledWord("char at p=", (dword)(*p) & 0xff);
#endif
      p++;
      continue;
    }

    /* copy line to buffer */
    /* terminate with eof, or eol */
    cmdp = cmdbuf;
    while (p < endp && *p != 0xff) {
      if (*p == '\r' || *p == '\n') {
	p++;
	if (*p == '\r' || *p == '\n') 
	  p++;
	break;
      }
      if ((cmdp - cmdbuf) >= sizeof(cmdbuf) - 1)
	  goto skip_to_eol;
      *cmdp++ = *p++;
    }
    *cmdp = '\0';
    cmdp = cmdbuf;

    putstr("+");
    putstr(cmdbuf);
    putstr("\r\n");

    if (just_show)
	continue;

    exec_string(cmdbuf);
    
#if 0
    memset(my_argv, 0, sizeof(my_argv));
    parseargs(cmdbuf, &my_argc, my_argv);
    execcmd(commands, my_argc, (const char **)my_argv);
#endif
  }
}


void command_params_eval(int argc, const char **argv)
{
  int	just_show = 0;
  const char*	prefix = PARAM_PREFIX;
    
  if (argc > 1) {
      just_show = (strncmp(argv[1], "-n", 2) == 2);
      if (just_show) {
	  if (argc > 2)
	      prefix = argv[2];
      }
      else
	  prefix = argv[1];
  }
  
  params_eval(prefix, just_show);
}

void command_partition(int argc, const char **argv)
{
  int i;
  if (argc > 1) {
      
    putstr("argv[1]="); putstr(argv[1]); putstr("\r\n");
    if (strcmp(argv[1], "show") == 0) {
      if (partition_table != NULL) {
        putLabeledWord("npartitions=", partition_table->npartitions);
        for (i = 0; i < partition_table->npartitions; i++) {
	    FlashRegion *partition = &partition_table->partition[i];
          putstr(&partition->name[0]); putstr("\r\n");
          putLabeledWord("  base: ", partition->base);
          putLabeledWord("  size: ", partition->size);
	  putLabeledWord("   end: ", partition->size + partition->base);
          putLabeledWord(" flags: ", partition->flags); 
        }
      }
    } else if (strcmp(argv[1], "save") == 0) {
	command_params_save(argc, argv);
    } else if (strcmp(argv[1], "delete") == 0) {
	if (argc == 3){
	    btflash_delete_partition(argv[2]);
	}
	else{
	    putstr("usage: partition delete part_name\r\n");
	}
	
  } else if (strcmp(argv[1], "reset") == 0) {
      btflash_reset_partitions();
    } else if (strcmp(argv[1], "define") == 0) {
      const char *namestr = argv[2];
      const char *basestr = argv[3];
      const char *sizestr = argv[4];
      const char *flagsstr = argv[5];
      char *name;
      unsigned long base, size, flags;
      
      if (argc < 6) {
        putstr("usage: partition define <name> <base> <size> <flags>\r\n");
        putstr("       flags  16 for JFFS2\r\n"); 
        return;
      }
      name = mmalloc(strlen(namestr)+1);
      if (name == NULL) {
        putstr("Error allocating space for name\r\n");
        return;
      }
      strcpy(name, namestr);

      base = strtoul(basestr, NULL, 0);
      size = strtoul(sizestr, NULL, 0);
      if (strcmp(flagsstr, "jffs2") == 0) {
        flags = LFR_JFFS2;
      } else { 
      flags = strtoul(flagsstr, NULL, 0);
      }

      if (base < 128000 && ((flags&LFR_PATCH_BOOTLDR) == 0)) {
        putstr("  bootldr sector must have flags set to 2\r\n"); 
        return;
      }

      btflash_define_partition(name, base, size, flags);
    } else {
      goto usage;
    }
  } else {
  usage:
    putstr("usage:\r\n"
           "  partition reset\r\n"
           "  partition show\r\n"
           "  partition define <name> <base> <size> <flags>\r\n"
           "            flags:   16 for JFFS2\r\n"
	   "  partition save\r\n"); 
  }
}

void
splash_file(char * fileName, char *partName)
{
    unsigned long size;
    unsigned long kernel_in_ram = 0xC0008000;
    const char *kernel_part_name = NULL;
    struct FlashRegion *kernelPartition;
    struct part_info part;
    
    // working space
    get_param_value("kernel_in_ram", &kernel_in_ram);   

    // partition to load file from
    if (partName){
	kernel_part_name = mmalloc(strlen(fileName)+1);
	strcpy(kernel_part_name,partName);
    }
    else{
	get_param_value("kernel_partition", &kernel_part_name);	
    }
    
    kernelPartition = btflash_get_partition(kernel_part_name);
    part.size = kernelPartition->size;
    /* for uniformly sized flash sectors */
    part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
    part.offset = ((char*)flashword) + kernelPartition->base;

    if (partName)
	mfree(kernel_part_name);
    
    if ((size = jffs2_1pass_load(kernel_in_ram,&part,fileName)) == 0){
	putstr("bad splash load for file: ");
	putstr(fileName);
	putstr("\r\n");
	return;
    }
    
    display_pbm(kernel_in_ram,size);
    

}

#if !defined (NO_SPLASH) && defined(CONFIG_BITSY)
#include "splashz.h"
void
splash()
{
    int		    rc;
    unsigned long   uncomplen;
    char*	    fb = lcd_get_image_buffer();
    

    /*  lcd_default_init(); */
    rc = uncompress(fb, &uncomplen, splash_zimg, sizeof(splash_zimg));
    if (rc != Z_OK) {
	putLabeledWord("splash: uncompress failed, rc: 0x", rc);
	return;
    }
    lcd_display_bitmap(fb, uncomplen, lcd_params, NULL);
}
#else
void
splash(
    void)
{
    putstr("splash code not compiled in.\n\r");
}
#endif

void
default_boot()
{
  while (1) {
     exec_string("boot");
     putstr("boot command failed, entering monitor\n\r");
     
     /* if default boot fails, drop into the bootmenu */
     bootmenu();
  }
}

int main(
    int	    argc,		/* how traditional! */
    char*   argv[])
{
  char	c;
  /*
   * in reality, we don't use argc and argv in the traditional sense.
   * If argc (the 0th arg) is 1 then we are to enter the bootmenu()
   * unconditionally.
   */
  
  int	call_bootmenu = argc != 0;
  long *p = DRAM_BASE0 + SZ_4K;
  
  

  /*
     At this stage, we assume:
        SDRAM is configured.
        UART is configured and ready to go.
        Initial PCI setup is done.
        Interrupts are OFF.
        MMU is DISABLED
  */

  CTL_REG_WRITE(SA1100_RTTR, 0x8000);        /* set up RTC divisor (--> ~1Hz) */
  auxm_init();			/* init the aux microcontroller i/f */
  

  /* to allow wakeup from linux, the bootloader, and wince we use a ram sig.
  *  we need to put the bootloader sig here so it can wake up properly.  Linux
  * takes care of its own.
  */
  
  *p = 0x4242; // the number is meaningless, wince is recognized by 0x0 here.
  
#ifdef DEBUG
  putstr("\r\nBOOT");
#endif

#ifdef CONFIG_PCI
  /* Configure the PCI devices */
#ifdef DEBUG
  putstr(" PCI");
#endif
  bootConfigurePCI();
#endif
#ifdef CONFIG_JORNADA720
  (*((volatile int *)SA1100_PPSR_REG)) &= ~(SA1100_PPDR_LFCLK | 0x80);
  (*((volatile int *)SA1100_PPDR_REG)) |= (SA1100_PPDR_LFCLK | 0x80);
#endif

  putLabeledWord("MMU table start=", (dword)mmu_table);
  putLabeledWord("Boot data start=", (dword)BOOT_DATA_START);
  putLabeledWord("Boot data size=", (dword)BOOT_DATA_SIZE);
  putLabeledWord("Stack data base=", (dword)STACK_BASE);
  putLabeledWord("Stack data size=", (dword)STACK_SIZE);
  
  /* 
   * set up MMU table:
   *  DRAM is cacheable
   *  cached and uncached images of Flash
   *  after MMU is enabled, bootldr is running in DRAM
   */
  bootConfigureMMU(); 
  writeBackDcache(CACHE_FLUSH_BASE);
  enableMMU();

  /* flashword logically should be assigned in bootConfigureMMU,
   * but we can't do it until the MMU is enabled
   * because before then it resides in flash 
   * -Jamey 9/4/2000
   */
  flashword = (unsigned long *)UNCACHED_FLASH_BASE;

  /* initialize the heap */
  mmalloc_init((unsigned char *)(HEAP_START), HEAP_SIZE);

  
  /* figure out what kind of flash we have */
  btflash_init();
  flashConfigureMMU(flash_size);

  {
     struct bootblk_param *system_rev_param = NULL;
     struct bootblk_param *dram_size_param = get_param("dram_size");
#ifdef CONFIG_SKIFF
     struct bootblk_param *mem_fclk_21285_param = get_param("mem_fclk_21285");
     struct bootblk_param *maclsbyte_param = get_param("maclsbyte");
     long system_rev = get_system_rev();
#endif
     long dram_size;
#ifdef CONFIG_SKIFF
     system_rev_param = get_param("system_rev");
     dram_size = ((system_rev&SYSTEM_REV_MAJOR_MASK) == SYSTEM_REV_SKIFF_V2) ? DRAM_SIZE : SZ_16M;
#endif
#if defined(CONFIG_BITSY) || defined(CONFIG_ASSABET) || defined(CONFIG_JORNADA720)
     /* assumes one bank of DRAM and that DRAM_BASE0 is not in the cache */
     *(long *)(DRAM_BASE0 + SZ_64M) = SZ_64M;
     *(long *)(DRAM_BASE0 + SZ_32M) = SZ_32M;
     *(long *)(DRAM_BASE0 + SZ_16M) = SZ_16M;
     dram_size = *(long *)DRAM_BASE0;
#endif /* Bitsy, Assabet, Jornada720 */
#ifdef CONFIG_SKIFF
#if   defined(MEMCLK_33MHZ)
#warning 33MHZ override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_33MHZ;
#elif defined(MEMCLK_48MHZ)
#warning 48MHz override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_48MHZ;
#elif defined(MEMCLK_60MHZ)
#warning 60MHZ override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_60MHZ;
#endif
     if (system_rev_param != NULL)
        system_rev_param->value = system_rev;
     putLabeledWord("system_rev: ", system_rev);
     switch (system_rev & SYSTEM_REV_MEMCLK_MASK) {
     case SYSTEM_REV_MEMCLK_60MHZ: 
        mem_fclk_21285_param->value = 60000000; break;
     case SYSTEM_REV_MEMCLK_48MHZ: 
        mem_fclk_21285_param->value = 48000000; break;
     case SYSTEM_REV_MEMCLK_33MHZ: 
     default:
        mem_fclk_21285_param->value = 33333333; break;
     }
     if (maclsbyte_param)
       maclsbyte_param->value = get_maclsbyte();

     putstr("memclk_hz: ");
     putstr(((system_rev & SYSTEM_REV_MEMCLK_MASK) == SYSTEM_REV_MEMCLK_33MHZ) 
            ? "33MHz\r\n" : "48MHz\r\n");
     /* this string should agree with the MAC in btpci.c */
     if (maclsbyte_param) {
       putstr("MAC=08:00:2b:00:01:"); putHexInt8(maclsbyte_param->value); putstr("\r\n");
     }
#endif /* CONFIG_SKIFF */
     if (dram_size_param != NULL)
        dram_size_param->value = dram_size;
     putLabeledWord("dram_size: ", dram_size);
     
     bootinfo.bt_memend = dram_size;
  }

#ifdef CONFIG_JORNADA720
  /* Take the MCU out of reset mode */
  (*((volatile int *)SA1100_PPSR_REG)) |= SA1100_PPDR_LFCLK;
#endif


  if (!jffs_init_1pass_list()){
      putstr("jffs_init_1pass_list failed to initialize\r\n");
  }



  /* print the opening banner */
  if (squelch_serial() || do_splash)
      splash();
  print_banner();

  if (squelch_serial()) {
      int suppress_splash = 0;
      /* normal boot, eval params */
      command_params_eval(0, NULL);
      get_param_value("suppress_splash", &suppress_splash);
      if (suppress_splash) {
         lcd_off();
      } else {
        lcd_on();
      }
  }
  else {
      putstr("DEBUG BOOT: not evaluating params\r\n");
      putstr("DEBUG BOOT: use `params eval' to evaluate parameters.\r\n");
  }

  if (get_param_value("autoboot_timeout", (void *)&autoboot_timeout) == -1)
      autoboot_timeout = TIMEOUT;

  /********************************************************************************
   * WARNING, IF YOU TURN ON THE D CACHE AND DONT CORRECTLY TURN IT OFF
   * DRAM GETS CORRUPTED, SINCE WE SEEM TO HANG WHEN WE FLUSH DCACHE, WE NEED TO
   * LOOK AT THIS MORE BEFORE RUNNING WITH DCACHE TURNED ON
   * 
   ********************************************************************************/  

  asmEnableCaches(0,1); // turn on icache
  
  if (autoboot_timeout == 0 || call_bootmenu)
      while (1) 
	  bootmenu(); /* does not return */
      
  /*
   * wait for a keystroke or a button press.
   * button presses cause commands to be run.
   */
  c = awaitkey(autoboot_timeout, NULL);
  if (((c != '\r') && (c != '\n') && (c != '\0'))) {
      putstr("type \"?\" or \"help\" for help.\r\n");
      bootmenu(); /* does not return */
  }

  default_boot();
}

void print_help_on_commands(struct bootblk_command *commands)
{
  int i = 0;
  while (commands[i].cmdstr != NULL) {
    putstr(commands[i].helpstr); putstr("\r\n");
    i++;
  }
}

/* no arguments */
void command_help(int argc, const char **argv)
{
  int i = 0;
  putstr("Available Commands:\r\n");
  if (argc == 2 && (strcmp(argv[1], "help") != 0)) {
     /* help <command>
      *   invoke <command> with 'help' as an argument
     */
     argv[0] = argv[1];
     argv[1] = "help";
     argv[2] = NULL;
     execcmd(commands, argc, argv);
     return;
  }
  print_help_on_commands(commands);
}


/* can have arguments or not */
void command_boot(int argc, const char **argv)
{   
    unsigned char* p;
    char *wp = "wince";
    
      
    if (argv[1] == NULL) {
      char *boot_type;
      get_param_value("boot_type", &boot_type);
      argv[1] = boot_type;
      argc++;
   }
    /* since we cant have a params table in wince, we need to check for the
     * wince partition here and force it
     */
    p = ((char*)flashword) + flashDescriptor->bootldr.base + flashDescriptor->bootldr.size;
    if (isWincePartition(p)){
	argv[1] = mmalloc(strlen(wp)+1);
	strcpy(argv[1],wp);
    }
    
   


   putstr("booting ");
   putstr(argv[1]);
   putstr("...\r\n");

   execcmd(boot_commands,argc-1,argv+1);
}

/* can have arguments or not */
void command_flash_type(int argc, const char **argv)
{
  if (argc == 1) {
     /* print the available flash types */
     btflash_print_types();
  } else {
    /* manually set the flash type */
     btflash_set_type(argv[1]);
  }
}

/* can have arguments or not */
void command_params(int argc, const char **argv)
{
   if (argc == 2) {
      if (strcmp(argv[1], "reset") == 0) {
         putstr("setting params to default values\r\n");
	 putLabeledWord("flashword = ",flashword);
	 putLabeledWord("bootldr_params = ",bootldr_params);
	 putLabeledWord("FLASH_BASE = ",FLASH_BASE);
	 putLabeledWord("sizeof(bootldr_params) = ",sizeof(bootldr_params));

         memcpy((char*)bootldr_params, (const char*)(((char*)flashword) + (((dword)bootldr_params) - FLASH_BASE)), sizeof(bootldr_params));
	 
         return;
      }
   } else if (argv[1] == NULL) {
      argv[1] = "show";
      argc++;
   }
   execcmd(params_commands,argc-1,argv+1);
}


#ifdef __linux__
void setup_linux_params(long bootimg_dest, long initrd_start, long initrd_size, 
			long memc_ctrl_reg, long mem_fclk_21285, long dram_size,
			const char *cmdline, int flashroot)
{
   struct param_struct *params = (struct param_struct *)(bootimg_dest + 0x100);
   int system_rev = get_param("system_rev")->value;
   long copy_ramdisk = 0;
   struct FlashRegion *ramdisk_region = btflash_get_partition("ramdisk");
   long first_word_of_ramdisk = (ramdisk_region != NULL) ? *(long *)(FLASH_BASE + ramdisk_region->base) : 0;
   int rootdev = 0x00ff;
   int using_ramdisk = flashroot;
   struct tag *tag;
   int i;
   
   get_param_value("copy_ramdisk", &copy_ramdisk);
   if (flashroot && (first_word_of_ramdisk == 0x28cd3d45)) {
     /* cramfs */
     copy_ramdisk = 0;
     rootdev = 0x3c02;
     using_ramdisk = 0;
   }
   if (!flashroot) {
     copy_ramdisk = 0;
   }

   // start with the core tag       
   tag = (struct tag *)(bootimg_dest + 0x100);
   
   putLabeledWord("Makeing core tag at ",(unsigned long) tag);
   
   tag->hdr.tag = ATAG_CORE;
   tag->hdr.size = tag_size(tag_core);
   tag->u.core.flags =0;
   tag->u.core.pagesize = LINUX_PAGE_SIZE;
   tag->u.core.rootdev = rootdev;
   tag = tag_next(tag);

   // now the cmdline tag
   putLabeledWord("Makeing cmdline tag at ",(unsigned long) tag);
   tag->hdr.tag = ATAG_CMDLINE;
   // must be at least +3!! 1 for the null and 2 for the ???
   tag->hdr.size = (strlen(cmdline) + 3 + sizeof(struct tag_header)) >> 2;
   //tag->hdr.size = (strlen(cmdline) + 10 + sizeof(struct tag_header)) >> 2;
   strcpy(tag->u.cmdline.cmdline,cmdline);
   tag = tag_next(tag);

       
   // now the mem32 tag
   putLabeledWord("Makeing mem32 tag at ",(unsigned long) tag);
   tag->hdr.tag = ATAG_MEM;
   tag->hdr.size = tag_size(tag_mem32);
   tag->u.mem.size = dram_size;
   tag->u.mem.start = 0xc0000000;
   tag = tag_next(tag);
       
       // and the ptable tag
   putLabeledWord("Makeing ptable tag at ",(unsigned long) tag);
   tag->hdr.tag = ATAG_PTABLE;
   tag->hdr.size = (sizeof(struct tag_ptable)
		    + sizeof(struct KernelFlashRegion) * partition_table->npartitions
		    + 3 + sizeof(struct tag_header)) >> 2;
   tag->u.ptable.magic = partition_table->magic;
   tag->u.ptable.npartitions = partition_table->npartitions;
   for (i=0; i < partition_table->npartitions;i++){
       strcpy(tag->u.ptable.partition[i].name,
	      partition_table->partition[i].name);
       tag->u.ptable.partition[i].base = partition_table->partition[i].base;
       tag->u.ptable.partition[i].size = partition_table->partition[i].size;
       tag->u.ptable.partition[i].flags = partition_table->partition[i].flags;
   }
       


   tag = tag_next(tag);
       
       
       // now the NULL tag
   tag->hdr.tag = ATAG_NONE;
   tag->hdr.size = 0;
       
   
   putLabeledWord("dram_size=", dram_size);
   putLabeledWord("nr_pages=", params->u1.s.nr_pages);
   putstr("command line is: ");
   putstr(cmdline);
   putstr("\r\n");
}
#endif

/*
 * boot_kernel
 *  root_filesystem_name:
 *  kernel_region_start: virtual address of region holding kernel image
 *  kernel_region_size:  size of region holding kernel image
 *  argc: number of arguments in argv (including argv[0], the command name)
 *  argv: argument vector (including argv[0], the command name)
 */
static void boot_kernel(const char *root_filesystem_name, 
                        vaddr_t kernel_region_start, size_t kernel_region_size, int argc, const char **argv, int no_copy)
{
  unsigned long *kernel_region_words;
  unsigned long *qnx_image_words = (unsigned long *)QNX_IMAGE_START;
  unsigned long kernel_magic;
  unsigned long kernel_image_source = (unsigned long)kernel_region_start ;
  unsigned long kernel_image_size = kernel_region_size;
  unsigned long kernel_image_dest;
  unsigned long kernel_image_maxsize = kernel_region_size;
  int i;
  char *os;
  long bootingLinux = 0;
  long linuxEntryPoint = 0;
  long memc_ctrl_reg = 0x110c;
  long mem_fclk_21285;
  long copy_ramdisk = 0;
  long dram_size = 0;
  long kernel_in_ram = 0;
  void (*qnx_start_func)( int );
  struct FlashRegion *ramdisk_region = btflash_get_partition("ramdisk");

  kernel_image_dest = DRAM_BASE;

  if (no_copy){
       // check from the ram copy       
      kernel_region_words = (unsigned long *) (0x8000 +
					       kernel_image_dest);      
  }
  else{
      kernel_region_words = (unsigned long *)kernel_region_start;// check the flash copy
  }
  kernel_magic = kernel_region_words[0];  
  
  get_param_value("dram_size", &dram_size);
  get_param_value("os", (long*)&os);
  get_param_value("entry", &linuxEntryPoint);
  get_param_value("kernel_in_ram", &kernel_in_ram);
  get_param_value("copy_ramdisk", &copy_ramdisk);

  /* Check for a QNX image */
  if ( strcmp( os, "autoselect" ) == 0 ||
       strcmp( os, "qnx" ) == 0 ) {

    /* Search the first 32 1K word boundries */ 
    for( i=0; i<32; i++ )
    {
        if ( qnx_image_words[ 256 * i ] == QNX_IMAGE_MAGIC ) {
          putLabeledWord( "Found a QNX kernel @ 0x",
			  (unsigned long)(&qnx_image_words[256*i]));
          qnx_start_func = (void *)QNX_IMAGE_START;
	  //flush_caches();
          let_uart_drain(SA1100_UART3_UTSR1);
          qnx_start_func( 0 );
          /* never gets here */
          return;
        }
    }
  }


  
  putLabeledWord("kernel_magic=", kernel_magic);
  if (kernel_magic == 0xFFFFFFFFL) {
    putstr("no boot image in flash\r\n");
    return;
  } 


  
  
  putLabeledWord("kernel_region_words[9]=", kernel_region_words[9]);
  
  if (kernel_region_words[9] == LINUX_ZIMAGE_MAGIC) {
    unsigned long compressed_rd_start = 0;
    unsigned long compressed_rd_size = 0;
    unsigned long initrd_size = SZ_8M;
    unsigned long initrd_start = STACK_BASE - initrd_size;
    unsigned long kernel_image_offset = 0x0;

    if (ramdisk_region != NULL) {
      compressed_rd_start = FLASH_BASE + ramdisk_region->base + 4;
      compressed_rd_size = *(unsigned long *)(FLASH_BASE + ramdisk_region->base);
    }          

#if 1
    if (   kernel_region_words[0] == SKIFF_ZIMAGE_MAGIC
           || kernel_region_words[9] == LINUX_ZIMAGE_MAGIC ) {
      /* standard Linux zImage gets loaded to phyasaddr 0x8000 */
      kernel_image_offset = 0x8000;
      linuxEntryPoint += 0x8000;
    }
#endif

    putstr("Linux ELF flash_imgstart=");
    putHexInt32((long)kernel_image_source);
    putstr(" size=");
    putHexInt32(kernel_image_size);
    putstr(" dest=");
    putHexInt32(kernel_image_dest);
    putstr(" offset=");
    putHexInt32(kernel_image_offset);
    putstr("\r\n");
     
    asmEnableCaches(0,0); // also flushes caches    
    putLabeledWord("MMU Control=", readCPR1());
    putLabeledWord("MMU PIDVAM=", readCPR13());

    if (((void *)kernel_image_dest != (void *)kernel_image_source) &&
	!no_copy) {
      putstr("copying Linux kernel ... ");
      memcpy((void*)(kernel_image_dest + kernel_image_offset),
	     (void*)kernel_image_source, kernel_image_size);
      putstr("done\r\n");
    }
    else if (no_copy)
	putstr("Skipping kernel copy by request.\n\r");

    for (i = 0; i < 40; i += 4) {
      /* we still have the MMU on, kernel_image_dest gets us to the right virtual address */
      putHexInt32(kernel_image_dest + kernel_image_offset + i);
      putstr(": ");
      putHexInt32(*(unsigned long *)(kernel_image_dest + kernel_image_offset +i));
      putstr("\r\n");
    }

    if (copy_ramdisk && compressed_rd_size != -1) {
      putstr("Copying compressed ramdisk from ");
      putHexInt32(compressed_rd_start);
      putstr(" to ");
      putHexInt32(initrd_start);
      putstr("...");
      memcpy((char*)initrd_start,(char*)compressed_rd_start,compressed_rd_size); 
      putstr("Done \r\n");
    } else if (compressed_rd_size != -1) {
      initrd_start = compressed_rd_start;
    }

    if (1) {
      char boot_args[MAX_BOOTARGS_LEN];
      char *linuxargs = NULL;
      int flashroot = 0;
      if (strcmp(root_filesystem_name, "ramdisk") == 0) {
        flashroot = 1;
      }

      putstr("root_filesystem_name="); putstr(root_filesystem_name); putstr("\r\n");
      memset(boot_args, 0, MAX_BOOTARGS_LEN);
      get_param_value("linuxargs", &linuxargs);
      putLabeledWord("Grabbed linuxargs, argc = ",argc);      
      if (linuxargs != NULL)
        strcat(boot_args, linuxargs);
      if (argc > 1) {
	  putstr("pre unparse setting boot parameters to\r\n");
	  putstr(boot_args);putstr("\r\n");
	  unparseargs(boot_args, argc-1, argv+1);
      }

      
      putstr("setting boot parameters to\r\n");
      putstr(boot_args);putstr("\r\n");

      
      
#ifdef __linux__
      setup_linux_params(kernel_image_dest, initrd_start, compressed_rd_size,
                         memc_ctrl_reg, mem_fclk_21285, dram_size, boot_args, flashroot);
#endif
    }

    putLabeledWord("linuxEntryPoint=", linuxEntryPoint);
    putstr("Booting Linux image\r\n");

    bootLinux(&bootinfo,
              MACH_TYPE,
#ifdef CONFIG_SKIFF
              /* hack -- linux entry point virtual address is 0x10008000, physaddr is 0x00008000 */
              /* after we disable the MMU we have to use the physical address */
              linuxEntryPoint&0x00FFFFFF
#else
              linuxEntryPoint
#endif

              );

  } else if (kernel_magic == NETBSD_KERNEL_MAGIC) {
    char boot_args[MAX_BOOTARGS_LEN];

    putstr("copying NetBSD kernel ... ");
    memcpy((void*)kernel_image_dest, (void*)kernel_image_source, kernel_image_size);
    putstr("done\r\n");

    /*bootinfo.bt_memavail = ROUNDUP(kernel_image_size, SZ_1M);*/
    bootinfo.bt_memavail = SZ_2M+SZ_1M;
    /*putLabeledWord("bt_memavail = ",bootinfo.bt_memavail);*/
      
    boot_args[0] = 0;
    strcat(boot_args, "netbsd ");
    if (argc > 1) {
      unparseargs(boot_args, argc-1, argv+1);
    }
    bootinfo.bt_args = boot_args;
    bootinfo.bt_vargp = (u_int32_t)boot_args & PAGE_MASK;
    bootinfo.bt_pargp = (u_int32_t)boot_args & PAGE_MASK;
    for(i = 0 ; i < (DRAM_SIZE0 - SZ_2M) ; i += SZ_1M) {
      unsigned long dram_paddr = DRAM_BASE + i;
      unsigned long dram_vaddr = 0xf0000000 + i;
      mmu_table[dram_vaddr >> 20] = dram_paddr | MMU_SECDESC | MMU_CACHEABLE;
    }
    asmEnableCaches(0,0);    
    putstr("done!\r\nJumping to 0xF0000020..\r\n");
    
    boot(&bootinfo,0xF0000020);
  } else {
    putstr("Unrecognized kernel image\r\n");
    return;
  }
}

/* can have arguments or not */
void command_boot_flash(int argc, const char **argv)
{
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   const char *kernel_part_name = NULL;
   char bootargs[MAX_BOOTARGS_LEN];
   struct FlashRegion *kernelPartition;

   get_param_value("ipaddr", &ipaddr);
   get_param_value("nfs_server_address", &serveraddr);
   get_param_value("gateway", &gatewayaddr);
   get_param_value("netmask", &netmask);
   get_param_value("hostname", &hostname);
   get_param_value("nfsroot", &nfsroot);
#define KERNEL_PARAM_STR    "kpart="
#define KERNEL_PARAM_STR_LEN sizeof(KERNEL_PARAM_STR)-1
   if (argc > 1 &&
       memcmp(KERNEL_PARAM_STR, argv[1], KERNEL_PARAM_STR_LEN) == 0) {
       kernel_part_name = argv[1] + KERNEL_PARAM_STR_LEN;
       /* skip over this param */
       argv[1] = argv[0];
       argv++;
       argc--;
   }
   else
       get_param_value("autoboot_kernel_part", &kernel_part_name);

   kernelPartition = btflash_get_partition(kernel_part_name);

   if (kernelPartition == NULL) {
       putstr("cannot find kernel partition named >");
       putstr(kernel_part_name);
       putstr("<\r\n");
     return;
   }
   else {
       putstr("booting kernel from partition >");
       putstr(kernel_part_name);
       putstr("<\r\n");
   }
   
   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   if ((ipaddr != NULL) || (serveraddr != NULL) || (gatewayaddr != NULL) || (netmask != NULL) || (hostname != NULL)) {
      strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
      strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
      strcat(bootargs, ":eth0 ");
   }
   
   argv[argc++] = bootargs;
   argv[argc] = NULL;

   boot_kernel("ramdisk",
               (vaddr_t)(((unsigned long)flashword) + kernelPartition->base), kernelPartition->size, argc, argv, 0);
}

/* can have arguments or not */
void command_boot_jffs2(int argc, const char **argv)
{
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   const char *kernel_file_name = NULL;
   const char *kernel_part_name = NULL;
   char bootargs[MAX_BOOTARGS_LEN];
   struct FlashRegion *kernelPartition;
   struct part_info part;
   int i;
   int size;
   const struct kernel_loader *the_loader = NULL;
   unsigned long ret;
   long kernel_in_ram = 0;


   bootargs[0] = '\0';
   get_param_value("ipaddr", &ipaddr);
   get_param_value("nfs_server_address", &serveraddr);
   get_param_value("gateway", &gatewayaddr);
   get_param_value("netmask", &netmask);
   get_param_value("hostname", &hostname);
   get_param_value("nfsroot", &nfsroot);
   get_param_value("kernel_partition", &kernel_part_name);


   kernelPartition = btflash_get_partition(kernel_part_name);
   if (argc > 1){
       kernel_file_name = argv[1];
       /* skip over this param */
       argv[1] = argv[0];
       argv++;
       argc--;
   }
   else
       get_param_value("kernel_filename", &kernel_file_name);   

   if (kernelPartition == NULL) {
       putstr("cannot find kernel partition named >");
       putstr(kernel_part_name);
       putstr("<\r\n");
     return;
   }
   else {
       putstr("booting ");
       putstr(kernel_file_name);
       putstr(" from partition >");
       putstr(kernel_part_name);
       putstr("<\r\n");
   }

   
   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   if ((ipaddr != NULL) || (serveraddr != NULL) || (gatewayaddr != NULL) || (netmask != NULL) || (hostname != NULL)) {
      strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
      strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
      strcat(bootargs, ":eth0 ");
   }
   
   if (strlen(bootargs)){
       argv[argc++] = bootargs;
       argv[argc] = NULL;
   }
   


   /* ok, so we copy the file to 32K ourselves and ask boot_kernel to skip the
    * copy.   that should be all that we need.
    */
   get_param_value("kernel_in_ram", &kernel_in_ram);   
   part.size = kernelPartition->size;
   /* for uniformly sized flash sectors */
   part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
   part.offset = ((char*)flashword) + kernelPartition->base;
    
   ret = jffs2_1pass_load((unsigned long *) kernel_in_ram,&part,kernel_file_name);   
   putstr("loaded file of size = 0x"); putHexInt32(ret);
   putstr(" at location 0x");putHexInt32(kernel_in_ram);putstr("\r\n");
   

   boot_kernel("ramdisk",
	       (vaddr_t)(((unsigned long)flashword) + kernelPartition->base), kernelPartition->size, argc, argv, 1);


}

/****************************************************************************/
/*
 *
 * THIS IS A TEST FOR POSSIBLY BOOTING WINCE
 *
 * **************************************************************************/
void command_boot_wince(int argc, const char **argv)
{
    void *pspr = NULL;
    unsigned long   v;
    unsigned long   mask;
    
    lcd_off();

    body_clearMem(0x100000,0xc0000000);
    
    bootLinux(0,
              2,
	      0x00040000
	);
   
}


#if 0
void load_ram_kernel(
    unsigned long img_dest)
{
  unsigned long offset;
  unsigned long img_size;

  /* parse the command line to fill img_size and img_dest */
  
  img_size = modem_receive((char*)img_dest, img_size);
  /* in case this is a kernel download, update bt_memavail */
  bootinfo.bt_memavail = ROUNDUP(img_size, SZ_1M);
  
  putHexInt32(img_size);
  putLabeledWord(" bytes loaded to ",img_dest);
  putstr("\r\n");

  return (img_size);
}
#endif

/* can have arguments or not */
void command_boot_addr(
    int		    argc,
    const char**    argv)
{
    const char *ipaddr = NULL;
    const char *serveraddr = NULL;
    const char *gatewayaddr = NULL;
    const char *netmask = NULL;
    const char *hostname = NULL;
    const char *nfsroot = NULL;
    char bootargs[MAX_BOOTARGS_LEN];
    unsigned long    img_dest;
    unsigned long    img_size;
    
    if (argc < 3) {
	putstr("not enough args, need <address> <size>\n\r");
	return;
    }
    
    img_dest = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("error parsing img_dest\r\n");
	return;
    }
    img_size = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("error parsing img_size\r\n");
	return;
    }
    
    get_param_value("ipaddr", &ipaddr);
    get_param_value("nfs_server_address", &serveraddr);
    get_param_value("gateway", &gatewayaddr);
    get_param_value("netmask", &netmask);
    get_param_value("hostname", &hostname);
    get_param_value("nfsroot", &nfsroot);
    
    if (nfsroot != NULL) {
	strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
    }
    if ((ipaddr != NULL) || (serveraddr != NULL) || (gatewayaddr != NULL) ||
	(netmask != NULL) || (hostname != NULL)) {
	strcat(bootargs, " ip=");
	strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
	strcat(bootargs, ":");
	strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
	strcat(bootargs, ":");
	strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
	strcat(bootargs, ":");
	strcat(bootargs, (netmask != NULL) ? netmask : "");
	strcat(bootargs, ":");
	strcat(bootargs, (hostname != NULL) ? hostname : "");
	strcat(bootargs, ":eth0 ");
    }

#if 0
    img_size = load_ram_kernel(img_dest);
    
    if ( image_size === 0) {
	putstr("load_ram_kernel() failed.\n\r");
	return;
    }
    image_size = (image_size + 4095) & ~4095;
#endif
    
    argv[argc++] = bootargs;
    argv[argc] = NULL;
    
    boot_kernel("ramdisk", (vaddr_t)img_dest, img_size, argc-2, argv+2,
		1);
}

/* can have arguments or not */
void command_boot_nfsroot(int argc, const char **argv)
{
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   char bootargs[MAX_BOOTARGS_LEN];
   struct FlashRegion *kernelPartition = btflash_get_partition("kernel");

   if (kernelPartition == NULL) {
     putstr("no kernel partition defined\r\n");
     return;
   }

   get_param_value("ipaddr", &ipaddr);
   get_param_value("nfs_server_address", &serveraddr);
   get_param_value("gateway", &gatewayaddr);
   get_param_value("netmask", &netmask);
   get_param_value("hostname", &hostname);
   get_param_value("nfsroot", &nfsroot);

   strcat(bootargs, " noinitrd root=/dev/nfs");
   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
   strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
   strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
   strcat(bootargs, ":eth0 ");
   
   argv[argc++] = bootargs;
   argv[argc] = NULL;

   boot_kernel("nfsroot",
               (vaddr_t)(((unsigned long)flashword) + kernelPartition->base), kernelPartition->size, argc, argv, 0);
}

void command_display(int argc, const char **argv)
{
#if defined(CONFIG_SA1100)
   putstr("\r\nSA1100 Registers:\r\n" );
     putLabeledWord("   SA1100_UART3:  ", (long) SA1100_UART3_UTCR0);
       putLabeledAddrWord( "      UTCR0   0x00 ", (long *)SA1100_UART3_UTCR0);
       putLabeledAddrWord( "      UTCR1   0x04 ", (long *)SA1100_UART3_UTCR1);
       putLabeledAddrWord( "      UTCR2   0x08 ", (long *)SA1100_UART3_UTCR2);
       putLabeledAddrWord( "      UTCR3   0x0c ", (long *)SA1100_UART3_UTCR3);
       putLabeledAddrWord( "      UTDR    0x10 ", (long *)SA1100_UART3_UTDR);
       putLabeledAddrWord( "      UTSR0   0x14 ", (long *)SA1100_UART3_UTSR0);
       putLabeledAddrWord( "      UTSR0+4 0x18 ", (long *)SA1100_UART3_UTSR0+4);
       putLabeledAddrWord( "      UTSR0+8 0x1c ", (long *)SA1100_UART3_UTSR0+8);
       putLabeledAddrWord( "      UTSR1   0x20 ", (long *)SA1100_UART3_UTSR1);
#endif
}


/* requires arguments */
void command_load_flash(int argc, const char **argv)
{
   unsigned long offset;

   unsigned long img_size = 0;
   unsigned long flash_dest = 0;
   unsigned long i;
   int override;

   get_param_value("override", &override);

   /* loading flash is different.. because writing flash is slow, we do
      not load it directly from the serial port. So we receive into
      memory first, and then program the flash... */
  
   if (argc < 2) { /* default to first command */
      putstr("usage: load flash <flashaddr>\r\n");
   } else {
      flash_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing flash_dest\r\n");
         return;
      }

      if (flash_dest < flashDescriptor->bootldr.size && !override) {
         putstr("That is bootloader space! Use load bootldr.  Operation canceled\r\n");
         return;
      }
      command_load_flash_region("flash", flash_dest, flash_size, 0);
   }
}


void command_load_ram(int argc, const char **argv)
{
  unsigned long offset;

  unsigned long img_size = 0;
  unsigned long img_dest = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("usage: load ram <ramaddr>\r\n");
  } else {
    /* parse the command line to fill img_size and img_dest */

    img_dest = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing img_dest\r\n");
      return;
    }
    
    img_size = modem_receive((char*)img_dest, img_size);
    /* in case this is a kernel download, update bt_memavail */
    bootinfo.bt_memavail = ROUNDUP(img_size, SZ_1M);

    putHexInt32(img_size);
    putLabeledWord(" bytes loaded to ",img_dest);
    putstr("\r\n");

    last_ram_load_address = img_dest;
  }
}

/**
 * does the work for command_load, etc.
 */

static void program_flash_region(const char *regionName, 
                                 unsigned long regionBase, size_t regionSize, 
                                 unsigned long src, size_t img_size, int flags)
{
   unsigned long hdr_size = 0; /* size of header */
   unsigned long i;
   int remaining_bytes;
   long noerase;
   get_param_value("noerase", &noerase);

   if (flags & LFR_PATCH_BOOTLDR) {
     unsigned int bsd_sum;
     /* check here that it's really a bootldr, otherwise print a message and exit */
     if (*(long *)(src + 0x20) != BOOTLDR_MAGIC) {
       putstr("Not loading bootldr into flash\r\n");
       putLabeledWord("Downloaded image does not have BOOTLDR_MAGIC: ", BOOTLDR_MAGIC);
       putLabeledWord("Instead found: ", *(long *)(src + 0x20));
       return;
       }
     /* check here to see if the architecture number matches */
     if (*(long *)(src + 0x2C) != ARCHITECTURE_MAGIC) {
       putstr( "Not loading bootldr into flash\r\n");
       putLabeledWord("Downloaded image does not have ARCHITECTURE_MAGIC=", ARCHITECTURE_MAGIC);
       putLabeledWord("Instead found=", *(long *)(src + 0x2C));
       return;
     }
     /* check here to be certain that the BSD sum value is zero */
     bsd_sum = bsd_sum_memory( src, img_size);
     putLabeledWord("BSD sum value is: ", bsd_sum);
     if (bsd_sum != 0) {
       putstr("BSD sum nonzero -- corrupt bootldr, not programming flash.\r\n");
       return;
     }
     {
	 unsigned long	start_addr = (unsigned long)(*(long *)(src + 0x28));

	 if (start_addr != 0x00000000) {
	     putLabeledWord("ERROR: will not program bootldr not linked at 0x00000000\n\r  start_addr: 0x", start_addr);
	     putstr("  download the version linked at 0x00000000\n\r");
	     return;
	 }
     }
   }

   if (flags & LFR_SIZE_PREFIX) {
      hdr_size += 4;
      src -= 4;
      *(long *)src = img_size;
      img_size += hdr_size;
   }

   if (img_size > regionSize) { // xple errors since minicom tends to eat some(all?)
      putLabeledWord("img_size is too large for region: ", regionSize);
      putLabeledWord("img_size is too large for region: ", regionSize);
      putLabeledWord("img_size is too large for region: ", regionSize);
      return;
   }

   putstr("\r\nprogramming flash...");

   set_vppen();
   if (flags & LFR_PATCH_BOOTLDR) {
      putstr( "\r\nunlocking boot sector of flash\r\n" );
      protectFlashRange( 0x0, img_size+hdr_size, 0);
   }
   if (!noerase) {
     int eraseSize = img_size + hdr_size;
     if (flags & LFR_JFFS2)
       eraseSize = regionSize;
     putstr("erasing ...\r\n");
     if (eraseFlashRange(regionBase,eraseSize)) {
       putstr("erase error!\r\n");
       goto clear_VPPEN;
     }
   }

   /* the magic number.. */
   putstr("writing flash..\r\n");
    
   i = 0;
   remaining_bytes = img_size;
   while(remaining_bytes > 0) {
      int bytes_programmed = 0;
      if ((i % SZ_64K) == 0) {
         putstr("addr: ");
         putHexInt32(regionBase+i);
         putstr(" data: ");
         putHexInt32(*(unsigned long *)(src+i));
         putstr("\r\n");
      }

      if (((i % 64) == 0) && (remaining_bytes > 64)) {
         /* program a block */
         if (programFlashBlock(regionBase+i, (unsigned long*)(src+i), 64)) {
            putstr("error while copying to flash!\r\n");
            goto clear_VPPEN;
            break;
         }
         bytes_programmed = 64;
      } else {
         if (programFlashWord(regionBase+i, *(unsigned long *)(src+i))) {
            putstr("error while copying to flash!\r\n");
            goto clear_VPPEN;
            break;
         }
         bytes_programmed = 4;
      }
      i += bytes_programmed;
      remaining_bytes -= bytes_programmed;
   }
   putstr("verifying ... ");
   {
      int i;
      int nwords = (img_size >> 2);
      unsigned long *srcwords = (unsigned long *)src;
      unsigned long *dstwords = (unsigned long *)&flashword[regionBase >> 2];
      for (i = 0; i < nwords; i++) {
         if (srcwords[i] != dstwords[i]) {
            putLabeledWord("error programming flash at offset=", i << 2);
            putLabeledWord("  src=", srcwords[i]);
            putLabeledWord("  flash=", dstwords[i]);
            putstr("not checking any more locations\r\n");
            goto clear_VPPEN;
         }
      }
   }
   if (flags & LFR_JFFS2) {
     unsigned long jffs2_sector_marker0 = 0xFFFFFFFF;
     unsigned long jffs2_sector_marker1 = 0xFFFFFFFF;
     unsigned long jffs2_sector_marker2 = 0xFFFFFFFF;
     get_param_value("jffs2_sector_marker0", &jffs2_sector_marker0);
     get_param_value("jffs2_sector_marker1", &jffs2_sector_marker1);
     get_param_value("jffs2_sector_marker2", &jffs2_sector_marker2);
     putstr("formatting ... ");
     btflash_jffs2_format_region(regionBase + img_size + hdr_size, regionSize - img_size - hdr_size, jffs2_sector_marker0, jffs2_sector_marker1, jffs2_sector_marker2);
   }
   putstr("done.\r\n");
   if (flags & LFR_PATCH_BOOTLDR) {
      protectFlashRange( 0x0, img_size+hdr_size, 1);
   }

 clear_VPPEN:
   clr_vppen();
}

static void command_load_flash_region(const char *regionName, unsigned long regionBase, unsigned long regionSize, int flags)
{
   unsigned long img_size = 0; /* size of data */
   unsigned long hdr_size = 0; /* size of header */
   unsigned long img_dest = VKERNEL_BASE + 1024; /* just a temporary holding area */
   unsigned long i;
   long xmodem = 0;
  
   putstr("loading flash region "); putstr(regionName); putstr("\r\n");

   get_param_value("xmodem", &xmodem);

#ifdef CONFIG_XYMODEM
   if (xmodem) {
#endif
      putstr("using xmodem\r\n");
      img_size = modem_receive((char*)img_dest, regionSize);
#ifdef CONFIG_XYMODEM
   } else {
      putstr("using xymodem\r\n");
      img_size = xymodem_receive((char*)img_dest, regionSize);
   }
#endif
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
    
   putHexInt32(img_size);
   putLabeledWord(" bytes loaded to ",img_dest);
   if ((img_size % 4) != 0) {
      putstr("img_size is not a multiple of 4 -- are we sure that's OK?\r\n");
   }

   program_flash_region(regionName, regionBase, regionSize, img_dest, img_size, flags);
}

/* requires arguments */
void command_load(int argc, const char **argv)
{
  if (argc > 1) {
    const char *partname = argv[1];
    struct FlashRegion *partition = btflash_get_partition(partname);
    if (partition == NULL)
      goto handle_subcommand;
    command_load_flash_region(partname, partition->base, partition->size, partition->flags);
    if (strcmp(partname, "params") == 0) {
      params_eval(PARAM_PREFIX, 0);
    }
  } else {
  handle_subcommand:
    execcmd(load_commands,argc-1,argv+1);
  }
}

/* can have arguments or not */
void command_save(int argc, const char **argv)
{
  if (argc > 1) {
    const char *partname = argv[1];
    struct FlashRegion *partition = btflash_get_partition(partname);
    if (partition == NULL)
      goto handle_subcommand;
    putstr("About to xmodem send "); putstr(partition->name); putstr("\r\n");
    putLabeledWord("  flashword=", (unsigned long)flashword);
    putLabeledWord("  base=", partition->base);
    putLabeledWord("  nbytes=", partition->size);
    if (!modem_send(partition->base + (char *)flashword, partition->size)) {
      putstr("download error.\r\n");
      return;
    } 
  } else {
  handle_subcommand:
    execcmd(save_commands,argc-1,argv+1);
  }
}

void command_save_all(int argc, const char **argv) 
{
   dword img_size;

   img_size = modem_send(flashword, flash_size); 
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   } 
}

void command_save_flash(int argc, const char **argv) {

   dword         img_size;
   unsigned long flash_size = 0;
   unsigned long flash_dest = 0;

   if (argc < 3) { /* default to first command */
      putstr("usage: load flash <flashaddr> <size>\r\n");
   } else {
      flash_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing <flashaddr>\r\n");
         return;
      }
      flash_size = strtoul(argv[2], NULL, 0);
      if (strtoul_err) {
	putstr("error parsing <size>\r\n");
      }
      putLabeledWord("flash_dest=", flash_dest );
      putLabeledWord("flash_size=", flash_size );
      img_size = modem_send( flash_dest + (char*)flashword, flash_size );
      if (!img_size) {
         putstr("download error. aborting.\r\n");
         return;
      }
   }
   return;
}

void command_save_ram(int argc, const char **argv) {

   dword         img_size;
   unsigned long ram_size = 0;
   unsigned long ram_dest = 0;

   if (argc < 3) { /* default to first command */
      putstr("usage: save ram <ramaddr> <size>\r\n");
   } else {
      ram_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing <ramaddr>\r\n");
         return;
      }
      ram_size = strtoul(argv[2], NULL, 0);
      if (strtoul_err) {
	putstr("error parsing <size>\r\n");
      }
      putLabeledWord("ram_dest=", ram_dest );
      putLabeledWord("ram_size=", ram_size );
      img_size = modem_send( ram_dest, ram_size );
      if (!img_size) {
         putstr("download error. aborting.\r\n");
         return;
      }
   }
   return;
}

void command_save_world(int argc, const char **argv) {

  putstr( "\r\nI don't know how to save the world.\r\n\r\n" );

}

void command_peek(int argc, const char **argv)
{
   execcmd(peek_commands,argc-1,argv+1);
}

void command_peek_ram(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long result = 0;
  unsigned long offset = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("peek ram requires arguments!\r\n");
  } else {
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    if (strcmp(argv[0], "byte") == 0)
       result = *(unsigned char *)(addr);
    else if (strcmp(argv[0], "short") == 0)
       result = *(unsigned short *)(addr);
    else if (strcmp(argv[0], "int") == 0)
       result = *(int *)(addr);
    else if (strcmp(argv[0], "flash") == 0)
       result = flashword[(addr&flash_address_mask) >> 2];
    else if (strcmp(argv[0], "gpio") == 0)
       /* read from the gpio read port */
       result = *(int *)(0x21800008 + addr);
    else 
       result = *(unsigned long *)(addr);

    putLabeledWord("  addr  = ",addr);
    putLabeledWord("  value = ",result);
  }
}

void command_poke(int argc, const char **argv)
{
   execcmd(poke_commands,argc-1,argv+1);
}

void command_poke_ram(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long value = 0;
  unsigned long offset = 0;
  
  if (argc < 3) { /* default to first command */
    putstr("poke ram requires arguments!\r\n");
  } else {

    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    value = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing value\r\n");
      return;
    }

    putstr("poke ram: \r\n");
    putLabeledWord(" addr=", addr);
    putLabeledWord(" value=", value);

    if (strcmp(argv[0], "byte") == 0) {
       *(unsigned char *)(addr) = value;
    } else if (strcmp(argv[0], "short") == 0) {
       *(unsigned short *)(addr) = value;
    } else if (strcmp(argv[0], "int") == 0) {
       *(int *)addr = value;
    } else if (strcmp(argv[0], "gpio") == 0) {
       /* write to the gpio write port */
       *(int *)(0x21800000 + addr) = value;
    } else if (strcmp(argv[0], "flash") == 0) {
       set_vppen();
       if (programFlashWord(addr & flash_address_mask, value))
          putstr("flash write failed!\r\n");
       clr_vppen();
    } else {
       *(unsigned long *)(addr) = value;
    }
  }
}


#if defined(CONFIG_SA1100)
extern unsigned long delayedBreakpointPC;
void command_breakpoint(int argc, const char **argv)
{
  if (argv[1] != NULL) {
    const char *pcstr = NULL;
    unsigned long pc;
    int delayed = 0;
    if (strcmp(argv[1], "delayed") == 0) {
      delayed = 1;
      pcstr = argv[2];
    } else {
      pcstr = argv[1];
    }
    pc = strtoul(pcstr, NULL, 0);
    pc &= 0xFFFFFFFC;
    pc |= 1;
    if (delayed) {
      putLabeledWord("PC breakpoint will be set after kernel unzip at: ", pc);
      delayedBreakpointPC = pc;
    } else {
      putLabeledWord("Setting hardware PC breakpoint at: ", pc);
      __asm__ ("mcr     p15, 0, %0, c14, c8, 0" : : "r" (pc));
    }
  } else {
    unsigned long pc = 0;
    putstr("Clearing PC breakpoint");
    __asm__ ("mcr     p15, 0, %0, c14, c8, 0" : : "r" (pc));
  }
}
#endif

void command_qflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long result = 0;
  unsigned long offset = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("qflash requires arguments!\r\n");
  } else {
    addr = strtoul(argv[argc-1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    if (strcmp(argv[1], "id") == 0) {
       result = queryFlashID(addr);
    } else if (strcmp(argv[1], "security") == 0) {
       result = queryFlashSecurity(addr);
    } else {
       result = queryFlash(addr);
    }
    putLabeledWord("value = ",result);
  }
}

void command_eflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long offset = 0;
  unsigned long len = 0;
  int override;

  get_param_value("override", &override);
  
  if (argc < 2) { /* default to first command */
    putstr("eflash requires arguments: <partition>|<addr> [<len>]|chip!\r\n");
    return;
  } else {
    struct FlashRegion *partition = btflash_get_partition(argv[1]);
    if (partition != NULL) {
      putstr("erasing partition "); putstr(argv[1]); putstr("\r\n");
      addr = partition->base;
      len = partition->size;
      goto erase;
    } else if (strncmp(argv[1], "chip", 4) == 0) {
      if (!override) {
	putstr("Cannot erase whole chip without setting override to 1.\r\n");
	return;
      }
      putstr("erasing flash chip\r\n");
      set_vppen();
      eraseFlashChip();
      clr_vppen();
      return;
    }
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr: "); putstr(argv[1]); putstr("\r\n");
    }
    putLabeledWord("addr=", addr);

    if (argc > 2) {
      len = strtoul(argv[2], NULL, 0);
      if (strtoul_err) {
        putstr("error parsing len: "); putstr(argv[2]); putstr("\r\n");
      }
      putLabeledWord("len=", len);

    }
  erase:
    if (addr == 0 && !override) {
      putstr("Cannot erase first sector without setting override to 1.\r\n");
      return;
    }

    set_vppen();
    if (len == 0) 
      eraseFlashSector(addr);
    else
      eraseFlashRange(addr, len);
    clr_vppen();
  }
}

void command_pflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long len = 0;
  unsigned long protect = 9;
  
  if (argc < 4) { /* default to first command */
    putstr("pflash requires arguments: <addr> <len> 0/1  (1 -> protect, 0 -> unprotect all)!\r\n");
  } else {
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr: "); putstr(argv[1]); putstr("\r\n"); return;
    }
    len = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing len: "); putstr(argv[2]); putstr("\r\n"); return;
    }
    protect = strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing protect: "); putstr(argv[3]); putstr("\r\n"); return;
    }
    /* be certain that protect is only 0/1 */
    if (!((protect == 0) || (protect == 1)) ) {
      putstr("error protect value must be 0/1: "); putstr(argv[3]); putstr("\r\n"); return;
    }
    
    if (verifiedFlashRange( addr, len )) {
       putLabeledWord("  addr=", addr);
       putLabeledWord("  len=", len);
       putLabeledWord("  protect=", protect);

       set_vppen();
       protectFlashRange(addr, len, protect);
       clr_vppen();
    } else { 
      putstr("Region specified is out of Range.\r\n" );
      putLabeledWord( " Please use a range less than:", flash_size );      
    }
  }
}

#if 0
void command_reboot(int argc, const char **argv)
{
  if (argc < 1) {
    reboot();
  } else {
    putstr("reboot does not need arguments!\r\n");
  }
}
#endif

void command_call(int argc, const char **argv)
{
  void (*fcn)(long a0, long a1, long a2, long a3) = NULL;
  int offset;
  
  if (argc < 2) {
    putstr("usage: call <addr>\r\n");
  } else {
    long a0 = 0;
    long a1 = 0;
    long a2 = 0;
    long a3 = 0;

    if (argv[1][0] == '.' || argv[1][0] == '=' || argv[1][0] == '-') {
	if (last_ram_load_address != 0) {
	    strtoul_err = 0;
	    fcn = (void*)last_ram_load_address;
	}
	else {
	    putstr("last_ram_load_address is 0.\n\r");
	    return;
	}
    }
    else
	fcn = (void*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing vaddr\r\n");
      return;
    }
    if (argc > 2) {
      a0 = strtoul(argv[2], NULL, 0);
    }
    if (argc > 3) {
      a1 = strtoul(argv[3], NULL, 0);
    }
    if (argc > 4) {
      a2 = strtoul(argv[4], NULL, 0);
    }
    if (argc > 5) {
      a3 = strtoul(argv[5], NULL, 0);
    }
    asmEnableCaches(0,0);
    putLabeledWord("Calling fcn=", (long)fcn);
    putLabeledWord("  a0=", a0);
    putLabeledWord("  a1=", a1);
    putLabeledWord("  a2=", a2);
    putLabeledWord("  a3=", a3);

    let_uart_drain(SA1100_UART3_UTSR1);
    
    fcn(a0, a1, a2, a3);
  }
}

void command_physaddr(int argc, const char **argv)
{
  unsigned long vaddr = 0;
  int offset;
  
  if (argc < 2) { /* default to first command */
    putstr("physaddr requires vaddr argument!\r\n");
  } else {
    vaddr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing vaddr\r\n");
      return;
    }
    {
      unsigned long mmu_table_entry = mmu_table[vaddr >> 20];
      unsigned long paddr = mmu_table_entry & ~((1L << 20) - 1);
      unsigned long sectionDescriptor = mmu_table_entry & ((1L << 20) - 1);
      putLabeledWord("vaddr=", vaddr);
      putLabeledWord("paddr=", paddr);
      putLabeledWord("sectionDescriptor=", sectionDescriptor);
    }
  }
}

void command_set(int argc, const char **argv)
{
   struct bootblk_param *param = bootldr_params;
   const char *paramValueString;
   char pName[256] = "";
   char pValue[256] = "";
   unsigned char set = 0;
   

   parseParamName(argc,argv,pName,pValue);
#if 0
   putstr("paramName <");
   putstr(pName);
   putstr(">\r\n");
   putstr("paramValue <");
   putstr(pValue);
   putstr(">\r\n");
#endif

   
   if ((argc == 1) && (pName[0] = '\0')) {
       /* no argses */
       int my_argc = 1;
       char *my_argv[128];
       my_argv[0] = "show";
       my_argv[1] = NULL;
       execcmd(commands, my_argc, (const char **)my_argv);
       return;
   }
       
   paramValueString = pValue;
   if (paramValueString == NULL)
      paramValueString = "";
   while (param->name != NULL) {
      int namelen = strlen(param->name);
      if (namelen == strlen(pName) &&
	  strncmp(pName, param->name, namelen) == 0) {
         long value;
         int paramValueLen = strlen(paramValueString);
	 set = 1;
	 putstr("  setting param <");
         putstr(param->name);
         putstr("> to value <");
         putstr(paramValueString);
         putstr(">\r\n");
         switch (param->paramFormat) {
         case PF_STRING: {
            if (paramValueLen == 0) {
               value = 0;
            } else {
               char *newstr = malloc(paramValueLen+1);
               memcpy(newstr, paramValueString, paramValueLen+1);
               value = (long)newstr;
            }
            break;
         }
         case PF_LOGICAL:
	   paramoldvalue = param->value;
           value = strtoul(paramValueString, NULL, 0);
         case PF_DECIMAL:
         case PF_HEX:
         default:
	    paramoldvalue = param->value;
            value = strtoul(paramValueString, NULL, 0);
         }
         param->value = value;
         if (param->update != NULL) {
            (param->update)(param);
         }
      }
      param++;
   }
   if (!set){
       putstr("I dont understand param ");
       putstr(pName);
       putstr("\r\n");
   }
   
}

void command_params_show(int argc, const char **argv)
{
   struct bootblk_param *params = bootldr_params;
   int i;
   unsigned char showMe;
   
   print_version_verbose(">> ");

   while (params->name != NULL) {
       showMe = 0;
       if (argc > 1){
	   for (i=1;i < argc; i++)
	       if (!strcmp(params->name,argv[i]))
		   showMe = 1;
       }
       else
	   showMe = 1;
       if (showMe){
	   putstr("  ");
	   putstr(params->name);
	   switch (params->paramFormat) {
	       case PF_STRING:
		   putstr("= <"); putstr((char*)params->value); putstr(">\r\n");
		   break;
	       case PF_DECIMAL: {
		   char buf[16];
		   dwordtodecimal(buf, params->value);
		   putstr("=<"); putstr(buf); putstr(">\r\n");
		   break;
	       }
	       case PF_HEX:
	       default:
		   putstr("=<0x"); putHexInt32(params->value); putstr(">\r\n");
		   break;
	   }
       }
       params++;
   }
}

#ifdef UPDATE_BAUDRATE
void update_baudrate(struct bootblk_param *param)
{
   volatile unsigned long *csr = (volatile unsigned long *)DC21285_ARMCSR_BASE;
#ifdef CONFIG_SKIFF
   unsigned long system_rev = get_system_rev();
#endif
   unsigned long fclk_hz = 33333333;
   int baudrate = param->value;
   unsigned long ubrlcr;
   unsigned long l_ubrlcr = 0;
   unsigned long m_ubrlcr = 0;
   unsigned long h_ubrlcr = 0;


   
   switch ( baudrate ) {
#if defined(CONFIG_SA1100)
   case 230400:
     break;
   case 115200:
     break;
#endif
   case 57600:
     break;
   case 38400:
     break;
   case 19200:
     break;
   case 9600:
     break;
   case 4800:
     break;
   case 2400:
     break;
   case 1200:
     break;
   case 300:
     break;
   case 110:
     break;
   default:
     param->value = paramoldvalue;
     putstr( "invalid baud rate:\r\n" );
     putstr( "     Please, try:   110,   300,  1200,   2400,  4800,  9600\r\n" );
     putstr( "                  19200, 38400, 57600, 115200 or 230,400\r\n" );
     return; 
   }
   
#ifdef CONFIG_SKIFF
   switch (system_rev&SYSTEM_REV_MEMCLK_MASK) {
   case SYSTEM_REV_MEMCLK_33MHZ:
      fclk_hz = 33333333;
      break;
   case SYSTEM_REV_MEMCLK_48MHZ:
      fclk_hz = 48000000;
      break;
   case SYSTEM_REV_MEMCLK_60MHZ:
      fclk_hz = 60000000;
      break;
   }

  fclk_hz = 48000000;

   ubrlcr = (fclk_hz / 64 / baudrate)-1;
   h_ubrlcr = csr[H_UBRLCR_REG];

#elif defined(CONFIG_SA1100)
   ubrlcr = (3686400 / 16 / baudrate) - 1;
#else
   #error no architecture defined in update_baudrate
#endif

   l_ubrlcr = ubrlcr & 0xFF;
   m_ubrlcr = (ubrlcr >> 8) & 0xFF;
   
   putLabeledWord("update_baudrate:  new baudrate=", baudrate);

#ifdef CONFIG_SKIFF
   /* wait for the TX FIFO to empty */
   while ((csr[UARTFLG_REG]&UART_TX_BUSY) != 0)
      /* spin */;

   /* disable the uart */
   csr[UARTCON_REG] = 0;

   /* set the new values, in the right order */
   csr[L_UBRLCR_REG] = l_ubrlcr;
   csr[M_UBRLCR_REG] = m_ubrlcr;
   csr[H_UBRLCR_REG] = h_ubrlcr;

   /* reenable the uart */
   csr[UARTCON_REG] = 1;
#else
   /* wait for the TX FIFO to empty */
  while (((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_TBY)); /* wait until TX not busy */
  *(volatile byte*)SA1100_UART3_UTCR3 = 0; /* disable UART */

  *(volatile byte*)SA1100_UART3_UTCR1 = m_ubrlcr; /* MSB of baudrate divisor */
  *(volatile byte*)SA1100_UART3_UTCR2 = l_ubrlcr; /* LSB of baudrate divisor */
  
  *(volatile byte*)SA1100_UART3_UTCR3 = SA1100_UTCR3_RXE|SA1100_UTCR3_TXE; /* enable UART */
#endif

   putLabeledWord(" baudrate changed to 0x", baudrate);

   /* set the value for the kernel too */

   bootinfo.bt_comspeed = baudrate;
}
#endif

void set_serial_number(struct bootblk_param *param)
{
}


#ifdef CONFIG_SKIFF
long get_system_rev()
{
   long word1 = *(long *)FLASH_BASE;
   long word2 = *(long *)(FLASH_BASE+0x0080000C);
   long revision = SYSTEM_REV_SKIFF_V1;
   if (word1 != word2) {
      /* Skiff-V2 -- GPIO register 0 contains the revision mask */
      revision = (word2 & SYSTEM_REV_MASK);
   }
   return revision;
}

void set_system_rev(struct bootblk_param *param)
{
   param->value = get_system_rev();
}
#endif


#if 0
void
pparam(
    const char*			t,
    const struct bootblk_param*	param)
{
    putstr(t);
    putstr("name>");
    putstr(param->name);
    putstr("<, ");
    
    if (param->paramType == PT_STRING) {
	char*	p = (char*)param->value;
	
	putstr("S>");
	if (p)
	    putstr(p);
	else
	    putstr("(NULL)");
	putstr("<\n\r");
    }
    else {
	putLabeledWord("I=0x", param->value);
    }
}
#endif
    
int
param_is_modified(
    const struct bootblk_param*	param,
    const struct bootblk_param* def_param)
{
    /*
     * compare params, taking into account the fact that
     * a string can have a different address and the same
     * value.
     */
#if 0
    pparam("par: ", param);
    pparam("def: ", def_param);
#endif
    
    if (param->paramType == PT_STRING) {
	if (param->value && def_param->value) {
	    return (strcmp((char*)(param->value),
			   (char*)(def_param->value)) != 0);
	}
	else {
	    /*
	     * we know both aren't non-null.
	     * if both are then that is a match
	     */
	    return (param->value || def_param->value);
	}
    }
    else {
	return (param->value != def_param->value);
    }
    
}

void command_params_save(int argc, const char **argv)
{
   char param_buf[SZ_16K];
   char *buf = param_buf;
   int partition_table_size = 0;
   int dont_save = 0;
   struct FlashRegion *paramsPart;
   struct bootblk_param *param = bootldr_params;
   /* default params points into the flash image as opposed to the copy in dram */
   const struct bootblk_param *defaultParam = 
      (const struct bootblk_param *)(((char*)flashword) + (((dword)bootldr_params) - FLASH_BASE));

   /* should pick up the old params from flash in case there are any non-bootldr lines */

   memset(param_buf, 0, sizeof(param_buf));

   partition_table_size = (sizeof(struct BootldrFlashPartitionTable) 
                           + partition_table->npartitions*sizeof(struct FlashRegion));
   memcpy(buf, partition_table, partition_table_size);

   buf += partition_table_size;

   if (argc > 1 && strcmp(argv[1], "-n") == 0) {
       PLW(flashword);
       PLW(bootldr_params);
       PLW(FLASH_BASE);
       PLW(defaultParam);
       dont_save = 1;
   }

   /* construct the parameter setting code */
   while (param->name != NULL) {
      int modifiedFromDefault = param_is_modified(param,
						  defaultParam); 
      if (modifiedFromDefault && !(param->paramType&PT_READONLY)) {
         strcat(buf, "bootldr: set ");
         strcat(buf, param->name);
         strcat(buf, " ");
         switch (param->paramFormat) {
         case PF_STRING:
	    strcat(buf, "\"");
            strcat(buf, (char *)param->value);
	    strcat(buf, "\"");
            break;
         case PF_DECIMAL: {
           dwordtodecimal(buf+strlen(buf), param->value);
         } break;
         case PF_HEX:
         default: {
            char num[16];
            strcat(buf, "0x");
            binarytohex(num, param->value,4);
            strcat(buf, num);
         }         
         }
         strcat(buf, "\r\n");
      }
      param++;
      defaultParam++;
   }
   putstr(buf);

   if (dont_save)
       putstr("Not erasing and writing params sector.\n\r");
   else {
       /* now erase and program the params sector */
       paramsPart = btflash_get_partition("params");
       if (paramsPart)
	   program_flash_region("params", paramsPart->base,
				paramsPart->size,
				(dword)param_buf,
				partition_table_size + strlen(buf)+1,
				paramsPart->flags);
   }
}


void command_cmdex(int argc, const char **argv)
{
    int	state;
    
    if (argc > 1)
	 use_extended_getcmd = strtoul(argv[1], NULL, 0);
    else
	use_extended_getcmd = !use_extended_getcmd;

    putLabeledWord("use_extended_getcmd=0x", use_extended_getcmd);    
}

void
update_autoboot_timeout(
    struct bootblk_param *param)
{
    autoboot_timeout = (unsigned long)param->value;
    if (autoboot_timeout == 0)
	putstr("Autoboot is DISABLED\n\r");
}

void
update_cmdex(
    struct bootblk_param *param)
{
    use_extended_getcmd = (unsigned long)param->value;
}

#if defined(CONFIG_BITSY)

void command_aux_ser(
    int argc,
    const char **argv)
{
    auxm_serial_dump();
}

void
command_lcd_test(
    int	    argc,
    const char*   argv[])
{
    int		    params_id;
    lcd_params_t*   params;
    
    if (argc > 1) {
	params_id = strtoul(argv[1], NULL, 0);
	if (params_id == LCDP_CUSTOM && argc > 2) {
	    params = (lcd_params_t*)strtoul(argv[2], NULL, 0);
	}
    }
    else {
	params_id = lcd_params;
	params = NULL;
    }
    
    lcd_default_init(params_id, params);
}

void
command_lcd_on(
    int	    argc,
    const char*   argv[])
{
    lcd_on();
}

void
command_lcd_off(
    int	    argc,
    const char*   argv[])
{
    lcd_off();
}

void
command_lcd_light(
    int	    argc,
    const char*   argv[])
{
    if (argc == 1)
	lcd_off();
    else {
	int level = strtoul(argv[1], NULL, 0);
	lcd_light(1, level);
    }
}

#if 0
void
command_lcd_pal(
    int	    argc,
    const char*   argv[])
{
    int	 palval;

    if (argc > 1) {
	palval = strtoul(argv[1], NULL, 0);
	setup_solid_palette(palval);
    }
    else
	putstr("Need arg: palval\n\r");
}
#endif


void
command_lcd_fill(
    int	    argc,
    const char*   argv[])
{
    int	color;
    int	inc;

    if (argc > 1) {
	color = strtoul(argv[1], NULL, 0);
	if (argc > 2)
	    inc = strtoul(argv[2], NULL, 0);
	else
	    inc = 0;
	lcd_fill(color, inc);
    }
    else
	putstr("Need args: color [inc]\n\r");
}

void
command_lcd_bar(
    int	    argc,
    const char*   argv[])
{
    int	color;
    int	percent;


    if (argc > 1) {
	color = strtoul(argv[1], NULL, 0);
	if (argc > 2)
	    percent = strtoul(argv[2], NULL, 0);
	else
	    percent = 100;
	lcd_bar(color, percent);
    }
    else
	putstr("Need args: color [percent]\n\r");

}


void
command_lcd_image(
    int	    argc,
    const char*   argv[])
{
    char*   fb = lcd_get_image_buffer();
    dword   img_size;
    
    img_size = modem_receive(fb, LCD_NUM_DISPLAY_BYTES(LCD_BPP));

    lcd_display_bitmap(fb, img_size, lcd_params, NULL);
}


void
command_led_blink(
    int	    argc,
    const char*   argv[])
{
    char ledData[] = {0x01,0x00,0x05,0x05};
    
    if (argc > 1) 
	ledData[2] = (char) strtoul(argv[1], NULL, 0);
    
    if (argc > 2)
	ledData[2] = (char) strtoul(argv[2], NULL, 0);

    if (argc > 3)
	ledData[0] = (char) strtoul(argv[3], NULL, 0);
    
    led_blink(ledData[0],ledData[1],ledData[2],ledData[3]);
    
}




#if 1

voidpf
zcalloc(
    voidpf opaque,
    unsigned items,
    unsigned size)
{
    void*   p;
    int	    totsize;

    p = mmalloc(totsize = items * size);
    if (p) {
	memset(p, 0x00, totsize);
    }
    return(p);
}

void
zcfree(
    voidpf opaque,
    voidpf ptr)
{
    mfree(ptr);
}

#undef free
void
free(
    void*   p)
{
    mfree(p);
}

char	zbuf[16384];
#endif

void
command_lcd_zimage(
    int	    argc,
    const char*   argv[])
{
    char*   fb = lcd_get_image_buffer();
    dword   img_size;
    uLongf  uncomplen = lcd_get_image_buffer_len();
    int	    rc;

    if (argc < 2) {
	putstr("need args: len\n\r");
	return;
    }
    
    img_size = strtoul(argv[1], NULL, 0);
    if (img_size > sizeof(zbuf)) {
	putLabeledWord("img too big, zbuf size: ", sizeof(zbuf));
	return;
    }
	
    rc = modem_receive(zbuf, img_size);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    if (rc == 0) {
	putstr("download failed.  Aborting\n\r");
	return;
    }

#if 0
    
    rc = uncompress(fb, &uncomplen, zbuf, img_size);
    putLabeledWord("uncompress returned, rc: 0x", rc);
    if (rc != Z_OK) {
	putLabeledWord("uncompress failed, rc: 0x", rc);
	return;
    }

#else
    
    {
	/*
	 * do it piecemeal to test out using zlib stream functions...
	 */
	z_stream stream;
	int	err;
	int	err2;
	char	tiny_buf[128];
	char*	fbp = fb;
	size_t tlen;
	
	stream.next_in = (Bytef*)zbuf;
	stream.avail_in = (uInt)img_size;
	
	stream.next_out = tiny_buf;
	stream.avail_out = (uInt)sizeof(tiny_buf);
	
	stream.zalloc = (alloc_func)0;
	stream.zfree = (free_func)0;
	
	err = inflateInit(&stream);
	if (err != Z_OK) {
	    putLabeledWord("inflateInit failed", err);
	    return;
	}

	while ((err = inflate(&stream, Z_SYNC_FLUSH)) == Z_OK) {
	    tlen = sizeof(tiny_buf) - stream.avail_out;

	    if (tlen) {
		memcpy(fbp, tiny_buf, tlen);
		fbp += tlen;
		/*  putLabeledWord("memcpy 0x", tlen); */
	    }

	    if (stream.avail_out == 0) {
		/*  putstr("reset obuf\n\r"); */
		stream.next_out = tiny_buf;
		stream.avail_out = (uInt)sizeof(tiny_buf);
	    }
	}

	if (err == Z_STREAM_END) {
	    tlen = sizeof(tiny_buf) - stream.avail_out;
	    if (tlen)
		memcpy(fbp, tiny_buf, tlen);
	}
	    
	err2 = inflateEnd(&stream);
	
	if (err != Z_STREAM_END) {
	    putLabeledWord("inflate failed", err);
	    return;
	}

	if (err2 != Z_OK) {
	    putLabeledWord("inflateEnd failed", err2);
	    return;
	}
	
	uncomplen = stream.total_out;
    }
    
#endif
    
    putstr("about to display\n\r");

#if 1
    lcd_display_bitmap(fb, uncomplen, lcd_params, NULL);
#else
    putstr("skipping pixel conversion");
#endif    
}


void
command_ttmode(
    int		argc,
    const char*	argv[])
{
    static int	ttmode = 0;
    int	new_mode;

    if (argc > 1)
	new_mode = strtoul(argv[1], NULL, 0);
    else
	new_mode = 1;		/* no args turns on */

    if (new_mode) {
	packetize_output = 1;
	ack_commands = 1;
	no_cmd_echo = 1;
	use_extended_getcmd = 0;
    }
    else {
	packetize_output = 0;
	ack_commands = 0;
	no_cmd_echo = 0;
	use_extended_getcmd = 1;
    }

    ttmode = new_mode;
}

void
command_ser_con(
    int		argc,
    const char* argv[])
{
}

void
command_irda_con(
    int		argc,
    const char* argv[])
{
    putstr("irda not available yet, starting serial console.\n\r");
}

void
command_splash(
    int		argc,
    const char* argv[])
{
    char *fileName = NULL;
    char *partName = NULL;
    
    if (argc > 2)
	partName = argv[2];
    if (argc > 1){
	fileName = argv[1];
	splash_file(fileName,partName);
    }
    else{
	splash();
    }
    
}

int
reboot_button_is_enabled(
    void)
{
    return (reboot_button_enabled > 0);
}

void
enable_reboot_button(
    void)
{
    reboot_button_enabled++;
}

void
disable_reboot_button(
    void)
{
    if (reboot_button_enabled > 0)
	reboot_button_enabled--;
}
#endif

void
command_memcpy(
    int		argc,
    const char* argv[])
{
    void*   dst;
    void*   src;
    int	    num;
    int	    size = 1;
    
    if (argc < 4) {
	putstr("memcpy needs args: dst src num [size]");
	return;
    }

    dst = (void*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("bad dst param\n\r");
	return;
    }
    src = (void*)strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("bad src param\n\r");
	return;
    }
    num = strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
	putstr("bad num param\n\r");
	return;
    }
    if (argc > 4) {
	size = strtoul(argv[4], NULL, 0);
	if (strtoul_err) {
	    putstr("bad size param\n\r");
	    return;
	}
    }

    putstr("memcpy\n\r");
    putLabeledWord("src: 0x", (unsigned long)src);
    putLabeledWord("dst: 0x", (unsigned long)dst);
    putLabeledWord("num: 0x", num);
    putLabeledWord("size: 0x", size);
    
    switch (size) {
	case 1:
	{
	    char*   d = (char*)dst;
	    char*   s = (char*)src;

	    while (num--)
		*d++ = *s++;
	}
	break;

	case 2:
	{
	    short*   d = (short*)dst;
	    short*   s = (short*)src;

	    while (num--)
		*d++ = *s++;
	}
	break;

	case 4:
	{
	    int*   d = (int*)dst;
	    int*   s = (int*)src;

	    while (num--)
		*d++ = *s++;
	}
	break;

	default:
	    putLabeledWord("Bad size: 0x", size);
	    break;
    }
    
}

void  hex_dump(
    unsigned char   *data,
    size_t	    num)
{
    int     i;
    long    oldNum;
    char    buf[90];
    char*   bufp;
    int	    line_resid;
    
    while (num)
    {
	bufp = buf;
	binarytohex(bufp, (unsigned long)data, 4);
	bufp += 8;
	*bufp++ = ':';
	*bufp++ = ' ';
	
	oldNum = num;
	
	for (i = 0; i < 16 && num; i++, num--) {
	    binarytohex(bufp, (unsigned long)data[i], 1);
	    bufp += 2;
	    *bufp++ = (i == 7) ? '-' : ' ';
	}

	line_resid = (16 - i) * 3;
	if (line_resid) {
	    memset(bufp, ' ', line_resid);
	    bufp += line_resid;
	}
	
	memcpy(bufp, "| ", 2);
	bufp += 2;
	
	for (i = 0; i < 16 && oldNum; i++, oldNum--)
	    *bufp++ = BL_ISPRINT(data[i]) ? data[i] : '.';

	line_resid = 16 - i;
	if (line_resid) {
	    memset(bufp, ' ', 16-i);
	    bufp += 16 - i;
	}
	
	*bufp++ = '\r';
	*bufp++ = '\n';
	*bufp++ = '\0';
	putstr(buf);
	data += 16;
    }
}

void
command_hex_dump(
    int		argc,
    const char*	argv[])
{
    size_t	    num;
    unsigned char*  p;

    if (argc == 3)		// size specified
	num = strtoul(argv[2], NULL, 0);
    else
	num = 16;

    p = (unsigned char*)strtoul(argv[1], NULL, 0);

    hex_dump(p, num);
}

void
bootldr_goto_sleep(
    void*   pspr)
{
    unsigned long   v;
    unsigned long   mask;

    /*
     * shut down all of the peripherals in an clean and
     * orderly fashsion.
     * (later)
     */

    /* XXX do it!!!! */
    lcd_off();
    auxm_fini_serial();
    CTL_REG_WRITE(SA1100_UART2_UTCR3, 0);
    CTL_REG_WRITE(SA1100_UART3_UTCR3, 0);
    
    
    /*  set pspr */
    if (pspr == NULL) {
	extern void SleepReset_Resume(void);
	extern void ResetEntryPoint(void);
	
	pspr = (void*)(SleepReset_Resume-ResetEntryPoint);
    }
	    
    CTL_REG_WRITE(SA1100_PSPR, (unsigned long)pspr);
    
    /*  clear reset register status bits */
    CTL_REG_WRITE(SA1100_RCSR, 0x0f);
    
    /*  setup GPIO outputs sleep state */
    /*  use current values ??? a good idea ??? */
    mask = CTL_REG_READ(SA1100_GPIO_BASE+SA1100_GPIO_GPDR_OFF);
    v = CTL_REG_READ(SA1100_GPIO_BASE+SA1100_GPIO_GPLR_OFF);
    v &= mask;
    CTL_REG_WRITE(SA1100_PGSR, v);
    
    /*  set wakeup conditions */
    /*  any gpio edge */
    CTL_REG_WRITE(SA1100_PWER, (1<<0));	/* power button */

    /* setup edge reggies
     * Wakeup on rising edge of power button.  It is inactive high
     * --------+        +-------
     *         +--------+
     *                  ^- wakeup here
     */
	
    CTL_REG_WRITE(SA1100_GPIO_BASE+SA1100_GPIO_GRER_OFF, 0x00000001);
    CTL_REG_WRITE(SA1100_GPIO_BASE+SA1100_GPIO_GFER_OFF, 0x00000000);

    /*  clear all previously detected edge bits */
    CTL_REG_WRITE(SA1100_GPIO_BASE+SA1100_GPIO_GEDR_OFF, 0xffffffff);

#if 0
    /*  set up an RTC int */
      v = regread(SA1100_RCNR)
      v += 10                             # 1hz-->10seconds
      regwrite(SA1100_RTAR, v)
      regread(SA1100_RCNR)
      regwrite(SA1100_RTSR, 1<<2)
#endif

#if 0				/* needed ??? */
    /*  enable some ints so we can wake up */
    CTL_REG_WRITE(SA1100_ICLR, 0);	/* make 'em all irqs */
    
    CTL_REG_WRITE(SA1100_ICMR,
#if 0
		  (1<<31)|	/* RTC match int */
		  (1<<17)|	/* gpio 27-11 (incl ACT button) */
#endif
		  (1<<0));	/* power button */
#endif
    
#if 0
    CTL_REG_WRITE(SA1100_RTSR, 0);
#endif
    
    CTL_REG_WRITE(SA1100_PCFR, PCFR_OPDE); /* kill the clock */
    
    /*  zzzzz */
    CTL_REG_WRITE(SA1100_PMCR, (1<<0));
}

void
command_reset(
    int		argc,
    const char*	argv[])
{
    bootldr_reset();
}

int
reportMismatch(
    void*	    addr1,
    void*	    addr2,
    unsigned long   w1,
    unsigned long   w2)
{
    putLabeledWord("addr1=0x", (unsigned long)addr1);
    putLabeledWord("addr2=0x", (unsigned long)addr2);
    putLabeledWord("w1=0x", w1);
    putLabeledWord("w2=0x", w2);

    /* signal no more comparisons */
    /* XXX add flag for 1 mismatch vs all mismatches */
    return(1);
}

void
command_memcmp(
    int		argc,
    const char* argv[])
{
    void*   dst;
    void*   src;
    int	    num;
    int	    size = 1;
    
    if (argc < 4) {
	putstr("memcpy needs args: dst src num [size]");
	return;
    }

    dst = (void*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("bad addr1 param\n\r");
	return;
    }
    src = (void*)strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("bad addr2 param\n\r");
	return;
    }
    num = strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
	putstr("bad num param\n\r");
	return;
    }
    if (argc > 4) {
	size = strtoul(argv[4], NULL, 0);
	if (strtoul_err) {
	    putstr("bad size param\n\r");
	    return;
	}
    }

    putstr("memcmp\n\r");
    putLabeledWord("a1: 0x", (unsigned long)src);
    putLabeledWord("a2: 0x", (unsigned long)dst);
    putLabeledWord("num: 0x", num);
    putLabeledWord("size: 0x", size);
    
    switch (size) {
	case 1:
	{
	    char*   d = (char*)dst;
	    char*   s = (char*)src;

	    while (num--) {
		if (*d != *s) {
		    if (reportMismatch(d, s, *d, *s))
			break;
		}
		s++;
		d++;
	    }
	}
	break;

	case 2:
	{
	    short*   d = (short*)dst;
	    short*   s = (short*)src;

	    while (num--) {
		if (*d != *s) {
		    if (reportMismatch(d, s, *d, *s))
			break;
		}
		s++;
		d++;
	    }
	}
	break;

	case 4:
	{
	    int*   d = (int*)dst;
	    int*   s = (int*)src;

	    while (num--) {
		if (*d != *s) {
		    if (reportMismatch(d, s, *d, *s))
			break;
		}
		s++;
		d++;
	    }
	}
	break;

	default:
	    putLabeledWord("Bad size: 0x", size);
	    break;
    }
    
}

void
set_machine_mitsy(
    void)
{
    /*
     * we are a monochrome bitsy.
     */

    /*
     * set up correct LCD params.
     */
    lcd_params = LCDP_MONO;

    /* turn off till we get this working. */
    do_splash = 0;
    
}

void
command_version(
    int		argc,
    const char*	argv[])
{
    print_version_verbose(NULL);
}

void
body_testJFFS2(char *filename,unsigned long *dest)
{
  struct part_info part;
  int i;
  int size;
  const struct kernel_loader *the_loader = NULL;
  struct FlashRegion *flashRegion = btflash_get_partition("root");
  unsigned long ret;
  
  
    putstr("Attempting to output file " );
    putstr(filename);
    putstr("\n");

    if (!flashRegion) {
	putstr("could not find partition "); putstr("root"); putstr("\r\n");
	return -1;
    }
    part.size = flashRegion->size;
    /* for uniformly sized flash sectors */
    part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
    part.offset = ((char*)flashword) + flashRegion->base;
    
    putLabeledWord("root part size =", part.size);

  for (i = 0; loader[i]; i++) {
    if (loader[i]->check_magic(&part)) {
      the_loader = loader[i];
      break;
    }
  }
  if (!the_loader) {
    putstr("no kernel found\r\n");
    return -1;
  }
  else {
      putstr("partition is of type ");
      putstr(the_loader->name);
      putstr(" \r\n");
  }

  ret = jffs2_test_load((unsigned long *) dest,&part,filename);

  putLabeledWord("returned  = ", ret); putstr("\r\n");
  
  
}


void
command_testJFFS2(
    int		argc,
    const char* argv[])
{
    char *filename;
    unsigned long * dst;
    
    if (argc > 1) {
	filename = argv[1];
    }
    else{
	putstr("read which file???\n\r");
	return;
    }
    if (argc > 2){
	
        dst = (void*)strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\r\n");
	    return;
	}
    }
    else{
	putstr("copy the file to where???\r\n");
	return;
    }
    
    body_testJFFS2(filename,dst);
}


void
body_infoJFFS2(char *partname)
{
    struct part_info part;
    int i;
    int size;
    const struct kernel_loader *the_loader = NULL;
    struct FlashRegion *flashRegion= btflash_get_partition(partname);
    unsigned long ret;

  
  
    if (!flashRegion) {
	putstr("could not find partition "); putstr("root"); putstr("\r\n");
	return -1;
    }
    part.size = flashRegion->size;
    /* for uniformly sized flash sectors */
    part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
    part.offset = ((char*)flashword) + flashRegion->base;
    
    ret = jffs2_info(&part);

    
}


void
command_infoJFFS2(
    int		argc,
    const char* argv[])
{
    char partname[256];
    unsigned long * dst;
    
    if (argc > 1) {
	strcpy(partname,argv[1]);	
    }
    else{
	strcpy(partname,"root");
    }
    
    body_infoJFFS2(partname);
}

void
body_timeFlashRead(char *partname)
{
    struct part_info part;
    int i;
    int size;
    const struct kernel_loader *the_loader = NULL;
    struct FlashRegion *flashRegion= btflash_get_partition(partname);
    unsigned long ret;

  
  
    if (!flashRegion) {
	putstr("could not find partition "); putstr("root"); putstr("\r\n");
	return -1;
    }
    part.size = flashRegion->size;
    /* for uniformly sized flash sectors */
    part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
    part.offset = ((char*)flashword) + flashRegion->base;
    
    for (i = 0; loader[i]; i++) {
	if (loader[i]->check_magic(&part)) {
	    the_loader = loader[i];
	    break;
	}
    }
    if (!the_loader) {
	putstr("no loader found for the partition\r\n");
	return -1;
    }
    ret = jffs2_scan_test(&part);

    
}


void
command_timeFlashRead(
    int		argc,
    const char* argv[])
{
    char partname[256];
    unsigned long * dst;
    
    if (argc > 1) {
	strcpy(partname,argv[1]);	
    }
    else{
	strcpy(partname,"root");
    }
    
    body_timeFlashRead(partname);
}

void
body_p1_ls(char *dir,char *partname)
{
    struct part_info part;
    int i;
    int size;
    struct FlashRegion *flashRegion= btflash_get_partition(partname);
    unsigned long ret;

  
  
    if (!flashRegion) {
	putstr("could not find partition "); putstr(partname); putstr("\r\n");
	return -1;
    }
    part.size = flashRegion->size;
    /* for uniformly sized flash sectors */
    part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
    part.offset = ((char*)flashword) + flashRegion->base;
    
    jffs2_1pass_ls(&part,dir);

    
}


void
command_p1_ls(
    int		argc,
    const char* argv[])
{
    char partname[256];
    char dirname[256];
    unsigned char * dest = (unsigned long *) 0xc0008000;
    const char *kernel_part_name = NULL;

    get_param_value("kernel_partition", &kernel_part_name);
   
    if (argc > 1) {
	strcpy(dirname,argv[1]);	
    }
    else{
	strcpy(dirname,"/");
    }
    if (argc > 2) {
	strcpy(partname,argv[2]);	
    }
    else{
	strcpy(partname,kernel_part_name);
    }

    body_p1_ls(dirname,partname);
}




long
body_p1_load_file(char *partname,char *filename,unsigned char *dest)
{
    struct part_info part;
    int i;
    int size;
    struct FlashRegion *flashRegion= btflash_get_partition(partname);
    unsigned long ret;

  
  
    if (!flashRegion) {
	putstr("could not find partition "); putstr(partname); putstr("\r\n");
	return -1;
    }
    part.size = flashRegion->size;
    /* for uniformly sized flash sectors */
    part.erasesize = flashDescriptor->sectors[1] - flashDescriptor->sectors[0];
    part.offset = ((char*)flashword) + flashRegion->base;
    
    ret = jffs2_1pass_load(dest,&part,filename);
#if 0
    putLabeledWord("loaded file of size =", ret);    
#endif

    return ret;
    
}


void
command_p1_load_file(
    int		argc,
    const char* argv[])
{
    char partname[256];
    char filename[256];
    unsigned char * dest = (unsigned long *) 0xc0008000;
    
    if (argc > 1) {
	strcpy(partname,argv[1]);	
    }
    else{
	strcpy(partname,"root");
    }
    if (argc > 2) {
	strcpy(filename,argv[2]);	
    }
    else{
	strcpy(filename,"cow");
    }

    if (argc > 3) {
	dest = strtoul(argv[3], NULL, 0);
	if (strtoul_err) {
	    putstr("error parsing <dest>\r\n");
	    return;
	}
    }

    
    body_p1_load_file(partname,filename,dest);
}



void
body_clearMem(unsigned long num,unsigned short *dst)
{
    unsigned short *ldst = dst;    
    int i;
    
    for (i=0; i < num/2; i++)
	*ldst++ = 0x0000;
}

void
command_clearMem(
    int		argc,
    const char* argv[])
{

    unsigned long num = 310410; //=644*482+2
    unsigned short *dst = (unsigned short *)0xc0000000;

    
    if (argc > 1) {
	num = (unsigned long) strtoul(argv[1], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\n\r");
	    return;
	}
    }
    if (argc > 2) {
	dst = (unsigned short *) strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\n\r");
	    return;
	}
    }


    
    body_clearMem(num,dst);
    

}



void
body_cmpKernels(char *partname,unsigned long *dstFlash,unsigned long *srcJFFS,unsigned long len)
{
  struct part_info part;
  int i;
  int size;
  const struct kernel_loader *the_loader = NULL;
  struct FlashRegion *flashRegion = btflash_get_partition(partname);
  unsigned long ret;
  unsigned char *pF = (unsigned char *) dstFlash;
  unsigned char *pJ = (unsigned char *) srcJFFS;
  
  
  if (!flashRegion){
      putstr("invalid partition ");putstr(partname);putstr("\r\n");
      return;
  }
  
      
      
  // copy the flash to ram
  putstr("copying Linux kernel ... ");
  memcpy((void*)(dstFlash),
	 (void*)flashRegion->base, flashRegion->size);
  putstr("done\r\n");
  
  // now do a bytewise compare.
  for (i=0; i<len;i++){
      if (*pF++ != *pJ++){
	  putLabeledWord("cmp failed at i =", (long) i);
	  putLabeledWord("pF =", (long) pF-1);
	  putLabeledWord("*pF =", (long) *(pF-1));
	  putLabeledWord("pJ =", (long) pJ-1);
	  putLabeledWord("*pJ =", (long) *(pJ-1));
	  return;	  
      }
  }
  putLabeledWord("kernels match up to ", len);         
}



void
command_cmpKernels(
    int		argc,
    const char* argv[])
{
    char *filename;
    unsigned long * dst;
    unsigned long * src;
    unsigned long len;
    char *partname;

    if (argc > 1)
	partname = argv[1];
    else{
	putstr("which kernel partition?\n\r");
	return;
    }

    if (argc > 2){
	
        dst = (void*)strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\n\r");
	    return;
	}
    }
    if (argc > 3){
	
        src = (void*)strtoul(argv[3], NULL, 0);
	if (strtoul_err) {
	    putstr("bad src param\n\r");
	    return;
	}
    }
    if (argc > 4){
	
        len = (void*)strtoul(argv[4], NULL, 0);
	if (strtoul_err) {
	    putstr("bad len param\n\r");
	    return;
	}
    }
    
    body_cmpKernels(partname,dst,src,len);
}

// This aviods the gcc assembler cache settings and seems to work more reliably.
//we should pull C-gccasm enabler if this proves solid

void
command_enable_caches(
    int		argc,
    const char* argv[])
{
    int   icache_en = 0;
    int   dcache_en = 0;
    unsigned long ret;
    
    if (argc > 1) {
	dcache_en = (int) strtoul(argv[1], NULL, 0);
	if (strtoul_err) {
	    putstr("bad  dcache param\n\r");
	    return;
	}
    }

    if (argc > 2) {
	icache_en = (int) strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad icache param\n\r");
	    return;
	}
    }

    ret = asmEnableCaches((unsigned long) dcache_en,(unsigned long) icache_en);
    putLabeledWord("  asmEnableCaches = ",ret);

}


void parseParamName(int argc,const char **argv,char *pName,char *pValue)
{
    int i;
    int len = 0;
    char *pAll;
    char *p;
    unsigned char canKill = 0;
    unsigned char valueSearch = 0;
    

    for (i=1; i < argc; i++){
#if 0
	putLabeledWord("i = ",i);
	putstr("argv = <");
	putstr(argv[i]);
	putstr(">\r\n");
#endif
	len+= strlen(argv[i]);
    }


    if ((pAll = (char *) mmalloc((len+1) * sizeof(char))) == NULL){
	putstr("out of room to build params list!\r\n");
	return;
    }
    pAll[0] = '\0';
    p = pAll;    
    for (i=1; i < argc; i++){
	strcpy(p,argv[i]);
	p += strlen(argv[i]);
	strcpy(p," ");
	p += 1;
    }
    // kill the last unneeded space
    *--p ='\0';
    

#if 0
    putstr("pAll = <");
    putstr(pAll);
    putstr(">\r\n");
#endif

    // eliminate = if it's there
    // we only kill the first one and only if its 
    p = pAll;
    while((p - pAll) < (strlen(argv[1]) + strlen(argv[2])+1)){

	if (canKill && (isblank(*p) || (*p == '=')))
	    valueSearch = 1;
	if (!canKill && isalnum(*p))
	    canKill = 1;
	else if (valueSearch && isalnum(*p))
	    break;
	if (canKill && (*p == '='))
	    *p = ' ';
	p++;
    }
    
#if 0
    putstr("pAll_post = <");
    putstr(pAll);
    putstr(">\r\n");
#endif

    p = pAll;
    while ((*p != ' ') && (*p != '\t'))
	*p++;
    *p++ = '\0';
    
    while ((*p == ' ') || (*p == '\t'))
	*p++;
    strcpy(pName,pAll);
    strcpy(pValue,p);
    
    free(pAll);
    return;
    
}

// this builds up the command buffer and either processes or prints it.
void parseParamFile(unsigned char *src,unsigned long size,int just_show)
{
    unsigned char *p = src;
    char cmdbuf[1024];
    char* cmdp;

    while ((p-src) < size){
	cmdp = cmdbuf;
	while (((p-src) < size) && (*p != '\r') && (*p != '\n'))
	    *cmdp++ = *p++;
	*cmdp = '\0';
	putstr("+ ");
	putstr(cmdbuf);
	putstr("\r\n");
	while ((*p == '\r') || (*p == '\n'))
	    p++;
	if (!just_show)
	    exec_string(cmdbuf);
    }
}

void command_pef(int argc, const char* argv[])
{
    int just_show = 0;

    if (argc > 1) 
	just_show = strtoul(argv[1],NULL,0);	

    params_eval_file(just_show);
}

void command_lli(int argc, const char* argv[])
{
    putLabeledWord("ARCH INFO(CPR0)=", readCPR0());
    putLabeledWord("MMU Control (CPR1)=", readCPR1());
    putLabeledWord("TRANSLATION TABLE BASE (CPR2)=", readCPR2());
    putLabeledWord("DOMAIN ACCESS CTL (CPR3)=", readCPR3());
    putLabeledWord("FAULT STATUS (CPR5)=", readCPR5());
    putLabeledWord("FAULT ADDRESS (CPR6)=", readCPR6());
    putLabeledWord("MMU PROC ID  (CPR13)=", readCPR13());
    putLabeledWord("BREAKPOINT  (CPR14)=", readCPR14());
    
}
	
void command_cat(int argc, const char* argv[])
{
    char partname[256];
    char filename[256];
    unsigned char * dest = (unsigned long *) 0xc0008000;
    long kernel_in_ram = 0;
    long size = 0;
    long i;
    
    get_param_value("kernel_in_ram", &kernel_in_ram);

    if (argc > 1) {
	strcpy(filename,argv[1]);	
    }
    else{
	putstr("cat what file?\r\n");
    }

    
    if (argc > 2) {
	strcpy(partname,argv[2]);	
    }
    else{
	strcpy(partname,"root");
    }

    size = body_p1_load_file(partname,filename,kernel_in_ram);
    for (i=0; i < size; i++){
	if (*((char *)(kernel_in_ram + i)) == '\n')
	    putc('\r');
	putc(*((char *)(kernel_in_ram + i)));
    }
}


unsigned char isWincePartition(unsigned char *src)
{
    unsigned long *p = (unsigned long *)src;
    unsigned char ret = 1;
    int i;
    
       
    //putLabeledWord("*p = ",*p);
    //putLabeledWord("WPM1 = ",WINCE_PARTITION_MAGIC_1);
    if (*p++ != WINCE_PARTITION_MAGIC_1)
	ret = 0;
    for (i=0; i < WINCE_PARTITION_LONG_0;i++){
	if (*p++ != 0x00000000)
	    ret = 0;
    }
    //putLabeledWord("*p = ",*p);
    //putLabeledWord("WPM2 = ",WINCE_PARTITION_MAGIC_2);
    if (*p++ != WINCE_PARTITION_MAGIC_2)
	ret = 0;
    
    
    return ret;
}
