/****************************************************************************/
/* Copyright 2000 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * btflash.c - Flash erase and program routines for Compaq Personal Server Monitor
 *
 */


#include "btflash.h"

#define FLASH_TIMEOUT 20000000

int btflash_verbose = 0;

/* default to physical address, update this variable when MMU gets changed */
volatile unsigned long *flashword = (volatile unsigned long*)FLASH_BASE;
#define doubleword(w_) ((((w_)&0xFFFF) << 16)|((w_)&0xFFFF))

#ifdef CONFIG_AMD_FLASH
static unsigned long flashSectors_Am29LV160BB[] = {
  0x00000000, /* first sector of bootldr */
  0x00008000, /* last sector of bootldr */
  0x0000c000, /* param sector */
  0x00010000, /* kernel starts here */
  0x00020000,
  0x00040000,
  0x00060000,
  0x00080000,
  0x000A0000,
  0x000C0000,
  0x000E0000,
  0x00100000,
  0x00120000,
  0x00140000,
  0x00160000,
  0x00180000,
  0x001A0000,
  0x001C0000,
  0x001E0000,
  0x00200000,
  0x00220000,
  0x00240000,
  0x00260000,
  0x00280000,
  0x002A0000,
  0x002C0000,
  0x002E0000,
  0x00300000, /* user code starts here */
  0x00320000,
  0x00340000,
  0x00360000,
  0x00380000,
  0x003A0000,
  0x003C0000,
  0x003E0000,
  0x00400000 /* guard sector -- not in flash */
};


static int amdFlashReset(void);
static int amdFlashProgramWord(unsigned long flashAddress, unsigned long value);
static int amdFlashProgramBlock(unsigned long flashAddress, unsigned long *values, int nbytes);
static int amdFlashEraseChip(void);
static int amdFlashEraseSector(unsigned long sectorAddress);
static int amdFlashEraseRange(unsigned long start, unsigned long len);
static int amdFlashProtectRange(unsigned long start, unsigned long len, int protect);

static FlashAlgorithm amdFlashAlgorithm = {
   amdFlashReset,
   amdFlashProgramWord,
   amdFlashProgramBlock,
   amdFlashEraseChip,
   amdFlashEraseSector,
   amdFlashEraseRange,
   amdFlashProtectRange
};

static struct FlashDescriptor flashDescriptor_Am29LV160BB = {
   "Am29LV160BB",
   1, 0x2249, 
   &amdFlashAlgorithm,
   sizeof(flashSectors_Am29LV160BB)/sizeof(dword) - 1, flashSectors_Am29LV160BB,
   { "bootldr",   0x00000000, 0x0000c000, LFR_PATCH_BOOTLDR },
   { "params",    0x0000c000, 0x00004000, 0 },
   { "kernel",    0x00010000, 0x00200000, LFR_KERNEL },
   { "ramdisk",   0x00100000, 0x00300000, LFR_SIZE_PREFIX },
   { "usercode",  0x00300000, 0x00100000, LFR_SIZE_PREFIX },
   { "altkernel", 0x00300000, 0x00100000, LFR_KERNEL }
};
#endif /* CONFIG_AMD_FLASH */

#ifdef CONFIG_AMD_FLASH
static unsigned long flashSectors_Am29DL323CT[] = {
    0x00000000,
    0x00020000,
    0x00040000,
    0x00060000,
    0x00080000,
    0x000a0000,
    0x000c0000,
    0x000e0000,
    0x00100000,
    0x00120000,
    0x00140000,
    0x00160000,
    0x00180000,
    0x001a0000,
    0x001c0000,
    0x001e0000,
    0x00200000,
    0x00220000,
    0x00240000,
    0x00260000,
    0x00280000,
    0x002a0000,
    0x002c0000,
    0x002e0000,
    0x00300000,
    0x00320000,
    0x00340000,
    0x00360000,
    0x00380000,
    0x003a0000,
    0x003c0000,
    0x003e0000,
    0x00400000,
    0x00420000,
    0x00440000,
    0x00460000,
    0x00480000,
    0x004a0000,
    0x004c0000,
    0x004e0000,
    0x00500000,
    0x00520000,
    0x00540000,
    0x00560000,
    0x00580000,
    0x005a0000,
    0x005c0000,
    0x005e0000,
    0x00600000,
    0x00620000,
    0x00640000,
    0x00660000,
    0x00680000,
    0x006a0000,
    0x006c0000,
    0x006e0000,
    0x00700000,
    0x00720000,
    0x00740000,
    0x00760000,
    0x00780000,
    0x007a0000,
    0x007c0000,
    0x007e0000,
    0x007e4000,
    0x007e8000,
    0x007ec000,
    0x007f0000,
    0x007f4000,
    0x007f8000,
    0x007fc000,
    0x00800000 /* guard entry */
};

static struct FlashDescriptor flashDescriptor_Am29DL323CT =  {
   "Am29DL323CT", 
   1, 0x2250, 
   &amdFlashAlgorithm,
   sizeof(flashSectors_Am29DL323CT)/sizeof(dword) - 1, flashSectors_Am29DL323CT,
   { "bootldr",   0x00000000, 0x00020000, LFR_PATCH_BOOTLDR },
   { "params",    0x007f4000, 0x00004000, 0 },
   { "kernel",    0x00020000, 0x000e0000, LFR_KERNEL },
   { "ramdisk",   0x00100000, 0x00300000, LFR_SIZE_PREFIX },
   { "usercode",  0x00100000, 0x00600000, 0 }, /* overlaps ramdisk */
   { "altkernel", 0x00700000, 0x000f0000, LFR_KERNEL },
   { "debugger",  0x00800000, 0x00000000, LFR_KERNEL }
};

static unsigned long flashSectors_Am29DL323CB[] = {
    0x00000000,
    0x00020000,
    0x00040000,
    0x00060000,
    0x00080000,
    0x000a0000,
    0x000c0000,
    0x000e0000,
    0x00100000,
    0x00120000,
    0x00140000,
    0x00160000,
    0x00180000,
    0x001a0000,
    0x001c0000,
    0x001e0000,
    0x00200000,
    0x00220000,
    0x00240000,
    0x00260000,
    0x00280000,
    0x002a0000,
    0x002c0000,
    0x002e0000,
    0x00300000,
    0x00320000,
    0x00340000,
    0x00360000,
    0x00380000,
    0x003a0000,
    0x003c0000,
    0x003e0000,
    0x00400000,
    0x00420000,
    0x00440000,
    0x00460000,
    0x00480000,
    0x004a0000,
    0x004c0000,
    0x004e0000,
    0x00500000,
    0x00520000,
    0x00540000,
    0x00560000,
    0x00580000,
    0x005a0000,
    0x005c0000,
    0x005e0000,
    0x00600000,
    0x00620000,
    0x00640000,
    0x00660000,
    0x00680000,
    0x006a0000,
    0x006c0000,
    0x006e0000,
    0x00700000,
    0x00720000,
    0x00740000,
    0x00760000,
    0x00780000,
    0x007a0000,
    0x007c0000,
    0x007e0000,
    0x00800000 /* guard entry */
};

static struct FlashDescriptor flashDescriptor_Am29DL323CB=  {
   "Am29DL323CB", 
   1, 0x2250, 
   &amdFlashAlgorithm,
   sizeof(flashSectors_Am29DL323CB)/sizeof(dword) - 1, flashSectors_Am29DL323CB,
   { "bootldr",   0x00000000, 0x00020000, LFR_PATCH_BOOTLDR },
   { "params",    0x007e0000, 0x00004000, 0 },
   { "kernel",    0x00020000, 0x000e0000, LFR_KERNEL },
   { "ramdisk",   0x00100000, 0x00400000, LFR_SIZE_PREFIX },
   { "usercode",  0x00100000, 0x00080000, 0 }, /* overlaps ramdisk */
   { "altkernel", 0x00600000, 0x00200000, LFR_KERNEL },
   { "debugger",  0x00700000, 0x00080000, LFR_KERNEL }
};
#endif /* CONFIG_AMD_FLASH */


#ifdef CONFIG_INTEL_FLASH
static unsigned long flashSectors_28F128J3A[] = {
  0x00000000, 
  0x00040000, 
  0x00080000, 
  0x000c0000, 
  0x00100000, 
  0x00140000, 
  0x00180000, 
  0x001c0000, 
  0x00200000, 
  0x00240000, 
  0x00280000, 
  0x002c0000, 
  0x00300000, 
  0x00340000, 
  0x00380000, 
  0x003c0000, 
  0x00400000, 
  0x00440000, 
  0x00480000, 
  0x004c0000, 
  0x00500000, 
  0x00540000, 
  0x00580000, 
  0x005c0000, 
  0x00600000, 
  0x00640000, 
  0x00680000, 
  0x006c0000, 
  0x00700000, 
  0x00740000, 
  0x00780000, 
  0x007c0000, 
  0x00800000, 
  0x00840000, 
  0x00880000, 
  0x008c0000, 
  0x00900000, 
  0x00940000, 
  0x00980000, 
  0x009c0000, 
  0x00a00000, 
  0x00a40000, 
  0x00a80000, 
  0x00ac0000, 
  0x00b00000, 
  0x00b40000, 
  0x00b80000, 
  0x00bc0000, 
  0x00c00000, 
  0x00c40000, 
  0x00c80000, 
  0x00cc0000, 
  0x00d00000, 
  0x00d40000, 
  0x00d80000, 
  0x00dc0000, 
  0x00e00000, 
  0x00e40000, 
  0x00e80000, 
  0x00ec0000, 
  0x00f00000, 
  0x00f40000, 
  0x00f80000, 
  0x00fc0000, 
  0x01000000, 
  0x01040000, 
  0x01080000, 
  0x010c0000, 
  0x01100000, 
  0x01140000, 
  0x01180000, 
  0x011c0000, 
  0x01200000, 
  0x01240000, 
  0x01280000, 
  0x012c0000, 
  0x01300000, 
  0x01340000, 
  0x01380000, 
  0x013c0000, 
  0x01400000, 
  0x01440000, 
  0x01480000, 
  0x014c0000, 
  0x01500000, 
  0x01540000, 
  0x01580000, 
  0x015c0000, 
  0x01600000, 
  0x01640000, 
  0x01680000, 
  0x016c0000, 
  0x01700000, 
  0x01740000, 
  0x01780000, 
  0x017c0000, 
  0x01800000, 
  0x01840000, 
  0x01880000, 
  0x018c0000, 
  0x01900000, 
  0x01940000, 
  0x01980000, 
  0x019c0000, 
  0x01a00000, 
  0x01a40000, 
  0x01a80000, 
  0x01ac0000, 
  0x01b00000, 
  0x01b40000, 
  0x01b80000, 
  0x01bc0000, 
  0x01c00000, 
  0x01c40000, 
  0x01c80000, 
  0x01cc0000, 
  0x01d00000, 
  0x01d40000, 
  0x01d80000, 
  0x01dc0000, 
  0x01e00000, 
  0x01e40000, 
  0x01e80000, 
  0x01ec0000, 
  0x01f00000, 
  0x01f40000, 
  0x01f80000, 
  0x01fc0000, 
  0x02000000 /* guard sector */
};

static int intelFlashReset(void);
static int intelFlashProgramBlock(unsigned long flashAddress, unsigned long *values, int nbytes);
static int intelFlashProgramWord(unsigned long flashAddress, unsigned long value);
static int intelFlashEraseChip(void);
static int intelFlashEraseSector(unsigned long sectorAddress);
static int intelFlashEraseRange(unsigned long start, unsigned long len);
static int intelFlashProtectRange(unsigned long start, unsigned long len, int protect);

static FlashAlgorithm intelFlashAlgorithm = {
   intelFlashReset,
   intelFlashProgramWord,
   intelFlashProgramBlock,
   intelFlashEraseChip,
   intelFlashEraseSector,
   intelFlashEraseRange,
   intelFlashProtectRange
};

static FlashDescriptor flashDescriptor_28F128J3A = {
   "28F128J3A", 
   0x89, 0x18, 
   &intelFlashAlgorithm,
   sizeof(flashSectors_28F128J3A)/sizeof(dword) - 1, flashSectors_28F128J3A,
   { "bootldr",   0x00000000, 0x00040000, LFR_PATCH_BOOTLDR },
   { "params",    0x000c0000, 0x00040000, 0 },
   { "kernel",    0x00040000, 0x00080000, LFR_KERNEL },
   { "ramdisk",   0x00100000, 0x00700000, LFR_SIZE_PREFIX },
   { "usercode",  0x00100000, 0x00700000, 0 }, /* for flash cramfs */
   { "altkernel", 0x00900000, 0x01100000, LFR_KERNEL },
   { "debugger",  0x00900000, 0x01100000, 0 }
};

#endif /* CONFIG_INTEL_FLASH */


static FlashDescriptor *flashDescriptors[] = {
#ifdef CONFIG_AMD_FLASH
   &flashDescriptor_Am29LV160BB,
   &flashDescriptor_Am29DL323CT,
   &flashDescriptor_Am29DL323CB,
#endif /* CONFIG_AMD_FLASH */
#ifdef CONFIG_BITSY
   &flashDescriptor_28F128J3A,
#endif /* CONFIG_BITSY */
   NULL
};

long flash_size = 0;
long flash_address_mask = -1;
int nsectors = 0;
unsigned long *flashSectors = NULL;
FlashDescriptor *flashDescriptor = NULL;

static int checkFlashDescriptor(const char *where)
{
   if (flashDescriptor == NULL) {
      putstr("No flash descriptor: "); putstr(where); putstr("\r\n");
      return -1;
   } else {
      return 0;
   }
}

int resetFlash ()
{
   if (!checkFlashDescriptor("reset flash")) {
      return flashDescriptor->algorithm->reset();
   } else {
      return 1;
   }
}

/*
 * Programs value at flashAddress
 * Sectors must be erased before they can be programmed.
 */
int programFlashWord(unsigned long flashAddress, unsigned long value)
{
   if (!checkFlashDescriptor("program flash word")) {
      return flashDescriptor->algorithm->programWord(flashAddress, value);
   } else {
      return 1;
   }
}

/*
 * Programs block of values starting at flashAddress
 * Sectors must be erased before they can be programmed.
 */
int programFlashBlock(unsigned long flashAddress, unsigned long *values, int nbytes)
{
   if (!checkFlashDescriptor("program flash word")) {
      return flashDescriptor->algorithm->programBlock(flashAddress, values, nbytes);
   } else {
      return 1;
   }
}

int eraseFlashChip ()
{
   if (!checkFlashDescriptor("erase flash chip")) {
      return flashDescriptor->algorithm->eraseChip();
   } else {
      return 1;
   }
}

/* sectorAddress must be a valid start of sector address. sectors
   must be erased before they can be programmed! */
int eraseFlashSector (unsigned long sectorAddress)
{
   if (!checkFlashDescriptor("erase flash sector")) {
      return flashDescriptor->algorithm->eraseSector(sectorAddress);
   } else {
      return 1;
   }
}

int eraseFlashRange(unsigned long startAddress, unsigned long len)
{
   if (!checkFlashDescriptor("erase flash range")) {
      return flashDescriptor->algorithm->eraseRange(startAddress, len);
   } else {
      return 1;
   }
}

int protectFlashRange(unsigned long startAddress, unsigned long len, int protect)
{
   if (!checkFlashDescriptor("erase flash range")) {
      return flashDescriptor->algorithm->protectRange(startAddress, len, protect);
   } else {
      return 1;
   }
}



unsigned long queryFlash(unsigned long flashWordAddress)
{
  /* there's a reason why we don't shift flashWordOffset by 2
   * here... we want the values we pass to queryFlash match the values on pages 15-17 of the spec.
   */
   unsigned long result;
   unsigned long flashWordOffset = flashWordAddress&flash_address_mask;

   /* put flash in query mode */
   flashword[0x55] = doubleword(0x98);
   result = flashword[flashWordOffset];
   /* reset flash */
   flashword[0x55] = doubleword(0xFF);
   return result;
}

static FlashAlgorithm *flashAlgorithm = NULL;
int updateFlashAlgorithm()
{
  int algorithm = queryFlash(0x13) & 0xFF;
  switch (algorithm) {
#ifdef CONFIG_INTEL_FLASH
  case 1:
    flashAlgorithm = &intelFlashAlgorithm;
    break;
#endif
#ifdef CONFIG_AMD_FLASH
  case 2:
    flashAlgorithm = &amdFlashAlgorithm;
    break;
#endif
  default:
    putLabeledWord("No flash algorithm known for CFI vendorID=", algorithm);
  }
  return algorithm;
}


unsigned long queryFlashID(unsigned long flashWordAddress)
{
   unsigned long result;
   unsigned long flashWordOffset = (flashWordAddress&flash_address_mask);
   int algorithm = queryFlash(0x13) & 0xFF;

   switch (algorithm) {
   case 1:
     /* reset flash -- Intel */
     flashword[0x55] = doubleword(0xFF);
     /* put flash in query mode */
     flashword[0x555] = doubleword(0x90);
     break;

   case 2:
     /* reset flash -- AMD */
     flashword[0x55] = doubleword(0xF0);
     /* put flash in query mode */
     flashword[0x555] = doubleword(0xAA);
     flashword[0x2AA] = doubleword(0x55);
     flashword[0x555] = doubleword(0x90);
     break;
   }

   /* read autoselect word */
   result = flashword[flashWordOffset];

   switch (algorithm) {
   case 1:
     /* reset flash -- Intel */
     flashword[0x55] = doubleword(0xFF);
     break;
   case 2:
     /* reset flash -- AMD */
     flashword[0x55] = doubleword(0xF0);
     break;
   }
   return result;
}

unsigned long queryFlashSecurity(unsigned long flashWordAddress)
{
  /* there's a reason why we don't shift flashWordOffset by 2
   * here... we want the values we pass to queryFlash match the values on pages 15-17 of the spec.
   */
   unsigned long result;
   unsigned long flashWordOffset = (flashWordAddress&flash_address_mask);

   /* reset flash */
   flashword[0x555] = doubleword(0xF0);
   flashword[0x555] = doubleword(0xF0);
   /* enter SecSi Sector Region */
   flashword[0x555] = doubleword(0xAA);
   flashword[0x2AA] = doubleword(0x55);
   flashword[0x555] = doubleword(0x88);

   /* read word from SecSi region */
   result = flashword[flashWordOffset];

   /* exit region */
   flashword[0x555] = doubleword(0xAA);
   flashword[0x2AA] = doubleword(0x55);
   flashword[0x555] = doubleword(0x90);
   flashword[0x000] = doubleword(0x00);

   return result;
}


#ifdef CONFIG_AMD_FLASH
int amdFlashReset ()
{
   /* send flash the reset command */
   flashword[0x55] = doubleword(0xF0);
   return 0;
}

/*
 * Programs value at flashAddress
 * Sectors must be erased before they can be programmed.
 */
static int amdFlashProgramWord(unsigned long flashAddress, unsigned long value)
{
   unsigned long flashWordOffset = (flashAddress&flash_address_mask) >> 2;
   long timeout = FLASH_TIMEOUT;
   unsigned long flashContents = flashword[flashWordOffset];
   unsigned long oldFlashContents;
   unsigned long hivalue = (value >> 16) & 0xFFFFl;
   unsigned long lovalue = (value >> 0) & 0xFFFFl;

   /* see if we can program the value without erasing */
   if ((flashContents & value) != value) {
      putstr("the flash sector needs to be erased first!\r\n");
      putLabeledWord("  flashAddress=", flashAddress);
      putLabeledWord("  flashWordOffset=", flashWordOffset);
      putLabeledWord("  &flashword[flashWordOffset]=", 
		     (dword)&flashword[flashWordOffset]);
      putLabeledWord("  flashContents=", flashContents);
      putLabeledWord("          value=", value);
   }
   
   /* send flash the program word command */
   flashword[0x555] = doubleword(0xAA);
   flashword[0x2AA] = doubleword(0x55);
   flashword[0x555] = doubleword(0xA0);
   flashword[flashWordOffset] = value;
   /* now wait for it to be programmed */
   while (((flashContents = flashword[flashWordOffset]) != value) &&
	  (timeout > 0)) {
      unsigned long hiword = (flashContents >> 16) & 0xFFFFl;
      unsigned long loword = (flashContents >> 0) & 0xFFFFl;
      if (0 && (hiword != hivalue) && (hiword & (1 << 5))) {
         /* programming upper bank of flash timed out */
         putstr("Upper bank of flash timed out!!");
	 timeout = 0;
         break;
      }
      if (0 && (loword != lovalue) && (loword & (1 << 5))) {
         /* programming upper bank of flash timed out */
         putstr("Lower bank of flash timed out!!");
	 timeout = 0;
         break;
      }
      oldFlashContents = flashContents;
      timeout--;
   }
   if (timeout <= 0) {
      putstr("programFlashWord timeout\r\n");
      putLabeledWord("  flashAddress=", flashAddress);
      putLabeledWord("  value=", value);
      putLabeledWord("  flashContents=", flashContents);
      putLabeledWord("  oldFlashContents=", oldFlashContents);
      return(-1);
   }
   return 0;
}

static int amdFlashProgramBlock(unsigned long flashAddress, unsigned long *values, int nbytes)
{
   int nwords = nbytes >> 2;
   int result = 0;
   int i;
   for (i = 0; i < nwords; i++) {
      result |= amdFlashProgramWord(flashAddress + (i*4), values[i]);
      if (result)
         break;
   }
   return result;
}

int amdFlashEraseChip ()
{
   int i;
   long timeout = FLASH_TIMEOUT;
   unsigned long flashWordOffset = 0;
   unsigned long flashContents;
   unsigned long oldFlashContents;
   const unsigned long hivalue = 0xFFFFl; /* when chip is erased, this is what we should read */
   const unsigned long lovalue = 0xFFFFl; /* when chip is erased, this is what we should read */
   
   flashword[0x555] = doubleword(0xAA);
   flashword[0x2AA] = doubleword(0x55);
   flashword[0x555] = doubleword(0x80);
   flashword[0x555] = doubleword(0xAA);
   flashword[0x2AA] = doubleword(0x55);
   flashword[0x555] = doubleword(0x10);
   while (((flashContents = flashword[flashWordOffset]) != 0xFFFFFFFFL) &&
          (timeout > 0)) {
      unsigned long hiword = (flashContents >> 16) & 0xFFFFl;
      unsigned long loword = (flashContents >> 0) & 0xFFFFl;
      if (0 && (hiword != hivalue) && (hiword & (1 << 5))) {
         /* programming upper bank of flash timed out */
         putstr("Upper bank of flash timed out!!");
         break;
      }
      if (0 && (loword != lovalue) && (loword & (1 << 5))) {
         /* programming upper bank of flash timed out */
         putstr("Lower bank of flash timed out!!");
         break;
      }
      oldFlashContents = flashContents; /* to check for toggle bits */
      timeout--;
   }
   if (timeout <= 0) {
      putstr("eraseFlashChip timeout\r\n");
      putLabeledWord("  flashp=", (dword)(&flashword[flashWordOffset]));
      putLabeledWord("  flashContents=", flashContents);
      putLabeledWord("  oldFlashContents=", oldFlashContents);
      return(-1);
   }
   return 0;
}

/* sectorAddress must be a valid start of sector address. sectors
   must be erased before they can be programmed! */
static int amdFlashEraseSector (unsigned long sectorAddress)
{
   int i;
   long timeout = FLASH_TIMEOUT;
   unsigned long flashWordOffset = (sectorAddress&flash_address_mask) >> 2;
   unsigned long flashContents;
   unsigned long oldFlashContents;
   const unsigned long hivalue = 0xFFFFl; /* when sector is erased, this is what we should read */
   const unsigned long lovalue = 0xFFFFl; /* when sector is erased, this is what we should read */
   
   for (i = 0; i < nsectors; i++) {
      if (flashSectors[i] == sectorAddress)
         break;
   }
   if (i >= nsectors) {
      putLabeledWord("eraseFlashSector: sectorAddress must be start of a sector! address=", sectorAddress);
      putLabeledWord("nsectors=", nsectors);
      return -1;
   }
   
   flashword[0x555] = doubleword(0xAA);
   flashword[0x2AA] = doubleword(0x55);
   flashword[0x555] = doubleword(0x80);
   flashword[0x555] = doubleword(0xAA);
   flashword[0x2AA] = doubleword(0x55);
   flashword[flashWordOffset] = doubleword(0x30);
   while (((flashContents = flashword[flashWordOffset]) != 0xFFFFFFFFL) &&
          (timeout > 0)) {
      unsigned long hiword = (flashContents >> 16) & 0xFFFFl;
      unsigned long loword = (flashContents >> 0) & 0xFFFFl;
      if (0 && (hiword != hivalue) && (hiword & (1 << 5))) {
         /* programming upper bank of flash timed out */
         putstr("Upper bank of flash timed out!!");
         break;
      }
      if (0 && (loword != lovalue) && (loword & (1 << 5))) {
         /* programming upper bank of flash timed out */
         putstr("Lower bank of flash timed out!!");
         break;
      }
      oldFlashContents = flashContents; /* to check for toggle bits */
      timeout--;
   }
   if (timeout <= 0) {
      putstr("eraseFlashSector timeout\r\n");
      putLabeledWord("  flashp=", (dword)(&flashword[flashWordOffset]));
      putLabeledWord("  flashAddress=", sectorAddress);
      putLabeledWord("  flashContents=", flashContents);
      putLabeledWord("  oldFlashContents=", oldFlashContents);
      putLabeledWord("  timeout=", timeout);
      return(-1);
   }
   return 0;
}

static int amdFlashEraseRange(unsigned long startAddress, unsigned long len)
{
  unsigned long lastSectorAddress = flashSectors[0];
  unsigned long limitAddress = startAddress + len;
  unsigned long sectorAddress, i;

  startAddress &= flash_address_mask;
  limitAddress &= flash_address_mask;

  for (i=0;i<nsectors;i++) {
     sectorAddress = flashSectors[i+1]; /* actually nsectors entries in this array -- last is fictitious guard sector */
    if ((lastSectorAddress <= startAddress) && (sectorAddress > startAddress)) {
      putLabeledWord("Erasing sector ",lastSectorAddress);
      if (eraseFlashSector(lastSectorAddress))
	return -1;
      len -= (sectorAddress - startAddress);
      startAddress = sectorAddress;
      if (startAddress >= limitAddress)
	break;
    }
    lastSectorAddress = sectorAddress;
  }
  return 0;
}

static int amdFlashProtectRange(unsigned long startAddress, unsigned long len, int protect)
{
   putstr("amdFlashProtectRange: unimplemented!!\r\n");
   return 0;
}
#endif CONFIG_AMD_FLASH




#ifdef CONFIG_INTEL_FLASH

int intelFlashReset ()
{
   /* send flash the reset command */
   /* address does not matter -- only the data */
   flashword[0x55] = doubleword(0xFF);
   return 0;
}

static int intelFlashReadStatus()
{
   /* address does not matter */
   flashword[0x55] = doubleword(0x70);
   return flashword[0x55];
}

static int intelFlashClearStatus()
{
   /* address does not matter */
   flashword[0x55] = doubleword(0x50);
}

/*
 * Programs value at flashAddress
 * Sectors must be erased before they can be programmed.
 */
static int intelFlashProgramWord(unsigned long flashAddress, unsigned long value)
{
   unsigned long flashWordOffset = (flashAddress&flash_address_mask) >> 2;
   long timeout = FLASH_TIMEOUT;
   unsigned long flashContents = flashword[flashWordOffset];
   unsigned long status = 0;

   /* see if we can program the value without erasing */
   if ((flashContents & value) != value) {
      putstr("the flash sector needs to be erased first!\r\n");
      putLabeledWord("  flashAddress=", flashAddress);
      putLabeledWord("  flashWordOffset=", flashWordOffset);
      putLabeledWord("  &flashword[flashWordOffset]=", 
		     (dword)&flashword[flashWordOffset]);
      putLabeledWord("  flashContents=", flashContents);
      return -1;
   }
   
   /* send flash the program word command */
   flashword[0x55] = doubleword(0x40);
   flashword[flashWordOffset] = value;
   /* now wait for it to be programmed */
   while (timeout > 0) {
      status = flashword[flashWordOffset];
      if ((status & doubleword(0x80)) == doubleword(0x80))
         break;
      timeout--;
   }
   intelFlashClearStatus();
   
   if ((timeout <= 0) || (status & doubleword(0x7f))) {
      putstr("programFlashWord error\r\n");
      putLabeledWord("  flashAddress=", flashAddress);
      putLabeledWord("  value=", value);
      putLabeledWord("  flashContents=", flashContents);
      putLabeledWord("  status=", status);
      return(-1);
   }
   return 0;
}

/* each flash chip has 32B block, 64B block for chip array */
static int intelFlashProgramBlock(unsigned long flashAddress, unsigned long *values, int nbytes)
{
   unsigned long flashWordOffset = (flashAddress&flash_address_mask) >> 2;
   unsigned long blockOffset = (flashWordOffset & 0xFFFFFFC0);
   int nwords = nbytes >> 2;
   int result = 0;
   long timeout = FLASH_TIMEOUT;
   unsigned long status = 0;
   int i;

   if (0) putLabeledWord("intelFlashProgramFlashBlock\r\n", flashAddress);

   /* send the "write to buffer" command */
   flashword[flashWordOffset] = doubleword(0xE8); 
   /* read the extended status register */
   do {
      status = flashword[flashWordOffset];
      if (0) putLabeledWord("  XSR=", status);
   } while ((status & 0x00800080) != 0x00800080);

   /* write word count at block start address */
   flashword[blockOffset] = doubleword(nwords-1); /* length minus one */

   /* send the data */
   for (i = 0; i < nwords; i++) {
      flashword[flashWordOffset+i] = values[i];
      if (0) putLabeledWord(" sr=", flashword[blockOffset]);
   }

   /* send the confirmation to program */
   flashword[blockOffset] = doubleword(0xD0);

   /* now wait for it to be programmed */
   timeout = FLASH_TIMEOUT;
   while (timeout > 0) {
      status = flashword[blockOffset];
      if (0) putLabeledWord("  status=", status);
      if ((status & doubleword(0x80)) == doubleword(0x80))
         break;
      timeout--;
   }
   status = intelFlashReadStatus();
   if (0) putLabeledWord("final status=", status);
   intelFlashClearStatus();
   
   if ((timeout <= 0) || (status & doubleword(0x7f))) {
      putstr("programFlashBlock error\r\n");
      putLabeledWord("  flashAddress=", flashAddress);
      putLabeledWord("  status=", status);
      return(-1);
   }
   return 0;
}

int intelFlashEraseChip ()
{
   int i;
   long timeout = FLASH_TIMEOUT;
   unsigned long flashWordOffset = 0;
   unsigned long flashContents;
   
   putstr("intelFlashEraseChip unimplemented\r\n");

   return 0;
}

/* sectorAddress must be a valid start of sector address. sectors
   must be erased before they can be programmed! */
static int intelFlashEraseSector (unsigned long sectorAddress)
{
   int i;
   long timeout = FLASH_TIMEOUT;
   unsigned long flashWordOffset = (sectorAddress&flash_address_mask) >> 2;
   unsigned long flashContents;
   unsigned long status;

   for (i = 0; i < nsectors; i++) {
      if (flashSectors[i] == sectorAddress)
         break;
   }
   if (i >= nsectors) {
      putLabeledWord("eraseFlashSector: sectorAddress must be start of a sector! address=", sectorAddress);
      putLabeledWord("nsectors=", nsectors);
      return -1;
   }

   /* send flash the erase sector command */
   flashword[0x55] = doubleword(0x20);
   flashword[flashWordOffset] = doubleword(0xD0);
   /* now wait for it to be programmed */
   while (timeout > 0) {
      status = flashword[flashWordOffset];
      if ((status & doubleword(0x80)) == doubleword(0x80))
         break;
      timeout--;
   }
   intelFlashClearStatus();
   flashContents = flashword[flashWordOffset];
   if ((timeout <= 0) || (status & doubleword(0x7f))) {
      putstr("eraseSector error\r\n");
      putLabeledWord("  sectorAddress=", sectorAddress);
      putLabeledWord("  flashContents=", flashContents);
      putLabeledWord("  status=", status);
      return(-1);
   }   

   return 0;
}

static int intelFlashEraseRange(unsigned long startAddress, unsigned long len)
{
  unsigned long lastSectorAddress = flashSectors[0];
  unsigned long limitAddress = startAddress + len;
  unsigned long sectorAddress, i;

  startAddress &= flash_address_mask;
  limitAddress &= flash_address_mask;

  for (i=0;i<nsectors;i++) {
     sectorAddress = flashSectors[i+1]; /* actually nsectors entries in this array -- last is fictitious guard sector */
    if ((lastSectorAddress <= startAddress) && (sectorAddress > startAddress)) {
      putLabeledWord("Erasing sector ",lastSectorAddress);
      putLabeledWord("  len=", len);
      if (eraseFlashSector(lastSectorAddress))
	return -1;
      len -= (sectorAddress - startAddress);
      startAddress = sectorAddress;
      if (startAddress >= limitAddress)
	break;
    }
    lastSectorAddress = sectorAddress;
  }
  return 0;
}

static int intelFlashProtectRange(unsigned long startAddress, unsigned long len, int protect)
{
  unsigned long lastSectorAddress = flashSectors[0];
  unsigned long limitAddress = startAddress + len;
  unsigned long sectorAddress, i;
  int result = 0;

  startAddress &= flash_address_mask;
  limitAddress &= flash_address_mask;

  if (protect) {
    for (i=0;i<nsectors;i++) {
      sectorAddress = flashSectors[i+1]; /* actually nsectors entries in this array -- last is fictitious guard sector */
      if ((lastSectorAddress <= startAddress) && (sectorAddress > startAddress)) {
	int status;
	putLabeledWord("Protecting sector ",lastSectorAddress);
	flashword[0] = doubleword(0x60);
	flashword[sectorAddress >> 2] = doubleword(0x01);
	status = intelFlashReadStatus();
	intelFlashClearStatus();
	if (status != doubleword(0x80)) {
	  return -1;
	}
	len -= (sectorAddress - startAddress);
	startAddress = sectorAddress;
	if (startAddress >= limitAddress)
	  break;
      }
      lastSectorAddress = sectorAddress;
    }
  } else {
    /* unprotects whole chip */
    flashword[0] = doubleword(0x60);
    flashword[0] = doubleword(0xd0);
    if (intelFlashReadStatus() != doubleword(0x80))
      result = -1;
    intelFlashClearStatus();
  }
  return result;
}

#endif /* CONFIG_INTEL_FLASH */


void btflash_print_types()
{
  int i;
  putstr("Flash types supported:\r\n");
  for (i = 0; flashDescriptors[i] != NULL; i++) {
    FlashDescriptor *fd = flashDescriptors[i];
    putstr("  "); putstr(fd->deviceName); putstr("\r\n");
  }
  putstr("Current flash type is "); putstr(flashDescriptor->deviceName); putstr("\r\n");
}

void btflash_set_type(const char *devicename)
{
   int i;
   for (i = 0; flashDescriptors[i] != NULL; i++) {
      FlashDescriptor *fd = flashDescriptors[i];
      if (strcmp(fd->deviceName, devicename) == 0) {
         putstr("setting flash type="); putstr(flashDescriptor->deviceName); putstr("\r\n");
         flashDescriptor = fd;

         nsectors = flashDescriptor->nsectors;
         flashSectors = flashDescriptor->sectors;
         flash_size = flashSectors[nsectors];
         flash_address_mask = flash_size-1;

         return;
      }
   }
   putstr("  Unknown flash device type: "); putstr(devicename); putstr("\r\n");
}


void btflash_init()
{
   int algorithm = updateFlashAlgorithm();
   int mfrid = queryFlashID(0x00);
   int devid = queryFlashID(0x01);
   int protected = queryFlashID(0x02);
   FlashDescriptor **fd = &flashDescriptors[0];

   putLabeledWord("btflash_init: mfrid=", mfrid);
   putLabeledWord("  devid=", devid);
   if ((devid&0xFFFF) != ((devid>>16)&0xFFFF)) {
#ifndef linux
      struct bootblk_param *noeraseParam = get_param("noerase");
      putLabeledWord("\r\n\r\n@@@@@@@@ mismatched flash parts @@@@@@@@\r\n\r\n", devid);
      if (noeraseParam != NULL)
         noeraseParam->value = 1;
#endif
      flashDescriptor = fd[0];
   }

   putstr("walking flash descriptors\r\n");
   for (flashDescriptor = *fd; 
        (flashDescriptor != NULL); 
        fd++, flashDescriptor = *fd) {
      if (doubleword(flashDescriptor->manufacturerID) == mfrid
          && doubleword(flashDescriptor->deviceID) == devid) {
         putstr("btflash_init: found flash "); putstr(flashDescriptor->deviceName); putstr("\r\n");
         nsectors = flashDescriptor->nsectors;
         flashSectors = flashDescriptor->sectors;
         flash_size = flashSectors[nsectors];
         flash_address_mask = flash_size-1;

         putLabeledWord("  flashDescriptor=", (long)flashDescriptor);
         putLabeledWord("  flashSectors=", (long)flashSectors);
         putLabeledWord("  nsectors=", nsectors);
         putLabeledWord("  flash_size=", flash_size);
         putLabeledWord("  flash_address_mask=", flash_address_mask);
         break;
      }
   }
}
