/****************************************************************************/
/* Copyright 2002 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * serial support file for Compaq Bootloadr
 *
 */

#include "bootldr.h"
#include "serial.h"
#include "buttons.h"
#include "sa1100.h"

int packetize_output = 0;
int getc_verbose_errors = 1;
int getc_errno = 0;

extern int get_pushed_char(void);
void serial_putnstr(struct serial_device *device,
                    const char	*str,
                    size_t	n);


#ifdef CONFIG_SKIFF
extern struct serial_device skiff_serial;
struct serial_device *serial = &skiff_serial;
#endif

#ifdef CONFIG_SA1100
int sa1100_char_ready(struct serial_device *sd) {
  return ((*(volatile long *)(sd->_serbase + SA1100_UTSR1)) & SA1100_UTSR1_RNE);
}

unsigned char sa1100_read_char(struct serial_device *sd) {
  return *(volatile unsigned char *)(sd->_serbase + SA1100_UTDR);
}

unsigned int sa1100_read_status(struct serial_device *sd) {
  return ((*(volatile byte *)(sd->_serbase + SA1100_UTSR1)) & SA1100_UTSR1_ERROR_MASK);
}

int sa1100_set_baudrate(struct serial_device *sd, int speed) {
  int brd;
  if (speed <= 115200) {
	brd = 23;
	while (speed > 9600) {
	  speed >>= 1;
	  brd++;
	}
	
	*(volatile byte *)(sd->_serbase + SA1100_UTCR3) = 0;
	*(volatile byte *)(sd->_serbase + SA1100_UTCR1) = brd >> 8;
	*(volatile byte *)(sd->_serbase + SA1100_UTCR2) = brd;
	
	/*
	 * Clear status register
	 */
	*(volatile byte *)(sd->_serbase + SA1100_UTSR0) = SA1100_UTSR0_REB | SA1100_UTSR0_RBB | SA1100_UTSR0_RID;
	*(volatile byte *)(sd->_serbase + SA1100_UTCR3) = SA1100_UTCR3_RIE | SA1100_UTCR3_RXE | SA1100_UTCR3_TXE;
	
   sd->_speed = speed;
  }
  return sd->_speed;
}

void *sa1100_serbase(struct serial_device *sd) {
  	return sd->_serbase;
}

void sa1100_write_char(struct serial_device *sd, unsigned char c) {
  if (!sd->enabled) return;
  while (!((*(volatile long *)(sd->_serbase + SA1100_UTSR1)) & SA1100_UTSR1_TNF)); /* wait until TX FIFO not full */
  *(byte *) (sd->_serbase + SA1100_UTDR) = c;
}

#endif // CONFIG_SA1100

#if defined(CONFIG_BITSY)

void
bitsy_putc(struct serial_device *sdevice,
		   char    c,
		   int	   encap_done)
{
  /*
   * silly person calling putc instead of putstr...
   * we encap the single char
   */
  
  if (!encap_done && packetize_output) {
	char	buf[2];
	buf[0] = c;
	buf[1] = '\0';
	serial_putnstr(sdevice, buf, 1);
  }
  else sa1100_write_char(sdevice, c);
}

void bitsy_write_char(struct serial_device *device, unsigned char c) {
  if (!device->enabled) return;
  bitsy_putc(device, c, 0);
}

/* bitsy serial is disabled by default, enabled if squelch serial is off */
static struct serial_device bitsy_serial = { 
  char_ready: sa1100_char_ready, 
  read_char: sa1100_read_char, 
  read_status: sa1100_read_status,
  write_char: bitsy_write_char, 
  set_baudrate: sa1100_set_baudrate, 
  serbase: sa1100_serbase,
  enabled: 0,
  _serbase: SA1100_UART3BASE
};

struct serial_device *serial = &bitsy_serial;

#elif defined(CONFIG_CONFIG_JORNADA720) || defined(CONFIG_JORNADA56X) 
static struct serial_device jornada_serial = { 
  char_ready: sa1100_char_ready, 
  read_char: sa1100_read_char, 
  read_status: sa1100_read_status,
  write_char: sa1100_write_char, 
  set_baudrate: sa1100_set_baudrate, 
  serbase: sa1100_serbase,
  enabled: 0,
  _serbase: SA1100_UART3BASE
};
#elif defined(CONFIG_ASSABET)

int assabet_char_ready() {
  return ((machine_has_neponset()  
		   ? (*(volatile long *)SA1100_UART3_UTSR1) 
		   : (*(volatile long *)SA1100_UART1_UTSR1))
		  & SA1100_UTSR1_RNE);
}

unsigned char assabet_read_char() {
  return (machine_has_neponset()
		  ? (*(volatile byte *)SA1100_UART3_UTDR) 
		  : (*(volatile byte *)SA1100_UART1_UTDR));
}

unsigned int assabet_read_status() {
  return ((machine_has_neponset()
		   ? (*(volatile long *)SA1100_UART3_UTSR1)
		   : (*(volatile long *)SA1100_UART1_UTSR1))
		  & SA1100_UTSR1_ERROR_MASK);
}

void assabet_write_char(struct serial_device *device, unsigned char c) {
  if (!device->enabled) return;
  while (!((machine_has_neponset() ? (*(volatile long *)SA1100_UART3_UTSR1) :
			(*(volatile long *)SA1100_UART1_UTSR1)) & SA1100_UTSR1_TNF));
  (machine_has_neponset() ? (*(byte *)SA1100_UART3_UTDR) :
   (*(byte *)SA1100_UART1_UTDR)) = c;
}

void *assabet_serbase() {
  return machine_has_neponset() ? Ser3Base : Ser1Base;
}
static struct serial_device assabet_serial = {
 assabet_char_ready, assabet_read_char, assabet_read_status, assabet_write_char, 0, 0, sa1100_set_baudrate, assabet_serbase, 1 
};


struct serial_device *serial = &assabet_serial;
#else
#error no architecture defined for SERIAL_CHAR_READY(), et al
#endif




void serial_putc(struct serial_device *device, char c) 
{
  if (!device->enabled) return;
  device->write_char(device, c);
}

byte serial_do_getc(struct serial_device *device,
    vfuncp	    idler,
    unsigned long   timeout,
    int*	    statp)
{
   byte c, rxstat;
   int	do_timeout = timeout != 0;
   int	ch;

   getc_errno = 0; /* reset errno */

   while (!device->char_ready(device)) {
       if ((ch = get_pushed_char()) != -1)
	   return (ch);
   
       if (do_timeout) {
	   if (!timeout)
	       break;
	   timeout--;
       }
       
       if (idler)
	   idler();
   }

   /* give priority to pushed chars */
   if ((ch = get_pushed_char()) != -1)
       return (ch);
   
   if (do_timeout && timeout == 0) {
       c = 0;
       rxstat = -1;
   }
   else {
       c = device->read_char(device);
       rxstat = device->read_status(device);
   }
   
   if (rxstat) {
      getc_errno = rxstat;
      if (getc_verbose_errors && !statp) {
         putLabeledWord("RXSTAT error: ", rxstat);
      }
      if (statp)
	  *statp = rxstat;
   }
   return(c);
}

byte
serial_getc(struct serial_device *device)
{
#ifdef CONFIG_SPOT
    return (serial_do_getc(device, spot_idle, 0, NULL));
#else
    return (serial_do_getc(device, button_check, 0, NULL));
#endif
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay iterations checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte
serial_awaitkey(struct serial_device *device, 
    unsigned long   delay,
    int*	    error_p)
{
    return (serial_do_getc(device, button_check, delay, error_p));
}

byte serial_do_getc_seconds(struct serial_device *device,
                     vfuncp	    idler,
                     unsigned long   timeout_seconds,
                     int*	    statp)
{
  byte c, rxstat;
  int	do_timeout = timeout_seconds != 0;
  int  timed_out = 0;
  unsigned long start_time = CTL_REG_READ(SA1100_RCNR);
  int	ch;

  getc_errno = 0; /* reset errno */

  while (!device->char_ready(device)) {
    if ((ch = get_pushed_char()) != -1)
      return (ch);
   
    if (do_timeout) {
      unsigned long time = CTL_REG_READ(SA1100_RCNR);
      if ((time - start_time) > timeout_seconds) {
        timed_out = 1;
        break;
      }
    }
       
    if (idler)
      idler();
  }

  /* give priority to pushed chars */
  if ((ch = get_pushed_char()) != -1)
    return (ch);
   
  if (do_timeout && timed_out) {
    c = 0;
    rxstat = -1;
  }
  else {
    c = device->read_char(device);
    rxstat = device->read_status(device);
  }
   
  if (rxstat) {
    getc_errno = rxstat;
    if (getc_verbose_errors && !statp) {
      putLabeledWord("RXSTAT error: ", rxstat);
    }
    if (statp)
      *statp = rxstat;
  }
  return(c);
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay seconds checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte
serial_awaitkey_seconds(struct serial_device *device,
    unsigned long   delay_seconds,
    int*	    error_p)
{
    return (serial_do_getc_seconds(device, button_check, delay_seconds, error_p));
}

void serial_do_putnstr(struct serial_device *device,
    const char* str,
    size_t	n)
{
   while (n && *str != '\0') {
      serial_putc(device, *str) ;
      str++ ;
      n--;
   }
}

void
serial_putnstr(struct serial_device *device,
    const char	*str,
    size_t	n)
{
   if (str == NULL)
      return;
   
#if defined(CONFIG_BITSY)
   if (packetize_output) {
       int  len;
       char len_str[16];

       len = strlen(str);
       if (n < len)
	   len = n;

       /* add the msg header */
       dwordtodecimal(len_str, len);
       serial_do_putnstr(device, "MSG/", 4);
       len = strlen(len_str);
       len_str[len] = '/';
       ++len;
       len_str[len] = '\0';
       serial_do_putnstr(device, len_str, len);
   }
#endif   

   serial_do_putnstr(device, str, n);
}

void serial_putstr(struct serial_device *device, const char *str)
{
    serial_putnstr(device, str, strlen(str));
}

void
serial_putstr_sync(struct serial_device *device, 
    const char*   s)
{
    serial_putstr(device, s);
    let_uart_drain(SA1100_UART3_UTSR1);
}

unsigned char getc() { return serial_getc(serial); }
void putc(char c) { serial_putc(serial, c); }
void putstr(const char *s) { serial_putstr(serial, s); }
void putnstr(const char *s, int n) { serial_putnstr(serial, s, n); }
byte awaitkey_seconds(unsigned long delay_seconds, int* error_p) {
  return serial_awaitkey_seconds(serial, delay_seconds, error_p);
}
void putstr_sync(const char* s) { serial_putstr_sync(serial, s); }
void do_putnstr(const char* s, int n) { serial_do_putnstr(serial, s, n); }
void putLabeledWord(const char *msg, unsigned long value)
{
   char buf[9];
   binarytohex(buf,value,4);
   putstr(msg);
   putstr(buf);
   putstr("\r\n");
}

void putLabeledAddrWord(const char *msg, unsigned long *value)
{
   char buf[9];

   putstr(msg);
   binarytohex(buf, (unsigned long) value, 4);
   putstr(" *0x" );
   putstr(buf);
   putstr( " == " );
   binarytohex(buf,*value,4);
   putstr(buf);
   putstr("\r\n");
}


void putHexInt32(unsigned long value)
{
  char buf[9];
  binarytohex(buf,value,4);
  putstr(buf);
}

void putHexInt16(word value)
{
  char buf[9];
  binarytohex(buf,value,2);
  putstr(buf);
}

void putHexInt8(byte value)
{
  char buf[3];
  binarytohex(buf,value,1);
  putstr(buf);
}

void putDecInt(unsigned long value)
{
  char buf[16];
  dwordtodecimal(buf, value);
  putstr(buf);
}

void putDecIntWidth(unsigned long value, unsigned int width)
{
  char buf[16];
  unsigned int length;

  dwordtodecimal(buf, value);

  for(length = strlen(buf); length < width; ++length)
    putc(' ');

  putstr(buf);
}

