/****************************************************************************/
/* Copyright 2002 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * flashif.c - Low level flash interface routines
 *
 */

#include "btflash.h"
#include "sa1100.h"
#include "params.h"

#ifdef CONFIG_BITSY
#include <asm/arch-sa1100/h3600_gpio.h>
#include <asm/arch-sa1100/h3600_asic.h>	
#endif

#define FLASH_TIMEOUT 20000000

/* default to physical address, update this variable when MMU gets changed */
volatile unsigned long *flashword = (volatile unsigned long*)FLASH_BASE;
#define bothbanks(w_) ((((w_)&0xFFFF) << 16)|((w_)&0xFFFF))


#ifdef CONFIG_BITSY
void set_vppen() {
    unsigned long mach_type = param_mach_type.value;
    
    switch (mach_type){
    case MACH_TYPE_H3100:
    case MACH_TYPE_H3600:
	set_egpio(EGPIO_BITSY_VPEN);
	break;
    case MACH_TYPE_H3800:
	//*((unsigned short *) 0x49001f00) = 0xf1e1;
	*((unsigned short *) H3800_FLASH_VPP_ADDR) = H3800_FLASH_VPP_ON;
    default:
	break;
    }
}

void clr_vppen() {
    unsigned long mach_type = param_mach_type.value;
    
    switch (mach_type){
    case MACH_TYPE_H3100:
    case MACH_TYPE_H3600:
	clr_egpio(EGPIO_BITSY_VPEN);
	break;
    case MACH_TYPE_H3800:
	// *((unsigned short *) 0x49001f00) = 0xf1e0;
	*((unsigned short *) H3800_FLASH_VPP_ADDR) = H3800_FLASH_VPP_OFF;
    default:
	break;
    }
}

#elif defined(CONFIG_JORNADA720)
void set_vppen()
{
    /* PPAR 0x00000000 PPDR 0x000049ff PPSR 0x00000870 */
    (*((volatile int *)SA1100_PPSR_REG)) |= 0x80;
    (*((volatile int *)SA1100_PPDR_REG)) |= 0x80;
}

void clr_vppen()
{
    (*((volatile int *)SA1100_PPSR_REG)) &= ~0x80;
}
#elif defined(CONFIG_JORNADA56X)
void set_vppen()
{
    SA1100_GPIO_GPDR_WRITE(0x0f5243fc);
    SA1100_GPIO_GPSR_WRITE(0x00100000);
    SA1100_GPIO_GPCR_WRITE(0x0f424000);
    SA1100_GPIO_GPSR_WRITE((1<<26));
}

void clr_vppen()
{
    SA1100_GPIO_GPCR_WRITE((1<<26));
}
#else
/* nothing to do here on Skiff */
void set_vppen() {
}
void clr_vppen() {
}
#endif



/*
 * XXX ugly hack.  we'll work this out to be more elegan later...
 */
/* two 16bit parts in parallel (original iPAQ) */
#define	BT_FLASH_ORGANIZATION_2x16	(1)

/* one single 16bit part (B&W iPAQ) */
#define	BT_FLASH_ORGANIZATION_1x16	(2)

static int  bt_flash_organization = BT_FLASH_ORGANIZATION_2x16;

#define	BTF_2x16()  (bt_flash_organization == BT_FLASH_ORGANIZATION_2x16)
#define	BTF_1x16()  (bt_flash_organization == BT_FLASH_ORGANIZATION_1x16)

#define	SET_BTF_2x16()  (bt_flash_organization = BT_FLASH_ORGANIZATION_2x16)
#define	SET_BTF_1x16()  (bt_flash_organization = BT_FLASH_ORGANIZATION_1x16)


int
flash_addr_shift(
    void)
{
    switch (bt_flash_organization) {
    case BT_FLASH_ORGANIZATION_2x16:
	return (2);
	break;
	    
    case BT_FLASH_ORGANIZATION_1x16:
	return (1);
	break;
	    
    default:
	putLabeledWord("flash_addr_shift(): "
		       "bad bt_flash_organization=",
		       bt_flash_organization);
	return (0);
	break;
    }

    return (-1);
}
    
unsigned long
flash_make_val(
    unsigned long   inval)
{
    switch (bt_flash_organization) {
    case BT_FLASH_ORGANIZATION_2x16:
	return (bothbanks(inval));
	break;
	    
    case BT_FLASH_ORGANIZATION_1x16:
	return (inval);
	break;
	    
    default:
	putLabeledWord("flash_make_val(): "
		       "bad bt_flash_organization=",
		       bt_flash_organization);
	return (0);
	break;
    }

    return (-1);
}

void
flash_write_cmd(
    unsigned	cmd_addr,
    int		cmd)
{
    switch (bt_flash_organization) {
    case BT_FLASH_ORGANIZATION_2x16:
	flashword[cmd_addr] = bothbanks(cmd);
	break;

    case BT_FLASH_ORGANIZATION_1x16:
#if 1
	putLabeledWord("fwc(1x16), Addr=0x",
		       (unsigned long)&((short*)flashword)[cmd_addr]);
#endif
	((short*)flashword)[cmd_addr] = cmd;
	break;

    default:
	putLabeledWord("flash_write_cmd(): bad bt_flash_organization=",
		       bt_flash_organization);
	break;
    }
}

unsigned long
flash_read_val(
    unsigned addr)
{
    unsigned long retval = 0;
    
    switch (bt_flash_organization) {
    case BT_FLASH_ORGANIZATION_2x16:
	retval = flashword[addr];
	break;

    case BT_FLASH_ORGANIZATION_1x16:
	retval = ((unsigned short*)flashword)[addr];
	break;

    default:
	putLabeledWord("flash_read_val(): bad bt_flash_organization=",
		       bt_flash_organization);
	break;
    }

    return (retval);
}

unsigned long
flash_read_array(
    unsigned	addr)
{
    char*   dst;

    /* force read array mode */
    flash_write_cmd(0x55, 0xff);
    
    switch (bt_flash_organization) {
    case BT_FLASH_ORGANIZATION_2x16:
	dst = (char*)flashword + (addr & ~0x3);
	return (*((unsigned long*)dst));

    case BT_FLASH_ORGANIZATION_1x16:
	dst = (char*)flashword + (addr & ~0x1);
#if 0
	putLabeledWord("fra(1x16), addr=0x", (unsigned long)dst);
#endif
	return (*((unsigned short*)dst));
	break;

    default:
	putLabeledWord("flash_read_array(): bad bt_flash_organization=",
		       bt_flash_organization);
	break;
    }
    return 0;
}

void
flash_write_val(
    unsigned	addr,
    int		val)
{
    char*   dst;
    
    switch (bt_flash_organization) {
    case BT_FLASH_ORGANIZATION_2x16:
	dst = (char*)flashword + (addr & ~0x3);
	*((unsigned long*)dst) = bothbanks(val);
	break;

    case BT_FLASH_ORGANIZATION_1x16:
	dst = (char*)flashword + (addr & ~0x1);
	*((unsigned short*)dst) = val;
	break;

    default:
	putLabeledWord("flash_write_val(): bad bt_flash_organization=",
		       bt_flash_organization);
	break;
    }
}

int
is_flash_16bit(void)
{
    int ret;
    /*
     * look at the memory control register for
     * the boot memory and see how wide it is.
     */
    ret = (CTL_REG_READ(SA1100_DRAM_CONFIGURATION_BASE+SA1100_MSC0) &
	   MSC_RBW16);
    if (ret)
	SET_BTF_1x16();
    return ret;
}

/* remove remaining direct accesses to flashword */
void nflash_write_cmd(unsigned long addr, unsigned long val)
{
    flashword[addr] = bothbanks(val);
}
void nflash_write_val( unsigned addr, int val)
{
    flashword[addr] = val;
}
unsigned long nflash_read_val( unsigned addr)
{
    return flashword[addr];
}
/* short read/write */
void flash_write_val_short( unsigned addr, int val)
{
    unsigned short* flashshort = (unsigned short*)flashword;
    flashshort[addr] = val;
}
unsigned long flash_read_val_short( unsigned addr)
{
    unsigned short* flashshort = (unsigned short*)flashword;
    return flashshort[addr];
}
