/****************************************************************************/
/* Copyright 2001 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * h3600 sleeve driver
 *
 */

#include "bootldr.h"
#ifdef __linux__
#include <asm-arm/setup.h>
#include <asm/arch-sa1100/h3600_gpio.h>
#include <asm/arch-sa1100/h3600_asic.h>	
#endif
#include "sa1100.h"
#include "bsdsum.h"
#include "architecture.h"
#include "hal.h"

#define __KERNEL__
#include "list.h"
#include <asm/arch/h3600-sleeve.h>

void msleep(unsigned int msec)
{
   int limit = msec * 3686;
   OSCR = 0;
   while (OSCR < limit)
      ;
}

static struct sleeve_dev *sleeve_dev;

/*******************************************************************************/


struct {
   char start_of_id; /* 0xaa */
   int data_len;
   char version;
   short vendor_id;
   short device_id;
} sleeve_header;


/*******************************************************************************/



static struct list_head sleeve_drivers;

static const struct sleeve_device_id *
h3600_sleeve_match_device(const struct sleeve_device_id *ids, struct sleeve_dev *dev)
{
   while (ids->vendor) {
      if ((ids->vendor == SLEEVE_ANY_ID || ids->vendor == dev->vendor) &&
          (ids->device == SLEEVE_ANY_ID || ids->device == dev->device))
         return ids;
      ids++;
   }
   return NULL;
}

static int h3600_sleeve_announce_device(struct sleeve_driver *drv, struct sleeve_dev *dev)
{
   const struct sleeve_device_id *id = NULL;

   if (drv->id_table) {
      id = h3600_sleeve_match_device(drv->id_table, dev);
      if (!id)
         return 0;
   }

   dev->driver = drv;
   if (drv->probe(dev, id) >= 0) {
      return 1;
   }

   return 0;
}

void h3600_sleeve_insert(void) 
{
   struct list_head *item;

   set_egpio(EGPIO_BITSY_OPT_NVRAM_ON);

   msleep(250);

   if (hal_spi_read(6, (char*)&sleeve_dev->vendor, 2) != 0) {
      putstr(__FUNCTION__ ": no spi read, defaulting sleeve vendor\r\n");
      sleeve_dev->vendor = COMPAQ_VENDOR_ID;
   }
   if (hal_spi_read(8, (char*)&sleeve_dev->device, 2) != 0) {
      putstr(__FUNCTION__ ": no spi read, defaulting sleeve deviceid\r\n");
      sleeve_dev->device = SINGLE_PCMCIA_SLEEVE;
   }

   putLabeledWord(" sleeve vendorid=", sleeve_dev->vendor);
   putLabeledWord(" sleeve deviceid=", sleeve_dev->device);
			
   for (item=sleeve_drivers.next; item != &sleeve_drivers; item=item->next) {
      struct sleeve_driver *drv = list_entry(item, struct sleeve_driver, node);
      if (h3600_sleeve_announce_device(drv, sleeve_dev)) {
         putstr(__FUNCTION__ ": matched driver "); putstr(drv->name); putstr("\r\n");
         return;
      }
   }

}

void h3600_sleeve_eject(void) 
{
   
   if (sleeve_dev->driver && sleeve_dev->driver->remove) {
      sleeve_dev->driver->remove(sleeve_dev);
   }

   memset(sleeve_dev, 0, sizeof(struct sleeve_dev));
   clr_egpio(EGPIO_BITSY_OPT_NVRAM_ON);
}

int h3600_current_sleeve( void )
{
   return H3600_SLEEVE_ID( sleeve_dev->vendor, sleeve_dev->device );
}

/*******************************************************************************/

int h3600_sleeve_register_driver(struct sleeve_driver *drv)
{
   putLabeledWord("registering sleeve driver ", (long)drv);
   list_add_tail(&drv->node, &sleeve_drivers);
   return 0;
}

void
h3600_sleeve_unregister_driver(struct sleeve_driver *drv)
{
   list_del(&drv->node);
}

/*******************************************************************************/

static int initialized = 0;

int h3600_sleeve_init_module(void)
{
    int result = 0;

    putstr(__FUNCTION__ "\r\n");

    if (!initialized) {
        clr_egpio(EGPIO_BITSY_OPT_NVRAM_ON);
        clr_egpio(EGPIO_BITSY_OPT_ON);

        sleeve_dev = (struct sleeve_dev *)mmalloc(sizeof(struct sleeve_dev));
        if (sleeve_dev == NULL) {
            putstr(__FUNCTION__ ": malloc failed\r\n");
            return -ENOMEM;
        }
        memset(sleeve_dev, 0, sizeof(struct sleeve_dev));
        initialized = 1;
    }

    return result;
}

void h3600_sleeve_cleanup_module(void)
{
}

void
command_sleeve(
               int		argc,
               const char*	argv[])
{
  if (!initialized) {
    h3600_sleeve_init_module();
    h3600_generic_pcmcia_init_module();
    initialized = 1;
  }
  if (strcmp(argv[1], "init") == 0) {
    h3600_sleeve_init_module();
    h3600_generic_pcmcia_init_module();
  } else if (strcmp(argv[1], "detect") == 0) {
    int detect = 0;
    hal_get_option_detect(&detect);
    putLabeledWord("sleeve detect=", detect);
  } else if (strcmp(argv[1], "insert") == 0) {
    h3600_sleeve_insert();
  } else if (strcmp(argv[1], "eject") == 0) {
    h3600_sleeve_eject();
  } else {
    putstr("usage: sleeve detect | insert | eject \r\n");
  }
}
