/****************************************************************************/
/* Copyright 2000 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * bootldr file for Compaq Personal Server Bootloader
 *
 */

/*
 * Maintainer: Jamey Hicks (jamey@crl.dec.com)
 * Original Authors: Edwin Foo, Jamey Hicks, Dave Panariti, Mike Schexnaydre, Chris Joerg
 * June 29, 2000 - save commands and xmodem send added.
 *                             - George France <france@crl.dec.com>
 * July  3, 2000 - add commands to allow the baudrate to be changed.
 *                             - George France <france@crl.dec.com>
 * July  5, 2000 - extended the range of baudrate change.
 *                             - George France <france@crl.dec.com>
 * July  5, 2000 - added PF_LOGICAL.
 *                             - George France <france@crl.dec.com>
 * July  6, 2000 - added display command.
 *                             - George France <france@crl.dec.com>
 * July 31, 2000 - add Architecture field to boot loader.
 *                             - George France <france@crl.dec.com>
 * Aug  04, 2000 - made the cached and uncached flash size the 
 *                 same as the actual flash size.
 *                             - George France <france@crl.dec.com>
 * Aug  04, 2000 - Auto protected the boot sector after the boot
 *                 loader is loaded.
 *                             - George France <france@crl.dec.com> 
 * Nov  22, 2000 - JFFS commands and params added.
 *                             - John G Dorsey <john+@cs.cmu.edu>
 * Nov  24, 2000 - bootldr.conf support added.
 *                             - John G Dorsey <john+@cs.cmu.edu>
 * Mar  14, 2001 - YMODEM (with MD5) support added.
 *                             - John G Dorsey <john+@cs.cmu.edu>
*/

static char bootprog_name[] = "Compaq OHH BootLoader";
static char bootprog_date[] = DATE;

static char cvsversion[] = "$Id: bootldr.c,v 1.100 2002/01/17 21:26:10 bavery Exp $";

#if defined(CONFIG_BITSY)
int do_splash = 1;
int packetize_output = 0;
int ack_commands = 0;
int no_cmd_echo = 0;
int use_extended_getcmd = 1;

#elif defined(CONFIG_ASSABET)
int use_extended_getcmd = 1;
int do_splash = 1;
int packetize_output = 0;
int ack_commands = 0;
int no_cmd_echo = 0;
int check_for_func_buttons = 0;
long reboot_button_enabled = 0;
#elif defined(CONFIG_SPOT) || defined(CONFIG_GATOR)
int use_extended_getcmd = 1;
int do_splash = 0;
int packetize_output = 0;
int ack_commands = 0;
int no_cmd_echo = 0;
int check_for_func_buttons = 0;
long reboot_button_enabled = 0;
#endif

unsigned long last_ram_load_address = 0;
unsigned long last_ram_load_size = 0;

#define USE_XMODEM

#include "bootldr.h"
#include "btpci.h"
#include "btflash.h"
#include "btusb.h"
#include "buttons.h"
#include "hal.h"
#include "pcmcia.h"
#include "heap.h"
#include "params.h"
#include "modem.h"
#include "xmodem.h"
#include "ymodem.h"
#include "lcd.h"
#include "pbm.h"
#if defined(CONFIG_BITSY)
#include "aux_micro.h"
#include "zlib.h"
#include "zUtils.h"

#endif
#ifdef __linux__
#include <asm-arm/setup.h>
#include <asm/arch-sa1100/h3600_gpio.h>
#include <asm/arch-sa1100/h3600_asic.h>	
#endif
#include "sa1100.h"
#include "bsdsum.h"
#include "architecture.h"
#ifdef CONFIG_USE_DATE_CODE
extern char _bootldr_date_code[];
#endif
#ifdef CONFIG_LOAD_KERNEL
#include "lkernel.h"
#endif
#ifdef CONFIG_MD5
#include "md5.h"
#endif


#ifdef BZIPPED_KERNELS
#define BZ_NO_STDIO
#include "bzip/bzlib.h"
#endif

#define	DEB(s)	putstr(s)
#define	PLW(x)	putLabeledWord(#x"=0x", (unsigned long)(x))

typedef unsigned long u_int32_t;
typedef unsigned long pd_entry_t;
typedef unsigned char BOOL;

long paramoldvalue;
#include "cyclone_boot.h"
		
#define ROUNDUP(v,szpwr2) (((v)+((szpwr2)-1))&~(szpwr2-1))

extern void *SerBase, *Ser1Base, *Ser3Base;

#if defined(CONFIG_ASSABET)
unsigned int *assabet_bcr = (unsigned int *)ASSABET_BCR;
unsigned int *assabet_scr = (unsigned int *)(HEAP_START - sizeof(int));
#endif

extern void getcmd_ex(char*, size_t);

extern void update_autoboot_timeout(struct bootblk_param *param);
extern void update_cmdex(struct bootblk_param *param);



/* Boot Loader Function Prototypes */
void print_help_on_commands(struct bootblk_command *commands);
static int  check_for_nested_help(int argc, const char **argv,struct bootblk_command *cmdlist);
void command_help(int argc, const char **argv,struct bootblk_command *cmdlist);
void command_boot(int argc, const char **argv);
void command_boot_flash(int argc, const char **argv);
void command_boot_jffs2(int argc, const char **argv);
void command_boot_wince(int argc, const char **argv);
void command_boot_nfsroot(int argc, const char **argv);
void command_display(int argc, const char **argv );
void command_load(int argc, const char **argv);
void command_load_flash(int argc, const char **argv);
void command_load_ram(int argc, const char **argv);
/* helper function */
static void command_load_flash_region(const char *regionName, unsigned long regionBase, unsigned long regionSize, int flags);
void command_save(int argc, const char **argv);
void command_save_all(int argc, const char **argv);
void command_save_flash(int argc, const char **argv);
void command_save_ram(int argc, const char **argv);
void command_save_world(int argc, const char **argv);
void command_peek(int argc, const char **argv);
void command_peek_ram(int argc, const char **argv);
#if defined(CONFIG_JFFS)
void command_peek_jffs(int argc, const char **argv);
#endif
void command_poke(int argc, const char **argv);
void command_poke_ram(int argc, const char **argv);
void command_breakpoint(int argc, const char **argv);
void command_qflash(int argc, const char **argv);
void command_eflash(int argc, const char **argv);
void command_pflash(int argc, const char **argv);
void command_physaddr(int argc, const char **argv);
void command_call(int argc, const char **argv);
void command_set(int argc, const char **argv);
int set_param_value(const char *name, char *value_p);
unsigned long strtoul(const char *str, char **endptr, int requestedbase);
void command_flash_type(int argc, const char **argv);
void command_params(int argc, const char **argv);
void command_params_show(int argc, const char **argv);
void command_params_eval(int argc, const char **argv);
void parseParamFile(unsigned char *src,unsigned long size,int just_show);
void params_eval_file(int just_show);
void params_eval_partition(const char* prefix_in,int just_show);
void command_params_save(int argc, const char **argv);
void command_partition(int argc, const char **argv);
void testCirrusCardbusBridge(int argc, const char **argv);
void command_cmdex(int argc, const char **argv);

#if defined(CONFIG_JFFS)
void command_ls(int argc, const char **argv);
void command_histogram(int argc, const char **argv);
#endif
#if defined(CONFIG_MD5)
void command_md5sum(int argc, const char **argv);
#endif

#if defined(CONFIG_BITSY)
void command_aux_ser(int argc, const char **argv);
#endif

void command_version(int argc, const char* argv[]);
void parseParamName(int argc,const char **argv,char *pName,char *pValue);

void command_memcpy(int argc, const char* argv[]);
void command_hex_dump(int argc, const char* argv[]);
void command_reset(int argc, const char* argv[]);
void command_enable_caches(int argc, const char* argv[]);
void command_memcmp(int argc, const char* argv[]);
void command_boot_addr(int argc, const char** argv);
void command_mem(int argc, const char** argv);
void command_memtest(int argc, const char* argv[]);

void command_hal(int argc, const char* argv[]);
void command_sleeve(int argc, const char* argv[]);

void command_testJFFS2(int argc, const char* argv[]);
void body_testJFFS2(const char *filename,unsigned long *dest);
void command_infoJFFS2(int argc, const char* argv[]);
void body_infoJFFS2(char *partname);
void command_timeFlashRead(int argc, const char* argv[]);
void body_timeFlashRead(char *partname);
void command_p1_load_file(int argc, const char* argv[]);
long body_p1_load_file(const char *partname,const char *filename,unsigned char *dest);
void command_p1_ls(int argc, const char* argv[]);
void body_p1_ls(char *dir,char *partname);

void command_cmpKernels(int argc, const char* argv[]);
void body_cmpKernels(const char *partname,unsigned long *dstFlash,unsigned long *srcJFFS,unsigned long len);

void command_clearMem(int argc, const char* argv[]);
void command_pef(int argc, const char* argv[]);
void command_lli(int argc, const char* argv[]);
void command_cat(int argc, const char* argv[]);
void body_clearMem(unsigned long num,unsigned short *dst);
unsigned char isWincePartition(unsigned char *p);
unsigned char amRunningFromRam(void);

void discover_machine_type();

void parseargs(char *argstr, int *argc_p, char **argv, char** resid);
void unparseargs(char *argstr, int argc, const char **argv);
void splash_file(const char * fileName, const char *partName);
void splash();
/**********************************************************
 * zlib tests
 * *******************************************************/

void command_tdz(int argc, const char* argv[]);
void body_tdz(unsigned long address,unsigned long size,unsigned long dAddress);



unsigned long autoboot_timeout = TIMEOUT;

byte awaitkey(unsigned long delay, int *error_p);
void bootmenu(void);

void monitorConfigureMMU(void);

#if defined(CONFIG_GZIP) || defined(CONFIG_GZIP_REGION)
int gunzip_region(char*   src,    char*   dst,
		  long len, const char *name);
#endif
static BOOL isValidBootloader(unsigned long p,unsigned long size);
static BOOL isValidOHHImage(unsigned long p,unsigned long size);
static BOOL isValidParrotImage(unsigned long p,unsigned long size);

/* Boot Loader Variables/Constants */

static struct bootblk_command commands[] = {
  { "?",      NULL, "?" , BB_RUN_FROM_RAM },
  { "help",   NULL, "help <command>" , BB_RUN_FROM_RAM },
  { "boot",   command_boot, "boot [jffs2|flash|nfsroot]" , BB_RUN_FROM_RAM },
  { "display",command_display, "display" , BB_RUN_FROM_RAM },
  { "load",   command_load, "load [ partition_name | ram]" , BB_RUN_FROM_RAM },
  { "upload",   command_save, "upload [all | bootldr | flash | kernel | ramdisk | params | world]" , BB_RUN_FROM_RAM },
  { "peek",   command_peek, "peek ram|flash|int|short|byte <addr>" , BB_RUN_FROM_RAM },
  { "poke",   command_poke, "poke ram|flash|int|short|byte <addr>" , BB_RUN_FROM_RAM },
#if defined(CONFIG_SA1100)
  { "breakpoint", command_breakpoint, "breakpoint" , BB_RUN_FROM_RAM },
#endif
  { "qflash", command_qflash, "qflash [cfi|id] <waddr> -- query flash" , BB_RUN_FROM_RAM },
  { "eflash", command_eflash, "eflash <partitionname>|<sectoraddr> [<size>]|chip -- erase partition, sector, range or chip" , BB_RUN_FROM_RAM },
  { "pflash", command_pflash, "pflash <addr> <len> 0/1 (1 -> protect, 0 -> unprotect all!) -- protect address range" , BB_RUN_FROM_RAM },
  { "reboot", command_reset, "reboot  -- same as reset" , BB_RUN_FROM_RAM },
  { "physaddr", command_physaddr, "physaddr <vaddr> -- returns <paddr>" , BB_RUN_FROM_RAM },
  /*  { "testusb", testUSB, "testusb" , BB_RUN_FROM_RAM },*/
  { "call",   command_call, "call <addr> args" , BB_RUN_FROM_RAM },
  { "jump",   command_call, "jump <addr>" , BB_RUN_FROM_RAM },
  { "set",    command_set, "set <param> <value>" , BB_RUN_FROM_RAM },
  { "show",    command_params_show, "show [<param1>] [<param2>] [<param3>]" , BB_RUN_FROM_RAM },
  { "evalparams", command_params_eval, "evalparams [prefix]", BB_RUN_FROM_RAM },
  { "params", command_params, "params [eval|show|save [-n]|reset]", BB_RUN_FROM_RAM },
  { "partition", command_partition, "partition [add <name> <base> <size> <flags>|delete <name>|show|save]  (flags 16 for JFFS2)", BB_RUN_FROM_RAM },
  { "flash_type", command_flash_type, "flash_type <flashtype>", BB_RUN_FROM_RAM },
#ifdef CONFIG_SKIFF
  { "cardbus", testCirrusCardbusBridge, "cardbus" , BB_RUN_FROM_RAM },
#endif
  { "cmdex", command_cmdex, "cmdex 0|1 (no|yes extended cmdline editing)", BB_RUN_FROM_RAM },
  
#if defined(CONFIG_BITSY)
  { "lcdon", command_lcd_on, "lcdon -- turn lcd back on", BB_RUN_FROM_RAM },
  { "lcdoff", command_lcd_off, "lcdoff -- turn lcd off", BB_RUN_FROM_RAM },
  { "lcdlight", command_lcd_light, "lcdlight level -- adjust lcd backlight", BB_RUN_FROM_RAM },
  { "splash", command_splash, "splash -- display splash image on lcd", BB_RUN_FROM_RAM },
  { "ver", command_version, "ver -- display version info.", BB_RUN_FROM_RAM },

#endif
#ifdef CONFIG_MD5
  { "md5sum", command_md5sum, "md5sum file <filename> | md5sum <partitionname>|<addr> [<size>]", BB_RUN_FROM_RAM },
#endif
  { "memcpy", command_memcpy, "memcpy -- dst src num [size]", BB_RUN_FROM_RAM },
  { "memcmp", command_memcmp, "memcmp -- addr1 addr2 num [size]", BB_RUN_FROM_RAM },
  { "hd", command_hex_dump, "hd -- hexdump memory -- addr [num=16]", BB_RUN_FROM_RAM },

#ifndef CONFIG_SMALL
  { "sleeve", command_sleeve, "sleeve detect).", BB_RUN_FROM_RAM },
  { "pcmcia", command_pcmcia, "pcmcia detect|insert|eject).", BB_RUN_FROM_RAM },
#endif
  { "reset", command_reset, "reset -- do software reset.", BB_RUN_FROM_RAM },
  { "baddr", command_boot_addr,
    "baddr <addr> <size> -- boot kernel loaded at <addr> of len <size>."},
  { "mem", command_mem, "mem -- show info about memory"},
  { "memtest", command_memtest, "memtest addr1 addr2 -- test mem between addrs"},
#ifdef CONFIG_LOAD_KERNEL
  
  { "p1_load_file", command_p1_load_file, "p1_load_file -- load from [partition] the [file] to [dest] ", BB_RUN_FROM_RAM },
  { "ls", command_p1_ls, "ls files from [dir] on [partition (default=param kernel_partition)]", BB_RUN_FROM_RAM },
  { "cat", command_cat, "cat -- show filename from [partition (default=param kernel_partition)]", BB_RUN_FROM_RAM },

#if defined(DEBUG_BOOTLDR_CMDS)
  { "auxser", command_aux_ser, "auxser -- dump aux micro's serial data)", BB_RUN_FROM_RAM },
  { "lcdt", command_lcd_test, "lcdt -- test lcd", BB_RUN_FROM_RAM },
  /*  { "lcdpal", command_lcd_pal, "lcdpal -- set lcd pal", BB_RUN_FROM_RAM }, */
  { "lcdfill", command_lcd_fill, "lcdfill -- fill lcd display", BB_RUN_FROM_RAM },
  { "lcdbar", command_lcd_bar, "lcdbar -- fill lcd bar with [color] to [percent] at [row offset]", BB_RUN_FROM_RAM },
  { "lcdinvertregion", command_lcd_invert_region,
    "lcdinvertregion -- invert region args:start_row start_col end_row end_col", BB_RUN_FROM_RAM },
  { "lcdimg", command_lcd_image, "lcdimg -- display image", BB_RUN_FROM_RAM },
  { "lcdzimg", command_lcd_zimage, "lcdzimg len -- display gzipped image", BB_RUN_FROM_RAM },
  { "ledblink", command_led_blink, "ledblink -- blink the led for [on time] [off time] [off = 0]", BB_RUN_FROM_RAM },
  { "ttmode", command_ttmode, "ttmode 1|0 -- go into tt mode", BB_RUN_FROM_RAM },
  { "ser_con", command_ser_con, "ser_con -- start a serial console session", BB_RUN_FROM_RAM },
  { "irda_con", command_irda_con, "irda_con -- start an irda console session", BB_RUN_FROM_RAM },
  { "hal", command_hal, "hal -- hardware abstraction layer.", BB_RUN_FROM_RAM },
  { "enable_caches", command_enable_caches, "enable_caches dcache[0,1] and icache[0,1]  .", BB_RUN_FROM_RAM },
#ifdef CONFIG_LOAD_KERNEL
  { "infoJFFS2", command_infoJFFS2, "infoJFFS2 -- scan JFFS2 partition for useful info ", BB_RUN_FROM_RAM },
  { "timeFlashRead", command_timeFlashRead, "timeFlashRead -- time a scan of the [partition] ", BB_RUN_FROM_RAM },

#endif // CONFIG_LOAD_KERNEL
  { "cmpKernels", command_cmpKernels, "cmpKernels -- copy flash kernel from [part] to [dst] and cmp with kernal at [src] of len [n]", BB_RUN_FROM_RAM },
  { "clearMem", command_clearMem, "clearMem -- write [num] 0x00 to a [dst]", BB_RUN_FROM_RAM },
  { "pef", command_pef, "pef -- [just_show]", BB_RUN_FROM_RAM },
  { "lli", command_lli, "lli -- low level coprocessor info", BB_RUN_FROM_RAM },
  { "discover", discover_machine_type, "discover -- guess what machine i am see mach_type for result", BB_RUN_FROM_RAM },
  { "tdz", command_tdz, "tdz -- test decompress routines (best with ascii) pre flashing", BB_RUN_FROM_RAM },

#endif // DEBUG_BOOTLDR_COMMANDS

#endif //CONFIG_LOAD_KERNEL

  { NULL,     NULL, NULL , BB_RUN_FROM_RAM },
};

static struct bootblk_command boot_commands[] = {
  { "flash",  command_boot_flash, "boot flash [bootargs ...] -- read kernel from partition" , BB_RUN_FROM_RAM },
  { "partition", command_boot_flash, "boot partition [bootargs ...] -- read kernel from partition" , BB_RUN_FROM_RAM },
#ifdef CONFIG_LOAD_KERNEL
  { "jffs2",  command_boot_jffs2, "boot jffs2 [bootargs ...] -- read kernel from filesystem" , BB_RUN_FROM_RAM },  
#endif //CONFIG_LOAD_KERNEL
#ifdef CONFIG_SKIFF
  { "nfsroot",   command_boot_nfsroot, "boot nfsroot [bootargs ...] -- read kernel from NFS" , BB_RUN_FROM_RAM },
#endif
  { "wince",   command_boot_wince, "boot wince [nothing yet ...]" , BB_RUN_FROM_RAM },
/*  { "ram",  command_boot_ram, "boot ram [bootargs ...]" , BB_RUN_FROM_RAM },*/
  
  { NULL,     NULL, NULL , BB_RUN_FROM_RAM }
};

static struct bootblk_command load_commands[] = {
  { "<partition>", command_load, "load <partition>" , BB_RUN_FROM_RAM },
  { "ram",     command_load_ram, "load ram <dstaddr> <size>" , BB_RUN_FROM_RAM },
  { "flash",   command_load_flash, "load flash <dstaddr>" , 0 },
  { NULL,     NULL, NULL , BB_RUN_FROM_RAM }
};

static struct bootblk_command save_commands[] = {
  { "<partition>", command_save,         "save <partition>" , BB_RUN_FROM_RAM },
  { "all",       command_save_all,       "save all" , BB_RUN_FROM_RAM },
  { "flash",     command_save_flash,     "save flash <addr> <len>" , BB_RUN_FROM_RAM },
  { "ram",       command_save_ram,       "save ram <addr> <len>" , BB_RUN_FROM_RAM },
  { "world",     command_save_world,     "save world" , BB_RUN_FROM_RAM },
  { NULL,        NULL,                   NULL , BB_RUN_FROM_RAM }
};

static struct bootblk_command peek_commands[] = {
  { "ram",    command_peek_ram, "peek ram <addr> reads 32 bits" , BB_RUN_FROM_RAM },
  { "byte",   command_peek_ram, "peek byte <addr> reads 8 bits" , BB_RUN_FROM_RAM },
  { "short",   command_peek_ram, "peek short <addr> reads 16 bits" , BB_RUN_FROM_RAM },
  { "int",    command_peek_ram,  "peek int <addr> reads 32 bits" , BB_RUN_FROM_RAM },
  { "flash",  command_peek_ram, "peek flash <offset>" , BB_RUN_FROM_RAM },
  { "gpio",   command_peek_ram, "peek gpio <offset>" , BB_RUN_FROM_RAM },
  { NULL,     NULL, NULL , BB_RUN_FROM_RAM }
};

static struct bootblk_command poke_commands[] = {
  { "ram",    command_poke_ram, "poke ram <addr> <dword>" , BB_RUN_FROM_RAM },
  { "byte",   command_poke_ram, "poke byte <addr> <byte> writes 8 bits" , BB_RUN_FROM_RAM },
  { "short",   command_poke_ram, "poke short <addr> <word> writes 16 bits" , BB_RUN_FROM_RAM },
  { "int",   command_poke_ram, "poke int <addr> <dword> writes 32 bits" , BB_RUN_FROM_RAM },
  { "flash",  command_poke_ram, "poke flash <offset> <dword>" , 0 },
  { "gpio",  command_poke_ram, "poke gpio <offset> <dword>" , BB_RUN_FROM_RAM },
  { NULL,     NULL, NULL , BB_RUN_FROM_RAM }
};

static struct bootblk_command params_commands[] = {
  { "eval",  command_params_eval, "params eval" , BB_RUN_FROM_RAM },
  { "show",  command_params_show, "params show" , BB_RUN_FROM_RAM },
  { "save",  command_params_save, "params save" , BB_RUN_FROM_RAM },
  { "reset",  command_params,     "params reset" , BB_RUN_FROM_RAM },
  { NULL,     NULL, NULL , BB_RUN_FROM_RAM }
};

#define UPDATE_BAUDRATE
#ifdef UPDATE_BAUDRATE
PARAM(baudrate, PT_INT, PF_DECIMAL, UART_BAUD_RATE, update_baudrate );
#endif
PARAM( os, PT_STRING, PF_STRING, (long)"autoselect", NULL );  /* "linux", "netbsd", "autoselect" */
PARAM( boot_type, PT_STRING, PF_STRING, (long)"jffs2", NULL );
PARAM( kernel_in_ram, PT_INT, PF_HEX, 0xC0008000, NULL ); /* set this to the address of where kernel is loaded in ram (0xC0008000) */
PARAM( autoboot_kernel_part, PT_STRING, PF_STRING, (long)"kernel", NULL );
PARAM( force_unzip, PT_INT, PF_HEX, 0, NULL ); 
PARAM( noerase, PT_INT, PF_HEX, 0, NULL ); 
PARAM( override, PT_INT, PF_HEX, 0, NULL ); 
PARAM( icache_enabled, PT_INT, PF_HEX, 1, update_icache_enabled ); 
PARAM( dcache_enabled, PT_INT, PF_HEX, 0, update_dcache_enabled ); 
#ifdef CONFIG_SKIFF
PARAM( entry, PT_INT, PF_HEX, 0x00000000, NULL ); 
#else
PARAM( entry, PT_INT, PF_HEX, 0xC0000000, NULL ); 
#endif
PARAM( copy_ramdisk, PT_INT, PF_HEX, 0, NULL );
PARAM( ramdisk_addr, PT_INT, PF_HEX, 0, NULL );
PARAM( dram_size, PT_INT|PT_READONLY, PF_HEX, DRAM_SIZE, NULL );
PARAM( memc_ctrl_reg, PT_INT, PF_HEX, 0x110c, NULL );
#ifdef CONFIG_PCI
PARAM( mem_fclk_21285, PT_INT|PT_READONLY, PF_DECIMAL, 48000000, NULL );
PARAM( maclsbyte, PT_INT, PF_HEX, 0xFF, program_all_eeprom );
#endif
#ifdef CONFIG_SKIFF
PARAM( serial_number, PT_INT|PT_READONLY, PF_HEX, 0xFF, set_serial_number );
PARAM( system_rev, PT_INT|PT_READONLY, PF_HEX, 0x01, set_system_rev );
PARAM( linuxargs, PT_STRING, PF_STRING, (long)" root=/dev/ram initrd ramdisk_size=8192", NULL );
#else
PARAM( linuxargs, PT_STRING, PF_STRING, (long)" noinitrd root=/dev/mtdblock1 init=/linuxrc console=ttySA0", NULL );
#endif
PARAM( mach_type, PT_INT, PF_DECIMAL, MACH_TYPE, NULL );
#ifdef CONFIG_LOAD_KERNEL
PARAM( kernel_partition, PT_STRING, PF_STRING, (long)"root", NULL );
PARAM( kernel_filename, PT_STRING, PF_STRING, (long)"boot/zImage", NULL );
PARAM( ptable_addr, PT_INT, PF_HEX, 0xC0004242, NULL );
#endif
PARAM( hostname, PT_STRING, PF_STRING, 0, NULL );
PARAM( domainname, PT_STRING, PF_STRING, 0, NULL );
PARAM( ipaddr, PT_STRING, PF_STRING, 0, NULL );
PARAM( gateway, PT_STRING, PF_STRING, 0, NULL );
PARAM( netmask, PT_STRING, PF_STRING, 0, NULL );
PARAM( dns1, PT_STRING, PF_STRING, 0, NULL );
PARAM( dns2, PT_STRING, PF_STRING, 0, NULL );
PARAM( netcfg, PT_STRING, PF_STRING, 0, NULL ); /* manual/dhcp/... */
PARAM( nfs_server_address, PT_STRING, PF_STRING, 0, NULL );
PARAM( nfsroot, PT_STRING, PF_STRING, 0, NULL );
PARAM( verbose, PT_INT, PF_HEX, 0, NULL );
PARAM( cmdex, PT_INT, PF_HEX, 1, update_cmdex );
    
    /* for formatting JFFS2 filesystem sectors */
PARAM( jffs2_sector_marker0, PT_INT, PF_HEX, 0x20031985, NULL );
PARAM( jffs2_sector_marker1, PT_INT, PF_HEX, 0x0000000C, NULL );
PARAM( jffs2_sector_marker2, PT_INT, PF_HEX, 0xE41EB0B1, NULL );

#if defined(CONFIG_JFFS)
PARAM( boot_file, PT_STRING, PF_STRING, 0, NULL );
PARAM( conf_file, PT_STRING, PF_STRING, (long)"/etc/bootldr.conf", NULL );
#endif

    /*
     * button macros.  the value is a string to be executed by the main
     * command processor
     */
PARAM( recb_cmd, PT_STRING, PF_STRING, 0, NULL);
PARAM( calb_cmd, PT_STRING, PF_STRING, (long)"ser_con", NULL);
PARAM( conb_cmd, PT_STRING, PF_STRING, (long)"irda_con", NULL);
PARAM( qb_cmd, PT_STRING, PF_STRING, 0, NULL);
PARAM( startb_cmd, PT_STRING, PF_STRING, (long)"boot", NULL);
PARAM( upb_cmd, PT_STRING, PF_STRING, 0, NULL);
PARAM( rightb_cmd, PT_STRING, PF_STRING, 0, NULL);
PARAM( leftb_cmd, PT_STRING, PF_STRING, 0, NULL);
PARAM( downb_cmd, PT_STRING, PF_STRING, 0, NULL);

    /* suppress the splash screen */
PARAM( suppress_splash, PT_STRING, PF_STRING, 0, NULL );

    /* reboot after this interval */
PARAM( autoboot_timeout, PT_INT, PF_DECIMAL, 10, update_autoboot_timeout );

struct bootblk_param *get_param(const char *name)
{
   struct bootblk_param *param = (struct bootblk_param *)&__params_begin;
   while (param < (struct bootblk_param *)&__params_end) {
      int namelen = strlen(param->name);
      if (strncmp(name, param->name, namelen) == 0
          && name[namelen] == 0) {
         return param;
      }
      param++;
   }
   putstr("get_param: could not find parameter "); putstr(name); putstr("\r\n");
   return NULL;
}

int get_param_value(const char *name, void *value_p)
{
   struct bootblk_param *param = get_param(name);
   if (param != NULL) {
      if (value_p != NULL) 
         *(long *)value_p = param->value;
      return 0;
   } else {
      return -1;
   }  
}

int set_param_value(const char *pName, char *pValue)
{
   struct bootblk_param *param = (struct bootblk_param *)&__params_begin;
   const char *paramValueString;
   unsigned char set = 0;

   paramValueString = pValue;
   if (paramValueString == NULL)
       paramValueString = "";
   
   while (param < (struct bootblk_param *)&__params_end) {
       int namelen = strlen(param->name);
       if (namelen == strlen(pName) &&
	   strncmp(pName, param->name, namelen) == 0) {
	   long value;
	   int paramValueLen = strlen(paramValueString);
	   set = 1;
	   putstr("  setting param <");
	   putstr(param->name);
	   putstr("> to value <");
	   putstr(paramValueString);
	   putstr(">\r\n");
	   switch (param->paramFormat) {
	       case PF_STRING: {
		   if (paramValueLen == 0) {
		       value = 0;
		   } else {
		       char *newstr = malloc(paramValueLen+1);
		       memcpy(newstr, paramValueString, paramValueLen+1);
		       value = (long)newstr;
		   }
		    break;
		}
		case PF_LOGICAL:
		    paramoldvalue = param->value;
		    value = strtoul(paramValueString, NULL, 0);
		case PF_DECIMAL:
		case PF_HEX:
		default:
		    paramoldvalue = param->value;
		    value = strtoul(paramValueString, NULL, 0);
	    }
	    param->value = value;
	    if (param->update != NULL) {
		(param->update)(param);
	    }
            return 1;
	}
       param++;
   }   
    if (!set){
	putstr("I dont understand param ");
	putstr(pName);
	putstr("\r\n");
    }
    return 0;
}




/*
 * this variable MUST be initialized explicitly to zero to prevent its
 * placement in the BSS.
 */
enum {
    mmue_enabled = 99,
    mmue_notenabled = 101
};

byte mmuEnabled = mmue_notenabled; /* we set this after we call enableMMU() */
#define isMmuEnabled()	(mmuEnabled == mmue_enabled)
#define setMmuEnabled()	(mmuEnabled =  mmue_enabled)

#ifdef NoLibC
/*
 * very simple memcpy to blast bits around from place to place
 */
void *memcpy(char *dst, const char *src, long n)
{
  if (   (((dword)dst)&3) == 0
	 && (((dword)src)&3) == 0
	 && (n&3) == 0 ) {
    unsigned long *ca,*cb;
    for (ca = (unsigned long *)dst, cb = (unsigned long *)src; n > 0;n -= 4)
      *ca++ = *cb++;
  } else {
    byte *ca,*cb;
    for (ca = (byte *)dst, cb = (byte *)src; n-- > 0;)
      *ca++ = *cb++;
  }
  return dst;
}


/*
 * memset routine: fills first n bytes of dst with value c
 */
void memset(void *dst, char c, unsigned long n)
{
  byte *ca;
  for (ca = (byte *)dst; n-- > 0; )
    *ca++ = c;
}

int strlen(const char *s)
{
   int l = 0;
   while (*s++ != 0)
      l++;
   return l;
}

int isalpha(const char c)
{
    if (((c >= 'a') && (c <= 'z')) ||
	((c >= 'A') && (c <= 'Z')))
	return 1;
    else
	return 0;
    
}

int isalpha(const char c)
{
    if (((c == ' ') || (c == '\t')))
	return 1;
    else
	return 0;
    
}


int isdigit(const char c)
{
    if (((c >= '0') && (c <= '9')))
	return 1;
    else
	return 0;
    
}

int isalnum(const char c)
{
    return (isdigit(c) || isalpha(c));
    
    
}



char *strcat(char *dest, const char *src)
{
   char c;
   int len = strlen(dest);
   dest += len;
   while (( *dest++ = *src++) != 0) {
   }
   return dest;
}

int strncmp(const char *s1, const char *s2, int len)
{
   int i = 0;
   int c1, c2;
   if (s1 == NULL || s2 == NULL)
      return -1;
   for (c1 = s1[i], c2 = s2[i]; i < len; c1 = s1[i], c2 = s2[i], i++) {
      if (c1 < c2)
         return -1;
      else if (c1 > c2)
         return 1;
      else
         continue;
   }
   return 0;
}

int strcmp(const char *s1, const char *s2)
{
   int l1 = strlen(s1);
   int l2 = strlen(s2);
   int l = (l1 < l2) ? l1 : l2;
   int result = strncmp(s1, s2, l);
   if (l == 0 && (l1 != l2)) {
      if (l1 < l2)
         return -1;
      else
         return 1;
   } else {
      return result;
   }
}

int memmem(const char *haystack,int num1, const char *needle,int num2)
{
   int l1 = num1
   int l2 = num2;
   int l = (l1 < l2) ? l1 : l2;
   int i;
   
   
   for (i=0; i < (l1 - l2)+1;i++)
       if (!strncmp((char *)(haystack+i),needle,l2)){
	   //putLabeledWord("returning i = ",i);
	   return (int)(haystack+i);
       }
   
   return (int) NULL;

}


char *strchr(const char *s, int c)
{
   char sc;
   while (((sc = *s) != c) && sc != 0) {
     s++;
   }
   if (sc == c)
      return (char *)s;
   else
      return NULL;
}
#endif /* NoLibC */

#ifdef CONFIG_SA1100
void * serbase(void){
#if defined(CONFIG_BITSY) || defined(CONFIG_JORNADA720) || defined(CONFIG_SPOT) || defined(CONFIG_GATOR)
  	return Ser3Base;
#elif defined(CONFIG_ASSABET)
	return machine_has_neponset() ? Ser3Base : Ser1Base;
#endif
}
#endif

#if defined(CONFIG_BITSY)
void
bitsy_putc(
    char    c,
    int	    encap_done)
{
    /*
     * silly person calling putc instead of putstr...
     * we encap the single char
     */
    
    if (!encap_done && packetize_output) {
	char	buf[2];

	buf[0] = c;
	buf[1] = '\0';

	putnstr(buf, 1);
    }
    else
	PrintChar(c, serbase());
}
#endif

void putc(char c) 
{
    if (squelch_serial())
	return;

    
  /* wait for space in the TX FIFO */
#ifdef CONFIG_SKIFF
  while (CSR_READ_BYTE(UARTFLG_REG) & UART_TX_FIFO_BUSY);
  CSR_WRITE_BYTE(UARTDR_REG,c);
#elif defined(CONFIG_BITSY)
  bitsy_putc(c, 0);
#elif defined(CONFIG_SPOT) || defined(CONFIG_JORNADA720) || defined(CONFIG_GATOR)
  while (!((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_TNF)); /* wait until TX FIFO not full */
  *(byte *)SA1100_UART3_UTDR = c;
#elif defined(CONFIG_ASSABET)
  while (!((machine_has_neponset() ? (*(volatile long *)SA1100_UART3_UTSR1) :
	    (*(volatile long *)SA1100_UART1_UTSR1)) & SA1100_UTSR1_TNF));
  (machine_has_neponset() ? (*(byte *)SA1100_UART3_UTDR) :
   (*(byte *)SA1100_UART1_UTDR)) = c;
#else
#error no definition for putc for this architecture
#endif
}

void
let_uart_drain(
    unsigned long status_addr)
{
    while (CTL_REG_READ(status_addr) & SA1100_UTSR1_TBY)
	;
}

void do_putnstr(
    const char* str,
    size_t	n)
{
   while (n && *str != '\0') {
#if defined(CONFIG_BITSY)
         bitsy_putc(*str, 1);
#elif defined(CONFIG_SA1100)
         PrintChar(*str, serbase()) ;
#else
         putc(*str) ;
#endif
      str++ ;
      n--;
   }
}

void
putnstr(
    const char	*str,
    size_t	n)
{
   if (str == NULL)
      return;
   
#if defined(CONFIG_BITSY)
   if (packetize_output) {
       int  len;
       char len_str[16];

       len = strlen(str);
       if (n < len)
	   len = n;

       /* add the msg header */
       dwordtodecimal(len_str, len);
       do_putnstr("MSG/", 4);
       len = strlen(len_str);
       len_str[len] = '/';
       ++len;
       len_str[len] = '\0';
       do_putnstr(len_str, len);
   }
#endif   

   do_putnstr(str, n);
}

void putstr(const char *str)
{
    putnstr(str, strlen(str));
}

void
putstr_sync(
    char*   s)
{
    putstr(s);
    let_uart_drain(SA1100_UART3_UTSR1);
}

#ifdef ZBSS
/*
 * zero bss (using no bss vars!)
 */
void
zbss(void)
{
    char*	    start;
    unsigned	    len;
    aout_header*    hdrp;
    
    hdrp = (aout_header*)FLASH_BASE;
    start = (char*)hdrp + hdrp->a_text + hdrp->a_data;
    
    len = hdrp->a_bss;
    if (len) {
	memset(start, 0x00, len);
    }
    
}
#endif /* ZBSS */

extern char HEX_TO_ASCII_TABLE[16];

void dwordtodecimal(char *buf, unsigned long x)
{
  int i = 0;
  int j = 0;
  char localbuf[16];

  if (x != 0) {
    while (x > 0) {
      unsigned long rem = x % 10;
      localbuf[i++] = HEX_TO_ASCII_TABLE[rem];
      x /= 10;
    }
    /* now reverse the characters into buf */
    while (i > 0) {
      i--;
      buf[j++] = localbuf[i];
    }
    buf[j] = '\0';
  } else {
    buf[0] = '0';
    buf[1] = '\0';
  }
}

void binarytohex(char *buf, long x, int nbytes)
{
  int i;
  int s = 4*(2*nbytes - 1);
  if (HEX_TO_ASCII_TABLE[0] != '0')
     putstr("HEX_TO_ASCII_TABLE corrupted\n");
  for (i = 0; i < 2*nbytes; i++) {
    buf[i] = HEX_TO_ASCII_TABLE[(x >> s) & 0xf];
    s -= 4;
  }
  buf[2*nbytes] = 0;
}

byte strtoul_err;
unsigned long strtoul(const char *str, char **endptr, int requestedbase)
{
   unsigned long num = 0;
   char c;
   byte digit;
   int base = 10;
   int nchars = 0;
   int leadingZero = 0;

   strtoul_err = 0;

   while ((c = *str) != 0) {
      if (nchars == 0 && c == '0') {
         leadingZero = 1;
         if (0) putLabeledWord("strtoul: leadingZero nchars=", nchars);
         goto step;
      } else if (leadingZero && nchars == 1) {
         if (c == 'x') {
            base = 16;
            if (0) putLabeledWord("strtoul: base16 nchars=", nchars);
            goto step;
         } else if (c == 'o') {
            base = 8;
            if (0) putLabeledWord("strtoul: base8 nchars=", nchars);
            goto step;
         }
      }
      if (0) putLabeledWord("strtoul: c=", c);
      if (c >= '0' && c <= '9') {
         digit = c - '0';
      } else if (c >= 'a' && c <= 'z') {
         digit = c - 'a' + 10;
      } else if (c >= 'A' && c <= 'Z') {
         digit = c - 'A' + 10;
      } else {
         strtoul_err = 3;
         return 0;
      }
      if (digit >= base) {
         strtoul_err = 4;
         return 0;
      }
      num *= base;
      num += digit;
   step:
      str++;
      nchars++;
   }
   return num;
}

void putLabeledWord(const char *msg, unsigned long value)
{
   char buf[9];
   binarytohex(buf,value,4);
   putstr(msg);
   putstr(buf);
   putstr("\r\n");
}

void putLabeledAddrWord(const char *msg, unsigned long *value)
{
   char buf[9];

   putstr(msg);
   binarytohex(buf, (unsigned long) value, 4);
   putstr(" *0x" );
   putstr(buf);
   putstr( " == " );
   binarytohex(buf,*value,4);
   putstr(buf);
   putstr("\r\n");
}


void putHexInt32(unsigned long value)
{
  char buf[9];
  binarytohex(buf,value,4);
  putstr(buf);
}

void putHexInt16(word value)
{
  char buf[9];
  binarytohex(buf,value,2);
  putstr(buf);
}

void putHexInt8(byte value)
{
  char buf[3];
  binarytohex(buf,value,1);
  putstr(buf);
}

void putDecInt(unsigned long value)
{
  char buf[16];
  dwordtodecimal(buf, value);
  putstr(buf);
}

void putDecIntWidth(unsigned long value, unsigned int width)
{
  char buf[16];
  unsigned int length;

  dwordtodecimal(buf, value);

  for(length = strlen(buf); length < width; ++length)
    putc(' ');

  putstr(buf);
}

static struct ebsaboot bootinfo = {
   BT_MAGIC_NUMBER,	/* boot info magic number */
   0x20000000,		/* virtual addr of arg page */
   0x20000000,		/* physical addr of arg page */
   NULL,		/* kernel args string pointer */
   (pd_entry_t *)MMU_TABLE_START,	/* active L1 page table */
   0,			/* start of physical memory */
   DRAM_SIZE,		/* end of physical memory */
   SZ_2M+SZ_1M,         /* start of avail phys memory */
   48000000,		/* fclk frequency */
   UART_BAUD_RATE,    	/* baudrate of serial console */
   1 			/* n stop bits */
};

#define	VER_SEP	    "-"

void print_bootldr_version2(
    char*   prefix)
{
  char vbuf[32];

  if (prefix)
      putstr(prefix);
  putstr(bootprog_name);
  putstr(", Rev ");
  dwordtodecimal(vbuf, VERSION_MAJOR); putstr(vbuf); putstr(VER_SEP);
  dwordtodecimal(vbuf, VERSION_MINOR); putstr(vbuf); putstr(VER_SEP);
  dwordtodecimal(vbuf, VERSION_MICRO); putstr(vbuf);
  if(strlen(VERSION_SPECIAL) > 0){
    putstr("-");
    putstr(VERSION_SPECIAL);
  }
#ifdef CONFIG_BIG_KERNEL
  putstr(" [BIG_KERNEL]");
#endif
#ifdef CONFIG_JFFS
  putstr(" [JFFS]");
#endif
#ifdef CONFIG_MD5
  putstr(" [MD5]");
#endif
  putstr(" [MONO]");
  putstr("\r\n");
}

void print_bootldr_version(void)
{
    print_bootldr_version2(">> ");
}

void
print_version_verbose(
    char*   prefix)
{
    print_bootldr_version2(prefix);

#ifdef CONFIG_USE_DATE_CODE
    if (prefix)
	putstr(prefix);
    putstr("Last link date: ");
    putstr(_bootldr_date_code);
    putstr("\n\r");
#endif
    if (prefix)
	putstr(prefix);

    putstr("Contact: bootldr@handhelds.org\r\n");
}

void print_banner(void)
{
  long armrev = 0;
  long cpsr = 0;
  __asm__("mrc p15, 0, %0, c0, c0, 0" : "=r" (armrev));
  __asm__("mrs %0, cpsr" : "=r" (cpsr));

  print_bootldr_version();

  putstr(">> ");
  putstr(bootprog_date);
#ifdef CONFIG_USE_DATE_CODE
  putstr("\n\r>> Last link date: ");
  putstr(_bootldr_date_code);
#endif
  putstr("\r\n>> Contact: bootldr@handhelds.org\r\n");
  
  putstr("\r\n");

  switch(armrev & ARM_ID_MASK){

  case ARM_ID_SA110:
    putLabeledWord(">> StrongARM SA-110 processor ID: ", armrev);
    break;

  case ARM_ID_SA1100:
    putLabeledWord(">> StrongARM SA-1100 processor ID: ", armrev);
    break;

  case ARM_ID_SA1110:
    putstr(">> StrongARM SA-1110 ");
    switch(armrev & ARM_VERSION_MASK){
    case  0: putstr("revision A0\r\n"); break;
    case  4: putstr("revision B0\r\n"); break;
    case  5: putstr("revision B1\r\n"); break;
    case  6: putstr("revision B2\r\n"); break;
    case  8: putstr("revision B4\r\n"); break;
    default: putLabeledWord("processor ID: ", armrev);
    }
    break;

  default:
    putLabeledWord(">> ARM Processor Rev=", armrev);

  }

#if CONFIG_SKIFF
  putLabeledWord(">>  Corelogic Rev=", *(volatile unsigned long *)(DC21285_ARMCSR_BASE + PCI_VENDOR_ID));
  putLabeledWord(">>  Corelogic fclk=", param_mem_fclk_21285.value;
#endif

#if defined(CONFIG_LOAD_KERNEL) || defined(CONFIG_ACCEPT_GPL)
  putstr(">> (c) 2000-2001 Compaq Computer Corporation, provided with NO WARRANTY under the terms of the GNU General Public License.\r\n");
#else
  putstr(">> (c) 2000-2001 Compaq Computer Corporation, provided with NO WARRANTY.\r\n");
#endif
  putstr(">> See http://www.handhelds.org/bootldr/ for full license and sources");
  putstr("Press Return to start the OS now, any other key for monitor menu\r\n");
}

void
bootldr_reset(
    void)
{
    clr_sleep_reset();
    putstr_sync("Rebooting...");
    CTL_REG_WRITE(SA1100_RSRR, RSRR_SWR);
}
    
#ifdef CONFIG_SKIFF

#define	SERIAL_CHAR_READY() (!(CSR_READ_BYTE(UARTFLG_REG) & UART_RX_FIFO_EMPTY))
   /* must read UARTDR_REG before RXSTAT_REG */
#define	SERIAL_READ_CHAR()	CSR_READ_BYTE(UARTDR_REG)
#define	SERIAL_READ_STATUS()	CSR_READ_BYTE(RXSTAT_REG)

#elif defined(CONFIG_BITSY) || defined(CONFIG_SPOT) || defined(CONFIG_JORNADA720) || defined(CONFIG_GATOR)

#define	SERIAL_CHAR_READY() ((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_RNE)
   /* must read UARTDR_REG before RXSTAT_REG */
#define	SERIAL_READ_CHAR()	(*(volatile byte *)SA1100_UART3_UTDR)
#define	SERIAL_READ_STATUS()	((*(volatile byte *)SA1100_UART3_UTSR1) & SA1100_UTSR1_ERROR_MASK)

#elif defined(CONFIG_ASSABET)

#define	SERIAL_CHAR_READY()  ((machine_has_neponset() ? \
                               (*(volatile long *)SA1100_UART3_UTSR1) : \
                               (*(volatile long *)SA1100_UART1_UTSR1)) & \
                              SA1100_UTSR1_RNE)
   /* must read UARTDR_REG before RXSTAT_REG */
#define	SERIAL_READ_CHAR()   (machine_has_neponset() ? \
                               (*(volatile byte *)SA1100_UART3_UTDR) : \
                               (*(volatile byte *)SA1100_UART1_UTDR))
#define	SERIAL_READ_STATUS() ((machine_has_neponset() ? \
                               (*(volatile long *)SA1100_UART3_UTSR1) : \
                               (*(volatile long *)SA1100_UART1_UTSR1)) & \
                              SA1100_UTSR1_ERROR_MASK)
#else
#error no architecture defined for SERIAL_CHAR_READY(), et al
#endif

int	pushed_chars_len = 0;
int	pushed_chars_idx = 0;
char	pushed_chars[128];


int
push_cmd_chars(
    char*   chars,
    int	    len)
{
    if (pushed_chars_len + len > sizeof(pushed_chars))
	return (-1);

    memcpy(&pushed_chars[pushed_chars_len], chars, len);
    pushed_chars_len += len;
    return 0;
}

int
get_pushed_char(
    void)
{
    int	    ret;

    if (pushed_chars_idx >= pushed_chars_len) {
	pushed_chars_idx = 0;
	pushed_chars_len = 0;
	ret = -1;
    }
    else
	ret = pushed_chars[pushed_chars_idx++] & 0xff;

    /*  putLabeledWord("gpc, ret: 0x", ret); */
    
    return(ret);
}

#ifdef CONFIG_SPOT
void spot_idle(void)
{
#define N_LIMIT 25

  unsigned long *olr = (unsigned long *)SPOT_OLR, olr_val;
  static int i = 0, n = 1, t = 0, greenish = 1, increasing = 1;
  static int times[] = {
    6000, 7243, 8409, 9423, 10222, 10755, 10990, 10911, 10524, 9853, 
    8939, 7841, 6627, 5373, 4159, 3061, 2147, 1476, 1089, 1010, 1245, 
    1778, 2577, 3591, 4757
  };

  if(greenish){

    if(i == n){
      i = 0;

      if(t < times[n - 1])
	++t;
      else {
	t = 0;
	n = increasing ? (n + 1) : (n - 1);
      }

      olr_val = 0xff00;
    } else {
      ++i;
      olr_val = 0x00ff;
    }

  } else {  /* reddish */

    if(i == n){
      i = 0;

      if(t < times[n - 1])
	++t;
      else {
	t = 0;
	n = increasing ? (n + 1) : (n - 1);
      }

      olr_val = 0x00ff;
    } else {
      ++i;
      olr_val = 0xff00;
    }

  }

  if(n == N_LIMIT)
    increasing = 0;
  else if(n == 0){
    n = 1;
    increasing = 1;
    greenish ^= 1;
  }

  *olr = olr_val;

}
#endif

int getc_verbose_errors = 1;
int getc_errno = 0;
byte do_getc(
    vfuncp	    idler,
    unsigned long   timeout,
    int*	    statp)
{
   byte c, rxstat;
   int	do_timeout = timeout != 0;
   int	ch;

   getc_errno = 0; /* reset errno */

   while (!SERIAL_CHAR_READY()) {
       if ((ch = get_pushed_char()) != -1)
	   return (ch);
   
       if (do_timeout) {
	   if (!timeout)
	       break;
	   timeout--;
       }
       
       if (idler)
	   idler();
   }

   /* give priority to pushed chars */
   if ((ch = get_pushed_char()) != -1)
       return (ch);
   
   if (do_timeout && timeout == 0) {
       c = 0;
       rxstat = -1;
   }
   else {
       c = SERIAL_READ_CHAR();
       rxstat = SERIAL_READ_STATUS();
   }
   
   if (rxstat) {
      getc_errno = rxstat;
      if (getc_verbose_errors && !statp) {
         putLabeledWord("RXSTAT error: ", rxstat);
      }
      if (statp)
	  *statp = rxstat;
   }
   return(c);
}

byte
getc(
    void)
{
#ifdef CONFIG_SPOT
    return (do_getc(spot_idle, 0, NULL));
#else
    return (do_getc(button_check, 0, NULL));
#endif
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay iterations checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte
awaitkey(
    unsigned long   delay,
    int*	    error_p)
{
    return (do_getc(button_check, delay, error_p));
}

byte do_getc_seconds(
                     vfuncp	    idler,
                     unsigned long   timeout_seconds,
                     int*	    statp)
{
  byte c, rxstat;
  int	do_timeout = timeout_seconds != 0;
  int  timed_out = 0;
  unsigned long start_time = CTL_REG_READ(SA1100_RCNR);
  int	ch;

  getc_errno = 0; /* reset errno */

  while (!SERIAL_CHAR_READY()) {
    if ((ch = get_pushed_char()) != -1)
      return (ch);
   
    if (do_timeout) {
      unsigned long time = CTL_REG_READ(SA1100_RCNR);
      if ((time - start_time) > timeout_seconds) {
        timed_out = 1;
        break;
      }
    }
       
    if (idler)
      idler();
  }

  /* give priority to pushed chars */
  if ((ch = get_pushed_char()) != -1)
    return (ch);
   
  if (do_timeout && timed_out) {
    c = 0;
    rxstat = -1;
  }
  else {
    c = SERIAL_READ_CHAR();
    rxstat = SERIAL_READ_STATUS();
  }
   
  if (rxstat) {
    getc_errno = rxstat;
    if (getc_verbose_errors && !statp) {
      putLabeledWord("RXSTAT error: ", rxstat);
    }
    if (statp)
      *statp = rxstat;
  }
  return(c);
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay seconds checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte
awaitkey_seconds(
    unsigned long   delay_seconds,
    int*	    error_p)
{
    return (do_getc_seconds(button_check, delay_seconds, error_p));
}

/* Iterations for delay loop, assumes caches on: */
#define DELAY_UNIT    (15000000)
#define AWAITKEY_UNIT (1000000)

void delay_seconds(int units)
{
  volatile int i;
  unsigned long start_time = CTL_REG_READ(SA1100_RCNR);
  for(i = 0; i < units * DELAY_UNIT; ++i) {
    unsigned long time = CTL_REG_READ(SA1100_RCNR);
    if ((time - start_time) > units) {
      break;
    }
  } 
}

#define CMDBUFSIZE 256
char cmdBuffer[CMDBUFSIZE];
void getcmd(void) {
   byte curpos = 0; /* curpos position - index into cmdBuffer */
   byte c;
   byte noreturn = 1;

   /* first clear out the buffer.. */
   memset(cmdBuffer, 0, CMDBUFSIZE);

   /* here we go..*/

   while (noreturn) {
      c = getc();
      switch (c)
         {
	 case 0x08:
         case 0x06:
         case 0x07:
         case 0x7E:
         case 0x7F: /* backspace / delete */
            if (curpos) { /* we're not at the beginning of the line */
               curpos--;
               putc(0x08); /* go backwards.. */
               putc(' ');  /* overwrite the char */
               putc(0x08); /* go back again */
            }
            cmdBuffer[curpos] = '\0';
            break;
         case '\r':
         case '\n':
         case '\0':
            noreturn = 0;
            putc('\r');
            putc('\n');
            break;

         case CTL_CH('x'):
	    curpos = 0;
	    break;
	     
         default:
            if (curpos < CMDBUFSIZE) {
               cmdBuffer[curpos] = c;
               /* echo it back out to the screen */
	       if (!no_cmd_echo)
		   putc(c);
               curpos++;
            }
            break;
         }
   }
   /*
     putstr("COMMAND: ");
     putstr(cmdBuffer);
     for (c=0;c<CMDBUFSIZE;c++) {
       putHexInt8(cmdBuffer[c]);
       putc(' ');
     }
     putstr("\r\n");
   */

}

int argc;

enum ParseState {
   PS_WHITESPACE,
   PS_TOKEN,
   PS_STRING,
   PS_ESCAPE
};

enum ParseState stackedState;

void parseargs(char *argstr, int *argc_p, char **argv, char** resid)
{
  // const char *whitespace = " \t";
  int argc = 0;
  char c;
  enum ParseState lastState = PS_WHITESPACE;
   
  /* tokenize the argstr */
  while ((c = *argstr) != 0) {
    enum ParseState newState;

    if (c == ';' && lastState != PS_STRING && lastState != PS_ESCAPE)
	break;
    
    if (lastState == PS_ESCAPE) {
      newState = stackedState;
    } else if (lastState == PS_STRING) {
      if (c == '"') {
        newState = PS_WHITESPACE;
        *argstr = 0;
      } else {
        newState = PS_STRING;
      }
    } else if ((c == ' ') || (c == '\t')) {
      /* whitespace character */
      *argstr = 0;
      newState = PS_WHITESPACE;
    } else if (c == '"') {
      newState = PS_STRING;
      *argstr = 0;
      argv[argc++] = argstr + 1;
    } else if (c == '\\') {
      stackedState = lastState;
      newState = PS_ESCAPE;
    } else {
      /* token */
      if (lastState == PS_WHITESPACE) {
        argv[argc++] = argstr;
      }      
      newState = PS_TOKEN;
    }

    lastState = newState;
    argstr++;
  }

  if (0) { 
    int i;
    putLabeledWord("parseargs: argc=", argc);
    for (i = 0; i < argc; i++) {
      putstr("   ");
      putstr(argv[i]);
      putstr("\r\n");
    }
  }

  argv[argc] = NULL;
  if (argc_p != NULL)
    *argc_p = argc;

  if (*argstr == ';') {
      *argstr++ = '\0';
  }
  *resid = argstr;
}



// this is getting more compliacated,  this function will averride any of the
// args in argstr with tthe args from argv.  this will allow you to override the
// param linuxargs from the commandline. e.g. init=/myRC will override
// init=linuxRC from the params.
void unparseargs(char *argstr, int argc, const char **argv)
{
   int i;
   char *cutStart;
   char *cutEnd;
   char *paramEnd;
   int delta;   
   int j;
   
   
   for (i = 0; i < argc; i++) {
      if (argv[i] != NULL) {
         if ((paramEnd = strchr(argv[i],'=')))// we have an a=b arg
            {
	       paramEnd++;
	       putstr("haystack = <");
	       putnstr(argstr,strlen(argstr));
	       putstr(">\r\n");
   
	       putstr("needle = <");
	       putnstr(argv[i],(int)(paramEnd - argv[i]));
	       putstr(">\r\n");

	       if ((cutStart = (char *)memmem(argstr,strlen(argstr),argv[i],(int)(paramEnd - argv[i]))) != NULL){
                  // found a match
                  if (!(cutEnd = strchr(cutStart,' '))) {
                     putstr(__FUNCTION__ ": was using result of strlen as pointer\r\n");  
                     cutEnd += strlen(argstr);
		  } else {
                     cutEnd++; // skip the space
                  }
                  delta = (int)(cutEnd - cutStart);		   
                  for (j=(int) (cutEnd - argstr);j < strlen(argstr); j++) 
                     argstr[j-delta] = argstr[j];
                  // new end of argstr
                  argstr[strlen(argstr) - delta] = '\0';
		   
	       }
            }	   
         strcat(argstr, " ");
         strcat(argstr, argv[i]);
      }
   }
}

int parsecmd(struct bootblk_command *cmdlist, int argc, const char **argv)
{
   /* find the command name */
   const char *cmdname = argv[0];
   int cmdnum = 0;

#if 0
   putLabeledWord("parsecmd: argc=", argc);   
   putstr("parsecmd: cmdname=>"); putstr(cmdname); putstr("<\r\n");
#endif
   if (argc < 1)
      return -1;
   /* do a string compare for the first offset characters of cmdstr
      against each member of the cmdlist */
   while (cmdlist[cmdnum].cmdstr != NULL) {
      if (strcmp(cmdlist[cmdnum].cmdstr, cmdname) == 0)
         return(cmdnum);
     cmdnum++;
   }
   return(-1);
}

void execcmd(struct bootblk_command *cmdlist, int argc, const char **argv)
{
   int cmdnum = parsecmd(cmdlist, argc, argv);
   
#if 0
   {
       int i;
       putLabeledWord("  argc=", argc);
       putLabeledWord("  cmdnum=", cmdnum);
       for (i=0; i < argc; i++){
	   putstr("execcmd: argv[i]=>"); putstr(argv[i]); putstr("<\r\n");
       }
   }
   
#endif
   // need to special case the help foo command.
   if ((strcmp(argv[0], "help") == 0) || 
       (strcmp(argv[0], "?") == 0)){
       command_help(argc,argv,cmdlist);
       
   }          
   else if ((cmdnum >= 0) && (cmdlist[cmdnum].cmdfunc)) {
       if (!amRunningFromRam() || (cmdlist[cmdnum].flags & BB_RUN_FROM_RAM)){
	   (*cmdlist[cmdnum].cmdfunc)(argc, argv);
       }
       else{
	   putstr("you can't execute the cmd <");
	   putstr(argv[0]);
	   putstr("> while running from ram\r\n");
       }
   } else {
#if 1
       putstr("Don't understand command>");
       putstr(argv[0]);
       putstr("< Try help for a command list.\n\r");
       // too annoying  print_help_on_commands(cmdlist);
#else
       /* default to the first command and insert it into the head of argv */
       int i;
       for (i = argc; i >= 0; i--) {
	   argv[i+1] = argv[i];
       }
       argc++;
       argv[0] = cmdlist[0].cmdstr;
       (*cmdlist[0].cmdfunc)(argc, argv);
#endif
   }
}

void
exec_string(
    char*   buf)
{
  int argc;
  char *argv[128];
  char*	resid;

  while (*buf) {
      memset(argv, 0, sizeof(argv));
      parseargs(buf, &argc, argv, &resid);
      if (argc > 0) {
	  execcmd(commands, argc, (const char **)argv);
      } else {
	  print_help_on_commands(commands);
      }
      buf = resid;
  }
}

void bootmenu(void)
{
  clr_squelch_serial();

  use_extended_getcmd = param_cmdex.value;

  while (1) {
    if (!packetize_output)
	putstr("boot> ");
    if (use_extended_getcmd)
	getcmd_ex(cmdBuffer, CMDBUFSIZE);
    else
	getcmd();
    
    if (cmdBuffer[0]) {
        /*
	 * useful if calling a modified bootldr's bootmenu
	 * function from another place.
	 */
	if (strcmp("quit", cmdBuffer) == 0)
	    return;
	
	exec_string(cmdBuffer);
    }
    
    if (ack_commands)
	do_putnstr("ACK0\n", 5);
  }
}

#define PARAM_PREFIX ("bootldr:")
#define DEBUG_COMMAND_PARAMS_EVAL 1 

void params_eval(
    const char*   prefix_in,
    int	    just_show)
{
  unsigned char* p;
  struct BootldrFlashPartitionTable *pt = 0;
  int partition_table_size = 0;
  // unsigned long ptable_addr = 0;

 
  putstr("\r\nparams_eval: prefix_in =");
  putstr(prefix_in);
  putLabeledWord("\r\nparams_eval: just_show =",just_show);

  // XXX this presumes that the params sector follows the bootloader sector!!
  p = ((char*)flashword) + flashDescriptor->bootldr.base + flashDescriptor->bootldr.size;
  

  pt = (struct BootldrFlashPartitionTable *)p;
  if (pt->magic == BOOTLDR_PARTITION_MAGIC) {
    params_eval_partition(prefix_in,just_show);
  }else if (isWincePartition(p)) {
    /* guess that wince is still installed and do not look for params */
  } else {
      params_eval_file(just_show);
  }
  // now we have our new params, lets copy them to the right place.
  // ptable_addr = param_ptable_addr.value;
  partition_table_size = (sizeof(struct BootldrFlashPartitionTable) 
			  + partition_table->npartitions*sizeof(struct FlashRegion));

  //memcpy(ptable_addr,partition_table,partition_table_size);
  
}

void params_eval_partition(
    const char*   prefix_in,
    int	    just_show)
{
  unsigned char* p;
  unsigned char* endp;
  const char* prefix;
  char cmdbuf[1024];
  char* cmdp;
  struct BootldrFlashPartitionTable *pt = 0;
  int partition_table_size = 0;
  int i;
  
  putstr("eval param blk\r\n");
  // presumes that the params sector follows the bootloader sector and is no larger!
  p = ((char*)flashword) + flashDescriptor->bootldr.base + flashDescriptor->bootldr.size;
  endp = p + flashDescriptor->bootldr.size;

  pt = (struct BootldrFlashPartitionTable *)p;
  if (pt->magic == BOOTLDR_PARTITION_MAGIC) {
    putstr("found partition table in params sector\r\n");
    putLabeledWord("pt->npartitions=", pt->npartitions);
    partition_table_size = 
      (sizeof(struct BootldrFlashPartitionTable) 
       + pt->npartitions * sizeof(struct FlashRegion));
    putLabeledWord("partition_table_size=", partition_table_size);
    for (i = 0; i < pt->npartitions; i++) {
      struct FlashRegion *partition = &pt->partition[i];
      btflash_define_partition(partition->name, partition->base, partition->size, partition->flags);
    }
  }
  p += partition_table_size;

  /* stops at end of sector or first unwritten byte */
  while (p < endp && *p != 0xff && *p != 0) {
#ifdef DEBUG_COMMAND_PARAMS_EVAL
    putLabeledWord("top of loop, p=", (dword)p);
#endif
    prefix = prefix_in;
    while (p < endp && *prefix && *p == *prefix) {
      p++;
      prefix++;
    }
#ifdef DEBUG_COMMAND_PARAMS_EVAL
    putLabeledWord("past prefix check, p=", (dword)p);
#endif

    if (*prefix != '\0') {
#ifdef DEBUG_COMMAND_PARAMS_EVAL
      putstr("no prefix match\r\n");
#endif

      skip_to_eol: 
      /* skip to end of line */
      while (p < endp && *p != '\n')
	p++;

#ifdef DEBUG_COMMAND_PARAMS_EVAL
      putLabeledWord("skipped line, p=", (dword)p);
      putLabeledWord("char at p=", (dword)(*p) & 0xff);
#endif
      p++;
      continue;
    }

    /* copy line to buffer */
    /* terminate with eof, or eol */
    cmdp = cmdbuf;
    while (p < endp && *p != 0xff) {
      if (*p == '\r' || *p == '\n') {
	p++;
	if (*p == '\r' || *p == '\n') 
	  p++;
	break;
      }
      if ((cmdp - cmdbuf) >= sizeof(cmdbuf) - 1)
	  goto skip_to_eol;
      *cmdp++ = *p++;
    }
    *cmdp = '\0';
    cmdp = cmdbuf;

    putstr("+");
    putstr(cmdbuf);
    putstr("\r\n");

    if (just_show)
	continue;

    exec_string(cmdbuf);
    
  }
}


void command_params_eval(int argc, const char **argv)
{
  int	just_show = 0;
  const char*	prefix = PARAM_PREFIX;
    
  if (argc > 1) {
      just_show = (strncmp(argv[1], "-n", 2) == 2);
      if (just_show) {
	  if (argc > 2)
	      prefix = argv[2];
      }
      else
	  prefix = argv[1];
  }
  
  params_eval(prefix, just_show);
}

void command_partition(int argc, const char **argv)
{
  int i;
  if (argc > 1) {
      
    putstr("argv[1]="); putstr(argv[1]); putstr("\r\n");
    if (strcmp(argv[1], "show") == 0) {
      if (partition_table != NULL) {
        putLabeledWord("npartitions=", partition_table->npartitions);
        for (i = 0; i < partition_table->npartitions; i++) {
	    FlashRegion *partition = &partition_table->partition[i];
          putstr(&partition->name[0]); putstr("\r\n");
          putLabeledWord("  base: ", partition->base);
          putLabeledWord("  size: ", partition->size);
	  putLabeledWord("   end: ", partition->size + partition->base);
          putLabeledWord(" flags: ", partition->flags); 
        }
      }
    } else if (strcmp(argv[1], "save") == 0) {
	command_params_save(argc, argv);
    } else if (strcmp(argv[1], "delete") == 0) {
	if (argc == 3){
	    btflash_delete_partition(argv[2]);
	}
	else{
	    putstr("usage: partition delete part_name\r\n");
	}
	
  } else if (strcmp(argv[1], "reset") == 0) {
      btflash_reset_partitions();
    } else if (strcmp(argv[1], "define") == 0) {
      const char *namestr = argv[2];
      const char *basestr = argv[3];
      const char *sizestr = argv[4];
      const char *flagsstr = argv[5];
      char *name;
      unsigned long base, size, flags;
      
      if (argc < 6) {
        putstr("usage: partition define <name> <base> <size> <flags>\r\n");
        putstr("       flags  16 for JFFS2\r\n"); 
        return;
      }
      name = mmalloc(strlen(namestr)+1);
      if (name == NULL) {
        putstr("Error allocating space for name\r\n");
        return;
      }
      strcpy(name, namestr);

      if (strcmp(basestr, "+") == 0) {
	  putstr("basestr is \"+\"\n\r");
	  base = 0xffffffff;
      }
      else
	  base = strtoul(basestr, NULL, 0);
      size = strtoul(sizestr, NULL, 0);
      if (strcmp(flagsstr, "jffs2") == 0) {
        flags = LFR_JFFS2;
      } else { 
      flags = strtoul(flagsstr, NULL, 0);
      }

      if (base < 128000 && ((flags&LFR_PATCH_BOOTLDR) == 0)) {
        putstr("  bootldr sector must have flags set to 2\r\n"); 
        return;
      }

      btflash_define_partition(name, base, size, flags);
    } else {
      goto usage;
    }
  } else {
  usage:
    putstr("usage:\r\n"
           "  partition reset\r\n"
           "  partition show\r\n"
           "  partition define <name> <base> <size> <flags>\r\n"
           "            flags:   16 for JFFS2\r\n"
	   "  partition save\r\n"); 
  }
}



void
default_boot()
{
  while (1) {
     exec_string("boot");
     putstr("boot command failed, entering monitor\n\r");
     
     /* if default boot fails, drop into the bootmenu */
     bootmenu();
  }
}

long mem_sizes[] = {
    SZ_64M,
    SZ_32M,
    SZ_16M,
    0				/* endicator */
};

long
probe_ram_size(
    void)
{
    int	    i;
    long    mem_size;

    /* assumes one bank of DRAM and that DRAM_BASE0 is not in the cache */

    /*
     * Each bank is only 128M long.  if we write to 128M,
     * we really write to the second bank. So
     * we store a 128M val @ 0, since if we have 128M, the other
     * writes will succeed to their respective addresses and won't
     * appear at zero.  If we have 128M, then nothing will
     * clobber the 128M @ 0 and that is what we'll read
     * as the mem size.
     * If we add more banks, this will need to change.
     */
    *(long *)(DRAM_BASE0) = SZ_128M;
    putstr("Probing bank0 memory size...\n\r");
    for (i = 0; ; i++) {
	mem_size = mem_sizes[i];
	if (!mem_size)
	    break;
	*(long *)(DRAM_BASE0 + mem_size) = mem_size;
    }
    putLabeledWord(" bank0 memory size=", *(long *)(DRAM_BASE0));

#ifdef CONFIG_DRAM_BANK1	
    *(long *)(DRAM_BASE1) = SZ_128M;
    putstr("Probing bank1 memory size...\n\r");
    for (i = 0; ; i++) {
	mem_size = mem_sizes[i];
	if (!mem_size)
	    break;
	*(long *)(DRAM_BASE1 + mem_size) = mem_size;
    }
    putLabeledWord(" bank1 memory size=", *(long *)(DRAM_BASE1));
#endif
    
    return (*(long *)DRAM_BASE0);
}

void
print_mem_size(
    char*   hdr,
    long    mem_size)
{
    char vbuf[32];
    putstr(hdr);
    putLabeledWord("0x", mem_size);
    dwordtodecimal(vbuf, mem_size >> 20);
    putstr(" in megs: "); putstr(vbuf); putstr("M\n\r");
}

int main(
    int	    argc,		/* how traditional! */
    char*   argv[])
{
  char	c;
  long icache_enabled, dcache_enabled;
  /*
   * in reality, we don't use argc and argv in the traditional sense.
   * If argc (the 0th arg) is 1 then we are to enter the bootmenu()
   * unconditionally.
   */
  
  int	call_bootmenu = argc != 0;
  long *p = (long*)(DRAM_BASE0 + SZ_4K);
  
  /*
     At this stage, we assume:
        SDRAM is configured.
        UART is configured and ready to go.
        Initial PCI setup is done.
        Interrupts are OFF.
        MMU is DISABLED
  */

  CTL_REG_WRITE(SA1100_RTTR, 0x8000);        /* set up RTC divisor (--> ~1Hz) */
  auxm_init();			/* init the aux microcontroller i/f */
  

  /* to allow wakeup from linux, the bootloader, and wince we use a ram sig.
  *  we need to put the bootloader sig here so it can wake up properly.  Linux
  * takes care of its own.
  */
  
  *p = 0x4242; // the number is meaningless, wince is recognized by 0x0 here.
  
#ifdef DEBUG
  putstr("\r\nBOOT");
#endif

#ifdef CONFIG_PCI
  /* Configure the PCI devices */
#ifdef DEBUG
  putstr(" PCI");
#endif
  bootConfigurePCI();
#endif
#ifdef CONFIG_JORNADA720
  (*((volatile int *)SA1100_PPSR_REG)) &= ~(SA1100_PPDR_LFCLK | 0x80);
  (*((volatile int *)SA1100_PPDR_REG)) |= (SA1100_PPDR_LFCLK | 0x80);
#endif

  putLabeledWord("MMU table start=", (dword)mmu_table);
  putLabeledWord("Boot data start=", (dword)BOOT_DATA_START);
  putLabeledWord("Boot data size=", (dword)BOOT_DATA_SIZE);
  putLabeledWord("Stack data base=", (dword)STACK_BASE);
  putLabeledWord("Stack data size=", (dword)STACK_SIZE);
  
  /* 
   * set up MMU table:
   *  DRAM is cacheable
   *  cached and uncached images of Flash
   *  after MMU is enabled, bootldr is running in DRAM
   */
  bootConfigureMMU(); 
  writeBackDcache(CACHE_FLUSH_BASE);
  enableMMU();

  /* flashword logically should be assigned in bootConfigureMMU,
   * but we can't do it until the MMU is enabled
   * because before then it resides in flash 
   * -Jamey 9/4/2000
   */
  flashword = (unsigned long *)UNCACHED_FLASH_BASE;

  /* initialize the heap */
  mmalloc_init((unsigned char *)(HEAP_START), HEAP_SIZE);

  
  /* figure out what kind of flash we have */
  btflash_init();
  // this must happen early, BEFORE the lcd is turned on
  discover_machine_type();
  initialize_by_mach_type();
  
  flashConfigureMMU(flash_size);

  {
#ifdef CONFIG_SKIFF
     long system_rev = get_system_rev();
#endif
     long dram_size;
#ifdef CONFIG_SKIFF
     dram_size = ((system_rev&SYSTEM_REV_MAJOR_MASK) == SYSTEM_REV_SKIFF_V2) ? DRAM_SIZE : SZ_16M;
#endif
#if defined(CONFIG_BITSY) || defined(CONFIG_ASSABET) || defined(CONFIG_JORNADA720)
     /* assumes one bank of DRAM and that DRAM_BASE0 is not in the cache */
     dram_size = probe_ram_size();
     if (dram_size == 0xffffffff)
	 putstr("DRAM size probe failed!\n\r");
#endif /* Bitsy, Assabet, Jornada720 */
#ifdef CONFIG_SKIFF
#if   defined(MEMCLK_33MHZ)
#warning 33MHZ override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_33MHZ;
#elif defined(MEMCLK_48MHZ)
#warning 48MHz override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_48MHZ;
#elif defined(MEMCLK_60MHZ)
#warning 60MHZ override     
     system_rev &= ~SYSTEM_REV_MEMCLK_MASK;
     system_rev |= SYSTEM_REV_MEMCLK_60MHZ;
#endif
     param_system_rev.value = system_rev;
     putLabeledWord("system_rev: ", system_rev);
     switch (system_rev & SYSTEM_REV_MEMCLK_MASK) {
     case SYSTEM_REV_MEMCLK_60MHZ: 
        param_mem_fclk_21285.value = 60000000; break;
     case SYSTEM_REV_MEMCLK_48MHZ: 
        param_mem_fclk_21285.value = 48000000; break;
     case SYSTEM_REV_MEMCLK_33MHZ: 
     default:
        param_mem_fclk_21285.value = 33333333; break;
     }
     if (maclsbyte_param)
       param_maclsbyte.value = get_maclsbyte();

     putstr("memclk_hz: ");
     putstr(((system_rev & SYSTEM_REV_MEMCLK_MASK) == SYSTEM_REV_MEMCLK_33MHZ) 
            ? "33MHz\r\n" : "48MHz\r\n");
     /* this string should agree with the MAC in btpci.c */
     if (maclsbyte_param) {
       putstr("MAC=08:00:2b:00:01:"); putHexInt8(param_maclsbyte.value); putstr("\r\n");
     }
#endif /* CONFIG_SKIFF */
     param_dram_size.value = dram_size;
     print_mem_size("SDRAM size: ", dram_size);
	 
     bootinfo.bt_memend = dram_size;
  }

#ifdef CONFIG_JORNADA720
  /* Take the MCU out of reset mode */
  (*((volatile int *)SA1100_PPSR_REG)) |= SA1100_PPDR_LFCLK;
#endif

#ifdef CONFIG_LOAD_KERNEL
  if (!jffs_init_1pass_list()){
      putstr("jffs_init_1pass_list failed to initialize\r\n");
  }
#endif //CONFIG_LOAD_KERNEL


  /* print the opening banner */
#if !defined (NO_SPLASH) && defined(CONFIG_BITSY)
  if (squelch_serial() || do_splash)
      splash();
#else
  // at least show something even if we didnt include a splash screen
  if (squelch_serial() || do_splash)    
      lcd_default_init(lcd_type, NULL);
#endif
  print_banner();

  if (squelch_serial()) {
      int suppress_splash = 0;
      /* normal boot, eval params */
      command_params_eval(0, NULL);
      suppress_splash = param_suppress_splash.value;
      if (suppress_splash) {
         lcd_off(lcd_type);
      } else {
        lcd_on();
      }
  }
  else {
      putstr("DEBUG BOOT: not evaluating params\r\n");
      putstr("DEBUG BOOT: use `params eval' to evaluate parameters.\r\n");
  }

  icache_enabled = param_icache_enabled.value;
  dcache_enabled = param_dcache_enabled.value;

  enable_caches(dcache_enabled, icache_enabled);

  autoboot_timeout = param_autoboot_timeout.value;

  /********************************************************************************
   * WARNING, IF YOU TURN ON THE D CACHE AND DONT CORRECTLY TURN IT OFF
   * DRAM GETS CORRUPTED, SINCE WE SEEM TO HANG WHEN WE FLUSH DCACHE, WE NEED TO
   * LOOK AT THIS MORE BEFORE RUNNING WITH DCACHE TURNED ON
   * 
   ********************************************************************************/  

  asmEnableCaches(dcache_enabled,icache_enabled); // turn on icache
  
  if (autoboot_timeout == 0 || call_bootmenu || !squelch_serial())
      while (1) 
	  bootmenu(); /* does not return */
      
  /*
   * wait for a keystroke or a button press.
   * button presses cause commands to be run.
   */
  c = awaitkey_seconds(autoboot_timeout, NULL);
  if (((c != '\r') && (c != '\n') && (c != '\0'))) {
      putstr("type \"?\" or \"help\" for help.\r\n");
      bootmenu(); /* does not return */
  }

  default_boot();
  return 0;
}

void print_help_on_commands(struct bootblk_command *commands)
{
  int i = 0;
  putstr("Available Commands:\r\n");
  while (commands[i].cmdstr != NULL) {
    putstr(commands[i].helpstr); putstr("\r\n");
    i++;
  }
}

/* no arguments */
void command_help(int argc, const char **argv,struct bootblk_command *cmdlist )
{
    /*cases :
     help foo
        parsecmd for foo and print the helpstr
     help
        print the help on command list
     help foo bar
        parsecmd for foo then call foo_cmd w/ argv[0]=help, then rest of args

	*/
   int cmdnum;

    
   if (argc == 1){       
       print_help_on_commands(cmdlist);
   }
   else if (argc == 2){
       cmdnum = parsecmd(cmdlist, argc-1, &argv[1]);
       if (cmdnum >= 0){
	   putstr(cmdlist[cmdnum].helpstr); putstr("\r\n");
       }
       else {
	   putstr("Command: ");putstr(argv[1]); putstr(" not known\r\n");
       }
       
   }   
   else{
       char *new_argv1 = mmalloc(strlen("help")+1);
       cmdnum = parsecmd(cmdlist, argc-1, &argv[1]);
       strcpy(new_argv1, "help");
       argv[1] = new_argv1;
       
        
       if (cmdnum >= 0)
	   (*cmdlist[cmdnum].cmdfunc)(argc-1, &argv[1]);
	
   }
   
}


/* can have arguments or not */
void command_boot(int argc, const char **argv)
{   
    unsigned char* p;
    char *wp = "wince";

    if (check_for_nested_help(argc,argv,boot_commands))
	return;
    
      
    if (argv[1] == NULL) {
      char *boot_type;
      boot_type = (char*)param_boot_type.value;
      argv[1] = boot_type;
      argc++;
   }
    /* since we cant have a params table in wince, we need to check for the
     * wince partition here and force it
     */
    p = ((char*)flashword) + flashDescriptor->bootldr.base + flashDescriptor->bootldr.size;
    if (isWincePartition(p)){
        char *new_argv1 = mmalloc(strlen(wp)+1);
	strcpy(new_argv1, wp);
	argv[1] = new_argv1;
    }
    
   


   putstr("booting ");
   putstr(argv[1]);
   putstr("...\r\n");

   execcmd(boot_commands,argc-1,argv+1);
}

/* can have arguments or not */
void command_flash_type(int argc, const char **argv)
{
  if (argc == 1) {
     /* print the available flash types */
     btflash_print_types();
  } else {
    /* manually set the flash type */
     btflash_set_type(argv[1]);
  }
}

#if defined(CONFIG_JFFS)

static int jffs_changed = 1;

/* can have arguments or not */
void command_ls(int argc, const char **argv)
{
  if(jffs_init(jffs_changed) < 0)
    return;

  jffs_changed = 0;

  jffs_list((argc == 1) ? "/" : (char *)argv[1]);
}

/* no arguments */
void command_histogram(int argc, const char **argv)
{
  jffs_statistics();
}

#endif  /* CONFIG_JFFS */

#if defined(CONFIG_MD5)
void command_md5sum(int argc, const char **argv)
{
   void *base = 0;
   unsigned long size = 0;
   unsigned int sum[MD5_SUM_WORDS];
   char sizebuf[32];
   /* default place to unpack file */
   base = (void*)param_kernel_in_ram.value;

   if (!(argc >= 2 || argc <= 4)) {
      putstr("Usage: md5sum file <filename> [<partition>]\r\n");
      putstr("Usage: md5sum <partitionname|address> [<len>]\r\n");
      return;
   }

   if (strcmp(argv[1], "file") == 0) {
#ifdef CONFIG_JFFS
      /* using John Dorsey's JFFS implementation */
      if (jffs_init(jffs_changed) < 0)
         return;

      jffs_changed = 0;
      jffs_md5sum((char *)argv[1]);
#else
      const char *partname = (argc == 4) ? argv[3] : "root";
      const char *filename = argv[2];
      /* fetch the file contents */
      size = body_p1_load_file(partname, filename, base);
#endif
   } else 
      {
         struct FlashRegion *region = btflash_get_partition(argv[1]);
         if (region != NULL) {
            base = (void*)region->base;
            size = region->size;
         } else {
            base = (void*)strtoul(argv[1], NULL, 0);
            if (strtoul_err) {
               putstr("error parsing base addr: "); putstr(argv[1]); putstr("\r\n");
               return;
            }
         }
         if (argc == 3) {
            size = strtoul(argv[2], NULL, 0);
            if (strtoul_err) {
               putstr("error parsing size: "); putstr(argv[2]); putstr("\r\n");
               return;
            }
         }
      }

   if (size > 0) {
      md5_sum(base, size, sum);
      md5_display(sum); putstr("    "); putstr(argv[1]); 
      /* now put the size */
      dwordtodecimal(sizebuf, size); putstr("    "); putstr(sizebuf); putstr("\r\n");
   }
}
#endif  /* CONFIG_MD5 */


/* can have arguments or not */
void command_params(int argc, const char **argv)  
{
    if (check_for_nested_help(argc,argv,params_commands))
	return;

   if (argc == 2) {
      if (strcmp(argv[1], "reset") == 0) {
         putstr("setting params to default values\r\n");
	 putLabeledWord("flashword = ",(long)flashword);
	 putLabeledWord("bootldr_params = ",(long)&__params_begin);
	 putLabeledWord("FLASH_BASE = ",FLASH_BASE);
	 putLabeledWord("sizeof(bootldr_params) = ", &__params_end - &__params_begin);

         memcpy((char*)&__params_begin, (const char*)(((char*)flashword) + (((dword)&__params_begin) - FLASH_BASE)), &__params_end - &__params_begin);
	 
         return;
      }
   } else if (argv[1] == NULL) {
      argv[1] = "show";
      argc++;
   }
   execcmd(params_commands,argc-1,argv+1);
}


#ifdef __linux__
void setup_linux_params(long bootimg_dest, long initrd_start, long initrd_size, 
			long memc_ctrl_reg, long dram_size,
			const char *cmdline, int flashroot)
{
   struct param_struct *params = (struct param_struct *)(bootimg_dest + 0x100);
   long copy_ramdisk = 0;
   struct FlashRegion *ramdisk_region = btflash_get_partition("ramdisk");
   long first_word_of_ramdisk = (ramdisk_region != NULL) ? *(long *)(FLASH_BASE + ramdisk_region->base) : 0;
   int rootdev = 0x00ff;
   int using_ramdisk = flashroot;
   struct tag *tag;
   int i;
   int npartitions = partition_table->npartitions;
   int newcmdlinelen = 0;
   char *newcmdline = NULL;

   copy_ramdisk = param_copy_ramdisk.value;
   if (flashroot && (first_word_of_ramdisk == 0x28cd3d45)) {
     /* cramfs */
     copy_ramdisk = 0;
     rootdev = 0x3c02;
     using_ramdisk = 0;
   }
   if (!flashroot) {
     copy_ramdisk = 0;
   }

   // add the partition table to the commandline
   //  Format suggested by Nicolas Pitre: mtdpart=<name1>@<size1>:<offset1>[,<name2>@<size2>:<offset2>[,...]]
   newcmdlinelen = strlen(cmdline) + 9 + 64*npartitions;
   newcmdline = mmalloc(newcmdlinelen);
   memset(newcmdline, 0, newcmdlinelen);
   strcpy(newcmdline, "mtdpart=");
   for (i = 0; i < npartitions; i++) {
     if (i != 0)
       strcat(newcmdline, ",");
     strcat(newcmdline, partition_table->partition[i].name);
     strcat(newcmdline, "@0x"); 
     binarytohex(newcmdline+strlen(newcmdline), partition_table->partition[i].base, 4);
     strcat(newcmdline, ":0x"); 
     binarytohex(newcmdline+strlen(newcmdline), partition_table->partition[i].size, 4);
   }
   strcat(newcmdline, " ");
   strcat(newcmdline, cmdline);
   cmdline = newcmdline;

   // start with the core tag       
   tag = (struct tag *)(bootimg_dest + 0x100);
   
   putLabeledWord("Makeing core tag at ",(unsigned long) tag);
   
   tag->hdr.tag = ATAG_CORE;
   tag->hdr.size = tag_size(tag_core);
   tag->u.core.flags =0;
   tag->u.core.pagesize = LINUX_PAGE_SIZE;
   tag->u.core.rootdev = rootdev;
   tag = tag_next(tag);

   // now the cmdline tag
   putLabeledWord("Makeing cmdline tag at ",(unsigned long) tag);
   tag->hdr.tag = ATAG_CMDLINE;
   // must be at least +3!! 1 for the null and 2 for the ???
   tag->hdr.size = (strlen(cmdline) + 3 + sizeof(struct tag_header)) >> 2;
   //tag->hdr.size = (strlen(cmdline) + 10 + sizeof(struct tag_header)) >> 2;
   strcpy(tag->u.cmdline.cmdline,cmdline);
   tag = tag_next(tag);

       
   // now the mem32 tag
   putLabeledWord("Makeing mem32 tag at ",(unsigned long) tag);
   tag->hdr.tag = ATAG_MEM;
   tag->hdr.size = tag_size(tag_mem32);
   tag->u.mem.size = dram_size;
   tag->u.mem.start = 0xc0000000;
   tag = tag_next(tag);
       
       // and the ptable tag
   putLabeledWord("Makeing ptable tag at ",(unsigned long) tag);
   tag->hdr.tag = ATAG_PTABLE_COMPAT;
   tag->hdr.size = (sizeof(struct tag_ptable)
		    + sizeof(struct bootldr_flash_region) * partition_table->npartitions
		    + 3 + sizeof(struct tag_header)) >> 2;
   tag->u.ptable.magic = partition_table->magic;
   tag->u.ptable.npartitions = partition_table->npartitions;
   for (i=0; i < partition_table->npartitions;i++){
       strcpy(tag->u.ptable.partition[i].name,
	      partition_table->partition[i].name);
       tag->u.ptable.partition[i].base = partition_table->partition[i].base;
       tag->u.ptable.partition[i].size = partition_table->partition[i].size;
       tag->u.ptable.partition[i].flags = partition_table->partition[i].flags;
   }
       


   tag = tag_next(tag);
       
       
       // now the NULL tag
   tag->hdr.tag = ATAG_NONE;
   tag->hdr.size = 0;
       
   
   putLabeledWord("dram_size=", dram_size);
   putLabeledWord("nr_pages=", params->u1.s.nr_pages);
   putstr("command line is: ");
   putstr(cmdline);
   putstr("\r\n");
}
#endif

/*
 * boot_kernel
 *  root_filesystem_name:
 *  kernel_region_start: virtual address of region holding kernel image
 *  kernel_region_size:  size of region holding kernel image
 *  argc: number of arguments in argv (including argv[0], the command name)
 *  argv: argument vector (including argv[0], the command name)
 */
void boot_kernel(const char *root_filesystem_name, 
                        vaddr_t kernel_region_start, size_t kernel_region_size, int argc, const char **argv, int no_copy)
{
  unsigned long *kernel_region_words;
  unsigned long *qnx_image_words = (unsigned long *)QNX_IMAGE_START;
  unsigned long kernel_magic;
  unsigned long kernel_image_source = (unsigned long)kernel_region_start ;
  unsigned long kernel_image_size = kernel_region_size;
  unsigned long kernel_image_dest;
  int i;
  char *os;
  long linuxEntryPoint = 0;
  long memc_ctrl_reg = 0x110c;
  long copy_ramdisk = 0;
  long ramdisk_addr = 0;
  long ymodem = 0;
  long dram_size = 0;
  long kernel_in_ram = 0;
  void (*qnx_start_func)( int );
  struct FlashRegion *ramdisk_region = btflash_get_partition("ramdisk");

  kernel_image_dest = DRAM_BASE;

  if (no_copy){
       // check from the ram copy       
      kernel_region_words = (unsigned long *) (0x8000 +
					       kernel_image_dest);      
  }
  else{
      kernel_region_words = (unsigned long *)kernel_region_start;// check the flash copy
  }
  kernel_magic = kernel_region_words[0];  
  
  dram_size = param_dram_size.value;
  os = (char*)param_os.value;
  linuxEntryPoint = param_entry.value; 
  kernel_in_ram = param_kernel_in_ram.value;
  copy_ramdisk = param_copy_ramdisk.value;
  ramdisk_addr = param_ramdisk_addr.value;
#ifdef CONFIG_YMODEM
  ymodem = param_ymodem.value;
#endif

  putLabeledWord("kernel partition base ", (unsigned long)kernel_region_words);

  /* Check for a QNX image */
  if ( strcmp( os, "autoselect" ) == 0 ||
       strcmp( os, "qnx" ) == 0 ) {

    /* Search the first 32 1K word boundries */ 
    for( i=0; i<32; i++ )
    {
        if ( qnx_image_words[ 256 * i ] == QNX_IMAGE_MAGIC ) {
          putLabeledWord( "Found a QNX kernel @ 0x",
			  (unsigned long)(&qnx_image_words[256*i]));
          qnx_start_func = (void *)QNX_IMAGE_START;
	  //flush_caches();
          let_uart_drain(SA1100_UART3_UTSR1);
          qnx_start_func( 0 );
          /* never gets here */
          return;
        }
    }
  }


  
  putLabeledWord("kernel_magic=", kernel_magic);
  if (kernel_magic == 0xFFFFFFFFL) {
    putstr("no boot image in flash\r\n");
    return;
  } 


  
  
  putLabeledWord("kernel_region_words[9]=", kernel_region_words[9]);
  
  if (kernel_region_words[9] == LINUX_ZIMAGE_MAGIC) {
    unsigned long compressed_rd_start = 0;
    unsigned long compressed_rd_size = 0;
    unsigned long initrd_size = SZ_8M;
    unsigned long initrd_start = STACK_BASE - initrd_size;
    unsigned long kernel_image_offset = 0x0;
    unsigned long mach_type = MACH_TYPE;
    if (ramdisk_region != NULL) {
      compressed_rd_start = FLASH_BASE + ramdisk_region->base + 4;
      compressed_rd_size = *(unsigned long *)(FLASH_BASE + ramdisk_region->base);
    }          

#if 1
    if (   kernel_region_words[0] == SKIFF_ZIMAGE_MAGIC
           || kernel_region_words[9] == LINUX_ZIMAGE_MAGIC ) {
      /* standard Linux zImage gets loaded to phyasaddr 0x8000 */
      kernel_image_offset = 0x8000;
      linuxEntryPoint += 0x8000;
    }
#endif

    putstr("Linux ELF flash_imgstart=");
    putHexInt32((long)kernel_image_source);
    putstr(" size=");
    putHexInt32(kernel_image_size);
    putstr(" dest=");
    putHexInt32(kernel_image_dest);
    putstr(" offset=");
    putHexInt32(kernel_image_offset);
    putstr("\r\n");
     
    asmEnableCaches(0,0); // also flushes caches    
    putLabeledWord("MMU Control=", readCPR1());
    putLabeledWord("MMU PIDVAM=", readCPR13());

    if (((void *)kernel_image_dest != (void *)kernel_image_source) &&
	!no_copy) {
      putstr("copying Linux kernel ... ");
      memcpy((void*)(kernel_image_dest + kernel_image_offset),
	     (void*)kernel_image_source, kernel_image_size);
      putstr("done\r\n");
    }
    else if (no_copy)
	putstr("Skipping kernel copy by request.\n\r");

    for (i = 0; i < 40; i += 4) {
      /* we still have the MMU on, kernel_image_dest gets us to the right virtual address */
      putHexInt32(kernel_image_dest + kernel_image_offset + i);
      putstr(": ");
      putHexInt32(*(unsigned long *)(kernel_image_dest + kernel_image_offset +i));
      putstr("\r\n");
    }

    if (copy_ramdisk && compressed_rd_size != -1) {

      initrd_start = (ramdisk_addr == 0) ? (STACK_BASE - initrd_size) :
	ramdisk_addr;

      putstr("Copying compressed ramdisk from ");
      putHexInt32(compressed_rd_start);
      putstr(" to ");
      putHexInt32(initrd_start);
      putstr("...");
      memcpy((char*)initrd_start,(char*)compressed_rd_start,compressed_rd_size); 
      putstr("Done \r\n");
    } else if (compressed_rd_size != -1) {
      initrd_start = compressed_rd_start;
    }

    if (1) {
      char boot_args[MAX_BOOTARGS_LEN];
      char *linuxargs = NULL;
      int flashroot = 0;
      if (strcmp(root_filesystem_name, "ramdisk") == 0) {
        flashroot = 1;
      }

      putstr("root_filesystem_name="); putstr(root_filesystem_name); putstr("\r\n");
      memset(boot_args, 0, MAX_BOOTARGS_LEN);
      linuxargs = (char *)param_linuxargs.value;
      putLabeledWord("Grabbed linuxargs, argc = ",argc);      
      mach_type = param_mach_type.value;
      putLabeledWord("Using mach_type ", mach_type);
      if (linuxargs != NULL)
        strcat(boot_args, linuxargs);
      if (argc > 1) {
	  putstr("pre unparse setting boot parameters to\r\n");
	  putstr(boot_args);putstr("\r\n");
	  unparseargs(boot_args, argc-1, argv+1);
      }

      
      putstr("setting boot parameters to\r\n");
      putstr(boot_args);putstr("\r\n");

      
      
#ifdef __linux__
      setup_linux_params(kernel_image_dest, initrd_start, compressed_rd_size,
                         param_memc_ctrl_reg.value, dram_size, boot_args, flashroot);
#endif
    }

#ifdef CONFIG_LCD
    lcd_off(lcd_type);
#endif

    putLabeledWord("linuxEntryPoint=", linuxEntryPoint);
    putstr("Booting Linux image\r\n");

    bootLinux(&bootinfo,
              mach_type,
#ifdef CONFIG_SKIFF
              /* hack -- linux entry point virtual address is 0x10008000, physaddr is 0x00008000 */
              /* after we disable the MMU we have to use the physical address */
              linuxEntryPoint&0x00FFFFFF
#else
              linuxEntryPoint
#endif

              );

  } else if (kernel_magic == NETBSD_KERNEL_MAGIC) {
    char boot_args[MAX_BOOTARGS_LEN];

    putstr("copying NetBSD kernel ... ");
    memcpy((void*)kernel_image_dest, (void*)kernel_image_source, kernel_image_size);
    putstr("done\r\n");

    /*bootinfo.bt_memavail = ROUNDUP(kernel_image_size, SZ_1M);*/
    bootinfo.bt_memavail = SZ_2M+SZ_1M;
    /*putLabeledWord("bt_memavail = ",bootinfo.bt_memavail);*/
      
    boot_args[0] = 0;
    strcat(boot_args, "netbsd ");
    if (argc > 1) {
      unparseargs(boot_args, argc-1, argv+1);
    }
    bootinfo.bt_args = boot_args;
    bootinfo.bt_vargp = (u_int32_t)boot_args & PAGE_MASK;
    bootinfo.bt_pargp = (u_int32_t)boot_args & PAGE_MASK;
    for(i = 0 ; i < (DRAM_SIZE0 - SZ_2M) ; i += SZ_1M) {
      unsigned long dram_paddr = DRAM_BASE + i;
      unsigned long dram_vaddr = 0xf0000000 + i;
      mmu_table[dram_vaddr >> 20] = dram_paddr | MMU_SECDESC | MMU_CACHEABLE;
    }
    asmEnableCaches(0,0);    
    putstr("done!\r\nJumping to 0xF0000020..\r\n");
    
    boot(&bootinfo,0xF0000020);
  } else {
    putstr("Unrecognized kernel image\r\n");
    return;
  }
}

/* can have arguments or not */
void command_boot_flash(int argc, const char **argv)
{
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   const char *kernel_part_name = NULL;
   char bootargs[MAX_BOOTARGS_LEN];
   struct FlashRegion *kernelPartition;

   ipaddr = (const char *)param_ipaddr.value;
   serveraddr = (const char *)param_nfs_server_address.value;
   gatewayaddr = (const char *)param_gateway.value;
   netmask = (const char *)param_netmask.value;
   hostname = (const char *)param_hostname.value;
   nfsroot = (const char *)param_nfsroot.value;
#define KERNEL_PARAM_STR    "kpart="
#define KERNEL_PARAM_STR_LEN sizeof(KERNEL_PARAM_STR)-1
   if (argc > 1 &&
       memcmp(KERNEL_PARAM_STR, argv[1], KERNEL_PARAM_STR_LEN) == 0) {
       kernel_part_name = argv[1] + KERNEL_PARAM_STR_LEN;
       /* skip over this param */
       argv[1] = argv[0];
       argv++;
       argc--;
   }
   else
       kernel_part_name = (const char *)param_autoboot_kernel_part.value;

   kernelPartition = btflash_get_partition(kernel_part_name);

   if (kernelPartition == NULL) {
       putstr("cannot find kernel partition named >");
       putstr(kernel_part_name);
       putstr("<\r\n");
     return;
   }
   else {
       putstr("booting kernel from partition >");
       putstr(kernel_part_name);
       putstr("<\r\n");
   }
   
   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   if ((ipaddr != NULL) || (serveraddr != NULL) || (gatewayaddr != NULL) || (netmask != NULL) || (hostname != NULL)) {
      strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
      strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
      strcat(bootargs, ":eth0 ");
   }
   
   argv[argc++] = bootargs;
   argv[argc] = NULL;

   boot_kernel("ramdisk",
               (vaddr_t)(((unsigned long)flashword) + kernelPartition->base), kernelPartition->size, argc, argv, 0);
}

/****************************************************************************/
/*
 *
 * THIS IS A TEST FOR POSSIBLY BOOTING WINCE
 *
 * **************************************************************************/
void command_boot_wince(int argc, const char **argv)
{
    
    lcd_off(lcd_type);

    body_clearMem(0x100000,(unsigned short *)0xc0000000);
    
    bootLinux(0,
              2,
	      0x00040000
	);
   
}


/* can have arguments or not */
void command_boot_addr(
    int		    argc,
    const char**    argv)
{
    const char *ipaddr = NULL;
    const char *serveraddr = NULL;
    const char *gatewayaddr = NULL;
    const char *netmask = NULL;
    const char *hostname = NULL;
    const char *nfsroot = NULL;
    char bootargs[MAX_BOOTARGS_LEN];
    unsigned long    img_dest;
    unsigned long    img_size;
    
    if (argc < 3) {
	putstr("not enough args, need <address> <size>\n\r");
	return;
    }
    
    img_dest = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("error parsing img_dest\r\n");
	return;
    }
    img_size = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("error parsing img_size\r\n");
	return;
    }
    
    ipaddr = (const char *)param_ipaddr.value;
    serveraddr = (const char *)param_nfs_server_address.value;
    gatewayaddr = (const char *)param_gateway.value;
    netmask = (const char *)param_netmask.value;
    hostname = (const char *)param_hostname.value;
    nfsroot = (const char *)param_nfsroot.value;
    
    if (nfsroot != NULL) {
	strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
    }
    if ((ipaddr != NULL) || (serveraddr != NULL) || (gatewayaddr != NULL) ||
	(netmask != NULL) || (hostname != NULL)) {
	strcat(bootargs, " ip=");
	strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
	strcat(bootargs, ":");
	strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
	strcat(bootargs, ":");
	strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
	strcat(bootargs, ":");
	strcat(bootargs, (netmask != NULL) ? netmask : "");
	strcat(bootargs, ":");
	strcat(bootargs, (hostname != NULL) ? hostname : "");
	strcat(bootargs, ":eth0 ");
    }

    argv[argc++] = bootargs;
    argv[argc] = NULL;
    
    boot_kernel("ramdisk", (vaddr_t)img_dest, img_size, argc-2, argv+2,
		1);
}

/* can have arguments or not */
void command_boot_nfsroot(int argc, const char **argv)
{
    const char *ipaddr = NULL;
    const char *serveraddr = NULL;
    const char *gatewayaddr = NULL;
    const char *netmask = NULL;
    const char *hostname = NULL;
    const char *nfsroot = NULL;
    char bootargs[MAX_BOOTARGS_LEN];
    struct FlashRegion *kernelPartition = btflash_get_partition("kernel");

    if (kernelPartition == NULL) {
        putstr("no kernel partition defined\r\n");
        return;
    }

    ipaddr = (const char *)param_ipaddr.value;
    serveraddr = (const char *)param_nfs_server_address.value;
    gatewayaddr = (const char *)param_gateway.value;
    netmask = (const char *)param_netmask.value;
    hostname = (const char *)param_hostname.value;
    nfsroot = (const char *)param_nfsroot.value;

    strcat(bootargs, " noinitrd root=/dev/nfs");
    if (nfsroot != NULL) {
        strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
    }
    strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
    strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
    strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
    strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
    strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
    strcat(bootargs, ":eth0 ");
   
    argv[argc++] = bootargs;
    argv[argc] = NULL;

    boot_kernel("nfsroot",
                (vaddr_t)(((unsigned long)flashword) + kernelPartition->base), kernelPartition->size, argc, argv, 0);
}

#if defined(CONFIG_JFFS)
/* can have arguments or not */
void command_boot_jffs(int argc, const char **argv)
{
   char *kernel;
   unsigned int length;
   const char *ipaddr = NULL;
   const char *serveraddr = NULL;
   const char *gatewayaddr = NULL;
   const char *netmask = NULL;
   const char *hostname = NULL;
   const char *nfsroot = NULL;
   const char *boot_file = NULL;
   const char *conf_file = NULL;
   char bootargs[2048], imagename[256];
   struct jffs_conf *conf;

   if(argc < 2)
       boot_file = param_boot_file.value; 
   else
       boot_file = argv[1];

   conf_file = param_conf_file.value;

   bootargs[0] = '\0';

   /* Start building the kernel argument list now, since we might append
    * to it or overwrite it depending on what we see in the conf file.
    */

   ipaddr = param_ipaddr.value;
   nfs_server_address = param_nfs_server_address.value;
   gatewayaddr = param_gatewayaddr.value;
   netmask = param_netmask.value;
   hostname = param_hostname.value;
   nfsroot = param_nfsroot.value;

   if (nfsroot != NULL) {
     strcat(bootargs, " nfsroot="); strcat(bootargs, nfsroot);
   }
   if ((ipaddr != NULL) || (serveraddr != NULL) || (gatewayaddr != NULL) || (netmask != NULL) || (hostname != NULL)) {
      strcat(bootargs, " ip="); strcat(bootargs, (ipaddr != NULL) ? ipaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (serveraddr != NULL) ? serveraddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (gatewayaddr != NULL) ? gatewayaddr : "");
      strcat(bootargs, ":"); strcat(bootargs, (netmask != NULL) ? netmask : "");
      strcat(bootargs, ":"); strcat(bootargs, (hostname != NULL) ? hostname : "");
      strcat(bootargs, ":eth0 ");
   }
   
   if(jffs_init(jffs_changed) < 0)
     return;
   
   jffs_changed = 0;

   if(conf_file == NULL || jffs_conf((char *)conf_file, &conf) < 0){

     if(strcmp(boot_file, "?") == 0){  /* query for default boot file */

       putstr("Default boot_file is \""); putstr(boot_file); putstr("\"\r\n");

       return;

     } else {  /* command was not a query; try to boot something */

       if(boot_file == NULL){
	 putstr("No conf file or default boot_file available\r\n");
	 return;
       }

       strcpy(imagename, boot_file);

     }

   } else {  /* use conf file */

     if(strcmp(boot_file, "?") == 0){  /* query for list of boot images */

       jffs_conf_list(conf);

       return;

     } else {  /* command was not a query; try to boot something */

       if(jffs_conf_image(conf, boot_file, imagename, bootargs) < 0){
	 putstr("Unable to determine path of bootable image\r\n");
	 return;
       }

     }

   }

   /* We should now have the name of a file which the user believes is a
    * bootable image. The moment of truth:
    */
   if((kernel = jffs_file(imagename, &length)) == NULL){
     putstr("Unable to boot file \""); putstr(imagename); putstr("\"\r\n");
     return;
   }
   
   /* Rock. */

   putstr("Booting file \""); putstr(imagename); putstr("\"...\r\n");

   /* boot_kernel() considers nearly everything in argv to be a kernel
    * argument, so don't pass things (like the boot file/label) that
    * we wouldn't want to send to the kernel:
    */
   if(argc > 1)
     argc = 1;

   argv[argc++] = bootargs;

   /* boot_kernel() will only consider our kernel arguments if argv is
    * sufficiently large:
    */
   if(argc <= 2)
     argv[argc++] = "";

   argv[argc] = NULL;


   boot_kernel("jffs", kernel, length, argc, argv, 0);
}
#endif  /* CONFIG_JFFS */

void command_display(int argc, const char **argv)
{
#if defined(CONFIG_SA1100)
  void *uart = serbase();

   putstr("\r\nSA1100 Registers:\r\n    SA1100_UART" );
   putc((uart == Ser1Base) ? '1' : '3');
   putLabeledWord(":  ", (unsigned int)uart);
   putLabeledAddrWord( "      UTCR0   0x00 ", (long *)(uart + SA1100_UTCR0));
   putLabeledAddrWord( "      UTCR1   0x04 ", (long *)(uart + SA1100_UTCR1));
   putLabeledAddrWord( "      UTCR2   0x08 ", (long *)(uart + SA1100_UTCR2));
   putLabeledAddrWord( "      UTCR3   0x0c ", (long *)(uart + SA1100_UTCR3));
   putLabeledAddrWord( "      UTDR    0x10 ", (long *)(uart + SA1100_UTDR));
   putLabeledAddrWord( "      UTSR0   0x14 ", (long *)(uart + SA1100_UTSR0));
   putLabeledAddrWord( "      UTSR0+4 0x18 ", (long *)(uart + SA1100_UTSR0+4));
   putLabeledAddrWord( "      UTSR0+8 0x1c ", (long *)(uart + SA1100_UTSR0+8));
   putLabeledAddrWord( "      UTSR1   0x20 ", (long *)(uart + SA1100_UTSR1));

#endif  /* defined(CONFIG_SA1100) */
}


/* requires arguments */
void command_load_flash(int argc, const char **argv)
{
   unsigned long flash_dest = 0;
   int override;

   override = param_override.value;

   /* loading flash is different.. because writing flash is slow, we do
      not load it directly from the serial port. So we receive into
      memory first, and then program the flash... */
  
   if (argc < 2) { /* default to first command */
      putstr("usage: load flash <flashaddr>\r\n");
   } else {
      flash_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing flash_dest\r\n");
         return;
      }

      if (flash_dest < flashDescriptor->bootldr.size && !override) {
         putstr("That is bootloader space! Use load bootldr.  Operation canceled\r\n");
         return;
      }
      command_load_flash_region("flash", flash_dest, flash_size, 0);
   }
}


void command_load_ram(int argc, const char **argv)
{
  unsigned long ymodem;
  unsigned long img_size = 0;
  unsigned long img_dest = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("usage: load ram <ramaddr>\r\n");
  } else {
    /* parse the command line to fill img_size and img_dest */

    img_dest = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing img_dest\r\n");
      return;
    }
    
#ifdef CONFIG_YMODEM
    ymodem = param_ymodem.value;

    if (ymodem)
      img_size = ymodem_receive((char*)img_dest, SZ_8M);
    else
#endif
      img_size = modem_receive((char*)img_dest, img_size);

    /* in case this is a kernel download, update bt_memavail */
    bootinfo.bt_memavail = ROUNDUP(img_size, SZ_1M);

    putHexInt32(img_size);
    putLabeledWord(" bytes loaded to ",img_dest);
    putstr("\r\n");


    last_ram_load_address = img_dest;
    last_ram_load_size = img_size;
    
  }
}

/**
 * does the work for command_load, etc.
 */

static void program_flash_region(const char *regionName, 
                                 unsigned long regionBase, size_t regionSize, 
                                 unsigned long src, size_t img_size, int flags)
{
   unsigned long hdr_size = 0; /* size of header */
   unsigned long i;
   int remaining_bytes;
   long noerase;
   int isGZip = 0;
   unsigned char buf[64]; //extra space incase
   unsigned long compr_img_size = img_size;
   struct bz_stream z;
   int num_bytes = 0;
   int bytes_programmed = 0;
   unsigned char *p = buf;

   
   
   noerase = param_noerase.value;
   if (isGZipRegion(src))
       isGZip = 1;

   

   if (flags & LFR_PATCH_BOOTLDR) {
       // in order to do the verify, bootldrs are special
       if (isGZip){
	   putstr("uncompressing gzipped bootldr\r\n");
	   putstr("uncompressing gzipped bootldr\r\n");	   
	   if (!gUnZip(src,&img_size,src + SZ_256K)){
	       putstr("bad unzip of gzipped bootldr\r\n");
	       return;
	   }
	   src += SZ_256K;
	   isGZip = 0;	   
       }
       
       if (!isValidBootloader(src,img_size)){
	   putstr("Not loading invalid bootldr into flash\r\n");
	   return;
       }
       
   }

   if (flags & LFR_SIZE_PREFIX) {
      hdr_size += 4;
      src -= 4;
      *(long *)src = img_size;
      img_size += hdr_size;
   }
   
   // let's see if we got a gzip image to play with.
   if (isGZip)
   {

       putstr("Looks like a gzipped image, let's verify it...\r\n");
       putstr("Looks like a gzipped image, let's verify it...\r\n");
       if (!verifyGZipImage(src,&img_size)){	   
	   putstr("invalid gzip image.  Sorry...\r\n");
	   return;
       }
       // img_size now has the uncomopressed size in it.
       gzInitStream(src,compr_img_size,&z);

   }
       

   

   if (img_size > regionSize) { // xple errors since minicom tends to eat some(all?)
      putLabeledWord("img_size is too large for region: ", regionSize);
      putLabeledWord("img_size is too large for region: ", regionSize);
      putLabeledWord("img_size is too large for region: ", regionSize);
      return;
   }

   putstr("\r\nprogramming flash...");

   set_vppen();
   if (flags & LFR_PATCH_BOOTLDR) {
      putstr( "\r\nunlocking boot sector of flash\r\n" );
      protectFlashRange( 0x0, img_size+hdr_size, 0);
   }
   if (!noerase) {
     int eraseSize = img_size + hdr_size;
     if (flags & LFR_JFFS2)
       eraseSize = regionSize;
     putstr("erasing ...\r\n");
     if (eraseFlashRange(regionBase,eraseSize)) {
       putstr("erase error!\r\n");
       goto clear_VPPEN;
     }
   }

   /* the magic number.. */
   putstr("writing flash..\r\n");
   i = 0;
   remaining_bytes = img_size;
   while(remaining_bytes > 0) { 
      
      if (isGZip){
	  num_bytes -= bytes_programmed;
	  p += bytes_programmed;
	  //putLabeledWord("pre num_bytes = 0x",num_bytes);
	  //putLabeledWord("pre remaining_bytes = 0x",remaining_bytes);
	  
	  if (!num_bytes){
	      num_bytes = gzRead(&z,buf,64);
	      p = buf;
	      //putLabeledWord("gzRead gives num_bytes = 0x",num_bytes);
	      
	  }

      }
      else{
	  num_bytes = 64;
	  p = buf;	  
	  memcpy((void *)p,(const void *)(src+i),num_bytes);
      }
      
      if ((i % SZ_64K) == 0) {
         putstr("addr: ");
         putHexInt32(regionBase+i);
         putstr(" data: ");
         putHexInt32(*(unsigned long *)p);
         putstr("\r\n");
      }

      if (((i % 64) == 0) && (remaining_bytes > 64) && (num_bytes == 64)) {
         /* program a block */
         if (programFlashBlock(regionBase+i, (unsigned long*)p, 64)) {
            putstr("error while copying to flash!\r\n");
            goto clear_VPPEN;
            break;
         }
         bytes_programmed = 64;
      } else {
         if (programFlashWord(regionBase+i, *(unsigned long *)p)) {
            putstr("error while copying to flash!\r\n");
            goto clear_VPPEN;
            break;
         }
         bytes_programmed = 4;
      }
      i += bytes_programmed;
      remaining_bytes -= bytes_programmed;
      
   }
   putstr("verifying ... \r\n");

   if (isGZip){
       unsigned long calc_crc32;
       unsigned char *dst = (unsigned char *)&flashword[regionBase >> 2];       
       calc_crc32 = crc32(0,dst,img_size);
       putLabeledWord("calculated crc32 = 0x",calc_crc32);
       putLabeledWord("desired    crc32 = 0x",z.read_crc32);
       if (calc_crc32 != z.read_crc32){
	   putstr("error programming flash (crc check failed)\r\n");
	   goto clear_VPPEN;
       }
   }
   else
   {
      int i;
      int nwords = (img_size >> 2);
      unsigned long *srcwords = (unsigned long *)src;
      unsigned long *dstwords = (unsigned long *)&flashword[regionBase >> 2];

      for (i = 0; i < nwords; i++) {
         if (srcwords[i] != dstwords[i]) {
            putLabeledWord("error programming flash at offset=", i << 2);
            putLabeledWord("  src=", srcwords[i]);
            putLabeledWord("  flash=", dstwords[i]);
            putstr("not checking any more locations\r\n");
            goto clear_VPPEN;
         }
      }
   }   
   if (flags & LFR_JFFS2) {
     unsigned long jffs2_sector_marker0 = 0xFFFFFFFF;
     unsigned long jffs2_sector_marker1 = 0xFFFFFFFF;
     unsigned long jffs2_sector_marker2 = 0xFFFFFFFF;
     jffs2_sector_marker0 = param_jffs2_sector_marker0.value;
     jffs2_sector_marker1 = param_jffs2_sector_marker1.value;
     jffs2_sector_marker2 = param_jffs2_sector_marker2.value;
     putstr("formatting ... ");
     btflash_jffs2_format_region(regionBase + img_size + hdr_size, regionSize - img_size - hdr_size, jffs2_sector_marker0, jffs2_sector_marker1, jffs2_sector_marker2);
   }
   putstr("done.\r\n");
   if (flags & LFR_PATCH_BOOTLDR) {
      protectFlashRange( 0x0, img_size+hdr_size, 1);
   }

 clear_VPPEN:
   clr_vppen();
}

static void command_load_flash_region(const char *regionName, unsigned long regionBase, unsigned long regionSize, int flags)
{
   unsigned long img_size = 0; /* size of data */
   unsigned long img_dest = VKERNEL_BASE + 1024; /* just a temporary holding area */
   long ymodem = 0;


   if (amRunningFromRam() && regionBase >= flashDescriptor->bootldr.size) {
       putstr("Can't load to partition <");
       putstr(regionName);
       putstr("> while running from ram.  Operation canceled\r\n");
       return;
   }

   
   putstr("loading flash region "); putstr(regionName); putstr("\r\n");

#ifdef CONFIG_YMODEM
   ymodem = param_ymodem.value;
#endif

   if (ymodem) {
#ifdef CONFIG_YMODEM
      putstr("using ymodem\r\n");
      img_size = ymodem_receive((char*)img_dest, regionSize);
#endif
   } else {
      putstr("using xmodem\r\n");
      img_size = modem_receive((char*)img_dest, regionSize);
   }
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   }
    
   putHexInt32(img_size);
   putLabeledWord(" bytes loaded to ",img_dest);
   if ((img_size % 4) != 0) {
      putstr("img_size is not a multiple of 4 -- are we sure that's OK?\r\n");
   }

   program_flash_region(regionName, regionBase, regionSize, img_dest, img_size, flags);
}

/* requires arguments */
void command_load(int argc, const char **argv)
{
    if (check_for_nested_help(argc,argv,load_commands))
	return;

  if (argc > 1) {
    const char *partname = argv[1];
    struct FlashRegion *partition = btflash_get_partition(partname);
    if (partition == NULL)
      goto handle_subcommand;
    command_load_flash_region(partname, partition->base, partition->size, partition->flags);
    if (strcmp(partname, "params") == 0) {
      params_eval(PARAM_PREFIX, 0);
    }
  } else {
  handle_subcommand:
    execcmd(load_commands,argc-1,argv+1);
  }
}

/* can have arguments or not */
void command_save(int argc, const char **argv)
{
    if (check_for_nested_help(argc,argv,save_commands))
	return;

  if (argc > 1) {
    const char *partname = argv[1];
    struct FlashRegion *partition = btflash_get_partition(partname);
    if (partition == NULL)
      goto handle_subcommand;
    putstr("About to xmodem send "); putstr(partition->name); putstr("\r\n");
    putLabeledWord("  flashword=", (unsigned long)flashword);
    putLabeledWord("  base=", partition->base);
    putLabeledWord("  nbytes=", partition->size);
    if (!modem_send(partition->base + (char *)flashword, partition->size)) {
      putstr("download error.\r\n");
      return;
    } 
  } else {
  handle_subcommand:
    execcmd(save_commands,argc-1,argv+1);
  }
}

void command_save_all(int argc, const char **argv) 
{
   dword img_size;

   img_size = modem_send((char*)flashword, flash_size); 
   if (!img_size) {
      putstr("download error. aborting.\r\n");
      return;
   } 
}

void command_save_flash(int argc, const char **argv) {

   dword         img_size;
   unsigned long flash_size = 0;
   unsigned long flash_dest = 0;

   if (argc < 3) { /* default to first command */
      putstr("usage: load flash <flashaddr> <size>\r\n");
   } else {
      flash_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing <flashaddr>\r\n");
         return;
      }
      flash_size = strtoul(argv[2], NULL, 0);
      if (strtoul_err) {
	putstr("error parsing <size>\r\n");
      }
      putLabeledWord("flash_dest=", flash_dest );
      putLabeledWord("flash_size=", flash_size );
      img_size = modem_send( flash_dest + (char*)flashword, flash_size );
      if (!img_size) {
         putstr("download error. aborting.\r\n");
         return;
      }
   }
   return;
}

void command_save_ram(int argc, const char **argv) {

   dword         img_size;
   unsigned long ram_size = 0;
   unsigned long ram_dest = 0;

   if (argc < 3) { /* default to first command */
      putstr("usage: save ram <ramaddr> <size>\r\n");
   } else {
      ram_dest = strtoul(argv[1], NULL, 0);
      if (strtoul_err) {
         putstr("error parsing <ramaddr>\r\n");
         return;
      }
      ram_size = strtoul(argv[2], NULL, 0);
      if (strtoul_err) {
	putstr("error parsing <size>\r\n");
      }
      putLabeledWord("ram_dest=", ram_dest );
      putLabeledWord("ram_size=", ram_size );
      img_size = modem_send((char*)ram_dest, ram_size );
      if (!img_size) {
         putstr("download error. aborting.\r\n");
         return;
      }
   }
   return;
}

void command_save_world(int argc, const char **argv) {

  putstr( "\r\nI don't know how to save the world.\r\n\r\n" );

}

void command_peek(int argc, const char **argv)
{
    if (check_for_nested_help(argc,argv,peek_commands))
	return;
    
   execcmd(peek_commands,argc-1,argv+1);
}

void command_peek_ram(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long result = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("peek ram requires arguments!\r\n");
  } else {
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    if (strcmp(argv[0], "byte") == 0)
       result = *(unsigned char *)(addr);
    else if (strcmp(argv[0], "short") == 0)
       result = *(unsigned short *)(addr);
    else if (strcmp(argv[0], "int") == 0)
       result = *(int *)(addr);
    else if (strcmp(argv[0], "flash") == 0)
       result = flashword[(addr&flash_address_mask) >> 2];
    else if (strcmp(argv[0], "gpio") == 0)
       /* read from the gpio read port */
       result = *(int *)(0x21800008 + addr);
    else 
       result = *(unsigned long *)(addr);

    putLabeledWord("  addr  = ",addr);
    putLabeledWord("  value = ",result);
  }
}

#if defined(CONFIG_JFFS)

void command_peek_jffs(int argc, const char **argv)
{
  unsigned int inumber, size;

  switch(argc){

  case 2:

    inumber = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing inumber\r\n");
      return;
    }
    
    jffs_dump_inode(inumber);

    break;

  case 3:
    size = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing size\r\n");
      return;
    }

    if(jffs_init(jffs_changed) < 0)
      return;
    
    jffs_changed = 0;
  
    jffs_dump_file((char *)argv[1], size);

    break;

  default:
    putstr("peek jffs requires inumber argument or filename and length\r\n");
  }

}

#endif  /* CONFIG_JFFS */

void command_poke(int argc, const char **argv)
{
    if (check_for_nested_help(argc,argv,poke_commands))
	return;
    execcmd(poke_commands,argc-1,argv+1);
}

void command_poke_ram(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long value = 0;
  
  if (argc < 3) { /* default to first command */
    putstr("poke ram requires arguments!\r\n");
  } else {

    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    value = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing value\r\n");
      return;
    }

    putstr("poke ram: \r\n");
    putLabeledWord(" addr=", addr);
    putLabeledWord(" value=", value);

    if (strcmp(argv[0], "byte") == 0) {
       *(unsigned char *)(addr) = value;
    } else if (strcmp(argv[0], "short") == 0) {
       *(unsigned short *)(addr) = value;
    } else if (strcmp(argv[0], "int") == 0) {
       *(int *)addr = value;
    } else if (strcmp(argv[0], "gpio") == 0) {
       /* write to the gpio write port */
       *(int *)(0x21800000 + addr) = value;
    } else if (strcmp(argv[0], "flash") == 0) {
       set_vppen();
       if (programFlashWord(addr & flash_address_mask, value))
          putstr("flash write failed!\r\n");
       clr_vppen();
    } else {
       *(unsigned long *)(addr) = value;
    }
  }
}


#if defined(CONFIG_SA1100)
extern unsigned long delayedBreakpointPC;
void command_breakpoint(int argc, const char **argv)
{
  if (argv[1] != NULL) {
    const char *pcstr = NULL;
    unsigned long pc;
    int delayed = 0;
    if (strcmp(argv[1], "delayed") == 0) {
      delayed = 1;
      pcstr = argv[2];
    } else {
      pcstr = argv[1];
    }
    pc = strtoul(pcstr, NULL, 0);
    pc &= 0xFFFFFFFC;
    pc |= 1;
    if (delayed) {
      putLabeledWord("PC breakpoint will be set after kernel unzip at: ", pc);
      delayedBreakpointPC = pc;
    } else {
      putLabeledWord("Setting hardware PC breakpoint at: ", pc);
      __asm__ ("mcr     p15, 0, %0, c14, c8, 0" : : "r" (pc));
    }
  } else {
    unsigned long pc = 0;
    putstr("Clearing PC breakpoint");
    __asm__ ("mcr     p15, 0, %0, c14, c8, 0" : : "r" (pc));
  }
}
#endif

void command_qflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long result = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("qflash requires arguments!\r\n");
  } else {
    addr = strtoul(argv[argc-1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr\r\n");
      return;
    }

    if (strcmp(argv[1], "id") == 0) {
       result = queryFlashID(addr);
    } else if (strcmp(argv[1], "security") == 0) {
       result = queryFlashSecurity(addr);
    } else {
       result = queryFlash(addr);
    }
    putLabeledWord("value = ",result);
  }
}

void command_eflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long len = 0;
  int override;

  override = param_override.value;

  if (argc < 2) { /* default to first command */
    putstr("eflash requires arguments: <partition>|<addr> [<len>]|chip!\r\n");
    return;
  } else {
    struct FlashRegion *partition = btflash_get_partition(argv[1]);
    if (partition != NULL) {
      putstr("erasing partition "); putstr(argv[1]); putstr("\r\n");
      addr = partition->base;
      len = partition->size;
      goto erase;
    } else if (strncmp(argv[1], "chip", 4) == 0) {
      if (!override) {
	putstr("Cannot erase whole chip without setting override to 1.\r\n");
	return;
      }
      putstr("erasing flash chip\r\n");
      set_vppen();
      eraseFlashChip();
      clr_vppen();
      return;
    }
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr: "); putstr(argv[1]); putstr("\r\n");
    }
    putLabeledWord("addr=", addr);

    if (argc > 2) {
      len = strtoul(argv[2], NULL, 0);
      if (strtoul_err) {
        putstr("error parsing len: "); putstr(argv[2]); putstr("\r\n");
      }
      putLabeledWord("len=", len);

    }
  erase:
    if (addr == 0 && !override) {
      putstr("Cannot erase first sector without setting override to 1.\r\n");
      return;
    }

    set_vppen();
    if (len == 0) 
      eraseFlashSector(addr);
    else
      eraseFlashRange(addr, len);
    clr_vppen();
  }
}

void command_pflash(int argc, const char **argv)
{
  unsigned long addr = 0;
  unsigned long len = 0;
  unsigned long protect = 9;
  
  if (argc < 4) { /* default to first command */
    putstr("pflash requires arguments: <addr> <len> 0/1  (1 -> protect, 0 -> unprotect all)!\r\n");
  } else {
    addr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing addr: "); putstr(argv[1]); putstr("\r\n"); return;
    }
    len = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing len: "); putstr(argv[2]); putstr("\r\n"); return;
    }
    protect = strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing protect: "); putstr(argv[3]); putstr("\r\n"); return;
    }
    /* be certain that protect is only 0/1 */
    if (!((protect == 0) || (protect == 1)) ) {
      putstr("error protect value must be 0/1: "); putstr(argv[3]); putstr("\r\n"); return;
    }
    
    if (verifiedFlashRange( addr, len )) {
       putLabeledWord("  addr=", addr);
       putLabeledWord("  len=", len);
       putLabeledWord("  protect=", protect);

       set_vppen();
       protectFlashRange(addr, len, protect);
       clr_vppen();
    } else { 
      putstr("Region specified is out of Range.\r\n" );
      putLabeledWord( " Please use a range less than:", flash_size );      
    }
  }
}


void command_call(int argc, const char **argv)
{
  void (*fcn)(long a0, long a1, long a2, long a3) = NULL;
  
  if (argc < 2) {
    putstr("usage: call <addr>\r\n");
  } else {
    long a0 = 0;
    long a1 = 0;
    long a2 = 0;
    long a3 = 0;

    if (argv[1][0] == '.' || argv[1][0] == '=' || argv[1][0] == '-') {
	if (last_ram_load_address != 0) {
	    strtoul_err = 0;
	    fcn = (void*)last_ram_load_address;
	}
	else {
	    putstr("last_ram_load_address is 0.\n\r");
	    return;
	}
    }
    else
	fcn = (void*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing vaddr\r\n");
      return;
    }
    if (argc > 2) {
      a0 = strtoul(argv[2], NULL, 0);
    }
    if (argc > 3) {
      a1 = strtoul(argv[3], NULL, 0);
    }
    if (argc > 4) {
      a2 = strtoul(argv[4], NULL, 0);
    }
    if (argc > 5) {
      a3 = strtoul(argv[5], NULL, 0);
    }
    asmEnableCaches(0,0);
    putLabeledWord("Calling fcn=", (long)fcn);
    putLabeledWord("  a0=", a0);
    putLabeledWord("  a1=", a1);
    putLabeledWord("  a2=", a2);
    putLabeledWord("  a3=", a3);

    let_uart_drain(SA1100_UART3_UTSR1);
    
    fcn(a0, a1, a2, a3);
  }
}

void command_physaddr(int argc, const char **argv)
{
  unsigned long vaddr = 0;
  
  if (argc < 2) { /* default to first command */
    putstr("physaddr requires vaddr argument!\r\n");
  } else {
    vaddr = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
      putstr("error parsing vaddr\r\n");
      return;
    }
    {
      unsigned long mmu_table_entry = mmu_table[vaddr >> 20];
      unsigned long paddr = mmu_table_entry & ~((1L << 20) - 1);
      unsigned long sectionDescriptor = mmu_table_entry & ((1L << 20) - 1);
      putLabeledWord("vaddr=", vaddr);
      putLabeledWord("paddr=", paddr);
      putLabeledWord("sectionDescriptor=", sectionDescriptor);
    }
  }
}

void command_set(int argc, const char **argv)
{
   char pName[256] = "";
   char pValue[256] = "";
   

   parseParamName(argc,argv,pName,pValue);
   
   if ((argc == 1) && (pName[0] = '\0')) {
       /* no argses */
       int my_argc = 1;
       char *my_argv[128];
       my_argv[0] = "show";
       my_argv[1] = NULL;
       execcmd(commands, my_argc, (const char **)my_argv);
       return;
   }

   set_param_value(pName,pValue);
}

void command_params_show(int argc, const char **argv)
{
   struct bootblk_param *params = (struct bootblk_param *)&__params_begin;
   int i;
   unsigned char showMe;
   
   print_version_verbose(">> ");

   while (params < (struct bootblk_param *)&__params_end) {
       showMe = 0;
       if (argc > 1){
	   for (i=1;i < argc; i++)
	       if (!strcmp(params->name,argv[i]))
		   showMe = 1;
       }
       else
	   showMe = 1;
       if (showMe){
	   putstr("  ");
	   putstr(params->name);
	   switch (params->paramFormat) {
	       case PF_STRING:
		   putstr("= <"); putstr((char*)params->value); putstr(">\r\n");
		   break;
	       case PF_DECIMAL: {
		   char buf[16];
		   dwordtodecimal(buf, params->value);
		   putstr("=<"); putstr(buf); putstr(">\r\n");
		   break;
	       }
	       case PF_HEX:
	       default:
		   putstr("=<0x"); putHexInt32(params->value); putstr(">\r\n");
		   break;
	   }
       }
       params++;
   }
}

#ifdef UPDATE_BAUDRATE
void update_baudrate(struct bootblk_param *param)
{
#ifdef CONFIG_SKIFF
   volatile unsigned long *csr = (volatile unsigned long *)DC21285_ARMCSR_BASE;
   unsigned long system_rev = get_system_rev();
   unsigned long fclk_hz = 33333333;
#endif
   int baudrate = param->value;
   unsigned long ubrlcr;
   unsigned long l_ubrlcr = 0;
   unsigned long m_ubrlcr = 0;
#ifdef CONFIG_SKIFF
   unsigned long h_ubrlcr = 0;
#endif


   
   switch ( baudrate ) {
#if defined(CONFIG_SA1100)
   case 230400:
     break;
   case 115200:
     break;
#endif
   case 57600:
     break;
   case 38400:
     break;
   case 19200:
     break;
   case 9600:
     break;
   case 4800:
     break;
   case 2400:
     break;
   case 1200:
     break;
   case 300:
     break;
   case 110:
     break;
   default:
     param->value = paramoldvalue;
     putstr( "invalid baud rate:\r\n" );
     putstr( "     Please, try:   110,   300,  1200,   2400,  4800,  9600\r\n" );
     putstr( "                  19200, 38400, 57600, 115200 or 230,400\r\n" );
     return; 
   }
   
#ifdef CONFIG_SKIFF
   switch (system_rev&SYSTEM_REV_MEMCLK_MASK) {
   case SYSTEM_REV_MEMCLK_33MHZ:
      fclk_hz = 33333333;
      break;
   case SYSTEM_REV_MEMCLK_48MHZ:
      fclk_hz = 48000000;
      break;
   case SYSTEM_REV_MEMCLK_60MHZ:
      fclk_hz = 60000000;
      break;
   }

  fclk_hz = 48000000;

   ubrlcr = (fclk_hz / 64 / baudrate)-1;
   h_ubrlcr = csr[H_UBRLCR_REG];

#elif defined(CONFIG_SA1100)
   ubrlcr = (3686400 / 16 / baudrate) - 1;
#else
   #error no architecture defined in update_baudrate
#endif

   l_ubrlcr = ubrlcr & 0xFF;
   m_ubrlcr = (ubrlcr >> 8) & 0xFF;
   
   putLabeledWord("update_baudrate:  new baudrate=", baudrate);

#if defined(CONFIG_SKIFF)
   /* wait for the TX FIFO to empty */
   while ((csr[UARTFLG_REG]&UART_TX_BUSY) != 0)
      /* spin */;

   /* disable the uart */
   csr[UARTCON_REG] = 0;

   /* set the new values, in the right order */
   csr[L_UBRLCR_REG] = l_ubrlcr;
   csr[M_UBRLCR_REG] = m_ubrlcr;
   csr[H_UBRLCR_REG] = h_ubrlcr;

   /* reenable the uart */
   csr[UARTCON_REG] = 1;
#elif defined(CONFIG_BITSY) || defined(CONFIG_SPOT) || defined(CONFIG_GATOR)
   /* wait for the TX FIFO to empty */
  while (((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_TBY)); /* wait until TX not busy */
  *(volatile byte*)SA1100_UART3_UTCR3 = 0; /* disable UART */

  *(volatile byte*)SA1100_UART3_UTCR1 = m_ubrlcr; /* MSB of baudrate divisor */
  *(volatile byte*)SA1100_UART3_UTCR2 = l_ubrlcr; /* LSB of baudrate divisor */
  
  *(volatile byte*)SA1100_UART3_UTCR3 = SA1100_UTCR3_RXE|SA1100_UTCR3_TXE; /* enable UART */
#elif defined(CONFIG_ASSABET)
   /* wait for the TX FIFO to empty */
  while (((machine_has_neponset() ? (*(volatile long *)SA1100_UART3_UTSR1) :
	   (*(volatile long *)SA1100_UART1_UTSR1)) & SA1100_UTSR1_TBY));
  (machine_has_neponset() ? (*(volatile byte*)SA1100_UART3_UTCR3) :
   (*(volatile byte*)SA1100_UART1_UTCR3)) = 0;

  (machine_has_neponset() ? (*(volatile byte*)SA1100_UART3_UTCR1) :
   (*(volatile byte*)SA1100_UART1_UTCR1)) = m_ubrlcr;
  (machine_has_neponset() ? (*(volatile byte*)SA1100_UART3_UTCR2) :
   (*(volatile byte*)SA1100_UART1_UTCR2)) = l_ubrlcr;
  
  (machine_has_neponset() ? (*(volatile byte*)SA1100_UART3_UTCR3) :
   (*(volatile byte*)SA1100_UART1_UTCR3)) = SA1100_UTCR3_RXE|SA1100_UTCR3_TXE;
#endif

   putLabeledWord(" baudrate changed to 0x", baudrate);

   /* set the value for the kernel too */

   bootinfo.bt_comspeed = baudrate;
}
#endif

void set_serial_number(struct bootblk_param *param)
{
}


#ifdef CONFIG_SKIFF
long get_system_rev()
{
   long word1 = *(long *)FLASH_BASE;
   long word2 = *(long *)(FLASH_BASE+0x0080000C);
   long revision = SYSTEM_REV_SKIFF_V1;
   if (word1 != word2) {
      /* Skiff-V2 -- GPIO register 0 contains the revision mask */
      revision = (word2 & SYSTEM_REV_MASK);
   }
   return revision;
}

void set_system_rev(struct bootblk_param *param)
{
   param->value = get_system_rev();
}
#endif

#ifdef CONFIG_SKIFF
/* seems not to work on sa1110 -- Brian Avery */
void flush_caches(void)
{
   writeBackDcache(CACHE_FLUSH_BASE);
   __asm__("  mcr p15, 0, r0, c7, c7, 0x00 /* flush I+D caches */\n"
           "  mcr p15, 0, r0, c7, c10, 4 /* drain the write buffer*/\n");
}
#else
void flush_caches(void)
{
}
#endif

long enable_caches(int dcache_enabled, int icache_enabled)
{
  long mmuc;
  long enabled = (icache_enabled ? 0x1000 : 0x0);
  enabled |= (dcache_enabled ? 0xC : 0);
  flush_caches();
  __asm__("  mrc p15, 0, %0, c1, c0, 0\n"
	  "  bic %0, %0, #0x1000\n"
	  "  bic %0, %0, #0x000c\n"
	  "  orr %0, %0, %1\n"
	  "  mcr p15, 0, %0, c1, c0, 0\n"
	  : "=&r" (mmuc) : "r" (enabled));
  flush_caches();
  return mmuc;
}

void update_dcache_enabled(struct bootblk_param *param)
{
   long dcache_enabled = param->value;
   long icache_enabled = 0;
   long mmuc;
   icache_enabled = param_icache_enabled.value;
   mmuc = enable_caches(dcache_enabled, icache_enabled);
   putLabeledWord("MMU Control word=", mmuc);
}

void update_icache_enabled(struct bootblk_param *param)
{
   long dcache_enabled = 0;
   long icache_enabled = param->value;
   long mmuc;
   dcache_enabled = param_dcache_enabled.value;
   mmuc = enable_caches(dcache_enabled, icache_enabled);
   putLabeledWord("MMU Control word=", mmuc);
}

int
param_is_modified(
    const struct bootblk_param*	param,
    const struct bootblk_param* def_param)
{
    /*
     * compare params, taking into account the fact that
     * a string can have a different address and the same
     * value.
     */
    
    if (param->paramType == PT_STRING) {
	if (param->value && def_param->value) {
	    return (strcmp((char*)(param->value),
			   (char*)(def_param->value)) != 0);
	}
	else {
	    /*
	     * we know both aren't non-null.
	     * if both are then that is a match
	     */
	    return (param->value || def_param->value);
	}
    }
    else {
	return (param->value != def_param->value);
    }
    
}

void command_params_save(int argc, const char **argv)
{
   char param_buf[SZ_16K];
   char *buf = param_buf;
   int partition_table_size = 0;
   int dont_save = 0;
   struct FlashRegion *paramsPart;
   struct bootblk_param *param = (struct bootblk_param *)&__params_begin;
   /* default params points into the flash image as opposed to the copy in dram */
   const struct bootblk_param *defaultParam = 
      (const struct bootblk_param *)(((char*)flashword) + (((dword)&__params_begin) - FLASH_BASE));

   /* should pick up the old params from flash in case there are any non-bootldr lines */

   memset(param_buf, 0, sizeof(param_buf));

   partition_table_size = (sizeof(struct BootldrFlashPartitionTable) 
                           + partition_table->npartitions*sizeof(struct FlashRegion));
   memcpy(buf, partition_table, partition_table_size);

   buf += partition_table_size;

   if (argc > 1 && strcmp(argv[1], "-n") == 0) {
       PLW(flashword);
       PLW(&__params_begin);
       PLW(FLASH_BASE);
       PLW(defaultParam);
       dont_save = 1;
   }

   /* construct the parameter setting code */
   while (param < (struct bootblk_param *)&__params_end) {
      int modifiedFromDefault = param_is_modified(param,
						  defaultParam); 
      if (modifiedFromDefault && !(param->paramType&PT_READONLY)) {
         strcat(buf, "bootldr: set ");
         strcat(buf, param->name);
         strcat(buf, " ");
         switch (param->paramFormat) {
         case PF_STRING:
	    strcat(buf, "\"");
            if(param->value)
	      strcat(buf, (char *)param->value);
	    strcat(buf, "\"");
            break;
         case PF_DECIMAL: {
           char num[32];
           dwordtodecimal(num, param->value);
           strcat(buf, num);
         } break;
         case PF_HEX:
         default: {
            char num[16];
            strcat(buf, "0x");
            binarytohex(num, param->value,4);
            strcat(buf, num);
         }         
         }
         strcat(buf, "\r\n");
      }
      param++;
      defaultParam++;
   }
   putstr(buf);

   if (dont_save)
       putstr("Not erasing and writing params sector.\n\r");
   else {
       /* now erase and program the params sector */
       paramsPart = btflash_get_partition("params");
       if (paramsPart)
	   program_flash_region("params", paramsPart->base,
				paramsPart->size,
				(dword)param_buf,
				partition_table_size + strlen(buf)+1,
				paramsPart->flags);
   
       else {
	   putstr("No params partition found.  I cannot save params into Wince or JFFS2 files!!!\r\n");
   }
   }
   

   
}


void command_cmdex(int argc, const char **argv)
{
    
    if (argc > 1)
	 use_extended_getcmd = strtoul(argv[1], NULL, 0);
    else
	use_extended_getcmd = !use_extended_getcmd;

    putLabeledWord("use_extended_getcmd=0x", use_extended_getcmd);    
}

void
update_autoboot_timeout(
    struct bootblk_param *param)
{
    autoboot_timeout = (unsigned long)param->value;
    if (autoboot_timeout == 0)
	putstr("Autoboot is DISABLED\n\r");
}

void
update_cmdex(
    struct bootblk_param *param)
{
    use_extended_getcmd = (unsigned long)param->value;
}

#if defined(CONFIG_BITSY)
void command_aux_ser(
    int argc,
    const char **argv)
{
    auxm_serial_dump();
}
#endif



void
command_led_blink(
    int	    argc,
    const char*   argv[])
{
    char ledData[] = {0x01,0x00,0x05,0x05};
    
    if (argc > 1) 
	ledData[2] = (char) strtoul(argv[1], NULL, 0);
    
    if (argc > 2)
	ledData[2] = (char) strtoul(argv[2], NULL, 0);

    if (argc > 3)
	ledData[0] = (char) strtoul(argv[3], NULL, 0);
    
    led_blink(ledData[0],ledData[1],ledData[2],ledData[3]);
    
}

#if 1

voidpf
zcalloc(
    voidpf opaque,
    unsigned items,
    unsigned size)
{
    void*   p;
    int	    totsize;

    p = mmalloc(totsize = items * size);
    if (p) {
	memset(p, 0x00, totsize);
    }
    return(p);
}

void
zcfree(
    voidpf opaque,
    voidpf ptr)
{
    mfree(ptr);
}

#undef free
void
free(
    void*   p)
{
    mfree(p);
}

char	zbuf[16384];
#endif

void
command_lcd_zimage(
    int	    argc,
    const char*   argv[])
{
    char*   fb = lcd_get_image_buffer();
    dword   img_size;
    uLongf  uncomplen = lcd_get_image_buffer_len();
    int	    rc;

    if (argc < 2) {
	putstr("need args: len\n\r");
	return;
    }
    
    img_size = strtoul(argv[1], NULL, 0);
    if (img_size > sizeof(zbuf)) {
	putLabeledWord("img too big, zbuf size: ", sizeof(zbuf));
	return;
    }
	
    rc = modem_receive(zbuf, img_size);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    putLabeledWord("modem_rx returned, rc: 0x", rc);
    if (rc == 0) {
	putstr("download failed.  Aborting\n\r");
	return;
    }

#if 0
    
    rc = uncompress(fb, &uncomplen, zbuf, img_size);
    putLabeledWord("uncompress returned, rc: 0x", rc);
    if (rc != Z_OK) {
	putLabeledWord("uncompress failed, rc: 0x", rc);
	return;
    }

#else
    
    {
	/*
	 * do it piecemeal to test out using zlib stream functions...
	 */
	z_stream stream;
	int	err;
	int	err2;
	char	tiny_buf[128];
	char*	fbp = fb;
	size_t tlen;
	
	stream.next_in = (Bytef*)zbuf;
	stream.avail_in = (uInt)img_size;
	
	stream.next_out = tiny_buf;
	stream.avail_out = (uInt)sizeof(tiny_buf);
	
	stream.zalloc = (alloc_func)0;
	stream.zfree = (free_func)0;
	
	err = inflateInit(&stream);
	if (err != Z_OK) {
	    putLabeledWord("inflateInit failed", err);
	    return;
	}

	while ((err = inflate(&stream, Z_SYNC_FLUSH)) == Z_OK) {
	    tlen = sizeof(tiny_buf) - stream.avail_out;

	    if (tlen) {
		memcpy(fbp, tiny_buf, tlen);
		fbp += tlen;
		putLabeledWord("memcpy 0x", tlen); 
	    }

	    if (stream.avail_out == 0) {
		  putstr("reset obuf\n\r"); 
		stream.next_out = tiny_buf;
		stream.avail_out = (uInt)sizeof(tiny_buf);
	    }
	}
	
	    
	if (err == Z_STREAM_END) {
	    tlen = sizeof(tiny_buf) - stream.avail_out;
	    if (tlen)
		memcpy(fbp, tiny_buf, tlen);
	}
	    
	err2 = inflateEnd(&stream);
	
	if (err != Z_STREAM_END) {
	    putLabeledWord("inflate failed", err);
	    return;
	}

	if (err2 != Z_OK) {
	    putLabeledWord("inflateEnd failed", err2);
	    return;
	}
	
	uncomplen = stream.total_out;
    }
    
#endif
    
    putstr("about to display\n\r");

#if 1
    lcd_display_bitmap(fb, uncomplen, lcd_type, NULL);
#else
    putstr("skipping pixel conversion");
#endif    
}

#if defined(CONFIG_BITSY) || defined(CONFIG_ASSABET)

void
command_ttmode(
    int		argc,
    const char*	argv[])
{
    static int	ttmode = 0;
    int	new_mode;

    if (argc > 1)
	new_mode = strtoul(argv[1], NULL, 0);
    else
	new_mode = 1;		/* no args turns on */

    if (new_mode) {
	packetize_output = 1;
	ack_commands = 1;
	no_cmd_echo = 1;
	use_extended_getcmd = 0;
    }
    else {
	packetize_output = 0;
	ack_commands = 0;
	no_cmd_echo = 0;
	use_extended_getcmd = 1;
    }

    ttmode = new_mode;
}

void
command_ser_con(
    int		argc,
    const char* argv[])
{
    putstr("serial console at your service...\n\r");    
}

void
command_irda_con(
    int		argc,
    const char* argv[])
{
    putstr("irda not available yet, starting serial console.\n\r");
}


int
reboot_button_is_enabled(
    void)
{
    return (reboot_button_enabled > 0);
}

void
enable_reboot_button(
    void)
{
    reboot_button_enabled++;
}

void
disable_reboot_button(
    void)
{
    if (reboot_button_enabled > 0)
	reboot_button_enabled--;
}
#endif

void
command_memcpy(
    int		argc,
    const char* argv[])
{
    void*   dst;
    void*   src;
    int	    num;
    int	    size = 1;
    
    if (argc < 4) {
	putstr("memcpy needs args: dst src num [size]");
	return;
    }

    dst = (void*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("bad dst param\n\r");
	return;
    }
    src = (void*)strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("bad src param\n\r");
	return;
    }
    num = strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
	putstr("bad num param\n\r");
	return;
    }
    if (argc > 4) {
	size = strtoul(argv[4], NULL, 0);
	if (strtoul_err) {
	    putstr("bad size param\n\r");
	    return;
	}
    }

    putstr("memcpy\n\r");
    putLabeledWord("src: 0x", (unsigned long)src);
    putLabeledWord("dst: 0x", (unsigned long)dst);
    putLabeledWord("num: 0x", num);
    putLabeledWord("size: 0x", size);
    
    switch (size) {
	case 1:
	{
	    char*   d = (char*)dst;
	    char*   s = (char*)src;

	    while (num--)
		*d++ = *s++;
	}
	break;

	case 2:
	{
	    short*   d = (short*)dst;
	    short*   s = (short*)src;

	    while (num--)
		*d++ = *s++;
	}
	break;

	case 4:
	{
	    int*   d = (int*)dst;
	    int*   s = (int*)src;

	    while (num--)
		*d++ = *s++;
	}
	break;

	default:
	    putLabeledWord("Bad size: 0x", size);
	    break;
    }
    
}

void  hex_dump(
    unsigned char   *data,
    size_t	    num)
{
    int     i;
    long    oldNum;
    char    buf[90];
    char*   bufp;
    int	    line_resid;
    
    while (num)
    {
	bufp = buf;
	binarytohex(bufp, (unsigned long)data, 4);
	bufp += 8;
	*bufp++ = ':';
	*bufp++ = ' ';
	
	oldNum = num;
	
	for (i = 0; i < 16 && num; i++, num--) {
	    binarytohex(bufp, (unsigned long)data[i], 1);
	    bufp += 2;
	    *bufp++ = (i == 7) ? '-' : ' ';
	}

	line_resid = (16 - i) * 3;
	if (line_resid) {
	    memset(bufp, ' ', line_resid);
	    bufp += line_resid;
	}
	
	memcpy(bufp, "| ", 2);
	bufp += 2;
	
	for (i = 0; i < 16 && oldNum; i++, oldNum--)
	    *bufp++ = BL_ISPRINT(data[i]) ? data[i] : '.';

	line_resid = 16 - i;
	if (line_resid) {
	    memset(bufp, ' ', 16-i);
	    bufp += 16 - i;
	}
	
	*bufp++ = '\r';
	*bufp++ = '\n';
	*bufp++ = '\0';
	putstr(buf);
	data += 16;
    }
}

void
command_hex_dump(
    int		argc,
    const char*	argv[])
{
    size_t	    num;
    unsigned char*  p;

    if (argc == 3)		// size specified
	num = strtoul(argv[2], NULL, 0);
    else
	num = 16;

    p = (unsigned char*)strtoul(argv[1], NULL, 0);

    hex_dump(p, num);
}

void
bootldr_goto_sleep(
    void*   pspr)
{
    /*
     * shut down all of the peripherals in an clean and
     * orderly fashsion.
     * (later)
     */

    /* XXX do it!!!! */
#ifdef CONFIG_LCD
    lcd_off(lcd_type);
#endif
#ifdef CONFIG_BITSY
    auxm_fini_serial();
#endif
    CTL_REG_WRITE(SA1100_UART2_UTCR3, 0);
    CTL_REG_WRITE(SA1100_UART3_UTCR3, 0);
    
    
    /*  set pspr */
    if (pspr == NULL) {
	extern void SleepReset_Resume(void);
	extern void ResetEntryPoint(void);
	
	pspr = (void*)(SleepReset_Resume-ResetEntryPoint);
    }
	    
    CTL_REG_WRITE(SA1100_PSPR, (unsigned long)pspr);
    
    /*  clear reset register status bits */
    CTL_REG_WRITE(SA1100_RCSR, 0x0f);
    
#if 0 
    /*  setup GPIO outputs sleep state */
    /*  use current values ??? a good idea ??? */
    mask = CTL_REG_READ(SA1100_GPIO_GPDR);
    v = CTL_REG_READ(SA1100_GPIO_GPLR);
    v &= mask;
    CTL_REG_WRITE(SA1100_PGSR, v);
#endif
    
    /*  set wakeup conditions */
    /*  any gpio edge */
    CTL_REG_WRITE(SA1100_PWER, (1<<0));	/* power button */

    /* setup edge reggies
     * Wakeup on rising edge of power button.  It is inactive high
     * --------+        +-------
     *         +--------+
     *                  ^- wakeup here
     */
	
    CTL_REG_WRITE(SA1100_GPIO_BASE+SA1100_GPIO_GRER_OFF, 0x00000001);
    CTL_REG_WRITE(SA1100_GPIO_BASE+SA1100_GPIO_GFER_OFF, 0x00000000);

    /*  clear all previously detected edge bits */
    CTL_REG_WRITE(SA1100_GPIO_BASE+SA1100_GPIO_GEDR_OFF, 0xffffffff);

#if 0
    /*  set up an RTC int */
      v = regread(SA1100_RCNR)
      v += 10                             # 1hz-->10seconds
      regwrite(SA1100_RTAR, v)
      regread(SA1100_RCNR)
      regwrite(SA1100_RTSR, 1<<2)
#endif

#if 0				/* needed ??? */
    /*  enable some ints so we can wake up */
    CTL_REG_WRITE(SA1100_ICLR, 0);	/* make 'em all irqs */
    
    CTL_REG_WRITE(SA1100_ICMR,
#if 0
		  (1<<31)|	/* RTC match int */
		  (1<<17)|	/* gpio 27-11 (incl ACT button) */
#endif
		  (1<<0));	/* power button */
#endif
    
#if 0
    CTL_REG_WRITE(SA1100_RTC_RTSR, 0);
#endif
    
    CTL_REG_WRITE(SA1100_PCFR, PCFR_OPDE); /* kill the clock */
    
    /*  zzzzz */
    CTL_REG_WRITE(SA1100_PMCR, (1<<0));
}

void
command_reset(
    int		argc,
    const char*	argv[])
{
    bootldr_reset();
}

void
command_halt(
    int		argc,
    const char*	argv[])
{
#ifdef CONFIG_SPOT
    putstr("Power down.\r\n");
    while(1)
        SA1100_GPIO_GPCR_WRITE(SPOT_POWER_LATCH);
#else
    putstr("Feature not available on this hardware.\r\n");
#endif
}

int
reportMismatch(
    void*	    addr1,
    void*	    addr2,
    unsigned long   w1,
    unsigned long   w2)
{
    putLabeledWord("addr1=0x", (unsigned long)addr1);
    putLabeledWord("addr2=0x", (unsigned long)addr2);
    putLabeledWord("w1=0x", w1);
    putLabeledWord("w2=0x", w2);

    /* signal no more comparisons */
    /* XXX add flag for 1 mismatch vs all mismatches */
    return(1);
}

void
command_memcmp(
    int		argc,
    const char* argv[])
{
    void*   dst;
    void*   src;
    int	    num;
    int	    size = 1;
    
    if (argc < 4) {
	putstr("memcpy needs args: dst src num [size]");
	return;
    }

    dst = (void*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("bad addr1 param\n\r");
	return;
    }
    src = (void*)strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("bad addr2 param\n\r");
	return;
    }
    num = strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
	putstr("bad num param\n\r");
	return;
    }
    if (argc > 4) {
	size = strtoul(argv[4], NULL, 0);
	if (strtoul_err) {
	    putstr("bad size param\n\r");
	    return;
	}
    }

    putstr("memcmp\n\r");
    putLabeledWord("a1: 0x", (unsigned long)src);
    putLabeledWord("a2: 0x", (unsigned long)dst);
    putLabeledWord("num: 0x", num);
    putLabeledWord("size: 0x", size);
    
    switch (size) {
	case 1:
	{
	    char*   d = (char*)dst;
	    char*   s = (char*)src;

	    while (num--) {
		if (*d != *s) {
		    if (reportMismatch(d, s, *d, *s))
			break;
		}
		s++;
		d++;
	    }
	}
	break;

	case 2:
	{
	    short*   d = (short*)dst;
	    short*   s = (short*)src;

	    while (num--) {
		if (*d != *s) {
		    if (reportMismatch(d, s, *d, *s))
			break;
		}
		s++;
		d++;
	    }
	}
	break;

	case 4:
	{
	    int*   d = (int*)dst;
	    int*   s = (int*)src;

	    while (num--) {
		if (*d != *s) {
		    if (reportMismatch(d, s, *d, *s))
			break;
		}
		s++;
		d++;
	    }
	}
	break;

	default:
	    putLabeledWord("Bad size: 0x", size);
	    break;
    }
    
}

void
command_version(
    int		argc,
    const char*	argv[])
{
    print_version_verbose(NULL);
}

void
command_mem(
    int		argc,
    const char*	argv[])
{
    putstr("Flash memory info:\n\r");
    btflash_print_types();

    putstr("\n\rSDRAM memory info:\n\r");

    print_mem_size("SDRAM size:", bootinfo.bt_memend);
    putstr("SDRAM bank0:\n\r");
    putLabeledWord("  mdcnfg = 0x",
		   CTL_REG_READ(SA1100_DRAM_CONFIGURATION_BASE+SA1100_MDCNFG));
    putLabeledWord("  mdrefr = 0x",
		   CTL_REG_READ(SA1100_DRAM_CONFIGURATION_BASE+SA1100_MDREFR));
    
}
    
static int rand_seed = 0;

void
srand(seed)
{
    seed += 0xabd826db;
    rand_seed = seed;
}

int
rand(void)
{
    int	ret = rand_seed;

    rand_seed += (rand_seed & 0x00010001) * 1000000;
    rand_seed += (rand_seed & 0x00100010) * 9999999;
    rand_seed += (rand_seed & 0x01100110) * 9999998;
    
    return (ret);
}

void
memtick(
    char*   start,
    char*   p,
    int	    tick_size,
    char*   tick_str)
{
    if ((p - start) % tick_size == 0)
	putstr(tick_str);
}
    
void
command_memtest(
    int		argc,
    const char* argv[])
{
    unsigned long*  start;
    unsigned long*  end;
    unsigned long*  p;
    int		    seed;
    int		    cmp;
    int		    rval;
    int		    tick_size = 64<<10;
    unsigned	    num_mismatches = 0;
    unsigned	    max_mismatches = 16;
    unsigned long   fillval = 0;
    int		    use_fillval = 0;
    
    if (argc < 4) {
	putstr("memtest needs args: addr1 addr2 seed");
	return;
    }

    start = (unsigned long*)strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("bad addr1 param\n\r");
	return;
    }
    end = (unsigned long*)strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("bad addr2 param\n\r");
	return;
    }
    seed = (int)strtoul(argv[3], NULL, 0);
    if (strtoul_err) {
	putstr("bad seed param\n\r");
	return;
    }
    if (argc > 4) {
	fillval = strtoul(argv[4], NULL, 0);
	if (strtoul_err) {
	    putstr("bad fillval param\n\r");
	    return;
	}
	use_fillval = 1;
    }
    

    srand(seed);
    putstr("store vals\n\r");
    for (p = start; p <= end; p++) {
	*p = use_fillval ? fillval : rand();
	memtick((char*)start, (char*)p, tick_size, ".");
    }

    srand(seed);
    putstr("\n\rcmp vals\n\r");
    for (p = start; p <= end; p++) {
	rval = use_fillval ? fillval : rand();
	cmp = *p - rval;
	if (cmp) {
	    putstr("\n\r");
	    putLabeledWord("mismatch, want: 0x", rval);
	    putLabeledWord("           got: 0x", *p);
	    putLabeledWord("          addr: 0x", (unsigned long)p);
	    num_mismatches++;
	    if (num_mismatches > max_mismatches) {
		putstr("max_mismatches exceeded, aborting.\n\r");
		break;
	    }
	}
	memtick((char*)start, (char*)p, tick_size, ".");
    }

    putstr("\n\rcmp done\n\r");
    if (num_mismatches)
	putLabeledWord("*********num_mismatches 0x", num_mismatches);
}


void
command_testJFFS2(
    int		argc,
    const char* argv[])
{
    const char *filename;
    unsigned long * dst;
    
    if (argc > 1) {
	filename = argv[1];
    }
    else{
	putstr("read which file???\n\r");
	return;
    }
    if (argc > 2){
	
        dst = (void*)strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\r\n");
	    return;
	}
    }
    else{
	putstr("copy the file to where???\r\n");
	return;
    }
    
    body_testJFFS2(filename,dst);
}


void
command_infoJFFS2(
    int		argc,
    const char* argv[])
{
    char partname[256];
    
    if (argc > 1) {
	strcpy(partname,argv[1]);	
    }
    else{
	strcpy(partname,"root");
    }
    
    body_infoJFFS2(partname);
}


void
command_timeFlashRead(
    int		argc,
    const char* argv[])
{
    char partname[256];
    
    if (argc > 1) {
	strcpy(partname,argv[1]);	
    }
    else{
	strcpy(partname,"root");
    }
    
    body_timeFlashRead(partname);
}

void
command_p1_ls(
    int		argc,
    const char* argv[])
{
    char partname[256];
    char dirname[256];
    const char *kernel_part_name = NULL;

    get_param_value("kernel_partition", &kernel_part_name);

    if (argc > 1) {
	strcpy(dirname,argv[1]);	
    }
    else{
	strcpy(dirname,"/");
    }
    if (argc > 2) {
	strcpy(partname,argv[2]);	
    }
    else{
	strcpy(partname,kernel_part_name);
    }

    body_p1_ls(dirname,partname);
}




void
command_p1_load_file(
    int		argc,
    const char* argv[])
{
    char partname[256];
    char filename[256];
    unsigned char * dest = (unsigned char *) 0xc0008000;
    
    if (argc > 1) {
	strcpy(partname,argv[1]);	
    }
    else{
	strcpy(partname,"root");
    }
    if (argc > 2) {
	strcpy(filename,argv[2]);	
    }
    else{
	strcpy(filename,"cow");
    }

    if (argc > 3) {
	dest = (unsigned char *)strtoul(argv[3], NULL, 0);
	if (strtoul_err) {
	    putstr("error parsing <dest>\r\n");
	    return;
	}
    }
    body_p1_load_file(partname,filename,dest);
}



void
body_clearMem(unsigned long num,unsigned short *dst)
{
    unsigned short *ldst = dst;    
    int i;
    
    for (i=0; i < num/2; i++)
	*ldst++ = 0x0000;
}

void
command_clearMem(
    int		argc,
    const char* argv[])
{

    unsigned long num = 310410; //=644*482+2
    unsigned short *dst = (unsigned short *)0xc0000000;

    
    if (argc > 1) {
	num = (unsigned long) strtoul(argv[1], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\n\r");
	    return;
	}
    }
    if (argc > 2) {
	dst = (unsigned short *) strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\n\r");
	    return;
	}
    }


    
    body_clearMem(num,dst);
    

}




void
command_cmpKernels(
    int		argc,
    const char* argv[])
{
    unsigned long * dst = 0;
    unsigned long * src = 0;
    unsigned long len = 0;
    const char *partname;

    if (argc > 1)
	partname = argv[1];
    else{
	putstr("which kernel partition?\n\r");
	return;
    }

    if (argc > 2){
	
        dst = (void*)strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad dst param\n\r");
	    return;
	}
    }
    if (argc > 3){
	
        src = (void*)strtoul(argv[3], NULL, 0);
	if (strtoul_err) {
	    putstr("bad src param\n\r");
	    return;
	}
    }
    if (argc > 4){
	
        len = strtoul(argv[4], NULL, 0);
	if (strtoul_err) {
	    putstr("bad len param\n\r");
	    return;
	}
    }
    
    body_cmpKernels(partname,dst,src,len);
}

// This aviods the gcc assembler cache settings and seems to work more reliably.
//we should pull C-gccasm enabler if this proves solid

void
command_enable_caches(
    int		argc,
    const char* argv[])
{
    int   icache_en = 0;
    int   dcache_en = 0;
    unsigned long ret;
    
    if (argc > 1) {
	dcache_en = (int) strtoul(argv[1], NULL, 0);
	if (strtoul_err) {
	    putstr("bad  dcache param\n\r");
	    return;
	}
    }

    if (argc > 2) {
	icache_en = (int) strtoul(argv[2], NULL, 0);
	if (strtoul_err) {
	    putstr("bad icache param\n\r");
	    return;
	}
    }

    ret = asmEnableCaches((unsigned long) dcache_en,(unsigned long) icache_en);
    putLabeledWord("  asmEnableCaches = ",ret);

}


void parseParamName(int argc,const char **argv,char *pName,char *pValue)
{
    int i;
    int len = 0;
    char *pAll;
    char *p;
    unsigned char canKill = 0;
    unsigned char valueSearch = 0;
    

    for (i=1; i < argc; i++){
#if 0
	putLabeledWord("i = ",i);
	putstr("argv = <");
	putstr(argv[i]);
	putstr(">\r\n");
#endif
	len+= strlen(argv[i]);
    }


    if ((pAll = (char *) mmalloc((len+1) * sizeof(char))) == NULL){
	putstr("out of room to build params list!\r\n");
	return;
    }
    pAll[0] = '\0';
    p = pAll;    
    for (i=1; i < argc; i++){
	strcpy(p,argv[i]);
	p += strlen(argv[i]);
	strcpy(p," ");
	p += 1;
    }
    // kill the last unneeded space
    *--p ='\0';
    

#if 0
    putstr("pAll = <");
    putstr(pAll);
    putstr(">\r\n");
#endif

    // eliminate = if it's there
    // we only kill the first one and only if its 
    p = pAll;
    while((p - pAll) < (strlen(argv[1]) + strlen(argv[2])+1)){

	if (canKill && (isblank(*p) || (*p == '=')))
	    valueSearch = 1;
	if (!canKill && isalnum(*p))
	    canKill = 1;
	else if (valueSearch && isalnum(*p))
	    break;
	if (canKill && (*p == '='))
	    *p = ' ';
	p++;
    }
    
#if 0
    putstr("pAll_post = <");
    putstr(pAll);
    putstr(">\r\n");
#endif

    p = pAll;
    while ((*p != ' ') && (*p != '\t'))
	p++;
    *p++ = '\0';
    
    while ((*p == ' ') || (*p == '\t'))
        p++;
    strcpy(pName,pAll);
    strcpy(pValue,p);
    
    free(pAll);
    return;
    
}

// this builds up the command buffer and either processes or prints it.
void parseParamFile(unsigned char *src,unsigned long size,int just_show)
{
    unsigned char *p = src;
    char cmdbuf[1024];
    char* cmdp;

    while ((p-src) < size){
	cmdp = cmdbuf;
	while (((p-src) < size) && (*p != '\r') && (*p != '\n'))
	    *cmdp++ = *p++;
	*cmdp = '\0';
	putstr("+ ");
	putstr(cmdbuf);
	putstr("\r\n");
	while ((*p == '\r') || (*p == '\n'))
	    p++;
	if (!just_show)
	    exec_string(cmdbuf);
    }
}

void command_pef(int argc, const char* argv[])
{
    int just_show = 0;

    if (argc > 1) 
	just_show = strtoul(argv[1],NULL,0);	

    params_eval_file(just_show);
}

void command_lli(int argc, const char* argv[])
{
    putLabeledWord("ARCH INFO(CPR0)=", readCPR0());
    putLabeledWord("MMU Control (CPR1)=", readCPR1());
    putLabeledWord("TRANSLATION TABLE BASE (CPR2)=", readCPR2());
    putLabeledWord("DOMAIN ACCESS CTL (CPR3)=", readCPR3());
    putLabeledWord("FAULT STATUS (CPR5)=", readCPR5());
    putLabeledWord("FAULT ADDRESS (CPR6)=", readCPR6());
    putLabeledWord("MMU PROC ID  (CPR13)=", readCPR13());
    putLabeledWord("BREAKPOINT  (CPR14)=", readCPR14());
    putLabeledWord("Program CounterC  (PC)=", readPC());
    
}
	
void command_cat(int argc, const char* argv[])
{
    char partname[256];
    char filename[256];
    long kernel_in_ram = 0;
    long size = 0;
    long i;
    
    kernel_in_ram = param_kernel_in_ram.value;

    if (argc > 1) {
	strcpy(filename,argv[1]);	
    }
    else{
	putstr("cat what file?\r\n");
    }

    
    if (argc > 2) {
	strcpy(partname,argv[2]);	
    }
    else{
	strcpy(partname,"root");
    }

    size = body_p1_load_file(partname,filename,(void*)kernel_in_ram);
    for (i=0; i < size; i++){
	if (*((char *)(kernel_in_ram + i)) == '\n')
	    putc('\r');
	putc(*((char *)(kernel_in_ram + i)));
    }
}


unsigned char isWincePartition(unsigned char *src)
{
    unsigned long *p = (unsigned long *)src;
    unsigned char ret = 1;
    int i;
    unsigned long tmp;
    
       
    //putLabeledWord("*p = ",*p);
    //putLabeledWord("WPM1 = ",WINCE_PARTITION_MAGIC_1);
    if (*p++ != WINCE_PARTITION_MAGIC_1)
	ret = 0;
    for (i=0; i < WINCE_PARTITION_LONG_0;i++){
	tmp = *p++;
	if ((tmp != 0x00000000) &&
	    (tmp != 0xFFFFFFFF))
	    ret = 0;
    }
    //putLabeledWord("*p = ",*p);
    //putLabeledWord("WPM2 = ",WINCE_PARTITION_MAGIC_2);
    if (*p++ != WINCE_PARTITION_MAGIC_2)
	ret = 0;
    
    
    return ret;
}
unsigned char amRunningFromRam(void)
{
    unsigned long pc = readPC();
    pc &= 0xF0000000;
    pc &= DRAM_BASE0;
    return (unsigned char) (pc>0);
    
    


}


static BOOL isValidBootloader(unsigned long p,unsigned long size)
{
	BOOL ret = TRUE;
	if ((!isValidOHHImage(p,size)) && (!isValidParrotImage(p,size)))
		ret=FALSE;

	return ret;
}
static BOOL isValidOHHImage(unsigned long p,unsigned long size)
{
	BOOL ret = TRUE;
	unsigned int bsd_sum;
	unsigned long start_addr;
	unsigned long mach_type = 0;
	unsigned long boot_caps;
	
	if (size < 0x2C) //exit early
		return FALSE;
	
	// right bootloader??
	if (*((unsigned long *)(p+0x20)) != BOOTLDR_MAGIC)
		ret = FALSE;
	
	// right arch??
	if (*((unsigned long *)(p+0x2C)) != ARCHITECTURE_MAGIC)
		ret = FALSE;

	// linked at 0x0??
	start_addr = (unsigned long)(*(unsigned long *)(p+0x28));
	if (start_addr != 0x0)
		ret = FALSE;

	// mach_type matches caps??
	boot_caps = (unsigned long)(*(unsigned long *)(p+0x30));
	mach_type = param_mach_type.value;
	if ((mach_type == MACH_TYPE_H3800) && !(boot_caps & BOOTCAP_3800_SUPPORT))
		ret = FALSE;


	//BSD Sum == 0??
	bsd_sum = bsd_sum_memory( p, size);
	if (bsd_sum != 0)
		ret = FALSE;

	return ret;
}
static BOOL isValidParrotImage(unsigned long p,unsigned long size)
{
	BOOL ret = TRUE;
	unsigned char* p2;

	
	if (size < 0x1000)
		return FALSE;
	if (size > 0x40000)
		return FALSE;
	
	// we'll just check a couple of magic numbers
	if (*((unsigned long *)(p+0x0)) != PARROT_MAGIC_0){
	    putLabeledWord("parrot word 0 ->0x", *((unsigned long *)(p+0x0)));
	    putLabeledWord("Expected      ->0x", PARROT_MAGIC_0);
	    ret = FALSE;
	}
	
	
	if ((*((unsigned long *)(p+0xFFC)) != PARROT_MAGIC_FFC) &&
	    (*((unsigned long *)(p+0xFFC)) != PARROT_MAGIC_FFC_ALT)){
	    putLabeledWord("parrot word 0xFFC ->0x", *((unsigned long *)(p+0xFFC)));
	    putLabeledWord("Expected      ->0x", PARROT_MAGIC_FFC);
	    putLabeledWord("or            ->0x", PARROT_MAGIC_FFC_ALT);
	    ret = FALSE;
	}
	
	if (*((unsigned long *)(p+0x1000)) != PARROT_MAGIC_1000){
    	    putLabeledWord("parrot word 0x1000 ->0x", *((unsigned long *)(p+0x1000)));
	    putLabeledWord("Expected           ->0x", PARROT_MAGIC_1000);
	    ret = FALSE;
	}

	// ok installling parrot w/out having previously installed wince is bad
	// so we'll add the wince partition check to the test
	// XXX this presumes that the params sector follows the bootloader sector!!
	p2 = ((char*)flashword) + flashDescriptor->bootldr.base + flashDescriptor->bootldr.size;

	if (!isWincePartition(p2)){
	    putstr("You must load wince BEFORE loading Parrot\r\n");
	    return FALSE;
	}
	
	return ret;
}
void discover_machine_type()
{
    volatile unsigned short *p = (unsigned short *) BITSY_EGPIO;
    volatile unsigned short *p2 = (unsigned short *) 0x0;
    volatile unsigned short tst;
    
    unsigned long mach_type = MACH_TYPE;
 
    // first check to see if we are a 3800
    // on the 3800 this is the gpio dir register so it is safe to set it
    // we will be able to read it back if we are a 3800 and not else
    // we are setting the serial port on bit so it's a safe op
    // for the 3(1,6,7)xx models too
    set_egpio(EGPIO_BITSY_RS232_ON);
    tst = *p2;
    tst = *p;
    mach_type = param_mach_type.value;

    if ((tst == EGPIO_BITSY_RS232_ON)){
	mach_type = MACH_TYPE_H3800;
        param_mach_type.value = MACH_TYPE_H3800;
    }
    else{
	if (is_flash_16bit())
            param_mach_type.value = MACH_TYPE_H3100;
	else
	    param_mach_type.value = MACH_TYPE_H3600;

    }

}

void initialize_by_mach_type()
{
    unsigned long mach_type = MACH_TYPE;

    mach_type = param_mach_type.value;
    
    putLabeledWord("mach_type  ->", mach_type);

#if 0
    putLabeledWord("asic1 Gpio Mask addr show up as ->", H3800_ASIC1_GPIO_MASK_ADDR);
    putLabeledWord("asic1 Gpio Dir addr show up as ->", H3800_ASIC1_GPIO_DIR_ADDR);
    putLabeledWord("asic1 Gpio Out addr show up as ->", H3800_ASIC1_GPIO_OUT_ADDR);	
#endif

    hal_init(mach_type);
    lcd_init(mach_type);
    btflash_reset_partitions();
}


/**************************************************************************************
 *
 * This is the section for our zlib experiments
 *
 * ***********************************************************************************/
void command_tdz(int argc, const char* argv[])
{

    unsigned long size;
    unsigned long address;
    unsigned long dAddress;
    int isGoodImg;
    
    size = last_ram_load_size;
	    
    
    if (argc < 3) {
	putstr("not enough args, need <srcAddress>  <destAddress>\n\r");
	return;
    }
    
    address = strtoul(argv[1], NULL, 0);
    if (strtoul_err) {
	putstr("error parsing address\r\n");
	return;
    }
    dAddress = strtoul(argv[2], NULL, 0);
    if (strtoul_err) {
	putstr("error parsing dAddress\r\n");
	return;
    }



    isGoodImg = verifyGZipImage(address,&size);
    putLabeledWord("verify img = ",isGoodImg);
    putLabeledWord("uncompr size = ",size);

    size = last_ram_load_size;
    body_tdz(address,size,dAddress);
}


void body_tdz(unsigned long address,unsigned long size,unsigned long dAddress)
{    
    char tiny_buf[ZLIB_CHUNK_SIZE];
    struct bz_stream z;
    Byte *p;
    unsigned long uncomplen;
    
    
    gzInitStream(address,size,&z);
    
    
    
    p = (Byte *) dAddress;
    
    while ((size = gzReadChunk(&z,tiny_buf))) {
	memcpy(p,tiny_buf,size);
	p += size;	
	if (z.err != Z_OK) break;
	
    }
    putLabeledWord("finished loading with runningCRC = 0x",z.crc32);
    

    uncomplen = z.stream.total_out;
    
    putLabeledWord("uncompressed length = 0x", uncomplen);
    putLabeledWord("total_in = 0x", z.stream.total_in);
    putLabeledWord("read_crc returns = 0x", z.read_crc32);
    
    size = crc32(0,(const void *) dAddress,uncomplen);

    putLabeledWord("crc32 static calc= 0x", size);
}

static int check_for_nested_help(int argc, const char **argv,struct bootblk_command *cmdlist)
{
    // if we have help subcommand pass it to exec to handle else continue with the cmd
    if ((strcmp(argv[0], "help") == 0) || 
	(strcmp(argv[0], "?") == 0)){
	execcmd(cmdlist,argc,argv);
	return 1;	
    }
    return 0;
}
