#include "bootldr.h"
#include "serial.h"
#include "buttons.h"
#include "sa1100.h"

int packetize_output = 0;
int getc_verbose_errors = 1;
int getc_errno = 0;

extern int get_pushed_char(void);
void serial_putnstr(struct serial_device *device,
    const char	*str,
			 size_t	n);

#ifdef CONFIG_SKIFF

void *skiff_serbase(void) {   /* Shouldn't use this */
  	return 0;
}

int skiff_char_ready() {
  return (!(CSR_READ_BYTE(UARTFLG_REG) & UART_RX_FIFO_EMPTY));
}

unsigned char skiff_read_char() {
  return CSR_READ_BYTE(UARTDR_REG);
}

unsigned int skiff_read_status() {
  return CSR_READ_BYTE(RXSTAT_REG);
}

void skiff_write_char(struct serial_device *device, unsigned char c) {
  if (!device->enabled) return;
  while (CSR_READ_BYTE(UARTFLG_REG) & UART_TX_FIFO_BUSY)
	; /* do nothing */
  CSR_WRITE_BYTE(UARTDR_REG,c);
}

static struct serial_device serial = { skiff_char_ready, skiff_read_char, skiff_read_status, skiff_write_char, 0, 0, skiff_serbase, 1 };

#elif defined(CONFIG_BITSY) || defined(CONFIG_SPOT) || defined(CONFIG_JORNADA720)  || defined(CONFIG_JORNADA56X) || defined(CONFIG_GATOR)
extern void *Ser3Base;

int sa1100_char_ready() {
  return ((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_RNE);
}

unsigned char sa1100_read_char() {
  return (*(volatile byte *)SA1100_UART3_UTDR);
}

unsigned int sa1100_read_status() {
  return ((*(volatile byte *)SA1100_UART3_UTSR1) & SA1100_UTSR1_ERROR_MASK);
}

void *sa1100_serbase(void) {
  	return Ser3Base;
}

#if defined(CONFIG_BITSY)

void
bitsy_putc(struct serial_device *device,
		   char    c,
		   int	   encap_done)
{
  /*
   * silly person calling putc instead of putstr...
   * we encap the single char
   */
  
  if (!encap_done && packetize_output) {
	char	buf[2];
	buf[0] = c;
	buf[1] = '\0';
	serial_putnstr(device, buf, 1);
  }
  else PrintChar(c, Ser3Base);
}

void bitsy_write_char(struct serial_device *device, unsigned char c) {
  if (!device->enabled) return;
  bitsy_putc(device, c, 0);
}

static struct serial_device serial = { sa1100_char_ready, sa1100_read_char, sa1100_read_status, bitsy_write_char, 0, 0, sa1100_serbase, 1 };
#else
void sa1100_write_char(struct serial_device *device, unsigned char c) {
  if (!device->enabled) return;
  while (!((*(volatile long *)SA1100_UART3_UTSR1) & SA1100_UTSR1_TNF)); /* wait until TX FIFO not full */
  *(byte *) SA1100_UART3_UTDR = c;
}
static struct serial_device serial = { sa1100_char_ready, sa1100_read_char, sa1100_read_status, sa1100_write_char, 0, 0, sa1100_serbase, 1 };
#endif


#elif defined(CONFIG_ASSABET)

int assabet_char_ready() {
  return ((machine_has_neponset()  
		   ? (*(volatile long *)SA1100_UART3_UTSR1) 
		   : (*(volatile long *)SA1100_UART1_UTSR1))
		  & SA1100_UTSR1_RNE);
}

unsigned char assabet_read_char() {
  return (machine_has_neponset()
		  ? (*(volatile byte *)SA1100_UART3_UTDR) 
		  : (*(volatile byte *)SA1100_UART1_UTDR));
}

unsigned int assabet_read_status() {
  return ((machine_has_neponset()
		   ? (*(volatile long *)SA1100_UART3_UTSR1)
		   : (*(volatile long *)SA1100_UART1_UTSR1))
		  & SA1100_UTSR1_ERROR_MASK);
}

void assabet_write_char(struct serial_device *device, unsigned char c) {
  if (!device->enabled) return;
  while (!((machine_has_neponset() ? (*(volatile long *)SA1100_UART3_UTSR1) :
			(*(volatile long *)SA1100_UART1_UTSR1)) & SA1100_UTSR1_TNF));
  (machine_has_neponset() ? (*(byte *)SA1100_UART3_UTDR) :
   (*(byte *)SA1100_UART1_UTDR)) = c;
}

void *assabet_serbase() {
  return machine_has_neponset() ? Ser3Base : Ser1Base;
}
static struct serial_device serial = { assabet_char_ready, assabet_read_char, assabet_read_status, assabet_write_char, 0, 0, assabet_serbase, 1 };

#else
#error no architecture defined for SERIAL_CHAR_READY(), et al
#endif




void serial_putc(struct serial_device *device, char c) 
{
  if (!device->enabled) return;
  device->write_char(device, c);
}

byte serial_do_getc(struct serial_device *device,
    vfuncp	    idler,
    unsigned long   timeout,
    int*	    statp)
{
   byte c, rxstat;
   int	do_timeout = timeout != 0;
   int	ch;

   getc_errno = 0; /* reset errno */

   while (!device->char_ready()) {
       if ((ch = get_pushed_char()) != -1)
	   return (ch);
   
       if (do_timeout) {
	   if (!timeout)
	       break;
	   timeout--;
       }
       
       if (idler)
	   idler();
   }

   /* give priority to pushed chars */
   if ((ch = get_pushed_char()) != -1)
       return (ch);
   
   if (do_timeout && timeout == 0) {
       c = 0;
       rxstat = -1;
   }
   else {
       c = device->read_char();
       rxstat = device->read_status();
   }
   
   if (rxstat) {
      getc_errno = rxstat;
      if (getc_verbose_errors && !statp) {
         putLabeledWord("RXSTAT error: ", rxstat);
      }
      if (statp)
	  *statp = rxstat;
   }
   return(c);
}

byte
serial_getc(struct serial_device *device)
{
#ifdef CONFIG_SPOT
    return (serial_do_getc(device, spot_idle, 0, NULL));
#else
    return (serial_do_getc(device, button_check, 0, NULL));
#endif
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay iterations checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte
serial_awaitkey(struct serial_device *device, 
    unsigned long   delay,
    int*	    error_p)
{
    return (serial_do_getc(device, button_check, delay, error_p));
}

byte serial_do_getc_seconds(struct serial_device *device,
                     vfuncp	    idler,
                     unsigned long   timeout_seconds,
                     int*	    statp)
{
  byte c, rxstat;
  int	do_timeout = timeout_seconds != 0;
  int  timed_out = 0;
  unsigned long start_time = CTL_REG_READ(SA1100_RCNR);
  int	ch;

  getc_errno = 0; /* reset errno */

  while (!device->char_ready()) {
    if ((ch = get_pushed_char()) != -1)
      return (ch);
   
    if (do_timeout) {
      unsigned long time = CTL_REG_READ(SA1100_RCNR);
      if ((time - start_time) > timeout_seconds) {
        timed_out = 1;
        break;
      }
    }
       
    if (idler)
      idler();
  }

  /* give priority to pushed chars */
  if ((ch = get_pushed_char()) != -1)
    return (ch);
   
  if (do_timeout && timed_out) {
    c = 0;
    rxstat = -1;
  }
  else {
    c = device->read_char();
    rxstat = device->read_status();
  }
   
  if (rxstat) {
    getc_errno = rxstat;
    if (getc_verbose_errors && !statp) {
      putLabeledWord("RXSTAT error: ", rxstat);
    }
    if (statp)
      *statp = rxstat;
  }
  return(c);
}

/*
 * Reads and returns a character from the serial port
 *  - Times out after delay seconds checking for presence of character
 *  - Sets *error_p to UART error bits or -1 on timeout
 *  - On timeout, sets *error_p to -1 and returns 0
 */
byte
serial_awaitkey_seconds(struct serial_device *device,
    unsigned long   delay_seconds,
    int*	    error_p)
{
    return (serial_do_getc_seconds(device, button_check, delay_seconds, error_p));
}

void serial_do_putnstr(struct serial_device *device,
    const char* str,
    size_t	n)
{
   while (n && *str != '\0') {
#if defined(CONFIG_BITSY)
         bitsy_putc(device, *str, 1);
#elif defined(CONFIG_SA1100)
         PrintChar(*str, device->serbase()) ;
#else
         serial_putc(device, *str) ;
#endif
      str++ ;
      n--;
   }
}

void
serial_putnstr(struct serial_device *device,
    const char	*str,
    size_t	n)
{
   if (str == NULL)
      return;
   
#if defined(CONFIG_BITSY)
   if (packetize_output) {
       int  len;
       char len_str[16];

       len = strlen(str);
       if (n < len)
	   len = n;

       /* add the msg header */
       dwordtodecimal(len_str, len);
       serial_do_putnstr(device, "MSG/", 4);
       len = strlen(len_str);
       len_str[len] = '/';
       ++len;
       len_str[len] = '\0';
       serial_do_putnstr(device, len_str, len);
   }
#endif   

   serial_do_putnstr(device, str, n);
}

void serial_putstr(struct serial_device *device, const char *str)
{
    serial_putnstr(device, str, strlen(str));
}

void
serial_putstr_sync(struct serial_device *device, 
    const char*   s)
{
    serial_putstr(device, s);
    let_uart_drain(SA1100_UART3_UTSR1);
}

unsigned char getc() { return serial_getc(&serial); }
void putc(char c) { serial_putc(&serial, c); }
void putstr(const char *s) { serial_putstr(&serial, s); }
void putnstr(const char *s, int n) { serial_putnstr(&serial, s, n); }
byte awaitkey_seconds(unsigned long delay_seconds, int* error_p) {
  return serial_awaitkey_seconds(&serial, delay_seconds, error_p);
}
void putstr_sync(const char* s) { serial_putstr_sync(&serial, s); }
void do_putnstr(const char* s, int n) { serial_do_putnstr(&serial, s, n); }

