/****************************************************************************/
/* Copyright 2002 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * Microsoft FAT filesystem support
 */

#include "bootldr.h"
#include "heap.h"
#include "fs.h"
#include "ide.h"
#include "vfat.h"
#include "commands.h"
#include <asm/errno.h>

static __inline__ int bpb_n_fats(struct bpb_info *info)
{
    return info->n_fats;
}
static __inline__ int bpb_bytes_per_sector(struct bpb_info *info)
{
    return (info->bytes_per_sector[1] << 8) + info->bytes_per_sector[0]; 
}
static __inline__ int bpb_sectors_per_cluster(struct bpb_info *info)
{
    return info->sectors_per_cluster;
}
static __inline__ int bpb_n_reserved_sectors(struct bpb_info *info)
{
    return info->n_reserved_sectors;
}

static __inline__ int bpb_n_root_entries(struct bpb_info *info)
{
    return (info->n_root_entries[1] << 8) + info->n_root_entries[0]; 
}

static __inline__ int bpb_root_dir_sectors(struct bpb_info *info)
{
    return (bpb_n_root_entries(info) * sizeof(struct fat_dir_entry) + (bpb_bytes_per_sector(info) - 1)) / bpb_bytes_per_sector(info);
}

static __inline__ int bpb_fat_size(struct bpb_info *info)
{
    int fat_size = 0;
    if (info->fat_size16 != 0) 
        fat_size = info->fat_size16;
    else 
        fat_size = info->fat_size32;
    return fat_size;
}
static __inline__ int bpb_first_root_dir_sector(struct bpb_info *info)
{
    return (bpb_n_reserved_sectors(info) 
            + (bpb_n_fats(info) * bpb_fat_size(info)));
}


static __inline__ int bpb_total_sectors(struct bpb_info *info)
{
    int total_sectors = 0;
    if (info->total_sectors16[0] || info->total_sectors16[1]) 
        total_sectors = (info->total_sectors16[1] << 8) | info->total_sectors16[0];
    else 
        total_sectors = info->total_sectors32;
    return total_sectors;
}

static __inline__ int bpb_n_data_sectors(struct bpb_info *info)
{
    return 
        bpb_total_sectors(info) - 
        (bpb_n_reserved_sectors(info) 
         + (bpb_n_fats(info) * bpb_fat_size(info))
         + bpb_root_dir_sectors(info));
}

static __inline__ int bpb_first_data_sector(struct bpb_info *info)
{
    return (bpb_n_reserved_sectors(info) 
            + (bpb_n_fats(info) * bpb_fat_size(info))
            + bpb_root_dir_sectors(info));
}

static __inline__ int bpb_n_clusters(struct bpb_info *info)
{
    return bpb_n_data_sectors(info) / bpb_sectors_per_cluster(info);
}

static __inline__ enum fat_type bpb_fat_type(struct bpb_info *info)
{
    u32 n_clusters = bpb_n_clusters(info);
    if (n_clusters < 4085)
        return ft_fat12;
    else if (n_clusters < 65535) 
        return ft_fat16;
    else 
        return ft_fat32;
}

static int vfat_read_bpb_info(struct vfat_filesystem *vfat, struct iohandle *ioh)
{
    /* read bpb_info from first sector of partition */
    int rc = ioh->ops->read(ioh, vfat->buf, 0, 512); 
    if (!rc)
        memcpy(&vfat->info, vfat->buf, sizeof(struct bpb_info));
    return rc;
}

static __inline__ int fat_entry_first_clusterno(struct fat_dir_entry *entry)
{
    return (entry->first_cluster_high << 16) | entry->first_cluster_low;
}

void fixup_shortname(unsigned char *dst, unsigned char *dosname)
{ 
    int i;
    int j = 0;
    int has_extension = 0;
    int dot_pos = 0;
    for (i = 0; i < 8; i++) {
        unsigned char c = dosname[i];
        if (c == ' ')
            break;
        if (c >= 'A' && c <= 'Z')
            c = c + 'a' - 'A';
        dst[j++] = c;
    } 
    dot_pos = j++;
    dst[dot_pos] = '.';
    for (i = 8; i < 11; i++) {
        unsigned char c = dosname[i];
        if (c == ' ')
            break;
        if (c >= 'A' && c <= 'Z')
            c = c + 'a' - 'A';
        has_extension = 1;
        dst[j++] = c;
    } 
    if (!has_extension)
        dst[dot_pos] = 0;
    dst[j++] = 0;
}

static __inline__ u32 vfat_next_clusterno(struct vfat_filesystem *vfat, u32 clusterno)
{
    enum fat_type ftype = vfat->fat_type;
    u32 entry = 0;
    switch (ftype) {
    case ft_fat16: {
        u16 *fat = (u16 *)vfat->fat; 
        entry = fat[clusterno];
        if (entry >= 0xFFF8)
            entry = VFAT_EOC; 
    } break; 
    case ft_fat32: {
        u32 *fat = (u32 *)vfat->fat; 
        entry = fat[clusterno];
        entry &= 0x00FFFFFF;
        if (entry >= 0x00FFFFF8)
            entry = VFAT_EOC;
    } break;
    case ft_fat12: {
       u8 *fat = (u8 *)vfat->fat;
       memcpy(&entry, fat + (clusterno + clusterno/2), sizeof(u16));
       if (clusterno & 1) /* if odd entry, shift right 4 bits */
          entry >>= 4;
       else
          entry &= 0xFFF; /* else use lower 12 bits */
    } break;
    default:
        return 0;
    }  
    // putLabeledWord("vfat_next_clusterno clusterno=", clusterno);
    // putLabeledWord("               next_clusterno=", entry);
    return entry;
}

u32 vfat_clusterno(struct vfat_filesystem *vfat, u32 clusterno, size_t byte_offset)
{
    size_t bytes_per_cluster = vfat->info.sectors_per_cluster * bpb_sectors_per_cluster(&vfat->info);
    while (byte_offset > bytes_per_cluster && clusterno != VFAT_EOC) {
        u32 next_clusterno = vfat_next_clusterno(vfat, clusterno);
        byte_offset -= bytes_per_cluster;
        clusterno = next_clusterno;
    }
    return clusterno;
}


/*
 * Note: first data clusterno is 2!
 */
int vfat_read_clusters(struct vfat_filesystem *vfat, char *buf, u32 clusterno, size_t nbytes)
{
    struct bpb_info *info = &vfat->info;
    u32 first_data_sector = bpb_first_data_sector(info);
    u32 sectors_per_cluster = bpb_sectors_per_cluster(info);
    u32 bytes_per_sector = bpb_bytes_per_sector(info);
    u32 bytes_per_cluster = bytes_per_sector * sectors_per_cluster;
    u32 sector_size = vfat->sector_size;
    size_t bytes_read = 0;
    if (nbytes == 0)
        nbytes = sector_size;
    while (clusterno != VFAT_EOC && bytes_read < nbytes) {
        u32 sector = first_data_sector + sectors_per_cluster * (clusterno - 2);
        size_t bytes_read_this_cluster = 0;
        // putLabeledWord("read_clusters: clusterno=", clusterno);
        while (bytes_read < nbytes && bytes_read_this_cluster < bytes_per_cluster) {
            int rc;
            // putLabeledWord("  sectorno=", sector);
            rc = vfat->iohandle->ops->read(vfat->iohandle, buf+bytes_read, sector*sector_size, bytes_per_sector);
            if (rc < 0) return rc;
            bytes_read += bytes_per_sector;
            bytes_read_this_cluster += bytes_per_sector;
            sector++;
        }
        clusterno = vfat_next_clusterno(vfat, clusterno);
    }
    return bytes_read;
}

void get_basename(char *bname, const char *fname)
{
    char *sep_pos = strrchr(fname, '/');
    if (sep_pos)
        strcpy(bname, sep_pos+1);
    else
        strcpy(bname, fname);
}

void get_dirname(char *dirname, const char *fname)
{
    char *sep_pos = strrchr(fname, '/');
    if (sep_pos) {
        while (fname < sep_pos) {
            *dirname++ = *fname++;
        }
    }
    *dirname = 0;
}

int vfat_find_dir_entry(struct vfat_filesystem *vfat, struct fat_dir_entry *outdir, const char *fname)
{
    char basename[128];
    char dirname[128];
    struct fat_dir_entry dir_storage;
    struct fat_dir_entry *dir = &dir_storage;
    struct fat_dir_entry entries[128];
    int rc = 0;
    get_basename(basename, fname);
    get_dirname(dirname, fname);
    putstr("vfat_find_dir_entry: fname='"); putstr(fname); putstr("'\r\n"); 
    putstr("                   dirname='"); putstr(dirname); putstr("'\r\n"); 
    putstr("                  basename='"); putstr(basename); putstr("'\r\n"); 
    if (dirname[0]) {
        if ((rc = vfat_find_dir_entry(vfat, &dir_storage, dirname)) == 0) {
            size_t n_bytes = dir->n_bytes;
            if (n_bytes == 0) n_bytes = 0x200;
            // entries = mmalloc(n_bytes); 
            vfat_read_clusters(vfat, (char *)entries, fat_entry_first_clusterno(dir), dir->n_bytes);
            dir = entries;
        } else {
            return rc; 
        }
    } else {
        putstr("  searching root_dir_entries\r\n");
        dir = vfat->root_dir_entries;
    }
    if (!basename[0])
        strcpy(basename, ".");

    while (dir->attr) {
        if ((dir->attr & vfat_attr_long_name) != vfat_attr_long_name) {
            char name[128];
            fixup_shortname(name, dir->name);
            if (strcmp(name, basename) == 0) {
               putstr("  vfat_find_dir_entry:\r\n");
               hex_dump((char *)dir, sizeof(struct fat_dir_entry));
                memcpy(outdir, dir, sizeof(struct fat_dir_entry));
                // if (entries)
                //    mfree(entries);
                if (!(outdir->attr & vfat_attr_directory))
                    return -ENOTDIR;
                return 0;
            }
        }
        dir++;
    }
    return -ENOENT;
}

int vfat_find_file_entry(struct vfat_filesystem *vfat, struct fat_dir_entry *outentry, const char *fname)
{
    char basename[128];
    char dirname[128];
    struct fat_dir_entry dir_storage;
    struct fat_dir_entry *dir = &dir_storage;
    struct fat_dir_entry entries[128];
    int rc = 0;
    get_basename(basename, fname);
    get_dirname(dirname, fname);
    putstr("vfat_find_file_entry: fname='"); putstr(fname); putstr("'\r\n"); 
    putstr("                    dirname='"); putstr(dirname); putstr("'\r\n"); 
    putstr("                   basename='"); putstr(basename); putstr("'\r\n"); 
    if (dirname[0]) {
        if ((rc = vfat_find_dir_entry(vfat, dir, dirname)) == 0) {
            size_t n_bytes = dir->n_bytes;
            putstr("vfat_find_file_entry: got dir entry for "); putstr(dirname); putstr("\r\n");
            putLabeledWord("  attr=", dir->attr);
            putLabeledWord("  clusterno=", fat_entry_first_clusterno(dir));
            putLabeledWord("  n_bytes=", n_bytes);
            if (n_bytes == 0) n_bytes = 0x200;
            // putstr(" calling malloc\r\n");
            // entries = mmalloc(n_bytes); 
            // putLabeledWord("   entries=", entries);
            vfat_read_clusters(vfat, (char *)entries, fat_entry_first_clusterno(dir), n_bytes);
            dir = entries;
        } else {
            return rc; 
        }
    } else {
        putstr("  looking for file entry in root_dir_entries:\r\n");
        dir = vfat->root_dir_entries;
    }
    while (dir->attr) {
        if ((dir->attr & vfat_attr_long_name) != vfat_attr_long_name) {
            char name[128];
            fixup_shortname(name, dir->name);
            if (strcmp(name, basename) == 0) {
               putstr("  vfat_find_file_entry:\r\n");
               hex_dump((char *)dir, sizeof(struct fat_dir_entry));
                memcpy(outentry, dir, sizeof(struct fat_dir_entry));
                // if (entries)
                //    mfree(entries);
                if (outentry->attr & vfat_attr_directory)
                    return -EISDIR;
                return 0;
            }
        }
        dir++;
    }
    return -ENOENT;
}

void vfat_list_one_entry(struct fat_dir_entry *dir)
{
    if ((dir->attr & vfat_attr_long_name) != vfat_attr_long_name) {
        /* is short name */
        char name[13];
        fixup_shortname(name, dir->name);
        putstr(" "); putstr(name); putstr("\r\n");
        putLabeledWord("   attr=", dir->attr);
        putLabeledWord("   first_cluster=", (dir->first_cluster_high << 16) | dir->first_cluster_low);
        putLabeledWord("   n_bytes=", dir->n_bytes);
    } else {
        /* is long name */
        char name[14];
        struct fat_ldir_entry *ldir = (struct fat_ldir_entry *)dir;
        int i;
        memset(name, 0, sizeof(name));
        for (i = 0; i < 5; i++)
            name[i+ 0] = ldir->name1[i*2];
        for (i = 0; i < 6; i++)
            name[i+ 5] = ldir->name2[i*2];
        for (i = 0; i < 2; i++)
            name[i+11] = ldir->name3[i*2];
        putstr(" ldir: "); putstr(name); putstr("\r\n");
        putLabeledWord("  ord=", ldir->ord);
    }
    dir++;
}

void vfat_list_dir_entries(struct fat_dir_entry *dir)
{
    putLabeledWord("  vfat_list_dir_entries: attr=", dir->attr);
    putstr("  skipping long entries\r\n");
    while (dir->attr) {
        if ((dir->attr & vfat_attr_long_name) != vfat_attr_long_name)
            vfat_list_one_entry(dir);
        dir++;
    }
}

/*
 * lists the file or directory named by fname
 */
int vfat_list(struct vfat_filesystem *vfat, const char *fname)
{
    struct fat_dir_entry entry;
    size_t n_bytes = 0x200;
    int rc = vfat_find_dir_entry(vfat, &entry, fname);
    if (!rc)
        return rc;
    if (entry.n_bytes)
        n_bytes = entry.n_bytes;
    if (entry.attr & vfat_attr_directory) {
        struct fat_dir_entry entries[128];
        // entries = mmalloc(n_bytes);
        if (entries) {
            vfat_read_clusters(vfat, (char *)entries, fat_entry_first_clusterno(&entry), n_bytes);
            vfat_list_dir_entries(entries);
            if (entries != vfat->root_dir_entries)
                mfree(entries);
        } else {
            return -EINVAL;
        }
    } else {
        /* file entry */
        vfat_list_one_entry(&entry);
    }
    return 0;
}



static int vfat_iohandle_file_read(struct iohandle *ioh, char *buf, size_t offset, size_t nbytes)
{
    struct vfat_filesystem *vfat = (struct vfat_filesystem *)ioh->ops->pdata;
    struct fat_dir_entry *entry = (struct fat_dir_entry *)ioh->pdata;
    u32 first_clusterno = fat_entry_first_clusterno(entry);
    u32 clusterno = vfat_clusterno(vfat, first_clusterno, offset);
    return vfat_read_clusters(vfat, buf, clusterno, nbytes);
}

int vfat_iohandle_release(struct iohandle *ioh)
{
    // mfree(ioh);
    return 0;
}

struct iohandle_ops vfat_iohandle_ops = {
    read: vfat_iohandle_file_read,
    close: vfat_iohandle_release
};

static char storage[sizeof(struct iohandle_ops)
                   + sizeof(struct iohandle)
                   + sizeof(struct fat_dir_entry)];

struct iohandle *vfat_get_file_entry_iohandle(struct vfat_filesystem *vfat, struct fat_dir_entry *entry)
{
    struct iohandle_ops *iohops = (struct iohandle_ops *)storage;
    struct iohandle *ioh = (struct iohandle *)(storage + sizeof(struct iohandle_ops));
    struct fat_dir_entry *fde = (struct fat_dir_entry *)(storage + sizeof(struct iohandle_ops) + sizeof(struct iohandle));
    memcpy(fde, entry, sizeof(struct fat_dir_entry));
    memcpy((char *)iohops, (char *)&vfat_iohandle_ops, sizeof(struct iohandle_ops));
    iohops->pdata = vfat;
    ioh->ops = iohops;
    ioh->pdata = fde;
    return ioh;
}


int vfat_mount(struct vfat_filesystem *vfat, struct iohandle *ioh)
{
    char oemname[9];
    struct bpb_info *info = &vfat->info;
    int rc = 0;
    size_t sector_size = 0x200;
    memset(vfat, 0, sizeof(struct vfat_filesystem));
    vfat->iohandle = ioh;
    putstr("vfat mount: reading bpb_info\r\n");
    if (vfat_read_bpb_info(vfat, ioh) == 0) {
        u32 sectors_per_cluster = bpb_sectors_per_cluster(info);
        u32 n_reserved_sectors = bpb_n_reserved_sectors(info);
        u32 n_root_entries = bpb_n_root_entries(info);
        u32 root_dir_sectors = bpb_root_dir_sectors(info);
        u32 first_root_dir_sector = bpb_first_root_dir_sector(info);
        u32 fat_size = bpb_fat_size(info);
        u32 fat_size_bytes = fat_size * sector_size;
        u32 n_fats = bpb_n_fats(info);
        u32 total_sectors = bpb_total_sectors(info);
        u32 first_data_sector = bpb_first_data_sector(info);
        u32 n_data_sectors = bpb_n_data_sectors(info);
        u32 n_clusters = bpb_n_clusters(info);
        u32 fat_type = bpb_fat_type(info);

        hex_dump((unsigned char *)&vfat->info, sizeof(struct bpb_info));
        memcpy(oemname, vfat->info.oemname, 8);
        putstr("  oemname="); putstr(oemname); putstr("\r\n");
        putLabeledWord("  sectors_per_cluster=", sectors_per_cluster);
        putLabeledWord("  n_reserved_sectors=", n_reserved_sectors);
        putLabeledWord("  n_root_entries=", n_root_entries);
        putLabeledWord("  root_dir_sectors=", root_dir_sectors);
        putLabeledWord("  first_root_dir_sector=", first_root_dir_sector);
        putLabeledWord("  fat_size=", fat_size);
        putLabeledWord("  fat_size_bytes=", fat_size_bytes);
        putLabeledWord("  n_fats=", n_fats);
        putLabeledWord("  total_sectors=", total_sectors);
        putLabeledWord("  n_data_sectors=", n_data_sectors);
        putLabeledWord("  first_data_sector=", first_data_sector);
        putLabeledWord("  n_clusters=", n_clusters);
        putLabeledWord("  fat_type=", fat_type);

        /* gather data */ 
        sector_size = bpb_bytes_per_sector(info);
        vfat->sector_size = sector_size;

        vfat->fat_type = fat_type;

        /* read the root_dir_entries */
        vfat->n_root_entries = n_root_entries;
        vfat->root_dir_entries = (struct fat_dir_entry *)mmalloc(n_root_entries * sizeof(struct fat_dir_entry));
        { 
            int offset = 0;
            u32 start_byte = sector_size * first_root_dir_sector;
            u32 nbytes = n_root_entries * sizeof(struct fat_dir_entry);
            char *buf = (char *)vfat->root_dir_entries;
            for (offset = 0; offset < nbytes; offset += sector_size) {
                rc = ioh->ops->read(ioh, buf+offset, start_byte+offset, sector_size);
                if (rc)
                    return rc;
            }
        }
        putstr("root_dir_entries:\r\n");
        hex_dump((char *)vfat->root_dir_entries, 0x96);

        /* read the fat */
        vfat->fat_size = fat_size;
        vfat->fat = mmalloc(sector_size * bpb_fat_size(info));
        { 
            int offset = 0;
            u32 start_byte = sector_size * bpb_n_reserved_sectors(info);
            u32 nbytes = sector_size * bpb_fat_size(info);
            char *buf = (char *)vfat->fat;
            for (offset = 0; offset < nbytes; offset += sector_size) {
                rc = ioh->ops->read(ioh, buf+offset, start_byte+offset, sector_size);
                if (rc)
                    return rc;
            }
        }
        putstr("fat:\r\n");
        hex_dump((char *)vfat->fat, 0x40);

        /* read some data */
        rc = ioh->ops->read(ioh, vfat->buf, sector_size * first_root_dir_sector, sector_size);
        putstr("first data:\r\n");
        hex_dump(vfat->buf, 0x40);
    }
    return rc;
}



struct vfat_filesystem *vfat = 0; 

SUBCOMMAND(vfat, mount, command_vfat, "-- fat partition on IDE partition <partid>", BB_RUN_FROM_RAM, 0);
SUBCOMMAND(vfat, ls, command_vfat,    "-- directory on mounted fat filesystem.  <dir> defaults to root.", BB_RUN_FROM_RAM, 0);
SUBCOMMAND(vfat, read, command_vfat,  "[<len>] -- copies file contents into DRAM.", BB_RUN_FROM_RAM, 0);
SUBCOMMAND(vfat, next_cluster, command_vfat, "debug -- computes next clusterno from fat", BB_RUN_FROM_RAM, 0); 

void command_vfat(int argc, const char **argv)
{
    if (strcmp(argv[1], "mount") == 0) {
        struct bpb_info *info;
        u32 partid = 2;
        struct iohandle *ioh = NULL;
        int rc = 0;

        vfat = mmalloc(sizeof(struct vfat_filesystem));;
        if (!vfat) {
           putstr("mmalloc failed\r\n");
           return;
        }
        info = &vfat->info;
        if (argv[2])
            partid = strtoul(argv[2], 0, 0);
        putLabeledWord("cmd vfat mount: partid=", partid);
        ioh = get_ide_partition_iohandle(partid);
        if (!ioh) {
           putstr("get_ide_partition_iohandle failed\r\n");
           return;
        }
        rc = vfat_mount(vfat, ioh);
        if (rc) {
          putLabeledWord("vfat_mount failed with rc=", rc);
          mfree(vfat);
          vfat = NULL;
          return;
        }          
        putstr("cmd vfat mount: listing the root directory \r\n");
        vfat_list_dir_entries(vfat->root_dir_entries);
    } else if (strcmp(argv[1], "ls") == 0) {
        const char *fname = argv[2];
        struct fat_dir_entry entries[128];

        if (vfat) {
            struct fat_dir_entry entry;
            memset(&entry, 0, sizeof(entry));
            if (!fname) {
                vfat_list_dir_entries(vfat->root_dir_entries); 
            } else {
                int rc = vfat_find_dir_entry(vfat, &entry, fname);
                putLabeledWord("vfat ls: rc=", rc);
                putLabeledWord("vfat ls: entry.n_bytes=", entry.n_bytes);
                vfat_read_clusters(vfat, (char *)entries, fat_entry_first_clusterno(&entry), 0x200);
                vfat_list_dir_entries(entries);
            }
        } else {
            putstr("no vfat mounted\r\n");
        }
    } else if (strcmp(argv[1], "next_cluster") == 0) {
        u32 clusterno = strtoul(argv[2], 0, 0);
        putLabeledWord("clusterno=", clusterno);
        putLabeledWord("nextcluster=", vfat_next_clusterno(vfat, clusterno));
    } else if (strcmp(argv[1], "read_cluster") == 0) {
        u32 dstaddr = strtoul(argv[2], 0, 0);
        u32 clusterno = strtoul(argv[3], 0, 0);
        u32 nbytes = strtoul(argv[4], 0, 0);
        if (!argv[2] || !argv[3] || !argv[4])
            goto usage;
        vfat_read_clusters(vfat, (char *)dstaddr, clusterno, nbytes);
    } else if (strcmp(argv[1], "read") == 0) {
        u32 dstaddr = strtoul(argv[2], 0, 0);
        const char *fname = argv[3];
        size_t len = strtoul(argv[4], 0, 0);
        struct fat_dir_entry entry;
        int rc;
        if (!dstaddr || !fname)
            goto usage;
        if (vfat) {
            rc = vfat_find_file_entry(vfat, &entry, fname);
            putLabeledWord("got file entry rc=", -rc);
            if (!rc) {
                struct iohandle *ioh = vfat_get_file_entry_iohandle(vfat, &entry);
                putLabeledWord(" entry.n_bytes=", entry.n_bytes);
                if (!len || len > entry.n_bytes) 
                    len = entry.n_bytes;
                if (ioh) {
                    rc = ioh->ops->read(ioh, (char*)dstaddr, 0, len);
                }
            }
            putLabeledWord("  rc=", rc);
        } else {
            putstr("no vfat mounted\r\n");
        }
    } else {
    usage:
        putstr("usage: vfat mount <partno> | ls [dir] | read_cluster <dramaddr> <clusterno> | read <dramaddr> file [nbytes]\r\n");
        putstr("       vfat next_cluster <clusterno>\r\n"); 
    }
    return 0;
}

/*
 * Mount vfat partition on ide partition partno (numbered from zero)
 */
int vfat_mount_partition(int partid)
{
   struct bpb_info *info;
   struct iohandle *ioh = NULL;

   vfat = mmalloc(sizeof(struct vfat_filesystem));;
   info = &vfat->info;
   putLabeledWord("cmd vfat mount: partid=", partid);
   ioh = get_ide_partition_iohandle(partid);
   return vfat_mount(vfat, ioh);
}


/* 
 * Reads nbytes of data from the named file into the provided buffer. 
 *  If nbytes is 0, read whole file.
 */
int vfat_read_file(char *buf, const char *filename, size_t nbytes)
{
   if (vfat) {
      struct fat_dir_entry entry;
      int rc = vfat_find_file_entry(vfat, &entry, filename);
      u32 first_clusterno;
      u32 clusterno; 
      if (rc)
         return rc;
      if (nbytes == 0)
         nbytes = entry.n_bytes;
      first_clusterno = fat_entry_first_clusterno(&entry);
      return vfat_read_clusters(vfat, buf, first_clusterno, nbytes);
   } else {
      return -EINVAL;
   }
}
