/****************************************************************************/
/* Copyright 2002 Compaq Computer Corporation.                              */
/*                                           .                              */
/* Copying or modifying this code for any purpose is permitted,             */
/* provided that this copyright notice is preserved in its entirety         */
/* in all copies or modifications.  COMPAQ COMPUTER CORPORATION             */
/* MAKES NO WARRANTIES, EXPRESSED OR IMPLIED, AS TO THE USEFULNESS          */
/* OR CORRECTNESS OF THIS CODE OR ITS FITNESS FOR ANY PARTICULAR            */
/* PURPOSE.                                                                 */
/****************************************************************************/
/*
 * iohandles, copy command and friends
 */

#include "commands.h"
#include "bootldr.h"
#include "btflash.h"
#include "hal.h"
#include "h3600_sleeve.h"
#include "pcmcia.h"
#include "ide.h"
#include "fs.h"
#include "vfat.h"
#include "heap.h"
#include "params.h"
#include "zlib.h"
#include "zUtils.h"

#define BUFSIZE 1024



struct flash_iohandle {
  u32 base;
  u32 size;
  u32 flags;
};



static int flash_iohandle_len(struct iohandle *ioh)
{
  if (ioh) {
    struct FlashRegion *partition = (struct FlashRegion *)ioh->pdata;
    return partition->size;
  } else {
    return 0;
  }
}

static int flash_iohandle_read(struct iohandle *ioh, char *buf, size_t offset, size_t len)
{
  if (ioh) {
    struct FlashRegion *partition = (struct FlashRegion *)ioh->pdata;
    if (!len || len > partition->size)
      len = partition->size;
    memcpy(buf, ((char *)flashword) + partition->base + offset, len);
    return len;
  } else {
    return -EINVAL;
  }
}

static int flash_iohandle_prewrite(struct iohandle *ioh, size_t offset, size_t len)
{
  if (ioh) {
    struct FlashRegion *partition = (struct FlashRegion *)ioh->pdata;
    len = partition->size;

    set_vppen();
    eraseFlashRange(partition->base, len);
    clr_vppen();

    return len;
  } else {
    return -EINVAL;
  }  
}

static int flash_iohandle_write(struct iohandle *ioh, char *src, size_t offset, size_t len)
{
  if (ioh) {
    struct FlashRegion *partition = (struct FlashRegion *)ioh->pdata;
    unsigned long regionBase = partition->base + offset;
    // unsigned long regionSize = partition->size;
    // int flags = partition->flags;
    unsigned long i, remaining_bytes, bytes_programmed;
    int num_bytes;
    char *p;
    u32 q;
    if (!len)
      return -EINVAL;
    i = 0;
    remaining_bytes = len;
    set_vppen();
    while(remaining_bytes > 0) { 

      num_bytes = 64;
      p = src + i;
      q = regionBase + i;

      if (((q % 64) == 0) && (remaining_bytes > 64) && (num_bytes == 64)) {
        /* program a block */
        if (programFlashBlock(q, (unsigned long*)p, 64)) {
          putstr("error while copying to flash!\r\n");
          break;
        }
        bytes_programmed = 64;
      } else {
        if (programFlashWord(q, *(unsigned long *)p)) {
          putstr("error while copying to flash!\r\n");
          break;
        }
        bytes_programmed = 4;
      }
      i += bytes_programmed;
      remaining_bytes -= bytes_programmed;
      
    }
    clr_vppen();
    return i;
  } else {
    return -EINVAL;
  }
}

static int flash_iohandle_close(struct iohandle *ioh)
{
  if (ioh) {
    mfree(ioh);
    return 0;
  } else {
    return -EINVAL;
  }
}

struct iohandle_ops flash_iohandle_ops = {
  len:  flash_iohandle_len,
  read: flash_iohandle_read,
  prewrite: flash_iohandle_prewrite,
  write: flash_iohandle_write,
  close: flash_iohandle_close,
};



struct iohandle *flash_open_iohandle(const char *partname)
{
  struct FlashRegion *partition = btflash_get_partition(partname);
  char *storage = mmalloc(sizeof(struct iohandle));
  struct iohandle *ioh = (struct iohandle *)storage;
  if (!partition) {
     putstr(__FUNCTION__ ": no partition named "); putstr(partname); putstr("\r\n");
     return (void*)-ENOENT;
  }
  if (!ioh) {
     putstr(__FUNCTION__ ": out of memory\r\n");
     return (void*)-ENOMEM;
  }
  ioh->sector_size = 512;
  ioh->pdata = storage + sizeof(struct iohandle);
  ioh->ops = &flash_iohandle_ops;
  ioh->pdata = partition;
  return ioh;
} 




struct iohandle *open_iohandle(const char *spec)
{
   struct FlashRegion *partition = NULL;
  if (!spec) {
    return NULL;
  } else if (strncmp(spec, "hda1:", 4) == 0) {
     return vfat_file_open(spec+5);
  } else if ((partition = btflash_get_partition(spec))) {
     return flash_open_iohandle(spec);
  } else {
     return NULL;
  }
}



COMMAND(copy, command_copy, "src dst [len] -- copy from src to dst", BB_RUN_FROM_RAM);
void command_copy(int argc, const char **argv)
{
  const char *srcspec = argv[1];
  const char *dstspec = argv[2];
  const char *lenspec = argv[3];
  struct iohandle *srchandle;
  struct iohandle *dsthandle;
  char buf[BUFSIZE];
  u32 len = 0;
  u32 srclen = 0;
  u32 dstlen = 0;
  u32 offset;
  u32 bytes_read = BUFSIZE;
  if (!srcspec || !dstspec) {
    goto usage;
  }
  srchandle = open_iohandle(srcspec);
  if (!srchandle) {
    putstr("Could not open source "); putstr(srcspec); putstr("\r\n");
    return;
  }
  dsthandle = open_iohandle(dstspec);
  if (!dsthandle) {
    putstr("Could not open destination "); putstr(dstspec); putstr("\r\n");
    return;
  }
  if (!dsthandle->ops || !dsthandle->ops->write) {
    putstr("destination "); putstr(dstspec); putstr(" does not support writing.\r\n");
    return;
  }
  if (lenspec) {
    len = strtoul(lenspec, 0, 0);
  }
  if (srchandle->ops->len && srchandle->ops->len(srchandle))
    srclen = srchandle->ops->len(srchandle);
  if (!len || (srclen && srclen < len))
      len = srclen;
  if (dsthandle->ops->len && dsthandle->ops->len(dsthandle)) {
    dstlen = dsthandle->ops->len(dsthandle);

  }
  putLabeledWord("srclen=", srclen);
  putLabeledWord("dstlen=", dstlen);
  putLabeledWord("nbytes=", len);

  if (!len)
    return;

  /*  if (dstlen && dstlen < len) {
    putstr("nbytes is too large for destination size ", dstlen);
    return;
    } */
  /* now prepare the destination */
  if (dsthandle->ops->prewrite)
    if (dsthandle->ops->prewrite(dsthandle, 0, len) < 0)
      return;
  /* now copy the data */
  for (offset = 0; offset < len; offset += bytes_read) {
    int rc;
    rc = srchandle->ops->read(srchandle, buf, offset, BUFSIZE);
    if (rc <= 0) {
      putLabeledWord(__FUNCTION__ ": read failed err=", rc);
      break;
    }
    bytes_read = rc;
    putstr(".");
    if ((offset % SZ_64K) == 0) {
      putstr("addr: ");
      putHexInt32(offset);
      putstr(" data: ");
      putHexInt32(*(unsigned long *) buf);
      putstr("\r\n");
    }
    rc = dsthandle->ops->write(dsthandle, buf, offset, bytes_read);
    if (rc < 0) {
      putLabeledWord(__FUNCTION__ ": write failed err=", rc);
      break;
    }
  }
  dsthandle->ops->close(dsthandle);
  srchandle->ops->close(srchandle);

  if (strcmp(dstspec, "params") == 0) {
	params_eval(PARAM_PREFIX, 0);
  }
  return;
 usage:
  putstr("usage: copy <srcspec> <dstspec>\r\n"
	 "    srcspec = <srcaddr>\r\n"
	 "            | <flashpartname> \r\n"    
	 "            | hda1:<filename> [<len>]\r\n"    
	 );
}
