#include <bios/config.h>
#include <bios/bootdev.h>
#include <bios/pci.h>
#include <bios/malloc.h>
#include <bios/time.h>
#include <bios/stdio.h>
#include <bios/stdioint.h>
#include <bios/string.h>
#include <bios/system.h>

typedef enum {
	ram_failure,
	ram_ok
} ram_test_t;

typedef enum {
	boot_failure,
	boot_single,
	boot_multiple
} boot_test_t;

extern struct bootdev boot_net;
extern struct bootdev boot_ide;
extern struct bootdev boot_scsi;

static struct bootdev *first;
static struct bootdev *bootdevs[] = {
#ifdef CONFIG_BOOT_IDE
	&boot_ide,
#endif
#ifdef CONFIG_BOOT_SCSI
	&boot_scsi,
#endif
#ifdef CONFIG_BOOT_NET
	&boot_net,
#endif
	NULL
};

extern unsigned long ramtest(unsigned long, unsigned long);

int ram_size;

static ram_test_t ram_test(void)
{
	unsigned int ptr;
	ram_test_t ret = ram_ok;

	printf("       KB SDRAM OK");

	for (ptr = 0; ptr < ram_size; ptr += 512) {
		if (ramtest(ptr, 0x55aacc33) != 0x55aacc33) {
			printf("\nMemory error detected at address 0x%08X\n",
				ptr);
			ret = ram_failure;
			break;
		}
		if ((ptr & 524280) == 0)
			printf("\r%6d", ptr >> 10);
	}

	if (ret == ram_ok)
		printf("\r%6d\n", ram_size >> 10);

	return ret;
}

static boot_test_t locate_boot_device(void)
{
	struct bootdev **prev;
	int i, found = 0;

	first = NULL;
	prev = &first;

	printf("Locating bootable devices: ");

	for (i = 0; bootdevs[i]; i++) {
		if (bootdevs[i]->init() == 0) {
			printf(found ? ", %s" : "%s", bootdevs[i]->name);
			*prev = bootdevs[i];
			prev = &bootdevs[i]->next;
			found++;
		}
	}

	if (!found)
		goto none;

	printf("\n");

	return found != 1 ? boot_multiple : boot_single;

none:
	printf("none\n");

	return boot_failure;
}

static void panic(const char *s)
{
	if (s)
		printf("%s", s);
	printf(" -- System Halted");
	while(1);
}

static int bdev;
int img_nr;
static int root_dev;

static void boot(void)
{
	printf("Now booting image...\n");

	sti();

	boot_kernel();
}

static void auto_boot(void)
{
	struct bootdev *dev;

	printf(" starting autoboot\n");
	wait_cs(200);

	switch (locate_boot_device()) {
	case boot_failure:
		panic("unable to locate boot device");

	case boot_multiple:
//		first = boot_select(first);

	case boot_single:
		break;
	}

	for (dev = first; dev; dev = dev->next) {
		printf("Trying %s...\n", dev->name);
		if (dev->start())
			continue;
		if (dev->load() == 0)
			break;
		printf("Warning: %s failed\n", dev->name);
		dev->stop();
	}

	if (!dev)
		panic("no boot devices found");

	dev->stop();

	boot();
}

static const char *bdevs[] = {
	"auto",
	"ide",
	"net",
	"scsi"
};

static void manual_boot(void)
{
	while (1) {
		int c;
		printf("\014EBSA285 manual boot\n");

		printf("1 - Boot method:  %-10s\n", bdevs[bdev]);
		printf("2 - Image number: %d\n", img_nr);
		printf("3 - Root device:  %04X\n", root_dev);
		printf("\n");
		printf("b - boot    p - pci config\n");

		printf("\ncfg> ");

		c = getc();

		printf("%c\n", c);

		switch (c) {
		case '1':
			bdev = (bdev + 1) & 3;
			break;

		case '2':
			img_nr = (img_nr + 1) & 3;

		case '3':
			break;

		case 'b':
			auto_boot();
			break;

		case 'p':
			pci_print_config();
			getc();
			break;
		}
	}

	boot();
}

/*
 * This is the main entry point for the BIOS.
 */
void start_main(void)
{
	int i;

	debug_init();	/* Initialise serial debug	*/
	malloc_init();	/* Initialise malloc pool	*/
	time_init();	/* Initialise timers		*/
	cli();

	/*
	 * Wait a while until the PCI
	 * devices have properly reset.
	 */
	wait_cs(1);

	pci_init();	/* Initialise PCI sub-system	*/
	vga_init();	/* Initialise VGA adapter	*/
#if 0
	isa_init();	/* Initialise ISA		*/
#else
	init_87338();	/* Initialise ISA		*/
#endif

	printf("EBSA285 Linux BIOS v"VERSION" (c) 1998-1999 Russell King (rmk@arm.linux.org.uk)\n\n");

	/* Check integrity of RAM */
	if (ram_test() == ram_failure)
		panic("ram failure");

	printf("\nHit 's' twice to abort autoboot...");

	i = 0;
	while (1) {
		int c;

		c = getc_timed(200);

		if (c == -1)
			break;

		printf("%c", c);

		if (c == 's')
			i += 1;
		else
			i = 0;

		if (i == 2)
			break;
	}

	if (i == 2)
		manual_boot();
	else
		auto_boot();

	while (1);
}
