#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <sys/fcntl.h>
#include <unistd.h>

struct exec
{
  __u32 a_info;		/* Use macros N_MAGIC, etc for access */
  __u32 a_text;		/* length of text, in bytes */
  __u32 a_data;		/* length of data, in bytes */
  __u32 a_bss;		/* length of uninitialized data area for file, in bytes */
  __u32 a_syms;		/* length of symbol table data in file, in bytes */
  __u32 a_entry;	/* start address */
  __u32 a_trsize;	/* length of relocation info for text, in bytes */
  __u32 a_drsize;	/* length of relocation info for data, in bytes */
};

struct aif {
	__u32	decomp_code;
	__u32	reloc_code;
	__u32	init_code;
	__u32	entry_code;
	__u32	exit_code;
	__u32	text_size;
	__u32	data_size;
	__u32	dbg_size;
	__u32	bss_size;
	__u32	dbg_type;
	__u32	image_base;
	__u32	workspace;
	__u32	reserved[4];
	__u32	unused[16];
};

char buffer[8192];

int main(int argc, char *argv[])
{
	struct exec exec;
	struct aif aifhdr;
	int aout, aif, bytes, total;

	if (argc != 3) {
		fprintf(stderr, "Usage: %s A.OUT AIF\n", argv[0]);
		exit (1);
	}

	aout = open(argv[1], O_RDONLY);

	if (aout < 0) {
		fprintf(stderr, "%s: %s\n", argv[1], strerror(errno));
		exit (1);
	}

	if (lseek(aout, 0, SEEK_SET) < 0) {
		fprintf(stderr, "%s: unable to lseek: %s\n", argv[1], strerror(errno));
		exit (1);
	}

	bytes = read(aout, &exec, sizeof(exec));

	if (bytes < 0) {
		fprintf(stderr, "%s: %s\n", argv[1], strerror(errno));
		exit (1);
	} else if (bytes != sizeof(exec)) {
		fprintf(stderr, "%s: file truncated\n", argv[1]);
		exit (1);
	}

	printf("Bios size: %d bytes text at 0x%X, %d bytes data, %d bytes bss\n",
		exec.a_text,
		exec.a_entry,
		exec.a_data,
		exec.a_bss);

	memset(&aifhdr, 0, sizeof(aifhdr));

	aifhdr.entry_code = 0xeb00001b;
	aifhdr.exit_code  = 0xeafffffe;
	aifhdr.text_size  = exec.a_text;
	aifhdr.data_size  = exec.a_data;
	aifhdr.bss_size   = exec.a_bss;
	aifhdr.image_base = exec.a_entry - 128;

	total = exec.a_text + exec.a_data;

	aif = open(argv[2], O_WRONLY | O_CREAT, 0666);
	if (aif < 0) {
		fprintf(stderr, "%s: %s\n", argv[2], strerror(errno));
		exit(1);
	}

	if (lseek(aif, 0, SEEK_SET) < 0) {
		fprintf(stderr, "%s: unable to lseek: %s\n", argv[2], strerror(errno));
		exit(1);
	}

	bytes = write(aif, &aifhdr, sizeof(aifhdr));

	if (bytes < 0) {
		fprintf(stderr, "%s: %s\n", argv[2], strerror(errno));
		exit (1);
	} else if (bytes != sizeof(aifhdr)) {
		fprintf(stderr, "%s: file truncated\n", argv[2]);
		exit (1);
	}

	do {
		bytes = total;
		if (bytes > sizeof(buffer))
			bytes = sizeof(buffer);

		bytes = read(aout, buffer, bytes);

		if (bytes > 0)
			bytes = write(aif, buffer, bytes);

		if (bytes < 0) {
			fprintf(stderr, "unable to copy: %s\n", strerror(errno));
			close(aif);
			unlink(argv[2]);
			exit(1);
		}

		total -= bytes;
	} while (total);

	close(aif);
	close(aout);
	return 0;
}
