/* arm specific support for Elf loading and relocation.
   Copyright 1996, 1997 Linux International.

   Contributed by wms <woody@corelcomputer.com>

   This file is part of the Linux modutils.

   This program is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by the
   Free Software Foundation; either version 2 of the License, or (at your
   option) any later version.

   This program is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software Foundation,
   Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

/*#ident "$Id: obj_arm.c,v 1.1 1997/09/10 22:20:02 wms Exp $"*/

#include <string.h>
#include <assert.h>

#include <module.h>
#include <obj.h>
#include <util.h>


/*======================================================================*/

struct arm_plt_entry
{
//  struct arm_plt_entry *next;  // make a linked list
//  ElfW(Addr) r_info;           // ??
    int offset;
    int allocated:1;
    int inited:1;                // has been set up
};

struct arm_got_entry
{
    int offset;
    int allocated:1;
    int reloc_done:1;
};

struct arm_file
{
  struct obj_file file;
  struct obj_section *plt;
  struct obj_section *got;
};

struct arm_symbol
{
  struct obj_symbol sym;
  struct arm_plt_entry plt_entry;
  struct arm_got_entry got_entry;
};

/*======================================================================*/

struct obj_file *
arch_new_file (void)
{
  struct arm_file *f;

  f = xmalloc(sizeof(struct arm_file));
  f->got = f->plt = NULL;
  return &f->file;
}

struct obj_section *
arch_new_section (void)
{
  return xmalloc(sizeof(struct obj_section));
}

struct obj_symbol *
arch_new_symbol (void)
{
  struct arm_symbol *p;

  p = xmalloc(sizeof(struct arm_symbol));
  return &p->sym;
}

/*======================================================================*/

enum obj_reloc
arch_apply_relocation (struct obj_file *f,
		       struct obj_section *targsec,
		       struct obj_section *symsec,
		       struct obj_symbol *sym,
		       Elf32_Rel *rel,
		       Elf32_Addr v)
{
  struct arm_file *afile = (struct arm_file *)f;
  struct arm_symbol *asym  = (struct arm_symbol *)sym;

  Elf32_Addr *loc = (Elf32_Addr *)(targsec->contents + rel->r_offset);
  Elf32_Addr dot = targsec->header.sh_addr + rel->r_offset;
  Elf32_Addr got = afile->got ? afile->got->header.sh_addr : 0;
  Elf32_Addr plt = afile->plt ? afile->plt->header.sh_addr : 0;

  struct arm_plt_entry *pe;
#define instruction long unsigned
  instruction *ip;

  
  enum obj_reloc ret = obj_reloc_ok;

  switch (ELF32_R_TYPE(rel->r_info))
    {
    case R_ARM_NONE:
      break;
 
    case R_ARM_ABS32:
      *loc += v;
      break;

    case R_ARM_GOT32:
      /* needs an entry in the .got: set it, once */
        if (!asym->got_entry.reloc_done)
        {
            asym->got_entry.reloc_done = 1;
            *(Elf32_Addr*)(afile->got->contents + asym->got_entry.offset) = v;
        }
        /* make the reloc with_respect_to_.got */
        *loc += asym->got_entry.offset;
        break;
        
      /* relative reloc, always to _GLOBAL_OFFSET_TABLE_ (which is .got)
       similar to branch, but is full 32 bits relative */
    case R_ARM_GOTPC:
        assert(got);
        *loc += got - dot;
        break;
        
    case R_ARM_PC24:
    case R_ARM_PLT32:
      /* find the plt entry and initialize it if necessary */
        assert(asym != NULL);
        pe = (struct arm_plt_entry*) &asym->plt_entry;
        assert(pe != NULL);
        if (!pe->inited)
	{
	  ip = (instruction *) (afile->plt->contents + pe->offset);
          ip[0] = 0xe51ff004;			/* ldr pc,[pc,#-4] */
	  ip[1] = v;	 			/* sym@ */
	  pe->inited = 1;
	}

      /* relative distance to target */
      v -= dot;
      /* if the target is too far away.... */
      if ((int)v < -0x02000000 || (int)v >= 0x02000000)
	{
	  /* go via the plt */
	  v = plt + pe->offset - dot;
	}
      if (v & 3)
	ret = obj_reloc_dangerous;
	
      v -= 8;		//ARM specific - adjust PC
      v >>= 2;		//ARM specific - offset in a branch is in dwords
      /* merge the offset into the instruction */
      *loc = (*loc & ~0x00ffffff) | (v & 0x00ffffff);
      break;

      /* address relative to the got */
    case R_ARM_GOTOFF:
        assert(got);
        *loc += v - got;
        break;
        
    case R_ARM_GLOB_DAT:
    case R_ARM_COPY:
    case R_ARM_JUMP_SLOT:
      printf("Warning: unhandled reloc %d\n",ELF32_R_TYPE(rel->r_info));
      printf("Warning: this module was compiled for PIC....\n"
             "This version of insmod does not load this type of module.\n");
      break;

    default:
      printf("Warning: unhandled reloc %d\n",ELF32_R_TYPE(rel->r_info));
      ret = obj_reloc_unhandled;
      break;
    }

  return ret;
}

/* For modules which may have been created PIC,
 we may need to make a .plt section and/or a .got
 section

 These are syntetic sections. The .plt is a collection
 of jump stubs, 2 words for each extern/global
 subroutine call. These can handle far calls.

 The .got is a simple array of pointers to objects;
 one word per pointer.

 This routine, arch_create_got, does a quick scan through
 the symbols, counting the requirements for .plt and .got,
 and creates the sections.
 */
int
arch_create_got (struct obj_file *f)
{
  struct arm_file *afile = (struct arm_file *) f;
  int i;
  struct obj_section *sec, *syms, *strs;
  ElfW(Rel) *rel, *relend;
  ElfW(Sym) *symtab, *extsym;
  const char *strtab, *name;
  struct arm_symbol *intsym;
  struct arm_plt_entry *pe;
  struct arm_got_entry *ge;
  int got_offset = 0, plt_offset = 0;

  for (i = 0; i < f->header.e_shnum; ++i)
    {
      sec = f->sections[i];
      if (sec->header.sh_type != SHT_RELM)
	continue;
      syms = f->sections[sec->header.sh_link];
      strs = f->sections[syms->header.sh_link];

      rel = (ElfW(RelM) *) sec->contents;
      relend = rel + (sec->header.sh_size / sizeof(ElfW(RelM)));
      symtab = (ElfW(Sym) *) syms->contents;
      strtab = (const char *) strs->contents;

      for (; rel < relend; ++rel)
      {
          extsym = &symtab[ELF32_R_SYM(rel->r_info)];

          switch(ELF32_R_TYPE(rel->r_info)) {
          case R_ARM_PC24:
          case R_ARM_PLT32:
              if (extsym->st_name)
                name = strtab + extsym->st_name;
              else
                name = f->sections[extsym->st_shndx]->name;
              intsym = (struct arm_symbol *) obj_find_symbol(f, name);

              pe = &intsym->plt_entry;

              if (!pe->allocated)
              {
                  pe->allocated = 1;
                  pe->offset = plt_offset;
                  plt_offset += 8;
                  pe->inited = 0;
              }
              break;

              /* these two don_t need got entries, but they need
               the .got to exist */
          case R_ARM_GOTOFF:
          case R_ARM_GOTPC:
              if (got_offset==0) got_offset = 4;
              break;
              
          case R_ARM_GOT32:
              if (extsym->st_name)
                name = strtab + extsym->st_name;
              else
                name = f->sections[extsym->st_shndx]->name;
              intsym = (struct arm_symbol *) obj_find_symbol(f, name);

              ge = (struct arm_got_entry*) &intsym->got_entry;
              if (!ge->allocated)
              {
                  ge->allocated = 1;
                  ge->offset = got_offset;
                  got_offset += sizeof(void*);
              }
              break;
          default:
              continue;
          }
	}
    }

  //  printf("creating .got section of size %d\n",got_offset);
  //  printf("creating .plt section of size %d\n",plt_offset);
  /* if there was a _GLOBAL_OFFSET_TABLE_, then the .got section
   exists already; find it and use it */
  if (got_offset)
  {
      struct obj_section* sec = obj_find_section(f, ".got");
      if (sec)
          obj_extend_section(sec, got_offset);
      else {
          sec = obj_create_alloced_section(f, ".got", 8, got_offset);
          assert(sec);
      }
      afile->got = sec;
  }
  if (plt_offset)
      afile->plt = obj_create_alloced_section(f, ".plt", 8, plt_offset);

  return 1;
}


int
arch_init_module (struct obj_file *f, struct new_module *mod)
{
  return 1;
}
