// mpeglayer3mktables.cc
// (C) 1999 Nicolas Pitre <nico@cam.org>
//
// This code generate the mpeglayer3tables.h file
//
// To compile: gcc -Wall -O2 -o mk mpeglayer3mktables.cc -lm
// To build the table: ./mk


#include <math.h>
#include <stdio.h>


#define OUT_FNAME "mpeglayer3tables.h"


typedef double REAL;

#define PI     M_PI
#define PI_12  (PI/12.0)
#define PI_18  (PI/18.0)
#define PI_24  (PI/24.0)
#define PI_36  (PI/36.0)
#define PI_72  (PI/72.0)

#define FOURTHIRDSTABLENUMBER (1<<13)

static REAL two_to_negative_half_pow[40];
static REAL TO_FOUR_THIRDSTABLE[FOURTHIRDSTABLENUMBER];
static REAL POW2[256];
static REAL POW2_1[8][2][16];
static REAL ca[8],cs[8];

static REAL win[4][36];
static REAL cos_18[9];
static REAL hsec_36[9],hsec_12[3];

REAL cos1_6 = cos(PI/6.0*1.0);
REAL cos2_6 = cos(PI/6.0*2.0);

typedef struct
{
  REAL l,r;
}RATIOS;

static RATIOS rat_1[16],rat_2[2][64];



int main(void)
{
  /* code for computing all tables */
  int i, j, k;

  // Calculate win
  {
    for(i=0;i<18;i++)
      win[0][i]=win[1][i]=0.5*sin(PI_72*(double)(2*i+1))/cos(PI_72*(double)(2*i+19));
    for(;i<36;i++)
      win[0][i]=win[3][i]=0.5*sin(PI_72*(double)(2*i+1))/cos(PI_72*(double)(2*i+19));

    for(i=0;i<6;i++)
    {
      win[1][i+18]=0.5/cos(PI_72*(double)(2*(i+18)+19));
      win[3][i+12]=0.5/cos(PI_72*(double)(2*(i+12)+19));
      win[1][i+24]=0.5*sin(PI_24*(double)(2*i+13))/cos(PI_72*(double)(2*(i+24)+19));
      win[1][i+30]=win[3][i]=0.0;
      win[3][i+6 ]=0.5*sin(PI_24*(double)(2*i+1))/cos(PI_72*(double)(2*(i+6)+19));
    }
    for(i=0;i<12;i++)
      win[2][i]=0.5*sin(PI_24*(double)(2*i+1))/cos(PI_24*(double)(2*i+7));
  }

  for(i=0;i<9;i++)
    cos_18[i]=cos(PI_18*double(i));
  for(i=0;i<9;i++)
    hsec_36[i]=0.5/cos(PI_36*double(i*2+1));
  for(i=0;i<3;i++)
    hsec_12[i]=0.5/cos(PI_12*double(i*2+1));

  for(i=0;i<40;i++)
    two_to_negative_half_pow[i]=(REAL)pow(2.0,-0.5*(double)i);

  for(i=0;i<FOURTHIRDSTABLENUMBER;i++)
    TO_FOUR_THIRDSTABLE[i]=(REAL)pow((double)i,4.0/3.0);
  for(i=0;i<256;i++)POW2[i]=(REAL)pow(2.0,(0.25*(i-210.0)));
  for(i=0;i<8;i++)
    for(j=0;j<2;j++)
      for(k=0;k<16;k++)POW2_1[i][j][k]=pow(2.0,(-2.0*i)-(0.5*(1.0+j)*k));

  {
    static REAL TAN12[16]=
    { 0.0,        0.26794919, 0.57735027  , 1.0,
      1.73205081, 3.73205081, 9.9999999e10,-3.73205081,
      -1.73205081,-1.01,      -0.57735027,  -0.26794919,
      0.0,        0.26794919, 0.57735027,   1.0};

    for(i=0;i<16;i++)
    {
      rat_1[i].l=TAN12[i]/(1.0+TAN12[i]);
      rat_1[i].r=1.0/(1.0+TAN12[i]);
    }
  }

#define IO0 ((double)0.840896415256)
#define IO1 ((double)0.707106781188)
  rat_2[0][0].l=rat_2[0][0].r=
  rat_2[1][0].l=rat_2[1][0].r=1.;

  for(i=1;i<64;i++)
    if((i%2)==1)
    {
      rat_2[0][i].l=pow(IO0,(i+1)/2);
      rat_2[1][i].l=pow(IO1,(i+1)/2);
      rat_2[0][i].r=
      rat_2[1][i].r=1.;
    }
    else
    {
      rat_2[0][i].l=
      rat_2[1][i].l=1.;
      rat_2[0][i].r=pow(IO0,i/2);
      rat_2[1][i].r=pow(IO1,i/2);
    }

  {
    static REAL Ci[8]=
    {-0.6f,-0.535f,-0.33f,-0.185f,-0.095f,-0.041f,-0.0142f,-0.0037f};
    REAL sq;

    for(i=0;i<8;i++)
    {
      sq=sqrt(1.0f+Ci[i]*Ci[i]);
      cs[i]=1.0f/sq;
      ca[i]=Ci[i]*cs[i];
    }
  }


  /* now print out those tables to the appropriate file */

  FILE *out = fopen( OUT_FNAME, "w" );
  if( !out ){
    perror( OUT_FNAME );
    exit(-1);
  }
  fprintf( out, "// " OUT_FNAME "\n" );
  fprintf( out, "// It is included by mpeglayer3.cc\n" );
  fprintf( out, "// This file was generated using mpeglayer3mktables.cc\n" );
  fprintf( out, "// Do not edit this file but mpeglayer3mktables.cc instead\n\n\n" );

  fprintf( out, "#include <values.h>\n" );
  fprintf( out, "#ifndef FIXED_POINT\n" );
  fprintf( out, "#define X( x )  (x)\n" );
  fprintf( out, "#define inf  MAXFLOAT\n" );
  fprintf( out, "#define Inf  MAXFLOAT\n" );
  fprintf( out, "#else\n" );
  fprintf( out, "#define inf  MAXINT\n" );
  fprintf( out, "#define Inf  MAXINT\n" );
  fprintf( out, "#endif\n\n\n" );

  int s;
  static char sp[] = "   ";


#define TABLE( DECL, I, J, K, FMT, WRAP, VAL... ) \
  fprintf( out, "static const " DECL " =\n{\n" ); \
  s = 0; \
  i = 0; \
  do{ \
    if( I ){ \
      s++; \
      fprintf( out, "%s{\n", sp+3-s ); \
    } \
    j = 0; \
    do{ \
      if( J ){ \
        s++; \
        fprintf( out, "%s{\n", sp+3-s ); \
      } \
      k = 0; \
      do{ \
        if( (k % WRAP) == 0 ) fprintf( out, "%s", sp+3-s ); \
        fprintf( out, " "FMT, ##VAL ); \
        if( k < K-1 ) fprintf( out, "," ); \
        if( ((k+1) % WRAP) == 0 ) fprintf( out, "\n" ); \
      }while( ++k < K ); \
      if( J ){ \
        fprintf( out, "%s}\n", sp+3-s ); \
        if( j < J-1 ) fprintf( out, "%s,\n", sp+3-s ); \
        s--; \
      } \
    }while( ++j < J ); \
    if( I ){ \
      fprintf( out, "%s}\n", sp+3-s ); \
      if( i < I-1 ) fprintf( out, "%s,\n", sp+3-s ); \
      s--; \
    } \
  }while( ++i < I ); \
  fprintf( out, "};\n\n" );



  fprintf( out, "\n#ifdef FIXED_POINT\n" );
  fprintf( out, "#undef X\n" );
  fprintf( out, "#define X( x )  {FL2FIX( x, FIXED_POINT_FRACBITS )}\n" );
  fprintf( out, "#endif\n\n" );

  TABLE( "REAL two_to_negative_half_pow[40]", 0, 0, 40, 
	 "X(%.15f)", 4, two_to_negative_half_pow[k] );

  TABLE( "REAL POW2_1[8][2][16]", 8, 2, 16, "X(%.15f)", 4, POW2_1[i][j][k] );

  TABLE( "REAL ca[8]", 0, 0, 8, "X(%.15f)", 4, ca[k] );
  TABLE( "REAL cs[8]", 0, 0, 8, "X(%.15f)", 4, cs[k] );

  TABLE( "REAL win[4][36]", 0, 4, 36, "X(%.15f)", 4, win[j][k] );

  TABLE( "REAL cos_18[9]", 0, 0, 9, "X(%.15f)", 3, cos_18[k] );

  TABLE( "REAL hsec_36[9]", 0, 0, 9, "X(%.15f)", 3, hsec_36[k] );

  TABLE( "REAL hsec_12[3]", 0, 0, 3, "X(%.15f)", 3, hsec_12[k] );

  TABLE( "RATIOS rat_1[16]", 0, 0, 16, "{X(% .15f), X(% .15f)}", 2,
	 rat_1[k].l, rat_1[k].r );
  TABLE( "RATIOS rat_2[2][64]", 0, 2, 64, "{X(%.15f), X(%.15f)}", 2,
	 rat_2[j][k].l, rat_2[j][k].r );


  fprintf( out, "\n#ifdef FIXED_POINT\n" );
  fprintf( out, "#undef X\n" );
  fprintf( out, "#define X( x )  {FL2FIX( x, 19 )}\n" );
  fprintf( out, "#endif\n\n" );

  TABLE( "REAL19 POW2[256]", 0, 0, 256, "X(%.15f)", 4, POW2[k] );


  fprintf( out, "\n#ifdef FIXED_POINT\n" );
  fprintf( out, "#undef X\n" );
  fprintf( out, "#define X( x )  {FL2FIX( x, 13 )}\n" );
  fprintf( out, "#endif\n\n" );

  TABLE( "REAL13 TO_FOUR_THIRDSTABLE[FOURTHIRDSTABLENUMBER]", 
	 0, 0, FOURTHIRDSTABLENUMBER, "X(%17.9f)", 4,
	 TO_FOUR_THIRDSTABLE[k] );

  fprintf( out, "#define cos1_6  %.15f\n", cos1_6 );
  fprintf( out, "#define cos2_6  %.15f\n", cos2_6 );
  fprintf( out, "\n" );

  fclose( out );
  printf( "File " OUT_FNAME " written.\n" );

  return 0;
}

