#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#include "dplib/dpmacs.h"
#include "dplib/eprintf.h"
#include "dplib/dl_list.h"
#include "dplib/fs-helpers.h"
#include "dplib/cmdproc.h"
#include "dplib/malloc-helpers.h"
#include "dplib/numio.h"
#include "dplib/mach.h"

#include "bootldr/partition.h"
#include "dplib/bl_params.h"
#include "dplib/nvmap.h"

name_val_map_t	lfr_flag_map[] =
{
    {"LFR_SIZE_PREFIX", LFR_SIZE_PREFIX},
    {"SP", LFR_SIZE_PREFIX},
    {"LFR_PATCH_BOOTLDR", LFR_PATCH_BOOTLDR},
    {"PB", LFR_PATCH_BOOTLDR},
    {"LFR_KERNEL", LFR_KERNEL},
    {"K", LFR_KERNEL},
    {"LFR_EXPAND", LFR_EXPAND},
    {"E", LFR_EXPAND},
    {"X", LFR_EXPAND},
    {"REST", LFR_EXPAND},
    {"LFR_JFFS2", LFR_JFFS2},
    {"JFFS2", LFR_JFFS2},
    {"JFFS", LFR_JFFS2},
    {"J", LFR_JFFS2},
};

/*
************************************************************************
*
* 
* 
************************************************************************
*/
char*
blp_write_part_tab(
    blp_params_t*   params)
{
    BootldrFlashPartitionTable*	pt;
    FlashRegion*		fp;
    blp_part_t*	item;
    dl_iter_t	iter;
    int	    n;

    /* nada to dooda */
    if (params->num_parts == 0)
	return((char*)params->sec_buf);

    pt = (BootldrFlashPartitionTable*)params->sec_buf;
    pt->magic = BOOTLDR_PARTITION_MAGIC;
    n = pt->npartitions = params->num_parts;
    
    fp = pt->partition;

    for (item = BLP_PARAM_FIRST_PART(params, &iter);
	 item;
	 item = BLP_PARAM_NEXT_PART(&iter)) {
	*fp = item->part;
	++fp;
    }
    return ((char*)fp);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
char*
blp_write_var(
    blp_var_t*	var,
    char*	bufp)
{
    int	    n;
    int	    n_rounded;
    
    
    if (*var->name) {
	/* we have a name so we need to write a set command */
	n = sprintf(bufp, "%s %s %s \"%s\"",
		    BOOTLDR_PREFIX,
		    BOOTLDR_SET_CMD,
		    var->name,
		    var->val);
	++n;			/* count the '\0' */
	n_rounded = long_round(n);
    }
    else {
	/* some unknown thing */
	n = sprintf(bufp, "%s", var->val);
	++n;
	n_rounded = long_round(n);
    }

    /* pad the final word out w/zeroes */
    bufp += n;
    while (n < n_rounded) {
	*bufp++ = '\0';
	n++;
    }
    return (bufp);
}
	
/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_write_vars(
    blp_params_t*   params,
    char*	    bufp)
{
    blp_var_t*	item;
    dl_iter_t	iter;
    
    for (item = BLP_PARAM_FIRST_VAR(params, &iter);
	 item;
	 item = BLP_PARAM_NEXT_VAR(&iter)) {
	bufp = blp_write_var(item, bufp);
    }
}


/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_write_params(
    blp_params_t*   params,
    char*	    outfile)
{
    char*   bufp;
    ssize_t rc;
    size_t  resid;
    int	    out_fd;

    if (!BLP_PARAMS_IS_DIRTY(params))
	return;

    if (*outfile) {
	out_fd = open(outfile, O_WRONLY);
	if (out_fd < 0)
	    eprintf(2, errno, "Cannot open outfile >%s<\n", outfile);
    }
    else
	out_fd = 1;
    
    memset(params->sec_buf, 0xff, params->sec_size);
    bufp = blp_write_part_tab(params);
    blp_write_vars(params, bufp);

    bufp = params->sec_buf;
    resid = params->sec_size;
    
    while (resid) {
	rc = write(out_fd, params->sec_buf, params->sec_size);
	if (rc < 0)
	    eprintf(2, errno, "Write failed.\n");
	else if (rc == 0)
	    break;
	else {
	    resid -= rc;
	    bufp += rc;
	}
    }

    if (*outfile && close(out_fd))
	eprintf(2, errno, "close of outfile >%s< failed.\n", outfile);
}

size_t
blp_sizeof_pe_var(
    blp_var_t*	var)
{
    size_t  namelen;
    size_t  vallen;
    size_t  totlen;

    namelen = strlen(var->name);
    vallen = strlen(var->val);

    if (namelen == 0){
	/*
	 * this is an unknown param.  we've saved it verbatim
	 * in val with an empty name
	 */
	totlen = strlen(var->val);
    }
    else {
	/* this is a bootldr set command.
	 * we will be quoting the val
	 * so + 1 for each '\0' and + 2 for the 2 "s
	 */
	totlen = BOOTLDR_PREFIX_LEN + 1 + BOOTLDR_SET_LEN + 1 +
	    namelen + 1 + vallen + 3;
    }

    /* now, round up.  The params seem to be on word boundaries */
    /* add one for \0 at end of string */
    totlen = (totlen + 1 + 3) & ~3;

    return (totlen);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
unsigned long
blp_flags_to_num(
    char*   flags)
{
    /* for now, just accept numeric values */
    /* later, bitwise or of flag names */

    return (strtoul(flags, NULL, 0));
}

void
blp_display_var(
    blp_var_t*	var)
{
    rprintf("'%s': '%s'\n", var->name, var->val);
}

void
blp_display_part(
    blp_part_t*	part)
{
    rprintf("name>%s<, ", part->part.name);
    rprintf("base: 0x%08lx, ", part->part.base);
    rprintf("size: 0x%08lx, ", part->part.size);
    rprintf("flags: 0x%08x\n", part->part.flags);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_show_vars(
    blp_params_t*   params)
{
    blp_var_t*	item;
    dl_iter_t	iter;
    size_t	plen = 0;
    
    for (item = BLP_PARAM_FIRST_VAR(params, &iter);
	 item;
	 item = BLP_PARAM_NEXT_VAR(&iter)) {
	blp_display_var(item);
	plen += blp_sizeof_pe_var(item);
    }
    rprintf("params len: %d\n", plen);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
blp_var_t*
blp_find_var_by_name(
    blp_params_t*   params,
    char*	    name)
{
    blp_var_t*	item;
    dl_iter_t	iter;
    
    for (item = BLP_PARAM_FIRST_VAR(params, &iter);
	 item;
	 item = BLP_PARAM_NEXT_VAR(&iter)) {
	if (strcmp(name, item->name) == 0)
	    return (item);
    }

    return (NULL);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_show_var(
    blp_params_t*   params,
    char*	    name)
{
    blp_var_t*	var;
    
    if ((var = blp_find_var_by_name(params, name)) != NULL)
	blp_display_var(var);
    else
	rprintf("No such var >%s<\n", name);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
blp_part_t*
blp_find_part_by_name(
    blp_params_t*   params,
    char*	    name)
{
    blp_part_t*	item;
    dl_iter_t	iter;
    
    for (item = BLP_PARAM_FIRST_PART(params, &iter);
	 item;
	 item = BLP_PARAM_NEXT_PART(&iter)) {
	if (strcmp(name, item->part.name) == 0)
	    return (item);
    }

    return (NULL);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_show_parts(
    blp_params_t*   params)
{
    blp_part_t*	item;
    dl_iter_t	iter;
    size_t	pt_len = sizeof(BootldrFlashPartitionTable);
    
    for (item = BLP_PARAM_FIRST_PART(params, &iter);
	 item;
	 item = BLP_PARAM_NEXT_PART(&iter)) {
	blp_display_part(item);
	pt_len += sizeof(item->part);
    }
    rprintf("part table len: %d\n", pt_len);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_show_part(
    blp_params_t*   params,
    char*	    name)
{
    blp_part_t*	pep;

    if ((pep = blp_find_part_by_name(params, name)) == NULL)
	rprintf("No part named >%s<\n", name);
    else
	blp_display_part(pep);
}

void
blp_show_part_or_var(
    blp_params_t*   params,
    char*	    name)
{
    blp_part_t*	pep;
    blp_var_t*	var;
    int		showed = 0;
    
    if ((var = blp_find_var_by_name(params, name)) != NULL) {
	blp_display_var(var);
	showed = 1;
    }
    if ((pep = blp_find_part_by_name(params, name)) != NULL) {
	blp_display_part(pep);
	showed = 1;
    }

    if (!showed)
	rprintf("No part or var named >%s<\n");
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
blp_var_t*
blp_var_define(
    blp_params_t*   params,
    char*	    name,
    char*	    val)
{
    blp_var_t*	var;
    var = blp_find_var_by_name(params, name);
    if (var == NULL) {
	blp_new_pe_var(var);
	BLP_PARAM_ADD_VAR(params, var);
	var->name = safe_zalloc(strlen(name)+1, "var_define(), name");
	strcpy(var->name, name);
    }
    else
	params->var_size -= blp_sizeof_pe_var(var);

    if (var->val)
	safe_free(var->val, "cmd_set(), free val");

    var->val = safe_zalloc(strlen(val)+1, "var_define(), val");
    strcpy(var->val, val);
    params->var_size += blp_sizeof_pe_var(var);
    BLP_PARAMS_MK_DIRTY(params);
    return (var);
}


/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_read_params_sector(
    char*	    infile,
    blp_params_t*   params)
{
    int	    fd;
    int	    rc;
    ssize_t num_read;
    
    if ((fd = open(infile, O_RDONLY)) < 0)
	eprintf(2, errno, "open of >%s< failed\n", infile);

    params->sec_buf = safe_zalloc(params->sec_size, "read_params_sector");
	
    if ((rc = fillbuf_fd(fd, params->sec_buf, params->sec_size, 0,
			 &num_read)) != 0)
	eprintf(2, 0, "read of sector data from file >%s< failed\n", infile);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_set_sector_size_s(
    blp_params_t*   params,
    char*	    s)
{
    params->sec_size = getsize(s, 1);
}

unsigned long
blp_get_sector_size(
    blp_params_t*   params)
{
    return (params->sec_size);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_parse_partition_table(
    blp_params_t*   params)
{
    BootldrFlashPartitionTable*	pt;
    FlashRegion*		fp;

    int	    n;
    int	    i;

    params->num_parts = 0;
    params->part_size = 0;
    params->var_size = 0;
    
    pt = (BootldrFlashPartitionTable*)params->sec_buf;
    if (pt->magic != BOOTLDR_PARTITION_MAGIC) {
	eprintf(0, 0, "No partition table magic number.  "
		"Resetting to empty.\n");
	return;
    }

    n = pt->npartitions;
    fp = pt->partition;
    for (i = 0; i < n; i++) {
	blp_part_define(params, fp->name, fp->base, fp->size, fp->flags);
	++fp;
    }
}


/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_parse_params(
    blp_params_t*   params)
{
    BootldrFlashPartitionTable*	pt;
    char*   rec_start;
    char*   p;
    char*   endp;
    size_t  resid;
    size_t  prefix_len = strlen(BOOTLDR_PREFIX);
    char*   prefix = BOOTLDR_PREFIX;
    char*   argv[100];
    int	    argc;

    /*
     * params are really just a buncha strings.
     * terminated by unwritten flash (0xff) or the end of the sector
     */
    pt = (BootldrFlashPartitionTable*)params->sec_buf;
    if (pt->magic != BOOTLDR_PARTITION_MAGIC)
	p = params->sec_buf;
    else
	p = params->sec_buf + sizeof(*pt) +
	    (sizeof(FlashRegion) * pt->npartitions);

    rprintf("p: 0x%p, delta: 0x%x\n", p, p-params->sec_buf);
    endp = params->sec_buf + params->sec_size;

    while ((p < endp) && (*p != (char)0xff)) {
	/*
	 * items are of the form:
	 * prefix data\0
	 * in our case:
	 * bootldr: bootldr cmd\0
	 * 
	 */
	rec_start = p;
	resid = endp - p;

	if (resid < prefix_len)
	    break;

	if (memcmp(p, prefix, prefix_len) == 0) {
	    /* looks like one of ours */
	    p += prefix_len;
	    if (*p != ' ')
		goto unknown;
	    ++p;
	    resid = endp - p;
	    /* parse param like a cmd */
	    cmdproc_parse(p, &argc, argv, DIM(argv));
	    if (strcmp(argv[0], BOOTLDR_SET_CMD) == 0) {
		/*
		 * 0   1   2   
		 * set var val
		 */
		blp_var_define(params, argv[1], argv[2]);
		rprintf("defining, name>%s<, val>%s<\n", argv[1], argv[2]);
	    }
	    else {
		goto unknown;
	    }
	}
	else {
	    
	  unknown:
	    /* some unknown param, save it verbatim */
	    blp_var_define(params, "", rec_start);
	    rprintf("defining, name>%s<, val>%s<\n", "", rec_start);
	}
	
	/* move to next record */
	p += strlen(p);
	while (p < endp && !*p)
	    p++;
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_free_parts(
    blp_params_t*   params)
{
    blp_part_t*	item;
    dl_iter_t	iter;

    while ((item = BLP_PARAM_FIRST_PART(params, &iter)) != NULL) {
	BLP_PARAM_DEL_PART(params, item);
	safe_free(item, "free_parts()");
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_free_vars(
    blp_params_t*   params)
{
    blp_var_t*	item;
    dl_iter_t	iter;

    while ((item = BLP_PARAM_FIRST_VAR(params, &iter)) != NULL) {
	BLP_PARAM_DEL_VAR(params, item);
	safe_free(item, "free_vars()");
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_free_buf(
    blp_params_t*   params)
{
    if (params->sec_buf) {
	safe_free(params->sec_buf, "free_params(), sec_buf");
	params->sec_buf = 0;
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_free_all_but_parts(
    blp_params_t*   params)
{
    blp_free_vars(params);
    blp_free_buf(params);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_free_all(
    blp_params_t*   params)
{
    blp_free_all_but_parts(params);
    blp_free_parts(params);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
blp_part_t*
blp_part_define(
    blp_params_t*   params,
    char*	    name,
    unsigned long   base,
    unsigned long   size,
    enum LFR_FLAGS flags)
{
    blp_part_t*	pep;

    /* edit existing if already defined */
    if ((pep = blp_find_part_by_name(params, name)) == NULL) {
	blp_new_pe_part(pep);
	BLP_PARAM_ADD_PART(params, pep);
	if (params->num_parts == 0) {
	    /* the first part, account for header */
	    params->part_size = sizeof(BootldrFlashPartitionTable);
	}
	params->part_size += sizeof(pep->part);
	params->num_parts++;
    }
    
    strcpy(pep->part.name, name);
    pep->part.base = base;
    pep->part.size = size;
    pep->part.flags = flags;
    BLP_PARAMS_MK_DIRTY(params);

    return (pep);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
blp_get_partition_table(
    char*	    infile,
    blp_params_t*   params)
{
    
    blp_read_params_sector(infile, params);
    blp_parse_partition_table(params);
    blp_free_buf(params);
}
