#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#include "dpmacs.h"
#include "eprintf.h"
#include "dl_list.h"
#include "fs-helpers.h"
#include "cmdproc.h"

int debug = 0;

#define	skipwhite(p)	while((*p) && isspace(*p)) p++
#define	skipchars(p)	while((*p) && !isspace(*p)) p++

#define	lenchk(word, wp, max)				\
    if ((wp)-(word) >= (max)-1)				\
    eprintf(2, 0, "Word too long >%s<\n", (word));

/*
************************************************************************
*
* cmds can be a bunch of alternatives separated by |s
* 
************************************************************************
*/
int
cmd_match(
    char*   s,
    char*   cmds,
    int	    substring_ok)		/* a simple prefix match is OK */
{
    size_t  slen = strlen(s);
    size_t  cmdlen = strlen(cmds);
    char*   cmdp = cmds;
    char    cmd_term;
    
    while (*cmdp) {
	if (cmdlen < slen) {
	    /* we cannot match, so bail */
	    return (0);
	}

	cmd_term = *(cmdp+slen);
	if (((substring_ok == CMDPROC_MATCH_SUBSTRING) ||
	     (cmd_term == '|' || cmd_term == '\0'))
	    &&
	    memcmp(s, cmdp, slen) == 0) {
#if 0
	    rprintf("s>%s<, cmdp>%s<, slen: %d\n",
		    s, cmdp, slen);
#endif
	    return (1);
	}

	/* no match, skip fwd to next alternative */
	while (*cmdp && *cmdp != '|') {
	    cmdlen--;
	    cmdp++;
	}
	if (*cmdp == '|') {
	    cmdlen--;
	    cmdp++;
	}
    }

    return (0);
}
	
/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmdproc_exec(
    int			    argc,
    char*		    argv[],
    cmdproc_cmd_chunk_t*    cmds)
{
    int	    i;
    
    cmdproc_entry_t*	cmd = cmds->cmd_list;
    size_t		num_cmds = cmds->num_cmds;

    if (argc < 1) {
	eprintf(0, 0, "Not enough args in cmdproc_exec()\n");
	return (-1);
    }
		
    for (i = 0; i < num_cmds; i++, cmd++) {
	if (!cmd->name)
	    return (-1);
	if (debug)
	    printf("argv[0]>%s<, name>%s<\n", argv[0], cmd->name);
	
	if (cmd_match(argv[0], cmd->name, CMDPROC_MATCH_EXACT)) {
	    void*   private;

	    if (cmd->private == CMDPROC_CURRENT_CMD_CHUNK)
		private = cmds;
	    else
		private = cmd->private;
	    if (cmd->func)
		return (cmd->func(argc, argv, cmd->help, private));
	    else {
		eprintf(0, 0, "%s has a NULL function pointer.\n",
			cmd->name);
		return (-1);
	    }
	}
    }

    eprintf(0, 0, "cmd >%s< not in command table.\n", argv[0]);
    
    return (-1);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static int
parsech(
    char**  pp,
    int*    quotedp)
{
    char*   p = *pp;
    int	    ch;
    char*   q;
    
    *quotedp = 0;
    
    if (*p == '\0')
	return (EOF);
    
    if (*p != '\\') 
	ch = *p++;
    else {
	/* got an escape, peek at next char */
	++p;
	*quotedp = 1;
	if (!*p) {
	    eprintf(0, 0, "End of string was escaped.\n");
	    ch = EOF;
	}
	else if ((q = strchr("a\ab\bn\nt\tr\r", *p)) != NULL) {
	    ch = *(q + 1);
	    ++p;
	}
	else if (strchr("01234567", *p)) {
	    /* octal constant */
	    if (debug)
		printf("Octal, looking at>%s<\n", p);
	    ch = strtoul(p, &p, 8) & 0xff;
	}
	else if (*p == 'x' || *p == 'X') {
	    char    buf[3];
	    /* hex constant */
	    if (debug)
		printf("Hex, looking at>%s<\n", p);
	    ++p;
	    if (!isxdigit(*p) || !isxdigit(*(p+1)))
		eprintf(2, 0, "Badly formed hex constant>%s<\n", p-1);
	    buf[0] = *p++;
	    buf[1] = *p++;
	    buf[2] = '\0';
	    ch = strtoul(buf, NULL, 16) & 0xff;
	    if (debug)
		printf("hex val: 0x%x\n", ch);
	}
	else {
	    ch = *p++;
	}
    }
    
    *pp = p;
    
/*      if (debug) */
/*  	printf("ch:%c, (%d)\n", ch, ch); */
    
    return (ch);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
enum {
    ST_BASE = 1,
    ST_END_OF_WORD = 2
};

/*
************************************************************************
*
* 
* 
************************************************************************
*/

static char*
parse_word(
    char**   cmdpp)
{
    char*   p = *cmdpp;
    int	    ch;
    char*   word;
    char*   wp;
    int	    quoted;
    int	    more;
    int	    state = ST_BASE;
    
    skipwhite(p);
    if (!*p) {
	*cmdpp = p;
	return (NULL);
    }
    
    if ((word = (char*)malloc(CMD_PROC_TOK_SIZE)) == NULL)
	eprintf(2, 0, "out of memory\n");
    wp = word;
    for (more = 1; more;) {
	
	ch = parsech(&p, &quoted);
	if (ch == EOF) {
	    if (state != ST_BASE && state != ST_END_OF_WORD)
		eprintf(0, 0, "Open quote detected and closed.\n");
	    break;
	}
	
	if (debug)
	    printf("ch>%c<, (%d), state: %d, quoted: %d\n", ch, ch, state,
		   quoted);
	
	switch (state) {
	    case ST_BASE:
		if (!quoted) {
		    if (strchr("'\"", ch))
			state = ch;
		    else if (isspace(ch)) {
			state = ST_END_OF_WORD;
			goto EOW;
		    }
		    else {
			lenchk(word, wp, CMD_PROC_TOK_SIZE);
			*wp++ = ch;
		    }
		}
		else {
		    lenchk(word, wp, CMD_PROC_TOK_SIZE);
		    *wp++ = ch;
		}
		break;
		
	    case '\'':
		if (!quoted && ch == '\'')
		    state = ST_BASE;
		else {
		    lenchk(word, wp, CMD_PROC_TOK_SIZE);
		    *wp++ = ch;
		}
		break;
		
	    case '\"':
		if (!quoted && ch == '\"')
		    state = ST_BASE;
		else {
		    lenchk(word, wp, CMD_PROC_TOK_SIZE);
		    *wp++ = ch;
		}
		break;
		
	    case ST_END_OF_WORD:
	  EOW:
	  skipwhite(p);
	  more = 0;
	  break;
	}
    }
    
    *wp = '\0';
    *cmdpp = p;
    if (debug)
	printf("returning >%s<, word: %p, wp: %p\n", word, word, wp);
    
    return (word);
}

/*
************************************************************************
*
* parse up to argv_size -1 words out of cmdline.
*             ------------
* Set all other elements of argv[] to NULL.
* command functions can count on at least 1 NULL after their
* last real arg.
************************************************************************
*/
void
cmdproc_parse(
    char*   cmdline,
    int*    argcp,
    char*   argv[],
    size_t  argv_size)
{
    char*   cmdp;
    int	    argc;
    
    argc = 0;
    
    while ((cmdp = parse_word(&cmdline)) != NULL) {
	if (argc >= argv_size - 1)
	    eprintf(2, 0, "Too many words on cmdline >%s<\n", cmdline);
	
	argv[argc++] = cmdp;
    }
    
    *argcp = argc;

    /* make sure all additional elements are null */
    while (argc < argv_size)
	argv[argc++] = NULL;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
cmdproc_free(
    int	    argc,
    char*   argv[])
{
    int	    i;
    
    for (i = 0; i < argc; i++)
	free(argv[i]);
}

static char*	help_argv[] = {
    "",
    "--help",
    NULL,
    NULL,
    NULL
};

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmdproc_do_help(
    cmdproc_cmd_chunk_t*    cmds,
    char*		    cmd_pat, /* pattern to match for help */
    char*		    prefix)
{
    cmdproc_entry_t*	cmd = cmds->cmd_list;
    size_t		num_cmds = cmds->num_cmds;
    int			i;
    int			pat_len;

    if (!prefix)
	prefix = "";

    if (cmd_pat)
	pat_len = strlen(cmd_pat); /* match this */
    else {
	cmd_pat = "";
	pat_len = 0;		/* match everything */
    }

    for (i = 0; i < num_cmds; i++, cmd++) {
	char*	help = cmd->help;
	int	cmp_len;
	int	slen = strlen(cmd->name);

	cmp_len = MIN(pat_len, slen);

	if (!cmp_len || cmd_match(cmd_pat, cmd->name,
				  CMDPROC_MATCH_SUBSTRING)) {
	    if (help == CMDPROC_CALL_FOR_HELP) {
		help_argv[0] = cmd->name;
		/* argv[1] == "--help" */
		help_argv[2] = "";	    /*  cmd_pat; */
		help_argv[3] = prefix;
		cmd->func(4, help_argv, cmd->help, cmd->private);
	    }
	    else {
		if (help == NULL || *help == '\0')
		    help = "No help specified (lazy SOB)";
		printf("%s%s: %s\n", prefix, cmd->name, help);
	    }
	}
    }
    
    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmdproc_help(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   cmd_chunk)
{
    return (cmdproc_do_help((cmdproc_cmd_chunk_t*)cmd_chunk, argv[1], NULL));
}

/*
************************************************************************
*
* Allow commands to have subcommands.
* 
************************************************************************
*/
int
cmdproc_subcommand(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   cmd_chunk)
{
    cmdproc_cmd_chunk_t*  subs = (cmdproc_cmd_chunk_t*)cmd_chunk;
    if (argc > 1 && strcmp(argv[1], "--help") == 0) {
	/*
	 * --help match-pat prefix
	 */
	char*	match_pat;
	char*	prefix;
	char	prefix_buf[256];

	match_pat = (argc > 2) ? argv[2] : "";
	prefix = (argc > 3) ? argv[3] : "";
	sprintf(prefix_buf, "%s%s ", prefix, argv[0]);
	return (cmdproc_do_help(subs, match_pat, prefix_buf));
    }
    else
	return (cmdproc_exec(argc-1, &argv[1], subs));
}
