/*
 * Atmel reading software for the iPAQ H3100/3600
 *
 * Copyright 2001 Compaq Computer Corporation.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Andrew Christian 
 *         <andrew.christian@compaq.com>
 *         October, 2001
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>

#define PDEBUG(format, args...) \
   if ( gDebug ) fprintf(stderr,format, ## args)

int gDebug = 0;

enum params_data {
	PARAMS_DONE = 0,
	PARAMS_TCHAR,
	PARAMS_WORD,
	PARAMS_LONG,
};

struct params_list {
	int   value;
	char *name;
};

struct params_table {
	int                 offset;
	enum params_data    type;
	int                 length;
	char *              name;
	struct params_list *list;
};

static struct params_list product_id[] = {
	{ 2, "Palm" },
	{ -1, "" },
};

static struct params_list page_mode[] = {
	{ 0, "Flash" },
	{ 1, "ROM" },
	{ -1, "" },
};

static struct params_list country_id[] = {
	{ 0, "USA" },
	{ -1, "" },
};

static struct params_table params_table[] = {
	{ 0, PARAMS_TCHAR, 5,  "HM Version" },
	{ 10, PARAMS_TCHAR, 20, "Serial #" },
	{ 50, PARAMS_TCHAR, 10, "Module ID" },
	{ 70, PARAMS_TCHAR, 5,  "Product Revision" },
	{ 80, PARAMS_WORD,  0,  "Product ID", product_id },
	{ 82, PARAMS_WORD,  0,  "Frame Rate" },
	{ 84, PARAMS_WORD,  0,  "Page Mode", page_mode },
	{ 86, PARAMS_WORD,  0,  "Country ID", country_id },
	{ 88, PARAMS_WORD,  0,  "Is Color Display" },
	{ 90, PARAMS_WORD,  0,  "ROM Size" },
	{ 92, PARAMS_WORD,  0,  "RAM Size" },
	{ 94, PARAMS_WORD,  0,  "Horizontal pixels" },
	{ 96, PARAMS_WORD,  0,  "Vertical pixels" },

	{ 128, PARAMS_LONG, 0,  "MDCNFG" },
	{ 132, PARAMS_LONG, 0,  "MDCAS00" },
	{ 136, PARAMS_LONG, 0,  "MDCAS01" },
	{ 140, PARAMS_LONG, 0,  "MDCAS02" },
	{ 144, PARAMS_LONG, 0,  "MSC0" },
	{ 148, PARAMS_LONG, 0,  "MSC1" },
	{ 152, PARAMS_LONG, 0,  "MECR" },
	{ 156, PARAMS_LONG, 0,  "MDREFR" },
	{ 160, PARAMS_LONG, 0,  "MDCAS20" },
	{ 164, PARAMS_LONG, 0,  "MDCAS21" },
	{ 168, PARAMS_LONG, 0,  "MDCAS22" },
	{ 172, PARAMS_LONG, 0,  "MSC2" },
	{ 176, PARAMS_LONG, 0,  "SMCNFG" },
	{ 180, PARAMS_LONG, 0,  "RTC RTTR" },
	{ 184, PARAMS_LONG, 0,  "Internal flash start" },
	{ 188, PARAMS_LONG, 0,  "Internal flash size" },
	{ 192, PARAMS_LONG, 0,  "Engineering no." },

	{ 252, PARAMS_WORD,  0,  "Valid Tag" },
	{ 254, PARAMS_WORD,  0,  "Checksum" },

	{ 0, PARAMS_DONE, 0, NULL  }
};

/* This should do better checking to make sure we don't fall of the edge */

int  fd;

static void read_atmel_flash( char *buf, int total_len )
{
	int   len;

	while ( (len = read( fd, buf, total_len )) > 0) {
		buf        += len;
		total_len  -= len;
	}

	if ( len < 0 ) {
		perror("Unable to read file\n");
		exit(1);
	}
}

#define BASIC_FORMAT "%20s : "
#define ATMEL_FLASH_LEN 256

static unsigned int read_word( char *p )
{
	unsigned int v = *p++;
	v <<= 8;
	return v | *p;
}

static char * lookup_params( struct params_list *list, int value )
{
	if ( !list )
		return NULL;

	while ( list->value != -1 && list->value != value )
		list++;
	return list->name;
}

static char * parse_eeprom( char *p, int index, int max, struct params_table *table )
{
	char data[ATMEL_FLASH_LEN]; 
	struct params_table *t = table;

	/* Suck in the data */
	read_atmel_flash( data, ATMEL_FLASH_LEN );

	for ( ; t->type != PARAMS_DONE && index < max ; t++ ) {
		char *data_ptr = &data[ t->offset ];

		switch ( t->type ) {
		case PARAMS_WORD: {
			unsigned short v = read_word( data_ptr );
			char *param;

			PDEBUG("Parsing word %s %x %x\n", t->name, *data_ptr, *(data_ptr+1));
			param = lookup_params(t->list,v);

			if ( param )
				p += sprintf(p, BASIC_FORMAT "%d (%s)\n", t->name, v, param );
			else
				p += sprintf(p, BASIC_FORMAT "%d\n", t->name, v);
			break;
		}
		case PARAMS_LONG: {
			unsigned int v = read_word( data_ptr );
			unsigned int w = read_word( data_ptr + 2 );

			PDEBUG("Parsing u32 %s\n", t->name);
			p += sprintf(p, BASIC_FORMAT "0x%08x\n", t->name, w << 16 | v);
			break;
		}
		case PARAMS_TCHAR: {
			int i;
			char *d = data_ptr;
			PDEBUG("Parsing unicode string %s\n", t->name );

			p += sprintf(p, BASIC_FORMAT, t->name );
			for ( i = 0 ; i < t->length ; i++ ) {
				if ( *++d != 0 )
					p += sprintf(p, "%c", *d );
				d++;
			}
			p += sprintf(p,"\n");
			break;
		}
		case PARAMS_DONE:
			PDEBUG("All done\n");
			return p;
		}
	}
	return p;
}
	
void usage( void )
{
	fprintf(stderr,"Usage: atmelparse [opts] [filename...]\n\n");
	fprintf(stderr,"Options\n");
	fprintf(stderr,"  -d        Debug\n");
	exit(1);
}

int get_opts( int argc, char **argv )
{
	int done = 0;
	
	while ( !done ) {
		int c;

		c = getopt( argc, argv, "d");
		switch (c) {
		case 'd':
			gDebug = 1;
			break;
		case EOF:
			done = 1;
			break;
		default:
			usage();
			break;
		}
	}
	return optind;
}

void dofile( void )
{
	char *p;
	char data[10000];

	p = parse_eeprom( data, 0, 10000, params_table );
	*p = 0;
	
	printf("%s", data);
	printf("\n");
}

int main( int argc, char **argv )
{
	int  index = get_opts( argc, argv );

	if ( index == argc ) {
		fd = 0;
		PDEBUG("Parsing stdin\n");
		dofile();
	}
	else {
		for ( ; index < argc ; index++ ) {
			PDEBUG("Parsing %s\n", argv[index]);
			fd = open( argv[index], O_RDONLY );
			if ( !fd ) {
				fprintf(stderr,"Unable to open %s for reading\n", argv[index]);
				exit(1);
			}
			dofile();
		}
	}
	return 1;
}

