/*
 * EEPROM programming software for iPAQ H3600
 *
 * Copyright 2001 Compaq Computer Corporation.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Andrew Christian 
 *         <andrew.christian@compaq.com>
 *         22 June 2001
 */

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>

typedef char           data8;
typedef short          data16;
typedef long           data32;
typedef unsigned char  udata8;
typedef unsigned short udata16;
typedef unsigned long  udata32;

struct id_data {
	udata32  length;
	udata8   version;
	udata16  vendor_id;
	udata16  product_id;
	char     description[255];
	udata8   type;
	udata8   interrupt_enable;
	udata8   external_battery;
	udata8   initial_power;
	udata8   suspend_power;
	udata16  reset_width_time;
	udata16  wait_stable_time;
	udata16  bootstrap_address;
	udata16  oem_information_address;
	
	/* These are only present in version == 2 */
	udata16  pcmcia_socket_0;
	udata16  pcmcia_socket_1;
	udata8   system_handle_opt_on;
};

static struct id_data default_id_data = {
	length:                  0,
	version:                 2,
	vendor_id:               0x1125,
	product_id:              0x0100,
	description:             "Compaq Mercury Backpaq",
	type:                    3,
	interrupt_enable:	 0x4e,
	external_battery:	 'Y',
	initial_power:	         0,
	suspend_power:	         0,
	reset_width_time:	 0,
	wait_stable_time:	 0,
	bootstrap_address:	 0,
	oem_information_address: 0,
	pcmcia_socket_0:	 0x03,
	pcmcia_socket_1:	 0x03,
	system_handle_opt_on:	 1
};

struct control_data {
	udata32  vendor_id;
	udata32  driver_id;
	udata32  memory_start;
	udata32  memory_stop;
};

static struct control_data default_control_data = {
	vendor_id:    0x22222222,
	driver_id:    0x33333333,
	memory_start: 0xae000000,
	memory_stop:  0xae1fffff
};

struct driver_entry {
	udata32  vendor_id;
	udata32  driver_id;
	char     file_name[255];
	char     display_name[255];
	char     stream_prefix[3];
};

struct driver_table {
	int      count;
	struct driver_entry entry[10];
};

struct blob {
	udata32  length;
	udata8   raw_data[255];
};

struct mercury_data {
	udata8   major_rev;     // Major hardware revision
	udata8   minor_rev;     // Minor hardware revision
	udata8   fpga_version; 
	udata8   camera;
	udata16  accel_offset_x;
	udata16  accel_scale_x;
	udata16  accel_offset_y;
	udata16  accel_scale_y;
	udata32  serial_number;
	udata32  flash_start;
	udata32  flash_length;
	udata32  sysctl_start;
};

static struct mercury_data default_mercury_data = {
	major_rev:	2,     // Major hardware revision
	minor_rev:	4,     // Minor hardware revision
	serial_number:	0,
	fpga_version:	1, 
	camera:	        2,     // Release color camera 
	accel_offset_x:	0x8000,
	accel_scale_x:	0x8000,
	accel_offset_y:	0x8000,
	accel_scale_y:	0x8000,
	flash_start:	0x0000000,
	flash_length:	0x2000000,
	sysctl_start:	0x2000000
};

struct eeprom_data {
	struct id_data      id;
	struct control_data control;
	struct driver_table driver;
	struct blob         config;
	struct blob         bootstrap;
	struct blob         oem;
};

/* Fill out the id_data structure; return bytes write */

int write_a_string( char *dest, char *source )
{
	int count = 1;
	while ( (*dest++ = *source++) != 0 )
		count++;
	
	return count;
}

int write_a_prefix( char *dest, char *source )
{
	int i;
	for ( i = 0 ; i < 3 ; i++ )
		*dest++ = *source++;
	return 3;
}

#define WRITE_8(data)       *((udata8*)s)=(data); s++;
#define WRITE_16(data)      WRITE_8((data)&0xff); WRITE_8((data)>>8)
#define WRITE_32(data)      WRITE_16((data)&0xffff); WRITE_16((data)>>16)
#define WRITE_STR(data)     s+=write_a_string(s,data)
#define WRITE_PREFIX(data)  s+=write_a_prefix(s,data)

int write_id( struct id_data *id, udata8 *dest, int max_len )
{
	udata8 *s = dest;

	WRITE_8( 0xaa );
	WRITE_32( id->length );
	WRITE_8( id->version );
	WRITE_16( id->vendor_id );
	WRITE_16( id->product_id );
	WRITE_STR( id->description );
	WRITE_8( id->type );
	WRITE_8( id->interrupt_enable );
	WRITE_8( id->external_battery );
	WRITE_8( id->initial_power );
	WRITE_8( id->suspend_power );
	WRITE_16( id->reset_width_time );
	WRITE_16( id->wait_stable_time );
	WRITE_16( id->bootstrap_address );
	WRITE_16( id->oem_information_address );
	
	if ( id->version == 2 ) {
		WRITE_16( id->pcmcia_socket_0 );
		WRITE_16( id->pcmcia_socket_1 );
		WRITE_8( id->system_handle_opt_on );
	}

	WRITE_32( 0x0f0f0f0f );
	return (s - dest);
}

int write_control( struct control_data *ctl, udata8 *dest, int max_len )
{
	udata8 * s = dest;
	WRITE_8( 0xbb );

	WRITE_32( ctl->vendor_id );
	WRITE_32( ctl->driver_id );
	WRITE_32( ctl->memory_start );
	WRITE_32( ctl->memory_stop );

	WRITE_32( 0x0f0f0f0f );
	return (s - dest);
}


int write_driver( struct driver_table *tbl, udata8 *dest, int max_len )
{
	udata8 * s = dest;
	int i;

	for ( i = 0 ; i < tbl->count ; i++ ) {
		struct driver_entry *e = &tbl->entry[tbl->count++];

		WRITE_32( e->vendor_id );
		WRITE_32( e->driver_id );
		WRITE_STR( e->file_name );
		WRITE_STR( e->display_name );
		WRITE_PREFIX( e->stream_prefix );
		WRITE_8( 0x03 );
	}

	WRITE_32( 0xffffffff );
	WRITE_32( 0x0f0f0f0f );
	return (s - dest);
}

int write_blob( struct blob *blob, udata8 *dest, int max_len )
{
	udata8 *s = dest;
	int i;

	WRITE_32( blob->length );
	for ( i = 0 ; i < blob->length ; i++ ) {
		WRITE_8( blob->raw_data[i] );
	}
	WRITE_32( 0x0f0f0f0f );
	return (s-dest);
}

void mercury_to_blob( struct mercury_data *mercury, struct blob *oem )
{
	udata8 *s = oem->raw_data;
		
	WRITE_8(mercury->major_rev);
	WRITE_8(mercury->minor_rev);
	WRITE_8(mercury->fpga_version);
	WRITE_8(mercury->camera);
	WRITE_16(mercury->accel_offset_x);
	WRITE_16(mercury->accel_scale_x);
	WRITE_16(mercury->accel_offset_y);
	WRITE_16(mercury->accel_scale_y);
	WRITE_32(mercury->serial_number);
	WRITE_32(mercury->flash_start);
	WRITE_32(mercury->flash_length);
	WRITE_32(mercury->sysctl_start);

	oem->length = s - oem->raw_data;
}


int fill_buffer( struct eeprom_data *data, udata8 *dest, int max_len )
{
	udata8 *p = dest;
	udata8 *s;
	int result;
	int total_size;
	int oem_offset;
	int bootstrap_offset;
	int control_offset;

	if ( (result = write_id( &data->id, p, max_len )) < 0 )
		return result;
	p += result;

	control_offset = p - dest;
	if ( (result = write_control( &data->control, p, max_len )) < 0 )
		return result;
	p += result;

	if ( (result = write_driver( &data->driver, p, max_len )) < 0 )
		return result;
	p += result;

	if ((result = write_blob( &data->config, p, max_len )) < 0)
		return result;
	p += result;

	bootstrap_offset = p - dest;
	if ((result = write_blob( &data->bootstrap, p, max_len )) < 0)
		return result;
	p += result;

	oem_offset = p - dest;
	if ((result = write_blob( &data->oem, p, max_len )) < 0)
		return result;
	p += result;

	// Update some fields in the id block
	total_size = ( p - dest );

	s = dest + 1; // Set to the length of data byte
	WRITE_32( total_size );

	s = dest + control_offset - 13;
	WRITE_16( bootstrap_offset );
	WRITE_16( oem_offset );

	return total_size;
}


int flush_buffer( int fd, udata8 *buf, int len )
{
	while ( len > 0 ) {
		int result = write( fd, buf, len );
		if ( result < 0 ) {
			perror("unable to write buffer");
			return result;
		}
		len -= result;
	}
	return 0;
}


	
void usage( void )
{
	fprintf(stderr,"Usage: eecreate [OPTION] SERIAL\n");
	fprintf(stderr,"Creates a Mercury Backpaq eeprom file and\n");
	fprintf(stderr,"sends it to stdout.\n\n");
	fprintf(stderr,"  -M, --major=REV      major revision number (default=2)\n");
	fprintf(stderr,"  -m, --minor=REV      minor revision number (default=2)\n");
	fprintf(stderr,"  -l, --landscape      camera in landscape mode\n");
	fprintf(stderr,"  -p, --portrait       camera in portrait mode (default)\n");
	fprintf(stderr,"  -c, --camera=MODEL   camera model (default=0)\n");
	fprintf(stderr,"  -f, --fpga=VER       set fpga version (default=1)\n");
	fprintf(stderr,"  -x, --xoffset=VAL    acclerometer x-axis duty cycle ratio (default=0.5)\n");
	fprintf(stderr,"  -y, --yoffset=VAL    acclerometer y-axis duty cycle ratio (default=0.5)\n");
	fprintf(stderr,"  -X, --xscale=VAL     acclerometer x-axis scaling (default=1.0)\n");
	fprintf(stderr,"  -Y, --yscale=VAL     acclerometer y-axis scaling (default=1.0)\n");
	fprintf(stderr,"      --fstart=ADDR    flash memory start address (default=0x0000000)\n");
	fprintf(stderr,"      --flength=VAL    flash memory length (default=0x2000000)\n");
	fprintf(stderr,"      --sysctl=VAL     sysctl address (default=0x2000000)\n");
	fprintf(stderr,"  -h, --help           display this help and exit\n");
	exit(1);
}

int get_opts( int argc, char **argv, struct mercury_data *data )
{
	int done = 0;
	
	while ( !done ) {
		int option_index = 0;
		int c;

		static struct option long_options[] = {
			{ "major", required_argument, NULL, 'M' },
			{ "minor", required_argument, NULL, 'm' },
			{ "landscape", no_argument, NULL, 'l' },
			{ "portrait", no_argument, NULL, 'p' },
			{ "camera", required_argument, NULL, 'c' },
			{ "fpga", required_argument, NULL, 'f' },
			{ "xoffset", required_argument, NULL, 'x' },
			{ "yoffset", required_argument, NULL, 'y' },
			{ "xscale", required_argument, NULL, 'X' },
			{ "yscale", required_argument, NULL, 'Y' },
			{ "fstart", required_argument, NULL, 1 },
			{ "flength", required_argument, NULL, 2 },
			{ "sysctl", required_argument, NULL, 3 },
			{ "help", no_argument, NULL, 'h' },
			{ 0, 0, 0, 0 }
		};

		c = getopt_long( argc, argv, "M:m:lpc:f:x:y:X:Y:h", 
				 long_options, &option_index);
		switch (c) {
		case EOF:
			done = 1;
			break;
		case 'M':
			data->major_rev = strtol(optarg,NULL,0);
			break;
		case 'm':
			data->minor_rev = strtol(optarg,NULL,0);
			break;
		case 'l':
			data->camera |= 0x80;
			break;
		case 'p':
			data->camera &= ~(0x80);
			break;
		case 'c':
			data->camera &= 0x80;  /* Clear the upper bits */
			data->camera |= strtol(optarg,NULL,0);
			break;
		case 'f':
			fprintf(stderr,"Handling %s\n", optarg);
			data->fpga_version = strtol(optarg,NULL,0);
			break;
		case 'x':
			data->accel_offset_x = 65536 * atof( optarg );
			break;
		case 'y':
			data->accel_offset_y = 65536 * atof( optarg );
			break;
		case 'X':
			data->accel_scale_x = 32768 * atof(optarg);
			break;
		case 'Y':
			data->accel_scale_y = 32768 * atof(optarg);
			break;
		case 1:
			data->flash_start = strtol(optarg,NULL,0);
			break;
		case 2:
			data->flash_length = strtol(optarg,NULL,0);
			break;
		case 3:
			data->sysctl_start = strtol(optarg,NULL,0);
			break;
		case 'h':
			usage();
			break;
		default:
			fprintf(stderr,"Unrecognized option %s\n\n", argv[optind-1] );
			usage();
			break;
		}
	}
	return optind;
}



int main( int argc, char **argv )
{
	udata8 buf[2000];
	int buf_len;
	struct eeprom_data form;
	
	int index = get_opts( argc, argv, &default_mercury_data );
	if ( index != argc - 1 ) {
		fprintf(stderr,"Must specify a serial number\n\n");
		usage();
	}
	default_mercury_data.serial_number = atoi(argv[index]);

	memcpy( &form.id,        &default_id_data,      sizeof(default_id_data));
	memcpy( &form.control,   &default_control_data, sizeof(default_id_data));
	form.driver.count  = 0;
	form.config.length = 0;
	form.bootstrap.length = 0;

	mercury_to_blob( &default_mercury_data, &form.oem );

	// Copy the structure to the data buffer
	buf_len = fill_buffer(&form, buf, 2000 );
	if ( buf_len < 0 )
		exit(1);

	flush_buffer(1,buf,buf_len);
	return 1;
}
