/*
 * EEPROM programming software for iPAQ H3600
 *
 * Copyright 2001 Compaq Computer Corporation.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Andrew Christian 
 *         <andrew.christian@compaq.com>
 *         22 June 2001
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <stdio.h>
#include <unistd.h>

#define PDEBUG(format, args...) \
   if ( gDebug ) fprintf(stderr,format, ## args)

int gDebug = 0;

enum sleeve_eeprom_datatype {
	EEPROM_DONE = 0,
	EEPROM_BYTE = 1,      // One byte
	EEPROM_SHORT = 2,     // Two bytes
	EEPROM_TRIPLE = 3,    // Three bytes
	EEPROM_LONG = 4,      // Four bytes
	EEPROM_STRING,        // Null terminated string
	EEPROM_SECTION,       // Section head
	EEPROM_START_REPEAT, 
	EEPROM_END_REPEAT,
	EEPROM_OFFSET,
	EEPROM_RAW, 
	EEPROM_SWITCH,        // Switch based on a number
	EEPROM_CASE,
	EEPROM_ENDCASE
};

struct sleeve_eeprom_table {
	enum sleeve_eeprom_datatype type;
	char *                      name;
	char *                      format;
	int                         magic_number;
};

static struct sleeve_eeprom_table eeprom_table[] = {
	{ EEPROM_SECTION, "ID Information", "===============", 0 },
	{ EEPROM_BYTE,   "Start of ID", "0x%x", 0xaa },
	{ EEPROM_LONG,   "Length of data", "%d", 0 },
	{ EEPROM_SWITCH, "Version Indicator", "0x%x", 1 },
	{ EEPROM_SHORT,  "Vendor ID", "0x%04x", 0 },
	{ EEPROM_SHORT,  "ID Number", "0x%04x", 0 },
	{ EEPROM_STRING, "Text description", "%s", 0 },
	{ EEPROM_BYTE,   "(Type)", "%d", 0 },
	{ EEPROM_BYTE,   "(Interrupt enable)", "%x", 0 },
	{ EEPROM_BYTE,   "Extended battery?", "%c", 0 },
	{ EEPROM_BYTE,   "(Initial power state)", "%x", 0 },
	{ EEPROM_BYTE,   "(Final power state)", "%x", 0 },
	{ EEPROM_SHORT,  "Reset width time", "0x%04x", 0 },
	{ EEPROM_SHORT,  "Wait stable time", "0x%04x", 0 },
	{ EEPROM_SHORT,  "Bootstrap address", "0x%04x", 0 },
	{ EEPROM_SHORT,  "OEM Information address", "0x%04x", 0 },

	{ EEPROM_CASE,   "", "", 2 },
       	{ EEPROM_SHORT,  "PCMCIA Socket 0", "0x%04x", 0 },
	{ EEPROM_SHORT,  "PCMCIA Socket 1", "0x%04x", 0 },
	{ EEPROM_BYTE,   "System handle OPT_ON", "0x%02x", 0 },
	{ EEPROM_ENDCASE,"", "", 0 },

	{ EEPROM_LONG,   "Terminator", "0x%08x", 0x0f0f0f0f },

	{ EEPROM_SECTION, "Control Information", "===================", 0 },
	{ EEPROM_BYTE,   "Start of Control", "0x%x", 0xbb },
	{ EEPROM_LONG,   "Vendor ID", "0x%08x", 0 },
	{ EEPROM_LONG,   "Driver ID", "0x%08x", 0 },
	{ EEPROM_LONG,   "Memory - Start of flash", "0x%08x", 0 },
	{ EEPROM_LONG,   "Memory - End of flash", "0x%08x", 0 },
	{ EEPROM_LONG,   "Terminator", "0x%08x", 0x0f0f0f0f },

	{ EEPROM_SECTION, "Driver Table", "============", 0 },
	{ EEPROM_START_REPEAT, "Vendor ID", "0x%08x", 0xffffffff },
	{ EEPROM_LONG,    "Driver ID", "0x%08x", 0 },
	{ EEPROM_STRING,  "Driver file name", "%s", 0 },
	{ EEPROM_STRING,  "Display name", "%s", 0 },
	{ EEPROM_TRIPLE,  "Stream prefix", "%s", 0 },
	{ EEPROM_BYTE,    "Record terminator", "%x", 0x03 },
	{ EEPROM_END_REPEAT, "\n", "", 0},
	{ EEPROM_LONG,    "Terminator", "0x%08x", 0x0f0f0f0f },

	{ EEPROM_SECTION, "Configuration", "=============", 0 },
	{ EEPROM_OFFSET,  "Size of Configuration", "0x%08x", 0 },
	{ EEPROM_RAW,     "Binary data", "%02x", 8 },
	{ EEPROM_LONG,    "Terminator", "0x%08x", 0x0f0f0f0f },

	{ EEPROM_SECTION, "Bootstrap", "=========", 0 },
	{ EEPROM_OFFSET,  "Size of Bootstrap", "0x%08x", 0 },
	{ EEPROM_RAW,     "Binary data", "%02x", 8 },
	{ EEPROM_LONG,    "Terminator", "0x%08x", 0x0f0f0f0f },

	{ EEPROM_SECTION, "OEM Area", "=========", 0 },
	{ EEPROM_OFFSET,  "Size of OEM Area", "0x%08x", 0 },
	{ EEPROM_RAW,     "Binary data", "%02x", 8 },
	{ EEPROM_LONG,    "Terminator", "0x%08x", 0x0f0f0f0f },

	{ EEPROM_DONE, NULL, NULL, 0 }
};

static struct sleeve_eeprom_table backpaq_oem_table[] = {
	{ EEPROM_SECTION, "Backpaq specific", "================", 0 },
	{ EEPROM_BYTE,    "Major revision", "0x%02x", 0 },
	{ EEPROM_BYTE,    "Minor revision", "0x%02x", 0 },
	{ EEPROM_BYTE,    "FPGA version", "0x%02x", 0 },
	{ EEPROM_BYTE,    "Camera", "0x%02x", 0 },
	{ EEPROM_SHORT,    "Accel offset x", "0x%02x", 0 },
	{ EEPROM_SHORT,    "Accel scale x", "0x%02x", 0 },
	{ EEPROM_SHORT,    "Accel offset y", "0x%02x", 0 },
	{ EEPROM_SHORT,    "Accel scale y", "0x%02x", 0 },
	{ EEPROM_LONG,    "Serial number", "%d", 0 },
	{ EEPROM_LONG,    "Flash start", "0x%08x", 0 },
	{ EEPROM_LONG,    "Flash length", "0x%08x", 0 },
	{ EEPROM_LONG,    "Sysctl start", "0x%08x", 0 },
};

/* This should do better checking to make sure we don't fall of the edge */

int  fd;
char gData[10000];
int  gDataLen = 0;

static void read_eeprom( void )
{
	char *buf = gData;
	int   max_len = 10000;
	int   len;

	while ( (len = read( fd, buf, max_len )) > 0) {
		buf      += len;
		gDataLen += len;
		max_len  -= len;
	}

	if ( len < 0 ) {
		perror("Unable to read file\n");
		exit(1);
	}
}

static void h3600_spi_read( int offset, char *store, int len )
{
	int i;
	char *buf = gData + offset;
	for ( i = 0 ; i < len ; i++ )
		*store++ = *buf++;
}

#define BASIC_FORMAT "%23s : "

static char * parse_eeprom( char *p, int index, int max, struct sleeve_eeprom_table *table )
{
	struct sleeve_eeprom_table *t = table;
	struct sleeve_eeprom_table *repeat = t;
	int data;
	int len;
	char triple[4];
	int offset = 0;
	int switch_value = 0;
	int i;

	for ( ; t->type != EEPROM_DONE && index < max ; t++ ) {
		switch ( t->type ) {
		case EEPROM_SECTION:
			PDEBUG("Parsing section %s, offset 0x%x\n", t->name, index );
			p += sprintf(p,"\n%s\n%s\n", t->name, t->format);
			break;
		case EEPROM_START_REPEAT:
			PDEBUG("Parsing start repeat %s\n", t->name );
			repeat = t;
			data = 0;
			h3600_spi_read( index, (char *) &data, 4 );
			index += 4;   // Increment to the next location
			p += sprintf(p, BASIC_FORMAT, t->name);
			p += sprintf(p, t->format, data);
			p += sprintf(p,"\n");
			if ( data == t->magic_number ) {
				while ( t->type != EEPROM_END_REPEAT )
					t++;
			}
			break;
		case EEPROM_END_REPEAT:
			PDEBUG("Parsing end repeat %s\n", t->name );
			p += sprintf(p,"%s\n%s\n", t->name, t->format);
			t = repeat;
			break;
		case EEPROM_OFFSET:
			PDEBUG("Parsing offset %s\n", t->name );
			data = 0;
			h3600_spi_read( index, (char *) &data, 4 );
			index += 4;   // Increment to the next location
			p += sprintf(p, BASIC_FORMAT, t->name);
			p += sprintf(p, t->format, data);
			p += sprintf(p,"\n");
			offset = data;
			break;
		case EEPROM_RAW:
			PDEBUG("Parsing raw data %s, offset 0x%x, bytes %d\n", t->name, index,offset );
			data = 0;
			for ( i = 0 ; i < offset ; i++ ) {
				if ( i % t->magic_number == 0 )
					p += sprintf(p, BASIC_FORMAT, t->name);
				data = 0;
				h3600_spi_read(index++,(char *)&data,1);
				p += sprintf(p, t->format, data );
				if ( (i+1) % t->magic_number == 0 )
					p += sprintf(p,"\n");
			}
			if ( i % t->magic_number != 0 )
				p += sprintf(p,"\n");
			break;
		case EEPROM_SWITCH:
			PDEBUG("Parsing switch %s\n", t->name );
			data = 0;
			h3600_spi_read( index, (char *) &data, t->magic_number );
			index += t->magic_number;   // Increment to the next location
			p += sprintf(p, BASIC_FORMAT, t->name);
			p += sprintf(p, t->format, data);
			p += sprintf(p,"\n");
			switch_value = data;
			break;
		case EEPROM_CASE:
			PDEBUG("Parsing case %s\n", t->name );
			if ( switch_value != t->magic_number ) {
				while ( t->type != EEPROM_ENDCASE )
					t++;
			}
			break;
		case EEPROM_ENDCASE:
			break;
		case EEPROM_BYTE:
		case EEPROM_SHORT:
		case EEPROM_LONG:
			PDEBUG("Parsing data %s, size %d\n", t->name, t->type );
			data = 0;
			h3600_spi_read( index, (char *) &data, t->type );
			index += t->type;   // Increment to the next location
			if ( t->magic_number != 0 && data != t->magic_number ) {
				p += sprintf(p, "%s -- bad magic number read %x should be %x\n",
					     t->name, data, t->magic_number);
				return p;
			}
			p += sprintf(p, BASIC_FORMAT, t->name);
			p += sprintf(p, t->format, data);
			p += sprintf(p,"\n");
			break;
		case EEPROM_TRIPLE:
			PDEBUG("Parsing triple %s\n", t->name );
			h3600_spi_read( index, (char *) &triple, 3 );
			triple[3] = 0;
			index += 3;
			p += sprintf(p, BASIC_FORMAT, t->name);
			p += sprintf(p, t->format, triple);
			p += sprintf(p,"\n");
			break;
		case EEPROM_STRING: 
			PDEBUG("Parsing string %s\n", t->name );
			len = 0;
			p += sprintf(p, BASIC_FORMAT "\"", t->name );
			do { 
				data = 0;
				h3600_spi_read( index++, (char *) &data, 1 );
				if (data)
					p += sprintf(p, "%c", data );
				len++;
			} while ( data );
			p += sprintf(p,"\"\n");
			break;
		case EEPROM_DONE:
			PDEBUG("All done\n");
			return p;
		}
	}
	return p;
}
	
void usage( void )
{
	fprintf(stderr,"Usage: eeparse [opts] [filename...]\n\n");
	fprintf(stderr,"Options\n");
	fprintf(stderr,"  -d        Debug\n");
	exit(1);
}

int get_opts( int argc, char **argv )
{
	int done = 0;
	
	while ( !done ) {
		int c;

		c = getopt( argc, argv, "d");
		switch (c) {
		case 'd':
			gDebug = 1;
			break;
		case EOF:
			done = 1;
			break;
		default:
			usage();
			break;
		}
	}
	return optind;
}

void dofile( void )
{
	char *p;
	char data[10000];
	short vendor, device;

	PDEBUG("Reading data\n");
	read_eeprom();

	PDEBUG("Parsing data\n");
	p = parse_eeprom( data, 0, 10000, eeprom_table );
	*p = 0;
	
	printf("%s", data);
	printf("\n");

	// Magic for backpaq 
	
	h3600_spi_read( 6, (char *)&vendor, 2 );
	h3600_spi_read( 8, (char *)&device, 2 );

	if ( vendor == 0x1125 && device == 0x100 ) {
		char d;
		short offset = 10;
		do {
			h3600_spi_read( offset++, &d, 1 );
		} while ( d != 0 );
		offset += 11;
		h3600_spi_read( offset, (char *)&offset, 2 );
		offset += 4; // Skip the length
		p = parse_eeprom( data, offset, 10000, backpaq_oem_table );
		*p = 0;
		printf("%s", data);
		printf("\n");
	}
}

int main( int argc, char **argv )
{
	int  index = get_opts( argc, argv );

	if ( index == argc ) {
		fd = 0;
		PDEBUG("Parsing stdin\n");
		dofile();
	}
	else {
		for ( ; index < argc ; index++ ) {
			PDEBUG("Parsing %s\n", argv[index]);
			fd = open( argv[index], O_RDONLY );
			if ( !fd ) {
				fprintf(stderr,"Unable to open %s for reading\n", argv[index]);
				exit(1);
			}
			dofile();
		}
	}
	return 1;
}

