/*
 * EEPROM programming software for iPAQ H3600
 *
 * Copyright 2001 Compaq Computer Corporation.
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 * Author: Andrew Christian 
 *         <andrew.christian@compaq.com>
 *         22 June 2001
 */

#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <sys/ioctl.h>

#include <stdio.h>
#include <unistd.h>
#include <getopt.h>
#include <stdlib.h>

#include <linux/h3600_ts.h>

int fill_buffer( int fd, char *buf, int max_len )
{
	int bytes_read = 0;
	
	while ( bytes_read < max_len ) {
		int result = read( fd, buf, max_len - bytes_read );
		if ( result == 0 )
			return bytes_read;
		
		if ( result < 0 ) {
			perror("Unable to read from file");
			return result;
		}
		buf += result;
		bytes_read += result;
	}
	fprintf(stderr,"Ran out of buffer space\n");
	return -1;
}


	
void usage( void )
{
	fprintf(stderr,"Usage: eeprogram [FILENAME]\n");
	fprintf(stderr,"Programs the EEPROM on a iPAQ sleeve.  If FILENAME is omitted,\n");
	fprintf(stderr,"program data is read from stdin.\n\n");
	exit(1);
}

#define BUFFER_SIZE  2048

#define CHUNK 4

int main( int argc, char **argv )
{
	int  fd = 0;
	char buf[BUFFER_SIZE];
	int  len;
	int i, j;


	if ( argc != 1 && argc != 2 )
		usage();

	if ( argc == 2 ) {
		fd = open( argv[1], O_RDONLY );
		if ( fd < 0 ) {
			perror("Unable to open file for reading\n");
			fprintf(stderr,"File %s for reading\n", argv[1]);
			return 1;
		}
	}

	len = fill_buffer( fd, buf, BUFFER_SIZE );
	close(fd);

	/* Run a few sanity checks */
	if ( len <= 0 ) {
		printf("Bad file size\n");
		return 1;
	}

	if ( buf[0] != 0xaa ) {
		printf("The magic byte is incorrect\n");
		return 1;
	}

	if ( buf[1] + buf[2] * 256 != len ) {
		printf("The size of the data doesn't agree internal\n");
		return 1;
	}

	/* Program the device */
	if ( (fd = open("/dev/h3600_ts", O_WRONLY)) < 0 ) {
		perror("Unable to open the touchscreen driver");
		return 1;
	}

	for ( i = 0 ; i < len ; i += CHUNK ) {
		struct h3600_spi_write_request request;
		struct h3600_spi_read_request read_request;
		request.len = ( len - i > CHUNK ? CHUNK : len - i );
		request.addr = i;
		memcpy( request.buff, &buf[i], request.len );
		printf("Writing to EEPROM address 0x%04x: ", request.addr);
		for ( j = 0 ; j < request.len ; j++ )
			printf(" %02x", request.buff[j]);
		       
		if ( ioctl( fd, WRITE_SPI, &request ) ) {
			perror("Failed to write!\n");
			return 1;
		}

		read_request.len = request.len;
		read_request.addr = request.addr;
		if ( ioctl( fd, READ_SPI, &read_request ) ) {
			perror("Failed to read!\n");
			return 1;
		}
		read_request.len = request.len;
		read_request.addr = request.addr;
		if ( ioctl( fd, READ_SPI, &read_request ) ) {
			perror("Failed to read!\n");
			return 1;
		}
		printf("{"); 
		for ( j = 0 ; j < read_request.len ; j++ )
			printf(" %02x", read_request.buff[j]);
		printf("{"); 
		printf("\n");

		sleep(0);
	}

	close(fd);
	printf("Wrote %d bytes\n", len );
	return 1;
}

