// gcardinfo.cc
//   1999 Atsushi Tagami
//
// Modified for use on the Compaq iPAQ
//   2001 Andrew Christian

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif // HAVE_CONFIG_H

#include <fstream>
#include <strstream>
#include <string>

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <signal.h>
#include <errno.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif // HAVE_UNISTD_H

#ifdef HAVE_FCNTL_H
#include <fcntl.h>
#endif // HAVE_FCNTL_H

#ifndef TIME_WITH_SYS_TIME
#include <time.h>
#endif // TIME_WITH_SYS_TIME

#ifdef HAVE_GETTIMEOFDAY
#include <sys/time.h>
#endif // HAVE_GETTIMEOFDAY

#ifdef HAVE_PCMCIA_CONFIG
#include <pcmcia/config.h>
#endif // HAVE_PCMCIA_CONFIG

#include <pcmcia/version.h>
#include <pcmcia/cs_types.h>
#include <pcmcia/cs.h>
#include <pcmcia/cistpl.h>
#include <pcmcia/ds.h>

#include <gtk/gtk.h>

//=== prototypes ===
void HandleDeviceMessage(gpointer, gint, GdkInputCondition);
void HandlePressSlotList(GtkWidget*, GdkEventButton*, gpointer);
void RequestMessage(GtkObject*, gpointer);
void RestartCardManager(GtkObject*, gpointer);
void ShowLogWindow(GtkObject*, gpointer);
void DestroyLogWindow(GtkObject*, gpointer);
gint CloseLogWindow(GtkObject*, gpointer);
void AppendLog(gchar *inMessage);
void Quit(GtkObject*, gpointer);
void SelectRow(GtkCList*, gint, gint, GdkEventButton* ,gpointer);
gint LookupDevice(gchar *inName);
gint OpenDeviceNode(dev_t inDevice);
gint UpdateCardInfo(gpointer data);
void UpdateCardName(void);
void ShowCardInfo(void);
void CreateWindow(gint inSlotNum);
void CreateLogWindow(void);
void CreatePopupMenu(void);

//=== constants ===
const gchar* const kDeviceProc   = "/proc/devices";
const gchar* const kPidFilePath  = "/var/run/cardmgr.pid";
const gchar* const kStabFilePath = "/var/run/stab";
const gchar* const kErrorPrefix  = "** ERROR **: ";
const gint kDefaultWindowWidth  = 240;
const gint kDefaultWindowHeight = 130;
const gint kLogWindowWidth  = 240;
const gint kLogWindowHeight = 200;
const gint kMaxSlot = 8;

const guint32 kPopupTimeoutDelay = 250;  // Millisecond delay until menu pops up

//=== structs ===
typedef enum {
  stateEmpty   = 0,
  statePresent = 1,
  stateReady   = 2,
  stateBusy    = 3,
  stateSuspend = 4
} State;

static const gchar* const gStateTag[] = {
  "EMPTY",
  "PRESENT",
  "READY",
  "BUSY",
  "SUSPEND"
};

typedef struct SSocketInfo {
  int    fd;
  gint   empty;
} SSocketInfo;

typedef struct SEventTag {
  event_t event;
  char *name;
} SEventTag;

static SEventTag kEventTag[] = {
  { CS_EVENT_CARD_INSERTION,    "card insertion" },
  { CS_EVENT_CARD_REMOVAL,      "card removal" },
  { CS_EVENT_RESET_PHYSICAL,    "prepare for reset" },
  { CS_EVENT_CARD_RESET,        "card reset successful" },
  { CS_EVENT_RESET_COMPLETE,    "reset request complete" },
  { CS_EVENT_EJECTION_REQUEST,  "user eject request" },
  { CS_EVENT_INSERTION_REQUEST, "user insert request" },
  { CS_EVENT_PM_SUSPEND,        "suspend card" },
  { CS_EVENT_PM_RESUME,         "resume card" },
  { CS_EVENT_REQUEST_ATTENTION, "request attention" },
};

const gint kNumTags  = sizeof(kEventTag)/sizeof(SEventTag);


typedef struct SRequestTag {
  gulong        request;
  const gchar*  label;
} SRequestTag;


static SRequestTag kRequestTag[] = {
  { DS_RESET_CARD,   "reset"},
  { DS_SUSPEND_CARD, "suspend"},
  { DS_RESUME_CARD,  "resume"},
  { DS_EJECT_CARD,   "eject" },
  { DS_INSERT_CARD,  "insert" }
};

enum {
  requestReset   = 0,
  requestSuspend = 1,
  requestResume  = 2,
  requestEject   = 3,
  requestInsert  = 4,
};

const gint kRequestTags  = sizeof(kRequestTag)/sizeof(SRequestTag);

typedef struct SMenuItem {
  const gchar*  label;
  GtkSignalFunc function;
} SMenuItem;

static SMenuItem kMenuItem[] = {
  { "restart cardmgr", GTK_SIGNAL_FUNC(RestartCardManager)},
  { "show log window", GTK_SIGNAL_FUNC(ShowLogWindow)},
  { "quit"           , GTK_SIGNAL_FUNC(Quit)}
};

const gint kMenuItems  = sizeof(kMenuItem)/sizeof(SMenuItem);

//=== global variable ===
static SSocketInfo gSlot[kMaxSlot];
static gint  gActiveSlot;
static State gActiveSlotState;
static GtkWidget* gSlotList = (GtkWidget*)NULL;
static GtkWidget* gInfoLabel[6];
static GtkWidget* gSlotPopup = (GtkWidget*)NULL;
static GtkWidget* gLogWindow = (GtkWidget*)NULL;
static GtkWidget* gLogText   = (GtkWidget*)NULL;
static GtkWidget* gSlotMenuItem[kRequestTags];

static gint gTimeCounter = 0;
static gint gTimmer = 0;
static gint    gPopupTimer  = 0;
static guint32 gPopupTimeAt = 0;


//---------- HandleDeviceMessage
// Handle message from device. when message is comming, this is called.
//
//  data      : slot number
//  souce     : device file id  (unused)
//  condition : input condition (unused) 
void
HandleDeviceMessage(gpointer data, gint source, GdkInputCondition condition)
{
  gint number = (gint)data;
  gint event;
  gint ret = read(gSlot[number].fd, &event, 4);
  for(gint i=0; i<kNumTags; i++){
    if(kEventTag[i].event == event){
      time_t seconds;
#ifdef HAVE_GETTIMEOFDAY
      struct timeval tv;
      gettimeofday(&tv, NULL);
      seconds = tv.tv_sec;
#else
      time(&seconds);
#endif // HAVE_GETTIMEOFDAY
      struct tm *theTime = localtime(&seconds);
      gchar* log 
	= g_strdup_printf("%02d:%02d:%02d slot:%d %s\n", 
			  theTime->tm_hour,theTime->tm_min,theTime->tm_sec,
			  number, kEventTag[i].name);
      AppendLog(log);
      break;
    }
  }

  // set timeout is 500ms (to update card name)
  // we need to wait update /var/run/stab
  gTimeCounter = 1;
  if(gTimmer == 0){
    gTimmer = gtk_timeout_add(500, UpdateCardInfo, (gpointer)NULL);
  }
}

//---------- HandlePressSlotList
// when press button for at least 1000 msecs, show popup menu

gint
DoPopupTimeout(gpointer data)
{
  if( gSlotPopup ){
    // all request menu items disable
    for(gint i=0; i<kRequestTags; i++){
      gtk_widget_set_sensitive(gSlotMenuItem[i], FALSE);
    }
    // enable menu item 
    switch(gActiveSlotState){
    case statePresent:
      gtk_widget_set_sensitive(gSlotMenuItem[requestInsert], TRUE);
      break;
    case stateReady:
    case stateBusy:
      gtk_widget_set_sensitive(gSlotMenuItem[requestReset], TRUE);
      gtk_widget_set_sensitive(gSlotMenuItem[requestSuspend], TRUE);
      gtk_widget_set_sensitive(gSlotMenuItem[requestEject], TRUE);
      break;
    case stateSuspend:
      gtk_widget_set_sensitive(gSlotMenuItem[requestReset], TRUE);
      gtk_widget_set_sensitive(gSlotMenuItem[requestResume], TRUE);
      gtk_widget_set_sensitive(gSlotMenuItem[requestEject], TRUE);
      break;
    }
    gtk_menu_popup(GTK_MENU(gSlotPopup), NULL, NULL, NULL, NULL,
		   GPOINTER_TO_UINT(data), gPopupTimeAt);
  }
  gPopupTimer = 0;
  return FALSE;   /* Do not continue to be called */
}


void
HandlePressSlotList(GtkWidget* clist, GdkEventButton *inEvent,
		    gpointer data)
{
  switch (inEvent->type) {
  case GDK_BUTTON_PRESS:
    if ( gPopupTimer )
      gtk_timeout_remove( gPopupTimer );

    gPopupTimeAt = kPopupTimeoutDelay + inEvent->time;
    gPopupTimer = gtk_timeout_add( kPopupTimeoutDelay, 
				     DoPopupTimeout, 
				     GUINT_TO_POINTER(inEvent->button) );
    break;
  case GDK_BUTTON_RELEASE:
    if ( gPopupTimer ) {
      gtk_timeout_remove( gPopupTimer );
      gPopupTimer = 0;
    }
    break;
  }
}


//---------- RequestMessage
// request message to active slot
//
//  inObject : slotlist [unused]
//  inData   : request
void
RequestMessage(GtkObject* /* inObject */, gpointer inData)
{
  gulong request = (gulong)inData;
  gint ret = ioctl(gSlot[gActiveSlot].fd, request);
  
  if(ret != 0){
    cerr << kErrorPrefix << "ioctl() operation failed: " 
	 << strerror(errno) << endl;
  }
}


//---------- RestartCardManager
// restart cardmgr
void
RestartCardManager(GtkObject* /* inObject */, gpointer /*inData*/)
{
  pid_t pid;
    
  ifstream ifs(kPidFilePath);
  if(!ifs){
    cerr << kErrorPrefix << "Could not open pidfile" << endl;
    return;
  }
  
  if(!(ifs >> pid)){
    cerr << kErrorPrefix << "Could not read pidfile" << endl;
    return;
  }
  
  if (kill(pid, SIGHUP) != 0){
    cerr << "Could not signal cardmgr" << endl;
  }
}


//---------- ShowLogWindow
// show log window
void
ShowLogWindow(GtkObject* /* inObject */, gpointer /*inData*/)
{
  if(!gLogWindow){
    CreateLogWindow();
  }
  if(gLogWindow){
    gtk_widget_show_all(gLogWindow);
  }    
}


//---------- DestroyLogWindow
// distroy log window
void
DestroyLogWindow(GtkObject* inObject, gpointer /*inData*/)
{
  gLogWindow = (GtkWidget*)NULL;
  gLogText = (GtkWidget*)NULL;
}


//---------- CloseLogWindow
// close log window
gint
CloseLogWindow(GtkObject* inObject, gpointer /*inData*/)
{
  if(gLogWindow){
    gtk_widget_hide_all(gLogWindow);
  }
  return TRUE;
}


//---------- AppendLog
// append message to log window
void
AppendLog(gchar *inMessage)
{
  if(!gLogText){
    CreateLogWindow();
  }
  if(gLogText){
    gtk_text_insert(GTK_TEXT(gLogText), NULL, NULL, NULL, 
		    inMessage, strlen(inMessage));
  }
}


//---------- Quit
// quit application
void
Quit(GtkObject* /* inObject */, gpointer /*inData*/)
{
  gtk_main_quit();
}


//---------- SelectRow
// change active slot
// callback when select slot row
void
SelectRow(GtkCList*       /*clist*/,
	  gint            inSlotNumber,
	  gint            /*column*/,
	  GdkEventButton* /*event*/,
	  gpointer        /*data*/)
{
  gActiveSlot = inSlotNumber;
  ShowCardInfo();
}


//---------- LookupDevice
// lookup device id by name
//
//  inName  : device name
// [return] : devive number or -1(if error)
gint
LookupDevice(gchar *inName)
{
  ifstream ifs(kDeviceProc);
  if(!ifs){
    cerr << kErrorPrefix << kDeviceProc << " cannot open" << endl;
    return -1;
  }
  
  string buffer;
  while(getline(ifs, buffer)){
    if(buffer.length() > 0){
      gint number;
      string name;
      istrstream iss(buffer.data(), buffer.length());

      if(iss >> number >> name){
	if(name == inName){
	  return number;
	}
      }
    }
  }
  cerr << kErrorPrefix << "No pcmcia driver in " << kDeviceProc << endl;
  return -1;
}


//--------- OpenDeviceNode
// open pcmcia slot
//  inDevice : device node number
//  [return] : file pointer or -1(if error)
gint
OpenDeviceNode(dev_t inDevice)
{
  gchar* filename;
  if((filename = tmpnam(NULL)) == NULL){
    return -1;
  }
  if(mknod(filename, S_IFCHR|S_IREAD, inDevice) != 0){
    return -1;
  }
  gint fp;
  if((fp = open(filename, O_RDONLY)) < 0){
    unlink(filename);
    return -1;
  }
  if(unlink(filename) != 0){
    close(fp);
    return -1;
  }
  return fp;
}



//-------- UpdateCardInfo
// update card name and info
// this function called by gtk_timeout
//  500ms timmer start when recive card event (stop after 2 times timeout)
gint
UpdateCardInfo(gpointer data)
{
  UpdateCardName();
  ShowCardInfo();
  
  if(--gTimeCounter < 0){
    gTimmer = 0;
    return FALSE;
  }else{
    return TRUE;
  }
}


//-------- UpdateCardName
// get card name from /var/run/stab
//
void
UpdateCardName(void)
{
  string buffer;
  gint   slotNumber;
  gint   indent = 99;
  
  ifstream ifs(kStabFilePath);
  lock(ifs);
  while(ifs >> buffer){
    // Socket <SocketNumber>: CardName
    if(buffer == "Socket"){
      ifs >> slotNumber;
      ifs >> buffer; // skip ':'
      getline(ifs, buffer);
      
      gSlot[slotNumber].empty = ((buffer == "empty") ? TRUE : FALSE);
      
      gchar* name = g_strndup(buffer.data(), buffer.length());
      gtk_clist_set_text(GTK_CLIST(gSlotList), slotNumber, 1, name);
      g_free(name);
      // clear device
      gtk_clist_set_text(GTK_CLIST(gSlotList), slotNumber, 2, "");
      
      indent = 0;
    }else{
      // 0 <type> <module> 0 <device>
      indent++;
      if(indent == 5 && slotNumber > -1){ // pickup <device>
	gchar* device = g_strndup(buffer.data(), buffer.length());
	gtk_clist_set_text(GTK_CLIST(gSlotList), slotNumber, 2, device);
	g_free(device);
      }
    }
  }
  unlock(ifs);
}


//---------- ShowCardInfo
// show card info
void
ShowCardInfo(void)
{
  if (gInfoLabel[0] == (GtkWidget*)NULL) return;  
  
  cs_status_t status;
  status.Function = 0;
  ioctl(gSlot[gActiveSlot].fd, DS_GET_STATUS, &status);
  State state = stateEmpty;
  if(gSlot[gActiveSlot].empty == TRUE){
    if(status.CardState & CS_EVENT_CARD_DETECT){
      state = statePresent;
    }
  }else if(status.CardState & CS_EVENT_PM_SUSPEND){
    state = stateSuspend;
  }else if(status.CardState & CS_EVENT_READY_CHANGE){
    state = stateBusy;
  }
  gActiveSlotState = state;

  config_info_t config;
  memset(&config, 0, sizeof(config));
  gint ret = ioctl(gSlot[gActiveSlot].fd, DS_GET_CONFIGURATION_INFO, &config);
  gchar *irq = NULL, *io = NULL;
  if(config.Attributes & CONF_VALID_CLIENT){
    if(config.AssignedIRQ != 0){
      irq = g_strdup_printf("%d", config.AssignedIRQ);
    }
    if(config.NumPorts1 > 0){
      ioaddr_t stop = config.BasePort1 + config.NumPorts1;
      if (config.NumPorts2 > 0) {
	if (stop == config.BasePort2){
	  io = g_strdup_printf("%#x-%#x",
			       config.BasePort1, stop+config.NumPorts2-1);
	}else{
	  io = g_strdup_printf("%#x-%#x, %#x-%#x", 
		  config.BasePort1, stop-1,
		  config.BasePort2, config.BasePort2+config.NumPorts2-1);
	}
      }else{
	io = g_strdup_printf("%#x-%#x", config.BasePort1, stop-1);
      }
    }
  }
  gtk_label_set_text(GTK_LABEL(gInfoLabel[0]), (irq ? irq : ""));
  if (irq) g_free(irq);
  
  gtk_label_set_text(GTK_LABEL(gInfoLabel[1]), (io ? io : ""));
  if (io) g_free(io);
  
  gchar* vcc = g_strdup_printf("%1.1f", ((gfloat)config.Vcc) / 10);
  gtk_label_set_text(GTK_LABEL(gInfoLabel[2]), vcc);
  g_free(vcc);
  
  gchar* vpp = g_strdup_printf("%1.1f", ((gfloat)config.Vpp1) / 10);
  gtk_label_set_text(GTK_LABEL(gInfoLabel[3]), vpp);
  g_free(vpp);

  gtk_label_set_text(GTK_LABEL(gInfoLabel[4]), gStateTag[state]);

  gtk_label_set_text(GTK_LABEL(gInfoLabel[5]), 
		   (status.CardState & CS_EVENT_WRITE_PROTECT ? "on" : "off"));
}


//---------- CreateWindow
// create gtk window
//
//  inSlotNum : number of slot(socket)
void
CreateWindow(gint inSlotNum)
{
  // create main window
  GtkWidget* window = gtk_window_new(GTK_WINDOW_TOPLEVEL);
  gtk_window_set_default_size(GTK_WINDOW(window), 
			      kDefaultWindowWidth, kDefaultWindowHeight);

  gtk_widget_set_name(window, "gcardinfo");
  gtk_window_set_title(GTK_WINDOW (window), "gcardinfo");
  gtk_signal_connect(GTK_OBJECT (window), "destroy",
		     GTK_SIGNAL_FUNC (gtk_main_quit), NULL);
  gtk_container_border_width (GTK_CONTAINER (window), 0);
  
  GtkWidget *vbox = gtk_vbox_new (FALSE, 0);
  gtk_container_add(GTK_CONTAINER(window), vbox);
  gtk_container_border_width (GTK_CONTAINER (vbox), 2);
  
  // create slot list
  const gchar* listTitle[] = { "#", "name", "device"};
  GtkWidget *list = gtk_clist_new_with_titles(3, (gchar**)listTitle);

  //gtk_clist_column_titles_hide(GTK_CLIST(list));
  gtk_clist_set_column_justification(GTK_CLIST(list), 0, GTK_JUSTIFY_LEFT);
  gtk_clist_set_column_justification(GTK_CLIST(list), 1, GTK_JUSTIFY_LEFT);
  gtk_clist_set_column_justification(GTK_CLIST(list), 2, GTK_JUSTIFY_LEFT);
  gtk_clist_set_selection_mode (GTK_CLIST(list), GTK_SELECTION_BROWSE);
  gtk_clist_set_column_width (GTK_CLIST (list), 0, 20);
  gtk_clist_set_column_width (GTK_CLIST (list), 1, 160);
  gtk_clist_set_column_width (GTK_CLIST (list), 2, 20);
  gtk_signal_connect(GTK_OBJECT(list), "select_row",
		     GTK_SIGNAL_FUNC(SelectRow), NULL);
  gtk_signal_connect_after(GTK_OBJECT(list), "button_press_event",
			   GTK_SIGNAL_FUNC(HandlePressSlotList), NULL);
  gtk_signal_connect_after(GTK_OBJECT(list), "button_release_event",
			   GTK_SIGNAL_FUNC(HandlePressSlotList), NULL);

  // append slot entry
  gchar number[2];
  gchar* entry[3] = {number , "empty", ""};
  for(gint i=0; i<inSlotNum; i++){
    sprintf(number, "%d", i);
    gtk_clist_append(GTK_CLIST(list), entry);
  }
  gSlotList = list;  
  
  //--
  GtkWidget *scroll
    = gtk_scrolled_window_new((GtkAdjustment*)NULL, (GtkAdjustment*)NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scroll),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll), list);
  gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);
  
  // info frame
  GtkWidget* frame = gtk_frame_new("info");
  gtk_box_pack_start(GTK_BOX(vbox), frame, FALSE, FALSE, 0);
  GtkWidget* table = gtk_table_new(4, 3, FALSE);
  
  GtkWidget *label, *content;
  gchar* infoTitle[] = { "IRQ", "I/O", "Vcc", "Vpp", "STATE", "Protect" };
  gint index = 0;
  for(gint j=0; j<3; j++){
    for(gint i=0; i<3; i+=2){
      label = gtk_label_new(infoTitle[index]);
      //gtk_label_set_justify(GTK_LABEL(label), GTK_JUSTIFY_RIGHT);
      gtk_misc_set_alignment(GTK_MISC(label), 1.0, 0.5);
      gtk_table_attach_defaults(GTK_TABLE(table), label, i, i+1, j, j+1);
      content = gtk_label_new("on");
      gtk_table_attach_defaults(GTK_TABLE(table), content, i+1, i+2, j, j+1);
      gInfoLabel[index] = content;
      index++;
    }
  }
  gtk_container_add(GTK_CONTAINER(frame), table);  
  
  // show window
  if (!GTK_WIDGET_VISIBLE (window)){
    gtk_widget_show_all(window);
  }else{
    gtk_widget_destroy(window);
  }
}


//---------- CreateLogWindow
// create log window
void
CreateLogWindow(void)
{
  // create main window
  gLogWindow = gtk_window_new(GTK_WINDOW_DIALOG);
  gtk_window_set_default_size(GTK_WINDOW(gLogWindow), 
			      kLogWindowWidth, kLogWindowHeight);
  gtk_widget_set_name(gLogWindow, "logwindow");
  gtk_window_set_title(GTK_WINDOW(gLogWindow), "gcardinfo log");
  gtk_signal_connect(GTK_OBJECT(gLogWindow), "delete_event",
		     GTK_SIGNAL_FUNC(CloseLogWindow), NULL);
  gtk_signal_connect(GTK_OBJECT(gLogWindow), "destroy",
		     GTK_SIGNAL_FUNC(DestroyLogWindow), NULL);
  gtk_container_border_width(GTK_CONTAINER(gLogWindow), 0);
  
  GtkWidget *vbox = gtk_vbox_new(FALSE, 0);
  gtk_container_add(GTK_CONTAINER(gLogWindow), vbox);
  gtk_container_border_width (GTK_CONTAINER(vbox), 2);
  
  // create slot list
  gLogText = gtk_text_new(NULL, NULL);
  gtk_text_set_editable(GTK_TEXT(gLogText), FALSE);
  GtkWidget *scroll
    = gtk_scrolled_window_new((GtkAdjustment*)NULL, (GtkAdjustment*)NULL);
  gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scroll),
				  GTK_POLICY_AUTOMATIC, 
				  GTK_POLICY_AUTOMATIC);
  gtk_container_add(GTK_CONTAINER(scroll), gLogText);
  gtk_box_pack_start(GTK_BOX(vbox), scroll, TRUE, TRUE, 0);
  
  // separator
  GtkWidget* separator = gtk_hseparator_new ();
  gtk_box_pack_start (GTK_BOX(vbox), separator, FALSE, TRUE, 5);
  // close button
  GtkWidget* button = gtk_button_new_with_label("Close");
  gtk_box_pack_start(GTK_BOX(vbox), button, FALSE, FALSE, 0);
  gtk_signal_connect(GTK_OBJECT(button), "clicked",
		     GTK_SIGNAL_FUNC(CloseLogWindow), NULL);
  // hide window
  gtk_widget_hide_all(gLogWindow);
}


//-------- CreatePopupMenu
void
CreatePopupMenu(void)
{
  gSlotPopup = gtk_menu_new();
  
  for(gint i=0; i<kRequestTags; i++){
    gSlotMenuItem[i] = gtk_menu_item_new_with_label(kRequestTag[i].label);
    gtk_menu_append(GTK_MENU(gSlotPopup), gSlotMenuItem[i]);
    gtk_signal_connect(GTK_OBJECT(gSlotMenuItem[i]), "activate",
		       GTK_SIGNAL_FUNC(RequestMessage),
		       gpointer(kRequestTag[i].request));
    gtk_widget_show(gSlotMenuItem[i]);
  }

  GtkWidget* item;
  
  // Separator
  item = gtk_menu_item_new();
  gtk_widget_set_sensitive(item, FALSE);
  gtk_menu_append(GTK_MENU(gSlotPopup), item);
  gtk_widget_show(item);

  // common menu
  for(gint i=0; i<kMenuItems; i++){
    item = gtk_menu_item_new_with_label(kMenuItem[i].label);
    gtk_menu_append(GTK_MENU(gSlotPopup), item);
    gtk_signal_connect(GTK_OBJECT(item), "activate",
		       GTK_SIGNAL_FUNC(kMenuItem[i].function),
		       NULL);
    gtk_widget_show(item);
  }
}


//-------- main
int
main(int argc, char *argv[])
{
  char name[12];

  if (geteuid() != 0) {
    cerr << kErrorPrefix << "must be setuid root" << endl;
    exit(-1);
  }

  // lookup pcmcia device  
  gint major = LookupDevice("pcmcia");
  if(major < 0){
    exit(-1);
  }
  // open pcmcia slot(socket)
  gint numSlot = 0;
  for(gint i = 0; i<kMaxSlot; i++, numSlot++){
    if((gSlot[i].fd = OpenDeviceNode((major << 8) + i)) < 0){
      break;
    }
    gSlot[i].empty = TRUE;
  }
  if(numSlot == 0){
    cerr << kErrorPrefix << "No slots found" << endl;
    exit(-1);
  }

  // get pcmcia service info
  servinfo_t serv;  
  if(ioctl(gSlot[0].fd, DS_GET_CARD_SERVICES_INFO, &serv) == 0){
    if (serv.Revision != CS_RELEASE_CODE) {
      cerr << kErrorPrefix << "Card Services release does not match!" << endl;
      cerr << kErrorPrefix << "Card services : " << serv.Revision 
	   << " ... I'm looking for " << CS_RELEASE_CODE << endl;
	      
//      exit(-1);
    }
  }else{
    cerr << kErrorPrefix << "Could not get CS revision info!" << endl;
    exit(-1);
  }

  // Switch back to real user privileges, to be safe 
  setuid(getuid());

  gtk_set_locale();       // set locale
  gtk_init(&argc, &argv); // init gtk+

  // watch slot socket
  for(gint i=0; i<numSlot; i++){
    gdk_input_add(gSlot[i].fd, GDK_INPUT_READ,
		  (GdkInputFunction)HandleDeviceMessage, (gpointer)i);
  }
  gInfoLabel[0] = (GtkWidget*)NULL;
  
  CreateWindow(numSlot);
  CreatePopupMenu();
  CreateLogWindow();
  
  gtk_clist_select_row(GTK_CLIST(gSlotList), 0, 0);

  gtk_main();
}
