/*
 * Itsy Utility Routines
 * Generic Hash Table Code
 *
 * Copyright (c) Compaq Computer Corporation, 1998, 1999
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 */

/*
 * Itsy Utility Routines
 * Generic Hash Table Code
 *
 * Modified version of DCPI "util/htable.c", Revision 1.19.
 * Original code written by Sanjay Ghemawat and Carl Waldspurger.
 *
 * $Log: htable.c,v $
 * Revision 1.1.1.1  2000/11/28 16:46:34  kerr
 * Compaq Profiling Infrastructure
 *
 * Revision 1.1  1999/08/13  23:21:08  caw
 * Initial revision (from DCPI sources).
 *
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include <util/util.h>

/*
 * support both macro and functional forms for hash, key equality
 *
 */

#ifdef	ELEMENT_HASH
#define	ELEMENT_HASH_PRIM(h, elt)	(ELEMENT_HASH(elt))		
#else	/* ELEMENT_HASH */
#define	ELEMENT_HASH_PRIM(h, elt)	(h->hash(elt))
#endif	/* ELEMENT_HASH */

#ifdef	ELEMENT_KEY_EQUAL
#define	ELEMENT_KEY_EQUAL_PRIM(h, a, b)	(ELEMENT_KEY_EQUAL(a, b))
#else	/* ELEMENT_KEY_EQUAL */
#define	ELEMENT_KEY_EQUAL_PRIM(h, a, b)	(h->key_equal(a, b))
#endif	/* ELEMENT_KEY_EQUAL */

/* Entries in "table" can have two special states:
	EMPTY			Entry is unused
	DELETED			Entry has been deleted
*/

static unsigned int hash_index(unsigned int x, unsigned int slot_bits) {
    /* The multiplier used below is "int(PHI * 2^k)" where PHI denotes
       the golden ratio.  This multiplier value is suggested as a good
       one by Knuth.
    
       The resulting hash value is converted into the range
       "[0..2^p-1]" by extracting the top "p" bits of the bottom k bits
       of the hash value.  An easy way to do this is to right shift the
       result by "k - p". */
    x = (x * 2654435769U);
#ifdef _X86_
    /*
     * On Intel processors, when slot_bits is zero, we try
     * to shift x right by 32 bits.  Intel's shift instruction
     * masks the # of bits to shift at 5 bits, causing it not
     * to shift the value at all.
     */
    if (slot_bits == 0) return 0;
#endif
    return (x >> (32U - slot_bits));
}

/* effects	Ignores current state of "h", and initializes it
		to an appropriate size for "n" elements. */
static void init(HTABLE* h, unsigned int n) {
    /* Search for the correct power of 2 */
    unsigned int min;
    unsigned int power, bits;
    unsigned int i;

    /* The table size is set-up so that at most 1/3 of the tbl is full */
    min = 3 * n;

    power = 1;
    bits = 0;
    while (power <= min) {
	power <<= 1;
	bits++;
    }

    h->tsize = power;
    h->slot_bits = bits;
    h->count = 0;
    h->delcount = 0;
    h->enlarge_size = power - power/3;
    if (h->enlarge_size <= n) h->enlarge_size = n+1;
    if (h->enlarge_size > power) h->enlarge_size = power;

    h->table = ALLOC_ARRAY(ELEMENT, power);
    for (i = 0; i < power; i++)
      ELEMENT_MARK_EMPTY(h, &h->table[i]);
}

#ifdef	NDEBUG
#define HSTAT(x)	0
#else	/* NDEBUG */
#define HSTAT(x)	x
#endif	/* NDEBUG */

/* requires	"key" is not in the table.
   		"h" does not have any deleted entries.
    		"h" is large enough to have valid rep after
		"h" is inserted. (i.e. "tsize > (count + 1)")
   effects	Inserts "key" into "tbl". */
static void unsafe_insert(HTABLE* h, ELEMENT key) {
    ELEMENT* table = h->table;
    unsigned int index = hash_index(ELEMENT_HASH_PRIM(h, key), h->slot_bits);

    HSTAT(h->searches++);
    HSTAT(h->probes++);
    if (!ELEMENT_IS_EMPTY(h, table[index])) {
	unsigned int mask = h->tsize - 1;
	do {
	    index = (index + 1) & mask;
	    HSTAT(h->probes++);
	} while (!ELEMENT_IS_EMPTY(h, table[index]));
    }

    table[index] = key;
    h->count++;
}

/* effects	If "key" exists in the table, return its index,
		else return the index of an unused location suitable
		for inserting the key. */
static unsigned int HFUN(find_index)(HTABLE* h, ELEMENT key) {
  ELEMENT* table = h->table;
  unsigned int index = hash_index(ELEMENT_HASH_PRIM(h, key), h->slot_bits);
  unsigned int mask;

  HSTAT(h->searches++);
  mask = h->tsize - 1;
  while (1) {
    HSTAT(h->probes++);
    if (ELEMENT_IS_EMPTY(h, table[index])) {
      return index;
    } else if (ELEMENT_IS_DELETED(h, table[index])) {
      /* This is a suitable candidate for insertion, but we need to
	 check that the element is not in the table. */
      unsigned int del = index;
      while (1) {
	HSTAT(h->probes++);
	index = (index + 1) & mask;
	if (ELEMENT_IS_EMPTY(h, table[index])) {
	  return del;
	} else if ((!ELEMENT_IS_DELETED(h, table[index])) &&
		   (ELEMENT_KEY_EQUAL_PRIM(h, table[index], key))) {
	  return index;
	}
      }
    } else if (ELEMENT_KEY_EQUAL_PRIM(h, table[index], key)) {
      return index;
    }
    index = (index + 1) & mask;
  }
}

/* effects	Resizes the table to appropriate size for "n" entries. */
static void HFUN(resize)(HTABLE* h, unsigned int n) {
  unsigned int old_size;
  unsigned int i;
  ELEMENT* old;

  if (n < h->count)
    n = h->count;

  /* Save old contents and make new correctly sized table */
  old = h->table;
  old_size = h->tsize;
  init(h, n);
  assert(h->tsize > n);
  
  /* Now restore the old contents */
  for (i = 0; i < old_size; i++) {
    ELEMENT key = old[i];
    if ((!ELEMENT_IS_EMPTY(h, key)) && 
	(!ELEMENT_IS_DELETED(h, key))) {
      unsafe_insert(h, key);
    }
  }

  free(old);
}

void HFUN(init)(HTABLE* h
#ifndef	ELEMENT_HASH
		, HTYPE(hasher) hash
#endif	/* ELEMENT_HASH */
#ifndef	ELEMENT_KEY_EQUAL
		, HTYPE(comparer) key_equal
#endif	/* ELEMENT_KEY_EQUAL */
		)
{
#ifndef	ELEMENT_HASH
  h->hash = hash;
#endif	/* ELEMENT_HASH */
#ifndef	ELEMENT_KEY_EQUAL
  h->key_equal = key_equal;
#endif	/* ELEMENT_KEY_EQUAL */
  HSTAT(h->probes = 0);
  HSTAT(h->searches = 0);
  init(h, 1);
}

void HFUN(destroy)(HTABLE* h) {
  free(h->table);
}

void HFUN(predict)(HTABLE* h, int n) {
  if (n >= 0)
    HFUN(resize)(h, (unsigned int) n);
}

void HFUN(shrink)(HTABLE* h)
{
  /*
   * modifies: h
   * effects:  Resizes hash table h to an appropriate size for the
   *           current number of elements that it contains.
   *
   */

  /* contract table when less than 1/6th occupied */
  if (h->tsize > (h->count * 6))
    HFUN(resize)(h, 0);
}

void HFUN(reset)(HTABLE* h)
{
  /*
   * modifies: h
   * effects:  Resets h to an empty hash table, discarding any
   *           existing elements that it contains.
   *
   */

  /* reclaim storage */
  free(h->table);

  /* reinitialize table */
  init(h, 1);
}

void HFUN(copy)(HTABLE* dst, HTABLE* src) {
  /* This check is necessary to deal with aliasing between the tables. */
  if (dst != src) {
    unsigned int i;
    unsigned int tsize = src->tsize;
    ELEMENT* table = src->table;
    
    HFUN(clear)(dst);
    HFUN(resize)(dst, src->count);
    
    for (i = 0; i < tsize; i++) {
      ELEMENT key = table[i];
      if ((!ELEMENT_IS_EMPTY(src, key)) &&
	  (!ELEMENT_IS_DELETED(src, key))) {
	unsafe_insert(dst, key);
      }
    }
  }
}

void HFUN(clear)(HTABLE* h) {
  unsigned int i;
  for (i = 0; i < h->tsize; i++)
    ELEMENT_MARK_EMPTY(h, &h->table[i]);
  h->count = 0;
  h->delcount = 0;
}

int HFUN(size)(HTABLE* h) {
  return (int)(h->count);
}

ELEMENT HFUN(find)(HTABLE* h, ELEMENT key) {
  unsigned int index = HFUN(find_index)(h, key);
  ELEMENT result = h->table[index];

  if (ELEMENT_IS_DELETED(h, result))
    ELEMENT_MARK_EMPTY(h, &result);

  return(result);
}

ELEMENT HFUN(try_insert)(HTABLE* h, ELEMENT key) {
  unsigned int index = HFUN(find_index)(h, key);
  ELEMENT old = h->table[index];
  if (ELEMENT_IS_EMPTY(h, old)) {
    /* Insert into empty slot */
    h->count++;
    h->table[index] = key;
    assert((h->count + h->delcount) <= h->tsize);
    if ((h->count + h->delcount) >= h->enlarge_size) {
      HFUN(resize)(h, h->count);
    }
    return key;
  }
  if (ELEMENT_IS_DELETED(h, old)) {
    /* Re-use the slot */
    h->delcount--;
    h->count++;
    h->table[index] = key;
    return key;
  }
  return old;
}

ELEMENT HFUN(insert)(HTABLE* h, ELEMENT key) {
  unsigned int index = HFUN(find_index)(h, key);
  ELEMENT old = h->table[index];
  h->table[index] = key;
  if (ELEMENT_IS_EMPTY(h, old)) {
    /* We added a brand-new entry */
    h->count++;
    assert((h->count + h->delcount) <= h->tsize);
    if ((h->count + h->delcount) >= h->enlarge_size) {
      HFUN(resize)(h, h->count);
    }
  } else if (ELEMENT_IS_DELETED(h, old)) {
    /* We re-used the slot */
    ELEMENT_MARK_EMPTY(h, &old);
    h->delcount--;
    h->count++;
  }
  return old;
}

ELEMENT HFUN(remove)(HTABLE* h, ELEMENT key) {
  unsigned int index = HFUN(find_index)(h, key);
  ELEMENT old = h->table[index];
  if ((!ELEMENT_IS_EMPTY(h, old)) &&
      (!ELEMENT_IS_DELETED(h, old))) {
    ELEMENT_MARK_DELETED(h, &h->table[index]);
    h->count--;
    h->delcount++;
    /* XXX Automatically shrink? */
  }
  return old;
}

void HFUN(check)(HTABLE* h) {
#ifndef	NDEBUG
  unsigned int i, j;
  unsigned int tsize = h->tsize;
  ELEMENT* table = h->table;

  /* Check for duplicates */
  for (i = 0; i < tsize; i++) {
    ELEMENT x = table[i];
    if (ELEMENT_IS_EMPTY(h, x)) continue;
    if (ELEMENT_IS_DELETED(h, x)) continue;
    
    for (j = i+1; j < tsize; j++) {
      ELEMENT y = table[j];
      if ((!ELEMENT_IS_EMPTY(h, y)) &&
	  (!ELEMENT_IS_DELETED(h, y))) {
	assert(!ELEMENT_KEY_EQUAL_PRIM(h, y, x));
      }
    }
  }

  /* Check that there are no empty slots in hash sequence for each elt */
  for (i = 0; i < tsize; i++) {
    unsigned int index;
    ELEMENT x = table[i];
    if (ELEMENT_IS_EMPTY(h, x)) continue;
    if (ELEMENT_IS_DELETED(h, x)) continue;

    index = hash_index(ELEMENT_HASH_PRIM(h, x), h->slot_bits);

    /* Loop at most tsize times */
    for (j = 0; j < tsize; j++) {
      /* Check for hole */
      assert(!ELEMENT_IS_EMPTY(h, table[index]));
      if (index == i) break;
      index = (index + 1) & (h->tsize - 1);
    }
  }

  /* count */
  j = 0;
  for (i = 0; i < tsize; i++) {
    ELEMENT x = table[i];
    if (ELEMENT_IS_EMPTY(h, x)) continue;
    if (ELEMENT_IS_DELETED(h, x)) continue;
    j++;
  }
  assert(h->count == j);
  
  /* delcount */
  j = 0;
  for (i = 0; i < tsize; i++) {
    if (ELEMENT_IS_DELETED(h, table[i])) j++;
  }
  assert(h->delcount == j);

  /* tsize is power of two */
  i = 1;
  while (i < tsize) {
    i = i << 1;
  }
  assert(i == tsize);

  /* Check that at least one slot is empty */
  assert(tsize > (h->count + h->delcount));
#endif	/* NDEBUG */
}

void HFUN(stats)(HTABLE* h, char* msg) {
#ifndef	NDEBUG
#if	0
    unsigned int* weight;
    unsigned int* run;
    unsigned int i;
#endif

    if (h->searches == 0) return;
    fprintf(stderr,
	    "%s: size= %5d, present= %5d, deleted= %5d, probes=%5.3f [%5d]\n",
	    msg, (int)h->tsize, (int)h->count, (int)h->delcount,
	    ((double) (h->probes + h->searches)) / ((double) h->searches),
	    (int)h->searches);

#if 0
    /* Compute number of primary hashes per bucket */
    weight = ALLOC_ARRAY(unsigned int, h->tsize);
    run = ALLOC_ARRAY(unsigned int, h->tsize);
    memset(weight, 0, sizeof(unsigned int) * h->tsize);
    memset(run, 0, sizeof(unsigned int) * h->tsize);
    for (i = 0; i < h->tsize; i++) {
	ELEMENT x = h->table[i];
	if (!ELEMENT_IS_EMPTY(h, x) && !ELEMENT_IS_DELETED(h, x)) {
	    unsigned int index = hash_index(ELEMENT_HASH_PRIM(h,x),
					    h->slot_bits);
	    weight[index]++;
	    if (index <= i) {
		run[i] = i + 1 - index;
	    } else {
		run[i] = h->tsize + i + 1 - index;
	    }
	}
    }
    for (i = 0; i < h->tsize; i++) {
	fprintf(stderr, "%s: %6d -> %6d, run %6d\n", msg, i, weight[i],run[i]);
    }
    free(weight);
    free(run);
#endif

#endif	/* NDEBUG */
}

void HGENFUN(init)(HTABLEGEN* g, HTABLE* h) {
  g->h = h;
  g->index = (int)h->tsize;
}

ELEMENT HGENFUN(next)(HTABLEGEN* g) {
  HTABLE* h = g->h;
  ELEMENT x;

  while (g->index > 0) {
    g->index--;
    x = h->table[g->index];
    if ((!ELEMENT_IS_EMPTY(h, x)) &&
	(!ELEMENT_IS_DELETED(h, x))) {
      return x;
    }
  }

  ELEMENT_MARK_EMPTY(h, &x);
  return(x);
}

void HGENFUN(remove_element)(HTABLEGEN* g) {
    int i = g->index;
    HTABLE* h = g->h;

    assert(i >= 0);
    assert(!ELEMENT_IS_EMPTY(h, h->table[i]));
    assert(!ELEMENT_IS_DELETED(h, h->table[i]));

    h->count--;
    h->delcount++;
    ELEMENT_MARK_DELETED(h, &h->table[i]);
}

void HGENFUN(replace_element)(HTABLEGEN* g, ELEMENT x) {
    int i = g->index;
    HTABLE* h = g->h;

    assert(i >= 0);
    assert(!ELEMENT_IS_EMPTY(h, h->table[i]));
    assert(!ELEMENT_IS_DELETED(h, h->table[i]));

    assert(ELEMENT_HASH_PRIM(h, h->table[i]) == ELEMENT_HASH_PRIM(h, x));
    assert(ELEMENT_KEY_EQUAL_PRIM(h, h->table[i], x));

    h->table[i] = x;
}

