#!/usr/bin/python
#
# Compile a text description of a keymap into a kernel .h file
#
# Usage:
#
#       makekernelmap [options] [filename]
#
# where valid options are
#
#         -k, --kernel        Create a file for the internal kernel table
#         -l, --linux         Create a file suitable for "loadkeys"
#         -x, --X             Create a file suitable for "xmodmap"

import sys
import os
import getopt

# Map Linux console key names ---> Xmodmap names
linuxtox = { "one" : "1",
             "two" : "2",
             "three" : "3",
             "four" : "4",
             "five" : "5",
             "six"  : "6",
             "seven" : "7",
             "eight" : "8",
             "nine"  : "9",
             "zero"  : "0",
             "euro"  : "EuroSign",
             "CtrlL" : "Control_L",
             "ShiftL" : "Shift_L",
             "AltGr" : "Mode_switch",
             "XFCalendar" : "XF86Calendar",
             "XTelephone" : "telephone",
             "XFMail"     : "XF86Mail",
             "XFStart"    : "XF86Start",
             "XFAudioRecord" : "XF86AudioRecord",
             "XFPowerDown" : "XF86PowerDown",
             "XFRockerEnter" : "XF86RockerEnter",
             }

def putxchar(c):
    if linuxtox.has_key(c):
        print linuxtox[c],
    else:
        print c,

# Map Linux console key names ---> Internal kernel driver constants
linuxskbd = { "Escape" : "SKEY_ESC",
              "one" : "SKEY_1",
              "two" : "SKEY_2",
              "three" : "SKEY_3", 
              "four" : "SKEY_4",
              "five" : "SKEY_5",
              "six" : "SKEY_6",
              "seven" : "SKEY_7", 
              "eight" : "SKEY_8",
              "nine" : "SKEY_9",
              "zero" : "SKEY_0",
              "minus" : "SKEY_MINUS", 
              "equal" : "SKEY_EQUAL",
              "BackSpace" : "SKEY_BACKSPACE",
              "Tab" : "SKEY_TAB",
              "q" : "SKEY_Q", 
              "w" : "SKEY_W",
              "e" : "SKEY_E",
              "r" : "SKEY_R",
              "t" : "SKEY_T", 
              "y" : "SKEY_Y",
              "u" : "SKEY_U",
              "i" : "SKEY_I",
              "o" : "SKEY_O", 
              "p" : "SKEY_P",
              "braceleft" : "SKEY_LEFTBRACE",
              "braceright" : "SKEY_RIGHTBRACE",
              "Return" : "SKEY_ENTER", 
              "CtrlL" : "SKEY_LEFTCTRL",
              "a" : "SKEY_A",
              "s" : "SKEY_S",
              "d" : "SKEY_D", 
              "f" : "SKEY_F",
              "g" : "SKEY_G",
              "h" : "SKEY_H",
              "j" : "SKEY_J", 
              "k" : "SKEY_K",
              "l" : "SKEY_L",
              "semicolon" : "SKEY_SEMICOLON",
              "apostrophe" : "SKEY_APOSTROPHE",
              "grave" : "SKEY_GRAVE",
              "ShiftL" : "SKEY_LEFTSHIFT",
              "backslash" : "SKEY_BACKSLASH",
              "z" : "SKEY_Z", 
              "x" : "SKEY_X",
              "c" : "SKEY_C",
              "v" : "SKEY_V",
              "b" : "SKEY_B", 
              "n" : "SKEY_N",
              "m" : "SKEY_M",
              "comma" : "SKEY_COMMA",
              "period" : "SKEY_DOT", 
              "slash" : "SKEY_SLASH",
              "ShiftR" : "SKEY_RIGHTSHIFT",
              "KP_Multiply" : "SKEY_KPASTERISK",
              "Alt" : "SKEY_LEFTALT",
              "AltGr" : "SKEY_RIGHTALT",
              "Prior" : "SKEY_PAGEUP",
              "Next"  : "SKEY_PAGEDOWN",
              "Up"    : "SKEY_UP",
              "Down"  : "SKEY_DOWN",
              "Left"  : "SKEY_LEFT",
              "Right" : "SKEY_RIGHT",
              "space" : "SKEY_SPACE",
              "Caps_Lock" : "SKEY_CAPSLOCK",
              "F1" : "SKEY_F1",
              "F2" : "SKEY_F2", 
              "F3" : "SKEY_F3",
              "F4" : "SKEY_F4",
              "F5" : "SKEY_F5",
              "F6" : "SKEY_F6", 
              "F7" : "SKEY_F7",
              "F8" : "SKEY_F8",
              "F9" : "SKEY_F9",
              "F10" : "SKEY_F10", 
              "Num_Lock" : "SKEY_NUMLOCK",
              "Scroll_Lock" : "SKEY_SCROLLLOCK",
              "KP_7" : "SKEY_KP7",
              "KP_8" : "SKEY_KP8", 
              "KP_9" : "SKEY_KP9",
              "KP_Subtract" : "SKEY_KPMINUS",
              "KP_4" : "SKEY_KP4",
              "KP_5" : "SKEY_KP5", 
              "KP_6" : "SKEY_KP6",
              "KP_Add" : "SKEY_KPPLUS",
              "KP_1" : "SKEY_KP1",
              "KP_2" : "SKEY_KP2", 
              "KP_3" : "SKEY_KP3",
              "KP_0" : "SKEY_KP0",
              "KP_Period" : "SKEY_KPDOT",
              "F11" : "SKEY_F11", 
              "F12" : "SKEY_F12",
              "KP_Enter" : "SKEY_KPENTER",
              "CtrlR" : "SKEY_RIGHTCTRL",
              "KP_Divide" : "SKEY_KPSLASH",
              "XFCalendar" : "H3600_SCANCODE_CALENDAR",
              "XTelephone" : "H3600_SCANCODE_CONTACTS",
              "XFMail"     : "H3600_SCANCODE_ENVELOPE",
              "XFStart"    : "H3600_SCANCODE_START" }

class ScancodeToSKBD:
    def __init__(self,path):
        self.dict = {}
        fd=open(path,'r')
        for line in fd.readlines():
            if line.startswith('#define'):
                values = line.split()
                if len(values) < 3:
                    continue
                (key,value) = values[1:3]
                if self.dict.has_key(value):
                    self.dict[key] = self.dict[value]
                else:
                    self.dict[key] = eval(value)

    def __getitem__(self,item):
        if self.dict.has_key(item):
            return self.dict[item]
        return None

    def printdict(self):
        keylist = self.dict.keys()
        keylist.sort()
        for k in keylist:
            print "%25s : %d" % (k, self.dict[k])
        

# The KeyboardToLinux maps from internal keyboard-specific scancodes to a desired
# set of Linux Console codes.
#
# Key dict files have the form
#    KEY   NAME   ALTNAME   SHIFTALTNAME
# where the last two are optional

class KeyboardToLinux:
    def __init__(self,fd,kscan):
        self.dict = {}           # Keyboard-specifc code  -> Linux console key names (list)
        self.scancodes = kscan   # SKEY_XXXX              -> Kernel keycode (#)
        self.extras  = { 'linux'   : [],
                         'xmodmap' : [],
                         '!'       : [] }
        # global linuxskbd         Linux console key name -> SKEY_XXXX
        for line in fd.readlines():
            self.addline(line)

    def addline(self,line):
            if line.startswith('#'):
                return

            if line.startswith('!'):
                self.extras['!'].append(line.strip())
                return
            
            for extra in self.extras.keys():
                if line.startswith(extra):
                    self.extras[extra].append(line[len(extra):].strip())
                    return

            values = line.split()
            if len(values) < 2:
                return
            key = eval(values[0])
            if self.dict.has_key( key ):
                sys.stderr.write("Redundant key %d %s\n" % key, values[0])
            else:
                self.dict[key] = values[1:]
        
    def __getitem__(self,key):
        if self.dict.has_key(key):
            return self.dict[key]
        return None

    def has_key(self,key):
        return self.dict.has_key(key)

    def printdict(self):
        keylist = self.dict.keys()
        keylist.sort()
        for k in keylist:
            print "%6d : %s" % (k, self.dict[k])

    def dumpforxmodmap(self):
        "Print mapping: kernel keycode (#) ---> Xmodmap file"
        # Fill in some defaults for the keys that are always available
        keycode_to_values = { 104 : ["KP_Enter"],
                              111 : ["Up"],
                              113 : ["Left"],
                              114 : ["Right"],
                              116 : ["Down"],
                              128 : ["XFAudioRecord"],
                              129 : ["XFPowerDown"],
                              130 : ["XFCalendar"],
                              131 : ["XTelephone"],
                              132 : ["XFMail"],
                              133 : ["XFStart"],
                              134 : ["XFRockerEnter"] }
        for k in self.dict.keys():
            keycode = self.scancodes[linuxskbd[self.dict[k][0]]]
            keycode_to_values[keycode + 8] = self.dict[k]

        for j in ('!', 'xmodmap'):
            for i in self.extras[j]:
                print i

        for i in range(8,136):
            print "keycode %3d = " % i,
            if keycode_to_values.has_key(i):
                has_doubled = 0
                vlist = keycode_to_values[i]
                putxchar(vlist[0])
                if len(vlist[0]) == 1 and vlist[0].islower():
                    putxchar(vlist[0].upper())
                    has_doubled = 1
                if len(vlist) > 1:
                    if not has_doubled:
                        putxchar(vlist[0])
                    putxchar(vlist[1])
                if len(vlist) > 2:
                    putxchar(vlist[2])
            print ""
        print "clear Shift"
        print "clear Lock"
        print "clear Control"
        print "clear Mod1"
        print "clear Mod2"
        print "clear Mod3"
        print "clear Mod4"
        print "clear Mod5"
        print "add Shift   = Shift_L"
        print "add Lock    = Caps_Lock"
        print "add Control = Control_L"
        print "add Mod1    = Mode_switch"
        
    def dumpforloadkeys(self):
        "Print mapping: kernel keycode (#) ---> Linux console key name"
        keycode_to_values = {  96 : ["KP_Enter"],
                              103 : ["Up"],
                              105 : ["Left"],
                              106 : ["Right"],
                              108 : ["Down"],
                              120 : ["XFAudioRecord"],
                              121 : ["XFPowerDown"],
                              122 : ["XFCalendar"],
                              123 : ["XTelephone"],
                              124 : ["XFMail"],
                              125 : ["XFStart"],
                              126 : ["XFRockerEnter"] }

        for k in self.dict.keys():
            keycode = self.scancodes[linuxskbd[self.dict[k][0]]]
            keycode_to_values[keycode] = self.dict[k]
        # By default we assume:
        #    0 = Normal
        #    1 = Shift
        #    2 = AltGr
        #    4 = Control
        #    8 = Alt       (with all appropriate combinations)
        # The Linux norm is 0-2,4-6,8-9,12
        for j in ('!', 'linux'):
            for i in self.extras[j]:
                print i

        print "keymaps 0-6,8-9,12"
        for i in range(1,128):
            print "keycode  %3d = " %i, 
            if keycode_to_values.has_key(i):
                v = keycode_to_values[i]
                print "%-14s" % v[0]
                if len(v) > 1:
                    print "        altgr keycode %3d = %s" % (i,v[1])
                if len(v) > 2:
                    print "        shift altgr keycode %3d = %s" % (i,v[2])
            else:
                print ""

    def dumpforkernel(self):
        "Print mapping: keyboard-specific code ---> SKEY_XXXX"
        print "/* Kernel mapping from scancode to keycode */"
        print "unsigned char raw_to_scancode[128] = {",
        bits = 0L
        for i in range(0,128):
            r = 0
            if self.dict.has_key(i):
                value = self.dict[i][0]
                if linuxskbd.has_key(value):
                    r = linuxskbd[value]  # r is something like SKEY_BACKSPACE"
                    bits |= 1L << self.scancodes[r]
                else:
                    print "WARNING!  MISSING KEY %s " % value
                    continue
            if not (i % 4):
                print "\n       ",
            print "%s, " % r,
        print " };\n"
        print "unsigned char raw_bitmask[] = { ",
        i = 0
        while ( bits ):
            if (i % 4) == 0:
                print "\n       ",
            i+=1
            tail = bits & 0xffL
            print "0x%02x," % tail,
            bits = bits >> 8
        print "};"


def usage():
    print "Usage: makekernelmap [opts] [filename]"
    print "If filename is not given, use stdin"
    print "Options:"
    print "      -k, --kernel           Dump kernel-specific file (default)"
    print "      -l, --loadkeys         Dump a loadkeys file"
    print "      -x, -X                 Dump an xmodmap file"
    print "      -p PATH, --path=PATH   Set kernel path (default=%s)" % kernelpath
    print "      -h, --help             This help message" 
    sys.exit(1)

def dofd(fd,kscan,dumptype):
    kd = KeyboardToLinux(fd,kscan)
    if dumptype == 'kernel':
        kd.dumpforkernel()
    elif dumptype == 'loadkeys':
        kd.dumpforloadkeys()
    elif dumptype == 'xmodmap':
        kd.dumpforxmodmap()
    else:
        print "Unknown dump type " + dumptype

# Scan the directories

kernelpath = "/rock/home/andyc/backpaq/linux/kernel"
dumptype = 'kernel'

try:
    (opts,pargs) = getopt.getopt(sys.argv[1:],'hp:klxX',['help','path=','kernel','loadkeys'])
except:
    usage()

for i in opts:
    if i[0] == '-h' or i[0] == '--help':
        usage()
    elif i[0] == '-p' or i[0] == '--path':
        kernelpath = i[1]
    elif i[0] == '-k' or i[0] == '--kernel':
        dumptype = 'kernel'
    elif i[0] == '-l' or i[0] == '--loadkeys':
        dumptype = 'loadkeys'
    elif i[0] == '-x' or i[0] == '-X':
        dumptype = 'xmodmap'
        
# Set kscan to be our SKBD to keycode mapping
kpath =  os.path.join(kernelpath,'include/linux/h3600_keyboard.h')
try:
    kscan = ScancodeToSKBD( kpath )
except:
    print "Unable to open %s" % kpath
    sys.exit(1)
    
if len(pargs) == 0:
    dofd(sys.stdin,kscan,dumptype)
else:
    for i in pargs:
        dofd(open(i,'r'),kscan,dumptype)

