#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#include "dplib/dpmacs.h"
#include "dplib/eprintf.h"
#include "dplib/dl_list.h"
#include "dplib/fs-helpers.h"
#include "dplib/cmdproc.h"
#include "dplib/malloc-helpers.h"
#include "dplib/numio.h"
#include "dplib/mach.h"

#include "bootldr/partition.h"
#include "dplib/bl_params.h"

#define	printf()    *** do not use printf, use rprintf instead ***


char*		progname;
blp_params_t	params;
blp_params_t*	paramsp = &params;
char*		outfile = "";	/* empty --> stdout */

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_show(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    if (argc == 1) {
	rprintf("Partitions:\n");
	blp_show_parts(paramsp);
	rprintf("\nParameters:\n");
	blp_show_vars(paramsp);
	rprintf("\nsize in flash: %d\n",
		long_round(paramsp->var_size+paramsp->part_size));
    }
    else {
	int	i;

	for (i = 1; i < argc; i++)
	    blp_show_part_or_var(paramsp, argv[i]);
    }

    return (0);
}
	
/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_set(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    blp_var_t*	var;

    if (argc != 3) {
	eprintf(0, 0, "Wrong number of params.\nusage: %s\n", help);
	return (-1);
    }

    var = blp_var_define(paramsp, argv[1], argv[2]);
    
    rprintf("set >%s< to >%s<\n", var->name, var->val);

    return (0);
}
    

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_samiam(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    int	    i;
    
    rprintf("%s called \n", argv[0]);
    for (i = 1; i < argc; i++)
	rprintf("%s ", argv[i]);
    if (i > 1)
	rprintf("\n");

    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_part_define(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    if (argc != 5) {
	eprintf(0, 0, "Wrong number of params.\nusage: %s\n", help);
	return (-1);
    }

     /*  "<name> <base> <size> <flags>\n  -- Define a partition", */
    blp_part_define(paramsp,
		    argv[1],
		    getsizeX(argv[2], 1, "sS", paramsp->sec_size),
		    getsizeX(argv[3], 1, "sS", paramsp->sec_size),
		    blp_flags_to_num(argv[4]));

    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_part_show(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    if (argc != 1)
	blp_show_parts(paramsp);
    else {
	int	i;

	for (i = 1; i < argc; i++)
	    blp_show_part(paramsp, argv[i]);
    }
    
    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_part_reset(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    if (argc != 1) {
	eprintf(0, 0, "Wrong number of params.\nusage: %s\n", help);
	return (-1);
    }

    /* temporary */
    return (cmd_samiam(argc, argv, help, private));
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_part_delete(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    blp_part_t*	part;
    
    if (argc != 2) {
	eprintf(0, 0, "Wrong number of params.\nusage: %s\n", help);
	return (-1);
    }

    part = blp_find_part_by_name(paramsp, argv[1]);
    if (part) {
	BLP_PARAM_DEL_PART(paramsp, part);
	return (0);
    }
    else {
	eprintf(0, 0, "No partition named >%s<\n", argv[1]);
	return (ENOENT);
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
cmd_save(
    int	    argc,
    char*   argv[],
    char*   help,
    void*   private)
{
    char*   file;
    
    if (argc > 2) {
	eprintf(0, 0, "Wrong number of params.\nusage: %s\n", help);
	return (-1);
    }
    if (argc == 2)
	file = argv[1];
    else
	file = outfile;

    blp_write_params(paramsp, file);

    return (0);
}
    
/*
************************************************************************
*
* Command Tables
* 
************************************************************************
*/
cmdproc_entry_t	params_commands[] =
{
    {"set", cmd_set, "<param> <value>\n -- sets param to value", NULL},
    {"show", cmd_show, "var...\n -- show values of var...", NULL},
    {"save", cmd_save, "[file]\n -- save params and partition table", NULL},
};

cmdproc_cmd_chunk_t params_cmd_chunk =
{
    DIM(params_commands),
    params_commands
};

cmdproc_entry_t	partition_commands[] =
{
    {"define", cmd_part_define,
     "<name> <base> <size> <flags>\n  -- Define a partition",
     NULL},

    {"show|list", cmd_part_show,
     "[part...]\n  -- Show named parts or all currently defined partitions",
     NULL},
    {"list", cmd_part_show,
     "\n  -- Show all currently defined partitions",
     NULL},

    {"reset", cmd_part_reset,
     "\n  -- Reset parition table to defaults.",
     NULL},

    {"delete", cmd_part_delete,
     "<name>\n  -- Delete the partition named <name>",
     NULL},
};

cmdproc_cmd_chunk_t part_cmd_chunk =
{
    DIM(partition_commands),
    partition_commands
};

cmdproc_entry_t	commands[] =
{
    {"partition|part|parts", cmdproc_subcommand, CMDPROC_CALL_FOR_HELP,
     (void*)&part_cmd_chunk},

    {"params", cmdproc_subcommand, CMDPROC_CALL_FOR_HELP,
     (void*)&params_cmd_chunk},

    {"set", cmd_set, "var val\n -- sets var to val", NULL},

    {"show", cmd_show, "var...\n -- show values of var...", NULL},
    
    {"?", cmdproc_help, "help", CMDPROC_CURRENT_CMD_CHUNK}
};

cmdproc_cmd_chunk_t cmd_chunk =
{
    DIM(commands),
    commands
};

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
pedit(
    FILE*	    cmd_stream,
    blp_params_t*   params,
    int		    interactive,
    char*	    prompt)
{
    char*   argv[100];
    int	    argc;
    char    cmdline[256];
    size_t  len;

    if (prompt == NULL)
	prompt = "> ";
	
    if (interactive)
	rprintf(prompt);
    
    while (fgets(cmdline, sizeof(cmdline)-1, cmd_stream)) {
	len = strlen(cmdline);
	if (len && cmdline[len-1] == '\n')
	    cmdline[len-1] = '\0';
	if (cmdline[0] == '\0' || cmdline[0] == '#')
	    goto loop;
	
	cmdproc_parse(cmdline, &argc, argv, DIM(argv));
	cmdproc_exec(argc, argv, &cmd_chunk);
	cmdproc_free(argc, argv);

      loop:
	if (interactive)
	    rprintf(prompt);
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
main(
    int	    argc,
    char*   argv[])
{
    char*	options = "s:o:i:";
    char*	file;
    int		opt;
    FILE*	fp;
    char*	infile;
    
    extern char*    optarg;
    extern int	    optind;
    extern int	    opterr;

    if (sizeof(long) != 4)
	eprintf(2, 0, "This program assumes longs are 4 bytes.  Sorry.\n"
		"You'll need to port it.\n");

    if (!is_little_endian())
	eprintf(2, 0, "This program assumes the processor is little endian.  "
		"Sorry.\n"
		"You'll need to port it.\n");
    
    infile = "/dev/mtdblock/1";
    memset((char*)&params, 0x00, sizeof(params));
    dl_list_init(&params.partition_list);
    dl_list_init(&params.variable_list);
    
    outfile = "";
    opterr = 1;

    blp_set_sector_size_s(&params, "0x40000");

    progname = basename(argv[0]);

    while ((opt = getopt(argc, argv, options)) != EOF) {
	switch (opt) {
	    case 'i':
		infile = optarg;
		break;
		
	    case 's':
		blp_set_sector_size_s(&params, optarg);
		break;
		
	    case 'o':
		outfile = optarg;
		break;
		
	    default:
		eprintf(2, 0, "Illegal option.\n");
		exit(2);
	}
    }
    
    blp_read_params_sector(infile, &params);
    blp_parse_partition_table(&params);
    blp_parse_params(&params);
    
    /* now, process arguments */
    if (optind >= argc) {
	/* no args, process stdin */
	pedit(stdin, &params, isatty(fileno(stdin)), "pedit> ");
    }
	
    while (optind < argc) {
	file = argv[optind++];
	if ((fp = fopen(file, "r")) == NULL)
	    eprintf(2, 0, "Cannot open input file >%s<\n", file);
	
	pedit(fp, &params, 0, "");
	
	if (fclose(fp))
	    eprintf(2, 0, "Error closing input file >%s<\n", file);
    }

    blp_write_params(paramsp, outfile);

    blp_free_all(paramsp);

    tmd_mem_check();
    
    return (0);
}
