/*
 * Itsy SA1100 Profiling Applications: Image Hash Implementation
 *
 * Copyright (c) Compaq Computer Corporation, 1999
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 */

/*
 * Author: Carl Waldspurger
 *
 * $Log: ihash.c,v $
 * Revision 1.1.1.1  2000/11/28 16:46:34  kerr
 * Compaq Profiling Infrastructure
 *
 * Revision 1.4  1999/08/27 17:38:10  caw
 * Replaced system <linux/{elf,a.out}.h> includes with generic "itsyhdr.h".
 *
 * Revision 1.3  1999/08/24 22:06:36  caw
 * Support ELF objects without program headers (e.g. loadable modules).
 *
 * Revision 1.2  1999/08/20 17:27:39  caw
 * Support both ELF and AOUT headers.
 *
 * Revision 1.1  1999/08/20 01:12:47  caw
 * Initial revision.
 *
 */

#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>

#include <util/util.h>
#include "ihash.h"
#include "itsyhdr.h"

/* 
 *  This implementation of CRC32 is from the Linux 2.0.30 sources 
 *  (drivers/net/smc9194.c), and is under the GNU Public License.
 *
 *  Finds the CRC32 of a set of bytes. 
 *  Again, from Peter Cammaert's code. 
 *
 */

static int crc32( char * s, int length ) { 
        /* indices */
        int perByte;
        int perBit;
        /* crc polynomial for Ethernet */
        const unsigned long poly = 0xedb88320;
        /* crc value - preinitialized to all 1's */
        unsigned long crc_value = 0xffffffff; 

        for ( perByte = 0; perByte < length; perByte ++ ) {
                unsigned char   c;

                c = *(s++);
                for ( perBit = 0; perBit < 8; perBit++ ) {
                        crc_value = (crc_value>>1)^
                                (((crc_value^c)&0x01)?poly:0);
                        c >>= 1;
                }
        }
        return  crc_value;
} 

#define	PH_MAX	(16)
#define	SH_MAX	(16)

static int image_file_hash_elf(int fd, int *hash)
{
  Elf32_Ehdr eh;
  int n, size;

  /* read ELF header, fail if not ELF */
  if ((lseek(fd, (off_t) 0, SEEK_SET) != (off_t) 0) ||
      (read(fd, &eh, sizeof(eh)) != sizeof(eh)) ||
      (eh.e_ident[0] != 0x7f) ||
      (strncmp(&eh.e_ident[1], "ELF", 3) != 0))
    return(FAILURE);

  /* hash programs headers, if any */
  if (eh.e_phoff != 0)
    {
      Elf32_Phdr ph[PH_MAX];

      /* read ELF executable program headers */
      n = MIN(PH_MAX, eh.e_phnum);
      size = n * eh.e_phentsize;
      if ((lseek(fd, (off_t) eh.e_phoff, SEEK_SET) != (off_t) eh.e_phoff) ||
	  (read(fd, ph, size) != size))
	return(FAILURE);

      /* compute hash value */
      *hash = crc32((char *) ph, size);
      return(SUCCESS);
    }

  /* hash section headers, if any */
  if (eh.e_shoff != 0)
    {
      Elf32_Shdr sh[SH_MAX];

      /* read ELF object section headers */
      n = MIN(SH_MAX, eh.e_shnum);
      size = n * eh.e_shentsize;
      if ((lseek(fd, (off_t) eh.e_shoff, SEEK_SET) != (off_t) eh.e_shoff) ||
	  (read(fd, sh, size) != size))
	return(FAILURE);

      /* compute hash value */
      *hash = crc32((char *) sh, size);
      return(SUCCESS);
    }

  /* no headers to hash */
  return(FAILURE);
}

static int image_file_hash_aout(int fd, int *hash)
{
  struct exec ah;

  /* read AOUT header, fail if not AOUT */
  if ((lseek(fd, (off_t) 0, SEEK_SET) != (off_t) 0) ||
      (read(fd, &ah, sizeof(ah)) != sizeof(ah)) ||
      (N_BADMAG(ah)))
    return(FAILURE);

  /* compute hash value */
  *hash = crc32((char *) &ah, sizeof(ah));

  /* everything OK */
  return(SUCCESS);
}

int image_file_hash(const char *name, int *hash)
{
  /*
   * modifies: hash
   * effects:  Sets hash to a hash value computed using header 
   *           information from the named executable image.
   *           Returns SUCCESS or FAILURE.
   *
   */

  int result, fd;

  /* initialize */
  result = FAILURE;
  *hash = 0;

  /* open file, fail if unable */
  if ((fd = open(name, O_RDONLY)) < 0)
    return(FAILURE);

  /* attempt hash assuming ELF or AOUT format */
  if ((image_file_hash_elf(fd, hash)  == SUCCESS) ||
      (image_file_hash_aout(fd, hash) == SUCCESS))
    result = SUCCESS;

  /* cleanup */
  (void) close(fd);
  return(result);
}


