/*
 * Itsy SA1100 Profiler Daemon
 *
 * Copyright (c) Compaq Computer Corporation, 1999
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 */

/*
 * Author: Carl Waldspurger
 *
 * $Log: profd.c,v $
 * Revision 1.2  2001/06/08 16:33:34  jamey
 * added -s option to skip directories, delayed signal handler installation until after databases created
 *
 * Revision 1.1.1.1  2000/11/28 16:46:34  kerr
 * Compaq Profiling Infrastructure
 *
 * Revision 1.19  1999/08/30  22:58:23  caw
 * Added "total" sample count field to profile header.
 *
 * Revision 1.18  1999/08/26 23:56:07  caw
 * Improved module info for "kernel.syms".
 *
 * Revision 1.17  1999/08/26 23:38:04  caw
 * Write "kernel.syms" file containing kernel+module info to database.
 * Renamed "profd_line" => "line_buf".  Other minor improvements.
 *
 * Revision 1.16  1999/08/26 22:13:59  caw
 * Allow multiple paths to be associated with each image.
 *
 * Revision 1.15  1999/08/24 01:09:55  caw
 * Added sampling pid, frequency, randomization to profile header.
 * Internally renamed unique image id from "hash" to "id".
 *
 * Revision 1.14  1999/08/20 17:51:57  caw
 * Removed device, inode from profile name; hash is sufficient.
 *
 * Revision 1.13  1999/08/20 01:42:20  caw
 * Renamed "hash" header field to "image".
 *
 * Revision 1.12  1999/08/20 01:14:14  caw
 * Use image header hash values for unique identification.
 *
 * Revision 1.11  1999/08/19 21:17:39  caw
 * Added profile header consisting of ASCII "keyword value" lines.
 * Other minor improvements.
 *
 * Revision 1.10  1999/08/17 01:19:56  caw
 * Distinguish special "kernel" and "nomap" profiles.
 * Add pid to profile names only when necessary.
 *
 * Revision 1.9  1999/08/16 22:15:22  caw
 * Added support for per-process profiles.
 *
 * Revision 1.8  1999/08/13 23:52:57  caw
 * Updated to use standard includes for hash table code.
 *
 * Revision 1.7  1999/08/13 01:11:46  caw
 * Added "-f" and "-r" options to control sampling frequency.
 *
 * Revision 1.6  1999/08/12 22:05:48  caw
 * Skip useless directories (e.g. /proc) when building image name map.
 *
 * Revision 1.5  1999/08/12 21:53:54  caw
 * Reduced default logging.  Added "-v" verbose logging option.
 *
 * Revision 1.4  1999/08/12 20:18:27  caw
 * Added support for (device, inode) => image name mappings.
 *
 * Revision 1.3  1999/08/12 00:33:26  caw
 * Specify "database" path on command-line.  Other minor improvements.
 *
 * Revision 1.2  1999/08/11 22:34:02  caw
 * Improved profile output: sorted by offset, less whitespace.
 *
 * Revision 1.1  1999/08/11 21:36:48  caw
 * Initial revision.
 *
 */

/*
 * includes
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <fcntl.h>
#include <signal.h>
#include <dirent.h>
#include <ctype.h>

#include <sys/stat.h>
#include <sys/ioctl.h>

#include <util/util.h>
#include <util/array.h>
#include <util/log.h>
#include <util/htable_generic.h>

#include <linux/profiler.h>

#include "profd.h"
#include "htable_sample.h"
#include "ihash.h"

/*
 * constants
 *
 */

#define	INAME_INVALID_CHARS	".+- %?*&<>/:|_\\"
#define	PROFD_MAX_LINE_LEN	(256)

#define	PROFD_IMAGE_ID_INVALID	(0)

#define	PROFD_NAME_UNKNOWN	"unknown"
#define	PROFD_NAME_KERNEL	"kernel"
#define	PROFD_NAME_NOMAP	"nomap"

#define	PROFD_PROFILE_VERSION	"itsy-0.01"

const char *skip_files[] = { ".", "..", NULL };
const char *skip_paths[32] = { "/proc", "/dev", NULL };

/*
 * types
 *
 */

typedef char line_buf[PROFD_MAX_LINE_LEN];

typedef ARRAY(char*) string_array;

typedef struct {
  ushort device;
  ulong inode;
  ulong id;
  char *name;
  string_array paths;
} iname;

typedef struct {
  /* image identifiers */
  ulong  inode;
  ushort device;

  /* associated process */
  ulong  pid;

  /* table of profd_sample */
  htable_sample data;
} profd_image;

typedef struct {
  /* profiling database */
  char *db_path;

  /* tables of iname */
  htable_generic inames;
  htable_generic modnames;

  /* profiling device */
  int fd;

  /* sampling frequency */
  ulong frequency;
  int randomize;

  /* per-process profiles */
  int by_pid;

  /* table of profd_image */
  htable_generic images;
} profd;

/*
 * globals
 *
 */

static int profd_terminate = 0;

/*
 * signal handlers
 *
 */

static void profd_signal_terminate(int ignore)
{
  profd_terminate = 1;
}

/*
 * string operations
 *
 */

static void string_destroy(char *s)
{
  /*
   * modifies: s
   * effects:  Reclaims all storage associated with s.
   *           Does nothing if s is NULL.
   *
   */
  
  if (s != NULL)
    free(s);
}

static char *string_clone(const char *s)
{
  /*
   * modifies: nothing
   * effects:  Returns an identical copy of s, or NULL if s is NULL.
   * storage:  Return value is NULL or heap-allocated.
   *
   */

  /* special-case: NULL */
  if (s == NULL)
    return(NULL);

  /* normal case */
  return(strdup(s));
}

static char *string_squeeze(const char *s, const char *remove)
{
  /*
   * modifies: nothing
   * effects:  Returns a new copy of s that excludes all 
   *           characters in remove.
   *
   */

  /* allocate storage */
  char *squeeze = (char *) malloc(STRING_LENGTH(s) + 1);
  char *p = squeeze;

  /* copy s to squeeze, removing unwanted chars */
  while (*s != '\0')
    {
      /* check if remove contains char */
      const char *r = remove;
      while ((*r != '\0') && (*r != *s))
	r++;

      /* if not, copy to squeeze */
      if (*r != *s)
	*p++ = *s;

      /* advance to next char */
      s++;
    }

  /* add terminating NUL */
  *p = '\0';

  /* everything OK */
  return(squeeze);
}

/*
 * file operations
 *
 */

static int dir_exists(const char* name) 
{
  struct stat s;

  if (stat(name, &s) < 0)
    return(0);
  else
    return(S_ISDIR(s.st_mode));
}

/*
 * iname operations
 *
 */

static iname *iname_create(ushort device, 
			   ulong inode,
			   ulong id,
			   const char *name)
{
  iname *n;

  /* allocate container */
  if ((n = ALLOC_NEW(iname)) == NULL)
    return(NULL);
  
  /* initialize */
  n->device = device;
  n->inode  = inode;
  n->id     = id;
  n->name   = string_clone(name);
  array_init(&n->paths, 1);

  /* everything OK */
  return(n);
}

static char *iname_name(iname *n)
{
  if ((n == NULL) || (n->name == NULL))
    return(PROFD_NAME_UNKNOWN);
  else
    return(n->name);
}

static void iname_destroy(iname *n)
{
  int i;

  /* reclaim path names */
  string_destroy(n->name);
  for (i = 0; i < n->paths.length; i++)
    string_destroy(n->paths.list[i]);
  array_destroy(&n->paths);

  /* reclaim container */
  free(n);
}

static void iname_add_path(iname *n, const char *path)
{
  array_append(&n->paths, string_clone(path));
}

static uint iname_hash(iname *n)
{
  /* 
   * modifies: nothing
   * effects:  Returns a hash value for n.
   *
   */

  return((uint) n->inode);
}

static int iname_match(iname *a, iname *b)
{
  /*
   * modifies: nothing
   * effects:  Returns TRUE iff a and b have the same device and inode.
   *
   */

  return((a->device == b->device) &&
	 (a->inode  == b->inode));
}

static int array_contains(const char **a, const char *s)
{
  /*
   * modifies: nothing
   * effects:  Returns TRUE iff array a contains string s.
   *
   */

  int i;

  /* search for s in a */
  for (i = 0; a[i] != NULL; i++)
    if (STRING_EQUAL(a[i], s))
      return(1);

  /* not found */
  return(0);
}

static int map_image_names(const char *path,
			   int recurse,
			   htable_generic *inames)
{
  /*
   * requires: inames is a table of iname
   * modifies: inames
   * effects:  Scans all files in directory identified by path, adding
   *           entries to inmaes.  If recurse is set, recursively scans all
   *           subdirectories of path.  Returns count of updates to inames.
   *
   */

  struct dirent *e;
  int count;
  DIR *d;
  
  log_verbose("map_image_names: %s\n", path);
  /* open specified directory, fail if error */
  if ((d = opendir(path)) == NULL)
    return(0);
  
  /* walk directory tree */
  count = 0;
  for (e = readdir(d); e != NULL; e = readdir(d))
    {
      line_buf name;
      int slash, id;
      struct stat s;
      iname *n, key;

      /* skip special files */
      if (array_contains(skip_files, e->d_name))
	continue;

      /* form complete path */
      slash = (path[STRING_LENGTH(path) - 1] == '/');
      (void) sprintf(name, 
		     "%s%s%s",
		     path,
		     slash ? "" : "/",
		     e->d_name);

      /* skip special paths */
      if (array_contains(skip_paths, name))
	continue;      

      /* skip if unable to query entry */
      if (stat(name, &s) < 0)
	continue;

      /* skip if already mapped */
      key.device = (ushort) s.st_dev;
      key.inode  = (ulong)  s.st_ino;
      if ((n = htable_generic_find(inames, &key)) != NULL)
	{
	  iname_add_path(n, name);
	  continue;
	}

      /* update map */
      if (image_file_hash(name, &id) != SUCCESS)
	id = PROFD_IMAGE_ID_INVALID;
      n = iname_create(key.device, key.inode, id, e->d_name);
      iname_add_path(n, name);
      htable_generic_insert(inames, n);
      count++;

      /* recurse if subdirectory */
      if (recurse && S_ISDIR(s.st_mode))
	count += map_image_names(name, recurse, inames);
    }

  /* cleanup */
  closedir(d);

  /* everything OK */
  return(count);
}


/*
 * profiler_sample operations
 *
 */

static void log_sample(const profiler_sample *s, const char *msg)
{
  log_verbose("%s %5d %04x:%05d %8lx %2x",
	      (msg == NULL) ? "" : msg,
	      s->pid, s->device, s->inode, s->offset, s->mode);
}

/*
 * sorting operations
 *
 */

/* profd_sample sorter */
#define SORTER		sort_profd_samples
#define ELEMENT		profd_sample
#define SORTER_PRIVATE	1
#define ELEMENT_LT(a,b)	((a).offset < (b).offset)
#include <util/gensort.h>

/*
 * profd_image operations
 *
 */

static profd_image *profd_image_create(ushort device, ulong inode, ulong pid)
{
  profd_image *image;

  /* allocate container */
  if ((image = ALLOC_NEW(profd_image)) == NULL)
    return(NULL);

  /* initialize */
  image->device = device;
  image->inode  = inode;
  image->pid    = pid;
  htable_sample_init(&image->data);
  
  /* everything OK */
  return(image);
}

static void profd_image_destroy(profd_image *image)
{
  /* reclaim container */
  free(image);
}

static int profd_image_samples(profd_image *image, profd_sample **samples)
{
  /*
   * modifies: samples
   * effects:  Allocates a new sample array and fills it will all 
   *           samples found in image.  Updates samples to point
   *           to the allocated sample array, and returns the number
   *           of samples in the array.
   *
   */

  profd_sample *in, *out;
  int i, count, tsize;

  /* XXX violate hash-table abstraction for efficiency */
  /* XXX using htable_sample_gen would be much slower  */
  in = image->data.table;
  tsize = image->data.tsize;

  /* allocate output array */
  out = ALLOC_ARRAY(profd_sample, image->data.count);

  /* drop all entries with zero counts (empty/deleted or useless) */
  count = 0;
  for (i = 0; i < tsize; i++) 
    if (in[i].count != 0) 
      out[count++] = in[i];

  /* everything OK */
  *samples = out;
  return(count);
}

static void profd_write_profile(profd *p, profd_image *image)
{
  line_buf label, fname;
  profd_sample *samples;
  int i, nsamples;
  ulong id, total;
  iname *n, key;
  FILE *stream;
  char *name;

  /* defaults */
  name = PROFD_NAME_UNKNOWN;
  id = PROFD_IMAGE_ID_INVALID;

  /* lookup image name, if any */
  key.device = image->device;
  key.inode  = image->inode;
  if ((n = htable_generic_find(&p->inames, &key)) != NULL)
    {
      name = string_squeeze(n->name, INAME_INVALID_CHARS);
      id = n->id;
    }

  /* construct pid label, if any */
  label[0] = '\0';
  if (image->pid != PROFILER_PID_NONE)
    (void) sprintf(label, "-%05ld", image->pid);

  /* construct profile name */
  (void) sprintf(fname,
		 "%s/%s-%08lx%s.prof", 
		 p->db_path,
		 name, 
		 id,
		 label);

  /* reclaim storage */
  if (n != NULL)
    string_destroy(name);

  /* open profile */
  stream = fopen(fname, "w");

  /* write profile header */
  (void) fprintf(stream, 
		 "version\t%s\n"
		 "image\t%08lx\n"
		 "name\t%s\n"
		 "device\t%04x\n"
		 "inode\t%ld\n"
		 "freq\t%ld\n"
		 "random\t%d\n",
		 PROFD_PROFILE_VERSION,
		 id,
		 iname_name(n),
		 image->device,
		 image->inode,
		 p->frequency,
		 p->randomize ? 1 : 0);

  /* report paths, if any */
  if (n != NULL)
    for (i = 0; i < n->paths.length; i++)
      (void) fprintf(stream, "path\t%s\n", n->paths.list[i]);

  /* report pid, if any */
  if (image->pid != PROFILER_PID_NONE)
    (void) fprintf(stream, "pid\t%ld\n", image->pid);

  /* extract samples from image, sort by offset */
  nsamples = profd_image_samples(image, &samples);
  sort_profd_samples(samples, nsamples);

  /* compute total sample count */
  total = 0;
  for (i = 0; i < nsamples; i++)
    total += samples[i].count;
  
  /* report total, end header with samples keyword */
  (void) fprintf(stream, 
		 "total\t%lu\n"
		 "samples\n",
		 total);

  /* emit samples to profile */
  for (i = 0; i < nsamples; i++)
    (void) fprintf(stream, "%lx %ld\n",
		   samples[i].offset,
		   samples[i].count);
  
  /* cleanup */
  free(samples);
  (void) fclose(stream);
}

static uint profd_image_hash(profd_image *image)
{
  /* 
   * modifies: nothing
   * effects:  Returns a hash value for image.
   *
   */

  return((uint) image->inode);
}

static int profd_image_match(profd_image *a, profd_image *b)
{
  /*
   * modifies: nothing
   * effects:  Returns TRUE iff a and b have the same device, inode, and pid.
   *
   */

  return((a->device == b->device) &&
	 (a->inode  == b->inode)  &&
	 (a->pid    == b->pid));
}

/*
 * profd operations
 *
 */

static void profd_merge_samples(profd *p,
				const profiler_sample *samples,
				int nsamples)
{
  profd_image key, *image;
  int i;

  for (i = 0; i < nsamples; i++)
    {
      const profiler_sample *s = &samples[i];
      profd_sample old, new;

      /* find image, create if necessary */
      key.device = s->device;
      key.inode  = s->inode;
      key.pid    = (p->by_pid ? s->pid : PROFILER_PID_NONE);
      if ((image = htable_generic_find(&p->images, &key)) == NULL)
	{
	  image = profd_image_create(key.device, key.inode, key.pid);
	  htable_generic_insert(&p->images, image);
	}

      /* create new sample */
      new.offset = s->offset;
      new.count  = s->count;

      /* merge with existing sample, if any */
      old = htable_sample_find(&image->data, new);
      new.count += old.count;
      (void) htable_sample_insert(&image->data, new);
    }
}

static void profd_read_samples(profd *p)
{
  profiler_sample samples[PROFD_READ_NSAMPLES];
  int nbytes, nsamples;

  nbytes   = read(p->fd, (void *) samples, sizeof(samples));
  nsamples = nbytes / sizeof(profiler_sample);
  if ((nsamples * sizeof(profiler_sample)) != nbytes)
    warning("profd_read_samples: partial sample (nbytes=%d)", nbytes);
  /* log first sample in batch */
  if (PROFD_DEBUG)
    log_sample(&samples[0], NULL);

  profd_merge_samples(p, samples, nsamples);
}

static void profd_write_profiles(profd *p)
{
  htable_generic_gen gen_image;
  profd_image *image;

  htable_generic_gen_init(&gen_image, &p->images);
  while ((image = htable_generic_gen_next(&gen_image)) != NULL)
    profd_write_profile(p, image);
}

static iname *profd_find_module(profd *p, const char *name)
{
  htable_generic_gen gen;
  int suffix_len, i;
  line_buf suffix;
  iname *n;

  /* heuristic conversion to path suffix */
  (void) sprintf(suffix, "/%s.o", name);
  suffix_len = STRING_LENGTH(suffix);

  /* find matching iname */
  htable_generic_gen_init(&gen, &p->modnames);
  while ((n = htable_generic_gen_next(&gen)) != NULL)
    for (i = 0; i < n->paths.length; i++)
      {
	char *path = n->paths.list[i];
	int path_len = STRING_LENGTH(path);

	if ((path_len >= suffix_len) &&
	    (STRING_EQUAL(suffix, &path[path_len - suffix_len])))
	  return(n);
      }

  /* not found */
  return(NULL);
}

static void profd_write_modules(profd *p, FILE *stream)
{
  /*
   * modifies: stream
   * effects:  Writes image name and id for each loaded kernel
   *           module to stream.
   *
   */

  line_buf line;
  FILE *mods;

  /* fail if unable to obtain module list */
  if ((mods = fopen("/proc/modules", "r")) == NULL)
    return;

  /* scan all loaded modules */
  while (fgets(line, sizeof(line), mods) != NULL)
    {
      iname *n;

      /* extract module name */
      char *s = line;
      while ((*s != '\0') && (!isspace(*s)))
	s++;
      *s = '\0';
      
      if ((n = profd_find_module(p, line)) != NULL)
	(void) fprintf(stream, "%s\t%08lx\n", line, n->id);
    }
  
  /* cleanup */
  (void) fclose(mods);
}

static void append_file_contents(FILE *stream, const char *path)
{
  line_buf line; 
  FILE *f;

  if ((f = fopen(path, "r")) != NULL)
    {
      /* copy file contents to stream */
      while (fgets(line, sizeof(line), f) != NULL)
	fputs(line, stream);

      /* cleanup */
      (void) fclose(f);
    }
}

static void profd_write_database(profd *p)
{
  line_buf fname;
  FILE *stream;

  /* write profiles */
  profd_write_profiles(p);

  /* write kernel symbol data */
  (void) sprintf(fname, "%s/kernel.syms", p->db_path);
  if ((stream = fopen(fname, "w")) != NULL)
    {
      /* append kernel module info */
      (void) fprintf(stream, "kmods\n");
      profd_write_modules(p, stream);

      /* append kernel version */
      (void) fprintf(stream, "kversion\n");
      append_file_contents(stream, "/proc/version");

      /* append kernel symbols */
      (void) fprintf(stream, "ksyms\n");
      append_file_contents(stream, "/proc/ksyms");

      /* cleanup */
      (void) fclose(stream);
    }
}

static int profd_setup(profd *p)
{
  profiler_ctl ctl;

  /* specify sampling frequency */
  ctl.frequency = p->frequency;
  ctl.randomize = p->randomize;

  return(ioctl(p->fd, PROFILER_SET_CTL, &ctl));
}

static void profd_map_inames(profd *p)
{
  int count;
  iname *n;

  /* add special inames */
  n = iname_create(PROFILER_DEVICE_NONE,
		   PROFILER_INODE_KERNEL,
		   PROFD_IMAGE_ID_INVALID,
		   PROFD_NAME_KERNEL);
  htable_generic_insert(&p->inames, n);
  n = iname_create(PROFILER_DEVICE_NONE,
		   PROFILER_INODE_NONE,
		   PROFD_IMAGE_ID_INVALID,
		   PROFD_NAME_NOMAP);
  htable_generic_insert(&p->inames, n);

  /* scan filesystems to collect names */
  count = map_image_names("/", 1, &p->inames);
  log_verbose("profd_map_inames: %d total", count);

  /* scan particular dirs for module names */
  count = map_image_names("/lib/modules", 1, &p->modnames);
  log_verbose("profd_map_inames: %d modules", count);
}

static profd *profd_create(const char *db_path, 
			   ulong frequency,
			   int randomize,
			   int by_pid)
{
  profd *p;
  int fd;

  /* fail if db_path doesn't exist */
  if (!dir_exists(db_path))
    {
      warning("directory %s does not exist", db_path);
      return(NULL);
    }

  /* open profiling device */
  if ((fd = open("/dev/prof", O_RDONLY)) < 0)
    {
      /* issue warning and fail */
      warning("unable to open /dev/prof");
      return(NULL);
    }

  /* allocate container */
  if ((p = ALLOC_NEW(profd)) == NULL)
    return(NULL);

  /* initialize */
  p->db_path = string_clone(db_path);
  p->fd = fd;
  p->frequency = frequency;
  p->randomize = randomize;
  p->by_pid = by_pid;
  htable_generic_init(&p->images, 
		      (htable_generic_hasher) profd_image_hash,
		      (htable_generic_comparer) profd_image_match);
  htable_generic_init(&p->inames, 
		      (htable_generic_hasher) iname_hash,
		      (htable_generic_comparer) iname_match);
  htable_generic_init(&p->modnames, 
		      (htable_generic_hasher) iname_hash,
		      (htable_generic_comparer) iname_match);
  
  /* everything OK */
  return(p);
}

static void profd_destroy(profd *p)
{
  htable_generic_gen gen;
  profd_image *image;
  iname *n;

  /* close profiling device */
  (void) close(p->fd);

  /* reclaim paths */
  string_destroy(p->db_path);

  /* reclaim images */
  htable_generic_gen_init(&gen, &p->images);
  while ((image = htable_generic_gen_next(&gen)) != NULL)
    profd_image_destroy(image);
  htable_generic_destroy(&p->images);

  /* reclaim inames */
  htable_generic_gen_init(&gen, &p->inames);
  while ((n = htable_generic_gen_next(&gen)) != NULL)
    iname_destroy(n);
  htable_generic_destroy(&p->inames);

  /* reclaim modnames */
  htable_generic_gen_init(&gen, &p->modnames);
  while ((n = htable_generic_gen_next(&gen)) != NULL)
    iname_destroy(n);
  htable_generic_destroy(&p->modnames);

  /* reclaim container */
  free(p);
}

/*
 * main program
 *
 */

static void usage(const char *program)
{
  /* log usage */
  (void) fprintf(stderr, 
                 "usage: %s: [options] <db>\n"
                 "\t-v | -verbose  ; Verbose logging\n"
		 "\t-p | -pids     ; Separate per-process profiles\n"
		 "\t-f | -freq <n> ; Sampling frequency (per sec) [%d]\n"
		 "\t-r | -rnd  <n> ; Randomize sampling frequency [%d]\n",
                 program,
		 PROFILER_FREQ_DEFAULT,
		 PROFILER_RND_DEFAULT);

  /* terminate program */
  exit(EXIT_FAILURE);
}

int main(int argc, char **argv)
{
  int optind, by_pid, randomize;
  ulong frequency;
  char *db_path;
  profd *p;

  /* initialize logging */
  log_set_name("profd");
  log_set_stream(stderr);

  /* defaults */
  frequency = PROFILER_FREQ_DEFAULT;
  randomize = PROFILER_RND_DEFAULT;
  by_pid = 0;

  /* parse command-line args */
  if (argc < 2)
    usage(argv[0]);
  for (optind = 1; (optind < argc) && (argv[optind][0] == '-'); optind++)
    {
      char *option = argv[optind];
      char *optarg = (optind + 1 < argc) ? argv[optind + 1] : NULL;
      
      /* -v | -verbose */
      if (STRING_EQUAL(option, "-v") || STRING_MATCH(option, "-verbose"))
	{
	  /* enable verbose logging */
	  log_set_verbose(1);
	  continue;
	}
      
      /* -p | -pids */
      if (STRING_EQUAL(option, "-p") || STRING_MATCH(option, "-pids"))
	{
	  by_pid = 1;
	  continue;
	}

      /* -f | -freq <n> */
      if (STRING_EQUAL(option, "-f") || STRING_MATCH(option, "-freq"))
        if (optarg != NULL)
          {
            frequency = atoi(optarg);
            optind++;
            continue;
          }      

      /* -r | -rnd <n> */
      if (STRING_EQUAL(option, "-r") || STRING_MATCH(option, "-rnd"))
	if (optarg != NULL)
	  {
	    randomize = atoi(optarg);
	    optind++;
	    continue;
	  }      

      /* -s | -skip <dir> */
      if (STRING_EQUAL(option, "-s") || STRING_MATCH(option, "-skip"))
	if (optarg != NULL)
	  {
            int i;
            for (i = 0; i < (sizeof(skip_paths)/sizeof(char *)); i++) {
              if (skip_paths[i] == NULL) {
                skip_paths[i] = strdup(optarg);
                break;
              }
            }
	    optind++;
	    continue;
	  }      

      /* unrecognized option */
      usage(argv[0]);
    }

  /* parse profile database */
  if ((argc - optind) != 1)
    usage(argv[0]);
  db_path = argv[optind];


  log_verbose("initializing profd database\n");
  /* initialize */
  if ((p = profd_create(db_path, frequency, randomize, by_pid)) == NULL)
    fatal("initialization failed");

  log_verbose("collecting image names\n");
  /* collect image names */
  profd_map_inames(p);

  /* install signal handlers */
  log_verbose("installing signal handlers\n");
  (void) signal(SIGINT,  profd_signal_terminate);
  (void) signal(SIGTERM, profd_signal_terminate);
  (void) signal(SIGQUIT, profd_signal_terminate);  

  log_verbose("setting up control parameters\n");
  /* specify control parameters */
  if (profd_setup(p) != 0)
    fatal("unable to control sampling rate");

  log_verbose("processing samples\n");
  /* process driver samples */
  while (!profd_terminate)
    profd_read_samples(p);

  log_verbose("saving profiles\n");
  /* write out profiles */
  log_verbose("saving profiles...");
  profd_write_database(p);

  /* cleanup */
  profd_destroy(p);

  /* everything OK */
  return(EXIT_SUCCESS);
}
