/*
 * Wince restoration:
 * Copped from 
 * $Id: reflash-mod.c,v 1.9 2001/08/14 21:25:55 davep Exp $
 *
 * Character-device access to raw MTD devices.
 *
 * by davep, 24-Apr-2001
 * 
 */


#include <linux/mtd/compatmac.h>
#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/module.h>
#include <linux/mtd/mtd.h>
#include <linux/mtd/partitions.h>
#include <linux/malloc.h>
#include <linux/proc_fs.h> 

#include "reflash.h"
#include "reflash-defs.h"

#define FR_PART_NAME	"RESTORATION partition(rw)"

/* Our partition node structure */
struct mtd_part {
	struct mtd_info mtd;
	struct mtd_info *master;
	u_long offset;
	int index;
	struct list_head list;
};

/*
 * Given a pointer to the MTD object in the mtd_part structure, we can retrieve
 * the pointer to that structure with this macro.
 */
#define PART(x)  ((struct mtd_part *)(x))
/*
 * the master_mtd is needed in to add new partitions :-<
 */

#define	mtd_get_master(mtd) PART(mtd)->master
/*  #define	mtd_get_offset(mtd) PART(mtd)->offset */

static int  fr_debug_level = 99;

#define debug(level,format...)	if (fr_debug_level > level) printk(format)
#define err_report(format...) printk(format)

#define	FLASH_SIZE(rip) ((rip)->master_mtd->size)

/*
************************************************************************
*
* 
* 
************************************************************************
*/
typedef struct restore_info_s
{
    /*
     * min and max addrs present amongst all of the
     * mtd devs / parts.
     */
    
    u_int32_t	min_addr;
    u_int32_t	max_addr;
    
    struct mtd_info*	master_mtd;
    
    struct mtd_info*	mtd;
}
restore_info_t;
static	restore_info_t	restore_info;

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static ssize_t
write_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params);

static ssize_t
verify_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params);

static int
unprotect_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params);

static int
protect_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params);

#define	MAX_PARAMS  (32)
flash_restore_params_t	param_list[MAX_PARAMS];
int params_index = 0;

static void erase_callback(struct erase_info *done)
{
    wait_queue_head_t *wait_q = (wait_queue_head_t *)done->priv;
    wake_up(wait_q);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static int
restore_erase(
    restore_info_t*	    rip,
    flash_restore_params_t* params)
{
    struct erase_info erase;
    DECLARE_WAITQUEUE(wait, current);
    wait_queue_head_t wait_q;
    int ret;

    debug(1, "restore_erase():\n");
    
    /*
     * let's erase the flash block.
     */
    
    init_waitqueue_head(&wait_q);
    erase.mtd = rip->mtd;
    erase.callback = erase_callback;
    erase.addr = params->flash_offset;
    
    if (params->buffer_len == REST_OF_FLASH) {
	erase.len = FLASH_SIZE(rip) - params->flash_offset;
	debug(1, "restore_erase(), REST_OF_FLASH: 0x%x\n", erase.len);
    }
    else
	erase.len = params->buffer_len;
    
    erase.priv = (u_long)&wait_q;
    
    set_current_state(TASK_INTERRUPTIBLE);
    add_wait_queue(&wait_q, &wait);
    
    ret = MTD_ERASE(rip->mtd, &erase);
    if (ret) {
	set_current_state(TASK_RUNNING);
	remove_wait_queue(&wait_q, &wait);
	printk (KERN_WARNING "mtdblock: erase of region [0x%lx, 0x%lx] "
		"on \"%s\" failed\n",
		params->flash_offset, params->buffer_len, rip->mtd->name);
	return ret;
    }
    
    schedule();  /* Wait for erase to finish. */
    remove_wait_queue(&wait_q, &wait);

    return 0;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static int
begin_restore(
    void)
{
    int	    i;
    flash_restore_params_t* pp;
    ssize_t rc;
    int	    ret;

    debug(1, "begin_restore():\n");
    for (i = 0; i < params_index; i++) {
	pp = &param_list[i];
	
	switch (pp->command) {
	    case FR_CMD_PROTECT:
		debug(1, "Protect command, vaddr: 0x%lx, len: 0x%lx, "
		      "offset: 0x%lx\n",
		      param_list[i].buffer_vaddr,
		      param_list[i].buffer_len, param_list[i].flash_offset);

		if ((ret = protect_segment(&restore_info, pp)) != 0)
		{
		    err_report("protect_segment failed, ret: 0x%x\n", ret);
		    goto out;
		}
		break;
	    
	    case FR_CMD_UNPROTECT:
		debug(1, "Unprotect command, vaddr: 0x%lx, len: 0x%lx, "
		      "offset: 0x%lx\n",
		      param_list[i].buffer_vaddr,
		      param_list[i].buffer_len, param_list[i].flash_offset);

		if ((ret = unprotect_segment(&restore_info, pp)) != 0)
		{
		    err_report("unprotect_segment failed, ret: 0x%x\n", ret);
		    goto out;
		}
		break;

	    case FR_CMD_WRITE:
		debug(1, "Write command, vaddr: 0x%lx, len: 0x%lx, "
		      "offset: 0x%lx\n",
		      param_list[i].buffer_vaddr,
		      param_list[i].buffer_len, param_list[i].flash_offset);

		if ((rc = write_segment(&restore_info, pp)) != pp->buffer_len)
		{
		    err_report("write_segment failed, rc: 0x%x\n", rc);
		    ret = -EIO;
		    goto out;
		}
		break;

	    case FR_CMD_VERIFY:
		debug(1, "Verify command, vaddr: 0x%lx, len: 0x%lx, "
		      "offset: 0x%lx\n",
		      param_list[i].buffer_vaddr,
		      param_list[i].buffer_len, param_list[i].flash_offset);

		if ((rc = verify_segment(&restore_info, pp)) != pp->buffer_len)
		{
		    err_report("verify_segment failed, rc: 0x%x\n", rc);
		    ret = -EIO;
		    goto out;
		}
		break;

	    case FR_CMD_ERASE:
		debug(1, "Erase command, len: 0x%lx, offset: 0x%lx\n",
		      param_list[i].buffer_len, param_list[i].flash_offset);
		rc = restore_erase(&restore_info, pp);
		if (rc != 0) {
		    err_report("restore_erase() failed, rc: 0x%x\n", rc);
		    ret = -EIO;
		    goto out;
		}
		break;

	    case FR_CMD_RESTORE:
		debug(1, "Restore command\n");
		break;
	}
    }

    return (0);

  out:
    return (ret);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
static ssize_t
restore_write(
    struct file *file,
    const char *buffer,
    unsigned long count,
    void *data)
{
    int	    rc;
    flash_restore_params_t* params = (flash_restore_params_t*)buffer;

    /* validate packet */
    if (params->version != REFLASH_IF_VERSION) {
	err_report("restore_write(), version mismatch, req: %d, mine: %d\n",
		   params->version, REFLASH_IF_VERSION);
	return (-EINVAL);
    }

    if (params->size != sizeof(*params)) {
	err_report("restore_write(), size mismatch, req: %d, mine: %d\n",
		   params->size, sizeof(*params));
	return (-EINVAL);
    }

    if (params->command == FR_CMD_RESTORE) {
	rc = begin_restore();
	params_index = 0;
	if (rc != 0)
	    return (rc);
    }
    else if (params->command == FR_CMD_RESET) {
	params_index = 0;
    }
    else {
	if (params_index >= MAX_PARAMS) {
	    err_report("restore_write(): too many param blocks, max: %d\n",
		       MAX_PARAMS);
	    return (-ENOMEM);
	}
	param_list[params_index++] = *params;
    }
    
    return sizeof(*params);
}

/*
 * we'll use the existing mtd partitions to determine the total
 * flash size.
 */
static int
find_flash_size(
    restore_info_t* rip)
{
    int	devnum;
    struct mtd_info *mtd;
    
    /* dump 'em all before we do anything.... */
    
    for (devnum = 0; devnum < MAX_MTD_DEVICES; devnum++) {
	mtd = get_mtd_device(NULL, devnum);
	
	if (!mtd) {
#if 0
	    debug(1, "find_flash_size(), "
		  "no mtd for devnum: %d\n", devnum);
#endif
	    continue;
	}

	rip->master_mtd = mtd_get_master(mtd);
	debug(1, "find_flash_size(), name>%s<, master->size: 0x%x\n",
	      mtd->name,
	      rip->master_mtd->size);
	    
	debug(1, "find_flash_size(), mtd->size: 0x%x\n", mtd->size);
	put_mtd_device(mtd);
    }
    
    rip->min_addr = 0;
    rip->max_addr = FLASH_SIZE(rip) - 1;
    
    debug(1, "find_flash_size(), len: 0x%x\n",
	  rip->max_addr - rip->min_addr + 1);
    
    return (0);
}

/*
 * add a partition that emcompases the entire flash device so that
 * we can write to any part of it easily
 */

static struct mtd_partition restore_partitions[] =
{
    {
	name: FR_PART_NAME,
	size: 0,		/* to be filled in  */
	offset: 0,
    }
};

/*
 * add in the partition that
 * encompases all of the other partitions.
 */
static void
add_encompassing_partition(
    restore_info_t* rip)
{
    /* patch in the size */
    restore_partitions[0].size = rip->max_addr - rip->min_addr + 1;
    
    debug(1, "add_encompassing_partition(), len: 0x%x\n",
	  restore_partitions[0].size);
    
    add_mtd_partitions(rip->master_mtd, restore_partitions, 1);
}

/*
 * find the dev we added which describes all of flash.
 */
static struct mtd_info*
find_encompassing_partition(
    restore_info_t* rip)
{
    struct mtd_info *mtd;
    int	    devnum;
    int	    e_devnum = -1;
    
    /*
     * for all possible devices
     */
    debug(1, "find_encompassing_partition(), min: 0x%x, max: 0x%x\n",
	  rip->min_addr, rip->max_addr);
    
    for (devnum = 0; devnum < MAX_MTD_DEVICES; devnum++) {
	mtd = get_mtd_device(NULL, devnum);
	
	if (!mtd) {
#if 0
	    debug(1, "find_encompassing_partition(), "
		  "no mtd for devnum: %d\n", devnum);
#endif
	    continue;
	}
	
	debug(1, "mtd: size: 0x%x\n", mtd->size);
	
	if (strcmp(mtd->name, FR_PART_NAME) == 0) {
	    if (e_devnum != -1) {
		err_report("find_encompassing_partition(), more than one "
			   "encompassing part found.\n");
		return NULL;
	    }
	    
	    e_devnum = devnum;
	}
	
	put_mtd_device(mtd);
    }
    
    if (e_devnum == -1) {
	err_report("find_encompassing_partition(), "
	      "could not find the encompassing part.\n");
	return (NULL);
    }
    
    return (get_mtd_device(NULL, e_devnum));
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static int
init_our_part(
    restore_info_t* rip)
{
    int	    rc;
    
    rc = find_flash_size(rip);
    
    if (rc != 0) {
	err_report("find_flash_size() failed, rc: %d\n", rc);
	return (rc);
    }
    
    /* see if our part is already there... */
    rip->mtd = find_encompassing_partition(rip);
    if (rip->mtd == NULL) {
	add_encompassing_partition(rip);
	
	rip->mtd = find_encompassing_partition(rip);
	
	if (rip->mtd == NULL) {
	    err_report("find_encompassing partition(), didn't.\n");
	}
    }
    
    return ((rip->mtd == NULL) ? ENODEV : 0);
}

#define	PROCFS_NODE_NAME    "sys/flash_restore"
static struct proc_dir_entry *restore_proc_entry = NULL;

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static int
init_proc_entry(
    void)
{
    int	    rc;
    
    restore_proc_entry = create_proc_entry(PROCFS_NODE_NAME, 0, NULL);
    
    if (restore_proc_entry) {
	restore_proc_entry->write_proc = restore_write;
	rc = 0;
    }
    else
	rc = ENOENT;
    
    return (rc);
}

/*====================================================================*/

#define MAX_KMALLOC_SIZE 0x20000

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static ssize_t
write_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params)
{
    struct mtd_info *mtd = rip->mtd;
    size_t retlen;
    size_t total_retlen=0;
    size_t count = params->buffer_len;
    loff_t off = params->flash_offset;
    int ret=0;
    int len;
    char *kbuf;
    char*	buf = (char*)params->buffer_vaddr;
    
    DEBUG(MTD_DEBUG_LEVEL0,"write_segment\n");
    
    if (off == mtd->size)
	return -ENOSPC;
    
    if (off + count > mtd->size)
	count = mtd->size - off;
    
    if (!count)
	return 0;
    
    kbuf=kmalloc(MAX_KMALLOC_SIZE, GFP_KERNEL);
    if (!kbuf) {
	err_report("write_segment(): kmalloc is null\n");
	return -ENOMEM;
    }
    
    while (count) {
	if (count > MAX_KMALLOC_SIZE) 
	    len = MAX_KMALLOC_SIZE;
	else
	    len = count;
	
	if (copy_from_user(kbuf, buf, len)) {
	    kfree(kbuf);
	    return -EFAULT;
	}
	
	ret = (*(mtd->write))(mtd, off, len, &retlen, kbuf);
	if (!ret) {
	    off += retlen;
	    total_retlen += retlen;
	    count -= retlen;
	    buf += retlen;
	}
	else {
	    kfree(kbuf);
	    return ret;
	}
    }
    
    kfree(kbuf);
    
    return total_retlen;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static ssize_t
verify_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params)
{
    struct mtd_info *mtd = rip->mtd;
    size_t retlen;
    size_t total_retlen=0;
    size_t count = params->buffer_len;
    loff_t off = params->flash_offset;
    int ret=0;
    int len;
    char*   rbuf;
    char*   ubuf;
    char*   buf = (char*)params->buffer_vaddr;
    
    DEBUG(MTD_DEBUG_LEVEL0,"verify_segment\n");
    
    if (off == mtd->size)
	return -ENOSPC;
    
    if (off + count > mtd->size)
	count = mtd->size - off;
    
    if (!count)
	return 0;
    
    rbuf=kmalloc(MAX_KMALLOC_SIZE, GFP_KERNEL);
    
    if (!rbuf) {
	err_report("verify_segment(): kmalloc is null\n");
	return -ENOMEM;
    }
    ubuf = rbuf + MAX_KMALLOC_SIZE/2;
    
    while (count) {
	if (count > MAX_KMALLOC_SIZE/2) 
	    len = MAX_KMALLOC_SIZE/2;
	else
	    len = count;
	
	err_report(" ubuf: %p, buf: %p, len: 0x%x.\n",
		   ubuf, buf, len);
	if (copy_from_user(ubuf, buf, len)) {
	    err_report("verify_segment(): copy_from_user() failed.\n");
	    err_report(" ubuf: %p, buf: %p, len: 0x%x.\n",
		       ubuf, buf, len);
	    kfree(rbuf);
	    return -EFAULT;
	}
	
	ret = (*(mtd->read))(mtd, off, len, &retlen, rbuf);
	if (!ret) {
	    if (memcmp(rbuf, ubuf, len) != 0) {
		err_report("verify_segment(), memcmp failed.\n");
		kfree(rbuf);
		return -EIO;
	    }
	    
	    off += retlen;
	    total_retlen += retlen;
	    count -= retlen;
	    buf += retlen;
	}
	else {
	    err_report("verify_segment(), read failed, rc: %d.\n", ret);
	    kfree(rbuf);
	    return ret;
	}
    }
    
    kfree(rbuf);

    debug(1, "verify_segment(), buf_len: 0x%lx, total_retlen: 0x%x\n",
	  params->buffer_len, total_retlen);
    
    return total_retlen;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static int
protect_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params)
{
    struct mtd_info *mtd = rip->mtd;
    int ret=0;
    
    if (!mtd->lock) {
	err_report("Lock not supported by mtd layer.\n");
	ret = -EOPNOTSUPP;
    }
    
    else
	ret = mtd->lock(mtd, params->flash_offset, params->buffer_len);

    debug(1, "mtd->lock(0x%lx, 0x%lx), ret: %d\n",
	  params->buffer_vaddr, params->buffer_len,ret);

    return (ret);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
static int
unprotect_segment(
    restore_info_t*	    rip,
    flash_restore_params_t* params)
{
    struct mtd_info *mtd = rip->mtd;
    int ret=0;
    
    if (!mtd->unlock) {
	err_report("Unlock not supported by mtd layer.\n");
	ret = -EOPNOTSUPP;
    }
    
    else
	ret = mtd->unlock(mtd, params->flash_offset, params->buffer_len);

    debug(1, "mtd->unlock(0x%lx, 0x%lx), ret: %d\n",
	  params->buffer_vaddr, params->buffer_len,ret);

    return (ret);
}

#if LINUX_VERSION_CODE < 0x20212 && defined(MODULE)
#define init_flash_restore init_module
#define cleanup_flash_restore cleanup_module
#endif

mod_init_t init_flash_restore(void)
{
    int	    rc;
    
    if ((rc = init_proc_entry()) != 0) {
	err_report("init_proc_entry() failed, rc: %d\n", rc);
	return (rc);
    }
    
    if ((rc = init_our_part(&restore_info)) != 0) {
	err_report("init_our_part() failed, rc: %d\n", rc);
	return (rc);
    }
    
    return 0;
}

mod_exit_t cleanup_flash_restore(void)
{
    debug(1, "enter cleanup_flash_restore()\n");
    
    if (restore_proc_entry)
	remove_proc_entry(PROCFS_NODE_NAME, 0);
    
    debug(1, "exit cleanup_flash_restore()\n");
}

module_init(init_flash_restore);
module_exit(cleanup_flash_restore);
