#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>
#include <setjmp.h>
#include <sys/types.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <signal.h>
#include <linux/reboot.h>
#include <sys/reboot.h>



/*  #define	CONFIG_GNU_READLINE (1) */

#ifdef CONFIG_GNU_READLINE
#include <readline.h>
#include <history.h>
#endif

#include "reflash.h"
#include "state.h"
#include "map.h"
#include "symtab.h"
#include "reflash-defs.h"
#include "dplib/dpmacs.h"
#include "dplib/eprintf.h"
#include "dplib/fs-helpers.h"
#include "dplib/getopt-helpers.h"
#include "bootldr/partition.h"
#include "dplib/dl_list.h"
#include "dplib/bl_params.h"

#define	RZ_INDICATOR	    "@rz"
#define	RZ_INDICATOR_LEN    (sizeof(RZ_INDICATOR)-1)
#define	RZ_SEPARATOR	    ('/')
#define	RZ_ANYFILE	    ("*")
char	*proc_file = "/proc/sys/flash_restore";
int	verbose = 0;
int	proc_file_fd = -1;
FILE*	ep_log = NULL;
char	*progname;
int	quietude = 0;		/* make us absolutely quiet */
char	rz_filename[PATH_MAX];
char	rz_cmd[PATH_MAX] = "rzcat-pipe";
int	do_reboot = 1;
#define	VS0(x)			#x
#define VERSION_STRING(x)	VS0(x) 
char	version_string[] = VERSION_STRING(CVS_VERSION_TAG);

int		use_ticker = 0;
unsigned long	tick_size;
char*		params_files[] = {
    "/dev/mtdblock1",		/* non-devfs */
    "/dev/mtdblock/1"		/* devfs */
};
int num_params_files = DIM(params_files);

char*		params_size = "0x40000";
char		symbol_indicator = '=';

#define declwrdir  char wrdir[PATH_MAX] = 
#include "wrdir-def.h"

int	printed_the_queue = 0;
unsigned int	rz_wait_time = 60*120;
int	yes_to_all_prompts = 0;

/*  #define	putstr(s)   eprintf(0, 0, "%s", s) */
#define	putstr(s)   rprintf("%s", s)

#define	MAX_PARAMS  (100)
flash_restore_params_t	params_queue[MAX_PARAMS];
int params_idx = 0;

enum {
    RZ_BAD_FILENAME = 1
};

#define	LORO_LEN	"len"
#define	LORO_OFF	"offset"
#define	LORO_ASSIGNMENT	"assignment"
#define LORO_IS_LEN(w)   (strcmp(w, LORO_LEN) == 0)

void
cmdloop(
    reflash_state_t*	rsp,
    char*		arg);

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
set_params_file(
    char*   arg)
{
	    
    params_files[0] = arg;
    num_params_files = 1;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
fix_tty(void)
{
    system("stty sane");
}

#if 0
/*
************************************************************************
*
* 
* 
************************************************************************
*/
char*
basename(
    char*   name)
{
    char*   progname;
    
    progname = name + strlen(name);
    while (progname >= name && *progname != '/')
	progname--;
    ++progname;
    return (progname);
}
#endif

int jump_on_exit_code = 0;
jmp_buf	error_vector;

#define	set_error_vector(v)  {			\
    (v) = setjmp(error_vector);			\
    jump_on_exit_code = 1;			\
}

#define unset_error_vector() {			\
    jump_on_exit_code = 0;			\
}

#define	error_vector_setq()	(jump_on_exit_code != 0)

char*	global_options = "O:t:o:n:f:e:E:rb:i:w:xd:D:v:V:u:l:c:XS:W:qh:p:CQ:"
                         "s:j=:m:P:k:F:T:N:yRa:L:M:B:";

#define get_options()	global_options

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
build_params(
    flash_restore_params_t* fp,
    int			    command,
    unsigned long	    offset,
    unsigned long	    size,
    void*		    buf)
{
    fp->version = REFLASH_IF_VERSION;
    fp->size = sizeof(*fp);
    
    fp->command = command;
    fp->buffer_vaddr = (unsigned long)buf;
    fp->buffer_len = size;
    fp->flash_offset = offset;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
eprint_str(
    char*   str)
{
    if (ep_log) {
	fprintf(ep_log, "%s%s", quietude ? "[Q]: " : "", str);
	fflush(ep_log);
    }
    
    if (quietude)
	return;
    fprintf(stderr, "%s", str);
    fflush(stderr);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
eprintf(
    int		exit_code,
    int		err,
    char*	fmt,
    ...)
{
    va_list ap;  // for variable args
    char    buf[1024];
    int	    len;
    extern  char*   progname;

    if (progname)
	len = sprintf(buf, "%s: ", progname);
    else
	len = 0;
    
    if (err)
	len += sprintf(buf+len, "%s: ", strerror(err));
    
    va_start(ap, fmt); // init specifying last non-var arg
    len = vsnprintf(buf+len, sizeof(buf)-2-len, fmt, ap);
    va_end(ap); // end var args

    eprint_str(buf);
    if (len < 0 || len > (int)sizeof(buf)) {
	eprint_str(" (error msg truncated)");
    }

    if (exit_code) {
	if (jump_on_exit_code)
	    longjmp(error_vector, exit_code);
	else 
	    exit(exit_code);
    }
}
/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
rprintf(
    char*   fmt,
    ...)
{
    va_list ap;  // for variable args
    int	    rc;
    
    if (quietude)
	return (-1);

    va_start(ap, fmt); // init specifying last non-var arg
    rc = vprintf(fmt, ap);
    fflush(stdout);
    return (rc);
}

#define	ARGV_MAX    (100)
/*
************************************************************************
*
* 
* 
************************************************************************
*/
pid_t
forkit(
    char*   procname,
    ...)
{
    pid_t   pid;
    char*   argv[ARGV_MAX];
    int	    argc;
    va_list ap;  // for variable args
    int	    num_retries = 0;

    /* build argv */
    argc = 1;
    va_start(ap, procname); // init specifying last non-var arg
    argv[0] = procname;
    
    for (argc = 1; argc < ARGV_MAX - 1; argc++) {
	argv[argc] = va_arg(ap, char*);
	if (!argv[argc])
	    break;
    }

  again:
    pid = fork();
    
    if (pid < 0) {
	eprintf(2, 0, "forkit, fork() failed\n");
    }
    else if (pid == 0) {
	/* child */
	execvp(procname, argv);
	eprintf(2, errno, "execvp(%s) failed.\n", procname);
    }
    else {
	/* parent */
	if (verbose)
	    eprintf(0, 0, "forkit() OK, child: %d.\n", pid);

	/*
	 * if rzCOMMAND fails by COMMAND returning non-zero,
	 * and if the file being sent is very small,
	 * (my guess is small enough to be rx'd in a single
	 * read or buffer) then zmodem exits via an unhandled
	 * signal.
	 * And the next invocation dies quickly.  So I stuck
	 * in this hack to try and make the system recover
	 * and try again.
	 */
	{
	    int	    stat;
	    int	    wpid;
	    
	    sleep(1);
	    wpid = waitpid(pid, &stat, WNOHANG);
	    if (wpid != 0) {
		fix_tty();
		if (++num_retries < 3) {
		    if (verbose)
			eprintf(0, 0, "%d died prematurely, retrying.\n",
				procname);
		    goto again;
		}
		else
		    eprintf(2, 0, "Cannot start up %s\n", procname);
	    }
	}
    }
    return (pid);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
option_needs_arg(
    char    c)
{
    char*   p;
    
    if (c == ':')
	eprintf(2, 0, "Illegal option: %c\n", c);
    if (c == '\0')
	eprintf(2, 0, "Illegal option: '\\0'\n", c);

    if ((p = index(get_options(), c)) == NULL)
	eprintf(2, 0, "Illegal option: %c\n", c);

    return ((*(p+1) == ':'));
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
close_proc_file(
    void)
{
    if (proc_file_fd != -1) {
	close(proc_file_fd);
	proc_file_fd = -1;
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
do_open_proc_file(
    char*   file_name)
{
    if (proc_file_fd < 0) {
	if ((proc_file_fd = open(file_name, O_WRONLY)) < 0) {
	    eprintf(2, errno, "cannot open proc_file: %s\n", proc_file);
	}
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
open_proc_file(
    void)
{
    return (do_open_proc_file(proc_file));
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
validate_len_or_offset(
    reflash_state_t*	rsp,
    char*		which,
    unsigned long	tlen)
{
    if (LORO_IS_LEN(which) && tlen == REST_OF_FLASH)
	return;
    
    if (tlen & rs_get_word_mask(rsp))
	eprintf(2, 0, "%s (%lu) is not a multiple of word size (%d).\n",
		which, tlen, (rs_get_word_mask(rsp)+1));
}

void
set_all_param_vars(
    char*   basename,
    unsigned long   base,
    unsigned long   size,
    unsigned long   flags)
{
    char		pname_buf[128];
    
    symbol_set_num(basename, base);

    sprintf(pname_buf, "%s.base", basename);
    symbol_set_num(pname_buf, base);
    
    sprintf(pname_buf, "%s.size", basename);
    symbol_set_num(pname_buf, size);
    
    sprintf(pname_buf, "%s.flags", basename);
    symbol_set_num(pname_buf, flags);
}

typedef struct defpart_s
{
    char*	    name;
    unsigned long   base;
    unsigned long   size;
    unsigned long   flags;
}
defpart_t;

defpart_t  defparts[] =
{
    {"bootldr",  0x00000, 0x40000, LFR_PATCH_BOOTLDR},
    {"params",   0x40000, 0x40000, 0},
    {"kernel",   0x80000, 0xc0000, 0},
#if 0
    {"root",	0x140000, (16<<20) - 0x140000, 0},
#endif
};

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
add_common_symbols(
    void)
{
    defpart_t*	pp;
    int		i;

    for (i = 0; i < DIM(defparts); i++) {
	pp = &defparts[i];

	set_all_param_vars(pp->name, pp->base, pp->size, pp->flags);
    }

    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
char*
find_params_file(
    void)
{
    int	    i;
    struct stat	stat_buf;

    for (i = 0; i < num_params_files; i++) {
	if (stat(params_files[i], &stat_buf) == 0)
	    return (params_files[i]);
    }

    return (NULL);
}
	
/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
read_partition_table_and_add_to_symtab(
    void)
{
    blp_params_t	params;
    dl_iter_t		iter;
    blp_part_t*		item;
    char*		params_file;

    params_file = find_params_file();
    if (!params_file)
	return (-1);
	
    memset((char*)&params, 0x00, sizeof(params));
    dl_list_init(&params.partition_list);
    dl_list_init(&params.variable_list);
    
    blp_set_sector_size_s(&params, params_size);

    /* zero for sec len --> do not read params */
    if (blp_get_sector_size(&params) == 0) {
	eprintf(0, 0, "Not reading params sector.\n");
	return (0);
    }
    
    blp_get_partition_table(params_file, &params);

    /*
     * add all pieces of the partition entry to the
     * symtable.
     * part (== base), part.size, part.flags
     */
    
    for (item = BLP_PARAM_FIRST_PART(&params, &iter);
	 item;
	 item = BLP_PARAM_NEXT_PART(&iter)) {

	set_all_param_vars(BLP_PART_NAME(item),
			   BLP_PART_BASE(item),
			   BLP_PART_SIZE(item),
			   BLP_PART_FLAGS(item));
    }
    
    blp_free_all(&params);

    return (0);
}


int
is_a_var(
    char*   s)
{
    return (s[0] == symbol_indicator);
}

char*
var_to_name(
    char*   arg)
{
    return (arg + 1);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
unsigned long
process_symbol(
    char*   symname)
{
    unsigned long   tlen;
    int		    rc;
    static int	    have_read_part_tab = 0;

    if (!have_read_part_tab) {
	read_partition_table_and_add_to_symtab();
	have_read_part_tab = 1;
    }
    
    rc = symbol_num_val(symname, &tlen);
    if (rc == 0)
	return (tlen);

    /*
     * symbol not in the table.
     * we want to try the partition table, so we need to
     * read that in if we haven't already.
     */
#ifndef CONFIG_UNSET_SYMBOL_IS_OK
    eprintf(1, 0, "Cannot find symbol >%s<\n", symname);
#endif
    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
unsigned long
get_len_or_offset(
    reflash_state_t*	rsp,
    char*		which,
    char*		arg)
{
    unsigned long   tlen;
    char*	    p;

    /* check for explicit symbol reference */
    if (is_a_var(arg))
	tlen = process_symbol(var_to_name(arg));
    else if (arg[0] >= '0' && arg[0] <= '9') {
	tlen = strtoul(arg, &p, 0);
	switch (*p) {
	    case 'k':
	    case 'K':
		tlen <<= 10;
		break;
		
	    case 'm':
	    case 'M':
		tlen <<= 20;
		break;
		
	    case 's':
	    case 'S':
		tlen *= rs_get_sec_size(rsp);
		break;
	}
	validate_len_or_offset(rsp, which, tlen);
    }
    else {
	/*
	 * let "bad numbers" be assumed to be
	 * variable names.
	 * It will die or succeed based on existence of
	 * variable and CONFIG_UNSET_SYMBOL_IS_OK
	 */
	tlen = process_symbol(arg);
    }

    return (tlen);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
size_t
getlen(
    reflash_state_t*	rsp,
    char*		arg)
{
    size_t  tlen;

    if (arg[0] == 'l' || arg[0] == 'L')
	tlen = rs_get_len(rsp);
    else
	tlen = get_len_or_offset(rsp, LORO_LEN, arg);
    
    return (tlen);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
send_request(
    reflash_state_t*	    rsp,
    flash_restore_params_t* paramsp)
{
    open_proc_file();
    
    if (write(proc_file_fd, (char*)paramsp, sizeof(*paramsp)) !=
	sizeof(*paramsp)) {
	eprintf(2, errno, "params write failed.\n");
    }

    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
send_queue(
    reflash_state_t*	rsp)
{
    int	    i;
    int	    rc;

    for (i = 0; i < params_idx; i++) {
	if ((rc = send_request(rsp, &params_queue[i])) != 0)
	    eprintf(2, 0, "Send request failed.\n");
    }

    params_idx = 0;
    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
clear_queue(
    reflash_state_t*	rsp)
{
    params_idx = 0;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
execute_queue(
    reflash_state_t*	rsp)
{
    flash_restore_params_t	params;

    send_queue(rsp);
    
    build_params(&params, FR_CMD_RESTORE, RESTRICTED_BASE, 0L, NULL);
    
    send_request(rsp, &params);
    
    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
print_params(
    flash_restore_params_t* fp,
    char*		    hdr)
{
    if (!hdr)
	hdr = "";
    
    eprintf(0, 0, "%s%17s, vaddr: 0x%08lx, off: 0x%08lx, len: 0x%08lx\n",
	    hdr,
	    map_cmd_to_name(fp->command),
	    fp->buffer_vaddr, fp->flash_offset, fp->buffer_len);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
do_queue_request(
    reflash_state_t*	rsp,
    int			command,
    unsigned long	offset,
    unsigned long	size,
    void*		buf)
{
    flash_restore_params_t	params;

    if (params_idx >= MAX_PARAMS) {
	eprintf(2, 0, "Param queue is full.  Not queuing item.\n");
    }
    
    if (cmd_restrictedq(command) &&
	rs_offset_restrictedq(rsp) && offset < RESTRICTED_BASE)
	eprintf(3, 0, "Offset (0x%lx) is in the bootldr "
		"segment (0... 0x%lx).\n"
		"   *** CMD NOT QUEUED ***\n",
		offset, RESTRICTED_BASE);
    
    build_params(&params, command, offset, size, buf);
    
    if (verbose) {
	print_params(&params, "q cmd: ");
    }

    params_queue[params_idx++] = params;

    return (0);
}

int
queue_request(
    reflash_state_t*	rsp,
    int			command,
    unsigned long	offset,
    unsigned long	size,
    void*		buf)
{
    if (params_idx == 0 && rs_resetq(rsp))
	do_queue_request(rsp, FR_CMD_RESET, RESTRICTED_BASE, 0L, NULL);

    return (do_queue_request(rsp, command, offset, size, buf));
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
resize_main_buf(
    reflash_state_t*	rsp,
    size_t  size)
{
    char*   p = rs_get_buf(rsp);

    if (p)
	free(p);

    p = malloc(size);
    if (!p)
	eprintf(2, 0, "Cannot alloc main_buf of size: %d\n", size);

    /* be NOR flash friendly */
    memset(p, 0xff, size);
    rs_set_buf_size(rsp, size);
    rs_set_buf(rsp, p);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
read_data(
    reflash_state_t*	rsp,
    int			read_fd,
    char*		buf,
    size_t		desired,
    int			till_eof, /* read till eof, no target size */
    unsigned long*	num_readp,
    int*		set_sizep)
{
    ssize_t	num_read;
    size_t	total_read = 0;
    size_t	resid = desired;
    char*	upto;
    unsigned	num_units = 0;
    unsigned	tmp_units;

    if (till_eof)
	upto = "up to ";
    else
	upto = "";

    eprintf(0, 0, "read_data, reading: %s%d bytes\n", upto, resid);

    *set_sizep = 0;

    while (1) {
	num_read = read(read_fd, buf, resid);
    
	/* all errors are fatal */
	if (num_read < 0)
	    eprintf(2, errno, "read failed.\n");

	total_read += num_read;
	resid -= num_read;
	buf += num_read;

	if (use_ticker) {
	    tmp_units = total_read / tick_size;
	    while (tmp_units > num_units) {
		rprintf(".");
		fflush(stdout);
		num_units++;
	    }
	}
	
	/* did we satsify our read ??? */
	if (resid == 0) {
	    if (!till_eof)
		break;
	    /*
	     * we've read all we were asked to.
	     * If till_eof is set, then desired is really the max
	     * that we can fit into our buffer.
	     * If resid is zero, that means we've hit the limit
	     * we can hold.  If there is more to read then we have
	     * a problem.  So we probe for EOF by reading one more.
	     */
	    num_read = read(read_fd, buf, 1);
	    if (num_read == 0) { /* EOF */
		*set_sizep = 1;
		*num_readp = total_read;
		break;
	    }
	
	    eprintf(2, 0, "end of buffer hit with more data present. "
		    "num_read: %d\n", num_read);
	}
	
	if (num_read == 0) {
	    /* till eof means eof is OK */
	    if (till_eof) {
		if (verbose)
		    eprintf(0, 0, "read_data, total_read: %d bytes\n",
			    total_read);
		*set_sizep = 1;
		*num_readp = total_read;
		break;
	    }
	
	    /*
	     * not in till_eof mode, this is an error since we didn't
	     * read everything we were asked to.
	     */
	    eprintf(2, 0, "premature EOF, read %d of %d bytes.\n",
		    total_read, desired);
	}
    }
    rprintf("\n");
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
get_rz_status(
    void)
{
    char    buf[1024];
    char*   bufp;
    int	    resid;
    int	    rc = -1;
    int	    fd;
    
    sprintf(rz_filename, "%s/%s", wrdir, "re-cmd-pipe");
    if (verbose)
	eprintf(0, 0, "opening %s to read\n", rz_filename);
    if ((fd = open(rz_filename, O_RDONLY)) < 0)
	eprintf(2, errno, "Failed to open command pipe >%s<\n", rz_filename);

    eprintf(0, 0, "opened %s\n", rz_filename);

    resid = sizeof(buf)-1;
    bufp = buf;
    while (resid) {
	rc = read(fd, bufp, resid);
	if (rc < 0)
	    eprintf(2, errno, "Error reading rz status.\n");

	if (rc == 0) {
	    int	    len;
	    
	    /* eof */
	    *bufp = '\0';
	    len = strlen(buf)-1;
	    if (buf[len] == '\n')
		buf[len] = '\0';
	    if (strcmp("-BADNAME", buf) == 0) {
		rc = RZ_BAD_FILENAME;
		goto out;
	    }
	    
	    if (strcmp("+OK", buf) == 0) {
		rc = 0;
		goto out;
	    }

	    fix_tty();
	    eprintf(2, 0, "Unknown response from rz >%s<\n", buf);
	}

	bufp += rc;
	resid -= rc;
    }

    eprintf(2, 0, "Response from rz is too long.\n");

  out:

    close(fd);
    return(rc);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
cmd_pipe_write(
    char*   msg,
    int	    len)
{
    int	    fd;
    int	    rc;
    int	    n;
    
    /* write message to the pipe */
    sprintf(rz_filename, "%s/%s", wrdir, "re-cmd-pipe");
    if (verbose)
	eprintf(0, 0, "opening %s to write\n", rz_filename);
    if ((fd = open(rz_filename, O_WRONLY)) < 0)
	eprintf(2, errno, "Failed to open command pipe >%s<\n", rz_filename);

    eprintf(0, 0, "opened %s\n", rz_filename);
    
    rc = write(fd, msg, len);
    eprintf(0, 0, "wrote %s\n", rz_filename);
    if (rc < 0)
	eprintf(2, errno, "Failed to write command pipe >%s<\n", rz_filename);
    if (rc != len)
	eprintf(2, errno, "Failed to write entire msg to command pipe >%s<\n",
		rz_filename);

    n = rc;
    
    rc = close(fd);
    if (rc != 0) 
	eprintf(2, errno, "Failed to close command pipe >%s<\n", rz_filename);

    eprintf(0, 0, "Wrote %d to %s\n", n, rz_filename);
    
    return;
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
cmd_pipe_printf(
    char*	fmt,
    ...)
{
    va_list ap;  // for variable args
    char    buf[1024];
    int	    len;

    va_start(ap, fmt); // init specifying last non-var arg
    len = vsnprintf(buf, sizeof(buf)-2, fmt, ap);
    va_end(ap); // end var args

    if (len < 0 || len > (int)sizeof(buf)) {
	eprintf(2, 0, "cmd pipe message truncated.\n");
    }

    if (verbose)
	eprintf(0, 0, "Writing cmd pipe>%s<, len: %d\n", buf, len);
    
    cmd_pipe_write(buf, len);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
char*
get_rz_filename(
    reflash_state_t*	rsp,
    char*		namep)
{
    char*	dest;
    size_t	len;
    
    static char	rz_name_buf[PATH_MAX];
    
    namep += RZ_INDICATOR_LEN;
    if (*namep == '-') {
	return (NULL);
    }
    
    if (*namep != RZ_SEPARATOR) {
	strcpy(rz_name_buf, RZ_ANYFILE);
	return (rz_name_buf);
    }
    
    ++namep;
    dest = rz_name_buf;
    len = 0;
    while (*namep && *namep != '\n' && len < PATH_MAX-1) {
	*dest++ = *namep++;
	len++;
    }
    if (*namep && *namep != '\n')
	eprintf(2, 0, "RZ file name is too long.\n");

    *dest = '\0';

    if (verbose)
	eprintf(0, 0, "rz_filename>%s<\n", rz_name_buf);

    return (basename(rz_name_buf));
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
drain_cmd_pipe(
    reflash_state_t*	rsp)
{
    return;
    
#if 0
    int	    fd;

    sprintf(rz_filename, "%s/%s", wrdir, "re-cmd-pipe");
    if ((fd = open(rz_filename, O_RDONLY)) < 0)
	eprintf(2, errno, "Failed to open command pipe >%s<\n", rz_filename);
#endif
}

void
rz_timeout(
    int	arg)
{
    eprintf(3, 0, "rz timeout\n");
    exit(3);
}


void
setup_wait_alarm(
    void)
{
    signal(SIGALRM, rz_timeout);
    alarm(rz_wait_time);
}

void
cancel_wait_alarm(
    void)
{
    alarm(0);
    signal(SIGALRM, SIG_DFL);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
input_data(
    reflash_state_t*	rsp,
    char*		arg,
    char*		buf)
{
    int	    till_eof;
    int	    read_fd = -1;
    size_t  tsize;
    int	    doing_rz = 0;
    int	    old_q = 0;
    int	    stat;
    pid_t   rz_pid = 0;
    int		    set_size;
    unsigned long   num_read;
    char*   requested_filename;
    char*   fname;

    old_q = quietude;

    /* evaluate it if it is a var name */
    if (is_a_var(arg)) {
	arg = symbol_str_val(var_to_name(arg));
    }

  again:
    /* input data */
    if (memcmp(arg, RZ_INDICATOR, 3) == 0) {
	/*
	 * special case for handling zmodem xfers...
	 * If the user specifies RZ_INDICATOR (@rz)
	 * as a file name, then
	 * we fire up our special link to rz.  This file
	 * is designed to copy its stdin to a special
	 * named pipe.
	 * We set up our foreground read to read this pipe.
	 * Once we hit eof, we wait for the rz process to
	 * finish.
	 * We also take care of silencing ourselves so we don't
	 * interfere with the zmodem xfer.
	 */
	drain_cmd_pipe(rsp);
	requested_filename = get_rz_filename(rsp, arg);

	sprintf(rz_filename, "%s/%s", wrdir, rz_cmd);
	if (verbose)
	    eprintf(0, 0, "About to start rz, cmd>%s<\n\n", rz_filename);

	if (requested_filename != NULL &&
	    (strcmp(requested_filename, "*") != 0))
	    rprintf("\nPlease use zmodem to send ``%s''\n",
		    requested_filename);
	
	fflush(stdout);
	fflush(stderr);
	rz_pid = forkit(rz_filename, "-y", NULL);

	/* we will block here till the rz command reads the pipe */
	if (requested_filename != NULL)
	    cmd_pipe_printf("%s", requested_filename);

	sprintf(rz_filename, "%s/%s", wrdir, "re-data-pipe");
	fname = rz_filename;
	
	quietude = 1;
	doing_rz = 1;
	
	goto readit;
    }
	
    if (arg[0] == '-') {
	till_eof = 1;
	/* size is a limit */
	tsize = rs_get_resid(rsp);
	read_fd = 0;
    }
    else if (arg[0] >= '0' && arg[0] <= '9') {
	/* looks like a number, len of bytes from stdin */
	till_eof = 0;
	tsize = getlen(rsp, arg);
	read_fd = 0;
    }
    else {
	fname = arg;
	
      readit:
	if (verbose)
	    eprintf(0, 0, "reading >%s<\n", arg);

	read_fd = open(fname, O_RDONLY);

	if (verbose)
	    eprintf(0, 0, "back from open, fd: %d\n", read_fd);
	
	if (read_fd < 0) {
	    eprintf(1, errno, "Cannot open %s for input\n", fname);
	}
	tsize = rs_get_resid(rsp);
	till_eof = 1;
    }
    
    read_data(rsp, read_fd, buf, tsize, till_eof, &num_read, &set_size);

    if (read_fd != 0)
	close(read_fd);

    if (doing_rz) {
	int	rc;
	int	ec;
	int	try_again;

	eprintf(0, 0, "getting rz_status\n");
	rc = get_rz_status();
	eprintf(0, 0, "got rz_status\n");
	try_again = (rc == RZ_BAD_FILENAME);

	quietude = old_q;

	setup_wait_alarm();
	
	/* wait for rz to exit */
	rc = waitpid(rz_pid, &stat, 0);

	cancel_wait_alarm();

	/*
	 * rz does something to the terminal such that the first few
	 * writes don't display.
	 * 1 worked, so 2 seems safe.
	 */
	sleep(2);
	
	if (rc < 0)
	    eprintf(2, errno, "wait() on rz process failed.\n");

	
	if (try_again)
	    ec = 0;
	else
	    ec = 2;
	    
	if (!WIFEXITED(stat)) {
	    fix_tty();
	    eprintf(ec, 0,
		    "rz process (%d) did not exit normally, stat: 0x%04x.\n",
		    rz_pid, stat);
	}
	else if (WEXITSTATUS(stat) != 0) {
	    errno = WEXITSTATUS(stat);
	    eprintf(ec, errno,
		    "rz process exited with an error, stat: 0x%04x.\n",
		    stat);
	}

	if (try_again) {
	    fix_tty();
	    eprintf(0, 0, "\n\nWrong filename.  Please try again.\n\n");
	    goto again;
	}
    }
    
    if (set_size) {
	if (verbose)
	    eprintf(0, 0, "input_data(), setting len: 0x%lx\n",
		    num_read);
	validate_len_or_offset(rsp, LORO_LEN, num_read);
	rs_set_len(rsp, num_read);
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
erase_cmd(
    reflash_state_t*	rsp,
    unsigned long	offset,
    size_t		tsize)
{
    size_t  sec_size = rs_get_sec_size(rsp);
    size_t	    old_size;
    unsigned long   old_off;
    unsigned long   old_end;
    unsigned long   new_end;
    
    if (sec_size) {
	/*
	 * remember where the end of the erase region was
	 * We don't want to round down the start such that the length
	 * no longer includes the all of the requested region.
	 * e.g.
	 * assume sector size = 0x40000
	 * offset = 0x3fffc
	 * len = 8
	 * the range requested is 0x7fffc... 0x80003
	 * rounding 0x7fffc down gives 0x40000
	 * Adding 8 does not get the 4 bytes in the next sector
	 * 
	 * What we really need to erase is
	 * 0x0000 0x40000 0x80000 0xc0000
	 *        [---------------)
	 */

	/*
	 * first, round down the starting address...
	 */
	old_off = offset;
	offset = (offset / sec_size) * sec_size;
	if (offset != old_off)
	    eprintf(0, 0, "erase_cmd(), offset (0x%x) was rounded to: 0x%x\n",
		    old_off, offset);
	
	if (tsize != REST_OF_FLASH) {
	    /*
	     * now, round up the original ending address to the
	     * end of its encompassing sector.
	     */
	    old_size = tsize;
	    old_end = old_off + old_size;
	    new_end = ((old_end + sec_size - 1) / sec_size) * sec_size;

	    /* compute the new len */
	    tsize = new_end - offset;
	    if (tsize != old_size) 
		eprintf(0, 0,
			"erase_cmd(), size (0x%x) was rounded to: 0x%x\n",
			old_size, tsize);
	}
    }

    /*
     * XXX ??? should we check the queue for an existing erase
     * and skip the queuing if one is found?
     */
    
    queue_request(rsp, FR_CMD_ERASE, offset, tsize, NULL);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
print_queue(
    reflash_state_t*	rsp)
{
    int	    i;
    char    num_buf[32];

    for (i = 0; i < params_idx; i++) {
	sprintf(num_buf, " %3d) ", i);
	print_params(&params_queue[i], num_buf);
    }

    if (i == 0)
	eprintf(0, 0, "   <empty>\n");
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
print_status(
    reflash_state_t*	rsp)
{
    char*   old_name = progname;
    extern char _reflash_date_code[];
    
    progname = NULL;

    eprintf(0, 0, "Version: %s\n", version_string);
    eprintf(0, 0, "Module i/f version: %d\n", REFLASH_IF_VERSION);
    eprintf(0, 0, "Build date: %s\n", _reflash_date_code);
    eprintf(0, 0, "Offset restriction: %s\n",
	    rs_offset_restrictedq(rsp) ? "ON" : "OFF");
    eprintf(0, 0, "buffer: %p, buffer len: %d (0x%x)\n",
	    rs_get_buf(rsp), rs_get_buf_size(rsp), rs_get_buf_size(rsp));
    eprintf(0, 0, "offset: 0x%lx,  length: 0x%x\n",
	    rs_get_offset(rsp), getlen(rsp, "l"));
    eprintf(0, 0, "word mask: 0x%lx, sec_size: %d (0x%x)\n",
	    rs_get_word_mask(rsp), rs_get_sec_size(rsp), rs_get_sec_size(rsp));
    eprintf(0, 0, "auto erase: %s, auto verify: %s\n",
	    rs_auto_eraseq(rsp) ? "ON" : "OFF",
	    rs_auto_verifyq(rsp) ? "ON" : "OFF");

    eprintf(0, 0, "Command queue:\n");
    print_queue(rsp);

    progname = old_name;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
hex_dump(
    unsigned char   *data,
    size_t	    num)
{
    int     i;
    long    oldNum;
    char    buf[90];
    char*   bufp;
    int	    line_resid;
    
    while (num)
    {
	bufp = buf;
	sprintf(bufp, "%08lx ", (unsigned long)data);
	bufp += 8;
	*bufp++ = ':';
	*bufp++ = ' ';
	
	oldNum = num;
	
	for (i = 0; i < 16 && num; i++, num--) {
	    sprintf(bufp, "%02lx", (unsigned long)data[i]);
	    bufp += 2;
	    *bufp++ = (i == 7) ? '-' : ' ';
	}

	line_resid = (16 - i) * 3;
	if (line_resid) {
	    memset(bufp, ' ', line_resid);
	    bufp += line_resid;
	}
	
	memcpy(bufp, "| ", 2);
	bufp += 2;
	
	for (i = 0; i < 16 && oldNum; i++, oldNum--)
	    *bufp++ = isprint(data[i]) ? data[i] : '.';

	line_resid = 16 - i;
	if (line_resid) {
	    memset(bufp, ' ', 16-i);
	    bufp += 16 - i;
	}
	
	*bufp++ = '\n';
	*bufp++ = '\0';
	putstr(buf);
	data += 16;
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
do_assignment(
    reflash_state_t*	rsp,
    char*		arg)
{
    char*   name;
    char*   name_end;
    unsigned long   val;
    char    quote;

    if (arg[0] == '=') {
	symbol_dump_table();
	return;
    }

    name = arg;

    /* parse out name */
    while (*arg && !isspace(*arg))
	arg++;
    name_end = arg;

    if (!*arg)
	eprintf(2, 0, "Badly formed assignment>%s<\n", arg);

    while (isspace(*arg))
	arg++;

    if (!*arg)
	eprintf(2, 0, "Badly formed assignment>%s<\n", arg);

    *name_end = '\0';
    
    if (*arg != '\'' && *arg != '"') {
	val = get_len_or_offset(rsp, LORO_ASSIGNMENT, arg);
	symbol_set_num(name, val);
    }
    else {
	char*	p;

	quote = *arg++;
	p = arg;
	while (*p && *p != quote)
	    p++;

	/* allow end of string to close the quote */
	*p = '\0';
	symbol_set_str(name, arg);
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
print_message(
    reflash_state_t*	rsp,
    char*		arg)
{
    char    quote;
    char*   p;

    p = arg;
    quote = *p;
    if (quote == '\'' || quote == '"') {
	arg++;
	p++;
	while (*p && *p != quote)
	    p++;
	if (*p == quote)
	    *p = '\0';
    }

    /*  eprintf(0, 0, "pm>%s<\n", arg); */
    
    rprintf("%s", arg);
    fflush(stdout);
}

void
cmd_reboot(
    void)
{
    eprintf(0, 0, "Preparing to reboot system.\n");
    sleep(2);
    
    sync();
    sync();

    reboot(RB_AUTOBOOT);

    /* should never get here */
    eprintf(2, 0, "Welcome to Milliways.\n");
}

    
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
exec_arg(
    reflash_state_t*	rsp,
    int			opt,
    char*		optarg)
{
    size_t	    tsize;

    if (verbose) {
	char*	oa = "<none>";
	if (option_needs_arg(opt))
	    oa = optarg;
	eprintf(0, 0, "process opt>%c<, arg>%s<\n", opt, oa);
    }
    
    switch (opt) {
	case 'a':
	    tick_size = strtoul(optarg, NULL, 0);
	    use_ticker = tick_size != 0;
	    break;
	    
	case 'R':
	    do_reboot = 0;
	    break;
	    
	case 'y':
	    yes_to_all_prompts = 1;
	    break;
	    
	case 'N':
	    verbose = strtoul(optarg, NULL, 0);
	    break;
	    
	case 'T':
	    rz_wait_time = strtoul(optarg, NULL, 0);
	    break;
	    
	case 'F':
	{
	    size_t  len;

	    len = strlen(optarg);
	    if (len >= sizeof(wrdir) - 1)
		eprintf(2, 0, "FIFO dir name too long\n");
	    memcpy(wrdir, optarg, len+1);
	}
	break;
	
	case 'k':
	    /*
	     * konvert a number to decimal
	     *  this should be handled in main and never get here.
	     */
	    eprintf(2, 0, "Welcome to Milliways.\n");
	    break;
	    
	case 'P':
	{
	    int	    c;
	    FILE*   instream;

	    if (strcmp(optarg, "-") == 0)
		optarg = "Commit (y/n)? ";

	    if ((instream = fopen("/dev/tty", "r")) == NULL)
		eprintf(2, 0, "Cannot open terminal device.\n");
	    
	    do {
		
		print_message(rsp, optarg);
		if (yes_to_all_prompts) {
		    rprintf("<auto-y>\n");
		    c = 'y';
		    break;
		}
		
		c = getc(instream);
		if (c == EOF) {
		    eprintf(0, 0, "Got EOF.  resetting.\n");
		    clearerr(stdin);
		    continue;
		}
	    }
	    while (!strchr("nNyY", c));
	    
	    if (strchr("nN", c))
		eprintf(42, 0, "Exiting at user's request.\n");
	}
	break;
	    
	case 'm':
	    print_message(rsp, optarg);
	    rprintf("\n");
	    break;
	    
	case '=':
	    do_assignment(rsp, optarg);
	    break;
	    
	case 'j':
	{
	    int	stat;
	    
	    /* join up w/forked process */
	    eprintf(0, 0, "Waiting...\n");
	    wait(&stat);
	    eprintf(0, 0, "...done, stat: 0x%x\n", stat);
	}
	break;
	    
	case 's':
	    /* spawn a process */
	    /* FE parse optarg w/, as arg separator.w */
	    forkit(optarg, NULL);
	    break;
	    
	case 'Q':
	    quietude = strtoul(optarg, NULL, 0);
	    break;
	    
	case 'C':
	    clear_queue(rsp);
	    if (verbose)
		eprintf(0, 0, "Command queue is empty.\n");
	    break;
	    
	case 'h': 
	{
	    unsigned char*   memp = (unsigned char*)rs_get_buf(rsp);

	    /* hexdump at offset */
	    memp += rs_get_offset(rsp);
	    tsize = getlen(rsp, optarg);
	    hex_dump(memp, tsize);
	    break;
	}
	    
	case 'q':
	    /* list queue and other useful info */
	    print_status(rsp);
	    printed_the_queue++;
	    break;
	    
	case 'W':
	    rs_set_word_mask(rsp, strtoul(optarg, NULL, 0));
	    break;

	case 'S':
	    rs_set_sec_size(rsp, strtoul(optarg, NULL, 0));
	    break;
	
	case 'X':
	    execute_queue(rsp);
	    break;
	    
	case 'c':
	    cmdloop(rsp, optarg);
	    break;
	    
	case 'l':
	    tsize = getlen(rsp, optarg);
	    rs_set_len(rsp, tsize);
	    break;

	case 'p':
	    /* protect region */
	    tsize = getlen(rsp, optarg);
	    queue_request(rsp, FR_CMD_PROTECT, rs_get_offset(rsp),
			tsize, NULL);
	    break;
	    
	case 'u':
	    /* unprotect region */
	    tsize = getlen(rsp, optarg);
	    queue_request(rsp, FR_CMD_UNPROTECT, rs_get_offset(rsp),
			tsize, NULL);
	    break;
	    
	case 'V':
	    rs_set_auto_verify(rsp, strtoul(optarg, NULL, 0));
	    if (verbose)
		eprintf(0, 0, "auto verify is %s\n",
			rs_get_auto_verify(rsp) ? "on" : "off");
	    break;
	
	case 'v':
	    /* verify region */
	    tsize = getlen(rsp, optarg);
	    queue_request(rsp, FR_CMD_VERIFY, rs_get_offset(rsp),
			tsize, rs_get_cur_bufp(rsp));
	    break;
	
	case 'D':
	    rs_set_data_offset(rsp, strtoul(optarg, NULL, 0));
	    break;
	    
	case 'd':
	{
	    char*   p = rs_get_data_bufp(rsp);
	    
	    *p = (char)strtoul(optarg, NULL, 0);
	    rs_data_offset_inc(rsp);
	}
	break;
	    
	case 'x':
	    exit(0);
	    break;
	    
	case 'w':
	    /* write a segment */
	    tsize = getlen(rsp, optarg);
	    if (verbose)
		eprintf(0, 0, "segwrite, tsize: 0x%lx, mb: %p, p: %p\n",
			tsize,
			rs_get_buf(rsp), rs_get_cur_bufp(rsp));
	    
	    if (rs_auto_eraseq(rsp))
		erase_cmd(rsp, rs_get_offset(rsp), tsize);
	    
	    queue_request(rsp, FR_CMD_WRITE, rs_get_offset(rsp), tsize,
			rs_get_cur_bufp(rsp));
	    
	    if (rs_auto_verifyq(rsp))
		queue_request(rsp, FR_CMD_VERIFY, rs_get_offset(rsp), tsize,
			    rs_get_cur_bufp(rsp));
	    break;
	
	case 'i':
	    input_data(rsp, optarg, rs_get_cur_bufp(rsp));
	    break;
	
	case 'b':
	    rs_set_test_base(rsp, strtoul(optarg, NULL, 0));
	    break;
	    
	case 'r':
	    rs_set_dont_reset(rsp, 1);
	    break;
	    
	case 'O':
	    rs_set_offset_restricted(rsp, strtoul(optarg, NULL, 0));
	    break;
	    
	case 'E':
	    rs_set_auto_erase(rsp, strtoul(optarg, NULL, 0));
	    if (verbose)
		eprintf(0, 0, "auto erase is %s\n",
			rs_auto_eraseq(rsp) ? "on" : "off");
	    break;
	    
	case 'e':
	    /* erase region */
	    tsize = getlen(rsp, optarg);
	    erase_cmd(rsp, rs_get_offset(rsp), tsize);
	    break;
	
	case 't':
	{
	    int		i;
#define	Q_TYPE	unsigned long		
	    Q_TYPE*		q;
	    int		limit;
	    
	    /* generate a test pattern */
	    tsize = getlen(rsp, optarg);
	    eprintf(0, 0, "test, tsize: 0x%lx, mb: %p, p: %p\n",
		    tsize, rs_get_buf(rsp), rs_get_cur_bufp(rsp));
	    i = 0;
	    limit = tsize/sizeof(Q_TYPE);
	    for (q = (Q_TYPE*)rs_get_cur_bufp(rsp); i < limit; i++)
		*q++ = (Q_TYPE)(i + rs_get_test_base(rsp));
	    
	    eprintf(0, 0, "test mem setting complete.\n");
	    eprintf(0, 0, "you must specify a write seg cmd to "
		    "actually do the write.\n");
	}
	break;
	
	case 'o':
	{
	    unsigned long   offset;

	    if (strcmp(optarg, "+") == 0)
		offset = rs_get_offset(rsp) + rs_get_len(rsp);
	    else 
		offset = get_len_or_offset(rsp, LORO_OFF, optarg);
	    
	    if (offset > rs_get_buf_size(rsp))
		eprintf(2, 0, "Offset is bigger than buffer\n");
		
	    rs_set_offset(rsp, offset);
	}
	break;
	    
	case 'n':
	    tsize = strtoul(optarg, NULL, 0);
	    resize_main_buf(rsp,  tsize);
	    break;
	    
	case 'f':
	    if ((proc_file = malloc(strlen(optarg)+1)) == NULL)
		eprintf(2, 0, "Cannot malloc space for file name.\n");
	    strcpy(proc_file, optarg);
	    /*
	     * ensure we've closed the old name so the next open gets
	     * us the correct file
	     */
	    close_proc_file();
	    break;

	case 'L':
	    /* lookup a symbol */
	    printf("0x%lx\n", process_symbol(optarg));
	    break;

	case 'M':
	    /* specify the file that holds the params */
	    set_params_file(optarg);
	    break;

	case 'B':
	    /* specify the params sector size */
	    params_size = optarg;
	    break;
	    
	default:
	    eprintf(2, 0, "Unknown option: %c\n", opt);
	    break;
	    
    }

    return (0);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
char*
getline(
    char*   prompt,
    FILE*   stream)
{
#ifdef CONFIG_GNU_READLINE
    {
	char*	ret;

	ret = readline(prompt);
	return (ret);
    }
#else
    {
	static char buf[256];
	char*	    ret;

	if (stream == stdin)
	    rprintf("%s", prompt);
	ret = fgets(buf, sizeof(buf)-1, stream);
	return (ret);
    }
#endif    
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
cmdloop(
    reflash_state_t*	rsp,
    char*		arg)
{
    char    prompt[80];
    char*   cmdline;
    char*   cmdend;
    char    cmd;
    char*   origcmd;
    char*   args;
    static int	loop_entered = 0;
    int	    dummy;
    FILE*   cmd_stream = NULL;
    int	    error_vec_set;
    int	    interactive;

    if (loop_entered)
	eprintf(0, 0, "cmdloop already running\n");
    loop_entered++;

    if (strcmp(arg, "-") == 0) {
	cmd_stream = stdin;
	interactive = 1;
    }
    else {
	cmd_stream = fopen(arg, "r");
	if (cmd_stream == NULL) {
	    eprintf(2, errno, "Cannot open cmd stream>%s<\n", arg);
	    exit(2);
	}
	interactive = 0;
    }

    if (interactive && !error_vector_setq()) {
	set_error_vector(dummy);
	error_vec_set = 1;
    }
    else
    	error_vec_set = 0;

    sprintf(prompt, "o=0x%lx; l=0x%x> ",
	    rs_get_offset(rsp), getlen(rsp, "l"));

    while ((cmdline = getline(prompt, cmd_stream)) != NULL) {
	origcmd = cmdline;
	cmdend = cmdline + strlen(cmdline) - 1;

	/* strip any trailing WS on command line */
	while (cmdend >= cmdline && isspace(*cmdend)) {
	    cmdend--;
	}
	if (cmdend > cmdline)
	    *(cmdend+1) = '\0';
	
	/* skip white space at beginning of command line */
	while (isspace(*cmdline)) {
	    cmdline++;
	}
	cmd = cmdline[0];

	/* allow comments and empty lines */
	if (cmd == '#' || cmd == '\0')
	    continue;

	++cmdline;
	
	if (isspace(*cmdline)) {
	    /* looks like an arg */
	    while (isspace(*cmdline)) {
		cmdline++;
	    }
	}
	
	args = cmdline;
	
	if (option_needs_arg(cmd) && args[0] == '\0')
	    eprintf(2, 0, "The ``%c'' command, needs an argument.\n", cmd);
	if (!option_needs_arg(cmd) && args[0] != '\0')
	    eprintf(2, 0, "This command, %c, does not need an argument.\n",
		    cmd);
	    

#if 0
	hex_dump(origcmd, 256);
	eprintf(0, 0, "cmd: %p, args: %p\n", &cmd, args);
	eprintf(0, 0, "cmd>%c<, arg>%s<\n", cmd, args);
#endif
	
	
	exec_arg(rsp, cmd, args);
	sprintf(prompt, "o=0x%lx; l=0x%x> ",
		rs_get_offset(rsp), getlen(rsp, "l"));

    }

#if 0
    eprintf(0, 0, "cowardly exit till testing complete\n");
    exit(1);
#endif

    if (error_vec_set)
	unset_error_vector();
    
    loop_entered--;
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
main(
    int	    argc,
    char*   argv[])
{
    int			opt;
    char*		options;
    int			iter;
    reflash_state_t	state;
    reflash_state_t*	rsp = &state;
    int			we_jumped;
    long		init_buffer_size = 16<<20;

    
    extern char*    optarg;
    extern int	    opterr;
    opterr = 1;

#if 0
    {	/* XXX - tmp debug code: remove me! */
	for (iter = 0; iter < argc; iter++) {
	    printf("%d>%s<\n", iter, argv[iter]);
	}
	
    }	/* XXX - tmp debug code: remove me! */
#endif

#if 1
    {
	char*	cwd = getcwd(NULL, 0);
	if (!cwd)
	    eprintf(2, 0, "Could not getcwd()\n");

	if (verbose)
	    eprintf(0, 0, "cwd>%s<\n", cwd);
	
	if (strcmp("/skiff/davep/tools/reflash", cwd) == 0) {
	    if ((ep_log = fopen("/skiff/davep/tools/reflash/reflash.err.log",
				"a")) == NULL) {
		eprintf(2, 0, "Cannot open error log file.\n");
	    }
	}
	free(cwd);
    }
#endif
    
    /*
     * special case
     * The default familiar shell does not handle hex numbers in the
     * arithmetic functions, so in order to prevent needing another
     * program, we'll just add a new function to this program.
     * We handle things early since there is no use in handling
     * all of the initialization needed for reflash.
     */
    if (argc > 1 && argv[1][0] == '-' && argv[1][1] == 'k') {
	unsigned num;
	
	if (argv[1][2] != '\0') {
	    num = strtoul(&argv[1][2], NULL, 0);
	}
	else if (argc > 2) {
	    num = strtoul(argv[2], NULL, 0);
	}
	else {
	    fprintf(stderr, "garbled 'k' command.\n");
	    exit(3);
	}
	printf("%u\n", num);
	exit(0);
    }

    progname = basename(argv[0]);
    options = get_options();

    add_common_symbols();

#if 1
    {
	char*	arg;
	
	arg = option_peek(argc, argv, options, "Q:");
	if (arg)
    	    quietude = strtoul(arg, NULL, 0);

	arg = option_peek(argc, argv, options, "n:");
	if (arg) {
	    init_buffer_size = strtoul(arg, NULL, 0);
	    printf("peeked a -n option, value: 0x%lx\n", init_buffer_size);
	}

	arg = option_peek(argc, argv, options, "B:");
	if (arg)
    	    params_size = arg;

	arg = option_peek(argc, argv, options, "M:");
	if (arg)
	    set_params_file(arg);
    }
#endif

#ifndef	RELEASE    
    eprintf(0, 0, "DEVELOPMENT version.\n"
	    "If you really need to use this utility,\n"
	    "please get the last tagged Vx_y_z version from CVS.\n");
#endif    

    if (verbose)
	eprintf(0, 0, "progname>%s<\n", progname);

    rs_init(rsp);
    rs_set_auto_erase(rsp, 1);
    rs_set_auto_verify(rsp, 1);
    resize_main_buf(rsp, init_buffer_size);
    rs_set_word_mask(rsp, 0x3);
    rs_set_sec_size(rsp, 0x40000);
    rs_set_offset_restricted(rsp, 1);

    if (verbose)
	eprintf(0, 0, "got main_buf, %p, len, 0x%lx\n",
		rs_get_buf(rsp), rs_get_buf_size(rsp));

    for (iter = 0; ((opt = getopt(argc, argv, options)) != EOF); iter++) {
#if 0
	if (iter == 1 && rs_resetq(rsp)) {
	    queue_request(rsp, FR_CMD_RESET, RESTRICTED_BASE, 0L, NULL);
	}
#endif
	exec_arg(rsp, opt, optarg);
    }

    eprintf(0, 0, "options processed.  Executing command queue...\n");
    sleep(1);

    /*  eprintf(1, 0, "cowardly exit till testing complete\n"); */

    /*
     * set up a long jump vector.  If something during the RESTORE
     * causes an error, then we'll come back here.  We then enter
     * a command loop to try and fix the problem
     */
    
    set_error_vector(we_jumped);
    while (we_jumped) {
	eprintf(0, 0, "\n\nError detected during parameter execution.\n"
		"  Entering command loop. Good luck.\n");
	exec_arg(rsp, 'c', "-");
	set_error_vector(we_jumped);
    }

    if (printed_the_queue == 0) {
	print_status(rsp);
	fflush(stdout);
	fflush(stderr);
	sleep(1);
    }

    execute_queue(rsp);

    rprintf("\n%s: operations completed successfully.\n\n",
	    progname);
    
    if (do_reboot)
	cmd_reboot();

    exit(0);
}
