#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <stdarg.h>
#include <string.h>
#include <unistd.h>
#include <errno.h>
#include <ctype.h>

#define SYM_NAME_MAX	(40)

typedef union symbol_u_s
{
    unsigned long   num;
    char*	    str;
}
symbol_u_t;

typedef enum symbol_type_e
{
    ST_NUM,
    ST_STRING
}
symbol_type_t;



typedef struct symbol_s
{
    struct symbol_s*	next;
    
    char	    name[SYM_NAME_MAX];
    int		    type;
    symbol_u_t	    datum;
}
symbol_t;

symbol_t*   symbol_table = NULL;

extern void
eprintf(
    int		exit_code,
    int		err,
    char*	fmt,
    ...);

extern int
rprintf(
    char* fmt,
    ...);

/*
************************************************************************
*
* 
* 
************************************************************************
*/
symbol_t*
symbol_lookup(
    char*   name)
{
    symbol_t*	sym;

    for (sym = symbol_table; sym; sym = sym->next) {
	if (strcmp(name, sym->name) == 0)
	    return (sym);
    }

    return (NULL);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
symbol_t*
symbol_new(
    void)
{
    symbol_t*	sym;

    sym = (symbol_t*)malloc(sizeof(*sym));
    if (sym)
	memset(sym, 0x00, sizeof(*sym));
    else
	eprintf(2, 0, "Cannot malloc symbol.");

    return (sym);
}
    
/*
************************************************************************
*
* 
* 
************************************************************************
*/
symbol_t*
symbol_add(
    char*	    name)
{
    symbol_t*	sym;

    sym = symbol_new();

    if (strlen(name) >= sizeof(sym->name)-1)
	eprintf(2, 0, "symbol_add(), name>%s< is too long\n", name);
    
    strcpy(sym->name, name);

    /* push it on the list */
    sym->next = symbol_table;
    symbol_table = sym;

    return (sym);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
int
symbol_num_val(
    char*	    name,
    unsigned long*  retval)
{
    symbol_t*	sym;

    if ((sym = symbol_lookup(name)) != NULL &&
	sym->type == ST_NUM) {
	*retval = sym->datum.num;
	return(0);
    }

    if (sym)
	eprintf(2, 0, "Symbol type is not numeric.\n");

    return (-1);
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
char*
symbol_str_val(
    char*   name)
{
    symbol_t*	sym;

    if ((sym = symbol_lookup(name)) != NULL &&
	sym->type == ST_STRING)
	return(sym->datum.str);

#ifndef CONFIG_UNSET_SYMBOL_IS_OK
    if (sym)
	eprintf(2, 0, "Symbol type is not string.\n");
    else
	eprintf(2, 0, "symbol >%s< not found.\n", name);
#endif

    return ("");
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
symbol_set_num(
    char*	    name,
    unsigned long   val)
{
    symbol_t*	sym;


    if ((sym = symbol_lookup(name)) == NULL)
	sym = symbol_add(name);

    if (!sym)
	eprintf(2, 0, "cannot find or add symbol >%s<\n", name);
    else {
	sym->type = ST_NUM;
	sym->datum.num = val;
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
symbol_set_str(
    char*   name,
    char*   val)
{
    symbol_t*	sym;


    if ((sym = symbol_lookup(name)) == NULL)
	sym = symbol_add(name);

    if (!sym)
	eprintf(2, 0, "cannot find or add symbol >%s<\n", name);
    else {
	sym->type = ST_STRING;
	if ((sym->datum.str = (char*)malloc(strlen(val)+1)) == NULL) {
	    eprintf(2, 0, "out of memory ub symbol_set_str()\n");
	}
	strcpy(sym->datum.str, val);
    }
}

/*
************************************************************************
*
* 
* 
************************************************************************
*/
void
symbol_dump_table(
    void)
{
    symbol_t*	sym;

    for (sym = symbol_table; sym; sym = sym->next) {
	rprintf("name>%s, val", sym->name);
	
	switch (sym->type) {
	    case ST_NUM:
		rprintf(": %ld == 0x%lx\n",
			sym->datum.num, sym->datum.num);
		break;

	    case ST_STRING:
		rprintf(" >%s<\n", sym->datum.str);
		break;

	    default:
		eprintf(2, 0, "\nUnknown symbol type: %d\n", sym->type);
	}
    }
}


