/*
 * Itsy Utility Routines
 * Generic Hash Table Test Driver
 *
 * Copyright (c) Compaq Computer Corporation, 1996,1997,1998,1999
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 */

/*
 * Itsy Utility Routines
 * Generic Hash Table Test Driver
 *
 * Modified version of DCPI "util/htest_generic.c", Revision 1.3.
 * Original code written by Sanjay Ghemawat.
 *
 * $Log: htest_generic.c,v $
 * Revision 1.1.1.1  2000/11/28 16:46:34  kerr
 * Compaq Profiling Infrastructure
 *
 * Revision 1.1  1999/08/14  00:03:00  caw
 * Initial revision (from DCPI sources).
 *
 */

/* Test hash table */

#include <assert.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>

#include <util/util.h>
#include <util/htable_generic.h>

#ifndef	TRUE
#define	TRUE	(1)
#define	FALSE	(0)
#endif

/* Element types */
typedef struct {
    int key;
    char* value;
} Elem;

/* Artificial restriction on hash value range */
static unsigned int hash_range = 0x80000000;

static unsigned int hash_elem(void* x) {
    return ((unsigned int)(((Elem*) x)->key)) % hash_range;
}

static int equal_elem(void* a, void* b) {
    return ((Elem*) a)->key == ((Elem*) b)->key;
}

/* Wrapper type for hash table */
typedef struct { htable_generic h; } imap;
typedef struct { htable_generic_gen g; } imap_Generator;

static imap* imap_new(void) {
    imap* m = (imap*) malloc(sizeof(imap));
    htable_generic_init(&m->h, hash_elem, equal_elem);
    return m;
}

static void imap_delete(imap* m) {
    Elem* e;
    htable_generic_gen g;

    htable_generic_gen_init(&g, &m->h);
    while ((e = htable_generic_gen_next(&g))) {
	free(e);
    }
    htable_generic_destroy(&m->h);
    free(m);
}

/*@unused@*/
static void imap_dump(imap* m) {
    Elem* e;
    htable_generic_gen g;

    htable_generic_gen_init(&g, &m->h);
    while ((e = htable_generic_gen_next(&g))) {
	fprintf(stderr, "%d->%s\n", e->key, e->value);
    }
    htable_generic_destroy(&m->h);
}

static int imap_size(imap* m) {
    return htable_generic_size(&m->h);
}

static void imap_clear(imap* m) {
    Elem* e;
    htable_generic_gen g;

    htable_generic_gen_init(&g, &m->h);
    while ((e = htable_generic_gen_next(&g))) {
	free(e);
    }
    htable_generic_clear(&m->h);
}

static void imap_copy(imap* d, imap* s) {
    Elem* e;
    htable_generic_gen g;

    /* First copy the table */
    htable_generic_copy(&d->h, &s->h);

    /* Now copy the elements */
    htable_generic_gen_init(&g, &d->h);
    while ((e = htable_generic_gen_next(&g))) {
	Elem* c = (Elem*) malloc(sizeof(Elem));
	memcpy(c, e, sizeof(Elem));
	htable_generic_gen_replace_element(&g, c);
    }
}

static int imap_contains(imap* m, int i) {
    Elem e;
    e.key = i;
    return (htable_generic_find(&m->h, &e) != NULL);
}

static int imap_find(imap* m, int i, char** v) {
    Elem* result;
    Elem e;

    e.key = i;
    result = htable_generic_find(&m->h, &e);
    if (result != NULL) {
	*v = result->value;
	return TRUE;
    } else {
	return FALSE;
    }
}

static void imap_insert(imap* m, int k, char* v) {
    Elem* e = (Elem*) malloc(sizeof(Elem));
    e->key = k;
    e->value = v;
    e = htable_generic_insert(&m->h, e);
    if (e != NULL) {
	free(e);
    }
}

static void imap_remove(imap* m, int k) {
    Elem* old;
    Elem e;

    e.key = k;
    old = htable_generic_remove(&m->h, &e);
    if (old != NULL) {
	free(old);
    }
}

static void imap_predict(imap* m, int n) {
    htable_generic_predict(&m->h, n);
}

static void imap_check(imap* m) {
    htable_generic_check(&m->h);
}

static imap_Generator* imap_new_generator(imap* m) {
    imap_Generator* g = (imap_Generator*) malloc(sizeof(imap_Generator));
    htable_generic_gen_init(&g->g, &m->h);
    return g;
}

static void imap_delete_generator(imap_Generator* g) {
    free(g);
}

static int imap_get_element(imap_Generator* g, int* k, char** v) {
    Elem* e = htable_generic_gen_next(&g->g);
    if (e == NULL) {
	return FALSE;
    } else {
	*k = e->key;
	*v = e->value;
	return TRUE;
    }
}

static void imap_remove_element(imap_Generator* g) {
    htable_generic_gen_remove_element(&g->g);
}

/*#define LOG_ASSERTS*/
#ifdef LOG_ASSERTS
#define ASSERT(x) do {fprintf(stderr, "ASSERT: %s\n", #x);assert(x);} while (0)
#else
#define ASSERT(x) assert(x)
#endif

/* Extract element from map */
char* fetch(imap* m, int k) {
    char* v;

    ASSERT(imap_contains(m, k));
    (void)imap_find(m, k, &v);
    return v;
}

/*
 * Check if M2 is a subset of M1.
 */
static int subset(imap* m1, imap* m2) {
    int k;
    int result;
    char* v;
    char* v2;
    imap_Generator* gen;

    result = TRUE;
    gen = imap_new_generator(m1);
    while (imap_get_element(gen, &k, &v)) {
	if (! imap_contains(m2, k)) {
	    result = FALSE;
	    break;
	}

	(void)imap_find(m2, k, &v2);
	if (v != v2) {
	    result = FALSE;
	    break;
	}
    }
    imap_delete_generator(gen);
    return result;
}

/*
 * Compare two maps.
 */
static int compare(imap* m1, imap* m2) {
    return (subset(m1, m2) && subset(m2, m1));
}

static void check_copy(imap* m) {
    imap copy;

    htable_generic_init(&copy.h, hash_elem, equal_elem);
    htable_generic_copy(&copy.h, &m->h);
    ASSERT(compare(&copy, m));
}

/*
 * Get map size by using iterators.
 */
static int num_iterations(imap* m) {
    int count = 0;
    int k;
    char* v;
    imap_Generator* gen = imap_new_generator(m);
    while (imap_get_element(gen, &k, &v)) {
	count++;
    }
    imap_delete_generator(gen);
    return count;
}

/*
 * Check that iteration over map yields specified key,value pair
 */
static int iteration_contains(imap* m, int key, char* value) {
    int k;
    char* v;
    imap_Generator* gen = imap_new_generator(m);
    while (imap_get_element(gen, &k, &v)) {
	if (k == key) {
	    ASSERT(v == value);
	    imap_delete_generator(gen);
	    return 1;
	}
    }
    imap_delete_generator(gen);
    return 0;
}

/* Check deletion from iterator */
static void check_del_iter(imap* m) {
    int n;
    imap* copy;
    copy = imap_new();

    for (n = 1; n <= imap_size(m); n++) {
	/* Delete the first "n" elements of set via iterator */
	int c = 0;
	int k;
	char* v;
	imap_Generator* gen;

	imap_copy(copy, m);
	gen = imap_new_generator(copy);
	while (imap_get_element(gen, &k, &v)) {
	    c++;
	    if (c <= n) imap_remove_element(gen);
	}
	ASSERT(imap_size(copy) == (imap_size(m) - n));
	imap_delete_generator(gen);

	/* Delete the last "n" elements of set via iterator */
	imap_copy(copy, m);
	c = imap_size(m);
	gen = imap_new_generator(copy);
	while (imap_get_element(gen, &k, &v)) {
	    if (c <= n) imap_remove_element(gen);
	    c--;
	}
	ASSERT(imap_size(copy) == (imap_size(m) - n));
	imap_delete_generator(gen);
    }

    imap_delete(copy);
}

/*
 * Some common strings.
 */
static char* one = "one";
static char* two = "two";
static char* three = "three";
static char* four = "four";
static char* five = "five";

static void black_empty(void);
static void black_single(void);
static void black_multiple(void);

static void black_box(void) {
    /*
     * Testing strategy -
     *
     * - Operations on empty maps.
     * - Operations on singleton maps.
     * - Operations on larger maps.
     */

    black_empty();
    black_single();
    black_multiple();
}

static void ximap_check(imap* m) {
    imap_check(m);
    check_copy(m);
}

static void black_empty() {
    /* Empty map tests. */

    int i;
    char* value;

    imap* empty = imap_new();
    ximap_check(empty);

    /* Check size */
    ASSERT(imap_size(empty) == 0);

    /* Check contains */
    for (i = -3; i <= 3; i++) {
	ASSERT(! imap_contains(empty, i));
    }

    /* Check find */
    for (i = -5; i <= 5; i++) {
	ASSERT(! imap_find(empty, i, &value));
    }

    /* Check iterator */
    {
	imap_Generator* gen;
	char* v;
	int k;

	check_del_iter(empty);
	ASSERT(num_iterations(empty) == 0);

	gen = imap_new_generator(empty);
	assert(! imap_get_element(gen, &k, &v));
	imap_delete_generator(gen);
    }

    /* Check insert */
    {
	imap* single;

	single = imap_new();
	ximap_check(single);
	imap_insert(single, 1, one);
	ximap_check(single);

	ASSERT(imap_size(single) == 1);
	ASSERT(imap_contains(single, 1));
	ASSERT(imap_find(single, 1, &value));
	ASSERT(value == one);

	imap_clear(single);
	ximap_check(single);
	ASSERT(imap_size(single) == 0);
	ASSERT(num_iterations(single) == 0);

	imap_delete(single);
    }

    /* Check remove */
    {
	imap* empty2;

	empty2 = imap_new();
	ximap_check(empty2);

	ASSERT(imap_size(empty2) == 0);
	imap_remove(empty2, 1);
	ximap_check(empty2);
	ASSERT(imap_size(empty2) == 0);

	imap_clear(empty2);
	ximap_check(empty2);
	ASSERT(imap_size(empty2) == 0);
	ASSERT(num_iterations(empty2) == 0);

	imap_delete(empty2);
    }

    /* Check clear */
    {
	imap* empty2;

	empty2 = imap_new();
	imap_clear(empty2);
	ximap_check(empty2);
	ASSERT(imap_size(empty2) == 0);
	ASSERT(num_iterations(empty2) == 0);
	imap_delete(empty2);
    }

    imap_delete(empty);
}

static void black_single() {
    /* Single element map tests */
    int i;
    char* value;
    imap* single;

    single = imap_new();
    ximap_check(single);
    imap_insert(single, 2, two);
    ximap_check(single);

    ASSERT(imap_size(single) == 1);

    /* Check contains */
    for (i = -3; i <= 3; i++) {
	/*@-intcompare@*/
	ASSERT(imap_contains(single, i) == (i == 2));
	/*@=intcompare@*/
    }

    /* Check find */
    ASSERT(fetch(single, 2) == two);

    /* Check find */
    for (i = -5; i <= 5; i++) {
	/*@-intcompare@*/
	ASSERT(imap_find(single, i, &value) == (i == 2));
	/*@=intcompare@*/
	if (i == 2) ASSERT(value == two);
    }

    /* Check iterator */
    {
	check_del_iter(single);
	ASSERT(num_iterations(single) == 1);
	ASSERT(iteration_contains(single, 2, two));
    }

    /* Check store */
    {
	imap* temp;

	temp = imap_new();
	ximap_check(temp);
	imap_copy(temp, single);
	ximap_check(temp);

	ASSERT(imap_size(temp) == 1);
	imap_insert(temp, 2, three);
	ximap_check(temp);
	ASSERT(imap_size(temp) == 1);
	ASSERT(imap_contains(temp, 2));
	ASSERT(fetch(temp, 2) == three);

	imap_copy(temp, single);
	ASSERT(imap_size(temp) == 1);
	imap_insert(temp, 3, three);
	ximap_check(temp);
	ASSERT(imap_size(temp) == 2);
	ASSERT(imap_contains(temp, 2));
	ASSERT(imap_contains(temp, 3));
	ASSERT(fetch(temp, 2) == two);
	ASSERT(fetch(temp, 3) == three);

	ASSERT(num_iterations(temp) == 2);
	ASSERT(iteration_contains(temp, 2, two));
	ASSERT(iteration_contains(temp, 3, three));

	imap_delete(temp);
    }

    /* Check insert */
    {
	imap* temp;

	temp = imap_new();
	ximap_check(temp);
	imap_copy(temp, single);
	ximap_check(temp);

	ASSERT(imap_size(temp) == 1);
	imap_insert(temp, 3, three);
	ximap_check(temp);
	ASSERT(imap_size(temp) == 2);
	ASSERT(imap_contains(temp, 2));
	ASSERT(imap_contains(temp, 3));
	ASSERT(fetch(temp, 2) == two);
	ASSERT(fetch(temp, 3) == three);

	ASSERT(num_iterations(temp) == 2);
	ASSERT(iteration_contains(temp, 2, two));
	ASSERT(iteration_contains(temp, 3, three));

	imap_delete(temp);
    }

    /* Check remove */
    {
	imap* temp;

	temp = imap_new();
	ximap_check(temp);
	imap_copy(temp, single);
	ximap_check(temp);

	imap_remove(temp, 5);
	ximap_check(temp);
	ASSERT(compare(temp, single));

	imap_remove(temp, 2);
	ximap_check(temp);
	ASSERT(imap_size(temp) == 0);
	ASSERT(! imap_contains(temp, 2));

	imap_delete(temp);
    }

    /* Check clear */
    {
	imap* temp;

	temp = imap_new();
	ximap_check(temp);
	imap_copy(temp, single);
	ximap_check(temp);

	imap_clear(temp);
	ximap_check(temp);
	ASSERT(imap_size(temp) == 0);
	ASSERT(num_iterations(temp) == 0);

	imap_delete(temp);
    }

    imap_delete(single);
}

static void black_multiple() {
    int i;
    char* value;
    imap* multi3;
    imap* multi4;
    imap* multi5;

    multi3 = imap_new();
    ximap_check(multi3);
    imap_insert(multi3, 1, one);
    ximap_check(multi3);
    imap_insert(multi3, 2, two);
    ximap_check(multi3);
    imap_insert(multi3, 3, three);
    ximap_check(multi3);

    multi4 = imap_new();
    ximap_check(multi4);
    imap_insert(multi4, 1, one);
    ximap_check(multi4);
    imap_insert(multi4, 2, two);
    ximap_check(multi4);
    imap_insert(multi4, 3, three);
    ximap_check(multi4);
    imap_insert(multi4, 4, four);
    ximap_check(multi4);

    multi5 = imap_new();
    ximap_check(multi5);
    imap_insert(multi5, 1, one);
    ximap_check(multi5);
    imap_insert(multi5, 2, two);
    ximap_check(multi5);
    imap_insert(multi5, 3, three);
    ximap_check(multi5);
    imap_insert(multi5, 4, four);
    ximap_check(multi5);
    imap_insert(multi5, 5, five);
    ximap_check(multi5);

    /* Check size */
    ASSERT(imap_size(multi3) == 3);
    ASSERT(imap_size(multi4) == 4);
    ASSERT(imap_size(multi5) == 5);

    /* Check contains. */
    ASSERT(imap_contains(multi3, 1));
    ASSERT(imap_contains(multi3, 2));
    ASSERT(imap_contains(multi3, 3));

    ASSERT(imap_contains(multi4, 1));
    ASSERT(imap_contains(multi4, 2));
    ASSERT(imap_contains(multi4, 3));
    ASSERT(imap_contains(multi4, 4));

    ASSERT(imap_contains(multi5, 1));
    ASSERT(imap_contains(multi5, 2));
    ASSERT(imap_contains(multi5, 3));
    ASSERT(imap_contains(multi5, 3));
    ASSERT(imap_contains(multi5, 5));

    /* Check find */
    ASSERT(fetch(multi3, 1) == one);
    ASSERT(fetch(multi3, 2) == two);
    ASSERT(fetch(multi3, 3) == three);

    ASSERT(fetch(multi4, 1) == one);
    ASSERT(fetch(multi4, 2) == two);
    ASSERT(fetch(multi4, 3) == three);
    ASSERT(fetch(multi4, 4) == four);

    ASSERT(fetch(multi5, 1) == one);
    ASSERT(fetch(multi5, 2) == two);
    ASSERT(fetch(multi5, 3) == three);
    ASSERT(fetch(multi5, 4) == four);
    ASSERT(fetch(multi5, 5) == five);

    /* Check find */
    for (i = -5; i <= 0; i++) {
	ASSERT(! imap_find(multi3, i, &value));
	ASSERT(! imap_find(multi4, i, &value));
	ASSERT(! imap_find(multi5, i, &value));
    }
    ASSERT(imap_find(multi3, 1, &value) && (value == one));
    ASSERT(imap_find(multi3, 2, &value) && (value == two));
    ASSERT(imap_find(multi3, 3, &value) && (value == three));
    ASSERT(! imap_find(multi3, 4, &value));
    ASSERT(! imap_find(multi3, 5, &value));

    ASSERT(imap_find(multi4, 1, &value) && (value == one));
    ASSERT(imap_find(multi4, 2, &value) && (value == two));
    ASSERT(imap_find(multi4, 3, &value) && (value == three));
    ASSERT(imap_find(multi4, 4, &value) && (value == four));
    ASSERT(! imap_find(multi4, 5, &value));

    ASSERT(imap_find(multi5, 1, &value) && (value == one));
    ASSERT(imap_find(multi5, 2, &value) && (value == two));
    ASSERT(imap_find(multi5, 3, &value) && (value == three));
    ASSERT(imap_find(multi5, 4, &value) && (value == four));
    ASSERT(imap_find(multi5, 5, &value) && (value == five));

    /* Check iterator */
    {
	check_del_iter(multi3);
	check_del_iter(multi4);
	check_del_iter(multi5);

	ASSERT(num_iterations(multi3) == 3);
	ASSERT(iteration_contains(multi3, 1, one));
	ASSERT(iteration_contains(multi3, 2, two));
	ASSERT(iteration_contains(multi3, 3, three));

	ASSERT(num_iterations(multi4) == 4);
	ASSERT(iteration_contains(multi4, 1, one));
	ASSERT(iteration_contains(multi4, 2, two));
	ASSERT(iteration_contains(multi4, 3, three));
	ASSERT(iteration_contains(multi4, 4, four));

	ASSERT(num_iterations(multi5) == 5);
	ASSERT(iteration_contains(multi5, 1, one));
	ASSERT(iteration_contains(multi5, 2, two));
	ASSERT(iteration_contains(multi5, 3, three));
	ASSERT(iteration_contains(multi5, 4, four));
	ASSERT(iteration_contains(multi5, 5, five));
    }

    /* Check store */
    {
	imap* temp = imap_new();
	ximap_check(temp);
	imap_copy(temp, multi3);
	ximap_check(temp);

	ASSERT(compare(multi3, temp));

	/* Store existing element */
	imap_insert(temp, 2, five);
	ximap_check(temp);
	ASSERT(imap_size(temp) == imap_size(multi3));
	ASSERT(imap_contains(temp, 2));
	ASSERT(fetch(temp, 2) == five);
	imap_remove(temp, 2);
	ximap_check(temp);
	imap_insert(temp, 2, fetch(multi3, 2));
	ximap_check(temp);
	ASSERT(compare(multi3, temp));

	/* Store non-existent element */
	imap_copy(temp, multi4);
	ximap_check(temp);
	ASSERT(compare(multi4, temp));
	imap_insert(temp, 5, five);
	ximap_check(temp);
	ASSERT(compare(multi5, temp));
	imap_remove(temp, 5);
	ximap_check(temp);
	ASSERT(compare(multi4, temp));

	imap_delete(temp);
    }

    /* Check insert */
    {
	imap* temp = imap_new();
	ximap_check(temp);
	imap_copy(temp, multi4);
	ximap_check(temp);

	ASSERT(compare(multi4, temp));
	ASSERT(imap_size(temp) == 4);
	imap_insert(temp, 5, five);
	ximap_check(temp);
	ASSERT(compare(multi5, temp));

	imap_copy(temp, multi3);
	imap_insert(temp, 4, four);
	ximap_check(temp);
	imap_insert(temp, 5, five);
	ximap_check(temp);
	ASSERT(compare(multi5, temp));

	imap_delete(temp);
    }

    /* Check remove */
    {
	imap* temp = imap_new();
	imap* empty = imap_new();

	/* Check removal of existing elements */
	ximap_check(temp);
	imap_copy(temp, multi3);
	ximap_check(temp);
	ASSERT(compare(multi3, temp));
	imap_remove(temp, 1);
	ximap_check(temp);
	imap_remove(temp, 2);
	ximap_check(temp);
	imap_remove(temp, 3);
	ximap_check(temp);
	ASSERT(compare(empty, temp));

	imap_copy(temp, multi3);
	ximap_check(temp);
	ASSERT(compare(multi3, temp));
	imap_remove(temp, 3);
	ximap_check(temp);
	imap_remove(temp, 2);
	ximap_check(temp);
	imap_remove(temp, 1);
	ximap_check(temp);
	ASSERT(compare(empty, temp));

	imap_copy(temp, multi5);
	ximap_check(temp);
	ASSERT(compare(multi5, temp));
	imap_remove(temp, 5);
	ximap_check(temp);
	ASSERT(compare(multi4, temp));
	imap_remove(temp, 4);
	ximap_check(temp);
	ASSERT(compare(multi3, temp));
	imap_remove(temp, 1);
	ximap_check(temp);
	imap_remove(temp, 2);
	ximap_check(temp);
	imap_remove(temp, 3);
	ximap_check(temp);
	ASSERT(compare(empty, temp));

	/* Check removal of non-existent elements */
	imap_copy(temp, multi4);
	ximap_check(temp);
	for (i = -5; i <= 0; i++) {
	    imap_remove(temp, i);
	    ximap_check(temp);
	    ASSERT(compare(multi4, temp));
	}
	for (i = 5; i <= 10; i++) {
	    imap_remove(temp, i);
	    ximap_check(temp);
	    ASSERT(compare(multi4, temp));
	}

	imap_delete(temp);
	imap_delete(empty);
    }

    /* Check clear */
    {
	imap* temp = imap_new();
	imap* empty = imap_new();

	/* Check removal of existing elements */
	ximap_check(temp);
	imap_copy(temp, multi3);
	ximap_check(temp);
	ASSERT(compare(multi3, temp));
	imap_clear(temp);
	ximap_check(temp);
	ASSERT(compare(empty, temp));

	imap_copy(temp, multi3);
	ximap_check(temp);
	ASSERT(compare(multi3, temp));
	imap_clear(temp);
	ximap_check(temp);
	ASSERT(compare(empty, temp));

	imap_copy(temp, multi5);
	ximap_check(temp);
	ASSERT(compare(multi5, temp));
	imap_clear(temp);
	ximap_check(temp);
	ASSERT(compare(empty, temp));

	imap_delete(temp);
	imap_delete(empty);
    }

    /* Check large number of entries */
    {
	imap* map = imap_new();
	char* copy;

	ximap_check(map);
	for (i = 0; i < 1000; i++) {
	    char* val = (char*) malloc(sizeof(char) * 20);
	    sprintf(val, "%d", i);
	    imap_insert(map, i, val);
	    ASSERT(num_iterations(map) == i+1);
	}
	ximap_check(map);

	copy = (char*) malloc(sizeof(char) * 20);
	for (i = 0; i < 1000; i++) {
	    char* val = fetch(map, i);
	    sprintf(copy, "%d", i);
	    ASSERT(strcmp(copy, val) == 0);
	}
	free(copy);

	for (i = 0; i < 1000; i++) {
	    free((char*)fetch(map, i));
	    imap_remove(map, i);
	    ASSERT(num_iterations(map) == (999-i));
	}
	ximap_check(map);

	imap_delete(map);
    }

    /* Check prediction */
    {
	imap* map = imap_new();
	char* copy;

	imap_predict(map, 1000);

	ximap_check(map);
	for (i = 0; i < 1000; i++) {
	    char* val = (char*) malloc(sizeof(char) * 20);
	    sprintf(val, "%d", i);
	    imap_insert(map, i, val);
	    ASSERT(num_iterations(map) == i+1);
	}
	ximap_check(map);

	copy = (char*) malloc(sizeof(char) * 20);
	for (i = 0; i < 1000; i++) {
	    char* val = fetch(map, i);
	    sprintf(copy, "%d", i);
	    ASSERT(strcmp(copy, val) == 0);
	}
	free(copy);

	for (i = 0; i < 1000; i++) {
	    free((char*)fetch(map, i));
	    imap_remove(map, i);
	    ASSERT(num_iterations(map) == (999-i));
	}
	ximap_check(map);

	imap_delete(map);
    }

    imap_delete(multi3);
    imap_delete(multi4);
    imap_delete(multi5);
}

/*
 * Glass box tests.
 */
static void glass_box(void) {
    /* ... */
}

static void run_tests(unsigned int hash_range_value) {
    unsigned int old_range = hash_range;
    hash_range = hash_range_value;
    fprintf(stderr, "testing with hash range 0x%08x\n", hash_range);
    black_box();
    glass_box();
    hash_range = old_range;
}

int
main(/*@unused@*/ int argc, /*@unused@*/ char** argv) {
    unsigned int i;

    run_tests(0x80000000);
    run_tests(0x40000000);
    run_tests(0x1000);
    run_tests(0x100);
    for (i = 1; i <= 16; i++) {
	run_tests(i);
    }

    return 0;
}
