/*
 * Itsy Utility Routines
 * Simple Logging Operations
 *
 * Copyright (c) Compaq Computer Corporation, 1999
 *
 * Use consistent with the GNU GPL is permitted,
 * provided that this copyright notice is
 * preserved in its entirety in all copies and derived works.
 *
 * COMPAQ COMPUTER CORPORATION MAKES NO WARRANTIES, EXPRESSED OR IMPLIED,
 * AS TO THE USEFULNESS OR CORRECTNESS OF THIS CODE OR ITS
 * FITNESS FOR ANY PARTICULAR PURPOSE.
 *
 */
/*
 * Author: Carl Waldspurger
 *
 * $Log: log.c,v $
 * Revision 1.1.1.1  2000/11/28 16:46:34  kerr
 * Compaq Profiling Infrastructure
 *
 * Revision 1.3  1999/02/26  20:05:10  caw
 * Changed default stream initialization to work with ELF.
 * Defined log_set_stream() operation.
 *
 * Revision 1.2  1999/02/25  18:43:10  kerr
 * Made enormous change to the makefile structure.  Now:
 *   1.  All makefiles get their basic parameters from the top level
 *   2.  All makefiles can be run hierarchically.
 *   3.  'make clean' and 'make libs' at the top level work.
 * This is in preparation for switching to elf.
 *
 * Revision 1.1  1999/01/22 22:54:34  caw
 * Initial revision.
 *
 */

/*
 * includes
 *
 */

#include <stdlib.h>
#include <stdio.h>
#include <stdarg.h>
#include <errno.h>
#include <unistd.h>
#include <string.h>

/*
 * constants
 *
 */

#define	DEFAULT_STREAM		stderr

/*
 * types
 *
 */

typedef struct {
  char *name;
  FILE *stream;
  int  verbose;
} log_state;

/*
 * module globals
 *
 */

static log_state global = { NULL, NULL, 0 };

/*
 * logging operations
 *
 */

void fatal(const char *format, ...)
{
  /*
   * modifies: global.stream (or DEFAULT_STREAM)
   * effects:  Logs error message and terminates program.
   *
   */

  FILE *stream = global.stream;
  char *name = global.name;

  va_list args;

  /* use DEFAULT_STREAM if none specified */
  if (stream == NULL)
    stream = DEFAULT_STREAM;

  /* log common prefix */
  if (name != NULL)
    {
      (void) fputs(name, stream);
      (void) fputs(": ", stream);
    }
  (void) fputs("failure: ", stream);

  /* log formatted args */
  va_start(args, format);
  (void) vfprintf(stream, format, args);
  va_end(args);

  /* log system error, if any */
  if (errno > 0 && errno < sys_nerr)
    (void) fprintf(stream, " (%s)\n", sys_errlist[errno]);
  else
    (void) fputc('\n', stream);
  
  /* terminate program */
  exit(EXIT_FAILURE);
}

void warning(const char *format, ...)
{
  /*
   * modifies: global.stream (or DEFAULT_STREAM)
   * effects:  Logs warning message.
   *
   */

  FILE *stream = global.stream;
  char *name = global.name;

  va_list args;

  /* use DEFAULT_STREAM if none specified */
  if (stream == NULL)
    stream = DEFAULT_STREAM;

  /* log common prefix */
  if (name != NULL)
    {
      (void) fputs(name, stream);
      (void) fputs(": ", stream);
    }
  (void) fputs("warning: ", stream);

  /* log formatted args */
  va_start(args, format);
  (void) vfprintf(stream, format, args);
  va_end(args);

  /* log system error, if any */
  if (errno > 0 && errno < sys_nerr)
    (void) fprintf(stream, " (%s)\n", sys_errlist[errno]);
  else
    (void) fputc('\n', stream);
}

void log_verbose(const char *format, ...)
{
  /*
   * modifies: global.stream (or DEFAULT_STREAM)
   * effects:  Logs verbose message iff global.verbose.
   *
   */

  FILE *stream = global.stream;
  char *name = global.name;

  va_list args;

  /* skip unless verbose logging enabled */
  if (!global.verbose)
    return;

  /* use DEFAULT_STREAM if none specified */
  if (stream == NULL)
    stream = DEFAULT_STREAM;

  /* log common prefix */
  if (name != NULL)
    {
      (void) fputs(name, stream);
      (void) fputs(": ", stream);
    }

  /* log formatted args */
  va_start(args, format);
  (void) vfprintf(stream, format, args);
  va_end(args);

  /* log newline */
  (void) fputc('\n', stream);
}

void log_set_name(const char *name)
{
  /*
   * modifies: global.name
   * effects:  Sets global.name to a copy of name.
   *
   */

  /* reclaim storage */
  if (global.name != NULL)
    free(global.name);

  /* clone name */
  global.name = NULL;
  if (name != NULL)
    global.name = strdup(name);
}

void log_set_verbose(int enable)
{
  /*
   * modifies: global.verbose (global)
   * effects:  Sets global.verbose flag to enable.
   *
   */

  global.verbose = enable;
}

void log_set_stream(FILE *stream)
{
  /*
   * modifies: global.stream (global)
   * effects:  Sets global.stream to stream.
   *
   */

  global.stream = stream;
}
