/* ***************************************************************************************
 * This file is a copy of sa11xx-uda1341 with the following differences:
 *
 * 2003-06-18   Fae Ghodrat     Modified driver to be used with the backpaq:
 *                              fake a direct connection to uda1341 L3 and dma.
 *                              transmit and receive are independent in the backpaq.  
 *                              No need to fill the buffers with zeros anymore.
 *************************************************************************************** */

/* sa11xx-uda1341 header */
/* ***************************************************************************************
 *  Driver for Philips UDA1341TS on Compaq iPAQ H3600 soundcard
 *  Copyright (C) 2002 Tomas Kasparek <tomas.kasparek@seznam.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License.
 * 
 * History:
 *
 * 2002-03-13	Tomas Kasparek	Initial release - based on h3600-uda1341.c from OSS
 * 2002-03-20   Tomas Kasparek  playback over ALSA is working
 * 2002-03-28   Tomas Kasparek  playback over OSS emulation is working
 * 2002-03-29   Tomas Kasparek  basic capture is working (native ALSA)
 * 2002-03-29   Tomas Kasparek  capture is working (OSS emulation)
 * 2002-04-04   Tomas Kasparek  better rates handling (allow non-standard rates)
 * 2002-02-11   Brian Avery     updated the driver to use the HAL layer so it works 
 *                              on 3800 as well as 3[61]00 ipaqs
 * 2002-02-14   Brian Avery     fixed full duplex mode
 * 2002-03-10   Brian Avery     fixed suspend resume, added hal ifdef abstraction layer
 */

/* sa11xx-uda1341.c,v 1.6 2002/12/04 18:52:05 perex Exp */
/****************************************************************************************
 *
 * To understand what Alsa Drivers should be doing look at "Writing an Alsa Driver" 
 * by Takashi Iwai available in the Alsa doc section on the website		
 * 
 * A few notes to make things clearer.  The UDA1341 is hooked up to Serial port 4 on 
 * the SA1100.  We are using  SSP mode to talk to the UDA1341.  The UDA1341 bit & 
 * wordselect clocks are generated by this UART. Unfortunately, the clock only runs if 
 * the transmit buffer has something in it.  So, if we are just recording, we feed the 
 * transmit DMA stream a bunch of 0x0000 so that the transmit buffer is full and the 
 * clock keeps going.  The zeroes come from FLUSH_BASE_PHYS which is a mem loc that 
 * always decodes to 0's w/ no off chip access.
 *
 * Some alsa terminology:
 *    frame => num_channels * sample_size  e.g stereo 16 bit is 2 * 16 = 32 bytes
 *    period => the least number of bytes that will generate an interrupt e.g. we 
 *              have a 1024 byte buffer and 4 periods in the runtime structure this 
 *              means we'll get an int every 256 bytes or 4 times per buffer.
 *    A number of the sizes are in frames rather than bytes, use frames_to_bytes 
 *    and bytes_to_frames to convert.  The easiest way to tell the units is to look 
 *    at the type i.e. runtime-> buffer_size is in frames and its type is snd_pcm_uframes_t
 *
 * Notes about the pointer fxn:
 *    The pointer fxn needs to return the offset into the dma buffer in frames
 *    Interruptsmust be  blocked before calling the  dma_get_pos fxn to avoid race with 
 *    interrupts
 *
 *
 * Notes about transfer methods:
 *    The async write calls fail.  I probably need to implement something else to 
 *    support them?
 * 
 ****************************************************************************************/


#include <linux/autoconf.h>
#include <sound/driver.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/errno.h>
#include <linux/ioctl.h>
#include <linux/delay.h>
#include <linux/slab.h>

#include <asm/hardware.h>
// #include <asm/irq.h>
// #include <asm/system.h>
#include <asm/arch/dma-backpaq.h>
/* #include <asm/dma.h> */

#include <sound/core.h>

#include <sound/pcm.h>
#include <sound/initval.h>

#include <linux/l3/l3.h>
#include <linux/l3/uda1341.h>

#include <asm/semaphore.h>
#include <asm/uaccess.h>
// included already -- #include <asm/hardware.h>

#if defined(CONFIG_IPAQ_HAL) || defined(CONFIG_IPAQ_HAL_MODULE)
#include <asm/arch/h3600_hal.h>
#endif

#define AUDIO_RATE_DEFAULT	44100

// #define DEBUG_MODE
// #define DEBUG_FUNCTION_NAMES

#include <sound/uda1341.h>

#if defined(CONFIG_H3600_BACKPAQ_FPGA)
#include <asm/arch/backpaq.h>
#endif


#define BUFSIZE 4*512  // buf * 2 samples * 2bytes/sample
#define BACKPAQ_DMA_BYTES 2*BUFSIZE
#define MASK_ALL_BACKPAQ_AUDIO_INT AUDIO_INT_PLAY_FIFO_EMPTY | AUDIO_INT_PLAY_FIFO_FULL | AUDIO_INT_PLAY_FIFO_1_2_FULL | AUDIO_INT_PLAY_FIFO_3_4_FULL | AUDIO_INT_REC_FIFO_EMPTY | AUDIO_INT_REC_FIFO_FULL | AUDIO_INT_REC_FIFO_1_2_FULL | AUDIO_INT_REC_FIFO_3_4_FULL | AUDIO_INT_RO | AUDIO_INT_WO

/* *** {{{ Type definitions *** */

MODULE_AUTHOR("Tomas Kasparek <tomas.kasparek@seznam.cz>");
MODULE_LICENSE("GPL");
MODULE_DESCRIPTION("SA1100/SA1111 + UDA1341TS driver for ALSA");
MODULE_CLASSES("{sound}");
MODULE_DEVICES("{{UDA1341,iPAQ H3600 UDA1341TS}}");

static char *id = NULL;	/* ID for this card */
static int max_periods_allowed = 255;


MODULE_PARM(id, "s");
MODULE_PARM_DESC(id, "ID string for SA1100/SA1111 + UDA1341TS soundcard.");
MODULE_PARM(max_periods_allowed, "i");
MODULE_PARM_DESC(max_periods_allowed, "max # of periods(interrupts) per buffer allowed. Modules parameter to work around oss limitations.");

#define chip_t sa11xx_uda1341_t

typedef enum stream_id_t{
	PLAYBACK=0,
	CAPTURE,
	MAX_STREAMS,
} stream_id_t;

typedef struct audio_regs {
        unsigned long intMask;
        unsigned long interrupt;
        unsigned long control;
        unsigned long l3Addr;
        unsigned long l3Data;
        unsigned long play;
        unsigned long record;
        unsigned long playCount;
        unsigned long recordCount;
        unsigned long debug;
} audio_regs_t;

typedef struct audio_stream {
        char *id;	/* identification string */
        int  stream_id;	/* numeric identification */	
        int active:1;	/* we are using this stream for transfer now */
        
        snd_pcm_substream_t *stream;

        /* fake a dma */
        dma_device_t dma_dev;	/* device identifier for DMA */
        dma_regs_t *dma_regs;	/* points to our DMA registers */
        int sent_periods;       /* # of sent periods from actual DMA buffer */
        int sent_total;         /* # of sent periods total (just for info & debug) */
        int sync;               /* are we recoding - flag used to do DMA trans. for sync */
} audio_stream_t;

static spinlock_t gLock;
typedef struct snd_card_backpaq_uda1341 {
        // struct pm_dev *pm_dev;        
        snd_card_t *card;
        snd_pcm_t *pcm;
        // snd_info_entry_t *proc_entry;

       /* fake a uda1341 device */
       struct l3_client *uda1341;

       audio_stream_t *s[MAX_STREAMS];
       audio_regs_t *audio_regs;

       unsigned long samplerate;
       unsigned char suspended;
       unsigned char initialized;
} sa11xx_uda1341_t;

static struct snd_card_backpaq_uda1341 *backpaq_uda1341 = NULL;

static unsigned int rates[] = {
       8000, 11025, 16000, 22050, 32000, 44100,
};

#define RATES sizeof(rates) / sizeof(rates[0])

static snd_pcm_hw_constraint_list_t hw_constraints_rates = {
	.count	= RATES,
	.list	= rates,
	.mask	= 0,
};

static unsigned int period_frames[] = {
	2, 8, 16, 32, 64, 128, 256, 384, 512,
};

#define PERIOD_FRAME sizeof(period_frames) / sizeof(period_frames[0])

static snd_pcm_hw_constraint_list_t hw_constraints_period_frames = {
	.count	= PERIOD_FRAME,
	.list	= period_frames,
	.mask	= 0,
};


/* }}} */

/* *** {{{ Clock and sample rate stuff *** */
#ifndef CONFIG_SA1100_IPAQ
#error This driver intended to serve H3x00 handhelds only!
#endif

// static void backpaq_uda1341_set_audio_clock(long rate);
static void backpaq_uda1341_set_audio_power(unsigned long rate);
static void backpaq_uda1341_set_audio_mute(unsigned char state);


static void backpaq_uda1341_get_control(unsigned long *control)
{
        /* read the hardware not the structure register */
        *control = BACKPAQ_AUDIO_CONTROL;
        // *control = backpaq_uda1341->audio_regs->control;
}

static void backpaq_uda1341_set_control(unsigned long *control)
{
	/* update structure */
	backpaq_uda1341->audio_regs->control = *control;

	/* update hardware */
	BACKPAQ_AUDIO_CONTROL = *control;
}


static void backpaq_uda1341_set_intMask(unsigned long intMask)
{
        // printk(KERN_ERR "%s:%d: backpaq_uda1341->audio_reg=%p\n", __FILE__, __LINE__, backpaq_uda1341->audio_regs);

	/* update structure */
	backpaq_uda1341->audio_regs->intMask = intMask;

	/* update hardware */
	BACKPAQ_AUDIO_INT_MASK = intMask;
}


static void backpaq_uda1341_set_samplerate(long rate)
{
        unsigned long control;
	unsigned short sampFreq;
	unsigned short multFactor;

	/* int clk_div = 0; */
	int clk=0;
  
	printk(__FILE__ ": setting audio sample rate to %ld ... \n", rate);
	DEBUG(KERN_DEBUG "set_samplerate rate: %ld\n", rate);

  
	// printk("%s: disabling audio play and record \n", __FUNCTION__);
	/* We don't want to mess with clocks when frames are in flight */
	/* Ser4SSCR0 &= ~SSCR0_SSE; */
	/* disable audio play and record and clear sample rate and mult factor */
	backpaq_uda1341_get_control(&control);
	control &= ((0 << BACKPAQ_AUDIO_CONTROL_PLAY) | (0 << BACKPAQ_AUDIO_CONTROL_RECORD));
	backpaq_uda1341_set_control(&control);
	// printk("%s: done disabling audio play and record. \n", __FUNCTION__);
	
	/* wait for any frame to complete */
	udelay(125);
  

	if (rate >= 44100) {
	        sampFreq = BACKPAQ_AUDIO_SF_44100;
		multFactor = BACKPAQ_AUDIO_FS_256;
		rate = 44100;
		clk = F256;
	} else if (rate >= 32000) {
	        sampFreq = BACKPAQ_AUDIO_SF_32000;
		multFactor = BACKPAQ_AUDIO_FS_384;
		rate = 32000;
		clk = F384;
	} else if (rate >= 22050) {
	        sampFreq = BACKPAQ_AUDIO_SF_22050;
		multFactor = BACKPAQ_AUDIO_FS_256;
		rate = 22050;
		clk = F256;
	} else if (rate >= 16000) {
	        sampFreq = BACKPAQ_AUDIO_SF_16000;
		multFactor = BACKPAQ_AUDIO_FS_384;
		rate = 16000;
		clk = F384;
	} else if (rate >= 11025) {
	        sampFreq = BACKPAQ_AUDIO_SF_11025;
		multFactor = BACKPAQ_AUDIO_FS_256;
		rate = 11025;
		clk = F256;
	} else {
	        sampFreq = BACKPAQ_AUDIO_SF_8000;
		multFactor = BACKPAQ_AUDIO_FS_512;
		rate = 8000;
		clk = F512;
	}

	/* Set the external clock generator */
	//h3600_audio_clock(rate);
	// backpaq_uda1341_set_audio_clock(rate);
  
	// printk("%s: setting uda1341 clock rate... \n", __FUNCTION__);
	/* send sampling rate and multiplication factor to FPGA */
	backpaq_uda1341_get_control(&control);
	control &= ((BACKPAQ_AUDIO_SF_CLR <<  BACKPAQ_AUDIO_CONTROL_SAMPLERATE) | 
		    (BACKPAQ_AUDIO_FS_CLR << BACKPAQ_AUDIO_CONTROL_MULTFACTOR));
	control |= ((multFactor << BACKPAQ_AUDIO_CONTROL_MULTFACTOR) | 
		    (sampFreq << BACKPAQ_AUDIO_CONTROL_SAMPLERATE));
	backpaq_uda1341_set_control(&control);
	
	l3_command(backpaq_uda1341->uda1341, CMD_FS, (void *)clk);
	l3_command(backpaq_uda1341->uda1341, CMD_FORMAT, (void *)FMT_MSB);
	/* l3_command(sa11xx_uda1341->uda1341, CMD_FS, (void *)clk); */
	/* l3_command(sa11xx_uda1341->uda1341, CMD_FORMAT, (void *)LSB16); */
	// printk("%s: done setting uda1341 clock rate. \n", __FUNCTION__);

	/* Ser4SSCR0 = (Ser4SSCR0 & ~0xff00) + clk_div + SSCR0_SSE; */
	DEBUG(KERN_DEBUG "set_samplerate done (new rate: %ld)\n", rate);
	// printk(__FILE__ ": done setting audio sample rate. \n");	
}

/* }}} */

/* *** {{{ HW init and shutdown *** */

static void backpaq_uda1341_audio_init(sa11xx_uda1341_t *backpaq_uda1341)
{
	unsigned long intMask;
	/* unsigned long flags; */

	printk(__FILE__ ": initializing backpaq audio... \n");
	DEBUG_NAME(KERN_DEBUG "backpaq_uda1341_audio_init(%p)\n", backpaq_uda1341);

	
	/* setup play and record buffers */
	if (backpaq_uda1341->s[PLAYBACK]) {
		backpaq_uda1341->s[PLAYBACK]->id = "BACKPAQ UDA1341 out";
		backpaq_uda1341->s[PLAYBACK]->stream_id = PLAYBACK;
		/* I'm using a fake device */
		backpaq_uda1341->s[PLAYBACK]->dma_dev = DMA_WR;
	}

	if (backpaq_uda1341->s[CAPTURE]) {
		backpaq_uda1341->s[CAPTURE]->id = "BACKPAQ UDA1341 in";
		backpaq_uda1341->s[CAPTURE]->stream_id = CAPTURE;
		/* I'm using a fake device */
		backpaq_uda1341->s[CAPTURE]->dma_dev = DMA_RD;
	}

	/* Initialize the UDA1341 internal state */
	backpaq_uda1341_set_audio_mute(1);
	
	/* Setup the uarts */
	/* ????
	local_irq_save(flags);
	GAFR |= (GPIO_SSP_CLK);
	GPDR &= ~(GPIO_SSP_CLK);
	Ser4SSCR0 = 0;
	Ser4SSCR0 = SSCR0_DataSize(16) + SSCR0_TI + SSCR0_SerClkDiv(8);
	Ser4SSCR1 = SSCR1_SClkIactL + SSCR1_SClk1P + SSCR1_ExtClk;
	Ser4SSCR0 |= SSCR0_SSE;
	local_irq_restore(flags);
	*/

	/* Enable the audio power */
	//h3600_audio_power(sa11xx_uda1341->samplerate);
	backpaq_uda1341_set_audio_power(backpaq_uda1341->samplerate);
	
	
	/* Wait for the UDA1341 to wake up */
	mdelay(1);

        
	// printk("%s: initializing uda1341... \n", __FUNCTION__);
	/* Initialize the UDA1341 internal state */
	l3_open(backpaq_uda1341->uda1341);
	// l3_open(sa11xx_uda1341->uda1341);
	// printk("%s: done initializing uda1341. \n", __FUNCTION__);

	/* make the left and right channels unswapped (flip the WS latch ) */
	// Ser4SSDR = 0;
       
	//h3600_audio_mute(0);
	backpaq_uda1341_set_audio_mute(0);

	/* turn volume up */
	/*
	printk(__FILE__ ": increasing volume ...\n");
	BACKPAQ_AUDIO_L3_ADDR = BACKPAQ_AUDIO_CODEC_DATA0;
	BACKPAQ_AUDIO_L3_DATA = (BACKPAQ_AUDIO_CODEC_DATA0_VOL_MASK & 
				 BACKPAQ_AUDIO_CODEC_DATA0_VOL_MID);
	*/
	// printk("%s: masking audio interrupts... \n", __FUNCTION__);
	/* mask all audio interrupts */
	/*
	intMask = (AUDIO_INT_PLAY_FIFO_EMPTY | AUDIO_INT_PLAY_FIFO_FULL |
		   AUDIO_INT_PLAY_FIFO_1_2_FULL | AUDIO_INT_PLAY_FIFO_3_4_FULL |
		   AUDIO_INT_REC_FIFO_EMPTY | AUDIO_INT_REC_FIFO_FULL |
		   AUDIO_INT_REC_FIFO_1_2_FULL | AUDIO_INT_REC_FIFO_3_4_FULL |
		   AUDIO_INT_RO | AUDIO_INT_WO);
	*/
	intMask = MASK_ALL_BACKPAQ_AUDIO_INT;
	backpaq_uda1341_set_intMask(intMask);  
	// unmask the audio in the top level interrupts
	BackpaqFPGAInterruptMask &= ~BACKPAQ_FPGA_INT_AUDIO;

	// printk("%s: done masking audio interrupts. \n", __FUNCTION__);

	printk(__FILE__ ": done initializing backpaq audio. \n");

}

static void backpaq_uda1341_audio_shutdown(sa11xx_uda1341_t *backpaq_uda1341)
{
	/* disable the audio power and all signals leading to the audio chip */
	//h3600_audio_mute(1);
	backpaq_uda1341_set_audio_mute(1);

	l3_close(backpaq_uda1341->uda1341);
	// l3_close(sa11xx_uda1341->uda1341);
	// Ser4SSCR0 = 0;
	/* backpaq_uda1341->uda1341->driver_data->active = 0; */

	//h3600_audio_power(0);
	/* a zero in the second variable does not turn 
	 * the power off, so why are we turning the power 
	 * on again ???? --
	 * doesn't matter because the audio_shutdown function isn't called 
	 * outside power management and that's commented out */
	// sa11xx_uda1341_set_audio_power(sa11xx_uda1341,0);
}


/* }}} */

/* *** {{{ BACKPAQ data transfer *** */

static void audio_process_data(audio_stream_t *s, dma_addr_t dma_ptr, u_int size)
{

	unsigned long fifoSize = BUFSIZE; // in bytes
	unsigned long fifoCount;
	unsigned long bytesXfered = 0;
	// dma_addr_t *curDmaPtr = NULL;
	dma_addr_t *curDmaPtr = dma_ptr;
	int ret;

	// printk(__FILE__ ": processing audio data... \n");
	DEBUG_NAME(KERN_DEBUG "process_data\n");

	// update the dma regs - start address and size
	ret = backpaq_start_dma((s)->dma_regs, dma_ptr, size);
	if (ret)
	        return; // something wen't wrong

	if((s->stream_id) == PLAYBACK) {

	        /* read the fifo count of fpga - how many transfers are allowed */
	        fifoCount = BACKPAQ_AUDIO_PLAY_FC * 2;
		/* change fifoCount to bytes */
		fifoCount = fifoCount*4;

		// printk("%s: waiting for fifo space ...\n", __FUNCTION__);
		/* wait till there's enough room in the fifo for this transfer */
		while ((fifoSize - fifoCount) < size) {
		        fifoCount = BACKPAQ_AUDIO_PLAY_FC * 2;
			fifoCount = fifoCount*4;
		}		  

		// printk("%s: before transfer -- fifo count in bytes = %ld \n", __FUNCTION__, fifoCount);
		// printk("%s: starting data transfer ...\n", __FUNCTION__);
		/* move data from kernel buffer to fpga */
		bytesXfered = 0;
		while ((fifoCount < fifoSize) & (bytesXfered < size)) {
			// curDmaPtr = ((long *)dma_ptr) + (bytesXfered/4);
	                BACKPAQ_AUDIO_PLAY_FIFO = *curDmaPtr;
		        curDmaPtr += 1;
			fifoCount += 4;
			bytesXfered += 4;
		}

	        /* make sure hardware saw all the transfered data */
	        fifoCount = BACKPAQ_AUDIO_PLAY_FC * 2;
		/* change fifoCount to bytes */
		fifoCount *= 4;
		// printk("%s: after transfer -- fifo count in bytes = %ld \n", __FUNCTION__, fifoCount);

		if (bytesXfered != size) {
			printk(KERN_ERR, "%s: didn't transfer all the bytes to hardware!\n", __FUNCTION__);
		}

	} else if ((s->stream_id) == CAPTURE) {

	        /* copy data from fpga to kernerl buffer */
	        fifoCount = BACKPAQ_AUDIO_REC_FC * 2;
		/* change fifoCount to bytes */
		fifoCount = fifoCount*4;

		bytesXfered = 0;
		/*
		while (fifoCount < size) {
		        fifoCount = BACKPAQ_AUDIO_REC_FC * 2;
			fifoCount = fifoCount*4;
		}	
		*/	  
		while ((fifoCount > 0) & (bytesXfered < size)) {
			// curDmaPtr = ((long *)dma_ptr) + (bytesXfered/4);
			*curDmaPtr = BACKPAQ_AUDIO_REC_FIFO;
		        curDmaPtr += 1;
			fifoCount += 4;
			bytesXfered += 4;
		}

		s->dma_regs->DBTA = bytesXfered;

	} else {
	          /* do nothing ???? */
	}	  

	// printk("%s: done transfering data. \n", __FUNCTION__);
	// printk("%s: bytes transfered = %ld \n", __FUNCTION__, bytesXfered);
	//printk(__FILE__ ": done processing audio data. \n");
	
	return;
}

/* }}} */


/* *** {{{ DMA staff *** */
/*********************
 * these are the address and sizes used to fill the xmit buffer so we can get a clock in
 * record only mode
 *********************/
#define FORCE_CLOCK_ADDR		(dma_addr_t)FLUSH_BASE_PHYS
#define FORCE_CLOCK_SIZE		4096 // was 2048

static void audio_dma_request(audio_stream_t *s, void (*callback)(void *))
{
	int ret;
	
	// printk(__FILE__ ": requesting audio dma... \n");
	DEBUG_NAME(KERN_DEBUG "audio_dma_request");

	DEBUG("\t request id <%s>\n", s->id);
	DEBUG("\t  request dma_dev = 0x%x \n", s->dma_dev);
	if((ret = backpaq_request_dma((s)->dma_dev, (s)->id, callback, s, &((s)->dma_regs))))
		printk("%s: backpaq_request_dma returned %d\n", __FUNCTION__, ret);

	// printk(__FILE__ ": done requesting audio dma. \n");

}

static void audio_dma_free(audio_stream_t *s)
{
	printk(__FILE__ ": freeing audio dma... \n");
	backpaq_free_dma((s)->dma_regs);
	(s)->dma_regs = 0;
	// printk(__FILE__ ": done freeing audio dma. \n");
}


static u_int audio_get_dma_pos(audio_stream_t *s)
{
	snd_pcm_substream_t *substream = s->stream;
	snd_pcm_runtime_t *runtime = substream->runtime;
	unsigned int offset;
	// unsigned long hw_int = 0;

	/* sa11xx_uda1341_t *chip = snd_pcm_substream_chip(s->stream); */
	// unsigned long flags;

	// printk(__FILE__ ": getting audio dma position... \n");
	DEBUG_NAME(KERN_DEBUG "get_dma_pos\n");

	// hw_int = BACKPAQ_AUDIO_INT;
	// printk("%s: audio hardware interrupt = 0x%x\n", __FUNCTION__);
        
	// this must be called w/ interrupts locked out see dma-sa1100.c in the kernel
	// spin_lock_irqsave(&gLock, flags);
	// printk("%s: runtime->dma_addr = 0x%x, ", __FUNCTION__, runtime->dma_addr);
	// offset = backpaq_get_dma_pos((s)->dma_regs) - runtime->dma_addr;
	// spin_unlock_irqrestore(&gLock, flags);
	
	// printk("%s: runtime->dma_addr = 0x%x\n", __FUNCTION__, runtime->dma_addr);
	offset = backpaq_get_dma_pos((s)->dma_regs);		
	// printk("%s: get dma position returned 0x%x\n", __FUNCTION__, offset);
	// offset = offset - runtime->dma_addr;
	// printk("%s: offset = %d [bytes], ", __FUNCTION__, offset);
	offset = bytes_to_frames(runtime,offset);	
	// printk(" %d [fr]\n", offset);

        // printk("%s: runtime->buffer_size = %d\n", __FUNCTION__, runtime->buffer_size);

	if (offset >= runtime->buffer_size){
		offset = runtime->buffer_size;
	}

	DEBUG(KERN_DEBUG "  hw_ptr_interrupt: 0x%lx\n",
	      (unsigned long)runtime->hw_ptr_interrupt);
	DEBUG(KERN_DEBUG "  updated pos [fr]: %ld\n",
	      offset - (offset % runtime->min_align));

	// printk(__FILE__ ": done getting audio dma position. \n");
 
 	return offset;
}

// this stops the dma and clears the dma ptrs
static void audio_stop_dma(audio_stream_t *s)
{
	long flags;

	// printk(__FILE__ ": stopping audio dma... \n");
	DEBUG_NAME(KERN_DEBUG "stop_dma\n");
	// zero filling streams (sync=1) don;t have alsa streams attached
	// but the 0 fill dma xfer still needs to be stopped
	if (!(s->stream || s->sync))
		return;

	spin_lock_irqsave(&gLock, flags);	
	s->active = 0;
	s->sent_periods = 0;
	s->sent_total = 0;
	s->sync = 0;	
	// this stops the dma channel and clears the buffer ptrs
	backpaq_clear_dma((s)->dma_regs);	
	spin_unlock_irqrestore(&gLock, flags);

	// printk(__FILE__ ": done stopping audio dma.\n");
}

static void audio_reset(audio_stream_t *s)
{
	printk(__FILE__ ": reseting audio... \n");
        DEBUG_NAME(KERN_DEBUG "dma_reset\n");
        
	if (s->stream) {
	  audio_stop_dma(s);
	}
	s->active = 0;
	// printk(__FILE__ ": done reseting audio. \n");
}

static void audio_process_dma(audio_stream_t *s)
{
	snd_pcm_substream_t *substream = s->stream;
	snd_pcm_runtime_t *runtime;
	// int ret;
	int ctr;
	static unsigned long totalBytesXfered = 0;

	// printk(__FILE__ ": processing audio dma... \n");
      	DEBUG_NAME(KERN_DEBUG "process_dma\n");

	/* don't need this anymore
	if(!s->active && s->sync){
		snd_assert(s->stream_id == PLAYBACK,return);		
		// fill the xmit dma buffers and return
		while (1) {
			DEBUG(KERN_DEBUG "sent zero dma  period (dma_size[B]: %d)\n", 
			FORCE_CLOCK_SIZE);
			ret = sa1100_start_dma((s)->dma_regs, FORCE_CLOCK_ADDR, 
			                       FORCE_CLOCK_SIZE);			
			if (ret)
				return;   
		}
	}
	*/

	runtime = substream->runtime;
        ctr=0;

	DEBUG("audio_process_dma hw_ptr_base = 0x%lx hw_ptr_interrupt = 0x%lx period_size = %ld  periods  = %d buffer_size = %ld sync=0x%lx  dma_area = 0x%lx \n",
	       runtime->hw_ptr_base,
	       runtime->hw_ptr_interrupt,
	       runtime->period_size,
	       runtime->periods,
	       runtime->buffer_size,
	       runtime->sync,
	       runtime->dma_area);

	DEBUG("audio_process_dma sent_total = %d  sent_period = %d\n",
	       s->sent_total,
	       s->sent_periods);

	if (!(s->active)) { 
		printk(KERN_ERR "%s: audio stream is not active\n", __FUNCTION__);
	        return;
	}

	// while (!(done)) {       
	if (s->active) { 
		unsigned int dma_size;		
		unsigned int offset;
                ctr++;

		dma_size = frames_to_bytes(runtime,runtime->period_size) ;
		if (dma_size > MAX_DMA_SIZE){
			/* this should not happen! */
			printk(KERN_ERR "-----> cut dma_size: %d -> ", dma_size);
			dma_size = CUT_DMA_SIZE;
			printk(KERN_ERR "%d <-----\n", dma_size);
		}
		offset = dma_size * s->sent_periods;
		
		// printk("%s: period_size = %d, periods = %d\n", __FUNCTION__, runtime->period_size, runtime->periods);
		// printk("%s: dma_size = %d, offset = %d, xfer_align = %d\n",  __FUNCTION__, dma_size, offset, runtime->xfer_align);

		/* ret = sa1100_start_dma((s)->dma_regs, runtime->dma_addr + offset, 
		                          dma_size);
		// the first time this while loop will run 3 times, i.e. it'll fill 
	        // the 2 dma buffers then get a -EBUSY, every other time it'll refill 
	        // the completed buffer and then get the -EBUSY
	        // so it'll just run twice
		*/

		/* replace above with hardware write/read */
		audio_process_data(s, runtime->dma_addr + offset, dma_size);
		
		DEBUG(KERN_DEBUG "sent period %d (%d total)(dma_size[B]: %d"
		      "offset[B]: %06d)\n",
		      s->sent_periods, s->sent_total, dma_size, offset);

#if 0
#ifdef DEBUG_MODE
		printk(KERN_DEBUG "  dma_area:");
		for (i=0; i < 32; i++) {
			printk(" %02x", *(char *)(runtime->dma_addr+offset+i));
		}
		printk("\n", __FUNCTION__);
#endif  
#endif  

		totalBytesXfered += (s->dma_regs->DBTA);
		//printk("%s: total bytes xfered = %d\n", __FUNCTION__, totalBytesXfered);
		if (totalBytesXfered == dma_size) {
			s->sent_total++;
			s->sent_periods++;
			// set dma total bytes transfered
			s->dma_regs->DBTA = dma_size * (s->sent_periods);
			// printk("%s: dma bytes xfered = %d\n", __FUNCTION__, s->dma_regs->DBTA);
			totalBytesXfered = 0;
			// inform alsa a period has elapsed
			snd_pcm_period_elapsed(s->stream);
			// printk("%s: runtime->status->state = %d\n", __FUNCTION__, runtime->status->state);

			s->sent_periods %= runtime->periods;
			if (!(s->sent_periods)) {
				/* set the dma done bit */
				// s->dma_regs->DCSR |= DCSR_DONEA;
				// s->dma_regs->DBTA = dma_size * (runtime->periods);
				printk("%s: dma wrap around\n", __FUNCTION__);
				// is there a special alsa call here ????
			}
		} else {
			printk("%s: partial transfers in progress ... \n", __FUNCTION__);
			while (totalBytesXfered < dma_size) {
				unsigned int partial_size = dma_size - totalBytesXfered;
				unsigned int partial_offset = runtime->dma_addr + offset + totalBytesXfered;
				printk("dma_offset = %d, dma_size = %d\n", partial_offset, partial_size);
				audio_process_data(s, partial_offset, partial_size);
				totalBytesXfered += (s->dma_regs->DBTA);
			}
		}

	}
	// printk(__FILE__ ": done processing audio dma. \n");
}


static void audio_dma_callback(void *data)
{
	audio_stream_t *s = data;
	unsigned long hw_int;

	// printk("%s: beginning dma callback ... \n", __FUNCTION__);

	DEBUG_NAME(KERN_DEBUG "dma_callback\n");
	DEBUG(KERN_DEBUG "----> period done <----\n");
#if 0
#ifdef DEBUG_MODE
	printk(KERN_DEBUG "  dma_area:");
	buf = s->stream->runtime->dma_addr +
		((s->sent_periods - 1 ) *
		 frames_to_bytes( s->stream->runtime, s->stream->runtime->period_size));	
	for (i=0; i < 32; i++) {
		printk(" %02x", *(char *)(buf + i));
	}
	printk("\n", __FUNCTION__);
#endif                      
#endif

	// clear hw interrupt
	// hw_int = BACKPAQ_AUDIO_INT;
	// printk("%s: audio hardware interrupt = 0x%lx\n", __FUNCTION__, hw_int);
	/*
	if (hw_int & AUDIO_INT_PLAY_FIFO_EMPTY) {
	        printk("%s: received play fifo empty interrupt\n", __FUNCTION__);
	}
	if (hw_int & AUDIO_INT_PLAY_FIFO_1_2_FULL) {
	        printk("%s: received play fifo 1/2 full interrupt\n", __FUNCTION__);
	}
	else if (hw_int & AUDIO_INT_REC_FIFO_FULL) {
	        printk("%s: received record fifo full interrupt\n", __FUNCTION__);
	}
	*/
// If we are getting a callback for an active stream then we inform the PCM middle layer
// we've finished a period
	// if (s->active) {
	if (s->active) {
	        // snd_pcm_period_elapsed(s->stream);
		audio_process_dma(s);
	}
}


/* }}} */


/* *** {{{ PCM setting *** */

/* *** {{{ trigger & timer *** */

static int snd_card_backpaq_uda1341_pcm_trigger(snd_pcm_substream_t *substream, stream_id_t stream_id, int cmd)
{
        sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime; 
	unsigned long intMask;

	// printk(__FILE__ ": begin pcm trigger... \n");

	DEBUG_NAME(KERN_DEBUG "pcm_trigger id: %d cmd: %d\n", stream_id, cmd);

	DEBUG(KERN_DEBUG "  sound: %d x %d [Hz]\n", runtime->channels, runtime->rate);
	DEBUG(KERN_DEBUG "  periods: %ld x %ld [fr]\n", (unsigned long)runtime->periods,
	      (unsigned long) runtime->period_size);
	DEBUG(KERN_DEBUG "  buffer_size: %ld [fr]\n", (unsigned long)runtime->buffer_size);
#if 0
	printk( "pcm_trigger id: %d cmd: %d\n", stream_id, cmd);

	printk( "  sound: %d channels at  %d [Hz]\n", runtime->channels, runtime->rate);
	printk( "  periods: %ld with a size of  %ld [bytes]\n", 
		(unsigned long)runtime->periods, 
		(unsigned long) frames_to_bytes(runtime,runtime->period_size));
	printk( "  buffer_size: %ld [bytes]\n", 
		(unsigned long)frames_to_bytes(runtime,runtime->buffer_size));
	printk( "  sample_bits = %d [bytes] \n", runtime->sample_bits/8);
	
#endif

#ifdef DEBUG_MODE
	/*
	printk(KERN_DEBUG "  dma_area:");
	for (i=0; i < 32; i++) {
		printk(" %02x", *(char *)(runtime->dma_addr+i));
	}
	printk("\n");
	*/
#endif
        
	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		/* requested stream startup */
		//printk("%s: turning active on ...\n", __FUNCTION__);
		backpaq_uda1341->s[stream_id]->active = 1;
		audio_process_dma(backpaq_uda1341->s[stream_id]);
		/* unmask play 1/2 full interrupts */
		intMask = (backpaq_uda1341->audio_regs->intMask & 
			   ~AUDIO_INT_PLAY_FIFO_1_2_FULL);
		backpaq_uda1341_set_intMask(intMask);
		//printk("%s: unmasking audio interrupts\n", __FUNCTION__);
		// unmask the audio interrupts
		BackpaqFPGAInterruptMask &= ~BACKPAQ_FPGA_INT_AUDIO;
		/*
		if (stream_id == PLAYBACK)
		  enable_irq(AUDIO_DMA_PLAY_INTERRUPT);
		else
		  enable_irq(AUDIO_DMA_REC_INTERRUPT);
		*/
		// printk("%s: leaving pcm trigger\n", __FUNCTION__);
		break;
	case SNDRV_PCM_TRIGGER_SUSPEND:
		/* requested stream suspend */
		backpaq_uda1341->s[stream_id]->active = 0;
		audio_stop_dma(backpaq_uda1341->s[stream_id]);
		/* mask all audio interrupts */
		BackpaqFPGAInterruptMask |= BACKPAQ_FPGA_INT_AUDIO;
		break;
	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		/* requested stream shutdown */
		backpaq_uda1341->s[stream_id]->active = 0;
		audio_stop_dma(backpaq_uda1341->s[stream_id]);
		/* mask all audio interrupts */
		BackpaqFPGAInterruptMask |= BACKPAQ_FPGA_INT_AUDIO;
		break;
	default:
		return -EINVAL;
		break;
	}
	// printk("%s: pcm trigger done. \n", __FUNCTION__);
	return 0;
}

/* }}} */

static snd_pcm_hardware_t snd_backpaq_uda1341_capture =
{

  /* SND_PCM_FLG_NONBLOCK -- do not block when play buffer is full/capture buffer empty */
	.info			= (SNDRV_PCM_INFO_NONINTERLEAVED |
				   SNDRV_PCM_INFO_BLOCK_TRANSFER |
				   SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats		= SNDRV_PCM_FMTBIT_S16_LE,
	.rates			= (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_11025 |\
				   SNDRV_PCM_RATE_16000 | SNDRV_PCM_RATE_22050 |\
				   SNDRV_PCM_RATE_32000 | SNDRV_PCM_RATE_44100),
	.rate_min		= 8000,
	.rate_max		= 44100,
	.channels_min		= 1,
	.channels_max		= 1,
	.buffer_bytes_max	= 65536, // was 65536 
	//.period_bytes_min	= 64,
	//.period_bytes_max	= 8190, /* <= MAX_DMA_SIZE from ams/arch-sa1100/dma.h */
	//.periods_min		= 2,
	.period_bytes_min	= 4,
	.period_bytes_max	= 1024,
	.periods_min		= 1,
	.periods_max		= 255, // dropped from 255 to 32 to  lower overhead costs
	//.fifo_size		= 0,
};

static snd_pcm_hardware_t snd_backpaq_uda1341_playback =
{
	.info			= (SNDRV_PCM_INFO_NONINTERLEAVED |
				   SNDRV_PCM_INFO_BLOCK_TRANSFER |
				   SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats		= SNDRV_PCM_FMTBIT_S16_LE,
	.rates			= (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_11025 |\
				   SNDRV_PCM_RATE_16000 | SNDRV_PCM_RATE_22050 |\
				   SNDRV_PCM_RATE_32000 | SNDRV_PCM_RATE_44100),
	/* | SNDRV_PCM_RATE_KNOT),  // No need for knot bit since we're not using 
	   unconventional rates */
	.rate_min		= 8000,
	.rate_max		= 44100,
	.channels_min		= 1,
	.channels_max		= 1,
	.buffer_bytes_max	= 65536, // was 65536 
	//.period_bytes_min	= 64,
	//.period_bytes_max	= 8190, /* <= MAX_DMA_SIZE from ams/arch-sa1100/dma.h */
	//.periods_min		= 2,
	.period_bytes_min	= 4,
	.period_bytes_max	= 1024,
	.periods_min		= 1,
	.periods_max		= 255, // dropped from 255 to 32 to  lower overhead costs
	//.fifo_size		= 0,
};



static int backpaq_UDA1341_hw_rule_period_size(snd_pcm_hw_params_t *params, snd_pcm_hw_rule_t *rule)
{
	snd_interval_t *r = hw_param_interval(params, SNDRV_PCM_HW_PARAM_PERIOD_SIZE);
	return snd_interval_list(r, (&hw_constraints_period_frames)->count, (&hw_constraints_period_frames)->list, (&hw_constraints_period_frames)->mask);
}


/* {{{ snd_card_sa11xx_uda1341_playback functions */
static int snd_card_backpaq_uda1341_playback_open(snd_pcm_substream_t *substream)
{
	sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	int err;
        
	DEBUG_NAME(KERN_DEBUG "playback_open\n");

	backpaq_uda1341->s[PLAYBACK]->stream = substream;
	backpaq_uda1341->s[PLAYBACK]->sent_periods = 0;
	backpaq_uda1341->s[PLAYBACK]->sent_total = 0;

	/* not true anymore */
	// no reset here since we may be zero filling the DMA
	// if we are, the dma stream will get reset in the pcm_trigger
	// i.e. when it actually starts to play
	audio_reset(backpaq_uda1341->s[PLAYBACK]);
 
	runtime->hw = snd_backpaq_uda1341_playback;
	runtime->dma_bytes = BACKPAQ_DMA_BYTES; 

	// runtime->hw.periods_max = max_periods_allowed;

	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					      &hw_constraints_rates)) < 0)
		return err;

#if 0
	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_PERIOD_SIZE,
					      &hw_constraints_period_frames)) < 0) {
		printk("%s: setting hw period size returned %d\n", __FUNCTION__, err);
		return err;
        }
#endif


#if 0
	if ((err = snd_pcm_hw_rule_add(runtime, 0, SNDRV_PCM_HW_PARAM_PERIOD_SIZE,
				       backpaq_UDA1341_hw_rule_period_size, backpaq_uda1341,
				       SNDRV_PCM_HW_PARAM_PERIOD_SIZE, -1)) < 0) {
		printk("%s: setting hw period size returned %d\n", __FUNCTION__, err);
		return err;
	}
#endif

#if 0
	snd_interval_list(hw_param_interval(backpaq_uda1341, SNDRV_PCM_HW_PARAM_PERIOD_SIZE), (&hw_constraints_period_frames)->count, (&hw_constraints_period_frames)->list, (&hw_constraints_period_frames)->mask);
#endif

	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0)
		return err;

#if 0
	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIOD_SIZE)) < 0)
		return err;
#endif 
	// printk("%s: period_size = %d, periods = %d\n", __FUNCTION__, runtime->period_size, runtime->periods);

#if 0
	// dont' have swparams
	printk("%s: setting xfer align ... \n", __FUNCTION__);
	if ((err = snd_pcm_sw_params_set_xfer_align(runtime, swparams, 2)) < 0) {
		printk("%s: setting xfer_align returned %d\n", __FUNCTION__, err);
		return err;
        }
#endif

#if 0
	printk("%s: runtime->xfer_align = %d\n", __FUNCTION__, runtime->xfer_align);
	runtime->xfer_align = 0x2; // align with 2 samples -- multiple of 4 bytes
	printk("%s: changing runtime->xfer_align to %d\n", __FUNCTION__, runtime->xfer_align);
#endif

#if 0
	if ((err = snd_pcm_hw_params_set_period_size_near(runtime, SNDRV_PCM_HW_PARAM_PERIOD_SIZE,
					      512, 0)) < 0) {
		printk("%s: setting hw period size returned %d\n", __FUNCTION__, err);
		return err;
        }
#endif 


	return 0;
}

static int snd_card_backpaq_uda1341_playback_close(snd_pcm_substream_t *substream)
{
	sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);

	DEBUG_NAME(KERN_DEBUG "playback_close\n");

	backpaq_uda1341->s[PLAYBACK]->stream = NULL;
      
	return 0;
}

static int snd_card_backpaq_uda1341_playback_ioctl(snd_pcm_substream_t *substream, unsigned int cmd, void *arg)
{

	DEBUG_NAME(KERN_DEBUG "playback_ioctl cmd: %d\n", cmd);
	return snd_pcm_lib_ioctl(substream, cmd, arg);
}

static int snd_card_backpaq_uda1341_playback_prepare(snd_pcm_substream_t *substream)
{
	// sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	// printk("%s: preparing for play ... \n", __FUNCTION__);        
	DEBUG_NAME(KERN_DEBUG "playback_prepare\n");
                
	// printk("%s: calling set_samplerate with %ld \n", __FUNCTION__, runtime->rate);        
	/* set requested samplerate */
	backpaq_uda1341_set_samplerate(runtime->rate);
	// printk("%s: done preparing for play. \n", __FUNCTION__);        
        
	return 0;
}

static int snd_card_backpaq_uda1341_playback_trigger(snd_pcm_substream_t *substream, int cmd)
{
        // sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	unsigned long control;

	// printk("%s: setting play enable = %d \n", __FUNCTION__, cmd);
	DEBUG_NAME(KERN_DEBUG "playback_trigger\n");
	
	/* enable/disable play if cmd is 1/0 (go/stop) */
	backpaq_uda1341_get_control(&control);
	control |= (cmd << BACKPAQ_AUDIO_CONTROL_PLAY);
	backpaq_uda1341_set_control(&control);

	return snd_card_backpaq_uda1341_pcm_trigger(substream, PLAYBACK, cmd);
}


static snd_pcm_uframes_t snd_card_backpaq_uda1341_playback_pointer(snd_pcm_substream_t *substream)
{
	sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	snd_pcm_uframes_t pos;
	
	DEBUG_NAME(KERN_DEBUG "playback_pointer\n");        
	pos = audio_get_dma_pos(backpaq_uda1341->s[PLAYBACK]);
	// printk("%s: position = %d \n", __FUNCTION__, pos);
	return pos;
	
}

/* }}} */

/* {{{ snd_card_sa11xx_uda1341_record functions */

static int snd_card_backpaq_uda1341_capture_open(snd_pcm_substream_t *substream)
{
	sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	int err;

	DEBUG_NAME(KERN_DEBUG "record_open\n");
	
		
	backpaq_uda1341->s[CAPTURE]->stream = substream;
	backpaq_uda1341->s[CAPTURE]->sent_periods = 0;
	backpaq_uda1341->s[CAPTURE]->sent_total = 0;
        
	audio_reset(backpaq_uda1341->s[CAPTURE]);        

	runtime->hw = snd_backpaq_uda1341_capture;
	runtime->hw.periods_max = max_periods_allowed;
	
	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0)
		return err;
	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					      &hw_constraints_rates)) < 0)
		return err;
#if 0
	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_PERIOD_SIZE,
					      &hw_constraints_period_sizes)) < 0)
		return err;
#endif   
	return 0;
}

static int snd_card_backpaq_uda1341_capture_close(snd_pcm_substream_t *substream)
{
	sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);

	DEBUG_NAME(KERN_DEBUG "record_close\n");
        
	backpaq_uda1341->s[CAPTURE]->stream = NULL;

	return 0;
}

static int snd_card_backpaq_uda1341_capture_ioctl(snd_pcm_substream_t *substream, unsigned int cmd, void *arg)
{

	DEBUG_NAME(KERN_DEBUG "record_ioctl cmd: %d\n", cmd);
	return snd_pcm_lib_ioctl(substream, cmd, arg);
	/* hardware specific stuff here */
}

static int snd_card_backpaq_uda1341_capture_prepare(snd_pcm_substream_t *substream)
{
	// sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
        
	printk("%s: preparing for record \n", __FUNCTION__);        
	DEBUG_NAME(KERN_DEBUG "record_prepare\n");

	/* set requested samplerate */
	backpaq_uda1341_set_samplerate(runtime->rate);

	return 0;
}


static int snd_card_backpaq_uda1341_capture_trigger(snd_pcm_substream_t *substream, int cmd)
{
	// sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	unsigned long control;

	DEBUG_NAME(KERN_DEBUG "record_trigger\n");

	/* enable/disable record if cmd is 1/0 (go/stop) */
	backpaq_uda1341_get_control(&control);
	control |= (cmd << BACKPAQ_AUDIO_CONTROL_RECORD);
	backpaq_uda1341_set_control(&control);
	
	return snd_card_backpaq_uda1341_pcm_trigger(substream, CAPTURE, cmd);
}


static snd_pcm_uframes_t snd_card_backpaq_uda1341_capture_pointer(snd_pcm_substream_t *substream)
{
	sa11xx_uda1341_t *backpaq_uda1341 = snd_pcm_substream_chip(substream);
	snd_pcm_uframes_t pos;

	DEBUG_NAME(KERN_DEBUG "record_pointer\n");
	pos = audio_get_dma_pos(backpaq_uda1341->s[CAPTURE]);
	return pos;
	
}

/* }}} */

/* {{{ HW params & free */

static int snd_backpaq_uda1341_hw_params(snd_pcm_substream_t *substream, snd_pcm_hw_params_t *hw_params)
{
        
	DEBUG_NAME(KERN_DEBUG "hw_params\n");
	
	return snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params));

	/* hardware setup and buffer allocation should be done here */
}

static int snd_backpaq_uda1341_hw_free(snd_pcm_substream_t *substream)
{

	DEBUG_NAME(KERN_DEBUG "hw_free\n");
		
	return snd_pcm_lib_free_pages(substream);
}

/* }}} */


static snd_pcm_ops_t snd_card_backpaq_uda1341_playback_ops = {
	.open		  = snd_card_backpaq_uda1341_playback_open,
	.close		  = snd_card_backpaq_uda1341_playback_close,
	.ioctl		  = snd_card_backpaq_uda1341_playback_ioctl,
	.hw_params        = snd_backpaq_uda1341_hw_params,
	.hw_free	  = snd_backpaq_uda1341_hw_free,
	.prepare	  = snd_card_backpaq_uda1341_playback_prepare,
	.trigger	  = snd_card_backpaq_uda1341_playback_trigger,
	.pointer          = snd_card_backpaq_uda1341_playback_pointer,
};

static snd_pcm_ops_t snd_card_backpaq_uda1341_capture_ops = {
	.open		  = snd_card_backpaq_uda1341_capture_open,
	.close		  = snd_card_backpaq_uda1341_capture_close,
	.ioctl		  = snd_card_backpaq_uda1341_capture_ioctl,
	.hw_params        = snd_backpaq_uda1341_hw_params,
	.hw_free	  = snd_backpaq_uda1341_hw_free,
	.prepare	  = snd_card_backpaq_uda1341_capture_prepare,
	.trigger	  = snd_card_backpaq_uda1341_capture_trigger,
	.pointer	  = snd_card_backpaq_uda1341_capture_pointer,
};


static int __init snd_card_backpaq_uda1341_pcm(sa11xx_uda1341_t *backpaq_uda1341, int device, int substreams)
{

	int err;
	
	printk(__FILE__ ": begin initializing pcm... \n");
	DEBUG_NAME(KERN_DEBUG "backpaq_uda1341_pcm\n");
	// audio power is turned on and a clock is set at the same time so 
	// this gives us a default
	// starting rate
	backpaq_uda1341->samplerate = AUDIO_RATE_DEFAULT;

	backpaq_uda1341->suspended = 0;
	
	printk("%s: new card setup... \n", __FUNCTION__);
	if ((err = snd_pcm_new(backpaq_uda1341->card, "UDA1341 PCM", device,
			       substreams, substreams, &(backpaq_uda1341->pcm))) < 0)
		return err;
	// printk("%s: done. \n", __FUNCTION__);

	printk("%s: allocating space for sound driver buffers... \n", __FUNCTION__);
	// this sets up our initial buffers and sets the dma_type to isa.
	// isa works but I'm not sure why (or if) it's the right choice
	// this may be too large, trying it for now
	snd_pcm_lib_preallocate_isa_pages_for_all(backpaq_uda1341->pcm, 64*1024, 64*1024);
	
	snd_pcm_set_ops(backpaq_uda1341->pcm, SNDRV_PCM_STREAM_PLAYBACK, 
			&snd_card_backpaq_uda1341_playback_ops);
	snd_pcm_set_ops(backpaq_uda1341->pcm, SNDRV_PCM_STREAM_CAPTURE, 
			&snd_card_backpaq_uda1341_capture_ops);

	backpaq_uda1341->pcm->private_data = backpaq_uda1341;
	backpaq_uda1341->pcm->info_flags = 0;
	strcpy(backpaq_uda1341->pcm->name, "UDA1341 PCM");

	backpaq_uda1341->audio_regs = snd_kcalloc(sizeof(audio_regs_t), GFP_KERNEL);
	backpaq_uda1341->s[PLAYBACK] = snd_kcalloc(sizeof(audio_stream_t), GFP_KERNEL);
	backpaq_uda1341->s[CAPTURE] = snd_kcalloc(sizeof(audio_stream_t), GFP_KERNEL);

	// printk("%s: done. \n", __FUNCTION__);

        backpaq_uda1341_audio_init(backpaq_uda1341);

	printk("%s: requesting dma... \n", __FUNCTION__);
	/* setup DMA controller */
	audio_dma_request(backpaq_uda1341->s[PLAYBACK], audio_dma_callback);
	audio_dma_request(backpaq_uda1341->s[CAPTURE], audio_dma_callback);
	// printk("%s: done. \n", __FUNCTION__);

	return 0;
}


/* }}} */

/* {{{ module init & exit */

// turn it off for now
#if 0
#ifdef CONFIG_PM

static int sa11xx_uda1341_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{
	sa11xx_uda1341_t *sa11xx_uda1341 = pm_dev->data;
	audio_stream_t *is, *os;	
	int stopstate;
	snd_card_t *card;

	
	DEBUG_NAME(KERN_DEBUG "pm_callback\n");
	printk(KERN_DEBUG "%s: req = %d\n", __FUNCTION__, req);
	
	
	is = sa11xx_uda1341->s[PLAYBACK];
	os = sa11xx_uda1341->s[CAPTURE];
	card = sa11xx_uda1341->card;
	
	switch (req) {
	case PM_SUSPEND: /* enter D1-D3 */
		if (card->power_state == SNDRV_CTL_POWER_D3hot){
			printk("%s: exiting eraly from suspend\n", __FUNCTION__);			
			return 0;
		}
		snd_power_lock(card);
		snd_pcm_suspend_all(sa11xx_uda1341->pcm);
		if (!sa11xx_uda1341->suspended){			
			// stop the hardware
			sa11xx_uda1341_audio_shutdown(sa11xx_uda1341);
			if (is->active)
				sa1100_stop_dma(is->dma_regs);
		
			if (os->active || os->sync)
				sa1100_stop_dma(os->dma_regs);
			audio_dma_free(sa11xx_uda1341->s[PLAYBACK]);
			audio_dma_free(sa11xx_uda1341->s[CAPTURE]);
			sa11xx_uda1341->suspended = 1;
		}		
		snd_power_change_state(card, SNDRV_CTL_POWER_D3hot);
		snd_power_unlock(card);		
		
		break;
	case PM_RESUME:  /* enter D0 */

		if (card->power_state == SNDRV_CTL_POWER_D0){
			printk("%s: exiting eraly from resume\n", __FUNCTION__);
			return 0;
		}
		snd_power_lock(card);


		if (sa11xx_uda1341->suspended){
			/* setup DMA controller */
			audio_dma_request(sa11xx_uda1341->s[PLAYBACK], audio_dma_callback);
			audio_dma_request(sa11xx_uda1341->s[CAPTURE], audio_dma_callback);
			sa11xx_uda1341_audio_init(sa11xx_uda1341);
			sa11xx_uda1341->suspended = 0;
		}		
		snd_power_change_state(card, SNDRV_CTL_POWER_D0);
		snd_power_unlock(card);

		break;
	}
	printk(KERN_DEBUG __FUNCTION__": exiting...\n");	
        return 0;
	
}
#endif
#endif

void snd_backpaq_uda1341_free(snd_card_t *card)
{
	sa11xx_uda1341_t *backpaq_uda1341 = snd_magic_cast(sa11xx_uda1341_t, card->private_data, return);

	printk(__FILE__ ": begin freeing sound card... \n");
	DEBUG_NAME(KERN_DEBUG "snd_backpaq_uda1341_free\n");

	if (backpaq_uda1341->s[PLAYBACK] != NULL) {
	        audio_dma_free(backpaq_uda1341->s[PLAYBACK]);
		kfree(backpaq_uda1341->s[PLAYBACK]);
		backpaq_uda1341->s[PLAYBACK] = NULL;
	}

	if (backpaq_uda1341->s[CAPTURE] != NULL) {
	        audio_dma_free(backpaq_uda1341->s[CAPTURE]);
		kfree(backpaq_uda1341->s[CAPTURE]);
		backpaq_uda1341->s[CAPTURE] = NULL;
	}

	snd_magic_kfree(backpaq_uda1341);
	card->private_data = NULL;
	// printk("%s: done. \n", __FUNCTION__);
}

/***************************************************************************************
 *
 *
 *  the hal layer abstraction interface.
 *
 *
 *
 *
 **************************************************************************************/

#if !(defined(CONFIG_IPAQ_HAL) || defined(CONFIG_IPAQ_HAL_MODULE))
#warning "Running w/o the IPAQ HAL layer, only 36xx and 31xx IPAQS will function"
#define GPIO_H3600_CLK_SET0		GPIO_GPIO (12)
#define GPIO_H3600_CLK_SET1		GPIO_GPIO (13)
#define	clr_sa11xx_uda1341_egpio(x)	clr_h3600_egpio(x)
#define set_sa11xx_uda1341_egpio(x)	set_h3600_egpio(x)
#endif




// static void backpaq_uda1341_set_audio_clock(long rate)
// {
  /*
#if defined(CONFIG_IPAQ_HAL) || defined(CONFIG_IPAQ_HAL_MODULE)

        // h3600_audio_clock(rate);
        h3600_audio_clock(44100);
#else
        GPCR = GPIO_H3600_CLK_SET0 | GPIO_H3600_CLK_SET1;
  */
  /*
	switch (rate) {
	case 24000: case 32000: case 48000:	// 00: 12.288 MHz
		GPCR = GPIO_H3600_CLK_SET0 | GPIO_H3600_CLK_SET1;
		break;

	case 22050: case 29400: case 44100:	// 01: 11.2896 MHz
		GPSR = GPIO_H3600_CLK_SET0;
		GPCR = GPIO_H3600_CLK_SET1;
		break;

	case 8000: case 10666: case 16000:	// 10: 4.096 MHz
		GPCR = GPIO_H3600_CLK_SET0;
		GPSR = GPIO_H3600_CLK_SET1;
		break;

	case 10985: case 14647: case 21970:	// 11: 5.6245 MHz
		GPSR = GPIO_H3600_CLK_SET0 | GPIO_H3600_CLK_SET1;
		break;
	}
  */
  // #endif
// }


static void backpaq_uda1341_set_audio_power(unsigned long default_rate)
{

#if defined(CONFIG_IPAQ_HAL) || defined(CONFIG_IPAQ_HAL_MODULE)
	h3600_audio_power(default_rate);
#else

	// volatile unsigned long *genControlReg = (unsigned long *) (_BackpaqSysctlGenControl);
	unsigned long controlData;
	unsigned long flags;

	// printk(__FILE__ ": setting audio power ... \n");
	controlData = BackpaqSysctlGenControl;
	/* ???? */
	local_irq_save(flags);

	/* turn fpga power on, including DACs */
	/* should this be done with fpga driver and I only turn the amps on ???? */
	BackpaqSysctlGenControl = controlData | BACKPAQ3_GENCTL_50VEN | 
	  BACKPAQ3_GENCTL_18VEN | BACKPAQ3_GENCTL_AUDIO_AMPEN;
  
	// printk(__FILE__ ": done. \n");

	/*
        if (default_rate){
		local_irq_save(flags);

		// Enable the audio power
		clr_sa11xx_uda1341_egpio(IPAQ_EGPIO_CODEC_NRESET);
		set_sa11xx_uda1341_egpio(IPAQ_EGPIO_AUDIO_ON);
		set_sa11xx_uda1341_egpio(IPAQ_EGPIO_QMUTE);
		local_irq_restore(flags);
		sa11xx_uda1341_set_samplerate(sa11xx_uda1341, default_rate); 
		set_sa11xx_uda1341_egpio(IPAQ_EGPIO_CODEC_NRESET);
	}
	else{
		clr_sa11xx_uda1341_egpio(IPAQ_EGPIO_CODEC_NRESET);
		clr_sa11xx_uda1341_egpio(IPAQ_EGPIO_AUDIO_ON);
	}
	
	*/
		
#endif

}

static void backpaq_uda1341_set_audio_mute(unsigned char state)
{
#if 0
        volatile unsigned long *audioL3AddrReg = (unsigned long *) (BACKPAQ_AUDIO_L3_ADDR);
	volatile unsigned long *audioL3DataReg = (unsigned long *) (BACKPAQ_AUDIO_L3_DATA);
#endif

// #if defined(CONFIG_IPAQ_HAL) || defined(CONFIG_IPAQ_HAL_MODULE)
	//	h3600_audio_mute(state);
// #else


	BACKPAQ_AUDIO_L3_ADDR = BACKPAQ_AUDIO_CODEC_DATA0;
	if (state) {
	        // printk(__FILE__ ": begin audio muting ... \n");
	        BACKPAQ_AUDIO_L3_DATA = (BACKPAQ_AUDIO_CODEC_DATA0_PP_DE_MT_M_EN | 
					 BACKPAQ_AUDIO_CODEC_DATA0_MT);
	} else {
	        // printk(__FILE__ ": begin audio unmuting ... \n");
	        BACKPAQ_AUDIO_L3_DATA = BACKPAQ_AUDIO_CODEC_DATA0_PP_DE_MT_M_EN;
	}

	// wait on l3 done interrupt ???

	// printk(__FILE__ ": done. \n");

	/*
	  if (state) {
	  *audioL3DataReg = (backpaq_uda1341->uda1341->regs.data0_2 | 
	  BACKPAQ_AUDIO_CODEC_DATA0_MT);
	  //	  set_sa11xx_uda1341_egpio(IPAQ_EGPIO_QMUTE);
	  } else {
	  *audioL3DataReg = (backpaq_uda1341->uda1341->regs.data0_2 & 
	  ~BACKPAQ_AUDIO_CODEC_DATA0_MT);
	  //	  clr_sa11xx_uda1341_egpio(IPAQ_EGPIO_QMUTE);
	  }
	*/
// #endif

}

/***************************************************************************************
 *
 *
 *  END of the hal layer abstraction interface.
 *
 *
 *
 *
 **************************************************************************************/

static int __init backpaq_uda1341_init(void)
{
	int err;
	snd_card_t *card;

	printk(__FILE__ ": begin backpaq_uda1314_init\n");
	DEBUG_NAME(KERN_DEBUG "backpaq_uda1341_uda1341_init\n");
        
	if (!machine_is_ipaq())
		return -ENODEV;

	/* register the soundcard */
	// printk(KERN_ERR "%s:%d: here\n", __FILE__, __LINE__);
	printk("%s: registering sound card ...\n", __FUNCTION__);
	card = snd_card_new(-1, id, THIS_MODULE, 0);
	if (card == NULL)
		return -ENOMEM;
	// printk("%s: done.\n", __FUNCTION__);
	printk("%s: allocating space for sound card driver ...\n", __FUNCTION__);
	backpaq_uda1341 = snd_magic_kcalloc(sa11xx_uda1341_t, 0, GFP_KERNEL);
	if (backpaq_uda1341 == NULL)
		return -ENOMEM;
         
	// printk("%s: done.\n", __FUNCTION__);
	card->private_data = (void *)backpaq_uda1341;
	card->private_free = snd_backpaq_uda1341_free;
        
	backpaq_uda1341->card = card;

	// mixer
	printk("%s: sending sound card info to mixer ...\n", __FUNCTION__);
	if ((err = snd_chip_backpaq_uda1341_mixer_new(backpaq_uda1341->card, 
						      &backpaq_uda1341->uda1341)))
		goto nodev;
	// printk("%s: done.\n", __FUNCTION__);

	// PCM
	printk("%s: sending sound card info to pcm ...\n", __FUNCTION__);
	if ((err = snd_card_backpaq_uda1341_pcm(backpaq_uda1341, 0, 2)) < 0)
		goto nodev;
	// printk("%s: done.\n", __FUNCTION__);
        
/* turn power managment off for now */
#if 0  
#ifdef CONFIG_PM
	sa11xx_uda1341->pm_dev = pm_register(PM_SYS_DEV, 0, sa11xx_uda1341_pm_callback);
	if (sa11xx_uda1341->pm_dev)
		sa11xx_uda1341->pm_dev->data = sa11xx_uda1341;
#endif
#endif
        
	strcpy(card->driver, "BACKPAQ UDA1341");
	strcpy(card->shortname, "H3600 BACKPAQ UDA1341TS");
	sprintf(card->longname, "Compaq iPAQ H3600 with BACKPAQ and Philips UDA1341TS");
        
	if ((err = snd_card_register(card)) == 0) {
		printk("%s: iPAQ audio support initialized\n", __FUNCTION__ );
		return 0;
	}

 nodev:
	printk(KERN_ERR "%s: ERROR %d occured\n", __FILE__, err);
	snd_card_free(card);
	return err;
}

static void __exit backpaq_uda1341_exit(void)
{
        // this vanished updating to alsa-driver 0.9.4, need to look at what changed in uda1341.c
	// snd_chip_uda1341_mixer_del(backpaq_uda1341->card);
	snd_card_free(backpaq_uda1341->card);
	backpaq_uda1341 = NULL;
}

module_init(backpaq_uda1341_init);
module_exit(backpaq_uda1341_exit);

/* }}} */

/*
 * Local variables:
 * indent-tabs-mode: t
 * End:
 */






