
/* 

Copyright (C) 2003 Christian Pellegrin

This code is released under GPL (GNU Public License) with absolutely no
warranty. Please see http://www.gnu.org/ for a complete discussion of the GPL.

based on code by:

Copyright (C) 2000 Lernout & Hauspie Speech Products, N.V.
Copyright (c) 2002 Tomas Kasparek <tomas.kasparek@seznam.cz>
Copyright (c) 2002 Hewlett-Packard Company
Copyright (c) 2000 Nicolas Pitre <nico@cam.org>

 */

/* 

   pxa-uda1380.c

   version 0.04 11/5/2003

   - PM working also when activated during play

   - fixed problems with not boundary aligned DMAs, tested with 8,22.5 and 44 kHz

   version 0.03 6/5/2003

   - PM working

   version 0.02 24/4/2003

   - initiali CVS

   TODO:

   - study why recording is not working (looks DMA is ok, we just keep getting FFFF from the codec I guess)

   - actually use master mute to avoid bumps

 */


#include <linux/module.h>
#include <sound/driver.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/i2c.h>
#include <linux/kmod.h>
#include "../../i2c/uda1380.h"

#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/dma.h>
#include <asm/arch-sa1100/h3600_hal.h>
#include <asm/arch-sa1100/h3600_asic.h>
#include <asm/arch-pxa/h3900_asic.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/initval.h>

#undef DEBUG
#define DEBUG 1
#ifdef DEBUG
#define DPRINTK( format, x... )  printk( format, ## x )
#else
#define DPRINTK( format, x... ) while(0)
#endif

#define chip_t pxa_uda1380_t

#define MAX_DMA_BUF 2

#define PXA_UDA1380_BSIZE 65536

static char *id = NULL;	/* ID for this card */

MODULE_PARM(id, "s");
MODULE_PARM_DESC(id, "ID string for PXA + UDA1380 soundcard.");

typedef enum stream_id_t{
	PLAYBACK=0,
	CAPTURE,
	MAX_STREAMS,
}stream_id_t;

typedef struct audio_stream {
	char *id;		/* identification string */
	int  stream_id;		/* numeric identification */	
	
	int dma_ch;		/* DMA channel used */
	u32 dev_addr;		/* parameter for DMA controller */
	u32 dcmd;		/* parameter for DMA controller */
	volatile u32 *drcmr;		/* the DMA request channel to use */

	pxa_dma_desc pxa_dma_desc[MAX_DMA_BUF] __attribute__ ((aligned (16))); /* uset to feed the DMA of the pxa250 */

	volatile int dma_running;		/* tells if DMA is running*/
	
	int cur_dma_buf;	/* which dma buffer is currently being played/captured */
	int next_period;	/* which period buffer is next to be played/captured */

  	snd_pcm_substream_t *stream;
}audio_stream_t;
/* TODO-CHRI: check that this is alligned on a 16 byte boundary */

typedef struct snd_card_pxa_uda1380 {
	snd_card_t *card;
	snd_pcm_t *pcm;
	struct i2c_client *uda1380;
	long samplerate;
	audio_stream_t *s[MAX_STREAMS];
	snd_info_entry_t *proc_entry;
#ifdef CONFIG_PM
	struct pm_dev *pm_dev;
	int after_suspend;
#endif
} pxa_uda1380_t;

static struct snd_card_pxa_uda1380 *pxa_uda1380 = NULL;

#define AUDIO_RATE_DEFAULT	44100
static long audio_samplerate = AUDIO_RATE_DEFAULT;


static unsigned int period_sizes[] = { 128, 256, 512, 1024, 2048, 4096 };
 
#define PERIOD_SIZES sizeof(period_sizes) / sizeof(period_sizes[0])
 
static snd_pcm_hw_constraint_list_t hw_constraints_period_sizes = {
        .count = PERIOD_SIZES,
        .list = period_sizes,
        .mask = 0
};

static void pxa_uda1380_set_samplerate(pxa_uda1380_t *pxa_uda1380, long val, int force)
{
	struct i2c_client *uda1380 = pxa_uda1380->uda1380;
	struct uda1380_cfg cfg;
	int clk_div = 0;
	int mute;

	DPRINTK("set_samplerate rate: %ld\n", val);

	if (val == pxa_uda1380->samplerate && !force)
		return;

	mute = uda1380_mute(uda1380, 1);

	udelay(125);

	if (val >= 48000)
		val = 48000;
	else if (val >= 44100)
		val = 44100;
	else if (val >= 22050)
		val = 22050;
	else if (val >= 16000)
		val = 16000;
	else
		val = 8000;

	/* Set the external clock generator */
//	h3600_audio_clock(val);

	/* Select the clock divisor */
	switch (val) {
	case 8000:
		cfg.fs = 256;
		cfg.pll = REG0_PLL_6TO12;
//		clk_div = SADIV_BITCLK_512_00_KHZ;
		clk_div = 0x48;
		break;
	case 16000:
		cfg.fs = 256;
		cfg.pll = REG0_PLL_12TO25;
		clk_div = SADIV_BITCLK_1_024_MHZ;
		break;
	case 22050:
		cfg.fs = 256;
		cfg.pll = REG0_PLL_12TO25;
		clk_div = SADIV_BITCLK_1_418_MHZ;
		break;
	case 44100:
		cfg.fs = 256;
		cfg.pll = REG0_PLL_25TO50;
		clk_div = SADIV_BITCLK_2_836_MHZ;
		break;
	case 48000:
		cfg.fs = 256;
		cfg.pll = REG0_PLL_25TO50;
		clk_div = SADIV_BITCLK_3_072_MHZ;
		break;
	}

	cfg.format = FMT_I2S;
	PXA_SADIV = clk_div;
	
	i2c_command(uda1380, I2C_UDA1380_CONFIGURE, &cfg);
	pxa_uda1380->samplerate = val;

	uda1380_mute(uda1380, mute);

	DPRINTK("set sample rate done\n");
}

static void pxa_uda1380_audio_init(pxa_uda1380_t *pxa_uda1380)
{
	unsigned long flags;
	
	DPRINTK("%s\n", __FUNCTION__);
	
	/* Setup the uarts */
	local_irq_save(flags);

	set_GPIO_mode(GPIO28_BITCLK_OUT_I2S_MD);
	set_GPIO_mode(GPIO29_SDATA_IN_I2S_MD);
	set_GPIO_mode(GPIO30_SDATA_OUT_I2S_MD);
	set_GPIO_mode(GPIO31_SYNC_I2S_MD);
	set_GPIO_mode(GPIO32_SYSCLK_I2S_MD);

	/* enable the clock to I2S unit */
	CKEN |= CKEN8_I2S;
	
	/* reset */
	PXA_SACR0 = (1<<3);
	PXA_SACR0 = 0;

	PXA_SACR0 = SACR0_ENB | SACR0_BCKD_OUTPUT | SACR0_RFTH(8) | SACR0_TFTH(8);
	PXA_SACR1 = 0;
	PXA_SADIV = SADIV_BITCLK_2_836_MHZ; /* 44.1 KHz */

 	local_irq_restore(flags);

	/* Blip the UDA1380 reset line */
	local_irq_save(flags);
	H3900_ASIC3_GPIO_B_OUT |= GPIO3_AUD_RESET;
 	local_irq_restore(flags);

	/* Enable the audio power */
	h3600_audio_power(audio_samplerate);

	mdelay(1);

	local_irq_save(flags);
	H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_AUD_RESET;
 	local_irq_restore(flags);

	/* Wait for the UDA1380 to wake up */
	mdelay(1);

	/* Initialize the UDA1380 internal state */
	i2c_command(pxa_uda1380->uda1380, I2C_UDA1380_OPEN, 0);

	pxa_uda1380_set_samplerate(pxa_uda1380, pxa_uda1380->samplerate, 1);

	h3600_audio_mute(0);

	DPRINTK("%s done\n", __FUNCTION__);
}

static void pxa_uda1380_audio_shutdown(pxa_uda1380_t *pxa_uda1380)
{
	DPRINTK("%s\n", __FUNCTION__);
	
	h3600_audio_mute(1);
	i2c_command(pxa_uda1380->uda1380, I2C_UDA1380_CLOSE, 0);

	PXA_SACR0 = 0;
	PXA_SACR1 = 0;

	h3600_audio_power(0);
	
	DPRINTK("%s done\n", __FUNCTION__);
}

static void audio_dma_irq(int ch, void *dev_id, struct pt_regs *regs);

static void audio_dma_request(audio_stream_t *s, void (*callback)(int , void *, struct pt_regs *))
{
	int err;

	DPRINTK("%s\n", __FUNCTION__);

	err = pxa_request_dma(s->id, DMA_PRIO_LOW, 
			      audio_dma_irq, s);
	if (err < 0) {
		printk("panic: cannot allocate DMA for %s\n", s->id);
		/* CHRI-TODO: handle this error condition gracefully */
	}

	s->dma_ch = err;
	if (s->stream_id == CAPTURE) {
	  DPRINTK("%s capture\n", __FUNCTION__);
	  s->drcmr = &DRCMRRXSADR;
	  s->dcmd = DCMD_RXPCDR;
	  s->dev_addr = __PREG(PXA_SADR);
	  *(s->drcmr) = s->dma_ch | DRCMR_MAPVLD;
	  printk ("drcmr capture: %x set to %x\n", (u32) s->drcmr, *(s->drcmr));
	}
	else {
	  DPRINTK("%s playback\n", __FUNCTION__);
	  s->drcmr = &DRCMRTXSADR;
	  s->dcmd = DCMD_TXPCDR;
	  s->dev_addr = __PREG(PXA_SADR);
	  *(s->drcmr) = s->dma_ch | DRCMR_MAPVLD;
	  printk ("drcmr playback: %x set to %x\n", (u32) s->drcmr, *(s->drcmr));

	}
	
	DPRINTK("%s done\n", __FUNCTION__);
}

static void audio_dma_free(audio_stream_t *s)
{
	DPRINTK("%s\n", __FUNCTION__);

	pxa_free_dma(s->dma_ch);

	DPRINTK("%s done\n", __FUNCTION__);
}


static u_int audio_get_dma_pos(audio_stream_t *s)
{
	snd_pcm_substream_t * substream = s->stream;
	snd_pcm_runtime_t *runtime = substream->runtime;
	unsigned int offset_bytes;
	unsigned int offset;
	int ch = s->dma_ch;
	u32 pos;

	DPRINTK("%s\n", __FUNCTION__);

	if (s->stream_id == CAPTURE)
		pos = DTADR(ch);
	else
		pos = DSADR(ch);
	offset_bytes =  pos - runtime->dma_addr; 
	offset = bytes_to_frames(runtime,offset_bytes);

	DPRINTK("%s done, dma position is %d (in bytes is %d, we are in buffer %d)\n",
		__FUNCTION__, offset , offset_bytes, s->cur_dma_buf);

	return offset;
}

static void audio_stop_dma(audio_stream_t *s)
{
	//int mute;
	//pxa_uda1380_t *chip = snd_pcm_substream_chip(s->stream);

	DPRINTK("%s\n", __FUNCTION__);

	//mute = uda1380_mute(chip->uda1380, 1);
	
	DCSR(s->dma_ch) = DCSR_STOPIRQEN;

	while (! (DCSR(s->dma_ch) & DCSR_STOPSTATE)) {
		if (!in_interrupt())
			schedule();
	}
	DPRINTK("%s finished waiting\n", __FUNCTION__);

	//uda1380_mute(chip->uda1380, mute);
}

static void audio_dma_fil_buff(audio_stream_t *s, int next_period, int cur_dma_buf)
{
  snd_pcm_substream_t * substream = s->stream;
  snd_pcm_runtime_t *runtime = substream->runtime;
  int next_dma_buf = (cur_dma_buf + 1) % MAX_DMA_BUF;
  int dma_size = frames_to_bytes(runtime,runtime->period_size) ;

  DPRINTK("%s next_period %d cur_dma_buf %d dma_size %d\n", __FUNCTION__, next_period, cur_dma_buf, dma_size);

  s->pxa_dma_desc[cur_dma_buf].ddadr = virt_to_bus((u32) &s->pxa_dma_desc[next_dma_buf]);
  if (s->stream_id == CAPTURE) {
    s->pxa_dma_desc[cur_dma_buf].dsadr = s->dev_addr;
    s->pxa_dma_desc[cur_dma_buf].dtadr = ((u32) runtime->dma_addr ) + dma_size * next_period;
  }
  else {
    s->pxa_dma_desc[cur_dma_buf].dsadr = ((u32) runtime->dma_addr ) + dma_size * next_period;
    s->pxa_dma_desc[cur_dma_buf].dtadr = s->dev_addr;
  }
  s->pxa_dma_desc[cur_dma_buf].dcmd = s->dcmd | dma_size | (1 << 21);
  DPRINTK("%s descriptor at %lx has ddadr %x dsadr %x dtadr %x dcmd %x\n", __FUNCTION__,
	  (unsigned long) &s->pxa_dma_desc[cur_dma_buf], 
	  s->pxa_dma_desc[cur_dma_buf].ddadr,
	  s->pxa_dma_desc[cur_dma_buf].dsadr,
	  s->pxa_dma_desc[cur_dma_buf].dtadr,
	  s->pxa_dma_desc[cur_dma_buf].dcmd
	  );
}

#define STEP_DMA \
  audio_dma_fil_buff(s, s->next_period, s->cur_dma_buf); \
  s->next_period = (s->next_period + 1) % runtime->periods; \
  s->cur_dma_buf = (s->cur_dma_buf + 1) % MAX_DMA_BUF

static void audio_process_dma(audio_stream_t *s)
{
  snd_pcm_substream_t * substream = s->stream;
  snd_pcm_runtime_t *runtime = substream->runtime;
  int dma_size = frames_to_bytes(runtime,runtime->period_size) ;
#ifdef WATCH_LOST_IRQS
  u32 pos;
#endif

  DPRINTK("%s\n", __FUNCTION__);
  /* first fill the DMA buffer with data for the next period we should play */

  STEP_DMA;

#if WATCH_LOST_IRQS
  /* check that actually the DMA pointer is in the period we expect (we are not
     losing irqs) */
  if (s->stream_id == CAPTURE) {
    pos = DTADR(s->dma_ch);
  }
  else {
    pos = DSADR(s->dma_ch);
  }
  while ( ! (s->pxa_dma_desc[s->cur_dma_buf].dsadr <= pos &&
	     (s->pxa_dma_desc[s->cur_dma_buf].dsadr + dma_size) > DDADR(s->dma_ch))) {
    DPRINTK("%s: losing irqs\n", __FUNCTION__);
    STEP_DMA;
  }
#else
  DPRINTK("pointer position: %x <- %x / %x -> %x\n", 
	  s->pxa_dma_desc[s->cur_dma_buf].dsadr, 
	  DSADR(s->dma_ch), DTADR(s->dma_ch),
	  (s->pxa_dma_desc[s->cur_dma_buf].dsadr + dma_size));
#endif
}

static void audio_start_dma(audio_stream_t *s, int restart)
{
	snd_pcm_substream_t * substream = s->stream;
	snd_pcm_runtime_t *runtime = substream->runtime;
	int i;

	DPRINTK("%s\n", __FUNCTION__);

	/* fille the descriptors */
	s->cur_dma_buf = 0;
	if (restart) {
		s->next_period = (s->next_period + runtime->periods - MAX_DMA_BUF) % runtime->periods;
	}
	else {
		s->next_period = 0;
	}
	for(i=0; i<MAX_DMA_BUF; i++) {
		STEP_DMA;
	}
	s->cur_dma_buf = 0;

	/* kick the DMA */
	DDADR(s->dma_ch) = virt_to_bus((u32) &s->pxa_dma_desc[s->cur_dma_buf]);
	DCSR(s->dma_ch) = DCSR_RUN;	
	s->dma_running = 1;
}

#undef STEP_DMA

static void audio_dma_irq(int chan, void *dev_id, struct pt_regs *regs)
{
	audio_stream_t *s = (audio_stream_t *) dev_id;
	int ch = s->dma_ch;
	u_int dcsr;
	
	DPRINTK("%s\n", __FUNCTION__);
	DPRINTK("DDADR %x, DSADR %x, DTADR %x, DCMD %x\n", DDADR(ch), DSADR(ch), DTADR(ch), DCMD(ch));
	DPRINTK("SASR0 %x\n", PXA_SASR0);

	dcsr = DCSR(ch);
	DCSR(ch) = dcsr & ~DCSR_STOPIRQEN;

	if (dcsr & DCSR_BUSERR) {
		printk("pxa-uda1380 DMA: bus error interrupt on channel %d\n", ch);
		return ;
	}

	if (dcsr & DCSR_ENDINTR) {
		DPRINTK("%s: period done\n", __FUNCTION__);
		
		audio_process_dma(s);
		snd_pcm_period_elapsed(s->stream);
		return;
	}


	if ((dcsr & DCSR_STOPIRQEN) && (dcsr & DCSR_STOPSTATE)) {
		DPRINTK("%s: stop irq\n", __FUNCTION__);
		s->dma_running = 0;
		return ;
	}

	DPRINTK("%s: unknown irq reason!\n", __FUNCTION__);
}

static int snd_card_pxa_uda1380_pcm_trigger(stream_id_t stream_id,
					    snd_pcm_substream_t * substream, int cmd)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime; 	
	
	DPRINTK("pcm_trigger id: %d cmd: %d\n", stream_id, cmd);

	DPRINTK("  sound: %d x %d [Hz]\n", runtime->channels, runtime->rate);
	DPRINTK("  periods: %ld x %ld [fr]\n", (unsigned long)runtime->periods,
	      (unsigned long) runtime->period_size);
	DPRINTK("  buffer_size: %ld [fr]\n", (unsigned long)runtime->buffer_size);
	DPRINTK("  dma_addr %p\n", (char *)runtime->dma_addr);
	DPRINTK("  dma_running PLAYBACK %d CAPTURE %d\n", 
		chip->s[PLAYBACK]->dma_running, chip->s[CAPTURE]->dma_running);


	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
		DPRINTK("trigger START\n");
		if (chip->s[PLAYBACK]->dma_running) {
			audio_stop_dma(chip->s[PLAYBACK]);
		}
		if (chip->s[CAPTURE]->dma_running) {
			audio_stop_dma(chip->s[CAPTURE]);
		}
		audio_start_dma(chip->s[stream_id],0);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
		DPRINTK("trigger STOP\n");
		audio_stop_dma(chip->s[stream_id]);
		break;
	default:
		DPRINTK("trigger UNKNOWN\n");
		return -EINVAL;
		break;
	}
	return 0;

}

static snd_pcm_hardware_t snd_pxa_uda1380_capture =
{
	.info			= (SNDRV_PCM_INFO_INTERLEAVED |
				   SNDRV_PCM_INFO_BLOCK_TRANSFER |
				   SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats		= SNDRV_PCM_FMTBIT_S16_LE,
	.rates			= (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
				   SNDRV_PCM_RATE_22050 | \
				   SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000 |\
				   SNDRV_PCM_RATE_KNOT),
	.rate_min		= 8000,
	.rate_max		= 48000,
	.channels_min		= 2,
	.channels_max		= 2,
	.buffer_bytes_max	= 65536, // was 16380
	.period_bytes_min	= 128,
	.period_bytes_max	= 4096, /* <= MAX_DMA_SIZE from ams/arch-sa1100/dma.h */
	.periods_min		= 2,
	.periods_max		= 32,
	.fifo_size		= 0,
};

static snd_pcm_hardware_t snd_pxa_uda1380_playback =
{
	.info			= (SNDRV_PCM_INFO_INTERLEAVED |
				   SNDRV_PCM_INFO_BLOCK_TRANSFER |
				   SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats		= SNDRV_PCM_FMTBIT_S16_LE,
	.rates			= (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
                                   SNDRV_PCM_RATE_22050 | \
				   SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000 |\
				   SNDRV_PCM_RATE_KNOT),
	.rate_min		= 8000,
	.rate_max		= 48000,
	.channels_min		= 2,
	.channels_max		= 2,
	.buffer_bytes_max	= 65536, 
	.period_bytes_min	= 128,
	.period_bytes_max	= 4096, /* <= MAX_DMA_SIZE from ams/arch-sa1100/dma.h */
	.periods_min		= 2,
	.periods_max		= 32, // dropped from 255 to  lower overhead costs
	.fifo_size		= 0,
};

static unsigned int rates[] = {
	8000,  
	16000, 22050,
	44100, 48000,
};

#define RATES sizeof(rates) / sizeof(rates[0])

static snd_pcm_hw_constraint_list_t hw_constraints_rates = {
	.count	= RATES,
	.list	= rates,
	.mask	= 0,
};

static int snd_card_pxa_uda1380_playback_open(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	int err;
        
	DPRINTK("playback_open\n");

	if (chip->after_suspend) {
	  chip->after_suspend = 0;
	}

	chip->s[PLAYBACK]->stream = substream;

	runtime->hw = snd_pxa_uda1380_playback;


#if 1
	snd_pcm_hw_constraint_list(runtime, 0,
                                   SNDRV_PCM_HW_PARAM_PERIOD_BYTES,
                                   &hw_constraints_period_sizes);
	snd_pcm_hw_constraint_minmax(runtime, SNDRV_PCM_HW_PARAM_BUFFER_BYTES, PXA_UDA1380_BSIZE, PXA_UDA1380_BSIZE);
#else
	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0)
		return err;
#endif

	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					      &hw_constraints_rates)) < 0)
		return err;
        
	return 0;
}

static int snd_card_pxa_uda1380_playback_close(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);

	DPRINTK("%s\n", __FUNCTION__);

	chip->s[PLAYBACK]->dma_running = 0;
	chip->s[PLAYBACK]->stream = NULL;
      
	return 0;
}

static int snd_card_pxa_uda1380_playback_ioctl(snd_pcm_substream_t * substream,
					       unsigned int cmd, void *arg)
{
	DPRINTK("playback_ioctl cmd: %d\n", cmd);
	return snd_pcm_lib_ioctl(substream, cmd, arg);
}

static int snd_card_pxa_uda1380_playback_prepare(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
        
	DPRINTK("playback_prepare\n");

	/* set requested samplerate */
	pxa_uda1380_set_samplerate(chip, runtime->rate, 0);
	chip->samplerate = runtime->rate;
        
	return 0;
}

static int snd_card_pxa_uda1380_playback_trigger(snd_pcm_substream_t * substream, int cmd)
{
	DPRINTK("playback_trigger\n");
	return snd_card_pxa_uda1380_pcm_trigger(PLAYBACK, substream, cmd);
}

static snd_pcm_uframes_t snd_card_pxa_uda1380_playback_pointer(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_uframes_t pos;

	DPRINTK("playback_pointer\n");        
	pos = audio_get_dma_pos(chip->s[PLAYBACK]);
	return pos;
}

static int snd_card_pxa_uda1380_capture_open(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	int err;
	
	DPRINTK("%s: \n", __FUNCTION__);

	if (chip->after_suspend) {
	  chip->after_suspend = 0;
	}

	chip->s[CAPTURE]->stream = substream;
	
	runtime->hw = snd_pxa_uda1380_capture;

#if 1
	snd_pcm_hw_constraint_list(runtime, 0,
                                   SNDRV_PCM_HW_PARAM_PERIOD_BYTES,
                                   &hw_constraints_period_sizes);
	snd_pcm_hw_constraint_minmax(runtime, SNDRV_PCM_HW_PARAM_BUFFER_BYTES, PXA_UDA1380_BSIZE, PXA_UDA1380_BSIZE);
#else
	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0)
		return err;
#endif

	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					      &hw_constraints_rates)) < 0)
		return err;
        
	return 0;
}

static int snd_card_pxa_uda1380_capture_close(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);

	DPRINTK("%s:\n", __FUNCTION__);

	chip->s[CAPTURE]->dma_running = 0;
	chip->s[CAPTURE]->stream = NULL;
      
	return 0;
}

static int snd_card_pxa_uda1380_capture_ioctl(snd_pcm_substream_t * substream,
					unsigned int cmd, void *arg)
{
	DPRINTK("%s:\n", __FUNCTION__);
	return snd_pcm_lib_ioctl(substream, cmd, arg);
}

static int snd_card_pxa_uda1380_capture_prepare(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	DPRINTK("%s\n", __FUNCTION__);

	/* set requested samplerate */
	pxa_uda1380_set_samplerate(chip, runtime->rate, 0);
        chip->samplerate = runtime->rate;

	return 0;
}

static int snd_card_pxa_uda1380_capture_trigger(snd_pcm_substream_t * substream, int cmd)
{
	DPRINTK("%s:\n", __FUNCTION__);
	return snd_card_pxa_uda1380_pcm_trigger(CAPTURE, substream, cmd);
}

static snd_pcm_uframes_t snd_card_pxa_uda1380_capture_pointer(snd_pcm_substream_t * substream)
{
	pxa_uda1380_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_uframes_t pos;

	DPRINTK("%s:\n", __FUNCTION__);
	pos = audio_get_dma_pos(chip->s[CAPTURE]);
	return pos;
}

static int snd_pxa_uda1380_hw_params(snd_pcm_substream_t * substream,
				     snd_pcm_hw_params_t * hw_params)
{
        int err;

	DPRINTK("hw_params, wanna allocate %d\n", params_buffer_bytes(hw_params));

	err = snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params));
	if (err<0) {
		printk("snd_pcm_lib_malloc_pages failed!\n");
	}
	DPRINTK("hw_params, err is %d\n", err);

	return err;
}

static int snd_pxa_uda1380_hw_free(snd_pcm_substream_t * substream)
{

	DPRINTK("hw_free\n");

	return snd_pcm_lib_free_pages(substream);
}

static snd_pcm_ops_t snd_card_pxa_uda1380_playback_ops = {
	.open			= snd_card_pxa_uda1380_playback_open,
	.close			= snd_card_pxa_uda1380_playback_close,
	.ioctl			= snd_card_pxa_uda1380_playback_ioctl,
	.hw_params	        = snd_pxa_uda1380_hw_params,
	.hw_free	        = snd_pxa_uda1380_hw_free,
	.prepare		= snd_card_pxa_uda1380_playback_prepare,
	.trigger		= snd_card_pxa_uda1380_playback_trigger,
	.pointer		= snd_card_pxa_uda1380_playback_pointer,
};

static snd_pcm_ops_t snd_card_pxa_uda1380_capture_ops = {
	.open			= snd_card_pxa_uda1380_capture_open,
	.close			= snd_card_pxa_uda1380_capture_close,
	.ioctl			= snd_card_pxa_uda1380_capture_ioctl,
	.hw_params	        = snd_pxa_uda1380_hw_params,
	.hw_free	        = snd_pxa_uda1380_hw_free,
	.prepare		= snd_card_pxa_uda1380_capture_prepare,
	.trigger		= snd_card_pxa_uda1380_capture_trigger,
	.pointer		= snd_card_pxa_uda1380_capture_pointer,
};


static int __init snd_card_pxa_uda1380_pcm(pxa_uda1380_t *pxa_uda1380, int device, int substreams)
{

	int err;
	
	DPRINTK("%s\n", __FUNCTION__);
	// audio power is turned on and a clock is set at the same time so this gives us a default
	// starting rate
	pxa_uda1380->samplerate = AUDIO_RATE_DEFAULT;
	
	if ((err = snd_pcm_new(pxa_uda1380->card, "UDA1380 PCM", device,
			       substreams, substreams, &(pxa_uda1380->pcm))) < 0)
		return err;

	// this sets up our initial buffers and sets the dma_type to isa.
	// isa works but I'm not sure why (or if) it's the right choice
	// this may be too large, trying it for now

	//err = snd_pcm_lib_preallocate_pages_for_all(pxa_uda1380->pcm, 64*1024, 64*1024, GFP_KERNEL | GFP_DMA);
	err = snd_pcm_lib_preallocate_isa_pages_for_all(pxa_uda1380->pcm, 64*1024, 64*1024);
	if (err < 0) {
	  printk("preallocate failed with code %d\n", err);
	}
	DPRINTK("%s memory prellocation done\n", __FUNCTION__);
	
	snd_pcm_set_ops(pxa_uda1380->pcm, SNDRV_PCM_STREAM_PLAYBACK, &snd_card_pxa_uda1380_playback_ops);
	snd_pcm_set_ops(pxa_uda1380->pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_card_pxa_uda1380_capture_ops);
	pxa_uda1380->pcm->private_data = pxa_uda1380;
	pxa_uda1380->pcm->info_flags = 0;
	strcpy(pxa_uda1380->pcm->name, "UDA1380 PCM");

	pxa_uda1380->s[PLAYBACK] = snd_kcalloc(sizeof(audio_stream_t), GFP_KERNEL);
	pxa_uda1380->s[CAPTURE] = snd_kcalloc(sizeof(audio_stream_t), GFP_KERNEL);

	pxa_uda1380_audio_init(pxa_uda1380);

	/* setup nameing */
	pxa_uda1380->s[PLAYBACK]->id = "UDA1380 PLAYBACK";
	pxa_uda1380->s[CAPTURE]->id = "UDA1380 CAPTURE";

	pxa_uda1380->s[PLAYBACK]->stream_id = PLAYBACK;
	pxa_uda1380->s[CAPTURE]->stream_id = CAPTURE;

	/* setup DMA controller */
	audio_dma_request(pxa_uda1380->s[PLAYBACK], audio_dma_irq);
	audio_dma_request(pxa_uda1380->s[CAPTURE], audio_dma_irq);

	pxa_uda1380->s[PLAYBACK]->dma_running = 0;
	pxa_uda1380->s[CAPTURE]->dma_running = 0;

	return 0;
}


/* }}} */

/* {{{ module init & exit */

#ifdef CONFIG_PM

static int pxa_uda1380_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{
	pxa_uda1380_t *pxa_uda1380 = pm_dev->data;
	audio_stream_t *is, *os;	
	snd_card_t *card;
	
	DPRINTK("pm_callback\n");
	DPRINTK("%s: req = %d\n", __FUNCTION__, req);
	
	is = pxa_uda1380->s[PLAYBACK];
	os = pxa_uda1380->s[CAPTURE];
	card = pxa_uda1380->card;
	
	switch (req) {
	case PM_SUSPEND: /* enter D1-D3 */
		if (os)
			audio_dma_free(os);
		if (is)
			audio_dma_free(is);
		pxa_uda1380_audio_shutdown(pxa_uda1380);
#ifdef CONFIG_ARCH_H3900
		H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_AUD_PWR_ON;
#endif
		break;
	case PM_RESUME:  /* enter D0 */
#ifdef CONFIG_ARCH_H3900
		H3900_ASIC3_GPIO_B_OUT |= GPIO3_AUD_PWR_ON;
#endif
		pxa_uda1380_audio_init(pxa_uda1380);
		if (is)
		  audio_dma_request(is, audio_dma_irq);
		if (os)
		  audio_dma_request(os, audio_dma_irq);
		pxa_uda1380->after_suspend = 1;
		if (is->dma_running) {
			audio_start_dma(is, 1);
		}
		if (os->dma_running) {
			audio_start_dma(os, 1);
		}
		break;
	}
	DPRINTK("%s: exiting...\n", __FUNCTION__);	
        return 0;
	
}

#endif

void snd_pxa_uda1380_free(snd_card_t *card)
{
	pxa_uda1380_t *chip = snd_magic_cast(pxa_uda1380_t, card->private_data, return);

	DPRINTK("snd_pxa_uda1380_free\n");

#ifdef CONFIG_PM
	pm_unregister(chip->pm_dev);
#endif

	chip->s[PLAYBACK]->drcmr = 0;
	chip->s[CAPTURE]->drcmr = 0;

	audio_dma_free(chip->s[PLAYBACK]);
	audio_dma_free(chip->s[CAPTURE]);

	kfree(chip->s[PLAYBACK]);
	kfree(chip->s[CAPTURE]);

	chip->s[PLAYBACK] = NULL;
	chip->s[CAPTURE] = NULL;

	snd_magic_kfree(chip);
	card->private_data = NULL;
}

static int __init pxa_uda1380_init(void)
{
	int err;
	snd_card_t *card;

	DPRINTK("pxa_uda1380_init\n");
        
	if (!machine_is_h3900())
		return -ENODEV;

	/* register the soundcard */
	card = snd_card_new(-1, id, THIS_MODULE, 0);
	if (card == NULL)
		return -ENOMEM;
	pxa_uda1380 = snd_magic_kcalloc(pxa_uda1380_t, 0, GFP_KERNEL);
	if (pxa_uda1380 == NULL)
		return -ENOMEM;
         
	card->private_data = (void *)pxa_uda1380;
	card->private_free = snd_pxa_uda1380_free;
        
	pxa_uda1380->card = card;

	/* try to bring in i2c support */
	request_module("i2c-adap-pxa");

	pxa_uda1380->uda1380 = i2c_get_client(I2C_DRIVERID_UDA1380, I2C_ALGO_PXA, NULL);
	if (!pxa_uda1380->uda1380) {
		printk("cannot find mixer!\n");
	}
	else {
		// mixer
		if ((err = snd_chip_uda1380_mixer_new(pxa_uda1380->card, &pxa_uda1380->uda1380))) {
			printk("snd_chip_uda1380_mixer_new failed\n");
			goto nodev;
		}
		// PCM
		if ((err = snd_card_pxa_uda1380_pcm(pxa_uda1380, 0, 2)) < 0) {
			printk("snd_chip_uda1380_pcm failed\n");
			goto nodev;
		}
	}

#ifdef CONFIG_PM
	pxa_uda1380->pm_dev = pm_register(PM_SYS_DEV, 0, pxa_uda1380_pm_callback);
	if (pxa_uda1380->pm_dev)
		pxa_uda1380->pm_dev->data = pxa_uda1380;
	pxa_uda1380->after_suspend = 0;
#endif
        
	strcpy(card->driver, "UDA1380");
	strcpy(card->shortname, "H3900 UDA1380");
	sprintf(card->longname, "Compaq iPAQ H3900 with Philips UDA1380");
        
	if ((err = snd_card_register(card)) == 0) {
		printk("iPAQ audio support initialized\n" );
		return 0;
	}

 nodev:
	snd_card_free(card);
	return err;
}

static void __exit pxa_uda1380_exit(void)
{
	snd_chip_uda1380_mixer_del(pxa_uda1380->card);
	snd_card_free(pxa_uda1380->card);
	pxa_uda1380 = NULL;
}

module_init(pxa_uda1380_init);
module_exit(pxa_uda1380_exit);
