
/*  
    
Copyright (C) 2003 Christian Pellegrin

This code is released under GPL (GNU Public License) with absolutely no
warranty. Please see http://www.gnu.org/ for a complete discussion of the GPL.

based on code by:

Copyright (C) 2000 Lernout & Hauspie Speech Products, N.V.
Copyright (c) 2002 Tomas Kasparek <tomas.kasparek@seznam.cz>
Copyright (c) 2002 Hewlett-Packard Company
Copyright (c) 2000 Nicolas Pitre <nico@cam.org>

*/

/* 

   uda1380.c

   version 0.03 11/5/2003

   - added full controls for playback

   - added the master mute control

   version 0.02 24/4/2003

   TODO:
   
   - study why pll is used instead of sysclock
   
   - add recording controls

   - write a mini I2C so we don't need the legacy one (cannot be used in IRQ
     routines since it sleeps)

 */

/* 

Supported Alsa controls:

- Master Playback Volume

- Tone Control - Strenght

- Tone Control - Bass

- Tone Control - Treble

- De-emphasis

 */

#undef DEBUG
#define DEBUG 1
#ifdef DEBUG
#define DPRINTK( format, x... )  printk( format, ## x )
#else
#define DPRINTK( format, x... ) while(0)
#endif

#include <sound/driver.h>
#include <linux/module.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/slab.h>
#include <linux/errno.h>
#include <linux/ioctl.h>
#include <linux/delay.h>
#include <linux/soundcard.h>
#include <linux/i2c.h>
#include "uda1380.h"

#include <sound/core.h>
#include <sound/control.h>
#include <sound/initval.h>
#include <sound/info.h>

#include <asm/uaccess.h>

#ifdef CONFIG_ARCH_H3900
#include <asm/arch-sa1100/h3600_asic.h>
#include <asm/arch-pxa/h3900_asic.h>
#endif

/* code from uda1380.c. basically just an ALSA interface around this */

#define DEF_VOLUME	65



#define REC_MASK	(SOUND_MASK_LINE | SOUND_MASK_MIC)
#define DEV_MASK	(REC_MASK | SOUND_MASK_VOLUME | SOUND_MASK_BASS | SOUND_MASK_TREBLE)

static struct uda1380_reg_info {
	unsigned short num;
	unsigned short default_value;
} uda1380_reg_info[] = {
    	{ REG0, (REG0_SC_256FS|REG0_DAC_USE_WSPLL|REG0_ADC_USE_WSPLL|REG0_PLL_25TO50
		 //{ REG0, (REG0_SC_256FS|REG0_DAC_USE_SYSCLK|REG0_ADC_USE_SYSCLK|REG0_PLL_25TO50
		 |REG0_EN_ADC|REG0_EN_DEC|REG0_EN_DAC|REG0_EN_INT) },
	{ I2S_REG, (FMT_I2S << I2S_REG_SFORO_SHIFT) | (FMT_I2S << I2S_REG_SFORI_SHIFT) },
	{ PWR_REG, ( PWR_REG_PON_PLL|PWR_REG_PON_HP|PWR_REG_PON_DAC|PWR_REG_PON_BIAS|PWR_REG_EN_ADC|PWR_REG_PON_ADC
		     //{ PWR_REG, ( PWR_REG_PON_HP|PWR_REG_PON_DAC|PWR_REG_PON_BIAS|PWR_REG_EN_ADC|PWR_REG_PON_ADC
		     |PWR_REG_PON_LNA|PWR_REG_PON_ADCL|PWR_REG_PON_ADCR|PWR_REG_PON_PGAL|PWR_REG_PON_PGAR ) },
	{ AMIX_REG, 0 },
	{ MASTER_VOL_REG, 0 },
	{ MIXER_VOL_REG, 0 },
	{ MBT_REG, 0 },
	{ MMCDM_REG, 0x0000 },
	{ MIXER_CTL_REG, 0 },
	{ DEC_VOL_REG, 0 },
	{ DPM_REG, 0x0000 },
	{ DEC_ADC_REG, 0x050c },
	{ DEC_AGC_REG, 0 } 
}; 

struct uda1380 {
	unsigned short regs[REG_MAX];
	int		active;
	unsigned short	volume;
	unsigned short	bass;
	unsigned short	treble;
	unsigned short	line;
	unsigned short	mic;
	int             mic_connected;
	int             line_connected;
	int		mod_cnt;
};

typedef struct uda1380 uda1380_t;

//hack for ALSA magic casting
typedef struct i2c_client i2c_client_t;
#define chip_t i2c_client_t      

static struct i2c_client *uda1380=NULL;

static void uda1380_write_reg(struct i2c_client *clnt, struct uda1380 *uda, int regaddr)
{
	char buffer[3];
	int r;

	DPRINTK("%s: writing %x in %x\n", __FUNCTION__, uda->regs[regaddr], regaddr);

	buffer[0] = regaddr;
	buffer[1] = uda->regs[regaddr] >> 8;
	buffer[2] = uda->regs[regaddr] & 0xff;

	r = i2c_master_send(clnt, buffer, 3);
	if (r != 3) {
		printk(KERN_ERR "uda1380: write failed, status %d\n", r);
	}
}

static void uda1380_sync(struct i2c_client *clnt)
{
	struct uda1380 *uda = clnt->data;
	int i;

	DPRINTK("%s:\n", __FUNCTION__);

	for (i = 0; i < ARRAY_SIZE(uda1380_reg_info); i++)
		uda1380_write_reg(clnt, uda, uda1380_reg_info[i].num);

}

static void uda1380_cmd_init(struct i2c_client *clnt)
{
	struct uda1380 *uda = clnt->data;

	DPRINTK("%s:\n", __FUNCTION__);

	uda->active = 1;

	/* resend all parameters */
	uda1380_sync(clnt);
}

static int uda1380_configure(struct i2c_client *clnt, struct uda1380_cfg *conf)
{
	struct uda1380 *uda = clnt->data;
	int ret = 0;

	DPRINTK("%s:\n", __FUNCTION__);

	mdelay(10);

	uda->regs[REG0] &= ~(REG0_SC_MASK);

	switch (conf->fs) {
	case 768: uda->regs[REG0] |= REG0_SC_768FS;	break;
	case 512: uda->regs[REG0] |= REG0_SC_512FS;	break;
	case 384: uda->regs[REG0] |= REG0_SC_384FS;	break;
	case 256: uda->regs[REG0] |= REG0_SC_256FS;	break;
	default:  ret = -EINVAL;			break;
	}

	uda->regs[REG0] &= ~(REG0_PLL_MASK);
	uda->regs[REG0] |= conf->pll;

	if (ret == 0 && uda->active) {
		uda1380_write_reg(clnt, uda, REG0);
	}

	switch (conf->format) {
	case FMT_I2S:
	case FMT_LSB16:
	case FMT_LSB18:
	case FMT_LSB20:
	case FMT_MSB:
		uda->regs[I2S_REG] &= ~(I2S_REG_SFORO_MASK|I2S_REG_SFORI_MASK); 
		uda->regs[I2S_REG] |= (conf->format << I2S_REG_SFORO_SHIFT);
		uda->regs[I2S_REG] |= (conf->format << I2S_REG_SFORI_SHIFT);
		break;
	default:  
		ret = -EINVAL;
		break;
	}
	if (ret == 0 && uda->active) {
		uda1380_write_reg(clnt, uda, I2S_REG);
	}

#if 1
	/* power off and on PLL (windows style) */
	/* this is needed after we change sample rates, guess something wrong with PLL start */
	/* perhaps we should use sysclock */
	uda->regs[PWR_REG] &= ~ PWR_REG_PON_PLL;
	uda1380_write_reg(clnt, uda, PWR_REG);
	mdelay(10);
	uda->regs[PWR_REG] |= PWR_REG_PON_PLL;
	uda1380_write_reg(clnt, uda, PWR_REG);
	mdelay(10);
#endif

	return ret;
}

static int uda1380_update(struct i2c_client *clnt, int cmd, void *arg)
{
	struct uda1380 *uda = clnt->data;
	struct i2c_gain *v = arg;
	char newregnum = 0;
	int val;

	DPRINTK("%s:\n", __FUNCTION__);

	switch (cmd) {
	case I2C_SET_VOLUME: /* set volume.  val =  0 to 100 => 255 to 0 */
		uda->regs[MASTER_VOL_REG] = 
			(MASTER_VCL(255 - ((v->left * 255) / 100))
			 |MASTER_VCR(255 - ((v->right * 255) / 100)));
		newregnum = MASTER_VOL_REG;
		break;

	case I2C_SET_BASS:   /* set bass.    val = 0 to 100 => 0 to 15 */
		val = (v->left * 15) / 100;
		uda->regs[MBT_REG] &= ~(MBT_BBL_MASK|MBT_BBR_MASK);
		uda->regs[MBT_REG] |= (MBT_BBL(val)|MBT_BBR(val));
		newregnum = MBT_REG;
		break;

	case I2C_SET_TREBLE: /* set treble.  val = 0 to 100 => 0 to 3 */
		val = (v->left * 3) / 100;
		uda->regs[MBT_REG] &= ~(MBT_TRL_MASK|MBT_TRR_MASK);
		uda->regs[MBT_REG] |= (MBT_TRL(val)|MBT_TRR(val));
		newregnum = MBT_REG;
		break;

	case I2C_SET_STR: /* set tone control.  val = 0 to 100 => 0 to 3 */
		val = (v->left * 3) / 100;
		uda->regs[MBT_REG] &= ~(MBT_MODE_MASK);
		uda->regs[MBT_REG] |= ( (val & 3) << MBT_MODE_SHIFT );
		newregnum = MBT_REG;
		break;

	case I2C_SET_DEEMP: /* set deemph.  val = 0 to 100 => 4 to 1 with 0->0 */
		if (v->left)
			val = 4 - ((val * 3) / 100);
		else
			val = 0;
		uda->regs[MMCDM_REG] &= ~(MMCDM_CHANNEL1_DEEMPHASIS_MASK | MMCDM_CHANNEL2_DEEMPHASIS_MASK);
		uda->regs[MMCDM_REG] |= ( (val & 7) << MMCDM_CHANNEL2_DEEMPHASIS_SHIFT ) || 
			( (val & 7) << MMCDM_CHANNEL1_DEEMPHASIS_SHIFT );
		newregnum = MMCDM_REG;
		break;

	case I2C_SET_LINE_GAIN:
		val = v->left * 15 / 100;
		uda->regs[DPM_REG] &= ~(DPM_GAINL_MASK|DPM_GAINR_MASK);
		uda->regs[DPM_REG] |= (DPM_GAINL(val)|DPM_GAINR(val));
		newregnum = DPM_REG;		
		break;		

	case I2C_SET_MIC_GAIN:
		val = v->left * 15 / 100;
		uda->regs[DEC_ADC_REG] &= ~(DEC_ADC_VGA_CTRL_MASK);
		uda->regs[DEC_ADC_REG] |= (DEC_ADC_VGA_CTRL(val));
		if (v->left) {
			uda->regs[DEC_ADC_REG] |= DEC_ADC_SEL_LNA | DEC_ADC_SEL_MIC;
		}
		else {
			uda->regs[DEC_ADC_REG] &= ~ (DEC_ADC_SEL_LNA | DEC_ADC_SEL_MIC);
		}
		newregnum = DEC_ADC_REG;		
		break;

	case I2C_SET_CAP_VOLUME:
		val = (v->left * 176 / 100) - 128;
		uda->regs[DEC_VOL_REG] &= ~(DEC_VCL_MASK|DEC_VCR_MASK);
		uda->regs[DEC_VOL_REG] |= (DEC_VCL(val)|DEC_VCR(val));
		newregnum = DEC_VOL_REG;		
		break;

	case I2C_SET_MIC_SW:
		if (v->left) {
			uda->regs[DEC_ADC_REG] |= DEC_ADC_SEL_LNA | DEC_ADC_SEL_MIC;
		}
		else {
			uda->regs[DEC_ADC_REG] &= ~ (DEC_ADC_SEL_LNA | DEC_ADC_SEL_MIC);
		}
		newregnum = DEC_ADC_REG;
		break;


	case I2C_SET_AGC:
		break;

	default:
		return -EINVAL;
	}		

	if (uda->active)
		uda1380_write_reg(clnt, uda, newregnum);
	return 0;
}

/* end code from uda1380.c */

int uda1380_mute(struct i2c_client *clnt, int mute) 
{
	int ret;
	struct uda1380 *uda = clnt->data;

	DPRINTK("%s: %d\n", __FUNCTION__, mute);
	
	ret = (uda->regs[MMCDM_REG] & MMCDM_MASTER_MUTE) > 0;

	if (mute) {
		uda->regs[MMCDM_REG] |= MMCDM_MASTER_MUTE;
	}
	else {
		uda->regs[MMCDM_REG] &= ~MMCDM_MASTER_MUTE;
	}
	uda1380_write_reg(clnt, uda, MMCDM_REG);

	return ret;
}

/* ALSA specific code */

/* TODO: for now doesn't support L/R balance, fix it! */

#define CMD_VOLUME 0
#define CMD_BASS 1
#define CMD_TREBBLE 2
#define CMD_MIC 3
#define CMD_LINE 4
#define CMD_CAP_VOLUME 5
#define CMD_MIC_SW 6
#define CMD_STR 7
#define CMD_DEEMP 8
#define CMD_LAST 9

static struct i2c_gain uda1380_gain[CMD_LAST];
static int uda1380_cmds[CMD_LAST] =
{
  I2C_SET_VOLUME,
  I2C_SET_BASS,
  I2C_SET_TREBLE,
  I2C_SET_MIC_GAIN,
  I2C_SET_LINE_GAIN,
  I2C_SET_CAP_VOLUME,
  I2C_SET_MIC_SW,
  I2C_SET_STR,
  I2C_SET_DEEMP
};

static int 
snd_uda1380_info_single(snd_kcontrol_t *kcontrol, snd_ctl_elem_info_t * uinfo)
{
	DPRINTK("%s called with private info: %ld\n", __FUNCTION__, kcontrol->private_value);
	switch(kcontrol->private_value) {
	case CMD_VOLUME:
	case CMD_BASS:
	case CMD_TREBBLE:
	case CMD_MIC:
	case CMD_LINE:
	case CMD_CAP_VOLUME:
	case CMD_STR:
	case CMD_DEEMP:
		uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
		uinfo->count = 1;
		uinfo->value.integer.min = 0;
		uinfo->value.integer.max = 100;
		return 0;
		break;
	case CMD_MIC_SW:
		uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
		uinfo->count = 1;
		uinfo->value.integer.min = 0;
		uinfo->value.integer.max = 1;
		return 0;
		break;
	}
	printk("%s called with bad private info: %ld\n", __FUNCTION__, kcontrol->private_value);
	return -EINVAL;
}

static int 
snd_uda1380_put_single(snd_kcontrol_t *kcontrol, snd_ctl_elem_value_t *ucontrol)
{
	struct i2c_client *clnt = uda1380;
	
	DPRINTK("%s called with private info: %ld (clnt is %p)\n", __FUNCTION__, kcontrol->private_value, clnt);
	switch(kcontrol->private_value) {
	case CMD_VOLUME:
	case CMD_BASS:
	case CMD_TREBBLE:
	case CMD_MIC:
	case CMD_LINE:
	case CMD_CAP_VOLUME:
	case CMD_MIC_SW:
	case CMD_STR:
	case CMD_DEEMP:
		uda1380_gain[kcontrol->private_value].left = uda1380_gain[kcontrol->private_value].right =
			ucontrol->value.integer.value[0];
		DPRINTK("setting %ld to %ld\n", kcontrol->private_value, ucontrol->value.integer.value[0]);
		uda1380_update(clnt, uda1380_cmds[kcontrol->private_value], &uda1380_gain[kcontrol->private_value]);
		return 0;
		break;
	}
	printk("%s called with bad private info: %ld\n", __FUNCTION__, kcontrol->private_value);
	return -EINVAL;
}

static int 
snd_uda1380_get_single(snd_kcontrol_t *kcontrol, snd_ctl_elem_value_t *ucontrol)
{
	DPRINTK("%s called with private info: %ld\n", __FUNCTION__, kcontrol->private_value);
	switch(kcontrol->private_value) {
	case CMD_VOLUME:
	case CMD_BASS:
	case CMD_TREBBLE:
	case CMD_MIC:
	case CMD_LINE:
	case CMD_CAP_VOLUME:
	case CMD_MIC_SW:
	case CMD_STR:
	case CMD_DEEMP:
		ucontrol->value.integer.value[0] = uda1380_gain[kcontrol->private_value].left;
		return 0;
		break;
	}
	printk("%s called with bad private info: %ld\n", __FUNCTION__, kcontrol->private_value);
	return -EINVAL;
}

#define UDA1380_SINGLE(xname, pvt_value) \
{ .iface = SNDRV_CTL_ELEM_IFACE_MIXER, \
  .name = xname, \
  .info = snd_uda1380_info_single, \
  .get = snd_uda1380_get_single, \
  .put = snd_uda1380_put_single, \
  .private_value = pvt_value \
}

static snd_kcontrol_new_t snd_uda1380_controls[] = {
	UDA1380_SINGLE("Master Playback Volume", CMD_VOLUME),
	UDA1380_SINGLE("Tone Control - Bass", CMD_BASS),
	UDA1380_SINGLE("Tone Control - Treble", CMD_TREBBLE),
	UDA1380_SINGLE("Tone Control - Strenght", CMD_STR),
	UDA1380_SINGLE("De-emphasis", CMD_DEEMP),
//	UDA1380_SINGLE("Mic Sensitivity Volume", CMD_MIC),
//	UDA1380_SINGLE("Line Sensitivity Volume", CMD_LINE),
//	UDA1380_SINGLE("Master Capture Volume", CMD_CAP_VOLUME),
//	UDA1380_SINGLE("Mic Capture Switch", CMD_MIC_SW),
};

#define UDA1380_CONTROLS (sizeof(snd_uda1380_controls)/sizeof(snd_kcontrol_new_t))

/* initialization stuff */

static struct i2c_driver uda1380_driver;

static int __init uda1380_init(void)
{
	DPRINTK("%s:\n", __FUNCTION__);

#ifdef CONFIG_ARCH_H3900
        H3900_ASIC3_GPIO_B_OUT |= GPIO3_AUD_PWR_ON;
#endif
	return i2c_add_driver(&uda1380_driver);
}

static void __exit uda1380_exit(void)
{
	DPRINTK("%s:\n", __FUNCTION__);

	i2c_del_driver(&uda1380_driver);
#ifdef CONFIG_ARCH_H3900
	H3900_ASIC3_GPIO_B_OUT &= ~GPIO3_AUD_PWR_ON;
#endif
}

static struct i2c_client client_template = {
	name: "(unset)",
	flags:  I2C_CLIENT_ALLOW_USE,
	driver: &uda1380_driver
};

static int uda1380_attach(struct i2c_adapter *adap, int addr, unsigned short flags, int kind)		
{
	struct uda1380 *uda;
	struct i2c_client *clnt;
	int i;

	DPRINTK("%s:\n", __FUNCTION__);

	clnt = kmalloc(sizeof(*clnt), GFP_KERNEL);
	memcpy(clnt, &client_template, sizeof(*clnt));
	clnt->adapter = adap;
	clnt->addr = addr;
	strcpy(clnt->name, "uda1380");

	uda = kmalloc(sizeof(*uda), GFP_KERNEL);
	if (!uda)
		return -ENODEV;

	memset(uda, 0, sizeof(*uda));

	/* set default values of registers */
	for (i = 0; i < ARRAY_SIZE(uda1380_reg_info); i++) {
		uda->regs[uda1380_reg_info[i].num] = uda1380_reg_info[i].default_value;
	}

	clnt->data = uda;

	i2c_attach_client(clnt);

	return 0;
}

static int uda1380_detach_client(struct i2c_client *clnt)
{
	DPRINTK("%s:\n", __FUNCTION__);

	i2c_detach_client(clnt);

	kfree(clnt->data);
	kfree(clnt);

	return 0;
}

/* Addresses to scan */
static unsigned short normal_i2c[] = {0x18,I2C_CLIENT_END};
static unsigned short normal_i2c_range[] = {I2C_CLIENT_END};
static unsigned short probe[]        = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short probe_range[]  = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore[]       = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short ignore_range[] = { I2C_CLIENT_END, I2C_CLIENT_END };
static unsigned short force[]        = { I2C_ALGO_PXA, 0x18, I2C_CLIENT_END, I2C_CLIENT_END };

static struct i2c_client_address_data addr_data = {
	normal_i2c, normal_i2c_range, 
	probe, probe_range, 
	ignore, ignore_range, 
	force
};

static int uda1380_attach_adapter(struct i2c_adapter *adap)
{
	DPRINTK("%s:\n", __FUNCTION__);

	return i2c_probe(adap, &addr_data, uda1380_attach);
}

static int uda1380_open(struct i2c_client *clnt)
{
	DPRINTK("%s:\n", __FUNCTION__);

	uda1380_cmd_init(clnt);
	return 0;
}

static void uda1380_close(struct i2c_client *clnt)
{
	struct uda1380 *uda = clnt->data;

	DPRINTK("%s:\n", __FUNCTION__);

	uda->active = 0;
}

static int uda1380_mixer_ioctl(struct i2c_client *clnt, int cmd, void *arg)
{
#if 0
	struct uda1380 *uda = clnt->data;
	struct i2c_gain gain;
	int val, nr = _IOC_NR(cmd), ret = 0;
#endif

	DPRINTK("%s:\n", __FUNCTION__);

#if 0

	if (cmd == SOUND_MIXER_INFO) {
		struct mixer_info mi;

		strncpy(mi.id, "UDA1380", sizeof(mi.id));
		strncpy(mi.name, "Philips UDA1380", sizeof(mi.name));
		mi.modify_counter = uda->mod_cnt;
		return copy_to_user(arg, &mi, sizeof(mi));
	}

	if (_IOC_DIR(cmd) & _IOC_WRITE) {
		ret = get_user(val, (int *)arg);
		if (ret)
			goto out;

		gain.left    = val & 255;
		gain.right   = val >> 8;

		switch (nr) {
		case SOUND_MIXER_VOLUME:
			uda->volume = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_VOLUME, &gain);
			break;

		case SOUND_MIXER_BASS:
			uda->bass = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_BASS, &gain);
			break;

		case SOUND_MIXER_TREBLE:
			uda->treble = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_TREBLE, &gain);
			break;

		case SOUND_MIXER_LINE:
			if (!uda->line_connected)
				return -EINVAL;
			uda->line = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_LINE_GAIN, &gain);
			break;

		case SOUND_MIXER_MIC:
			if (!uda->mic_connected)
				return -EINVAL;
			uda->mic = val;
			uda->mod_cnt++;
			uda1380_update(clnt, I2C_SET_MIC_GAIN, &gain);
			break;

		case SOUND_MIXER_RECSRC:
			break;

		case SOUND_MIXER_AGC:
			uda1380_update(clnt, I2C_SET_AGC, (void *) val);
			break;

		default:
			ret = -EINVAL;
		}
	}

	if (ret == 0 && _IOC_DIR(cmd) & _IOC_READ) {
		int nr = _IOC_NR(cmd);
		ret = 0;

		switch (nr) {
		case SOUND_MIXER_VOLUME:     val = uda->volume;	break;
		case SOUND_MIXER_BASS:       val = uda->bass;	break;
		case SOUND_MIXER_TREBLE:     val = uda->treble;	break;
		case SOUND_MIXER_LINE:       val = uda->line;	break;
		case SOUND_MIXER_MIC:        val = uda->mic;	break;
		case SOUND_MIXER_RECSRC:     val = REC_MASK;	break;
		case SOUND_MIXER_RECMASK:    val = REC_MASK;	break;
		case SOUND_MIXER_DEVMASK:    val = DEV_MASK;	break;
		case SOUND_MIXER_CAPS:       val = 0;		break;
		case SOUND_MIXER_STEREODEVS: val = 0;		break;
		default:	val = 0;     ret = -EINVAL;	break;
		}

		if (ret == 0)
			ret = put_user(val, (int *)arg);
	}
out:
	return ret;
#endif
	return -ENOTTY;
}

int uda1380_command(struct i2c_client *clnt, unsigned int cmd, void *arg)
{
	int ret = -EINVAL;

	DPRINTK("%s:\n", __FUNCTION__);

	if (_IOC_TYPE(cmd) == 'M')
		ret = uda1380_mixer_ioctl(clnt, cmd, arg);
	else if (cmd == I2C_UDA1380_CONFIGURE)
		ret = uda1380_configure(clnt, arg);
	else if (cmd == I2C_UDA1380_OPEN)
		ret = uda1380_open(clnt);
	else if (cmd == I2C_UDA1380_CLOSE)
		(ret = 0), uda1380_close(clnt);

	return ret;
}

static void uda1380_inc_use(struct i2c_client *clnt)
{
	DPRINTK("%s:\n", __FUNCTION__);

	MOD_INC_USE_COUNT;
}

static void uda1380_dec_use(struct i2c_client *clnt)
{
	DPRINTK("%s:\n", __FUNCTION__);

	MOD_DEC_USE_COUNT;
}

static struct i2c_driver uda1380_driver = {
	name:		UDA1380_NAME,
	id:		I2C_DRIVERID_UDA1380,
	flags:		I2C_DF_NOTIFY,
	attach_adapter:	uda1380_attach_adapter,
	detach_client:	uda1380_detach_client,
	command:        uda1380_command,
	inc_use:	uda1380_inc_use,
	dec_use:	uda1380_dec_use
};

int snd_chip_uda1380_mixer_new(snd_card_t *card, struct i2c_client **clnt)
{
	int idx, err;

	DPRINTK("%s:\n", __FUNCTION__);
        
	snd_assert(card != NULL, return -EINVAL);

	*clnt = i2c_get_client(I2C_DRIVERID_UDA1380, I2C_ALGO_PXA, NULL);
	if (!clnt)
	  printk("uda1380 panic: connot get i2c client\n");
	uda1380 = *clnt;

	for (idx = 0; idx < UDA1380_CONTROLS; idx++) {
		if ((err = snd_ctl_add(card, snd_ctl_new1(&snd_uda1380_controls[idx], uda1380))) < 0)
			return err;
	}

	DPRINTK("added %d controls\n", UDA1380_CONTROLS);

	/* set default values */
	{
	  snd_kcontrol_t * tmp_k = kmalloc(sizeof(snd_kcontrol_t), GFP_KERNEL);
	  snd_ctl_elem_value_t * tmp_v = kmalloc(sizeof(snd_ctl_elem_value_t), GFP_KERNEL);

	  tmp_k->private_value = CMD_VOLUME;
	  tmp_v->value.integer.value[0] = 0;
	  snd_uda1380_put_single(tmp_k, tmp_v);

	  kfree(tmp_k);
	  kfree(tmp_v);
	}

	strcpy(card->mixername, "UDA1380 Mixer");

	MOD_INC_USE_COUNT;
        
	return 0;
}

void snd_chip_uda1380_mixer_del(snd_card_t *card)
{
	DPRINTK("uda1380 mixer_del\n");
        
	MOD_DEC_USE_COUNT;
}

module_init(uda1380_init);
module_exit(uda1380_exit);

MODULE_AUTHOR("Christian Pellegrin");
MODULE_DESCRIPTION("Philips UDA1380 CODEC driver");

