/* 
 *
 * Copyright (C) 2003 Hewlett-Packard Company
 *
 * This code is released under GPL (GNU Public License) with
 * absolutely no warranty. Please see http://www.gnu.org/ for a
 * complete discussion of the GPL.
 *
 * based on code by:
 *
 * Copyright (C) 2003 Christian Pellegrin
 * Copyright (C) 2000 Lernout & Hauspie Speech Products, N.V.
 * Copyright (c) 2002 Tomas Kasparek <tomas.kasparek@seznam.cz>
 * Copyright (c) 2002 Hewlett-Packard Company
 * Copyright (c) 2000 Nicolas Pitre <nico@cam.org>
 */

#include <linux/module.h>
#include <sound/driver.h>
#include <linux/init.h>
#include <linux/types.h>
#include <linux/fs.h>
#include <linux/delay.h>
#include <linux/pm.h>
#include <linux/errno.h>
#include <linux/slab.h>
#include <linux/sound.h>
#include <linux/soundcard.h>
#include <linux/i2c.h>
#include <linux/kmod.h>

#include <asm/semaphore.h>
#include <asm/uaccess.h>
#include <asm/hardware.h>
#include <asm/dma.h>
#include <asm/arch-sa1100/h3600_hal.h>
#include <asm/arch-sa1100/h3600_asic.h>
#include <asm/arch-pxa/h5400-asic.h>
#include <asm/arch-pxa/pxa-i2s.h>

#include <sound/core.h>
#include <sound/pcm.h>
#include <sound/driver.h>
#include <sound/control.h>
#include <sound/initval.h>
#include <sound/info.h>

#include "../i2c/ak4535.h"

#undef DEBUG
#define DEBUG 1
#ifdef DEBUG
#define DPRINTK( format, x... )  printk( format, ## x )
#else
#define DPRINTK( format, x... ) while(0)
#endif

#define chip_t snd_card_h5400_ak4535_t

#define MAX_DMA_BUF 2

#define H5400_AK4535_BSIZE 65536

static char *id = NULL;	/* ID for this card */
static int max_periods_allowed = 255; /* @@@@ check on this */

MODULE_PARM(id, "s");
MODULE_PARM_DESC(id, "ID string for PXA + AK4535 soundcard.");


struct h5400_audio_stream {
	char *id;		/* identification string */
	int  stream_id;		/* numeric identification */	
	
	int dma_ch;		/* DMA channel used */
	u32 dev_addr;		/* parameter for DMA controller */
	u32 dcmd;		/* parameter for DMA controller */
	volatile u32 *drcmr;		/* the DMA request channel to use */

	pxa_dma_desc pxa_dma_desc[MAX_DMA_BUF] __attribute__ ((aligned (16))); /* used to feed the DMA of the pxa250 */

	volatile int dma_running;		/* tells if DMA is running*/
	
	int cur_dma_buf;	/* which dma buffer is currently being played/captured */
	int next_period;	/* which period buffer is next to be played/captured */

	int active:1;		/* we are using this stream for transfer now */

	int sent_periods;       /* # of sent periods from actual DMA buffer */
	int sent_total;         /* # of sent periods total (just for info & debug) */

  	snd_pcm_substream_t *stream;
};
/* TODO-CHRI: check that this is alligned on a 16 byte boundary */

static struct snd_card_h5400_ak4535 *h5400_ak4535 = NULL;

#define AUDIO_RATE_DEFAULT	44100
static long audio_samplerate = AUDIO_RATE_DEFAULT;


static unsigned int period_sizes[] = { 128, 256, 512, 1024, 2048, 4096 };
 
#define PERIOD_SIZES sizeof(period_sizes) / sizeof(period_sizes[0])
 
static snd_pcm_hw_constraint_list_t hw_constraints_period_sizes = {
        .count = PERIOD_SIZES,
        .list = period_sizes,
        .mask = 0
};

static void h5400_ak4535_set_samplerate(snd_card_h5400_ak4535_t *h5400_ak4535, long val, int force)
{
	struct i2c_client *ak4535 = h5400_ak4535->ak4535;
	struct ak4535_cfg cfg;
	int clk_div = 0;
	int mute;

	DPRINTK("set_samplerate rate: %ld\n", val);

	if (val == h5400_ak4535->samplerate && !force)
		return;

	cfg.fs = pxa_i2s_set_samplerate (val);
	cfg.format = FMT_I2S;
	
	i2c_command(ak4535, I2C_AK4535_CONFIGURE, &cfg);
	h5400_ak4535->samplerate = val;

	DPRINTK("set sample rate done\n");
}

static void h5400_ak4535_audio_init(snd_card_h5400_ak4535_t *h5400_ak4535)
{
	unsigned long flags;
	
	DPRINTK("%s\n", __FUNCTION__);
	
	h3600_audio_power(0);

	pxa_i2s_init ();
	mdelay(1);

	/* Enable the audio power */
	if (h3600_audio_power(audio_samplerate))
		printk("Unable to enable audio power\n");

	/* Wait for the AK4535 to wake up */
	mdelay(1);

	/* Initialize the AK4535 internal state */
	i2c_command(h5400_ak4535->ak4535, I2C_AK4535_OPEN, 0);

	h3600_audio_mute(0);

	h5400_ak4535_set_samplerate(h5400_ak4535, h5400_ak4535->samplerate, 1);

	DPRINTK("%s done\n", __FUNCTION__);
}

static void h5400_ak4535_audio_shutdown(snd_card_h5400_ak4535_t *h5400_ak4535)
{
	DPRINTK("%s\n", __FUNCTION__);
	

	/* disable the audio power and all signals leading to the audio chip */
	h3600_audio_mute(1);
	i2c_command(h5400_ak4535->ak4535, I2C_AK4535_CLOSE, 0);

	pxa_i2s_shutdown ();

	h3600_audio_power(0);

	i2c_dec_use_client(h5400_ak4535->ak4535);

	DPRINTK("%s done\n", __FUNCTION__);
}

static void audio_dma_irq(int ch, void *dev_id, struct pt_regs *regs);

static void audio_dma_request(h5400_audio_stream_t *s, void (*callback)(int , void *, struct pt_regs *))
{
	int err;

	DPRINTK("%s\n", __FUNCTION__);

	err = pxa_request_dma(s->id, DMA_PRIO_LOW, 
			      audio_dma_irq, s);
	if (err < 0) {
		printk("panic: cannot allocate DMA for %s\n", s->id);
		/* CHRI-TODO: handle this error condition gracefully */
	}

	s->dma_ch = err;
	if (s->stream_id == CAPTURE) {
		DPRINTK("%s capture\n", __FUNCTION__);
		s->drcmr = &DRCMRRXSADR;
		s->dcmd = DCMD_RXPCDR;
		s->dev_addr = __PREG(PXA_SADR);
		*(s->drcmr) = s->dma_ch | DRCMR_MAPVLD;
		printk ("drcmr capture: %x set to %x\n", (u32) s->drcmr, *(s->drcmr));
	}
	else {
		DPRINTK("%s playback\n", __FUNCTION__);
		s->drcmr = &DRCMRTXSADR;
		s->dcmd = DCMD_TXPCDR;
		s->dev_addr = __PREG(PXA_SADR);
		*(s->drcmr) = s->dma_ch | DRCMR_MAPVLD;
		printk ("drcmr playback: %x set to %x\n", (u32) s->drcmr, *(s->drcmr));

	}
	
	DPRINTK("%s done\n", __FUNCTION__);
}

static void audio_dma_free(h5400_audio_stream_t *s)
{
	DPRINTK("%s\n", __FUNCTION__);

	pxa_free_dma(s->dma_ch);

	DPRINTK("%s done\n", __FUNCTION__);
}


static u_int audio_get_dma_pos(h5400_audio_stream_t *s)
{
	snd_pcm_substream_t * substream = s->stream;
	snd_pcm_runtime_t *runtime = substream->runtime;
	unsigned int offset_bytes;
	unsigned int offset;
	int ch = s->dma_ch;
	u32 pos;

	DPRINTK("%s\n", __FUNCTION__);

	if (s->stream_id == CAPTURE)
		pos = DTADR(ch);
	else
		pos = DSADR(ch);
	offset_bytes =  pos - runtime->dma_addr; 
	offset = bytes_to_frames(runtime,offset_bytes);

	DPRINTK("%s done, dma position is %d (in bytes is %d, we are in buffer %d)\n",
		__FUNCTION__, offset , offset_bytes, s->cur_dma_buf);

	return offset;
}

static void audio_stop_dma(h5400_audio_stream_t *s)
{
	//int mute;
	//snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(s->stream);

	DPRINTK("%s\n", __FUNCTION__);

	//mute = ak4535_mute(chip->ak4535, 1);
	
	DCSR(s->dma_ch) = DCSR_STOPIRQEN;

	while (! (DCSR(s->dma_ch) & DCSR_STOPSTATE)) {
		if (!in_interrupt())
			schedule();
	}
	DPRINTK("%s finished waiting\n", __FUNCTION__);

	//ak4535_mute(chip->ak4535, mute);
}

static void audio_dma_fill_buff(h5400_audio_stream_t *s, int next_period, int cur_dma_buf)
{
	snd_pcm_substream_t * substream = s->stream;
	snd_pcm_runtime_t *runtime = substream->runtime;
	int next_dma_buf = (cur_dma_buf + 1) % MAX_DMA_BUF;
	int dma_size = frames_to_bytes(runtime,runtime->period_size) ;

	DPRINTK("%s next_period %d cur_dma_buf %d dma_size %d\n", __FUNCTION__, next_period, cur_dma_buf, dma_size);

	s->pxa_dma_desc[cur_dma_buf].ddadr = virt_to_bus((u32) &s->pxa_dma_desc[next_dma_buf]);
	if (s->stream_id == CAPTURE) {
		s->pxa_dma_desc[cur_dma_buf].dsadr = s->dev_addr;
		s->pxa_dma_desc[cur_dma_buf].dtadr = ((u32) runtime->dma_addr ) + dma_size * next_period;
	}
	else {
		s->pxa_dma_desc[cur_dma_buf].dsadr = ((u32) runtime->dma_addr ) + dma_size * next_period;
		s->pxa_dma_desc[cur_dma_buf].dtadr = s->dev_addr;
	}
	s->pxa_dma_desc[cur_dma_buf].dcmd = s->dcmd | dma_size | (1 << 21);
	DPRINTK("%s descriptor at %lx has ddadr %x dsadr %x dtadr %x dcmd %x\n", __FUNCTION__,
		(unsigned long) &s->pxa_dma_desc[cur_dma_buf], 
		s->pxa_dma_desc[cur_dma_buf].ddadr,
		s->pxa_dma_desc[cur_dma_buf].dsadr,
		s->pxa_dma_desc[cur_dma_buf].dtadr,
		s->pxa_dma_desc[cur_dma_buf].dcmd
		);
}

#define STEP_DMA \
  audio_dma_fill_buff(s, s->next_period, s->cur_dma_buf); \
  s->next_period = (s->next_period + 1) % runtime->periods; \
  s->cur_dma_buf = (s->cur_dma_buf + 1) % MAX_DMA_BUF

static void audio_process_dma(h5400_audio_stream_t *s)
{
	snd_pcm_substream_t * substream = s->stream;
	snd_pcm_runtime_t *runtime = substream->runtime;
	int dma_size = frames_to_bytes(runtime,runtime->period_size) ;
#ifdef WATCH_LOST_IRQS
	u32 pos;
#endif

	DPRINTK("%s\n", __FUNCTION__);
	/* first fill the DMA buffer with data for the next period we should play */

	STEP_DMA;

#if WATCH_LOST_IRQS
	/* check that actually the DMA pointer is in the period we expect (we are not
	   losing irqs) */
	if (s->stream_id == CAPTURE) {
		pos = DTADR(s->dma_ch);
	}
	else {
		pos = DSADR(s->dma_ch);
	}
	while ( ! (s->pxa_dma_desc[s->cur_dma_buf].dsadr <= pos &&
		   (s->pxa_dma_desc[s->cur_dma_buf].dsadr + dma_size) > DDADR(s->dma_ch))) {
		DPRINTK("%s: losing irqs\n", __FUNCTION__);
		STEP_DMA;
	}
#else
	DPRINTK("pointer position: %x <- %x / %x -> %x\n", 
		s->pxa_dma_desc[s->cur_dma_buf].dsadr, 
		DSADR(s->dma_ch), DTADR(s->dma_ch),
		(s->pxa_dma_desc[s->cur_dma_buf].dsadr + dma_size));
#endif
}

static void audio_start_dma(h5400_audio_stream_t *s, int restart)
{
	snd_pcm_substream_t * substream = s->stream;
	snd_pcm_runtime_t *runtime = substream->runtime;
	int i;

	DPRINTK("%s\n", __FUNCTION__);

	/* fille the descriptors */
	s->cur_dma_buf = 0;
	if (restart) {
		s->next_period = (s->next_period + runtime->periods - MAX_DMA_BUF) % runtime->periods;
	}
	else {
		s->next_period = 0;
	}
	for(i=0; i<MAX_DMA_BUF; i++) {
		STEP_DMA;
	}
	s->cur_dma_buf = 0;

	/* kick the DMA */
	DDADR(s->dma_ch) = virt_to_bus((u32) &s->pxa_dma_desc[s->cur_dma_buf]);
	DCSR(s->dma_ch) = DCSR_RUN;	
	s->dma_running = 1;
}

#undef STEP_DMA

static void audio_dma_irq(int chan, void *dev_id, struct pt_regs *regs)
{
	h5400_audio_stream_t *s = (h5400_audio_stream_t *) dev_id;
	int ch = s->dma_ch;
	u_int dcsr;
	
	DPRINTK("%s\n", __FUNCTION__);
	DPRINTK("DDADR %x, DSADR %x, DTADR %x, DCMD %x\n", DDADR(ch), DSADR(ch), DTADR(ch), DCMD(ch));
	DPRINTK("SASR0 %x\n", PXA_SASR0);

	dcsr = DCSR(ch);
	DCSR(ch) = dcsr & ~DCSR_STOPIRQEN;

	if (dcsr & DCSR_BUSERR) {
		printk("pxa-ak4535 DMA: bus error interrupt on channel %d\n", ch);
		return ;
	}

	if (dcsr & DCSR_ENDINTR) {
		DPRINTK("%s: period done\n", __FUNCTION__);
		
		/* @@@@ check this out */
		audio_process_dma(s);
		snd_pcm_period_elapsed(s->stream);
		return;
	}


	if ((dcsr & DCSR_STOPIRQEN) && (dcsr & DCSR_STOPSTATE)) {
		DPRINTK("%s: stop irq\n", __FUNCTION__);
		s->dma_running = 0;
		return ;
	}

	DPRINTK("%s: unknown irq reason!\n", __FUNCTION__);
}

static int snd_card_h5400_ak4535_pcm_trigger(stream_id_t stream_id,
					   snd_pcm_substream_t * substream, int cmd)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime; 	
	
	DPRINTK("pcm_trigger id: %d cmd: %d\n", stream_id, cmd);

	DPRINTK("  sound: %d x %d [Hz]\n", runtime->channels, runtime->rate);
	DPRINTK("  periods: %ld x %ld [fr]\n", (unsigned long)runtime->periods,
		(unsigned long) runtime->period_size);
	DPRINTK("  buffer_size: %ld [fr]\n", (unsigned long)runtime->buffer_size);
	DPRINTK("  dma_addr %p\n", (char *)runtime->dma_addr);
	DPRINTK("  dma_running PLAYBACK %d CAPTURE %d\n", 
		chip->s[PLAYBACK]->dma_running, chip->s[CAPTURE]->dma_running);


	switch (cmd) {
	case SNDRV_PCM_TRIGGER_START:
	case SNDRV_PCM_TRIGGER_RESUME:
	case SNDRV_PCM_TRIGGER_PAUSE_PUSH:
		DPRINTK("trigger START\n");
		if (chip->s[PLAYBACK]->dma_running) {
			audio_stop_dma(chip->s[PLAYBACK]);
		}
		if (chip->s[CAPTURE]->dma_running) {
			audio_stop_dma(chip->s[CAPTURE]);
		}
		chip->s[stream_id]->active = 0;
		audio_start_dma(chip->s[stream_id],0);
		break;
	case SNDRV_PCM_TRIGGER_SUSPEND:
		/* requested stream suspend */
		audio_stop_dma(chip->s[stream_id]);
		break;
	case SNDRV_PCM_TRIGGER_STOP:
	case SNDRV_PCM_TRIGGER_PAUSE_RELEASE:
		DPRINTK("trigger STOP\n");
		chip->s[stream_id]->active = 0;
		audio_stop_dma(chip->s[stream_id]);
		break;
	default:
		DPRINTK("trigger UNKNOWN\n");
		return -EINVAL;
		break;
	}
	return 0;

}

static snd_pcm_hardware_t snd_h5400_ak4535_capture =
{
	.info			= (SNDRV_PCM_INFO_INTERLEAVED |
				   SNDRV_PCM_INFO_BLOCK_TRANSFER |
				   SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID),
	.formats		= SNDRV_PCM_FMTBIT_S16_LE,
	.rates			= (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |\
				   SNDRV_PCM_RATE_22050 | \
				   SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000 |\
				   SNDRV_PCM_RATE_KNOT),
	.rate_min		= 8000,
	.rate_max		= 48000,
	.channels_min		= 2,
	.channels_max		= 2,
	.buffer_bytes_max	= 65536, // was 16380
	.period_bytes_min	= 64, // was 128 but this is what sa11xx-uda1341 uses
	.period_bytes_max	= 4096, /* <= MAX_DMA_SIZE from ams/arch-sa1100/dma.h */
	.periods_min		= 2,
	.periods_max		= 32,
	.fifo_size		= 0,
};

static snd_pcm_hardware_t snd_h5400_ak4535_playback =
{
	.info			= (SNDRV_PCM_INFO_INTERLEAVED |
				   SNDRV_PCM_INFO_BLOCK_TRANSFER |
				   SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_MMAP_VALID |
				   SNDRV_PCM_INFO_PAUSE | SNDRV_PCM_INFO_RESUME),
	.formats		= SNDRV_PCM_FMTBIT_S16_LE,
	.rates			= (SNDRV_PCM_RATE_8000 | SNDRV_PCM_RATE_16000 |
                                   SNDRV_PCM_RATE_22050 | 
				   SNDRV_PCM_RATE_44100 | SNDRV_PCM_RATE_48000 |
				   SNDRV_PCM_RATE_KNOT),
	.rate_min		= 8000,
	.rate_max		= 48000,
	.channels_min		= 2,
	.channels_max		= 2,
	.buffer_bytes_max	= 65536, 
	.period_bytes_min	= 64, /* was 128 @@@@ changed to same as sa11xx-uda1341 jamey 2003-nov-20 */
	.period_bytes_max	= 4096, /* <= MAX_DMA_SIZE from ams/arch-sa1100/dma.h */
	.periods_min		= 2,
	.periods_max		= 32, // dropped from 255 to  lower overhead costs
	.fifo_size		= 0,
};

static unsigned int rates[] = {
	8000,  
	16000, 22050,
	44100, 48000,
};

#define RATES sizeof(rates) / sizeof(rates[0])

static snd_pcm_hw_constraint_list_t hw_constraints_rates = {
	.count	= RATES,
	.list	= rates,
	.mask	= 0,
};

static int snd_card_h5400_ak4535_playback_open(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	int err;
        
	DPRINTK("playback_open\n");

	if (chip->after_suspend) {
		chip->after_suspend = 0;
	}

	chip->s[PLAYBACK]->stream = substream;
	chip->s[PLAYBACK]->sent_periods = 0;
	chip->s[PLAYBACK]->sent_total = 0;

	runtime->hw = snd_h5400_ak4535_playback;
	runtime->hw.periods_max = max_periods_allowed;

/* @@@@ check this out also -jamey 2003-nov-20 */
#if 1
	snd_pcm_hw_constraint_list(runtime, 0,
                                   SNDRV_PCM_HW_PARAM_PERIOD_BYTES,
                                   &hw_constraints_period_sizes);
	snd_pcm_hw_constraint_minmax(runtime, SNDRV_PCM_HW_PARAM_BUFFER_BYTES, H5400_AK4535_BSIZE, H5400_AK4535_BSIZE);
#else
	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0)
		return err;
#endif

	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					      &hw_constraints_rates)) < 0)
		return err;
        
	return 0;
}

static int snd_card_h5400_ak4535_playback_close(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);

	DPRINTK("%s\n", __FUNCTION__);

	chip->s[PLAYBACK]->dma_running = 0;
	chip->s[PLAYBACK]->stream = NULL;
      
	return 0;
}

static int snd_card_h5400_ak4535_playback_ioctl(snd_pcm_substream_t * substream,
					      unsigned int cmd, void *arg)
{
	DPRINTK("playback_ioctl cmd: %d\n", cmd);
	return snd_pcm_lib_ioctl(substream, cmd, arg);
}

static int snd_card_h5400_ak4535_playback_prepare(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
        
	DPRINTK("playback_prepare\n");

	/* set requested samplerate */
	h5400_ak4535_set_samplerate(chip, runtime->rate, 0);
        
	return 0;
}

static int snd_card_h5400_ak4535_playback_trigger(snd_pcm_substream_t * substream, int cmd)
{
	DPRINTK("playback_trigger\n");
	return snd_card_h5400_ak4535_pcm_trigger(PLAYBACK, substream, cmd);
}

static snd_pcm_uframes_t snd_card_h5400_ak4535_playback_pointer(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_uframes_t pos;

	DPRINTK("playback_pointer\n");        
	pos = audio_get_dma_pos(chip->s[PLAYBACK]);
	return pos;
}

static int snd_card_h5400_ak4535_capture_open(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;
	int err;
	
	DPRINTK("%s: \n", __FUNCTION__);

	if (chip->after_suspend) {
		chip->after_suspend = 0;
	}

	chip->s[CAPTURE]->stream = substream;
	chip->s[CAPTURE]->sent_periods = 0;
	chip->s[CAPTURE]->sent_total = 0;
	
	runtime->hw = snd_h5400_ak4535_capture;
	runtime->hw.periods_max = max_periods_allowed;

/* @@@@ check this out also -jamey 2003-nov-20 */
#if 1 
	snd_pcm_hw_constraint_list(runtime, 0,
                                   SNDRV_PCM_HW_PARAM_PERIOD_BYTES,
                                   &hw_constraints_period_sizes);
	snd_pcm_hw_constraint_minmax(runtime, SNDRV_PCM_HW_PARAM_BUFFER_BYTES, H5400_AK4535_BSIZE, H5400_AK4535_BSIZE);
#else
	if ((err = snd_pcm_hw_constraint_integer(runtime, SNDRV_PCM_HW_PARAM_PERIODS)) < 0)
		return err;
#endif

	if ((err = snd_pcm_hw_constraint_list(runtime, 0, SNDRV_PCM_HW_PARAM_RATE,
					      &hw_constraints_rates)) < 0)
		return err;
        
	return 0;
}

static int snd_card_h5400_ak4535_capture_close(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);

	DPRINTK("%s:\n", __FUNCTION__);

	/* @@@ check this -- sa11xx-uda1341 does not do this */
	chip->s[CAPTURE]->dma_running = 0;
	chip->s[CAPTURE]->stream = NULL;
      
	return 0;
}

static int snd_card_h5400_ak4535_capture_ioctl(snd_pcm_substream_t * substream,
					     unsigned int cmd, void *arg)
{
	DPRINTK("%s:\n", __FUNCTION__);
	return snd_pcm_lib_ioctl(substream, cmd, arg);
}

static int snd_card_h5400_ak4535_capture_prepare(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_runtime_t *runtime = substream->runtime;

	DPRINTK("%s\n", __FUNCTION__);

	/* set requested samplerate */
	h5400_ak4535_set_samplerate(chip, runtime->rate, 0);
        chip->samplerate = runtime->rate;

	return 0;
}

static int snd_card_h5400_ak4535_capture_trigger(snd_pcm_substream_t * substream, int cmd)
{
	DPRINTK("%s:\n", __FUNCTION__);
	return snd_card_h5400_ak4535_pcm_trigger(CAPTURE, substream, cmd);
}

static snd_pcm_uframes_t snd_card_h5400_ak4535_capture_pointer(snd_pcm_substream_t * substream)
{
	snd_card_h5400_ak4535_t *chip = snd_pcm_substream_chip(substream);
	snd_pcm_uframes_t pos;

	DPRINTK("%s:\n", __FUNCTION__);
	pos = audio_get_dma_pos(chip->s[CAPTURE]);
	return pos;
}

static int snd_h5400_ak4535_hw_params(snd_pcm_substream_t * substream,
				    snd_pcm_hw_params_t * hw_params)
{
        int err;

	DPRINTK("hw_params, wanna allocate %d\n", params_buffer_bytes(hw_params));

	err = snd_pcm_lib_malloc_pages(substream, params_buffer_bytes(hw_params));
	if (err<0) {
		printk("snd_pcm_lib_malloc_pages failed!\n");
	}
	DPRINTK("hw_params, err is %d\n", err);

	return err;
}

static int snd_h5400_ak4535_hw_free(snd_pcm_substream_t * substream)
{

	DPRINTK("hw_free\n");

	return snd_pcm_lib_free_pages(substream);
}

static snd_pcm_ops_t snd_card_h5400_ak4535_playback_ops = {
	.open			= snd_card_h5400_ak4535_playback_open,
	.close			= snd_card_h5400_ak4535_playback_close,
	.ioctl			= snd_card_h5400_ak4535_playback_ioctl,
	.hw_params	        = snd_h5400_ak4535_hw_params,
	.hw_free	        = snd_h5400_ak4535_hw_free,
	.prepare		= snd_card_h5400_ak4535_playback_prepare,
	.trigger		= snd_card_h5400_ak4535_playback_trigger,
	.pointer		= snd_card_h5400_ak4535_playback_pointer,
};

static snd_pcm_ops_t snd_card_h5400_ak4535_capture_ops = {
	.open			= snd_card_h5400_ak4535_capture_open,
	.close			= snd_card_h5400_ak4535_capture_close,
	.ioctl			= snd_card_h5400_ak4535_capture_ioctl,
	.hw_params	        = snd_h5400_ak4535_hw_params,
	.hw_free	        = snd_h5400_ak4535_hw_free,
	.prepare		= snd_card_h5400_ak4535_capture_prepare,
	.trigger		= snd_card_h5400_ak4535_capture_trigger,
	.pointer		= snd_card_h5400_ak4535_capture_pointer,
};


static int __init snd_card_h5400_ak4535_pcm(snd_card_h5400_ak4535_t *h5400_ak4535, int device, int substreams)
{

	int err;
	
	DPRINTK("%s\n", __FUNCTION__);
	// audio power is turned on and a clock is set at the same time so this gives us a default
	// starting rate
	h5400_ak4535->samplerate = AUDIO_RATE_DEFAULT;
	
	if ((err = snd_pcm_new(h5400_ak4535->card, "AK4535 PCM", device,
			       substreams, substreams, &(h5400_ak4535->pcm))) < 0)
		return err;

	// this sets up our initial buffers and sets the dma_type to isa.
	// isa works but I'm not sure why (or if) it's the right choice
	// this may be too large, trying it for now

	err = snd_pcm_lib_preallocate_isa_pages_for_all(h5400_ak4535->pcm, 64*1024, 64*1024);
	if (err < 0) {
		printk("preallocate failed with code %d\n", err);
	}
	DPRINTK("%s memory prellocation done\n", __FUNCTION__);
	
	snd_pcm_set_ops(h5400_ak4535->pcm, SNDRV_PCM_STREAM_PLAYBACK, &snd_card_h5400_ak4535_playback_ops);
	snd_pcm_set_ops(h5400_ak4535->pcm, SNDRV_PCM_STREAM_CAPTURE, &snd_card_h5400_ak4535_capture_ops);
	h5400_ak4535->pcm->private_data = h5400_ak4535;
	h5400_ak4535->pcm->info_flags = 0;
	strcpy(h5400_ak4535->pcm->name, "AK4535 PCM");

	h5400_ak4535->s[PLAYBACK] = snd_kcalloc(sizeof(h5400_audio_stream_t), GFP_KERNEL);
	h5400_ak4535->s[CAPTURE] = snd_kcalloc(sizeof(h5400_audio_stream_t), GFP_KERNEL);

	h5400_ak4535_audio_init(h5400_ak4535);

	/* setup naming */
	h5400_ak4535->s[PLAYBACK]->id = "AK4535 PLAYBACK";
	h5400_ak4535->s[CAPTURE]->id = "AK4535 CAPTURE";

	h5400_ak4535->s[PLAYBACK]->stream_id = PLAYBACK;
	h5400_ak4535->s[CAPTURE]->stream_id = CAPTURE;

	/* setup DMA controller */
	audio_dma_request(h5400_ak4535->s[PLAYBACK], audio_dma_irq);
	audio_dma_request(h5400_ak4535->s[CAPTURE], audio_dma_irq);

	h5400_ak4535->s[PLAYBACK]->dma_running = 0;
	h5400_ak4535->s[CAPTURE]->dma_running = 0;

	return 0;
}


/* }}} */

/* {{{ module init & exit */

#ifdef CONFIG_PM

static int h5400_ak4535_pm_callback(struct pm_dev *pm_dev, pm_request_t req, void *data)
{
	snd_card_h5400_ak4535_t *h5400_ak4535 = pm_dev->data;
	h5400_audio_stream_t *is, *os;	
	snd_card_t *card;
	
	DPRINTK("pm_callback\n");
	DPRINTK("%s: req = %d\n", __FUNCTION__, req);
	
	is = h5400_ak4535->s[PLAYBACK];
	os = h5400_ak4535->s[CAPTURE];
	card = h5400_ak4535->card;
	
	switch (req) {
	case PM_SUSPEND: /* enter D1-D3 */
		if (os)
			audio_dma_free(os);
		if (is)
			audio_dma_free(is);
		h5400_ak4535_audio_shutdown(h5400_ak4535);
		break;
	case PM_RESUME:  /* enter D0 */
		h5400_ak4535_audio_init(h5400_ak4535);
		if (is)
			audio_dma_request(is, audio_dma_irq);
		if (os)
			audio_dma_request(os, audio_dma_irq);
		h5400_ak4535->after_suspend = 1;
		if (is->dma_running) {
			audio_start_dma(is, 1);
		}
		if (os->dma_running) {
			audio_start_dma(os, 1);
		}
		break;
	}
	DPRINTK("%s: exiting...\n", __FUNCTION__);	
        return 0;
	
}

#endif

void snd_h5400_ak4535_free(snd_card_t *card)
{
	snd_card_h5400_ak4535_t *chip = snd_magic_cast(snd_card_h5400_ak4535_t, card->private_data, return);

	DPRINTK("snd_h5400_ak4535_free\n");

#ifdef CONFIG_PM
	pm_unregister(chip->pm_dev);
#endif

	chip->s[PLAYBACK]->drcmr = 0;
	chip->s[CAPTURE]->drcmr = 0;

	audio_dma_free(chip->s[PLAYBACK]);
	audio_dma_free(chip->s[CAPTURE]);

	kfree(chip->s[PLAYBACK]);
	kfree(chip->s[CAPTURE]);

	chip->s[PLAYBACK] = NULL;
	chip->s[CAPTURE] = NULL;

	snd_magic_kfree(chip);
	card->private_data = NULL;
}

static int __init h5400_ak4535_init(void)
{
	int err;
	snd_card_t *card;

	DPRINTK("h5400_ak4535_init\n");
        
	if (!machine_is_h5400())
		return -ENODEV;

	/* register the soundcard */
	card = snd_card_new(-1, id, THIS_MODULE, 0);
	if (card == NULL)
		return -ENOMEM;
	h5400_ak4535 = snd_magic_kcalloc(snd_card_h5400_ak4535_t, 0, GFP_KERNEL);
	if (h5400_ak4535 == NULL)
		return -ENOMEM;
         
	card->private_data = (void *)h5400_ak4535;
	card->private_free = snd_h5400_ak4535_free;
        
	h5400_ak4535->card = card;

#ifdef CONFIG_PM
	h5400_ak4535->pm_dev = pm_register(PM_SYS_DEV, 0, h5400_ak4535_pm_callback);
	if (h5400_ak4535->pm_dev)
		h5400_ak4535->pm_dev->data = h5400_ak4535;
	h5400_ak4535->after_suspend = 0;
#endif
        
	strcpy(card->driver, "AK4535");
	strcpy(card->shortname, "H5xxx AK4535");
	sprintf(card->longname, "Compaq iPAQ H5xxx with AKM AK4535");
        
	/* try to bring in i2c support */
	request_module("i2c-adap-pxa");

	h5400_ak4535->ak4535 = i2c_get_client(I2C_DRIVERID_AK4535, I2C_ALGO_PXA, NULL);
	printk("h5400_ak4535_init: card=%p chip=%p clnt=%p\n",
	       h5400_ak4535->card, 
	       h5400_ak4535, 
	       h5400_ak4535->ak4535);
	if (!h5400_ak4535->ak4535) {
		printk("cannot find ak4535!\n");
	}
	else {
		// mixer
		if ((err = snd_card_ak4535_mixer_new(h5400_ak4535->card))) {
			printk("snd_card_ak4535_mixer_new failed\n");
			goto nodev;
		}
		// PCM
		if ((err = snd_card_h5400_ak4535_pcm(h5400_ak4535, 0, 2)) < 0) {
			printk("snd_card_ak4535_pcm failed\n");
			goto nodev;
		}
	}

	if ((err = snd_card_register(card)) == 0) {
		printk("iPAQ audio support initialized\n" );
		return 0;
	}

 nodev:
	snd_card_free(card);
	return err;
}

static void __exit h5400_ak4535_exit(void)
{
	if (h5400_ak4535) {
		snd_card_ak4535_mixer_del(h5400_ak4535->card);
		snd_card_free(h5400_ak4535->card);
		h5400_ak4535 = NULL;
	}
}

module_init(h5400_ak4535_init);
module_exit(h5400_ak4535_exit);
