/*
 *  Bluez SCO soundcard
 *  Copyright (c) 2003 by Jonathan Paisley <jp@dcs.gla.ac.uk>
 *
 *  Based on dummy.c which is
 *  Copyright (c) by Jaroslav Kysela <perex@suse.cz>
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 */

#define chip_t snd_card_bluez_sco_t

#include <sound/driver.h>
#include <linux/version.h>
#include <linux/init.h>
#include <linux/jiffies.h>
#include <linux/slab.h>
#include <linux/time.h>
#include <linux/wait.h>
#include <linux/socket.h>
#include <linux/file.h>
#include <linux/completion.h>
#include <linux/smp_lock.h>
#include <net/sock.h>
#include <net/bluetooth/bluetooth.h>

#include <sound/bluezsco.h>
#include <sound/core.h>
#include <sound/control.h>
#include <sound/pcm.h>
#include <sound/rawmidi.h>
#include <sound/hwdep.h>
#define SNDRV_GET_ID
#include <sound/initval.h>

MODULE_AUTHOR("Jonathan Paisley <jp@dcs.gla.ac.uk>");
MODULE_DESCRIPTION("Bluez SCO Headset Soundcard");
MODULE_LICENSE("GPL");
MODULE_CLASSES("{sound}");
MODULE_DEVICES("{{ALSA,Bluez SCO Soundcard}}");

static char *mod_revision = "$Revision: 1.2 $";

#undef dprintk
#if 1
#define dprintk(fmt...) printk(KERN_INFO "snd-bluez-sco: " fmt)
#else
#define dprintk(fmt...) do {} while(0)
#endif

#define MAX_BUFFER_SIZE		(64*1024)

#define MIXER_ADDR_MASTER	0
#define MIXER_ADDR_MIC		1
#define MIXER_ADDR_LAST		1

#define MIXER_MASK_MASTER	1
#define MIXER_MASK_MIC		2

#define MIXER_MIN_VOLUME	1
#define MIXER_MAX_VOLUME	15

struct snd_card_bluez_sco_pcm;

typedef struct snd_card_bluez_sco {
	snd_card_t 		*card;
	spinlock_t 		mixer_lock;
	int 			mixer_volume[MIXER_ADDR_LAST + 1];
	snd_kcontrol_t		*mixer_controls[MIXER_ADDR_LAST + 2]; /* also loopback */
	volatile int		loopback;
	
	spinlock_t		mixer_changed_lock;
	volatile int		mixer_changed;
	wait_queue_head_t 	hwdep_wait;
	
	int 			thread_pid;
	struct completion 	thread_done;

	volatile int 		thread_exit;
	struct semaphore 	thread_sem;

	volatile struct socket 		*sco_sock;
	struct semaphore 	sock_sem;
	wait_queue_head_t 	wait;

	struct semaphore 	playback_sem;
	struct snd_card_bluez_sco_pcm *playback;
	struct semaphore 	capture_sem;
	struct snd_card_bluez_sco_pcm *capture;
} snd_card_bluez_sco_t;

typedef struct snd_card_bluez_sco_pcm {
	snd_card_bluez_sco_t *bluez_sco;
	spinlock_t lock;
	unsigned int pcm_size;
	unsigned int pcm_count;
	unsigned int pcm_bps;		/* bytes per second */
	unsigned int pcm_irq_pos;	/* IRQ position */
	unsigned int pcm_buf_pos;	/* position in buffer */
	snd_pcm_substream_t *substream;
} snd_card_bluez_sco_pcm_t;

static snd_card_t *snd_bluez_sco_cards[SNDRV_CARDS] = SNDRV_DEFAULT_PTR;

static int snd_card_bluez_sco_playback_trigger(snd_pcm_substream_t *
					       substream, int cmd)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_pcm_t *bspcm =
	    snd_magic_cast(snd_card_bluez_sco_pcm_t, runtime->private_data,
			   return -ENXIO);
	snd_card_bluez_sco_t *bluez_sco = snd_pcm_substream_chip(substream);

	dprintk("playback_trigger %d\n", cmd);

	if (cmd == SNDRV_PCM_TRIGGER_START) {
		bluez_sco->playback = bspcm;
		dprintk("setting playback to bspcm\n");
	} else if (cmd == SNDRV_PCM_TRIGGER_STOP) {
		bluez_sco->playback = NULL;
		dprintk("setting playback to NULL\n");
	} else {
		return -EINVAL;
	}
	return 0;
}

static int snd_card_bluez_sco_capture_trigger(snd_pcm_substream_t *
					      substream, int cmd)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_pcm_t *bspcm =
	    snd_magic_cast(snd_card_bluez_sco_pcm_t, runtime->private_data,
			   return -ENXIO);
	snd_card_bluez_sco_t *bluez_sco = snd_pcm_substream_chip(substream);

	dprintk("capture_trigger %d\n", cmd);

	if (cmd == SNDRV_PCM_TRIGGER_START) {
		bluez_sco->capture = bspcm;
		dprintk("setting capture to bspcm\n");
	} else if (cmd == SNDRV_PCM_TRIGGER_STOP) {
		bluez_sco->capture = NULL;
		dprintk("setting capture to NULL\n");
	} else {
		return -EINVAL;
	}
	return 0;
}

static int snd_card_bluez_sco_pcm_prepare(snd_pcm_substream_t * substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_pcm_t *bspcm =
	    snd_magic_cast(snd_card_bluez_sco_pcm_t, runtime->private_data,
			   return -ENXIO);
	unsigned int bps;

	bps = runtime->rate * runtime->channels;
	bps *= snd_pcm_format_width(runtime->format);
	bps /= 8;
	if (bps <= 0)
		return -EINVAL;
	bspcm->pcm_bps = bps;
	bspcm->pcm_size = snd_pcm_lib_buffer_bytes(substream);
	bspcm->pcm_count = snd_pcm_lib_period_bytes(substream);
	bspcm->pcm_irq_pos = 0;
	bspcm->pcm_buf_pos = 0;
	dprintk(
	       "prepare ok bps: %d size: %d count: %d\n",
	       bspcm->pcm_bps, bspcm->pcm_size,
	       bspcm->pcm_count);
	return 0;
}

static int snd_card_bluez_sco_playback_prepare(snd_pcm_substream_t *
					       substream)
{
	return snd_card_bluez_sco_pcm_prepare(substream);
}

static int snd_card_bluez_sco_capture_prepare(snd_pcm_substream_t *
					      substream)
{
	dprintk("capture_prepare\n");
	return snd_card_bluez_sco_pcm_prepare(substream);
}

static void snd_card_bluez_sco_pcm_receive(snd_card_bluez_sco_pcm_t * bspcm,
					   unsigned char *data,
					   unsigned int len)
{
	unsigned long flags;
	unsigned int oldptr;

	spin_lock_irqsave(&bspcm->lock, flags);
	oldptr = bspcm->pcm_buf_pos;
	bspcm->pcm_irq_pos += len;
	bspcm->pcm_buf_pos += len;
	bspcm->pcm_buf_pos %= bspcm->pcm_size;
	spin_unlock_irqrestore(&bspcm->lock, flags);
	/* copy a data chunk */
	if (oldptr + len > bspcm->pcm_size) {
		unsigned int cnt = bspcm->pcm_size - oldptr;
		memcpy(bspcm->substream->runtime->dma_area + oldptr, data,
		       cnt);
		memcpy(bspcm->substream->runtime->dma_area, data + cnt,
		       len - cnt);
	} else {
		memcpy(bspcm->substream->runtime->dma_area + oldptr, data,
		       len);
	}
	/* update the pointer, call callback if necessary */
	spin_lock_irqsave(&bspcm->lock, flags);
	if (bspcm->pcm_irq_pos >= bspcm->pcm_count) {
		bspcm->pcm_irq_pos %= bspcm->pcm_count;
		spin_unlock_irqrestore(&bspcm->lock, flags);
		snd_pcm_period_elapsed(bspcm->substream);
	} else
		spin_unlock_irqrestore(&bspcm->lock, flags);

}

static void snd_card_bluez_sco_pcm_send(snd_card_bluez_sco_pcm_t * bspcm,
					unsigned char *data,
					unsigned int len)
{
	unsigned long flags;
	unsigned int oldptr;

	spin_lock_irqsave(&bspcm->lock, flags);
	oldptr = bspcm->pcm_buf_pos;
	bspcm->pcm_irq_pos += len;
	bspcm->pcm_buf_pos += len;
	bspcm->pcm_buf_pos %= bspcm->pcm_size;
	spin_unlock_irqrestore(&bspcm->lock, flags);
	/* copy a data chunk */
	if (oldptr + len > bspcm->pcm_size) {
		unsigned int cnt = bspcm->pcm_size - oldptr;
		memcpy(data, bspcm->substream->runtime->dma_area + oldptr,
		       cnt);
		memcpy(data + cnt, bspcm->substream->runtime->dma_area,
		       len - cnt);
	} else {
		memcpy(data, bspcm->substream->runtime->dma_area + oldptr,
		       len);
	}
	/* update the pointer, call callback if necessary */
	spin_lock_irqsave(&bspcm->lock, flags);
	if (bspcm->pcm_irq_pos >= bspcm->pcm_count) {
		bspcm->pcm_irq_pos %= bspcm->pcm_count;
		spin_unlock_irqrestore(&bspcm->lock, flags);
		snd_pcm_period_elapsed(bspcm->substream);
	} else
		spin_unlock_irqrestore(&bspcm->lock, flags);
}

static snd_pcm_uframes_t
snd_card_bluez_sco_playback_pointer(snd_pcm_substream_t * substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_pcm_t *bspcm =
	    snd_magic_cast(snd_card_bluez_sco_pcm_t, runtime->private_data,
			   return -ENXIO);

	return bytes_to_frames(runtime, bspcm->pcm_buf_pos);
}

static snd_pcm_uframes_t
snd_card_bluez_sco_capture_pointer(snd_pcm_substream_t * substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_pcm_t *bspcm =
	    snd_magic_cast(snd_card_bluez_sco_pcm_t, runtime->private_data,
			   return -ENXIO);

	return bytes_to_frames(runtime, bspcm->pcm_buf_pos);
}

static snd_pcm_hardware_t snd_card_bluez_sco_playback = {
	.info = (SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
		 SNDRV_PCM_INFO_MMAP_VALID),
	.formats = SNDRV_PCM_FMTBIT_S8 /* | SNDRV_PCM_FMTBIT_S16_LE */ ,
	.rates = SNDRV_PCM_RATE_8000,
	.rate_min = 8000,
	.rate_max = 8000,
	.channels_min = 1,
	.channels_max = 1,
	.buffer_bytes_max = MAX_BUFFER_SIZE,
	.period_bytes_min = 64,
	.period_bytes_max = MAX_BUFFER_SIZE,
	.periods_min = 1,
	.periods_max = 1024,
	.fifo_size = 0,
};

static snd_pcm_hardware_t snd_card_bluez_sco_capture = {
	.info = (SNDRV_PCM_INFO_MMAP | SNDRV_PCM_INFO_INTERLEAVED |
		 SNDRV_PCM_INFO_MMAP_VALID),
	.formats = SNDRV_PCM_FMTBIT_S8 /* | SNDRV_PCM_FMTBIT_S16_LE */ ,
	.rates = SNDRV_PCM_RATE_8000,
	.rate_min = 8000,
	.rate_max = 8000,
	.channels_min = 1,
	.channels_max = 1,
	.buffer_bytes_max = MAX_BUFFER_SIZE,
	.period_bytes_min = 64,
	.period_bytes_max = MAX_BUFFER_SIZE,
	.periods_min = 1,
	.periods_max = 1024,
	.fifo_size = 0,
};

static void snd_card_bluez_sco_runtime_free(snd_pcm_runtime_t * runtime)
{
	snd_card_bluez_sco_pcm_t *bspcm =
	    snd_magic_cast(snd_card_bluez_sco_pcm_t, runtime->private_data,
			   return);
	snd_magic_kfree(bspcm);
}

static int snd_card_bluez_sco_playback_open(snd_pcm_substream_t *
					    substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_pcm_t *bspcm;

	dprintk("playback_open\n");
	
	bspcm = snd_magic_kcalloc(snd_card_bluez_sco_pcm_t, 0, GFP_KERNEL);
	if (bspcm == NULL)
		return -ENOMEM;
	if ((runtime->dma_area =
	     snd_malloc_pages_fallback(MAX_BUFFER_SIZE, GFP_KERNEL,
				       &runtime->dma_bytes)) == NULL) {
		snd_magic_kfree(bspcm);
		return -ENOMEM;
	}
	spin_lock_init(&bspcm->lock);
	bspcm->substream = substream;
	runtime->private_data = bspcm;
	runtime->private_free = snd_card_bluez_sco_runtime_free;
	runtime->hw = snd_card_bluez_sco_playback;
	return 0;
}

static int snd_card_bluez_sco_capture_open(snd_pcm_substream_t * substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_pcm_t *bspcm;

	dprintk("capture_open\n");

	bspcm = snd_magic_kcalloc(snd_card_bluez_sco_pcm_t, 0, GFP_KERNEL);
	if (bspcm == NULL)
		return -ENOMEM;
	if ((runtime->dma_area =
	     snd_malloc_pages_fallback(MAX_BUFFER_SIZE, GFP_KERNEL,
				       &runtime->dma_bytes)) == NULL) {
		snd_magic_kfree(bspcm);
		return -ENOMEM;
	}
	memset(runtime->dma_area, 0, runtime->dma_bytes);
	spin_lock_init(&bspcm->lock);
	bspcm->substream = substream;
	runtime->private_data = bspcm;
	runtime->private_free = snd_card_bluez_sco_runtime_free;
	runtime->hw = snd_card_bluez_sco_capture;
	return 0;
}

static int snd_card_bluez_sco_playback_close(snd_pcm_substream_t *
					     substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	snd_card_bluez_sco_t *bluez_sco = snd_pcm_substream_chip(substream);

	snd_assert(bluez_sco->playback == NULL, ;);
	
    	/* Ensure any references to this in our thread have finished */
	down(&bluez_sco->playback_sem);
	up(&bluez_sco->playback_sem);

	snd_free_pages(runtime->dma_area, runtime->dma_bytes);
	return 0;
}

static int snd_card_bluez_sco_capture_close(snd_pcm_substream_t *
					    substream)
{
	snd_pcm_runtime_t *runtime = substream->runtime;
	struct snd_card_bluez_sco *bluez_sco = (struct snd_card_bluez_sco *) substream->private_data;

	snd_assert(bluez_sco->capture == NULL, ;);

    	/* Ensure any references to this in our thread have finished */
	down(&bluez_sco->capture_sem);
	up(&bluez_sco->capture_sem);

	snd_free_pages(runtime->dma_area, runtime->dma_bytes);
	return 0;
}

static snd_pcm_ops_t snd_card_bluez_sco_playback_ops = {
	.open = snd_card_bluez_sco_playback_open,
	.close = snd_card_bluez_sco_playback_close,
	.ioctl = snd_pcm_lib_ioctl,
	.prepare = snd_card_bluez_sco_playback_prepare,
	.trigger = snd_card_bluez_sco_playback_trigger,
	.pointer = snd_card_bluez_sco_playback_pointer,
};

static snd_pcm_ops_t snd_card_bluez_sco_capture_ops = {
	.open = snd_card_bluez_sco_capture_open,
	.close = snd_card_bluez_sco_capture_close,
	.ioctl = snd_pcm_lib_ioctl,
	.prepare = snd_card_bluez_sco_capture_prepare,
	.trigger = snd_card_bluez_sco_capture_trigger,
	.pointer = snd_card_bluez_sco_capture_pointer,
};

static int __init snd_card_bluez_sco_pcm(snd_card_bluez_sco_t * bluez_sco)
{
	snd_pcm_t *pcm;
	int err;

	if ((err =
	     snd_pcm_new(bluez_sco->card, "Bluez SCO PCM", 0, 1, 1,
			 &pcm)) < 0)
		return err;
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_PLAYBACK,
			&snd_card_bluez_sco_playback_ops);
	snd_pcm_set_ops(pcm, SNDRV_PCM_STREAM_CAPTURE,
			&snd_card_bluez_sco_capture_ops);
	pcm->private_data = bluez_sco;
	pcm->info_flags = 0;
	strcpy(pcm->name, "Bluez SCO PCM");
	return 0;
}

#define BLUEZ_SCO_VOLUME(xname, xindex, addr) \
{ .iface = SNDRV_CTL_ELEM_IFACE_MIXER, .name = xname, .index = xindex, \
                        .info = snd_bluez_sco_volume_info, \
                                .get = snd_bluez_sco_volume_get, .put = snd_bluez_sco_volume_put, \
                                                                        .private_value = addr }

static int snd_bluez_sco_volume_info(snd_kcontrol_t * kcontrol,
				     snd_ctl_elem_info_t * uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_INTEGER;
	uinfo->count = 1;
	uinfo->value.integer.min = MIXER_MIN_VOLUME;
	uinfo->value.integer.max = MIXER_MAX_VOLUME;
	return 0;
}

static int snd_bluez_sco_volume_get(snd_kcontrol_t * kcontrol,
				    snd_ctl_elem_value_t * ucontrol)
{
	snd_card_bluez_sco_t *bluez_sco = _snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int addr = kcontrol->private_value;

	spin_lock_irqsave(&bluez_sco->mixer_lock, flags);
	ucontrol->value.integer.value[0] = bluez_sco->mixer_volume[addr];
	spin_unlock_irqrestore(&bluez_sco->mixer_lock, flags);
	return 0;
}

static int snd_bluez_sco_volume_put(snd_kcontrol_t * kcontrol,
				    snd_ctl_elem_value_t * ucontrol)
{
	snd_card_bluez_sco_t *bluez_sco = _snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int changed, addr = kcontrol->private_value;
	int vol;

	vol = ucontrol->value.integer.value[0];
	if (vol < MIXER_MIN_VOLUME)
		vol =MIXER_MIN_VOLUME;
	if (vol > MIXER_MAX_VOLUME)
		vol = MIXER_MAX_VOLUME;
	spin_lock_irqsave(&bluez_sco->mixer_lock, flags);
	changed = bluez_sco->mixer_volume[addr] != vol;
	bluez_sco->mixer_volume[addr] = vol;
	spin_unlock_irqrestore(&bluez_sco->mixer_lock, flags);
	if (changed) {
		spin_lock_irqsave(&bluez_sco->mixer_changed_lock, flags);
		bluez_sco->mixer_changed = 1;
		spin_unlock_irqrestore(&bluez_sco->mixer_changed_lock, flags);
		wake_up(&bluez_sco->hwdep_wait);
	}
	return changed;
}

static int snd_bluez_sco_boolean_info(snd_kcontrol_t *kcontrol,
                          snd_ctl_elem_info_t *uinfo)
{
	uinfo->type = SNDRV_CTL_ELEM_TYPE_BOOLEAN;
	uinfo->count = 1;
	uinfo->value.integer.min = 0;
	uinfo->value.integer.max = 1;
	return 0;
}

static int snd_bluez_sco_loopback_get(snd_kcontrol_t * kcontrol,
				    snd_ctl_elem_value_t * ucontrol)
{
	snd_card_bluez_sco_t *bluez_sco = _snd_kcontrol_chip(kcontrol);
	unsigned long flags;

	spin_lock_irqsave(&bluez_sco->mixer_lock, flags);
	ucontrol->value.integer.value[0] = bluez_sco->loopback;
	spin_unlock_irqrestore(&bluez_sco->mixer_lock, flags);
	return 0;
}

static int snd_bluez_sco_loopback_put(snd_kcontrol_t * kcontrol,
				    snd_ctl_elem_value_t * ucontrol)
{
	snd_card_bluez_sco_t *bluez_sco = _snd_kcontrol_chip(kcontrol);
	unsigned long flags;
	int changed;
	int loopback;

	loopback = !!ucontrol->value.integer.value[0];
	
	spin_lock_irqsave(&bluez_sco->mixer_lock, flags);
	changed = bluez_sco->loopback != loopback;
	bluez_sco->loopback = loopback;
	spin_unlock_irqrestore(&bluez_sco->mixer_lock, flags);
	return changed;
}

#define BLUEZ_SCO_CONTROLS (sizeof(snd_bluez_sco_controls)/sizeof(snd_kcontrol_new_t))

static snd_kcontrol_new_t snd_bluez_sco_controls[] = {
	BLUEZ_SCO_VOLUME("Master Playback Volume", 0, MIXER_ADDR_MASTER),
	BLUEZ_SCO_VOLUME("Master Capture Volume", 0, MIXER_ADDR_MIC),
	{	.iface = SNDRV_CTL_ELEM_IFACE_MIXER, 
		.name = "Loopback Switch",
		.index = 0, 
                .info = snd_bluez_sco_boolean_info,
                .get = snd_bluez_sco_loopback_get,
		.put = snd_bluez_sco_loopback_put,
	}
};

int __init snd_card_bluez_sco_new_mixer(snd_card_bluez_sco_t *bluez_sco)
{
	snd_card_t *card = bluez_sco->card;
	
	unsigned int idx;
	int err;

	snd_assert(bluez_sco != NULL, return -EINVAL);
	spin_lock_init(&bluez_sco->mixer_lock);
	strcpy(card->mixername, "Bluez Headset Mixer");

	for (idx = 0; idx < BLUEZ_SCO_CONTROLS; idx++) {
		bluez_sco->mixer_controls[idx] = 
			snd_ctl_new1(&snd_bluez_sco_controls[idx],bluez_sco);
	
		if ((err = snd_ctl_add(card,bluez_sco->mixer_controls[idx])) < 0)
			return err;
	}
	return 0;
}

/**
 *	The following three functions are copied from net/socket.c
 */

static inline struct socket *socki_lookup(struct inode *inode)
{
	return &inode->u.socket_i;
}

static struct socket *sockfd_lookup(int fd, int *err)
{
	struct file *file;
	struct inode *inode;
	struct socket *sock;

	if (!(file = fget(fd))) {
		*err = -EBADF;
		return NULL;
	}

	inode = file->f_dentry->d_inode;
	if (!inode->i_sock || !(sock = socki_lookup(inode))) {
		*err = -ENOTSOCK;
		fput(file);
		return NULL;
	}

	if (sock->file != file) {
		printk(KERN_ERR "socki_lookup: socket file changed!\n");
		sock->file = file;
	}
	return sock;
}

static __inline__ void sockfd_put(struct socket *sock)
{
	fput(sock->file);
}


static int snd_card_bluez_open(snd_hwdep_t * hw, struct file *file)
{
	return 0;
}

static int snd_card_bluez_release(snd_hwdep_t * hw, struct file *file)
{
	return 0;
}


static int snd_card_bluez_ioctl(snd_hwdep_t * hw, struct file *file,
				unsigned int cmd, unsigned long arg)
{
	snd_card_bluez_sco_t *bluez_sco = 
		snd_magic_cast(snd_card_bluez_sco_t,hw->card->private_data,return -ENXIO);
	struct socket *sock;
	int err = -EINVAL;
	int fd = arg;

	switch (cmd) {
	case SNDRV_BLUEZ_SCO_IOCTL_SET_SCO_SOCKET:
		err = 0;
		/*  Interrupt any socket operations, so that we may
		 *  change the socket */
    	    	down(&bluez_sco->sock_sem);
		kill_proc(bluez_sco->thread_pid, SIGINT, 1);
		if (bluez_sco->sco_sock) {
			dprintk("Disposing of previous socket count %d\n",file_count(bluez_sco->sco_sock->file));
			sockfd_put((struct socket*) bluez_sco->sco_sock);

			bluez_sco->sco_sock = NULL;
		}
		
		if (fd>=0) {
			err = -EINVAL;
			sock = sockfd_lookup(fd, &err);
			if (sock) {
				if (sock->sk->family == PF_BLUETOOTH &&
				    sock->sk->protocol == BTPROTO_SCO) {
					bluez_sco->sco_sock = sock;
					wake_up(&bluez_sco->wait);
					err = 0;
				} else {
					dprintk("Not a bluetooth SCO socket %d:%d\n",
						sock->sk->family,
						sock->sk->protocol);
					sockfd_put(sock);
				}
			}
		}
    	    	up(&bluez_sco->sock_sem);
		break;
	}
	return err;
}

static long snd_card_bluez_write(snd_hwdep_t * hw, const char *buf, long count,
				loff_t * offset)
{
	snd_card_bluez_sco_t *bluez_sco = 
		snd_magic_cast(snd_card_bluez_sco_t,hw->card->private_data,return -ENXIO);
	int mixer_volume[MIXER_ADDR_LAST + 1];
	int retval;
	int i;
	
	if (count != sizeof(mixer_volume))
		return -EINVAL;
	
	if (copy_from_user(mixer_volume,buf,sizeof(mixer_volume)))
		return -EFAULT;
		
	retval = sizeof(mixer_volume);
		
	spin_lock_irq(&bluez_sco->mixer_lock);
	for (i=0;i<=MIXER_ADDR_LAST;i++) {
		if (bluez_sco->mixer_volume[i] != mixer_volume[i]) {
			bluez_sco->mixer_volume[i] = mixer_volume[i];
			snd_ctl_notify(bluez_sco->card, 
				SNDRV_CTL_EVENT_MASK_VALUE, 
				&bluez_sco->mixer_controls[i]->id);
		}
	}
	spin_unlock_irq(&bluez_sco->mixer_lock);
	
	return retval;
}

static long snd_card_bluez_read(snd_hwdep_t * hw, char *buf, long count,
				loff_t * offset)
{
	snd_card_bluez_sco_t *bluez_sco = 
		snd_magic_cast(snd_card_bluez_sco_t,hw->card->private_data,return -ENXIO);
	DECLARE_WAITQUEUE(wait, current);
	ssize_t retval;

	if (count != sizeof(bluez_sco->mixer_volume))
		return -EINVAL;

	add_wait_queue(&bluez_sco->hwdep_wait, &wait);
	current->state = TASK_INTERRUPTIBLE;
	do {
		int changed;
		spin_lock_irq(&bluez_sco->mixer_changed_lock);
		changed = bluez_sco->mixer_changed;
		bluez_sco->mixer_changed = 0;
		spin_unlock_irq(&bluez_sco->mixer_changed_lock);

		if (changed != 0)
			break;

		if (signal_pending(current)) {
			retval = -ERESTARTSYS;
			goto out;
		}
		schedule();
	} while (1);
	if (copy_to_user(buf, bluez_sco->mixer_volume, sizeof(bluez_sco->mixer_volume)))
		retval = -EFAULT;
	else	retval = sizeof(bluez_sco->mixer_volume);
	
      out:
	current->state = TASK_RUNNING;
	remove_wait_queue(&bluez_sco->hwdep_wait, &wait);
	return retval;
}

static unsigned int snd_card_bluez_poll(snd_hwdep_t * hw,
					struct file *file,
					poll_table * wait)
{
	snd_card_bluez_sco_t *bluez_sco = 
		snd_magic_cast(snd_card_bluez_sco_t,hw->card->private_data,return 0);
	int changed;

	poll_wait(file, &bluez_sco->hwdep_wait, wait);

	spin_lock_irq(&bluez_sco->mixer_changed_lock);
	changed = bluez_sco->mixer_changed;
	spin_unlock_irq(&bluez_sco->mixer_changed_lock);

	if (changed != 0)
		return POLLIN | POLLRDNORM;
	return 0;
}

static int snd_card_bluez_sco_thread(void *data)
{
	snd_card_t *card = (snd_card_t *) data;
	snd_card_bluez_sco_t *bluez_sco = 
		snd_magic_cast(snd_card_bluez_sco_t,card->private_data,return 0);
	struct socket *sock;
	int len;
#define BUF_SIZE 256
	unsigned char buf[BUF_SIZE];
	struct msghdr msg;
	struct iovec iov;
    	mm_segment_t fs;
	
	lock_kernel();

	daemonize();
	reparent_to_init();

	sprintf(current->comm, "snd-bluez-sco");

	up(&bluez_sco->thread_sem);

	do {
		if (signal_pending(current))
			flush_signals(current);

		/*	This may be woken up by a wake_up() when
		 *	a new socket is installed, or by a signal.
		 *	Signals are sent to terminate the thread,
		 *	in which case thread_exit is set, and to force
		 *	recvmesg() to wake up (from the ioctl handler)
		 */
    	    	wait_event_interruptible(bluez_sco->wait,
		    	bluez_sco->sco_sock!=0);
		if (bluez_sco->thread_exit)
			break;

    	    	down(&bluez_sco->sock_sem);
		sock = (struct socket*) bluez_sco->sco_sock;
		if (sock)
		    get_file(sock->file);
		up(&bluez_sco->sock_sem);

		if (!sock)
		    continue;

		/* We have a socket, let's read from it and write to it... */

		memset(&msg,0,sizeof(msg));
		msg.msg_iov = &iov;
		iov.iov_base = buf;
		iov.iov_len = BUF_SIZE;
		
		/* Pretend so that copy_to_user and friends work */
		fs = get_fs();
		set_fs(get_ds());

		/* This will block until we receive data or a signal */
		len = sock->ops->recvmsg(sock, &msg, BUF_SIZE, 0, NULL);
		if (len > 0) {

			down(&bluez_sco->capture_sem);
			if (bluez_sco->capture) {
				snd_card_bluez_sco_pcm_receive
				    (bluez_sco->capture,
				     buf, len);
			}
			up(&bluez_sco->capture_sem);

			down(&bluez_sco->playback_sem);
			if (bluez_sco->playback || !bluez_sco->loopback)
				memset(buf, 0, len);
			if (bluez_sco->playback) {
				snd_card_bluez_sco_pcm_send
				    (bluez_sco->playback,
				     buf, len);
			}
			up(&bluez_sco->playback_sem);

			msg.msg_flags = 0;
			msg.msg_iov = &iov;
			iov.iov_base = buf;
			iov.iov_len = BUF_SIZE;
			sock->ops->sendmsg(sock, &msg, len,
					   NULL);
		}

		set_fs(fs);
		/* Expect this to be 3 because we have a copy,
		   the bluez_sco has one, and so does the app.
		   */
		if (file_count(sock->file) != 3) {
			dprintk("file_count is %d\n",file_count(sock->file));
		}
    	    	fput(sock->file);

		schedule();
	} while (!bluez_sco->thread_exit);

	dprintk("thread exiting\n");

	unlock_kernel();
	complete_and_exit(&bluez_sco->thread_done, 0);
}

static void snd_card_bluez_private_free(snd_card_t * card)
{
	snd_card_bluez_sco_t *bluez_sco = snd_magic_cast(snd_card_bluez_sco_t,card->private_data,return);

	dprintk("private_free, killing thread\n");
	bluez_sco->thread_exit = 1;
	kill_proc(bluez_sco->thread_pid, SIGTERM, 1);
	wait_for_completion(&bluez_sco->thread_done);

	if (bluez_sco->sco_sock) {
		dprintk("shutdown: freeing socket count %d\n",file_count(bluez_sco->sco_sock->file));
		sockfd_put((struct socket*) bluez_sco->sco_sock);
	}
	
	snd_magic_kfree(bluez_sco);
}

static int __init snd_card_bluez_sco_probe(int dev)
{
	snd_card_t *card;
	snd_card_bluez_sco_t *bluez_sco;
	int err;
	snd_hwdep_t *hw;

	card =
	    snd_card_new(SNDRV_DEFAULT_IDX1, SNDRV_DEFAULT_STR1,
			 THIS_MODULE, 0);
	if (card == NULL)
		return -ENOMEM;
		
	bluez_sco = snd_magic_kcalloc(snd_card_bluez_sco_t, 0, GFP_KERNEL);
	card->private_data = bluez_sco;
	card->private_free = snd_card_bluez_private_free;

	bluez_sco->card = card;

	init_completion(&bluez_sco->thread_done);
	init_MUTEX_LOCKED(&bluez_sco->thread_sem);
	init_MUTEX(&bluez_sco->sock_sem);
	init_MUTEX(&bluez_sco->capture_sem);
	init_MUTEX(&bluez_sco->playback_sem);
	init_waitqueue_head(&bluez_sco->wait);
	init_waitqueue_head(&bluez_sco->hwdep_wait);
	spin_lock_init(&bluez_sco->mixer_changed_lock);

	bluez_sco->thread_pid =
	    kernel_thread(snd_card_bluez_sco_thread, card,
			  CLONE_FS | CLONE_FILES | CLONE_SIGHAND);
	if (bluez_sco->thread_pid < 0) {
		err = bluez_sco->thread_pid;
		goto __nodev;
	}

	down(&bluez_sco->thread_sem);

	if ((err = snd_card_bluez_sco_pcm(bluez_sco)) < 0)
		goto __nodev;
	if ((err = snd_card_bluez_sco_new_mixer(bluez_sco)) < 0)
		goto __nodev;
	strcpy(card->driver, "Bluez SCO");
	strcpy(card->shortname, "Bluez Headset");
	sprintf(card->longname, "Bluez Headset %i", dev + 1);

	err = snd_hwdep_new(card, "BluezSCO", 0, &hw);
	if (err < 0)
		goto __nodev;

	sprintf(hw->name, "BluezSCO");
	hw->iface = SNDRV_HWDEP_IFACE_BLUEZ_SCO;
	hw->ops.open 	= snd_card_bluez_open;
	hw->ops.ioctl 	= snd_card_bluez_ioctl;
	hw->ops.release	= snd_card_bluez_release;
	hw->ops.read 	= snd_card_bluez_read;
	hw->ops.write 	= snd_card_bluez_write;
	hw->ops.poll 	= snd_card_bluez_poll;

	if ((err = snd_card_register(card)) == 0) {
		snd_bluez_sco_cards[dev] = card;
		return 0;
	}
      __nodev:
	snd_card_free(card);
	return err;
}

static int __init alsa_card_bluez_sco_init(void)
{
	printk(KERN_INFO "snd-bluez-sco revision %s\n",mod_revision+11);
	
	if (snd_card_bluez_sco_probe(0) < 0) {
#ifdef MODULE
		printk(KERN_ERR
		       "Bluez SCO soundcard not found or device busy\n");
#endif
		return -ENODEV;
	}
	return 0;
}

static void __exit alsa_card_bluez_sco_exit(void)
{
	int idx;

	for (idx = 0; idx < SNDRV_CARDS; idx++)
		snd_card_free(snd_bluez_sco_cards[idx]);
}

module_init(alsa_card_bluez_sco_init)
    module_exit(alsa_card_bluez_sco_exit)
#ifndef MODULE
static int __init alsa_card_bluez_sco_setup(char *str)
{
	static unsigned __initdata nr_dev = 0;

	if (nr_dev >= SNDRV_CARDS)
		return 0;
	nr_dev++;
	return 1;
}

__setup("snd-bluez-sco=", alsa_card_bluez_sco_setup);

#endif				/* ifndef MODULE */
