/*
 * gpio.c: Simple program to configure/read/write PXA GPIO's.
 *
 *  Copyright (C) 2003, Jamey Hicks <jamey.hicks@hp.com>
 *  Copyright (C) 2000, Jan-Derk Bakker (J.D.Bakker@its.tudelft.nl)
 *
 *
 * This software has been developed for the LART computing board
 * (http://www.lart.tudelft.nl/). The development has been sponsored by
 * the Mobile MultiMedia Communications (http://www.mmc.tudelft.nl/)
 * and Ubiquitous Communications (http://www.ubicom.tudelft.nl/)
 * projects.
 *
 * The author can be reached at:
 *
 *  Jan-Derk Bakker
 *  Information and Communication Theory Group
 *  Faculty of Information Technology and Systems
 *  Delft University of Technology
 *  P.O. Box 5031
 *  2600 GA Delft
 *  The Netherlands
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <getopt.h>
#include <string.h>
#include <errno.h>
#include <signal.h>
#include <fcntl.h>
#include <ctype.h>
#include <termios.h>
#include <sys/types.h>
#include <sys/mman.h>
  
#define FATAL do { fprintf(stderr, "Error at line %d, file %s (%d) [%s]\n", \
  __LINE__, __FILE__, errno, strerror(errno)); exit(1); } while(0)
 
#define MAP_SIZE 4096UL
#define MAP_MASK (MAP_SIZE - 1)

int main(int argc, char **argv) 
{
     int fd;
     volatile unsigned long *gpio_base; 
     off_t gpio_nr;
     unsigned long gplr, gpdr, gafr;
	
     int dir = 0, set_dir = 0;
     int alt_fcn = 0, set_alt_fcn = 0;
     int clear = 0, set = 0;
     int rising = 0, falling = 0;
     int verbose = 0;

     if (argc < 2) {
	  fprintf(stderr, "\nUsage:\t%s gpio_nr\n",
		  argv[0]);
	  exit(1);
     }
     while (1) {
	  int c = getopt_long(argc, argv, "a:cfiorsv", NULL, NULL);
	  if (c == -1)
	       break;
	  switch (c) {
	  case 'a':
	       alt_fcn = optarg;
	       set_alt_fcn = 1;
	       break;
	  case 'c':
	       clear = 1;
	       break;
	  case 'f':
	       falling = 1;
	       break;
	  case 'i':
	       dir = 0;
	       set_dir = 1;
	       break;
	  case 'o':
	       dir = 1;
	       set_dir = 1;
	       break;
	  case 'r':
	       rising = 1;
	       break;
	  case 's':
	       set = 1;
	       break;
	  case 'v':
	       verbose = 1;
	       break;
	  default:
	  }
     }
     gpio_nr = strtoul(argv[optind], 0, 0);

     if ((fd = open("/dev/mem", O_RDWR | O_SYNC)) == -1) FATAL;
    
     /* Map one page */
     gpio_base = mmap(0, MAP_SIZE, PROT_READ | PROT_WRITE, MAP_SHARED, fd, 0x40e00000 & ~MAP_MASK);
     if (gpio_base == (void *) -1) FATAL;
    
     gplr = *(gpio_base + (0x0000 >> 2) + (gpio_nr >> 5));
     gpdr = *(gpio_base + (0x000c >> 2) + (gpio_nr >> 5));
     gafr = *(gpio_base + (0x0054 >> 2) + (gpio_nr >> 4));

     if (clear)
	  *(gpio_base + (0x0024 >> 2) + (gpio_nr >> 5)) = (1 << (gpio_nr&0x1f));
     if (set)
	  *(gpio_base + (0x0018 >> 2) + (gpio_nr >> 5)) = (1 << (gpio_nr&0x1f));
     if (set_alt_fcn)
	  *(gpio_base + (0x0054 >> 2) + (gpio_nr >> 4)) = (alt_fcn << 2*(gpio_nr&0x0f));
     if (set_dir)
	  *(gpio_base + (0x000c >> 2) + (gpio_nr >> 5)) = (dir << (gpio_nr&0x1f));
     if (falling)
	  *(gpio_base + (0x003c >> 2) + (gpio_nr >> 5)) = (1 << (gpio_nr&0x1f));
     if (rising)
	  *(gpio_base + (0x0030 >> 2) + (gpio_nr >> 5)) = (1 << (gpio_nr&0x1f));

     if (verbose) {
	  unsigned long grer = *(gpio_base + (0x0030 >> 2) + (gpio_nr >> 5));
	  unsigned long gfer = *(gpio_base + (0x003c >> 2) + (gpio_nr >> 5));
	  unsigned long gedr = *(gpio_base + (0x0048 >> 2) + (gpio_nr >> 5));
	  printf("GPLR(%d)=0x%08x\n", gpio_nr >> 5, gplr);
	  printf("GPDR(%d)=0x%08x\n", gpio_nr >> 5, gpdr);
	  printf("GAFR(%d)=0x%08x\n", gpio_nr >> 4, gafr);
	  printf("GRER(%d)=0x%08x\n", gpio_nr >> 5, grer);
	  printf("GFER(%d)=0x%08x\n", gpio_nr >> 5, gfer);
	  printf("GEDR(%d)=0x%08x\n", gpio_nr >> 5, gedr);
     }

     printf("GPLR[%d]=0x%08x", gpio_nr, gplr & (1 << (gpio_nr&0x1f)));
     printf("\t GPDR[%d]=0x%08x", gpio_nr, gpdr & (1 << (gpio_nr&0x1f)));
     printf("\t GAFR[%d]=0x%08x\n", gpio_nr, gafr & (3 << 2*(gpio_nr&0x0f)));
	
     if (munmap(gpio_base, MAP_SIZE) == -1) FATAL;
     close(fd);
     return 0;
}

