/* sprintf_alloc.c -- like sprintf with memory allocation

   Carl D. Worth

   Copyright (C) 2001 University of Southern California

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.
*/

#include "ipkg.h"
#include <stdarg.h>

#include "sprintf_alloc.h"

static int vsprintf_alloc(char **str, const char *fmt, va_list ap);

int sprintf_alloc(char **str, const char *fmt, ...)
{
    int ret;
    va_list ap;

    if (!str) {
      fprintf(stderr, "Null string pointer passed to sprintf_alloc\n");
      return -1;
    }
    if (!fmt) {
      fprintf(stderr, "Null fmt string passed to sprintf_alloc\n");
      return -1;
    }
    va_start(ap, fmt);
    ret = vsprintf_alloc(str, fmt, ap);
    va_end(ap);

    return ret;
}

/* ripped more or less straight out of PRINTF(3) */
static int vsprintf_alloc(char **str, const char *fmt, va_list ap)
{
    char *new_str;
    /* Guess we need no more than 100 bytes. */
    int n, size = 100;
 
    if ((new_str = malloc (size)) == NULL)
	return -1;
    *str = new_str;
    while (1) {
	/* Try to print in the allocated space. */
	n = vsnprintf (new_str, size, fmt, ap);
	/* If that worked, return the size. */
	if (n > -1 && n < size)
	    return n;
	/* Else try again with more space. */
	if (n > -1)    /* glibc 2.1 */
	    size = n+1; /* precisely what is needed */
	else           /* glibc 2.0 */
	    size *= 2;  /* twice the old size */
	new_str = realloc(new_str, size);
	if (new_str == NULL) {
	    free(new_str);
	    *str = NULL;
	    return -1;
	}
	*str = new_str;
    }
}
