/*
 * (Unnamed) mixer - v0.04
 * Mixer for OSS/X11
 * Copyright 1998, Ben Buxton (bb@zip.com.au)
 *
 * Button1 over the letters selects the device show/changed by
 * the slider.
 * Button3 selects the input device.
 * Ctrl-Button1 moves the window around
 * Shift-Button1 quits
 *
 * check http://www.zip.com.au/~bb/linux/ for new versions
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */


#include <X11/X.h>
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xresource.h>
#include <X11/Xatom.h>
#include <linux/soundcard.h>
#include <X11/cursorfont.h>
#include <fcntl.h>
#include <stdio.h>

#define WIDTH 35
#define HEIGHT 100
#define FONT "-adobe-times-*-r-*-*-10-*-*-*-*-*-*-*"
#define VOLTEXT "V"
#define LINTEXT "L"
#define DSPTEXT "D"
#define CDTEXT "C"
#define MICTEXT "M"
#define QTEXT "Q"
#define RESOURCE_FILE "/etc/X11/app-defaults/Emix"

void DoBevels();
void GetVolume();
void OpenMixer();
void DrawSlider();
void SetVolume();
void DrawTicks();
void DrawButtons();
void dosizecursor(int,int,char,XColor *);

Display *disp;
int screen;
Window win, root;
Bool sliding=False;
Bool override;
Bool nogeom;
GC textagc, textbgc, topshad, botshad, tickgc, ingc, greygc;
XEvent event;

static XrmOptionDescRec opts[] = {
     {"-geometry", ".geometry", XrmoptionSepArg, (caddr_t) NULL},
     {"-display",  ".display",  XrmoptionSepArg, (caddr_t) NULL},
     {"-device",   ".device",   XrmoptionSepArg, (caddr_t) NULL},
     {"-override", ".override", XrmoptionNoArg, (caddr_t) "True"},
     {"-font",     ".font",     XrmoptionSepArg, (caddr_t) NULL},
     {"-nogeom",     ".nogeom",     XrmoptionNoArg, (caddr_t) "True"},
   NULL
};

/* Dimensions */
int width=WIDTH, height=HEIGHT, x, y;
/* Slider dims */
int swidth, sheight;
int volume;
/* Total slider area height, button area height */
int sw, bw;

/* Info for 'buttons'*/
XTextProperty prop;
XFontStruct *font;
char *fontname = FONT;
int volht, linht, cdht, dspht, micht, qht;
int liny, cdy, voly, dspy, micy, qy;

/* Info on current mixer dev */
int mixerfd;
int device = 1; /* 1 = main, 2 = line, 4 = cd, 8 = dsp 16 = mic */
int unavail = 0; /* List of unavailable mixers (mask)*/
int devmask;
int numdev;
char *mixer;
int indevice; /* Input device SOUND_MASK_LINE, SOUND_MASK_MIC, etc */


int main(int argc, char **argv) {
   Colormap cmap;
   XGCValues vals;
   unsigned long mask;
   XColor texta, textb, in, topgrey, botgrey, grey, tick;
   XSizeHints hints;
   XTextProperty prop;
   XSetWindowAttributes attributes;
   Window rr, cr;
   Bool moving = False;
   int rx, ry, px, py, mr, wx, wy;
   int tx, newh;
   int gstatus = 0;
   XrmDatabase rdb, srvdb = NULL, cmdb;
   XrmValue val;
   char *em = "";
   char *str_type, buffer[50];
   char *display = NULL;
   int rwidth, rheight;
   int d;
   
   /* Go through cmdline options, etc. Also connect to disp */

   XrmInitialize();

   cmdb = XrmGetStringDatabase(em);

   XrmParseCommand(&cmdb, opts, 6, "emix", &argc, argv);

   if (XrmGetResource(cmdb, "emix.display", "Emix.display", &str_type, &val))
     {
	display = (void *)malloc(val.size+1);
	strncpy(display, val.addr, val.size);
     }

   /* Open the display */
   disp = XOpenDisplay(display);
   if (disp == NULL)
     {
	printf("Error: Cannot open display %s!\n", display);
	exit(1);
     }
   
   XrmCombineFileDatabase(RESOURCE_FILE,&cmdb,False);
   
   if(XResourceManagerString(disp) != NULL)
     {
	srvdb = XrmGetStringDatabase(XResourceManagerString(disp));
     }
   
   XrmMergeDatabases(srvdb, &cmdb);
   
   rdb = cmdb;

   if (XrmGetResource(rdb, "emix.geometry", "Emix.geometry", &str_type, &val))
     {
	strncpy(buffer, val.addr, (int) val.size);
	gstatus = XParseGeometry(buffer, &x, &y, &width, &height);
	if((gstatus & XValue) || (gstatus & YValue))
	  hints.flags = USPosition;
	if((gstatus & WidthValue) || (gstatus & HeightValue))
	  hints.flags |= USSize;
     }

   if (XrmGetResource(rdb, "emix.device", "Emix.device", &str_type, &val))
     {
	mixer = (void *)malloc(val.size+1);
	strncpy(mixer, val.addr, val.size);
     } else {
	mixer = "/dev/mixer";
     }
   
   if (XrmGetResource(rdb, "emix.override", "Emix.override", &str_type, &val))
     {
	if(!strncasecmp("true", val.addr, val.size))
	  override = 1;
     }

   if (XrmGetResource(rdb, "emix.nogeom", "Emix.nogeom", &str_type, &val))
     {
	if(!strncasecmp("true", val.addr, val.size))
	  nogeom = 1;
     }

   if (XrmGetResource(rdb, "emix.font", "Emix.font", &str_type, &val))
     {
	fontname = (void *)malloc(val.size+1);
	strncpy(font, val.addr, val.size);
     }

   if(argc != 1) {
      printf("Usage: %s [-geom geometry] [-nogeom] [-dev mixer] [-override] [-font fn]\n", argv[0]);
      exit(1);
   }

   /* Do stuff with '-geom' command line option */
   if( !(gstatus & WidthValue))
     width = WIDTH;
   
   if( !(gstatus & HeightValue))
     height = HEIGHT;

   if( !(gstatus & XValue))
     x = 1;
   else if (gstatus & XNegative)
     x = rwidth + x - width;
   
   if( !(gstatus & YValue))
     y = 1;
   else if (gstatus & YNegative)
     y = rheight + y - height;

   hints.x = x;
   hints.y = y;
   hints.min_width = width;
   hints.min_height = height;
   hints.flags |= PMinSize;
   
/*   printf("Opening the mixer and getting the volume\n"); */
   OpenMixer();
   GetVolume();
   screen = DefaultScreen(disp);
   cmap = DefaultColormap(disp, screen);
   root = RootWindow(disp, screen);

/*   printf("Loading the font\n"); */
   /* Do stuff with font, etc */

   font = XLoadQueryFont(disp, fontname);
   if(font == NULL)
     {
        printf("Can't load font %s!\n", fontname);
        exit(1);
     }
/*   printf("Calculating bounds\n"); */
   volht = linht = cdht = dspht = micht = qht = font->max_bounds.ascent + font->max_bounds.descent;
   bw = font->max_bounds.width+2;
   sw = width - bw;
   
   numdev = 1;
   if (devmask & SOUND_MASK_VOLUME)
     numdev++;
   if (devmask & SOUND_MASK_LINE)
     numdev++;
   if (devmask & SOUND_MASK_PCM)
     numdev++;
   if (devmask & SOUND_MASK_CD)
     numdev++;
   if (devmask & SOUND_MASK_MIC)
     numdev++;


/*   printf("Calcing d, height=%d, voly=%d, numdev=%d\n",height,voly,numdev);  */
   d = (height - voly - 12)/(numdev-1);
   
   voly = volht;
   if (devmask & SOUND_MASK_LINE)
     liny = voly + d;
   else
     liny = voly;

   if (devmask & SOUND_MASK_PCM)
     dspy = liny + d;
   else
     dspy = liny;
	
   if (devmask & SOUND_MASK_CD)
     cdy = dspy + d;
   else
     cdy = dspy;
    
   if (devmask & SOUND_MASK_MIC)
     micy = cdy + d;
   else
     micy = cdy;
    
   qy = micy + d;

/*   printf("Setting colors\n"); */
   /* Colours we will use */
   topgrey.red = topgrey.blue = topgrey.green = 40000;
   botgrey.red = botgrey.blue = botgrey.green = 10000;
   grey.red = grey.blue = grey.green = 18724;
   tick.red = tick.blue = tick.green = 50000;
   texta.red = texta.green = 65535;
   texta.blue = 10000;
   textb.red = 50000;
   textb.blue = 50000;
   textb.green = 50000;
   in.red = 5000;
   in.green = 60000;
   in.blue = 20000;
   XAllocColor(disp, cmap, &topgrey);
   XAllocColor(disp, cmap, &botgrey);
   XAllocColor(disp, cmap, &grey);
   XAllocColor(disp, cmap, &in);
   XAllocColor(disp, cmap, &tick);
   XAllocColor(disp, cmap, &texta);
   XAllocColor(disp, cmap, &textb);
   
   /* Create main window */
   mask = CWOverrideRedirect | CWBackPixel | CWCursor;
   attributes.override_redirect = override;
   attributes.background_pixel = grey.pixel;
   
   /*	attributes.cursor = XCreateFontCursor(disp, XC_top_left_arrow);*/
   attributes.cursor = XCreateFontCursor(disp, XC_left_ptr);
   win = XCreateWindow(disp, root, x, y, width, height, 0,
		       CopyFromParent, CopyFromParent, CopyFromParent,
		       mask, &attributes);
   
   /* Set the size hints for the window */
   XSetWMNormalHints(disp, win, &hints);
   
   /* Set the name of the app for the WM */
   prop.value = "Emix";
   prop.encoding = XA_STRING;
   prop.format = 8;
   prop.nitems = strlen(prop.value);
   XSetWMName(disp, win, &prop);
   XMapWindow(disp, win);

   /* Create GCs */
   mask = GCForeground;
   vals.foreground = topgrey.pixel;
   topshad = XCreateGC(disp, win, mask, &vals);
   
   vals.foreground = botgrey.pixel;
   botshad = XCreateGC(disp, win, mask, &vals);

   vals.foreground = tick.pixel;
   tickgc = XCreateGC(disp, win, mask, &vals);

   vals.foreground = in.pixel;
   ingc = XCreateGC(disp, win, mask, &vals);
   vals.foreground = grey.pixel;
   greygc = XCreateGC(disp, win, mask, &vals);

   mask |= GCFont;
   vals.foreground = texta.pixel;
   vals.font = font->fid;
   textagc = XCreateGC(disp, win, mask, &vals);

   vals.foreground = textb.pixel;
   vals.font = font->fid;
   textbgc = XCreateGC(disp, win, mask, &vals);

   XSelectInput(disp, win, ButtonPressMask|ButtonReleaseMask|
		KeyReleaseMask|PointerMotionMask|EnterWindowMask|
                ExposureMask);
   
   swidth = sw*8/10;
   sheight = height/10;

   DoBevels(win, 0, 0, width, height, topshad, botshad);
   DrawTicks();

   GetVolume();
   DrawSlider();

   /* Main event loop */
   for(;;) {
      XNextEvent(disp, &event);
      switch(event.type)
        {
	case EnterNotify:
	   /* Update mixer vals */
	   GetVolume();
	   DrawSlider();
	   break;
        case KeyRelease:
	  switch (event.xkey.keycode) {
	  case 24: // q
	    exit(0);
	    break;
	  case 46: // l
	    device = SOUND_MASK_LINE;
	    break;
	  case 55: // v
	    device = SOUND_MASK_VOLUME;
	    break;
	  case 58: // m
	    device = SOUND_MASK_MIC;
	    break;
	  }
	  fprintf(stderr, "keyrelease keycode=%d device=%d\n", event.xkey.keycode, device);
	  goto update_device;
	  break;
	case ButtonPress:
	   /* Pointer button pressed */
	   /* Ctrl-Button1 moves window */
	   if(event.xbutton.button == Button1 &&
	      event.xbutton.state == ControlMask)
	     {
                XQueryPointer(disp, win, &rr,
			      &cr, &rx, &ry, &px, &py, &mr);
                XRaiseWindow(disp, win);
                moving = True; 
	     }
	   /* Shift-button1 quits */
	   else if(event.xbutton.button == Button1 &&
		   event.xbutton.state == ShiftMask)
	     {
                exit(0);
	     }
	   /* Button1 moves slider/buttons */
	   else if (event.xbutton.button == Button1 || event.xbutton.button == Button3)
	     {
		XQueryPointer(disp, win, &rr,&cr, &rx, &ry, &px, &py, &mr);
		/* Select mixer device */
                if (px > sw) {
                    if(py < voly - volht) {
			// nothing
		    } else if(py < voly) {
			if (event.xbutton.button != Button1)
			    break;
			if (devmask & SOUND_MASK_VOLUME)
			    device = SOUND_MASK_VOLUME;
                    } else if (py < liny) {
			if (event.xbutton.button == Button3) {
			    indevice = SOUND_MASK_LINE;
			    ioctl(mixerfd, SOUND_MIXER_WRITE_RECSRC, &indevice);
			}
			else
			    if (devmask & SOUND_MASK_LINE)
				device = SOUND_MASK_LINE;
                    } else if (py < dspy) {
			if (event.xbutton.button != Button1)
			    break;
			if (devmask & SOUND_MASK_PCM)
			    device = SOUND_MASK_PCM;
                    } else if (py < cdy) {
			if (devmask & SOUND_MASK_CD)
			    device = SOUND_MASK_CD;
                    } else if (py < micy) {
			if (devmask & SOUND_MASK_MIC)
			    device = SOUND_MASK_MIC;
                    } else if (py < qy) {
			exit(0);
                    }
		update_device: 
		    DrawButtons();
		    GetVolume();
		    DrawSlider();
		    break;
		}
		
		/* Begin slider movement */
		sliding = True;
		newh = height - py - 2;
		volume = newh*100/height;
		SetVolume();
		DrawSlider();
	     }
	   break;
	   /* Button-release stops stuff happening */
	 case ButtonRelease:
	   if(moving) {
	      moving = False;
	      mask = CWCursor;
	      attributes.cursor = 0;
	      XChangeWindowAttributes(disp, win, mask, &attributes);
	   }
	   if(sliding) {
	      sliding = False;
	      GetVolume();
	      DrawSlider();
	   }
	   break;
	   /* Handle moving of mouse with button1 events */
	 case MotionNotify:
	   if(moving) {
	      while(XCheckTypedEvent(disp, MotionNotify, &event));
	      XQueryPointer(disp,root,&rr,&cr,&rx,&ry,&wx,&wy,&mr);
	      dosizecursor(wx-px, wy-py, '+', &texta);
	      XMoveWindow(disp, win, wx-px, wy-py);
	   } else if (sliding) {
	      while(XCheckTypedEvent(disp, MotionNotify, &event));
	      XQueryPointer(disp,win,&rr,&cr,&rx,&ry,&wx,&wy,&mr);
	      newh = height - wy - 2;
	      volume = newh*100/height;
	      if (volume < 0) volume = 0;
	      if (volume > 100) volume = 100;
	      SetVolume();
	      DrawSlider();
	   }
	   break;
	   /* redraw the window on exposure */
	 case Expose:
	   GetVolume();
	   DrawSlider();
	   DrawTicks();
	   DoBevels(win, 0, 0, width, height, topshad, botshad);
	   break;
        }
      
   }
}


/* Draw bevels */
void DoBevels(Window win, int x, int y, int w, int h, GC top, GC bot) 
{
   XPoint points1[3];
   XPoint points2[3];
   
   /* Window bevel coordinates */
   points1[0].x = points1[1].x = x;
   points1[1].y = points1[2].y = y;
   points1[0].y = y+h-1;
   points1[2].x = x+w-1;
   
   points2[0].x = x;
   points2[0].y = points2[1].y = y+h-1;
   points2[1].x = points2[2].x = x+w-1;
   points2[2].y = y+1;
   XDrawLines(disp, win, top, points1, 3, CoordModeOrigin);
   XDrawLines(disp, win, bot, points2, 3, CoordModeOrigin);
}


void OpenMixer() {
   mixerfd = open(mixer, O_RDWR);
   if (mixerfd == -1)
     {
	perror("Can't open mixer device");
        exit(1);
     }
   ioctl(mixerfd, MIXER_READ(SOUND_MIXER_DEVMASK), &devmask);
}


/* retrive current mixer volume */
void GetVolume()
{
   
   int vol;
   int res;

   switch(device) {
    case SOUND_MASK_VOLUME:
      res = ioctl(mixerfd, MIXER_READ(SOUND_MIXER_VOLUME), &vol);
      break;
    case SOUND_MASK_LINE:
      res = ioctl(mixerfd, MIXER_READ(SOUND_MIXER_LINE), &vol);
      break;
    case SOUND_MASK_PCM:
      res = ioctl(mixerfd, MIXER_READ(SOUND_MIXER_PCM), &vol);
      break;
    case SOUND_MASK_CD:
      res = ioctl(mixerfd, MIXER_READ(SOUND_MIXER_CD), &vol);
      break;
   case SOUND_MASK_MIC:
     res = ioctl(mixerfd, MIXER_READ(SOUND_MIXER_MIC), &vol);
     break;
   }
   
   if (res==-1) {
      /* An undefined mixer channel was accessed - mark it*/
      unavail |= device;
   }

   /* Get current input source */
   res = ioctl(mixerfd, MIXER_READ(SOUND_MIXER_RECSRC), &indevice);
   volume = vol*100/32768;
}


void DrawSlider()
{
   int sx, sy;

   sx = sw/10 + 1;
   sy = height - sheight - ((height-sheight)*volume/71);
   if (sy < 2)
     sy=2;
   if (sy > height-sheight-2)
     sy=height-sheight-2;
   /* Vertical groove plus clear area for slider */
   XClearArea(disp, win, sx, 1, swidth, height-2, False);
   DrawButtons();
   XDrawLine(disp, win, botshad, sw/2-1, sheight/2, sw/2-1, height-sheight/2);
   XDrawLine(disp, win, topshad, sw/2, sheight/2, sw/2, height-sheight/2);
   XFillRectangle(disp, win, tickgc, sx, sy, swidth, sheight);
   /* Slider outer bevels */
   if(sliding)
     DoBevels(win, sx, sy, swidth, sheight, botshad, topshad);
   else
     DoBevels(win, sx, sy, swidth, sheight, topshad, botshad);
   /* Thumb grip */
   XDrawLine(disp, win, botshad, 7, sy+(sheight/2), sw-8, sy+(sheight/2));
   XDrawLine(disp, win, topshad, 7, sy+(sheight/2)-1, sw-8, sy+(sheight/2)-1);
   
   XSync(disp, False);
}

/* Set mixer volume then retrive the actual resultant setting */
void SetVolume()
{
   int vol;
   int res;
   unsigned char v;
   v = volume;
   vol = volume;
   vol = vol << 8;
   vol += v;

   switch(device) {
    case SOUND_MASK_VOLUME:
      res = ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_VOLUME), &vol);
      break;
    case SOUND_MASK_LINE:
      res = ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_LINE), &vol);
      break;
    case SOUND_MASK_PCM:
      res = ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_PCM), &vol);
      break;
    case SOUND_MASK_CD:
      res = ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_CD), &vol);
      break;
   case SOUND_MASK_MIC:
     res = ioctl(mixerfd, MIXER_WRITE(SOUND_MIXER_MIC), &vol);
     break;
   }

   if (res==-1)
     {
	/* An undefined mixer channel was accessed */
	unavail |= device;
     }
   GetVolume();
}

/* Draws slider tick marks */
void DrawTicks()
{
   int tx;
   int tw = sw/10;
   for (tx = height-sheight/2 ; tx>0 ; tx -= (height-sheight/2)/10) 
     {
	
        XDrawLine(disp, win, tickgc, 1, tx, tw, tx);
        XDrawLine(disp, win, tickgc, sw-tw, tx, sw-1, tx);
     }
}

/* Draw text buttons for mixer selection */
void DrawButtons()
{
   GC s1,s2,s3,s4,s5,s6;
   int x=sw+4;
   GC i1, i2;

   i1 = i2 = greygc;
   switch(device) {
    case SOUND_MASK_VOLUME:
      s1 = textagc;
      s2 = s3 = s4 = s5 = textbgc;
      break;
    case SOUND_MASK_LINE:
      s2 = textagc;
      s1 = s3 = s4 = s5 = textbgc;
      break;
    case SOUND_MASK_PCM:
      s3 = textagc;
      s1 = s2 = s4 = s5 = textbgc;
      break;
    case SOUND_MASK_CD:
      s4 = textagc;
      s1 = s2 = s3 = s5 = textbgc;
      break;
   case SOUND_MASK_MIC:
     s5 = textagc;
     s1 = s2 = s3 = s4 = textbgc;
     break;
   }
   s6 = textbgc;
   
   if (indevice & SOUND_MASK_LINE)
     i1 = ingc;
   if (indevice & SOUND_MASK_CD)
     i2 = ingc;
   
   XDrawLine(disp, win, botshad, sw+1, 1, sw+1, height-2);
   XDrawLine(disp, win, topshad, sw+2, 1, sw+2, height-2);
   if (devmask & SOUND_MASK_VOLUME)
     XDrawString(disp, win, s1, x, voly, VOLTEXT, strlen(VOLTEXT));
   if (devmask & SOUND_MASK_LINE)
     XDrawString(disp, win, s2, x, liny, LINTEXT, strlen(LINTEXT));
   if (devmask & SOUND_MASK_PCM)
     XDrawString(disp, win, s3, x, dspy, DSPTEXT, strlen(DSPTEXT));
   if (devmask & SOUND_MASK_CD)
     XDrawString(disp, win, s4, x, cdy, CDTEXT, strlen(CDTEXT));
   if (devmask & SOUND_MASK_MIC)
     XDrawString(disp, win, s5, x, micy, MICTEXT, strlen(MICTEXT));
   
   XDrawString(disp, win, s5, x, qy, QTEXT, strlen(QTEXT));
   XDrawLine(disp, win, i1, x, liny+2, width-3, liny+2);
   XDrawLine(disp, win, i2, x, cdy+2, width-3, cdy+2);
}


/* Set the cursor to the toplevel window to a string of
 * the form "wseph" eg "25x50". Cursor colour is given
 * by 'col'.
 */
void dosizecursor(int w, int h, char sep, XColor *col) 
{
   
   int width,height;
   Pixmap pix, mask;
   GC gc;
   XGCValues gcv;
   XSetWindowAttributes wattr;
   unsigned long msk;
   char geom[15];
   XPoint points[3];
   
   if (nogeom)
     return;
   sprintf(geom, "%d%c%d", w, sep, h);
   
   /* Obtain required geometryies and create 2 suitably-sized pixmaps of
    * depth 1, one for the cursor, the other for the mask.
    */
   width = XTextWidth(font, geom, strlen(geom)) +7;
   height = font->max_bounds.ascent + font->max_bounds.descent +2;
   pix = XCreatePixmap(disp, root, width, height, 1);
   mask = XCreatePixmap(disp, root, width, height, 1);

   /* Create the GC with foreground 0 and suitable font */
   msk = GCForeground | GCFont;
   gcv.foreground = 0;
   gcv.font = font->fid;
   gc = XCreateGC(disp, pix, msk, &gcv);
   
   /* Fill both pixmaps with 0 */
   XFillRectangle(disp, mask, gc, 0, 0, width, height);
   XFillRectangle(disp, pix, gc, 0, 0, width, height);
   
   /* Chenge the foreground to '1' and draw the string */
   XSetForeground(disp, gc, 1);
   points[0].x = points[0].y = 0;
   points[1].y = points[2].x = 2;
   points[1].x = points[2].y = 5;
   XFillPolygon(disp, mask, gc, points, 3, Convex, CoordModeOrigin);
   XDrawString(disp, mask, gc, 6, height-1, geom, strlen(geom));

   /* Generate the cursor from the pixmaps and set it for the window */
   wattr.cursor = XCreatePixmapCursor(disp, pix, mask, col, col, 0, 0);
   msk = CWCursor;
   XChangeWindowAttributes(disp, win, msk, &wattr);
   /* Free the resources */
   XFreePixmap(disp, pix);
   XFreePixmap(disp, mask);
   XFreeGC(disp, gc);
}
