/*
 * loader.c - load and execute an elf executable
 *
 * Copyright (C) 1999 Steve Hill <sjhill@plutonium.net>
 * Copyright (C) 1999 Bradley D. LaRonde <brad@ltc.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software Foundation,
 * Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 */
/*
 *  MODIFICATIONS: 
 *  11-Dec-2000  - added a function to return cpu specific settings
 *
 *               - removed some member functions that weren't 
 *                 needed because of above.
 *
 *               Dirk <dirk.vanhennekeler@compaq.com>
 */


#include <windows.h>
#include <commctrl.h>
#include "../pbsdboot/pbsdboot.h"
#include "bdl/misc.h"
#include "bdl/system.h"
#include "bdl/vector.h"
#include "bdl/dialog.h"
#include "resource.h"
#include "bdl/progress.h"
#include "../lib/config.h"
#include "loader.h"
#include "ASystemInfo.h"

extern "C" startprog_nec_vr41xx(caddr_t map);
extern "C" startprog_philips_pr31700(caddr_t map);
extern "C" int startprog_intel_sa11xx(caddr_t map);

const unsigned char PRID_NEC_VR41XX = 0x0c;
const unsigned char PRID_PHILIPS_PR31700 = 0x22;

CPUSETTINGS cpusettings[] = {
#if defined(MIPS)
	{PRID_NEC_VR41XX,		0x80000000, startprog_nec_vr41xx,		TRUE	},
	{PRID_PHILIPS_PR31700,	0x44000000, startprog_philips_pr31700,	FALSE	},
#elif defined(ARM)
	{PROCESSOR_STRONGARM ,	0xc0000000, startprog_intel_sa11xx,		TRUE	},
#endif
	{0,						0,			0,							FALSE	}, // end marker
};

class PbsdbootProgressCallbackAdapter : public ProgressCallbackAdapter
{
private:
	static unsigned int m_nCalls;
	static unsigned int m_nSkip;

public:
	PbsdbootProgressCallbackAdapter(ProgressDialog* pProgressDialog)
		: ProgressCallbackAdapter(pProgressDialog)
	{
	}

	static void SetSkip(unsigned int nSkip)
	{
		m_nSkip = nSkip;
	}

	static BOOL CheckCancel(int nProgress)
	{
		// only update every so many calls
		// this speeds things up a little bit
		if ( PbsdbootProgressCallbackAdapter::m_nCalls++ % m_nSkip == 0 )
			return Update(nProgress);
		else
			return ProgressCallbackAdapter::m_pProgressDialog->ShouldCancel();
	}
};

unsigned int PbsdbootProgressCallbackAdapter::m_nCalls = 0;
unsigned int PbsdbootProgressCallbackAdapter::m_nSkip = 1;

CPUSETTINGS* Loader::CPUSettings() const
{
	CPUSETTINGS *		pCPUSettings = cpusettings;
	ASystemInfo			si;

	while (pCPUSettings->pfnStartProg != 0)
	{
		if ( (si.wProcessorArchitecture == PROCESSOR_ARCHITECTURE_MIPS) &&
			 ((si.wProcessorRevision >> 8) == (unsigned char)pCPUSettings->nPID) )
		{
			break; 
		}
		else if ( (si.wProcessorArchitecture == PROCESSOR_ARCHITECTURE_ARM) &&
				  (si.dwProcessorType == pCPUSettings->nPID) )
		{
			break; 
		}

		pCPUSettings++;
	}

	return pCPUSettings;
}

BOOL Loader::GetProcessorSpecificSettings(unsigned int& pps, startprog_t& psp) const
{
	CPUSETTINGS *		pCPUSettings = CPUSettings();

	if (pCPUSettings->pfnStartProg)
	{
		pps = pCPUSettings->ppStartAddr;
		psp = pCPUSettings->pfnStartProg;
		return TRUE; 
	}

	return FALSE;
}

void Loader::Load(ImageSection* pSection, LPCTSTR pszParameters)
{
	TCHAR msg[512];

	// get the processor specific settings
	BOOL bResult = GetProcessorSpecificSettings(phys_start, startprog);
	if ( !bResult )
	{
		_stprintf(msg, _T("Unsupported processor revision 0x%.4x"),
			ASystemInfo().wProcessorRevision);
		System::ErrorMessageBox(msg);
		return;
	}

	// don't load debug_info
	pref.load_debug_info = 0;

	// figure the kernel image file name
	TCHAR szImage[MAX_PATH];
	System::AbsolutePath(szImage, pSection->GetImage());

	// Open the kernel file for reading.
	int fd = open(szImage, O_RDONLY);
	if (fd == -1)
	{
		_stprintf(msg, _T("Error %d opening kernel file %s."),
			GetLastError(), szImage);
		System::ErrorMessageBox(msg);
		return;
	}

	// open the serial port to ensure transceiver is enabled
	// TODO: make port and message a config option, no port = no open
	const TCHAR szSerialPort[] = _T("COM1:");
	HANDLE hSerial = CreateFile(szSerialPort, GENERIC_READ | GENERIC_WRITE,
		0, 0, OPEN_EXISTING, 0, 0);
	if ( hSerial == INVALID_HANDLE_VALUE )
	{
		_stprintf(msg, _T("Error %d opening serial port %s."),
			GetLastError(), szSerialPort);
		System::ErrorMessageBox(msg);
	}
	else
	{
		debug_printf(_T("Serial port opened.\n"));
		char szMessage[256];
		wcstombs(szMessage, System::GetApplicationName(), MAX_CHARS(szMessage));
		strcat(szMessage, "\r\n");
		DWORD dwActual;
		WriteFile(hSerial, szMessage, strlen(szMessage), &dwActual, 0);
		FlushFileBuffers(hSerial);
	}

	// display the boot progress dialog
	ProgressDialog progress;
	progress.SetCaption(System::GetApplicationName());
	HWND hwndProgress = progress.Create(System::GetModuleInstance(), 0);
	if ( hwndProgress == 0 )
	{
		_stprintf(msg, _T("Error %d creating progress dialog."), GetLastError());
		System::ErrorMessageBox(msg);
		// keep going anyway
	}

	PbsdbootProgressCallbackAdapter pca(&progress);
	CheckCancel = pca.CheckCancel;

	_stprintf(msg, _T("Loading %s %s..."), pSection->GetLabel(), pszParameters);
	progress.Update(0, _T(""), msg);

	// convert append
	char szAppend[2048];
	wcstombs(szAppend, pSection->GetAppend(), MAX_CHARS(szAppend));

	// convert parameters
	char szParameters[2048];
	wcstombs(szParameters, pszParameters, MAX_CHARS(szParameters));

	// merge them
	strcat(szParameters, " ");
	strcat(szParameters, szAppend);

	// figure program name for first arg
	char szProgram[MAX_PATH];
	wcstombs(szProgram, System::GetModuleFileName(), MAX_CHARS(szProgram));

	// hold arguments
	char *argv[256];
	int argc = 0;

	// first arg is the program name
	argv[argc++] = szProgram;

	// parse pszParameters into standard argc/argv
	// only to cat them back together again in prom_init,
	// then split them back apart again in parse_options - lol - bdl
	char *parg = strtok(szParameters, " ");
	while ( parg )
	{
		argv[argc++] = parg;
		parg = strtok(0, " ");
	}

	// last arg is null
	argv[argc] = 0;

	progress.Update(0, _T("Figuring start and end addresses..."));

	// Get the start and end address of the kernel memory.
	caddr_t start, end;
	int nResult = getinfo(fd, &start, &end);
	if ( nResult < 0)
	{
		_stprintf(msg, _T("Error %d getting elf file info for %s."),
			GetLastError(), szImage);
		System::ErrorMessageBox(msg);
		return;
	}

	progress.Update(0, _T("Initializing memory..."));
	pca.SetSkip(40);

	// Initialize the virtual memory for the kernel pages.
	nResult = vmem_init(start, end);
	if ( nResult < 0)
		return;

	progress.Update(0, _T("Allocating memory for arguments..."));

	// Allocate memory for kernel arguments.
	caddr_t argbuf = vmem_alloc();
	if ( argbuf == 0 )
	{
		_stprintf(msg, _T("Error %d allocating memory for kernel arguments."),
			GetLastError());
		System::ErrorMessageBox(msg);
		vmem_free();
		return;
	}

	progress.Update(0, _T("Allocating memory for bootinfo struct..."));

	// Allocate memory for kernel arguments.
	struct bootinfo* bibuf = (struct bootinfo*)vmem_alloc();
	if ( bibuf == 0 )
	{
		_stprintf(msg, _T("Error %d allocating memory for bootinfo struct."),
			GetLastError());
		System::ErrorMessageBox(msg);
		vmem_free();
		return;
	}

	progress.Update(0, _T("Copying arguments..."));

	// Copy kernel arguments into newly allocated memory.
	// skip past the argv[] array for the argument storage
	caddr_t p = &argbuf[sizeof(char *)* argc];
	int i;
	for (i = 0; i < argc; i++)
	{
		int arglen = strlen(argv[i]) + 1;
		((char **) argbuf)[i] = p;
		memcpy (p, argv[i], arglen);
		p += arglen;
	}

	progress.Update(0, _T("Loading the kernel file..."));
	pca.SetSkip(3);

	// Load the kernel file into memory.
	nResult = loadfile(fd, &start);
	if ( nResult < 0)
	{
		if ( !progress.ShouldCancel() )
		{
			_stprintf(msg, _T("Error %d loading elf file %s."),
				GetLastError(), szImage);
			System::ErrorMessageBox(msg);
		}
		vmem_free();
		return;
	}

	progress.Update(0, _T("Preparing and executing kernel..."));

	// Go execute initialization code and boot the kernel.
	// if it works, it won't return
	nResult = vmem_exec(start, argc, (char **)argbuf, bibuf);

	// guess it didn't work
	_stprintf(msg, _T("Error %d, result %d executing kernel."),
		GetLastError(), nResult);
	System::ErrorMessageBox(msg);
	vmem_free();
}


class LoadDialog : public ModalDialog
{
private:
	int m_nTimeout;
	int m_nCountdown;
	TCHAR m_szLabels[1024];
	TCHAR m_szCommand[1024];

	void ShowCountdown()
	{
		TCHAR szMsg[512];
		_stprintf(szMsg,
			_T("This program will automatically execute the load")
			_T(" command in %d seconds. Press Esc to cancel the countdown."),
			m_nCountdown);
		SetItemText(IDC_MESSAGE, szMsg);

		// set the progress position
		SendItemMessage(IDC_PROGRESS, PBM_SETPOS, (m_nCountdown * 100) / m_nTimeout, 0);
	}

	void ShowReady()
	{
		SetItemText(IDC_MESSAGE, _T("Ready for load command."));
		::ShowWindow(GetItem(IDC_PROGRESS), SW_HIDE);
	}

	BOOL OnInitDialog()
	{
		if ( m_nTimeout > 0 )
		{
			m_nCountdown = m_nTimeout;
			ShowCountdown();
			SetTimer(m_hwndDlg, 1, 1000, 0);
		}
		else
			ShowReady();

		SetItemText(IDC_LABELS, m_szLabels);

		SetItemText(IDC_COMMAND, m_szCommand);
		return TRUE;
	}

	BOOL OnOK()
	{
		// retrieve the boot command
		GetItemText(IDC_COMMAND, m_szCommand, MAX_CHARS(m_szCommand));

		return ModalDialog::OnOK();
	}

	BOOL OnTimer(int nTimer, TIMERPROC* pTimerproc)
	{
		if ( nTimer == 1 )
		{
			if ( --m_nCountdown == 0 )
			{
				OnOK();
				return TRUE;
			}
			
			ShowCountdown();
			return TRUE;
		}

		return FALSE;
	}

	BOOL OnCancel()
	{
		if ( m_nCountdown > 0 )
		{
			KillTimer(m_hwndDlg, 1);
			m_nCountdown = 0;
			ShowReady();
			return TRUE;
		}

		return ModalDialog::OnCancel();
	}

public:
	LoadDialog() :
		ModalDialog(MAKEINTRESOURCE(IDD_LOAD))
	{
	}

	void SetTimeout(int n)
	{
		m_nTimeout = n;
	}

	void SetLabels(LPCTSTR psz)
	{
		_tcscpy(m_szLabels, psz);
	}
	
	void SetCommand(LPCTSTR psz)
	{
		_tcscpy(m_szCommand, psz);
	}

	LPCTSTR GetCommand()
	{
		return m_szCommand;
	}
};

void Loader::GetSystemProcessorInfo(LPTSTR psz)
{
	ASystemInfo sysinfo;

	_stprintf(psz, _T("arch %d, type %d, level 0x%.4x, rev 0x%.4x"),
		sysinfo.wProcessorArchitecture, sysinfo.dwProcessorType,
		sysinfo.wProcessorLevel, sysinfo.wProcessorRevision);
}

BOOL Loader::VerifyProcessorSupport()
{
	CPUSETTINGS *		pCPUSettings = CPUSettings();

	if (pCPUSettings->bSupported == FALSE)
	{
		TCHAR info[256];
		GetSystemProcessorInfo(info);
		TCHAR msg[256];
		_stprintf(msg, _T("Unsupported processor - %s. Continue anyway?"), info);
		int nResult = System::YesNoMessageBox(msg);
		if ( nResult != IDYES )
			return FALSE;
	}

	return TRUE;
}

int Loader::Main()
{
	TCHAR szMsg[MAX_PATH * 2];

	m_nVerboseLevel = 0;

	if ( m_nVerboseLevel > 0 )
	{
		TCHAR msg[256];
		GetSystemProcessorInfo(msg);
		System::InfoMessageBox(msg);
	}

	// let them know up front
	if ( !VerifyProcessorSupport() )
		return 1;

	// progress bar needs this
	InitCommonControls();

	// figure the absolute config file name
	TCHAR szConfigFile[MAX_PATH];
	System::AbsolutePath(szConfigFile, _T("cyacecfg.txt"));

	// load the boot configuration file
	BootConfig config;
	BOOL bResult = config.Load(szConfigFile);
	if ( !bResult ) {
		_stprintf(szMsg, _T("Error %d loading boot configuration file %s."),
			GetLastError(), szConfigFile);
		System::ErrorMessageBox(szMsg);
		return 1;
	}

	// no image sections?
	if ( config.GetImageSections()->Size() == 0 ) {
		_stprintf(szMsg, _T("Error - no image sections in configuration file %s."),
			szConfigFile);
		System::ErrorMessageBox(szMsg);
		return 1;
	}

	// get the default image section
	ImageSection* pDefaultSection = config.GetImageSections()->At(0);
	if ( pDefaultSection == 0 ) {
		_stprintf(szMsg, _T("Error - no image sections defined in configuration file %s."),
			szConfigFile);
		System::ErrorMessageBox(szMsg);
		return 1;
	}

	// figure the default boot command
	TCHAR szBootCommand[1024];
	_tcscpy(szBootCommand, pDefaultSection->GetLabel());

	// display the boot dialog?
	if ( config.GetTimeout() != 0 )
	{
		LoadDialog dlg;
		dlg.SetCaption(System::GetApplicationName());
		dlg.SetCommand(pDefaultSection->GetLabel());
		dlg.SetTimeout(config.GetTimeout());

		// make a section list
		TCHAR szLabels[1024];
		_tcscpy(szLabels, _T("Available labels: "));
		int i;
		for ( i = 0; i < config.GetImageSections()->Size(); i++ )
		{
			if ( i > 0 )
				_tcscat(szLabels, _T(", "));
			_tcscat(szLabels, config.GetImageSections()->At(i)->GetLabel());
		}
		dlg.SetLabels(szLabels);

		int nResult = dlg.Do(System::GetModuleInstance(), 0);
		if ( nResult != IDOK )
			return 1;

		// get the specified boot command
		_tcscpy(szBootCommand, dlg.GetCommand());
	}

	// split off the section label
	TCHAR szLabel[256];
	TCHAR szParameters[2048];
	_stscanf(szBootCommand, _T("%s %[^\0]"), szLabel, szParameters);

	// find the specified section
	ImageSection* pSection = 0;
	int i;
	for ( i = 0; i < config.GetImageSections()->Size(); i++ )
	{
		if ( _tcsicmp(szLabel, config.GetImageSections()->At(i)->GetLabel()) == 0 )
		{
			pSection = config.GetImageSections()->At(i);
			break;
		}
	}

	// didn't find the section?
	if ( pSection == 0 ) {
		_stprintf(szMsg, _T("Error - can't find section %s in configuration file %s."),
			szLabel, szConfigFile);
		System::ErrorMessageBox(szMsg);
		return 1;
	}

	// boot it
	Load(pSection, szParameters);

	// getting here means that it failed

	return 1;
}
