/*
 *  mbpanel
 *
 *  A 'system tray' - Matthew Allum <mallum@handhelds.org>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 *
 */


#include "mbmenu.h"
#include "mbutil.h"

#include "dock.h"
#include "msg.h"

MBPanel *G_panel = NULL; 

#include "scrollbutton.xpm"
#include "menubutt.xpm"

/* Scroll based funcs */

static 
void panel_paint_scroller(MBPanel *panel, int state)
{
  XCopyArea(panel->dpy, panel->scroller_pxm,
	    panel->win_scroll_button, panel->gc, 
	    0, 0, 16, 16, 0, 2 );
  
  if (state)
    {
      XSetFunction(panel->dpy, panel->gc, GXxor);
      if (state == LEFT)
	XFillRectangle(panel->dpy, panel->win_scroll_button, panel->gc, 0,2,8,16);
      if (state == RIGHT)
	XFillRectangle(panel->dpy, panel->win_scroll_button, panel->gc, 8,2,8,16);
      XSetFunction(panel->dpy, panel->gc, GXcopy);
    }
}

void 
panel_activate_scroll(MBPanel *d)
{
  DBG("scroll button activated\n");
  d->scroll_active = 1;
  XMoveWindow(d->dpy, d->win_scroll_button, d->w - SCROLL_WIN_WIDTH, 0);
  XMapRaised(d->dpy, d->win_scroll_button);
}

void 
panel_deactivate_scroll(MBPanel *d)
{
  DBG("scroll button activated\n");
  d->scroll_active = 0;
  XUnmapWindow(d->dpy, d->win_scroll_button);
}

void 
panel_scroll(MBPanel *panel, int dir, MBPanelApp *ignore)
{
  MBPanelApp *p;
  Bool client_off_right = 0;

 if (dir == LEFT) 
   {
     if (panel->scroll_offset <= 0)
       { panel->scroll_offset += 2; }
     else
       { return; }
   }

 if (dir == RIGHT )
   { 
     /* Check there are actually some icons of the right of the screen */
     for (p=panel->panel_app_head; p != NULL; p = p->next)
       {
	 if (ignore != NULL && p == ignore)
	   continue;

	 if ( (p->x+p->w) > (panel->w - SCROLL_WIN_WIDTH) )
	   {
	     client_off_right = 1;
	     break;
	   }
       }
     
     if (!client_off_right) return;
     
     panel->scroll_offset -= 2; 
   }
 

  for (p=panel->panel_app_head; p != NULL; p = p->next)
    {
      if (ignore != NULL && p == ignore)
	continue;

      if (dir == LEFT)
	{ p->x += 2; }
      if (dir == RIGHT)
	{ p->x -= 2; }

      panel_app_move_to(panel, p, p->x);

    }
  XMapRaised(panel->dpy, panel->win_scroll_button);
}

void 
panel_scroll_update_visibility(MBPanel *panel)
{
  MBPanelApp *papp = panel_app_list_get_last(panel); /* XXX panel->tail be
						            faster !      */
  if (papp)
    {
      if ((papp->x + papp->w) > (panel->w + SCROLL_ACTIVATE_SENSITIVITY))
	{
	  if (!panel->scroll_active)
	    {
	      XLowerWindow(panel->dpy, papp->win);
	      panel_activate_scroll(panel);
	    }
	} else {
	  if (panel->scroll_active)
	    panel_deactivate_scroll(panel);
	}
    }
}

/* misc  */

void
panel_toggle_visibilty(MBPanel *d)
{

#define PANEL_HIDDEN_SIZE 6 
  DBG("%s() called, x: %i, y: %i, w: %i, h: %i\n", __func__,
      d->x, d->y, d->w, d->h);

  if (d->is_hidden)
    {
      if (PANEL_IS_VERTICAL(d))
	{
	  XMoveResizeWindow(d->dpy, d->win, d->x, d->y, PANEL_SIZE, d->h );
	  XMapSubwindows(d->dpy, d->win);
	} else {
	  XMoveResizeWindow(d->dpy, d->win, d->x, d->y, d->w, PANEL_SIZE);
	  XMapSubwindows(d->dpy, d->win);
	}
      d->is_hidden = False;
    } else {
      if (PANEL_IS_VERTICAL(d))
	{
	  XMoveResizeWindow(d->dpy, d->win, d->x, d->y, 
			    PANEL_HIDDEN_SIZE, d->h );
	  XMapSubwindows(d->dpy, d->win);
	} else {
	  XMoveResizeWindow(d->dpy, d->win, d->x, d->y, 
			    d->w, PANEL_HIDDEN_SIZE);
	  XMapSubwindows(d->dpy, d->win);
	}
      d->is_hidden = True;
    }
}

static void panel_paint_menubutt(MBPanel *panel, int state)
{
  XGCValues gc_vals;
  
  XClearWindow(panel->dpy, panel->win_menu_button);
  gc_vals.clip_x_origin = 2;
  gc_vals.clip_y_origin = 2 + state;
  XChangeGC(panel->dpy, panel->gc,
	    GCClipXOrigin|GCClipYOrigin, &gc_vals);

  XSetClipMask(panel->dpy, panel->gc, panel->menubutt_mask);
  XCopyArea(panel->dpy, panel->menubutt_pxm,
	    panel->win_menu_button, panel->gc, 
	    0, 0, 6, 8, 2, 2+state );
  
  XSetClipMask(panel->dpy, panel->gc, None);
}


void
panel_set_bg(MBPanel *panel, int bg_type, char *bg_spec)
{
  MBPixbufImage *img_tmp, *img_bg;
  int dx, dy, dw, dh;

  Pixmap tmp_pxm;
  char xprop_def[32] = { 0 };
   
  switch (bg_type)
    {
    case BG_PIXMAP:
      if ((img_tmp = mb_pixbuf_img_new_from_file(panel->pb, bg_spec)) == NULL)
	{
	  fprintf(stderr, "mbdock: failed to load %s\n", bg_spec);
	  panel_set_bg(panel, BG_SOLID_COLOR, DEFAULT_COLOR_SPEC);
	  return;
	}
      
      img_bg = mb_pixbuf_img_new(panel->pb, panel->w, panel->h);
      
      for (dy=0; dy < panel->h; dy += img_tmp->height)
	for (dx=0; dx < panel->w; dx += img_tmp->width)
	  {
	    if ( (dx + img_tmp->width) > panel->w )
	      dw = img_tmp->width - ((dx + img_tmp->width)-panel->w);
	    else
	      dw = img_tmp->width;
	    
	    if ( (dy + img_tmp->height) > panel->h )
	      dh = img_tmp->height-((dy + img_tmp->height)-panel->h);
	    else
	      dh = img_tmp->height;
	    mb_pixbuf_img_copy(panel->pb, img_bg, img_tmp,
			       0, 0, dw, dh, dx, dy);
	  }
      
      mb_pixbuf_img_free(panel->pb, img_tmp);
      
      if (panel->bg_pxm != None) XFreePixmap(panel->dpy, panel->bg_pxm);
      
      panel->bg_pxm = XCreatePixmap(panel->dpy, panel->win_root, 
				    panel->w, panel->h,
				    panel->pb->depth ); 
      
      mb_pixbuf_img_render_to_drawable(panel->pb, img_bg, panel->bg_pxm, 0, 0);
      
      mb_pixbuf_img_free(panel->pb, img_bg);
      

      XSetWindowBackgroundPixmap(panel->dpy, panel->win_menu_button, 
				 panel->bg_pxm);
      XClearWindow(panel->dpy, panel->win_menu_button);
      
      XSetWindowBackgroundPixmap(panel->dpy, panel->win, panel->bg_pxm);
      XClearWindow(panel->dpy, panel->win);
      
      snprintf(xprop_def, 32, "pxm:%li", panel->bg_pxm);

      break;
    case BG_SOLID_COLOR:
      if (XParseColor(panel->dpy, 
		      DefaultColormap(panel->dpy, panel->screen),
		      bg_spec,
		      &panel->xcol ))
	{

	  XAllocColor(panel->dpy, 
		      DefaultColormap(panel->dpy, panel->screen),
		      &panel->xcol);

	  XSetWindowBackground(panel->dpy, panel->win, 
			       panel->xcol.pixel);
	  XClearWindow(panel->dpy, panel->win);

	  /* XXX Below needed ? */
	  XSetWindowBackground(panel->dpy, 
			       panel->win_scroll_button,panel->xcol.pixel);
	  XClearWindow(panel->dpy, panel->win_scroll_button);
	  XSetWindowBackground(panel->dpy, 
			       panel->win_menu_button,panel->xcol.pixel);
	  XClearWindow(panel->dpy, panel->win_menu_button);
	  snprintf(xprop_def, 32, "rgb:%li", panel->xcol.pixel);
	}
      break;
    case BG_TRANS:
      tmp_pxm = util_get_root_pixmap(panel);
      if (tmp_pxm != None)
	{
	  int trans_level = 0;

	  if (bg_spec) trans_level = atoi(bg_spec);

	  DBG("%s() Getting root pixmap\n", __func__);

	  if (panel->bg_pxm != None) XFreePixmap(panel->dpy, panel->bg_pxm);

	  panel->bg_pxm = XCreatePixmap(panel->dpy, panel->win_root, 
					panel->w, panel->h,
					panel->pb->depth ); 
	  
	  /* if (trans_level > 0)
	     { */ 
	  img_tmp = mb_pixbuf_img_new_from_drawable(panel->pb, 
						    tmp_pxm, None,
						    panel->x, panel->y, 
						    panel->w, panel->h); 

	  if (img_tmp == NULL)
	    {
	      XFreePixmap(panel->dpy, panel->bg_pxm);
	      fprintf(stderr, "Failed to get root pixmap id\n");
	      panel_set_bg(panel, BG_SOLID_COLOR, DEFAULT_COLOR_SPEC);
	      return;
	    }


	  DBG("%s() First pixel of root looks like %i, %i, %i, %i\n", 
	      __func__, img_tmp->rgba[0], img_tmp->rgba[1],
	      img_tmp->rgba[2], img_tmp->rgba[3]);
	  
	  if (trans_level > 0)
	    for (dx = 0; dx < panel->w; dx++)
	      for (dy = 0; dy < panel->h; dy++)
		mb_pixbuf_img_composite_pixel(img_tmp, dx, dy, 
					      255, 255, 255,
					      trans_level);
	  
	  
	  mb_pixbuf_img_render_to_drawable(panel->pb, img_tmp, 
					   panel->bg_pxm, 0, 0);
	  
	  mb_pixbuf_img_free(panel->pb, img_tmp);
	  
	  
	  XSetWindowBackgroundPixmap(panel->dpy, panel->win, 
				     panel->bg_pxm);
	  XClearWindow(panel->dpy, panel->win);
	  XSetWindowBackgroundPixmap(panel->dpy, panel->win_menu_button, 
				     panel->bg_pxm);
	  XClearWindow(panel->dpy, panel->win_menu_button);
	  
	  snprintf(xprop_def, 32, "pxm:%li", panel->bg_pxm);
	  
	} else {
	  fprintf(stderr, "Failed to get root pixmap id\n");
	  panel_set_bg(panel, BG_SOLID_COLOR, DEFAULT_COLOR_SPEC);
	  return;
	}
      
      break;
    }

  if (xprop_def)
    {
      DBG("setting _MB_PANEL_BG to %s\n", xprop_def);
      
      XChangeProperty(panel->dpy, panel->win, 
		      panel->atoms[ATOM_MB_PANEL_BG],
		      XA_STRING, 8,
		      PropModeReplace, xprop_def, 
		      strlen(xprop_def));
  
      XSync(panel->dpy, False);

    }
}

Bool
panel_set_theme_from_root_prop(MBPanel *panel)
{
  Atom realType;
  unsigned long n;
  unsigned long extra;
  int format;
  int status;
  char * value;
  struct stat stat_info;
  char panel_cfg[256];

  DBG("%s() called\n", __func__);
		
  if ( panel->use_themes == False )
    {
      DBG("%s() panel themeing disabled by command line options\n", __func__ );
      return False;
    }
  
  status = XGetWindowProperty(panel->dpy, panel->win_root,
			      panel->atoms[ATOM_MB_THEME], 
			      0L, 512L, False,
			      AnyPropertyType, &realType,
			      &format, &n, &extra,
			      (unsigned char **) &value);
	    
  if (status != Success || value == 0
      || *value == 0 || n == 0)
    {
      DBG("%s() no _MB_THEME set on root window\n", __func__ );
      return False;
    } else {
      strcpy(panel_cfg, value);
      strcat(panel_cfg, "/theme.desktop");
      if (stat(panel_cfg, &stat_info) != -1)
	{
	  MBDotDesktop *theme  = NULL;
	  theme = mb_dotdesktop_new_from_file(panel_cfg);
	  if (theme)
	    {
	      if (mb_dotdesktop_get(theme, "DockBgColor"))
		{
		  panel_set_bg(panel, BG_SOLID_COLOR, 
			       mb_dotdesktop_get(theme, "DockBgColor"));
		}
	      if (mb_dotdesktop_get(theme, "DockBgTrans"))
		{
		  panel_set_bg(panel, BG_TRANS, 
			       mb_dotdesktop_get(theme, "DockBgTrans")); 
		}
	      if (mb_dotdesktop_get(theme, "DockBgPixmap"))
		{
		  panel_set_bg(panel, BG_PIXMAP, 
			       mb_dotdesktop_get(theme, "DockBgPixmap")); 
		}
	      if (mb_dotdesktop_get(theme, "MenuBgColor"))
		{
		  mb_menu_set_col(panel->mbmenu, BG_COL, 
				  mb_dotdesktop_get(theme, "MenuBgColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuFgColor"))
		{
		  mb_menu_set_col(panel->mbmenu, FG_COL, 
				  mb_dotdesktop_get(theme, "MenuFgColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuHlColor"))
		{
		  mb_menu_set_col(panel->mbmenu, HL_COL, 
				  mb_dotdesktop_get(theme, "MenuHlColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuBdColor"))
		{
		  mb_menu_set_col(panel->mbmenu, BD_COL, 
				  mb_dotdesktop_get(theme, "MenuBdColor"));
		}
	      if (mb_dotdesktop_get(theme, "MenuTransparency"))
		{
		  mb_menu_set_trans(panel->mbmenu,  
				    atoi(mb_dotdesktop_get(theme, 
						      "MenuTransparency")));
		}
	      if (mb_dotdesktop_get(theme, "MenuFont"))
		{
		  mb_menu_set_font(panel->mbmenu, 
				   mb_dotdesktop_get(theme, "MenuFont"));
		}
	      mb_dotdesktop_free(theme);
	      return True;
	    } 
	}
    }
  return False;
}

void 
panel_usage(char *bin_name)
{
  fprintf(stderr, "%s usage: %s [Options...]\n"
	  "Where options are;\n"
          "-display, -d         <X11 Display name>\n"
          "-geometry, -g        <X11 geometry spec>\n"
	  "--id                 <integer>\n"
          "--orientation        <north|east|south|west>\n\n"
          "--no-session, -ns\n"
          "--no-menu, -nm\n"
          "--overide-bubbles, -o\n\n"
	  "Background options:\n"
          "--bgcolor,  -c       <color spec>\n"
	  "--bgpixmap, -b       <image file>\n"
	  "--bgtrans,  -bt      <'yes'|transparency percentage>\n"
	  "*NOTE* setting the background here will disable the effect\n"
	  "       of any external theme changes. \n"
	  , bin_name, bin_name);
  exit(1);
}

/* XEMBED */

int
panel_get_map_state(MBPanel *d, MBPanelApp *c)
{
  Atom realType;
  unsigned long n;
  unsigned long extra;
  int format;
  int status;
  CARD32 * value = NULL;

  status = XGetWindowProperty(d->dpy, c->win,
			      d->atoms[ATOM_XEMBED_INFO],
			      0L, 10L,
			      0, XA_ATOM, &realType, &format,
			      &n, &extra, (unsigned char **) &value);
  if (status == Success)
    {
      if (realType == XA_CARDINAL && format == 32)
	{
	  /*
	    printf("VERSION: %i\n", value[0]);
	    printf("MAPPED:  %i\n", value[1]);
	  */
	  return (int)value[1];
	}
    }

  return -1;
}

void panel_send_xembed_message(
			      MBPanel *d,
			      MBPanelApp *c,
			      long message, /* message opcode */
			      long detail,  /* message detail */
			      long data1,  /* message data 1 */
			      long data2  /* message data 2 */
			      ){
  XEvent ev;
  memset(&ev, 0, sizeof(ev));
  ev.xclient.type = ClientMessage;
  ev.xclient.window = c->win;
  ev.xclient.message_type = d->atoms[ATOM_XEMBED_MESSAGE];
  ev.xclient.format = 32;
  ev.xclient.data.l[0] = CurrentTime;
  ev.xclient.data.l[1] = message;
  ev.xclient.data.l[2] = detail;
  ev.xclient.data.l[3] = data1;
  ev.xclient.data.l[4] = data2;
  XSendEvent(d->dpy, c->win, False, NoEventMask, &ev);
  XSync(d->dpy, False);
}

void panel_send_manage_message( MBPanel *d )
{
  XEvent ev;
  memset(&ev, 0, sizeof(ev));
  ev.xclient.type = ClientMessage;
  ev.xclient.window = d->win_root;
  ev.xclient.message_type = d->atoms[ATOM_MANAGER];
  ev.xclient.format = 32;
  ev.xclient.data.l[0] = CurrentTime;
  ev.xclient.data.l[1] = d->atoms[ATOM_SYSTEM_TRAY];
  ev.xclient.data.l[2] = d->win;

  XSendEvent(d->dpy, d->win_root, False, StructureNotifyMask, &ev);
  XSync(d->dpy, False);
}



/* Events */

void
panel_handle_button_event(MBPanel *panel, XButtonEvent *e)
{
  XEvent ev;
  int done = 0;
  Bool menu_was_active = 0;
  struct timeval then, now;
  DBG("%s() called, subwindow : %li\n", __func__, e->subwindow );

  if (!panel->use_menu) return; /* menu disabled */

  if (e->window == panel->win_scroll_button)
    {
      if (e->x < 8)
	{
	  panel_paint_scroller(panel, LEFT);
	} else {
	  panel_paint_scroller(panel, RIGHT);
	}
      while (!XCheckMaskEvent(panel->dpy, ButtonReleaseMask, &ev)) 
	{
	  if (e->x < 8)
	    {
	      panel_scroll(panel, LEFT, NULL);
	    }
	  else
	    {
	      panel_scroll(panel, RIGHT, NULL);
	    }
	  usleep(10);
	}
      panel_paint_scroller(panel, 0);
      session_save(panel);
      return;
    }

  if (e->window == panel->win_menu_button)
    {
      panel_paint_menubutt(panel, 1);
      
      while (!done) {
	if (XCheckMaskEvent(panel->dpy,ButtonReleaseMask, &ev))
	  if (ev.type == ButtonRelease) done=1;
      }
      menu_was_active = mb_menu_is_active(panel->mbmenu);
      panel_paint_menubutt(panel, 0);

      util_get_mouse_position(panel, &panel->click_x, &panel->click_y);
      if (!menu_was_active)
	{
	  mb_menu_activate(panel->mbmenu,panel->click_x,panel->click_y);
	}
      return;
    }

  if (e->window != panel->win) return;

  if (panel->is_hidden)
    {
      panel_toggle_visibilty(panel);
      return;
    }

  gettimeofday(&then, NULL);
  while (!done) {
    if (XCheckMaskEvent(panel->dpy,ButtonReleaseMask, &ev))
      if (ev.type == ButtonRelease) done=1;
    gettimeofday(&now, NULL);
    if ( (now.tv_usec-then.tv_usec) > (panel->click_time*1000) )
      done=2;
  }
  
  if (done == 2)
    {
      int dpy_h = DisplayHeight(panel->dpy, panel->screen);
      util_get_mouse_position(panel, &panel->click_x, &panel->click_y);
      mb_menu_activate(panel->mbmenu,
		       (panel->click_x-5 < 0) ? 2 : panel->click_x-5,
		       (panel->click_y+5 > dpy_h) ? dpy_h-2 : panel->click_y+5);
    }
}

void
panel_handle_expose(MBPanel *panel, XExposeEvent *e)
{
  panel_paint_menubutt(panel, 0);
  if (panel->scroll_active) panel_paint_scroller(panel, 0);
}

void
panel_handle_dock_request(MBPanel *panel, Window win)
{
  int app_origin_dist = 0;
  char *cmd_str = NULL;
  MBPanelApp *new_papp = NULL;

  util_get_command_str_from_win(panel, win, &cmd_str); /* cmd_str freed l8r */

  if (session_preexisting_win_matches_wanted(panel, win, cmd_str))
    {
      app_origin_dist = panel->session_init_offset;
      session_preexisting_clear_current(panel);
    }

  new_papp = panel_app_new(panel, win, cmd_str, app_origin_dist);

  XMapWindow(panel->dpy, new_papp->win);	
  
  XSelectInput(panel->dpy, new_papp->win, PropertyChangeMask );
  
  /* tell app its docked - panel app should now map its self */
  panel_send_xembed_message(panel, new_papp,
			    XEMBED_EMBEDDED_NOTIFY,
			    0, panel->win, 0);
  
  /* sent when it gets focus */
  panel_send_xembed_message(panel, new_papp,
			    XEMBED_WINDOW_ACTIVATE,
			    0,0,0);
  
  new_papp->mapped = True;


  panel_scroll_update_visibility(panel);

  panel_menu_add_remove_item(panel, new_papp);

  session_save(panel);

  session_preexisting_start_next(panel);
}

void
panel_handle_client_message(MBPanel *panel, XClientMessageEvent *e)
{
  DBG("%s() called\n", __func__ );
  if (e->message_type
      == panel->atoms[ATOM_SYSTEM_TRAY_OPCODE])
    {
      DBG("%s() got system tray message\n", __func__ );
      switch (e->data.l[1])
	{
	case SYSTEM_TRAY_REQUEST_DOCK:
	  DBG("%s() is SYSTEM_TRAY_REQUEST_DOCK\n", __func__ );
	  panel_handle_dock_request(panel, e->data.l[2]);
	  break;
	case SYSTEM_TRAY_BEGIN_MESSAGE:
	  DBG("%s() is SYSTEM_TRAY_BEGIN_MESSAGE\n", __func__ );
	  msg_new(panel, e);
	  break;
	case SYSTEM_TRAY_CANCEL_MESSAGE:
	  DBG("%s() is SYSTEM_TRAY_CANCEL_MESSAGE\n", __func__ );
	  break;
	}
      return;
    }

  if (e->message_type
      == panel->atoms[ATOM_NET_SYSTEM_TRAY_MESSAGE_DATA])
    {
      DBG("%s() got system tray message _data_\n", __func__ );
      msg_add_data(panel, e);
      return;
    }

  if (e->message_type
      == panel->atoms[ATOM_WM_DELETE_WINDOW])
    {
      util_cleanup_children(0);
    }
   
  return;
}

void
panel_handle_property_notify(MBPanel *panel, XPropertyEvent *e)
{
  MBPanelApp *papp;

  if (e->atom == panel->atoms[ATOM_XEMBED_INFO])
    {
      int i; 
      papp = panel_app_get_from_window(panel, e->window); 
      DBG("%s() got XEMBED_INFO property notify for %s\n", 
	  __func__, papp->name );
      if (papp != NULL)
	{
	  i = panel_get_map_state(panel, papp);
	  if (i == 1)
	    {
	      XMapRaised(panel->dpy, papp->win);
	      papp->mapped = True;
	    }
	  if (i == 0)
	    {
	      XUnmapWindow(panel->dpy, papp->win);
	      papp->mapped = False;
	    }
	}
      return;
    }

  if (e->atom == panel->atoms[ATOM_MB_THEME])
    {
      panel_set_theme_from_root_prop(panel);
      return;
    }
  
  


  /* XXX catch root pixmap chnaging .. */
}

static Bool
get_xevent_timed(Display* dpy, XEvent* event_return, struct timeval *tv)
{
  if (tv == NULL) 
    {
      XNextEvent(dpy, event_return);
      return True;
    }

  XFlush(dpy);

  if (XPending(dpy) == 0) 
    {
      int fd = ConnectionNumber(dpy);
      fd_set readset;
      FD_ZERO(&readset);
      FD_SET(fd, &readset);
      if (select(fd+1, &readset, NULL, NULL, tv) == 0) 
	{
	  return False;
	} else {
	  XNextEvent(dpy, event_return);
	  return True;
	}
    } else {
      XNextEvent(dpy, event_return);
      return True;
    }
}

void 
panel_main(MBPanel *panel)
{
  MBPanelApp *papp;
  XEvent an_event;
  int xfd;

  XSelectInput (panel->dpy, panel->win, StructureNotifyMask|ExposureMask|
		SubstructureRedirectMask|SubstructureNotifyMask|
		ButtonPressMask|ButtonReleaseMask|PointerMotionMask|
		PropertyChangeMask);

  XSelectInput (panel->dpy, panel->win_scroll_button, 
		ExposureMask|ButtonPressMask|ButtonReleaseMask);

  XSelectInput (panel->dpy, panel->win_menu_button, 
		ExposureMask|ButtonPressMask|ButtonReleaseMask);

  XSelectInput (panel->dpy, panel->win_root, 
		PropertyChangeMask|StructureNotifyMask);

  xfd = ConnectionNumber (panel->dpy);

  XFlush(panel->dpy);

  while(1)
    {
      struct timeval tvt;
      tvt.tv_usec = 500;
      tvt.tv_sec = 0;
      
#if 0 /* XXX replace below with dnotify  */
      if (panel->reload_pending && !mb_menu_is_active(panel->mbmenu))
	{
	  mb_menu_free(panel->mbmenu);
	  panel_menu_init(panel);
	  for (papp = panel->panel_app_head; papp != NULL; papp = papp->next)
	    if (papp->name)
	      mb_menu_add_item_to_menu(panel->mbmenu, panel->remove_menu, 
				       papp->name, NULL, 
				       papp->name, panel_menu_kill_cb, 
				       (void *)papp, MBMENU_NO_SORT);
	  panel->reload_pending = False;
	}
#endif
	
      /*	while (XPending(panel->dpy))
		{ 
	    XNextEvent(panel->dpy, &an_event);
      */
      if (get_xevent_timed(panel->dpy, &an_event, &tvt))
	{
	  mb_menu_handle_xevent(panel->mbmenu, &an_event);
	  switch (an_event.type)
	    {
	    case ButtonPress:
	      panel_handle_button_event(panel, &an_event.xbutton);
	      break;
	    case ClientMessage:
	      panel_handle_client_message(panel, &an_event.xclient);
	      break;
	    case PropertyNotify:
	      panel_handle_property_notify(panel, &an_event.xproperty);
	      break;
	    case MapRequest:
	      break;
	    case UnmapNotify:
	      /* window should unmap ... we destroy it */
	      papp = panel_app_get_from_window(panel, 
					       an_event.xunmap.window);
	      if (papp)
		{
		  panel_app_destroy(panel, papp);

		  /* remove any connected message windows */
		  if (panel->msg_win && panel->msg_win_sender == papp)
		    {
		      XDestroyWindow(panel->dpy, panel->msg_win);
		      panel->msg_win = None;
		    }

		  panel_scroll_update_visibility(panel);
		  /* 
		   *  We set an alarm here, so the session file is only
		   *  updated after a couple fo seconds. 
		   *  This is done so if the xserver is closing and bringing 
		   *  down all the clients. Its very likely it'll kill 
		   *  a panelapp before the panel, and we dont really want it
		   *  to be refmoved from the session file.
		   */
		  alarm(3);
		}
	      break;
	    case Expose:
	      panel_handle_expose(panel, &an_event.xexpose);
	      break;
	    case DestroyNotify:
	      break;
	    case ConfigureRequest:
	      panel_app_handle_configure_request(panel, 
						 &an_event.xconfigurerequest);
	      break;
	    case ConfigureNotify:
	      
	      if (an_event.xconfigure.window == panel->win)
		{
		  DBG("%s(): configureNotify on panel\n", __func__);
		  panel->x = an_event.xconfigure.x;
		  panel->y = an_event.xconfigure.y;
		  panel->w = an_event.xconfigure.width;
		  panel->h = an_event.xconfigure.height;
		  
		  for(papp = panel->panel_app_head; 
		      papp != NULL;
		      papp = papp->next)
		    panel_app_deliver_config_event(panel, papp);
		  
		  panel_scroll_update_visibility(panel);

    
		} 
	      else if (an_event.xconfigure.window == panel->win_root
		       && panel->use_flip)
		{
		  /* rotation */
		  switch (panel->orientation)
		    {
		    case South:
		      panel_change_orientation(panel, West); 
		      break;
		    case North:
		      panel_change_orientation(panel, East); 
		      break;
		    case West:
		      panel_change_orientation(panel, South); 
		      break;
		    case East:
		      panel_change_orientation(panel, North); 
		      break;
		    }
		}
	      break;
	    }
	  msg_handle_events(panel, &an_event);
	}
	    /* } */
      session_preexisting_handle_timeouts(panel);
      msg_handle_timeouts(panel);

      /*
	    XFlush(panel->dpy);

	    FD_ZERO (&fd);
	    FD_SET (xfd, &fd);
	    
	    if (select (xfd+1, &fd, NULL, NULL, &tvt) == 0)
	      {
		session_preexisting_handle_timeouts(panel);
		msg_handle_timeouts(panel);
	      }
      */    
    }
}

void
panel_change_orientation(MBPanel *panel, 
			 MBPanelOrientation new_orientation)
{
  /* 
     - Unmap the dock.

     has it changed vertical -> horizontal ?

     Yes 
       -> resize dock
       -> change offsets, call move_to

     No
       -> simply move the dock window

     - Remap the dock

  */
  XUnmapWindow(panel->dpy, panel->win);

  if ( ( (panel->orientation == East || panel->orientation == West)
	 && (new_orientation == North || new_orientation == South) )
       || 
       ( (panel->orientation == North || panel->orientation == South)
	 && (new_orientation == East || new_orientation == West ) )
       )
    {
      MBPanelApp *papp_cur = panel->panel_app_head;

      panel->x = 0;
      panel->y = 0;
      panel->h = PANEL_HEIGHT;
      panel->w = DisplayWidth(panel->dpy, panel->screen); 

      switch (new_orientation)
	{
	case South:
	  panel->y = DisplayHeight(panel->dpy, panel->screen) - panel->h; 
	  break;
	case North:
	  break;
	case West:
	  panel->w = PANEL_HEIGHT;
	  panel->h = DisplayHeight(panel->dpy, panel->screen);
	  break;
	case East:
	  panel->w = PANEL_HEIGHT;
	  panel->h = DisplayHeight(panel->dpy, panel->screen);
	  panel->x = DisplayWidth(panel->dpy, panel->screen) - panel->w; 
	  break;
	}

      panel->orientation = new_orientation;

      XMoveResizeWindow( panel->dpy, panel->win, panel->x, panel->y,
			 panel->w, panel->h );

      /* move_to() will reposition each app */
      while (papp_cur != NULL)
	{
	  panel_app_move_to(panel, papp_cur, papp_cur->offset);
	  papp_cur = papp_cur->next;
	}
    }
  else
    {
      switch (new_orientation)
	{
	case South:
	  panel->y = DisplayHeight(panel->dpy, panel->screen) - panel->h; 
	  break;
	case North:
	  panel->y = 0;
	  break;
	case West:
	  panel->x = 0;
	  break;
	case East:
	  panel->x = DisplayWidth(panel->dpy, panel->screen) - panel->w; 
	  break;
	}

      panel->orientation = new_orientation;

      XMoveResizeWindow( panel->dpy, panel->win, panel->x, panel->y,
			 panel->w, panel->h );
    }

  XMapWindow(panel->dpy, panel->win);
  XMapSubwindows(panel->dpy, panel->win);

  XFlush(panel->dpy);
}


MBPanel 
*panel_init(int argc, char *argv[])
{

  XGCValues gv;
  XSetWindowAttributes dattr;
  XpmAttributes xpm_attr;
  XSizeHints size_hints;

  unsigned long wm_struct_vals[4];

  char *geometry_str = NULL;
  char *color_def = NULL;
  char *bg_pixmap_def = NULL;
  char *display_name = (char *)getenv("DISPLAY");

  char tray_atom_spec[128] = { 0 }; 
  char tray_id_env_str[16] = { 0 };

  MBPanelApp *papp_menu_button = NULL;
  MBPanel    *panel;

  int panel_border_sz = 0;
  char *want_trans = NULL;

#ifdef USE_XFT  
  XRenderColor colortmp;
#endif
  int i = 0, j = 0;

  struct {
    char *name;
    MBPanelOrientation orientation;
  } orientation_lookup[] = {
    { "north", North },
    { "south", South },
    { "east",  East  },
    { "west",  West  },
    {  NULL,   North }
  };

  XSetErrorHandler(util_handle_xerror);
   
  panel = NEW(MBPanel);
  memset(panel, sizeof(MBPanel), 0);

  /* defualts */
  panel->init_padding = 4;
  panel->padding = 2;
  panel->use_session  = True;
  panel->use_menu     = True;
  panel->click_time = 400;
  panel->use_overide_wins = False;
  panel->orientation = South;
  panel->system_tray_id = 0;
  panel->use_flip = False;

  for (i = 1; i < argc; i++) {
    if (!strcmp ("-display", argv[i]) || !strcmp ("-d", argv[i])) {
      if (++i>=argc) panel_usage (argv[0]);
      display_name = argv[i];
      continue;
    }
    if (!strcmp ("--no-session", argv[i]) || !strcmp ("-ns", argv[i])) {
      panel->use_session  = False;
      continue;
    }
    if (!strcmp ("--no-menu", argv[i]) || !strcmp ("-nm", argv[i])) {
      panel->use_menu  = False;
      continue;
    }
    if (!strcmp ("--overide-bubbles", argv[i]) || !strcmp ("-o", argv[i])) {
      panel->use_overide_wins = True;
      continue;
    }
    if (!strcmp ("--bgcolor", argv[i]) || !strcmp ("-c", argv[i])) {
      if (++i>=argc) panel_usage (argv[0]);
      color_def = argv[i];
      continue;
    }
    if (strstr (argv[i], "-geometry") || !strcmp ("-g", argv[i])) {
      if (++i>=argc) panel_usage (argv[0]);
      geometry_str = argv[i];
      continue;
    }
    if (!strcmp ("--bgpixmap", argv[i]) || !strcmp ("-b", argv[i])) {
      if (++i>=argc) panel_usage (argv[0]);
      bg_pixmap_def = argv[i];
      continue;
    }
    if (!strcmp ("--bgtrans", argv[i]) || !strcmp ("-bt", argv[i])) {
      if (++i>=argc) panel_usage (argv[0]);
      want_trans = argv[i];
      continue;
    }
    if (!strcmp ("--id", argv[i]) || !strcmp ("-i", argv[i])) {
      if (++i>=argc) panel_usage (argv[0]);
      panel->system_tray_id = atoi(argv[i]);
      continue;

    }
    if (!strcmp ("--border", argv[i])) {
      if (++i>=argc) panel_usage (argv[0]);
      panel_border_sz = atoi(argv[i]);
      continue;
    }
    if (!strcmp ("--orientation", argv[i])) {
      Bool found = False;
      if (++i>=argc) panel_usage (argv[0]);
      j = 0;
      while (orientation_lookup[j].name != NULL)
	{
	  if (!strcasecmp(orientation_lookup[j].name, argv[i]))
	    {
	      panel->orientation = orientation_lookup[j].orientation;
	      found = True;
	    }
	  j++;
	}
      if (found) continue;
    }
    panel_usage(argv[0]);
  }

  if ((panel->dpy = XOpenDisplay(display_name)) == NULL)
    {
      fprintf(stderr, "%s: failed to open display\n", argv[0]);
      exit(1);
    }

  panel->screen = DefaultScreen(panel->dpy);
  panel->win_root = RootWindow(panel->dpy, panel->screen);

  panel->panel_app_head = NULL;

  panel->x = 0;
  panel->y = 0;
  panel->h = PANEL_HEIGHT;
  panel->w = DisplayWidth(panel->dpy, panel->screen); 
  

  switch (panel->orientation)
    {
    case South:
      panel->y = DisplayHeight(panel->dpy, panel->screen) - panel->h; 
      break;
    case North:
      break;
    case West:
      panel->w = PANEL_HEIGHT;
      panel->h = DisplayHeight(panel->dpy, panel->screen);
      break;
    case East:
      panel->w = PANEL_HEIGHT;
      panel->h = DisplayHeight(panel->dpy, panel->screen);
      panel->x = DisplayWidth(panel->dpy, panel->screen) - panel->w; 
      break;
    }

  panel->bg_pxm = None;

  if (geometry_str)
    XParseGeometry(geometry_str, &panel->x, &panel->y, &panel->w, &panel->h);  

  /* XXX use defines */
  panel->atoms[0] = XInternAtom(panel->dpy, "_NET_WM_WINDOW_TYPE"     ,False);
   
  panel->atoms[1] = XInternAtom(panel->dpy, "_NET_WM_WINDOW_TYPE_DOCK",False);


  panel->atoms[3] = XInternAtom(panel->dpy, "_NET_SYSTEM_TRAY_OPCODE", False);
  panel->atoms[4] = XInternAtom(panel->dpy, "_XEMBED_INFO",            False);
  panel->atoms[5] = XInternAtom(panel->dpy, "_XEMBED",                 False);
  panel->atoms[6] = XInternAtom(panel->dpy, "MANAGER",                 False);

  panel->atoms[7] = XInternAtom(panel->dpy, "_MB_DOCK_ALIGN",          False);
  panel->atoms[8] = XInternAtom(panel->dpy, "_MB_DOCK_ALIGN_EAST",     False);

  panel->atoms[9] = XInternAtom(panel->dpy, "_NET_SYSTEM_TRAY_MESSAGE_DATA", False);
  panel->atoms[10] = XInternAtom(panel->dpy, "_NET_WM_WINDOW_TYPE_SPLASH", False);

  panel->atoms[11] = XInternAtom(panel->dpy, "WM_PROTOCOLS", False);
  panel->atoms[12] = XInternAtom(panel->dpy, "WM_DELETE_WINDOW", False);

  panel->atoms[13] = XInternAtom(panel->dpy, "_MB_THEME", False);

  panel->atoms[14] = XInternAtom(panel->dpy, "_MB_PANEL_TIMESTAMP", False);

  panel->atoms[15] = XInternAtom(panel->dpy, "_NET_WM_STRUT", False);

  panel->atoms[16] = XInternAtom(panel->dpy, "_MB_PANEL_BG", False);

  panel->atoms[17] = XInternAtom(panel->dpy, "WM_CLIENT_LEADER", False);

  panel->atoms[18] = XInternAtom(panel->dpy, "_NET_WM_ICON", False);

  panel->atoms[ATOM_NET_WM_PID] 
    = XInternAtom(panel->dpy, "_NET_WM_PID", False); 

  panel->atoms[ATOM_XROOTPMAP_ID] 
    = XInternAtom(panel->dpy, "_XROOTPMAP_ID", False);

  /* Set selection atom */
  snprintf(tray_atom_spec, 128,"_NET_SYSTEM_TRAY_S%i", panel->system_tray_id); 
  panel->atoms[ATOM_SYSTEM_TRAY] 
    = XInternAtom(panel->dpy, tray_atom_spec, False);
  snprintf(tray_id_env_str, 16, "%i", panel->system_tray_id);
  setenv("SYSTEM_TRAY_ID", tray_id_env_str, 1);

  panel->pb = mb_pixbuf_new(panel->dpy, panel->screen);
   
  gv.graphics_exposures = False;
  gv.function   = GXcopy;
  gv.foreground = WhitePixel(panel->dpy, panel->screen);

  panel->gc = XCreateGC(panel->dpy, panel->win_root,
		    GCGraphicsExposures|GCFunction|GCForeground, &gv);

  /*
  if (color_def != NULL)
    util_xcol_from_spec(panel, &panel->xcol,  color_def);
  else 
    util_xcol_from_spec(panel, &panel->xcol,  DEFAULT_COLOR_SPEC);
  */

  dattr.background_pixel = panel->xcol.pixel;
   
  panel->win = XCreateWindow(panel->dpy, panel->win_root,
			 panel->x, panel->y, 
			 panel->w, panel->h,
			 panel_border_sz,
			 CopyFromParent,
			 CopyFromParent,
			 CopyFromParent,
			 CWBackPixel,
			 &dattr);

  size_hints.flags      = PPosition | PSize | PMinSize;
  size_hints.x          = panel->x;
  size_hints.y          = panel->y;
  size_hints.width      = panel->w;
  size_hints.height     = panel->h;
  size_hints.min_width  = panel->w;
  size_hints.min_height = panel->h;
    
  XSetStandardProperties(panel->dpy, panel->win, tray_atom_spec, 
			 tray_atom_spec, 0, argv, argc, &size_hints);


  panel->win_scroll_button = XCreateWindow(panel->dpy, panel->win,
					   panel->w - SCROLL_WIN_WIDTH, 
					   0, 
					   SCROLL_WIN_WIDTH, 
					   panel->h, 
					   0,
					   CopyFromParent,
					   CopyFromParent,
					   CopyFromParent,
					   CWBackPixel,
					   &dattr);

  panel->win_menu_button = XCreateWindow(panel->dpy, panel->win_root,
					 0, 0, 
					 PANEL_IS_VERTICAL(panel) ?
					 panel->w : 8, 
					 PANEL_IS_VERTICAL(panel) ?
					 8 : panel->h, 
					 0,
					 CopyFromParent,
					 CopyFromParent,
					 CopyFromParent,
					 CWBackPixel,
					 &dattr);


  xpm_attr.valuemask = XpmCloseness;
  xpm_attr.closeness = 40000;

  if (XpmCreatePixmapFromData( panel->dpy, 
			       panel->win_scroll_button, scrollbutton_xpm,
			       &panel->scroller_pxm, &panel->scroller_mask, 
			       &xpm_attr)
      != XpmSuccess )
    {
      fprintf(stderr, "failed to get scrollbutton image\n");
      exit(1);
    }
  
  if (XpmCreatePixmapFromData( panel->dpy, 
			       panel->win_menu_button, menubutt_xpm,
			       &panel->menubutt_pxm, &panel->menubutt_mask, 
			       &xpm_attr)
      != XpmSuccess )
    {
      fprintf(stderr, "failed to get menu button image\n");
      exit(1);
    }

  XChangeProperty(panel->dpy, panel->win, 
		  panel->atoms[ATOM_WM_WINDOW_TYPE], XA_ATOM, 32, 
		  PropModeReplace,
		  (unsigned char *) &panel->atoms[ATOM_WM_WINDOW_TYPE_DOCK], 
		  1);

  /* Set our ewmh reserved space XXX reset this when we hide */
  wm_struct_vals[0] = ( panel->orientation == West ) ? panel->w : 0;
  wm_struct_vals[1] = ( panel->orientation == East ) ? panel->w : 0;
  wm_struct_vals[2] = ( panel->orientation == North ) ? panel->h : 0;
  wm_struct_vals[3] = ( panel->orientation == South ) ? panel->h : 0;

  XChangeProperty(panel->dpy, panel->win, panel->atoms[ATOM_NET_WM_STRUT], 
		  XA_CARDINAL, 32, PropModeReplace,
		  (unsigned char *)wm_struct_vals, 4);
  
  panel->msg_queue_start = NULL;
  panel->msg_queue_end   = NULL;
  panel->msg_win = None;   

  util_xcol_from_spec(panel, &panel->msg_xcol, "gold");
  util_xcol_from_spec(panel, &panel->msg_urgent_xcol, "orange");

  panel->msg_gc = XCreateGC(panel->dpy, panel->win_root,
			GCGraphicsExposures|GCFunction|GCForeground, &gv);

#ifdef USE_XFT
  if ((panel->msg_font = XftFontOpenName(panel->dpy, panel->screen, 
					 "sans,verdana-6:bold")) == NULL)
    { printf("Cant open XFT font\n"); exit(0); }

  colortmp.red   = 0;
  colortmp.green = 0;
  colortmp.blue  = 0;
  colortmp.alpha = 0xffff;
  XftColorAllocValue(panel->dpy,
		     DefaultVisual(panel->dpy, panel->screen),
		     DefaultColormap(panel->dpy, panel->screen),
		     &colortmp,
		     &panel->fg_xftcol);
#else
  if (!(panel->msg_font = XLoadQueryFont(panel->dpy, "5x7")))
    {
      printf("5x7 font not found, giving up...\n");
      exit(1);
    }
  XSetFont(panel->dpy, panel->msg_gc, panel->msg_font->fid);
#endif

  panel->msg_font_height = panel->msg_font->ascent + panel->msg_font->descent; 

  panel->next_click_is_not_double = True;
  panel->is_hidden = False;
  
  XSetWMProtocols(panel->dpy, panel->win , &panel->atoms[11], 2);

  panel->mbmenu = NULL;
  panel_menu_init(panel); 
  
  G_panel = panel;                /* global for sig handlers :(  */
  panel->reload_pending = False;

  util_install_signal_handlers();

  /* Set the theme etc */

  panel->use_themes = True;

  if (color_def != NULL) 
    { 
      panel_set_bg(panel, BG_SOLID_COLOR, color_def); 
      mb_menu_set_col(panel->mbmenu, BG_COL, color_def);
      panel->use_themes = False;
    }
  else if (bg_pixmap_def) 
    {  
      panel_set_bg(panel, BG_PIXMAP, bg_pixmap_def); 
      panel->use_themes = False;
    }
  else if (want_trans)
    {  
      panel_set_bg(panel, BG_TRANS,  want_trans ); 
      panel->use_themes = False;
    }
  else
    {  
      if (!panel_set_theme_from_root_prop(panel)) 
	{
	  panel_set_bg(panel, BG_SOLID_COLOR, DEFAULT_COLOR_SPEC );
	  mb_menu_set_col(panel->mbmenu, BG_COL, DEFAULT_COLOR_SPEC );
	}
    }

  panel->click_x = 0;
  panel->click_y = 0;
  panel->scroll_active = False;
  panel->scroll_offset = 0;

  panel->cursors[CURS_SLIDER] 
    = XCreateFontCursor(panel->dpy, XC_sb_h_double_arrow);
  panel->cursors[CURS_ORIG] 
    = XCreateFontCursor(panel->dpy, XC_left_ptr);

  papp_menu_button = panel_app_new(panel, panel->win_menu_button, NULL, 0);
  papp_menu_button->ignore = True;

  if (panel->use_menu)
    XMapRaised(panel->dpy, panel->win_menu_button);

  return panel;

}

int main(int argc, char *argv[])
{
  MBPanel *panel;

  panel = panel_init(argc, argv);

  /* Attempt to own the system tray selection  */
  if (!XGetSelectionOwner(panel->dpy, panel->atoms[ATOM_SYSTEM_TRAY]))
    {
      XSetSelectionOwner(panel->dpy, panel->atoms[ATOM_SYSTEM_TRAY],
			 panel->win, CurrentTime);
    } else {
      fprintf(stderr, "Panel already exists. aborting. Try running mbdock with the --id switch.\n"); 
      exit(0);
    }

  /* Announce to any clients that are interested that we have it */
  panel_send_manage_message( panel ); 

  panel_paint_menubutt(panel, 0);

  XMapWindow(panel->dpy, panel->win);

  session_init(panel);

  panel_main(panel);

  return 0;
}

#ifdef USE_XSETTINGS

#if 0

system_tray_id;

/MATCHBOX/PANEL/XX/ORIENTATION north|south|east|west
/MATCHBOX/PANEL/XX/BACKGROUND  default|rgb:XXX|pxm:XXX|trans:XXX


#define XSET_UNKNOWN    0
#define XSET_THEME      1
#define XSET_CURSOR     2
#define XSET_LOWLIGHT   3
#define XSET_TITLEBARS  4

static void
wm_xsettings_notify_cb (const char       *name,
			XSettingsAction   action,
			XSettingsSetting *setting,
			void             *data)
{
  Wm *w = (Wm *)data;
  int i = 0;
  int key = XSET_UNKNOWN;
  
  struct _mb_xsettings { char *name; int value; } mb_xsettings[] = {
    { "/MATCHBOX/THEME",     XSET_THEME     },
    { "/MATCHBOX/CURSOR",    XSET_CURSOR    },
    { "/MATCHBOX/LOWLIGHT",  XSET_LOWLIGHT  },
    { "/MATCHBOX/TITLEBARS", XSET_TITLEBARS },
    { NULL,       -1 } 
  };

  while(  mb_xsettings[i].name != NULL )
    {
      if (!strcmp(name, mb_xsettings[i].name)
	  && setting != NULL 	/* XXX set to NULL when action deleted */
	  && setting->type == XSETTINGS_TYPE_STRING )
	{
	  key = mb_xsettings[i].value;
	  break;
	}
      i++;
    }
    
  if (key == XSET_UNKNOWN) return;

  switch (action)
    {
    case XSETTINGS_ACTION_NEW:
    case XSETTINGS_ACTION_CHANGED:
      switch (key)
	{
	case XSET_THEME:
	  if (w->flags & STARTUP_FLAG)
	      w->config->theme_file = strdup(setting->data.v_string);
	  else
	      mbtheme_switch(w, setting->data.v_string);
	  break;
	case XSET_LOWLIGHT:
	  if (!strcasecmp("true", setting->data.v_string))
	      w->config->dialog_shade = True;
	  else
	      w->config->dialog_shade = False;
	  break;
	case XSET_CURSOR:
	  if (!strcasecmp("true", setting->data.v_string))
	    wm_set_cursor_visibility(w, True);
	  else 
	    wm_set_cursor_visibility(w, False);
	  break;
	case XSET_TITLEBARS:
	  /* XXX todo */
	  break;

	}
    case XSETTINGS_ACTION_DELETED:
      /* Do nothing for now */
      break;
    }
}

#endif

#endif
